************************************************************************
* This file is part of OpenMolcas.                                     *
*                                                                      *
* OpenMolcas is free software; you can redistribute it and/or modify   *
* it under the terms of the GNU Lesser General Public License, v. 2.1. *
* OpenMolcas is distributed in the hope that it will be useful, but it *
* is provided "as is" and without any express or implied warranties.   *
* For more details see the full text of the license in the file        *
* LICENSE or in <http://www.gnu.org/licenses/>.                        *
*                                                                      *
* Copyright (C) 2007, Ten-no Research Group                            *
*               2012, Thomas Bondo Pedersen                            *
************************************************************************
*NOTICE===NOTICE========NOTICE========NOTICE========NOTICE========NOTICE
*                                                                      *
*  Thomas Bondo Pedersen, Nov. 28, 2012:                               *
*  Part of this code (subroutine Remez and everything called in that   *
*  routine) was obtained from S. Ten-no; conditions (license or other  *
*  terms) are unknown at present. Please do not use without consulting *
*  T.B. Pedersen first!                                                *
*                                                                      *
*NOTICE===NOTICE========NOTICE========NOTICE========NOTICE========NOTICE
      Subroutine MinimaxLaplace(Verbose,N,xmin,xmax,l_wt,w,t,irc)
C
C     Thomas Bondo Pedersen, November 2012.
C
C     Generate grid (weights w and points t) for the
C     Laplace transform of 1/x.
C
C     The minimax approximation for setting up the grid
C     [Takatsuka, Ten-no, Hackbusch; JCP 129, 044112 (2008)]
C     is used. The source code for the minimax approximation
C     was provided by Ten-no on Nov. 28, 2012 (to TBP).
C
C     Verbose   -- boolean to control printing
C     N         -- number of grid points requested. If N=0 on input,
C                  the number of points required to get an accuracy
C                  of 1.0d-6 (in the Laplace transform) is used. In
C                  this case, N is the required number of grid points
C                  on return.
C     xmin,xmax -- range of x values to be covered.
C     l_wt      -- allocated dimension of the arrays w and t.
C                  If N(input)=0 and l_wt<N(output), only l_wt grid
C                  points are returned in w and t. N(output) is still
C                  the full number of grid points [this situation is
C                  also flagged by the return code irc, see below].
C     w         -- grid weights are returned in array w.
C     t         -- grid points are returned in array t.
C     irc       -- return code:
C                  <0: input error (parameter -irc is the culprit)
C                   0: all OK
C                   1: l_wt is too small to store full result and only
C                      l_wt grid weights/points are returned.
C
      Implicit None
      Logical Verbose
      Integer N
      Real*8  xmin
      Real*8  xmax
      Integer l_wt
      Real*8  w(l_wt)
      Real*8  t(l_wt)
      Integer irc
#include "stdalloc.fh"

      Integer mGrid
      Parameter (mGrid=20) ! limited by Remez implementation

      Character*8 DefaultGrid
      Parameter (DefaultGrid='MICRO   ')

      Integer     l_Coeff
      Real*8, Allocatable :: Coeff(:)
      Integer     K_Lap
      Character*8 Demand
      Logical     Inf

      Integer i

      irc=0
      If (N.lt.0 .or. N.gt.mGrid) Then
         irc=-1
         Return
      End If
      If (xmin.lt.0.0d0) Then
         irc=-2
         Return
      End If
      If ((xmax-xmin).lt.0.0d0) Then
         irc=-3
         Return
      End If
      If (l_wt.lt.1) Then
         irc=-4
         Return
      End If

      K_Lap=N
      If (N.eq.0) Then
         Demand=DefaultGrid
      Else
         Demand='        '
      End If
      l_Coeff=2*mGrid
      Call mma_Allocate(Coeff,l_Coeff,Label='LapCoef')
      Inf=.false.
      Call Remez(Verbose,K_Lap,xmin,xmax,Coeff,Demand,Inf)
      If (K_Lap.lt.0) Then
         Call mma_Deallocate(Coeff)
         irc=-1
         Write(6,'(A,I10)')
     &   'MinimaxLaplace: Remez returned K_Lap=',K_Lap
         Return
      End If
      If (N.eq.0) N=K_Lap
      If (l_wt.lt.K_Lap) Then
         Do i=1,l_wt
            w(i)=Coeff(2*i-1)
            t(i)=Coeff(2*i)
         End Do
         irc=2
      Else
         Do i=1,K_Lap
            w(i)=Coeff(2*i-1)
            t(i)=Coeff(2*i)
         End Do
      End If
      Call mma_Deallocate(Coeff)

      End
      Subroutine Remez_SetupPrint(print_to_molcas_log)
      Implicit None
      Logical print_to_molcas_log
#include "remez.fh"
      Integer ini
      Integer  isFreeUnit
      External isFreeUnit
      If (print_to_molcas_log) Then
         IW=6
      Else
         ini=7
         IW=isFreeUnit(ini)
         Call Molcas_Open(IW,'REMEZ')
      End If
      End
      Subroutine Remez_ShutDownPrint(print_to_molcas_log)
      Implicit None
      Logical print_to_molcas_log
#include "remez.fh"
      If (.not.print_to_molcas_log .and. IW.gt.0) Then
         Close(IW)
         IW=-999
      End If
      End
      Integer Function TestMinimaxLaplace(Tolerance,Verbose)
C
C     Thomas Bondo Pedersen, December 2012.
C
C     Test that the minimax code from Ten-no gives correct results
C     (test case provided by Ten-no). The minimax code is called
C     through routine MinimaxLaplace.
C
C     Input:
C        Tolerance -- >=0: defines tolerance for test
C                     <0:  default tolerance used
C        Verbose   -- If .true., print information
C
C     Returns:
C        -1: nonzero return code from MinimaxLaplace
C         0: weights OK, grid points OK
C         1: weights wrong, grid points OK
C         2: weights OK, grid points wrong
C         3: weights wrong, grid points wrong
C
      Implicit None
      Real*8  Tolerance
      Logical Verbose
#include "stdalloc.fh"
      Real*8, Allocatable:: tmlwr(:), tmltr(:), tmlw(:), tmlt(:)

      Real*8   dDot_
      external ddot_

      Integer N
      Parameter (N=8)

      Real*8 xmin, xmax
      Parameter (xmin=1.08976414D0, xmax=1.08976414D0)

      Integer K_Lap
      Integer irc
      Integer l_w_ref
      Integer l_t_ref
      Integer l_w
      Integer l_t
      Integer l_wt

      Real*8 Tol
      Real*8 Emin, Emax
      Real*8 RMSw, RMSt

      If (Verbose) Then
         Write(6,'(//,A)')
     &                  '>>>>>>>>>>>>>>>>>>>>>><<<<<<<<<<<<<<<<<<<<<<<<'
         Write(6,'(A)') '>>>>>>>>>> Enter TestMinimaxLaplace <<<<<<<<<<'
         Write(6,'(A,//)')
     &                  '>>>>>>>>>>>>>>>>>>>>>><<<<<<<<<<<<<<<<<<<<<<<<'
         Call xFlush(6)
      End If

      If (Tolerance.lt.0.0d0) Then
         Tol=1.0d-7
      Else
         Tol=Tolerance
      End If

      l_w_ref=N
      l_t_ref=N
      l_w=N
      l_t=N
      Call mma_allocate(tmlwr,l_w_ref,Label='tmlwr')
      Call mma_allocate(tmltr,l_t_ref,Label='tmltr')
      Call mma_allocate(tmlw,l_w,Label='tmlw')
      Call mma_allocate(tmlt,l_t,Label='tmlt')

      tmlwr(1)=0.097293042801116517
      tmlwr(2)=0.237233954792368945
      tmlwr(3)=0.407050561470959249
      tmlwr(4)=0.635894859784969846
      tmlwr(5)=0.973101625527261427
      tmlwr(6)=1.505487565587009913
      tmlwr(7)=2.419319385961121949
      tmlwr(8)=4.393171503147499379

      tmltr(1)=0.03771106629456916
      tmltr(2)=0.20333950465288628
      tmltr(3)=0.52200685559948912
      tmltr(4)=1.03690003666258002
      tmltr(5)=1.82953857756319826
      tmltr(6)=3.04727454214513793
      tmltr(7)=4.96421466224179220
      tmltr(8)=8.21146012481546705

      K_Lap=N
      Emin=xmin
      Emax=xmax
      l_wt=N
      Call MinimaxLaplace(Verbose,K_Lap,Emin,Emax,l_wt,tmlw,tmlt,irc)
      If (Verbose) Then
         Write(6,'(A,I6)') 'Return code from MinimaxLaplace=',irc
         Call xFlush(6)
      End If
      If (irc.ne.0) Then
         irc=-1
      Else
         Call dAXPY_(N,-1.0d0,tmlwr,1,tmlw,1)
         Call dAXPY_(N,-1.0d0,tmltr,1,tmlt,1)
         RMSw=sqrt(dDot_(N,tmlw,1,tmlw,1)/dble(N))
         RMSt=sqrt(dDot_(N,tmlt,1,tmlt,1)/dble(N))
         If (Verbose) Then
            Write(6,'(A,1P,D25.16)') 'Weight RMS error=    ',RMSw
            Write(6,'(A,1P,D25.16)') 'Grid point RMS error=',RMSt
            Write(6,'(A,1P,D25.16)') 'Tolerance=           ',Tol
            Call xFlush(6)
         End If
         irc=0
         If (RMSw.gt.Tol) irc=irc+1
         If (RMSt.gt.Tol) irc=irc+2
      End If

      Call mma_deallocate(tmlt)
      Call mma_deallocate(tmlw)
      Call mma_deallocate(tmltr)
      Call mma_deallocate(tmlwr)

      If (Verbose) Then
         Write(6,'(A,I3)') 'TestMinimaxLaplace=',irc
         Write(6,'(//,A)')
     &                  '>>>>>>>>>>>>>>>>>>>>>><<<<<<<<<<<<<<<<<<<<<<<'
         Write(6,'(A)') '>>>>>>>>>> Exit TestMinimaxLaplace <<<<<<<<<<'
         Write(6,'(A,//)')
     &                  '>>>>>>>>>>>>>>>>>>>>>><<<<<<<<<<<<<<<<<<<<<<<'
         Call xFlush(6)
      End If

      TestMinimaxLaplace=irc

      End
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C  Thomas Bondo Pedersen, Nov. 28, 2012:
C  This code was obtained from S. Ten-no; conditions (license or other
C  terms) are unknown at present.
C  It implements the minimax approximation for the numerical
C  quadrature for Laplace transformation of 1/x. Weights and
C  points are returned in Coeff:
C     Coeff(1) = Omega(1)   Coeff(2) = Alpha(1)
C     Coeff(3) = Omega(2)   Coeff(4) = Alpha(2)
C     etc.
C  Reference: Takatsuka, Ten-no, Hackbusch; JCP 129, 044112 (2008)
C
*MODULE BESTAP   *DECK Remez
      SUBROUTINE Remez(Vrbse,K_Lap,EMin,EMax,Coeff,Demand,Inf)
*======================================================================*
*----------------------------------------------------------------------*
*                                                                      *
*     Function : Main program of Remez algorithm                       *
*                                                                      *
*                1      K_Lap                                          *
*             ------- := SUM Omega(N)*EXP(-Alpha(N)*DELTA)             *
*              DELTA     N=1                                           *
*                                                                      *
*              DELTA  := EA+EB-EI-EJ                                   *
*                                                                      *
*     Author(s): Akio Takatsuka (2007)                                 *
*                                                                      *
*     K_Lap : The number of quadrature points (K_Lap=1,...,20)         *
*             K_Lap=0 : K_Lap is automatically determined.             *
*                                                                      *
*----------------------------------------------------------------------*
*======================================================================*
      IMPLICIT REAL*8 (A-H,O-Z)
C
#include "remez.fh"
C
      Logical Vrbse
      LOGICAL Inf
C
      PARAMETER (ItrEnd=50,ZERO=0.0D+00,ONE=1.0D+00,MxList=30)
C
      CHARACTER*8 Demand
      Logical Verbose
C
Ctbp  DIMENSION Alpha(20),Omega(20),Xi(40),Coeff(40),V(40),T(40),DD(82),
Ctbp *          CofOld(40),RList(MxList),IRMax(20),RMin(20),RMax(20)
      DIMENSION Coeff(40),T(40),DD(82),
     *          CofOld(40),RList(MxList),IRMax(20),RMin(20),RMax(20)
C
      DATA RList / 2.0D+00, 5.0D+00, 1.0D+01, 2.0D+01, 3.0D+01, 4.0D+01,
     *             5.0D+01, 6.0D+01, 7.0D+01, 8.0D+01, 9.0D+01, 1.0D+02,
     *             2.0D+02, 3.0D+02, 4.0D+02, 5.0D+02, 6.0D+02, 7.0D+02,
     *             8.0D+02, 9.0D+02, 1.0D+03, 2.0D+03, 3.0D+03, 4.0D+03,
     *             5.0D+03, 6.0D+03, 7.0D+03, 8.0D+03, 9.0D+03, 1.0D+04/
      DATA RMin  / 2.0D+00, 2.0D+00, 2.0D+00, 2.0D+00, 2.0D+00, 2.0D+00,
     *             2.0D+00, 1.0D+01, 1.0D+01, 1.0D+01, 1.0D+01, 1.0D+01,
     *             1.0D+01, 1.0D+01, 1.0D+01, 1.0D+01, 2.0D+01, 4.0D+01,
     *             5.0D+01, 6.0D+01/
      DATA RMax  / 8.667D+00, 4.154D+01, 1.468D+02, 4.361D+02,
     *             1.154D+03, 2.802D+03, 6.373D+03, 1.375D+04,
     *             2.839D+04, 5.650D+04, 1.090D+05, 2.045D+05,
     *             3.738D+05, 6.692D+05, 1.175D+06, 2.080D+06,
     *             3.450D+06, 5.754D+06, 9.494D+06, 1.550D+07/
      DATA IRMax / 3, 7,13,16,22, 23,27,31,31,31,
     *            31,31,31,31,31, 31,31,31,31,31/
C
      LOGICAL Change,Dbg,SkpRem,StopBA
C
      Call Untested('Laplace quadrature generation (subroutine remez)')
C
      Dbg = .FALSE.
      SkpRem = .FALSE.
      Change = .FALSE.
      IMes = -1
C
      Verbose=Vrbse.or.Dbg
C
      Call Remez_SetupPrint(Verbose)
C
      WRITE(IW,'(/" Remez: minimax approximation for Laplace grid " )')
      WRITE(IW,'( " ============================================= "/)')
C
C     ===== Check input variables =====
C
      IF (EMin.GT.EMax) THEN
         WRITE(IW,'(/" Input values of energy are unsuitable. "/)')
         WRITE(IW,*)'EMin, EMax',EMin,EMax
         K_Lap=-1
         Call Remez_ShutDownPrint(Verbose)
         RETURN
      ELSE IF ((K_Lap.LT.0).OR.(K_Lap.GT.20)) THEN
         WRITE(IW,'(/" Input value of K is unsuitable. ")')
         WRITE(IW,*)'K_Lap',K_Lap
         K_Lap=-2
         Call Remez_ShutDownPrint(Verbose)
         RETURN
      ENDIF
C
C     ===== Set initial values =====
C           InitR : initial R
C
      EMinIv = ONE/EMin
      R = EMax*EMinIv
C
      J = MxList
      DO I=1,MxList
         EDiff = R-RList(J)
         IF (EDiff.GT.ZERO) THEN
            InitR = J
            GOTO 100
         ENDIF
         J = J-1
      ENDDO
      InitR = 1
  100 CONTINUE
C
C     ===== Determine K_Lap value (If K_Lap isn't determined.) =====
C
      IF (K_Lap.EQ.0) CALL DfineK(K_Lap,R,InitR,Demand)
      IDim = 2*K_Lap
C
      RMax0=RMax(K_Lap)
      WRITE(IW,'(1X,"# of quadrature points =",I3/)')K_Lap
      WRITE(IW,'(1X,"Maximum R =",F20.10)')RMax0
      WRITE(IW,'(" MO energy")')
      WRITE(IW,'(4X,"Min =",2X,F14.8)')EMin
      WRITE(IW,'(4X,"Max =",2X,F14.8)')EMax
      WRITE(IW,'(4X,"R   =",F16.8)')R
C
C     ===== Check the R value =====
C       R.LE.RMin(K_Lap) or R.GT.RMax(K_Lap)
C
      IF (R.LE.RMin(K_Lap)) THEN
         SkpRem = .TRUE.
         InitR = 1
         IMes = 0
         RLim = RMin(K_Lap)
      ELSE IF ((R.GT.RList(IRMax(K_Lap)-1)).AND.(R.LT.RMax(K_Lap))) THEN
         InitR = IRMax(K_Lap)-1
      ELSE IF (R.GT.RMax(K_Lap)) THEN
         SkpRem = .TRUE.
         InitR = IRMax(K_Lap)
         IMes = 1
      ENDIF
      IF (Inf) THEN
         SkpRem = .TRUE.
         InitR = IRMax(K_Lap)
         IMes = 2
      ENDIF
      IF (SkpRem) THEN
         WRITE(IW,'(/" Remez step skipped.")')
         IF (IMes.EQ.0)
     *      WRITE(IW,'(" Because R is smaller than",F8.3,".")') RLim
         IF (IMes.EQ.1)
     *      WRITE(IW,'(" Because R is larger than",F8.3,".")')
     *      RMax(K_Lap)
         GOTO 999
      ENDIF
C
C     ===== Set initial values (Omega,Alpha,T) =====
C
      CALL SetExp(K_Lap,InitR,R,RIni,Coeff,T)
C
      IF (Dbg) THEN
         WRITE(IW,'(/" Initial data for Remez algorithm"/)')
         DO I=1,K_Lap
            Idx = 2*I-1
            WRITE(IW,
     *      '(" Omega(",I3,") = ",F21.18,2X,"Alpha(",I3,") = ",F20.16)')
     *      I,Coeff(Idx),I,Coeff(Idx+1)
         ENDDO
      ENDIF
C
C     ===== Start Remez step =====
C
      Theta  = ZERO
      Theta2 = ONE
C
      WRITE(IW,'(" Remez step starts from ",F5.0/)') RIni
      CALL FdExtr(K_Lap,T,Coeff,R,Theta,DD,StopBA)
      IF (Dbg) THEN
         WRITE(IW,'(" Error output (initial)"/)')
         DO I=1,2*(2*K_Lap+1)
            WRITE(IW,*)'dd',I,DD(I)
         ENDDO
      ENDIF
      IF (StopBA) THEN
         Change =.TRUE.
         GOTO 999
      ENDIF
C
C     ===== Iteration =====
C
      DO Iter=1,ItrEnd
         CALL DCOPY_(IDim,Coeff,1,CofOld,1)
         CALL SlvNt2(K_Lap,R,Coeff,T,Theta2,VVMax,StopBA)
         IF (StopBA) THEN
            Change =.TRUE.
            GOTO 999
         ENDIF
c        CALL TStat(" Newton2",1)
         CALL SlvNt1(K_Lap,100,Coeff,T)
c        CALL TStat(" Newton1",1)
C
         IF (Dbg) THEN
            WRITE(IW,'(/"Iter =",I3)')Iter
            CALL FdExtr(K_Lap,T,Coeff,R,Theta,DD,StopBA)
            DO J=1,IDim+1
               WRITE(IW,
     *         '(" Error(",I2,") = ",F20.17,2X," X = ",F20.14)')
     *         J,DD(J),DD(2*K_Lap+1+J)
            ENDDO
            WRITE(IW,*)
         ENDIF
C
C     ===== Check convergence =====
C     CofOld is updated by the mean of Coeff
C
         DO J=1,IDim
            CofOld(J) = ABS(CofOld(J)-Coeff(J))
         ENDDO
         CALL FindAM(IDim+1,DD,DDMax,IMax)
         DCofMx = FindMx(IDim,CofOld)
C
         IF (Dbg) THEN
            CALL FdExtr(K_Lap,T,Coeff,R,Theta,DD,StopBA)
            CALL FindAM(IDim+1,DD,DDMax,IMax)
            IF (Iter.EQ.1)
     *         WRITE(IW,
     *         '(" Iter",5X,"Max Change",15X,"Max DD",17X,"Max error")')
            WRITE(IW,'(I3,3(2X,E23.15E3)," (",I2,")")')
     *            Iter,DCofMx,VVMax,DDMax,IMax
         ELSE
            IF (Iter.EQ.1)
     *         WRITE(IW,'(" Iter",5X,"Max Change",15X,"Max DD")')
            WRITE(IW,'(I3,2(2X,E23.15E3))') Iter,DCofMx,VVMax
         ENDIF
C
         IF ((VVMax.LT.5.0D-16).AND.(DCofMx.LT.1.0D-05)) GOTO 777
C
c        CALL TStat(" Check T",1)
      ENDDO
C
      WRITE(IW,'(" The number of iterations exceeded ",I3," .")')
     *        ItrEnd
C
C     ===== Check the result =====
C
  777 CONTINUE
      CALL FdExtr(K_Lap,T,Coeff,R,Theta,DD,StopBA)
      IF (StopBA) THEN
         Change =.TRUE.
         GOTO 999
      ENDIF
      CALL FindAM(IDim+1,DD,DDMax,IMax)
      CALL ChkAcc(K_Lap,InitR,DDMax,R,Change)
C
C     ===== Change parameters =====
C     If the convergence isn't good, the larger R's parameters are used.
C
      WRITE(IW,'(/" Iteration =",I6)') Iter
  999 CONTINUE
      IF (Change.OR.SkpRem) THEN
         WRITE(IW,'(1X,"Change!!")')
         CALL SetExp(K_Lap,InitR,R,RIni,Coeff,T)
      ENDIF
C
C     ===== Determine Xi =====
C
c     CALL TStat(" Check T",1)
      CALL SlvNt1(K_Lap,1,Coeff,T)
      IF (Change) THEN
         R = RIni
      ENDIF
c     CALL TStat(" Newton1",1)
C
      WRITE(IW,'(/" Optimized solution in R = ",F10.4/)')R
      DO I=1,K_Lap
         Idx = 2*I-1
         WRITE(IW,
     *   '(" Omega(",I3,") = ",F21.18,2X,"Alpha(",I3,") = ",F20.17)')
     *   I,Coeff(Idx),I,Coeff(Idx+1)
      ENDDO
      WRITE(IW,*)
      CALL FdExtr(K_Lap,T,Coeff,R,Theta,DD,StopBA)
      DO I=1,IDim+1
         WRITE(IW,'(" Error(",I2,") = ",F20.17,2X," X = ",F20.14)')
     *      I,DD(I),DD(2*K_Lap+1+I)
      ENDDO
      WRITE(IW,*)
      DO I=1,IDim
         WRITE(IW,'(" Xi(",I2,") = ",F20.15)')I,T(I)
      ENDDO
      WRITE(IW,'()')
C
C     ===== Maltiple the scaling parameter (EMinIv = 1/EMin) =====
C
      CALL DSCAL_(IDim,EMinIv,Coeff,1)
      WRITE(IW,'(/" Final solution "/)')
      DO I=1,K_Lap
         Idx = 2*I-1
         WRITE(IW,
     *   '(" Omega(",I3,") = ",F21.18,2X,"Alpha(",I3,") = ",F20.17)')
     *   I,Coeff(Idx),I,Coeff(Idx+1)
      ENDDO
C
      Call Remez_ShutDownPrint(Verbose)
C
      RETURN
      END
C*MODULE BESTAP *DECK EXPSUM
      FUNCTION ExpSum(X,K,Coeff)
*======================================================================*
*----------------------------------------------------------------------*
*                                                                      *
*     FUNCTION : Compute the Function in Exponential sum method.       *
*             K                                                        *
*     1/X =: SUM Omega(I)*EXP(-Alpha(I)*X)                             *
*            I=1                                                       *
*                                                                      *
*     Author(s): Akio Takatsuka (2007)                                 *
*                                                                      *
*----------------------------------------------------------------------*
*======================================================================*
      IMPLICIT REAL*8 (A-H,O-Z)
C
      DIMENSION Coeff(40)
C
      Dum = 0.0D+00
      DO I=1,K
         Idx = 2*I-1
         Omega = Coeff(Idx)
         Alpha = Coeff(Idx+1)
         Dum = Dum+Omega*EXP(-X*Alpha)
      ENDDO
      ExpSum = Dum
C
      RETURN
      END
C*MODULE BESTAP *DECK GLASUM
      FUNCTION GLaSum(K,X,W,Delta)
*======================================================================*
*----------------------------------------------------------------------*
*                                                                      *
*     FUNCTION : Compute the Function in Gauss-Laguerre method         *
*             K                                                        *
*     1/X =: SUM W(I)*Func(X(I))                                       *
*            I=1                                                       *
*                                                                      *
*     Author(s): Akio Takatsuka (2007)                                 *
*                                                                      *
*----------------------------------------------------------------------*
*======================================================================*
      IMPLICIT REAL*8 (A-H,O-Z)
C
      PARAMETER (ZERO=0.0D+00,ONE=1.0D+00)
C
      DIMENSION W(20),X(20)
C
      Dum = ZERO
      DO I=1,K
         Val = EXP((-Delta+ONE)*X(I))
         Dum = Dum+W(I)*Val
      ENDDO
      GLaSum = Dum
C
      RETURN
      END
C*MODULE BESTAP *DECK GLESUM
      FUNCTION GLeSum(K,X,W,Delta)
*======================================================================*
*----------------------------------------------------------------------*
*                                                                      *
*     FUNCTION : Compute the Function in Gauss-Legendre method         *
*             K                                                        *
*     1/X =: SUM W(I)*Func(X(I))                                       *
*            I=1                                                       *
*                                                                      *
*     Author(s): Akio Takatsuka (2007)                                 *
*                                                                      *
*----------------------------------------------------------------------*
*======================================================================*
      IMPLICIT REAL*8 (A-H,O-Z)
C
      PARAMETER (ZERO=0.0D+00,ONE=1.0D+00)
C
      DIMENSION W(20),X(20)
C
      Dum = ZERO
      DO I=1,K
         A    = ONE/(ONE-X(I))
         Varl = X(I)*A
         Derv = Varl*A
         Val = Derv*EXP(-Delta*Varl)
         Dum = Dum+W(I)*Val
      ENDDO
      GLeSum = Dum
C
      RETURN
      END
C*MODULE BESTAP *DECK FUNCLA
      FUNCTION FuncLa(X,Delta)
C----------------------------------------------------------------------
C----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
      FuncLa = EXP((-Delta+1.0D+00)*X)
      RETURN
      END
C*MODULE BESTAP *DECK FUNCLE
      FUNCTION FuncLe(X,Delta)
C----------------------------------------------------------------------
C     Function : Define function after CoV for Gauss-Legendre Method
C                H.Koch,A.Sanchez de Meras (J.Chem.Phys. 113,508)
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
      PARAMETER (ONE=1.0D+00)
C
      Varl = X/(ONE-X)
      Derv = ONE/((ONE-X)*(ONE-X))
      FuncLe = Derv*EXP(-Delta*Varl)
      RETURN
      END
C*MODULE BESTAP *DECK SUMLAG
      FUNCTION SumLag(K_Lap,W,X,Delta)
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
      DIMENSION W(K_Lap),X(K_Lap)
C
      Dum = 0.0D+00
      DO I=1,K_Lap
         Dum = Dum+W(I)*FuncLa(X(I),Delta)
      ENDDO
      SumLag = Dum
C
      RETURN
      END
C*MODULE BESTAP *DECK SUMLEG
      FUNCTION SumLeg(K_Lap,W,X,Delta)
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
      DIMENSION W(K_Lap),X(K_Lap)
C
      Dum = 0.0D+00
      DO I=1,K_Lap
         Dum = Dum+W(I)*FuncLe(X(I),Delta)
      ENDDO
      SumLeg = Dum
C
      RETURN
      END
C*MODULE BESTAP *DECK SLVNT1
      SUBROUTINE SlvNt1(K_Lap,NTimes,Coeff,T)
C-----------------------------------------------------------------------
C     Function : Newton-Raphson method (type 1)
C                obtain omega & alpha
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
#include "remez.fh"
C
      PARAMETER (TOL=1.0D-22,ONE=1.0D+00,TWO=2.0D+00,THTMIN=1.0D-04,
     *           PT5=0.5D+00)
C
      DIMENSION Coeff(40),CoeffOld(40),W(40),VV(40),T(40),A(40,40)
C
      LOGICAL Error,Dbg
C
      Theta = 1.0D+00
      Dbg = .FALSE.
      IDim = 2*K_Lap
C
      DO Iter=1,NTimes
         IRes = 1
         CALL AltErr(K_Lap,Coeff,T,VV,Eps0)
         IF (Eps0.GT.Tol) THEN
            CALL PtDiff(IDim,Coeff,T,A)
C
            IF (Dbg) THEN
               WRITE(IW,'("Check values of derivatives")')
               CALL Laplace_PRSQ(A,IDim,IDim,IDim)
            ENDIF
C
            CALL SlvEqs(IDim,A,W,VV,Error)
            IF (Dbg) THEN
               WRITE(IW,*)'Check solutions of equations',Error
               DO I=1,IDim
                  WRITE(IW,*)I,W(I)
               ENDDO
            ENDIF
            IF (.NOT.Error) GOTO 555
            CALL DCOPY_(IDim,Coeff,1,CoeffOld,1)
C
  222       CONTINUE
            DO I=1,IDim
               Coeff(I) = CoeffOld(I)-Theta*W(I)
            ENDDO
            CALL AltErr(K_Lap,Coeff,T,VV,Eps1)
            IF (Eps1.LT.Eps0) THEN
               Theta = TWO*Theta
               IF (Theta.GT.ONE) Theta = ONE
               GOTO 888
            ENDIF
            IF (Theta.LT.THTMIN) THEN
               GOTO 888
            ELSE
               Theta = Theta*PT5
               GOTO 222
            ENDIF
  555       CONTINUE
         ENDIF
         IRes = 0
  888    CONTINUE
         IF(IRes.EQ.0) GOTO 999
      ENDDO
  999 CONTINUE
      RETURN
      END
C*MODULE BESTAP *DECK SLVNT2
      SUBROUTINE SlvNt2(K_Lap,R,Coeff,T,Theta2,VVMax,StopBA)
C-----------------------------------------------------------------------
C     Function : Newton-Raphson method (type 2)
C
C         R      : The size of approximation range
C         T      : clossing points
C         TOld   : saved data of the first T
C         Coeff  : Omega & Alpha
C         CofOld : saved data of the first Coeff
C         DD     : maximum error & its coordinate
C         VVMax  : Maximum mean of VV array
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
#include "remez.fh"
C
      PARAMETER (TOL=1.0D-22,ZERO=0.0D+00,ONE=1.0D+00,TWO=2.0D+00,
     *           PT5=0.5D+00,TLim=2.0D-05)
C
      LOGICAL Error,NG,Dbg,StopBA
C
      DIMENSION Coeff(40),CofOld(40),T(40),TOld(40),VV(40),W(40),DD(82),
     *          A(40,40)
C
      IDim = 2*K_Lap
      New2 = 10
      Theta2Mx = ONE
      NG = .FALSE.
      Dbg = .FALSE.
      IF (Dbg) WRITE(IW,*)'theta2',Theta2
C
      CALL FdExtr(K_Lap,T,Coeff,R,Theta,DD,StopBA)
      IF (StopBA) RETURN
      CALL DCOPY_(IDim,Coeff,1,CofOld,1)
C
      DO I=1,IDim
         VV(I) = DD(I)+DD(I+1)
      ENDDO
      Eps0 = FindMx(IDim,VV)
C-tbp: avoid the risk of using eps1 later:
      Eps1=Eps0
C
      IF (Eps0.GT.TOL) THEN
         Eps = 1.0D-03
         DO J=1,IDim
            Eprel = T(J)*Eps
            EprInv = ONE/Eprel
            TCpy = T(J)
            T(J) = T(J)*(ONE+Eps)
            CALL SlvNt1(K_Lap,New2,Coeff,T)
            CALL FdExtr(K_Lap,T,Coeff,R,Theta,DD,StopBA)
            IF (StopBA) RETURN
            DO I=1,IDim
               Temp = DD(I)+DD(I+1)-VV(I)
               A(I,J) = Temp*EprInv
            ENDDO
            CALL DCOPY_(IDim,CofOld,1,Coeff,1)
            T(J) = TCpy
         ENDDO
         CALL SlvEqs(IDim,A,W,VV,Error)
         IF (.NOT.Error) GOTO 555
         CALL DCOPY_(IDim,T,1,TOld,1)
C
  222    CONTINUE
         DO I=1,IDim
            T(I) = TOld(I)-Theta2*W(I)
         ENDDO
         CALL CkAltT(K_Lap,R,T,NG)
         IF (NG) THEN
            WRITE(IW,'("!! wrong T-values !!")')
            CALL AbortG
            New2 = 100
            CALL DCOPY_(IDim,TOld,1,T,1)
            GOTO 333
         ENDIF
         CALL DCOPY_(IDim,CofOld,1,Coeff,1)
         CALL SlvNt1(K_Lap,New2,Coeff,T)
         CALL FdExtr(K_Lap,T,Coeff,R,Theta,DD,StopBA)
         IF (StopBA) RETURN
         DO I=1,IDim
            VV(I) = DD(I)+DD(I+1)
         ENDDO
         Eps1 = FindMx(IDim,VV)
         IF (Dbg) WRITE(IW,*)'eps1',Eps1
         IF (Eps1.LT.Eps0) THEN
            Theta2 = TWO*Theta2
            IF (Theta2.GT.Theta2Mx) Theta2 = Theta2Mx
         ENDIF
         GOTO 999
C
  333    CONTINUE
         IF (Theta2.LT.TLim) THEN
            WRITE(IW,'(" Theta2 becomes too small.")')
         ELSE
            Theta2 = Theta2*PT5
            GOTO 222
         ENDIF
C
  555    CONTINUE
      ENDIF
C
  999 CONTINUE
      VVMax = Eps1
      CALL SlvNt1(K_Lap,New2,Coeff,T)
C
      IF (Dbg) THEN
         WRITE(IW,*)'eps1',VVMax
         DO I=1,IDim
            WRITE(IW,*)'coeff',i,coeff(i)
         ENDDO
         DO I=1,IDim
            WRITE(IW,*)'T',i,t(i)
         ENDDO
      ENDIF
C
      RETURN
      END
C*MODULE BESTAP *DECK FDEXTR
      SUBROUTINE FdExtr(K_Lap,T,Coeff,R,Theta,DD,StpBA)
C-----------------------------------------------------------------------
C     Function : Find extrema in each interval
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
#include "remez.fh"
C
      PARAMETER (ZERO=0.0D+00,ONE=1.0D+00,PT5=5.0D-01,MxIter=10000,
     *           Thrs=1.0D-09,TWO=2.0D+00)
C
      DIMENSION Coeff(40),T(40),DD(82)
C
      LOGICAL Dbg,StpBA
C
      IDim = 2*K_Lap
      IDimEnd = IDim+1
      Dbg =.FALSE.
      StpBA =.FALSE.
C
      DO I=1,IDimEnd
C
C     ===== End points =====
C
         IF (I.EQ.1) THEN
            XXMax = ONE
            GOTO 888
         ELSE IF (I.EQ.IDimEnd) THEN
            XXMax = R
            GOTO 888
         ENDIF
C
C     ===== Initial values (midpoint)
C
         X1 = T(I-1)
         X2 = T(I)
         X = (X1+X2)*PT5
C
C     ===== Solve Equations =====
C
         Theta = ONE
         DO Iter=1,MxIter
            FX  = GetDr1(K_Lap,X,Coeff)
            DFX = GetDr2(K_Lap,X,Coeff)
  100       CONTINUE
            XNew = X-Theta*FX/DFX
            DifX = ABS(XNew-X)
            IF (Dbg) WRITE(IW,*)Iter,XNew,Difx
            IF (DifX.LT.Thrs) GOTO 777
            FNew = GetDr1(K_Lap,XNew,Coeff)
            FFF  = (ONE-Theta*PT5)*FX
            IF (ABS(FNew).GE.ABS(FFF)) THEN
               IF (Dbg) WRITE(IW,*)FNew,FFF
               Theta = Theta*PT5
               GOTO 100
            ENDIF
            X = XNew
         ENDDO
C
         WRITE(IW,'("*************** Max Iteration in FdExtr")')
         WRITE(IW,'("I =",I3," Max DifX. =",E23.15E3)')I,DifX
!!       StpBA =.TRUE.
!!       RETURN
C
         FMax  = ZERO
         XXMax = ZERO
C
         IX = -1
         XErr2 = 9.99D+02
         XErr3 = XErr2
         IDr = 1000
         IDrEnd = IDr+1
         DrInv = ONE/DBLE(IDr)
         DrDif = (X2-X1)*DrInv
C
         DO J=1,IDrEnd
            X = X1+(J-1)*DrDif
            FF = QuadErr(K_Lap,X,Coeff)
            XErr1 = XErr2
            XErr2 = XErr3
            XErr3 = FF
            IF (ABS(FF).GT.ABS(FMax)) THEN
               FMax = FF
               XXMax = X
               IX = J
            ENDIF
         ENDDO
         XXMax = -XXMax
C
         IF (IX.EQ.1) GOTO 888
         IF (IX.EQ.IDrEnd) GOTO 888
C
         Dum = X1+(IX-2)*DrDif
         XM1 = QuadErr(K_Lap,Dum,Coeff)
C
         Dum = X1+(IX-1)*DrDif
         XM2 = QuadErr(K_Lap,Dum,Coeff)
C
         Dum = X1+IX*DrDif
         XM3 = QuadErr(K_Lap,Dum,Coeff)
C
         XErr1  = (XM3-XM1)/(TWO*(XM3-TWO*XM2+XM1))
         XErr2  = XErr1*DrInv-XXMax
         XMx = MAX(ABS(XM1),ABS(XM3))
         IF (ABS(XM2).LE.XMx) GOTO 888
         XXMax = XErr2
C
  777    CONTINUE
         XXMax = XNew
C
C     ===== Save data =====
C
  888    CONTINUE
         FMax = QuadErr(K_Lap,XXMax,Coeff)
         DD(I) = FMax
         DD(I+IDimEnd) = XXMax
      ENDDO
      RETURN
      END
C*MODULE BESTAP *DECK GETDR1
      FUNCTION GetDr1(K_Lap,X,Coeff)
C-----------------------------------------------------------------------
C     Function : Get 1st derivative of error function
C
C                     K_Lap
C     Dr1(X) = 1/X*X - SUM Omega(I)*Alpha(I)*Exp(-Alpha(I)*X)
C                      I=1
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
      PARAMETER (ZERO=0.0D+00,ONE=1.0D+00)
C
      DIMENSION Coeff(40)
C
      Dum = ZERO
      DO I=1,K_Lap
         Idx = 2*I-1
         Omega = Coeff(Idx)
         Alpha = Coeff(Idx+1)
         Dum = Dum+Omega*Alpha*EXP(-Alpha*X)
      ENDDO
C
      GetDr1 = Dum-ONE/(X*X)
C
      RETURN
      END
C*MODULE BESTAP *DECK GETDR2
      FUNCTION GetDr2(K_Lap,X,Coeff)
C-----------------------------------------------------------------------
C     Function : Get 2nd derivative of error function
C
C     K_Lap
C      SUM Omega(I)*Alpha(I)*Alpha(I)*Exp(-Alpha(I)*X)
C      I=1
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
      PARAMETER (ZERO=0.0D+00,ONE=1.0D+00,TWO=2.0D+00)
C
      DIMENSION Coeff(40)
C
      Dum = ZERO
      DO I=1,K_Lap
         Idx = 2*I-1
         Omega = Coeff(Idx)
         Alpha = Coeff(Idx+1)
         Dum = Dum+Omega*Alpha*Alpha*EXP(-Alpha*X)
      ENDDO
C
      GetDr2 = TWO/(X*X*X)-Dum
C
      RETURN
      END
C*MODULE BESTAP *DECK SLVEQS
      SUBROUTINE SlvEqs(N,A,X,B,Error)
C-----------------------------------------------------------------------
C     Function : Solve equations (Loesen)
C                Solve AX = B
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
      PARAMETER (ZERO=0.0D+00,ONE=1.0D+00,G_Eps=1.0D-19)
C
      LOGICAL Error
C
      DIMENSION A(40,40),B(40),X(40)
C
      Error = .FALSE.
      DO I=1,N
         DO K=I,N
            S = ZERO
            DO J=I,N
               Temp = A(K,J)*A(K,J)
               S = S+Temp
            ENDDO
            IF (S.EQ.ZERO) GOTO 999
            S = SQRT(S)
            SInv = ONE/S
            B(K) = B(K)*SInv
            DO J=1,N
               A(K,J) = A(K,J)*SInv
            ENDDO
         ENDDO
C
         P = A(I,I)
         K = I
         DO J=I,N
            S = A(J,I)
            IF (ABS(S).GT.ABS(P)) THEN
               P = S
               K = J
            ENDIF
         ENDDO
         IF (K.GT.I) THEN
            DO J=I,N
               S = A(I,J)
               A(I,J) = A(K,J)
               A(K,J) = S
            ENDDO
            S = B(I)
            B(I) = B(K)
            B(K) = S
         ENDIF
         IF ((ABS(P).LT.G_Eps).AND.(P.EQ.ZERO)) GOTO 999
C
         PInv = ONE/P
         B(I) = B(I)*PInv
         DO J=I+1,N
            A(I,J) = A(I,J)*PInv
         ENDDO
         DO K=I+1,N
            S = A(K,I)
            IF (S.NE.ZERO) THEN
               Temp = S*B(I)
               B(K) = B(K)-Temp
               DO J=I+1,N
                  Temp = S*A(I,J)
                  A(K,J) = A(K,J)-Temp
               ENDDO
            ENDIF
         ENDDO
      ENDDO
C
      II = N
      DO I=1,N
         S = B(II)
         DO J=II+1,N
            Temp = A(II,J)*X(J)
            S = S-Temp
         ENDDO
         X(II) = S
         II = II-1
      ENDDO
C
      Error = .TRUE.
  999 CONTINUE
      RETURN
      END
C*MODULE BESTAP *DECK PTDIFF
      SUBROUTINE PtDiff(IDim,Coeff,T,A)
C-----------------------------------------------------------------------
C     Function : Calculate derivative
C                j = odd  ; A(I,J) = EXP(-X*Coeff(J+1))
C                j = even ; A(I,J) = -X*Coeff(J-1)*EXP(-X*Coeff(J))
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
      DIMENSION Coeff(40),T(40),A(40,40)
C
      DO I=1,IDim
         DO J=1,IDim
            X = T(IDim+1-I)
            IF (MOD(J,2).EQ.1) THEN
               A(I,J) = EXP(-X*Coeff(J+1))
            ELSE
               A(I,J) = -X*Coeff(J-1)*EXP(-X*Coeff(J))
            ENDIF
         ENDDO
      ENDDO
      RETURN
      END
C*MODULE BESTAP *DECK QUADERR
      FUNCTION QuadErr(K_Lap,X,Coeff)
C-----------------------------------------------------------------------
C     Function : Calculate the error between Exponensial sum and 1/x
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
      PARAMETER (ONE=1.0D+00)
C
      DIMENSION Coeff(40)
C
      S = ExpSum(X,K_Lap,Coeff)
C
      QuadErr = S-ONE/X
      RETURN
      END
C*MODULE BESTAP *DECK ALTERR
      SUBROUTINE AltErr(K_Lap,Coeff,T,VV,Eps)
C-----------------------------------------------------------------------
C     Function : Calculate errors in each points of alternant
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
      DIMENSION VV(40),T(40),Coeff(40)
C
      IDim = 2*K_Lap
      Eps = 0.0D+00
      J = IDim
      DO I=1,IDim
         X = T(J)
         VV(I) = QuadErr(K_Lap,X,Coeff)
         Eps = MAX(Eps,ABS(VV(I)))
         J = J-1
      ENDDO
      RETURN
      END
C*MODULE BESTAP *DECK CKALTD
      SUBROUTINE CkAltD(K_Lap,DD,NG)
C-----------------------------------------------------------------------
C     Function : Check the alternation of array DD
C-----------------------------------------------------------------------
      IMPLICIT REAL*8(A-H,O-Z)
C
#include "remez.fh"
C
      PARAMETER (ZERO=0.0D+00)
C
      DIMENSION DD(82)
C
      LOGICAL NG
C
      IDimEnd = 2*K_Lap
      NG = .FALSE.
C
      DO I=1,IDimEnd
         A = DD(I)*DD(I+1)
         IF (A.GE.ZERO) THEN
            NG = .TRUE.
            WRITE(IW,'("DD sign is wrong at I =",I3)') I
         ENDIF
      ENDDO
      RETURN
      END
C*MODULE BESTAP *DECK CkAltT
      SUBROUTINE CkAltT(K_Lap,R,T,NG)
C-----------------------------------------------------------------------
C     Function : Check the alternation of array T
C-----------------------------------------------------------------------
      IMPLICIT REAL*8(A-H,O-Z)
C
#include "remez.fh"
C
      PARAMETER (ONE=1.0D+00)
C
      DIMENSION T(40)
C
      LOGICAL NG
C
      IDimEnd = 2*K_Lap+1
      NG = .FALSE.
C
      DO I=1,IDimEnd
         IF (I.EQ.1) THEN
            A = ONE
         ELSE
            A = T(I-1)
         ENDIF
         IF (I.EQ.IDimEnd) THEN
            B = R
         ELSE
            B = T(I)
         ENDIF
         IF (A.GE.B) THEN
            WRITE(IW,'("The sign of T is wrong at I =",I3)')I
            NG = .TRUE.
            GOTO 999
         ENDIF
      ENDDO
  999 CONTINUE
      RETURN
      END
C*MODULE BESTAP *DECK INITT
      SUBROUTINE InitT(K_Lap,T,RIni,RNew)
C-----------------------------------------------------------------------
C     Function : Set the initial T()
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
#include "remez.fh"
C
      PARAMETER (ONE=1.0D+00)
C
      DIMENSION T(40)
C
      LOGICAL Dbg
C
      Dbg = .FALSE.
      Val = (RNew-ONE)/(RIni-ONE)
C
      IF (Dbg) WRITE(IW,'("T(I)")')
      DO I=1,2*K_Lap
         T(I) = ONE+(T(I)-ONE)*Val
         IF (Dbg) WRITE(IW,'(I3,2X,F20.14)')I,T(I)
      ENDDO
C
      RETURN
      END
C*MODULE BESTAP *DECK FINDMX
      FUNCTION FindMx(n,X)
C-----------------------------------------------------------------------
C     Function : Find Maximumnorm in X(IDim)
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
      DIMENSION X(n)
C
      XMax = 0.0D+00
      DO I=1,n
         XMax = MAX(XMax,ABS(X(I)))
      ENDDO
      FindMx = XMax
      RETURN
      END
C*MODULE BESTAP *DECK FINDAM
      SUBROUTINE FindAM(n,X,XMax,IMax)
C-----------------------------------------------------------------------
C     Function : Find Absolute Maximum in X(n)    XMax = X(IMax)
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
      DIMENSION X(n)
C
      IMax = 1
      XMax = 0.0D+00
C
      DO I=1,n
         XVal = ABS(X(I))
         IF (XMax.LT.XVal) THEN
            XMax = XVal
            IMax = I
         ENDIF
      ENDDO
      RETURN
      END
C*MODULE BESTAP *DECK DFINEK
      SUBROUTINE DfineK(K_Lap,R,InitR,Demand)
C-----------------------------------------------------------------------
C     Function : Define optimal K value
C
C     Demand : Select the accuracy which you want
C            = MILLI ... 10E-03 accuracy (K = 2-)
C            = MICRO ... 10E-06 accuracy (K = 4-)
C            = NANO  ... 10E-09 accuracy (K = 7-)
C            = PICO  ... 10E-12 accuracy (K =11-)
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
#include "remez.fh"
C
      PARAMETER (MxList=31,MxK=20,ZERO=0.0D+00)
C
      CHARACTER*8 Milli,Micro,Nano,Pico,Demand
C
      DIMENSION ErLst1(MxK),ErLst2(MxK),ErLst3(MxK),ErLst4(MxK),
     *          ErLst5(MxK),IdxMin(MxK),KList1(MxList),KList2(MxList),
     *          KList3(MxList),KList4(MxList),RList(30)
C
      DATA Milli,Micro /'MILLI   ','MICRO   '/
      DATA Nano,Pico   /'NANO    ','PICO    '/
C
      DATA IdxMin / 1, 1, 1, 1, 1,  1, 1, 3, 3, 3,
     *              3, 3, 3, 3, 3,  3, 4, 6, 7, 8/
C
      DATA KList1 / 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
     *              3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3/
      DATA KList2 / 3, 4, 5, 6, 6, 7, 7, 7, 7, 7, 7, 7, 8, 8, 9, 9,
     *              9, 9, 9, 9, 9,10,10,10,10,10,10,10,10,10,10/
      DATA KList3 / 5, 7, 8, 9,10,10,11,11,11,11,12,12,13,14,14,14,
     *             15,15,15,15,15,16,17,17,17,18,18,18,18,18,18/
      DATA KList4 / 6,10,10,12,13,14,14,15,15,16,16,16,18,19,19,20,
     *             20,20,20,20,20,20,20,20,20,20,20,20,20,20,20/
C
      DATA ErLst1 / 0.000D+00, 8.752D-03, 5.052D-03, 1.700D-03,
     *              6.428D-04, 2.646D-04, 1.163D-04, 5.392D-05,
     *              2.611D-05, 1.312D-05, 6.807D-06, 3.630D-06,
     *              1.984D-06, 1.108D-06, 6.311D-07, 3.659D-07,
     *              2.155D-07, 1.289D-07, 7.811D-08, 4.794D-08/
      DATA ErLst2 / 0.000D+00, 0.000D+00, 0.000D+00, 6.258D-06,
     *              4.243D-06, 7.379D-06, 9.841D-06, 8.724D-06,
     *              7.468D-06, 9.296D-06, 3.844D-06, 1.582D-06,
     *              6.481D-07, 2.648D-07, 1.079D-07, 4.388D-08,
     *              1.780D-08, 7.213D-09, 2.918D-09, 1.179D-09/
      DATA ErLst3 / 0.000D+00, 0.000D+00, 0.000D+00, 0.000D+00,
     *              0.000D+00, 7.741D-09, 1.716D-09, 4.045D-09,
     *              6.822D-09, 9.323D-09, 3.357D-09, 4.301D-09,
     *              7.587D-09, 7.555D-09, 7.134D-09, 8.484D-09,
     *              7.213D-09, 8.602D-09, 8.602D-09, 9.700D-09/
      DATA ErLst4 / 0.000D+00, 0.000D+00, 0.000D+00, 0.000D+00,
     *              0.000D+00, 0.000D+00, 0.000D+00, 0.000D+00,
     *              0.000D+00, 9.021D-12, 6.492D-13, 5.389D-12,
     *              5.735D-12, 9.405D-12, 6.723D-12, 5.298D-12,
     *              1.050D-12, 3.128D-12, 7.099D-12, 9.371D-12/
      DATA ErLst5 / 8.556D-02, 1.785D-02, 5.052D-03, 1.700D-03,
     *              6.428D-04, 2.646D-04, 1.163D-04, 5.392D-05,
     *              2.611D-05, 1.312D-05, 6.807D-06, 3.630D-06,
     *              1.984D-06, 1.108D-06, 6.311D-07, 3.659D-07,
     *              2.155D-07, 1.289D-07, 7.811D-08, 4.794D-08/
      DATA RList  / 2.0D+00, 5.0D+00, 1.0D+01, 2.0D+01, 3.0D+01,
     *              4.0D+01, 5.0D+01, 6.0D+01, 7.0D+01, 8.0D+01,
     *              9.0D+01, 1.0D+02, 2.0D+02, 3.0D+02, 4.0D+02,
     *              5.0D+02, 6.0D+02, 7.0D+02, 8.0D+02, 9.0D+02,
     *              1.0D+03, 2.0D+03, 3.0D+03, 4.0D+03, 5.0D+03,
     *              6.0D+03, 7.0D+03, 8.0D+03, 9.0D+03, 1.0D+04/
C
      LOGICAL Trial
C
C     ===== Check a larger R value =====
C
      Trial = .FALSE.
C
      WRITE(IW,'("Demanded accuracy is ",A8,".")')Demand
C
      IF (InitR.EQ.31) THEN
         IF (Demand.EQ.Milli) THEN
            K_Lap = 3
            ErrVal = ErLst5(K_Lap)
            WRITE(IW,
     *            '(/" This K gurantees the error less than ",E11.4E2)')
     *            ErrVal
         ELSE IF (Demand.EQ.Micro) THEN
            K_Lap = 11
            ErrVal = ErLst5(K_Lap)
            WRITE(IW,
     *      '(/" This K gurantees the error less than ",E11.4E2)')
     *            ErrVal
         ELSE
            K_Lap = 20
            ErrVal = ErLst5(K_Lap)
            IF (Demand.EQ.Nano) THEN
               IF (R.LE.3.0D+04) THEN
                  K_Lap = 19
                  ErrVal = ErLst3(14)
                  WRITE(IW,
     *            '(/" This K gurantees the error less than ",E11.4E2)')
     *            ErrVal
                  GOTO 100
               ELSE IF (R.LE.1.0D+05) THEN
                  ErrVal = ErLst3(15)
                  WRITE(IW,
     *            '(/" This K gurantees the error less than ",E11.4E2)')
     *            ErrVal
                  GOTO 100
               ENDIF
            ENDIF
            WRITE(IW,'(/"!!! Caution !!!")')
            WRITE(IW,
     *'("In this R value, we can only guarantee",E11.4E2," accuracy.")')
     *      ErrVal
         ENDIF
  100    CONTINUE
         RETURN
      ENDIF
C
      IF (InitR.LE.8) Trial = .TRUE.
C
      IVal = InitR+1
C
      IF (Demand.EQ.Milli) THEN
         K_Lap = KList1(IVal)
         IF (Trial) THEN
            IdxR  = IdxMin(K_Lap)
            R_Val = RList(IdxR)
            IF ((R-R_Val).LT.ZERO) THEN
               IVal = IdxR
            ENDIF
         ENDIF
         ErrVal = ErLst1(K_Lap)
         WRITE(IW,
     *       '(/" This K gurantees the error less than ",E11.4E2," .")')
     *   ErLst1(K_Lap)
      ELSE IF (Demand.EQ.Micro) THEN
         K_Lap = KList2(IVal)
         IF (Trial) THEN
            IdxR  = IdxMin(K_Lap)
            R_Val = RList(IdxR)
            IF ((R-R_Val).LT.ZERO) THEN
               IVal = IdxR
            ENDIF
         ENDIF
         ErrVal = ErLst2(K_Lap)
         WRITE(IW,
     *   '(/" This K gurantees the error less than ",E11.4E2," .")')
     *   ErLst2(K_Lap)
      ELSE IF (Demand.EQ.Nano) THEN
         K_Lap = KList3(IVal)
         IF (Trial) THEN
            IdxR  = IdxMin(K_Lap)
            R_Val = RList(IdxR)
            IF ((R-R_Val).LT.ZERO) THEN
               IVal = IdxR
            ENDIF
         ENDIF
         ErrVal = ErLst3(K_Lap)
         WRITE(IW,
     *   '(/" This K gurantees the error less than ",E11.4E2," .")')
     *   ErLst3(K_Lap)
      ELSE IF (Demand.EQ.Pico) THEN
         K_Lap = KList4(IVal)
         IF (Trial) THEN
            IdxR  = IdxMin(K_Lap)
            R_Val = RList(IdxR)
            IF ((R-R_Val).LT.ZERO) THEN
               IVal = IdxR
            ENDIF
         ENDIF
         ErrVal = ErLst4(K_Lap)
         WRITE(IW,
     *   '(/" This K gurantees the error less than ",E11.4E2," .")')
     *    ErLst4(K_Lap)
      ENDIF
C
      RETURN
      END
*MODULE BESTAP *DECK DFMP2E
      SUBROUTINE DfMP2E(NVar,NOcc,NCore,E,EMin,EMax)
C-----------------------------------------------------------------------
C     Function : Define MP2 energy
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
      PARAMETER (TWO=2.0D+00)
C
      DIMENSION E(NVar),IDX(NVar)
C
      DO I=1,NVar
         IDX(I) = I
      ENDDO
c     CALL OrdV(E,IDX,NVar)
C
      EMin = TWO*(E(IDX(NCore+NOcc+1))-E(IDX(NCore+NOcc)))
      EMax = TWO*(E(IDX(NVar))-E(IDX(NCore+1)))
C
      RETURN
      END

      SUBROUTINE AbortG
      IMPLICIT NONE
C-tbp STOP
      Call WarningMessage(2,'Remez aborting!')
      Call Abend()
      END

C*MODULE GELLIB  *DECK Laplace_PRSQ
      SUBROUTINE Laplace_PRSQ(V,M,N,NDIM)
C
C     USE IO_Files
C
      IMPLICIT REAL*8(A-H,O-Z)
C
#include "remez.fh"
C
      DIMENSION V(NDIM,M)
C
C     ----- PRINT OUT A SQUARE MATRIX -----
C     -V- IS -N- ROWS BY -M- COLUMNS, WITH LEADING DIMENSION -NDIM-
C
      MAX = 10
      IMAX = 0
  100 IMIN = IMAX+1
      IMAX = IMAX+MAX
      IF (IMAX .GT. M) IMAX = M
      WRITE (IW,9008)
      WRITE (IW,9028)(I,I = IMIN,IMAX)
      DO J = 1,N
         WRITE (IW,9048)J,(V(J,I),I = IMIN,IMAX)
      ENDDO
      IF (IMAX .LT. M) GO TO 100
      RETURN
 9008 FORMAT(1X)
 9028 FORMAT(10X,10(4X,I4,4X))
 9048 FORMAT(I5,1X,10F12.7)
      END

C*MODULE GELLIB  *DECK LVCLR
      SUBROUTINE LVCLR(A,INCA,N)
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      DIMENSION A(*)
C
      PARAMETER (ZERO=0.0D+00)
C
C     ----- ZERO OUT VECTOR -A-, USING INCREMENT -INCA- -----
C
      IF (INCA .NE. 1) GO TO 100
      DO L=1,N
         A(L) = ZERO
      ENDDO
      RETURN
C
  100 CONTINUE
      LA=1-INCA
      DO L=1,N
         LA=LA+INCA
         A(LA) = ZERO
      ENDDO
      RETURN
      END
C*MODULE *DECK SETEXP
      SUBROUTINE SetExp(K,InitR,R,RIni,Coeff,T)
*======================================================================*
*----------------------------------------------------------------------*
*                                                                      *
C     Function : Determine the initial values for Exponential sum      *
C
C     All data were prepared by Prof. W. Hackbusch and shown on the
C     site below.
C
C         www.mis.mpg.de/scicomp/EXP_SUM/1_x/
C
C     Author(s): Akio Takatsuka (2007)
C-----------------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)
C
#include "remez.fh"
C
      PARAMETER (ZERO=0.0D+00,ONE=1.0D+00,THREE=3.0D+00,KMAX=20,
     *           KMAX2=2*KMAX)
C
      DIMENSION Alpha(KMAX),Omega(KMAX),Xi(KMAX2),Coeff(KMAX2),T(KMAX2)
C
      LOGICAL Dbg
C
      Dbg = .FALSE.
C
      IDim = 2*K
      CALL LVCLR(Alpha,1,KMAX)
      CALL LVCLR(Omega,1,KMAX)
      CALL LVCLR(Xi,1,KMAX2)
      CALL LVCLR(Coeff,1,KMAX2)
      CALL LVCLR(T,1,KMAX2)
C
C     ===== Check K value =====
C
      GOTO ( 100, 200, 300, 400, 500, 600, 700, 800, 900,1000,
     *      1100,1200,1300,1400,1500,1600,1700,1800,1900,2000),K
C
C     ===== K = 01 =====
C
  100 CONTINUE
      GOTO ( 102, 105, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199,
     *       199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199,
     *       199, 199, 199, 199, 199, 199, 199),InitR
C
C     ----- K = 01 , R = 2E0 -----
C
  102 CONTINUE
C
      RIni = 2.0D+00
C
      Omega( 1) = 2.0009458905089673305330300046023239701753D+00
C
      Alpha( 1) = 0.7151291879699048470680859679315943822075D+00
C
      Xi( 1) = 1.088487745833368534800061899759D+00
      Xi( 2) = 1.762080703894060781608023324996D+00
      GOTO 9999
C
C     ----- K = 01 , R = 5E0 -----
C
  105 CONTINUE
C
      RIni = 5.0D+00
C
      Omega( 1) = 1.5216296303352122180679514840129229469312D+00
C
      Alpha( 1) = 0.4931630669040556423090519017060628925719D+00
C
      Xi( 1) = 1.170595465206690831828743870613D+00
      Xi( 2) = 3.226441104658823615493681846722D+00
      GOTO 9999
C
C     ----- K = 01 , R = 1E1 (Best : R = 8.667) -----
C
  199 CONTINUE
      RIni = 1.0D+01
C
      Omega( 1) = 1.4290997869805788563629667109466936381068D+00
C
      Alpha( 1) = 0.4464926062994777657404622123182846848977D+00
C
      Xi( 1) = 1.190833867508896858262469609091D+00
      Xi( 2) = 3.774644895827677148602424805901D+00
      GOTO 9999
C
C     ===== K = 02 =====
C
  200 CONTINUE
C
      GOTO ( 202, 205, 211, 212, 213, 214, 299, 299, 299, 299, 299, 299,
     *       299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 299,
     *       299, 299, 299, 299, 299, 299, 299),InitR
C
C     ----- K = 02 , R = 2E0 -----
C
  202 CONTINUE
C
      RIni = 2.0D+00
C
      Omega( 1) = 1.0724401884999553583380227372856552392477D+00
      Omega( 2) = 3.2918896543848644533659941746606136803166D+00
C
      Alpha( 1) = 0.4072692399097063106511596580761036534568D+00
      Alpha( 2) = 2.4428035233716464261818218650645917477959D+00
C
      Xi( 1) = 1.024669084390886258668224018109D+00
      Xi( 2) = 1.221540551568295593040137936303D+00
      Xi( 3) = 1.586346135830852440849920992338D+00
      Xi( 4) = 1.940702043262358547767373884163D+00
      GOTO 9999
C
C     ----- K = 02 , R = 5E0 -----
C
  205 CONTINUE
C
      RIni = 5.0D+00
C
      Omega( 1) = 0.6544138215791260005066258542694157540609D+00
      Omega( 2) = 2.5128880395953934736782975356561564694857D+00
C
      Alpha( 1) = 0.2412189720039558034640671951853718724124D+00
      Alpha( 2) = 1.6503932424456509387919456166571308131097D+00
C
      Xi( 1) = 1.053800078348716762481791786588D+00
      Xi( 2) = 1.530809276455422984720630696920D+00
      Xi( 3) = 2.701138971986026049171975005336D+00
      Xi( 4) = 4.545078945463444416996701136213D+00
      GOTO 9999
C
C     ----- K = 02 , R = 1E1 -----
C
  211 CONTINUE
C
      RIni = 1.0D+01
C
      Omega( 1) = 0.4596450781301019298055472034336688125222D+00
      Omega( 2) = 2.2190206174604986150578761483487255645741D+00
C
      Alpha( 1) = 0.1616927398937998487085578730448887085913D+00
      Alpha( 2) = 1.3075141330652339404817807722025690964074D+00
C
      Xi( 1) = 1.072446291884906891152262620182D+00
      Xi( 2) = 1.752717114846459290429699084957D+00
      Xi( 3) = 3.698767419816329259920836824982D+00
      Xi( 4) = 8.233481847579251056146176779293D+00
      GOTO 9999
C
C     ----- K = 02 , R = 2E1 -----
C
  212 CONTINUE
C
      RIni = 2.0D+01
C
      Omega( 1) = 0.3456883945557714279787908662200379694696D+00
      Omega( 2) = 2.0588931989746813778271311567991119773069D+00
C
      Alpha( 1) = 0.1137705533481849198753493856983709520136D+00
      Alpha( 2) = 1.1122420832263719987081929119199230626691D+00
C
      Xi( 1) = 1.086388801185830942520188502343D+00
      Xi( 2) = 1.931390216502200829159531514367D+00
      Xi( 3) = 4.620528669639594092268669589529D+00
      Xi( 4) = 13.614404138913966666943333727602D+00
      GOTO 9999
C
C     ----- K = 02 , R = 3E1 -----
C
  213 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.3089269872856084493337180951755271962611D+00
      Omega( 2) = 2.0075705464402703406302796995674952995614D+00
C
      Alpha( 1) = 0.0979831430245671137333374374478456303450D+00
      Alpha( 2) = 1.0494466863388992488356071808297542702348D+00
C
      Xi( 1) = 1.091649420750634746956134191187D+00
      Xi( 2) = 2.001884214128133870425410578697D+00
      Xi( 3) = 5.015778281485991329114587644611D+00
      Xi( 4) = 16.815914202468247307109550092719D+00
      GOTO 9999
C
C     ----- K = 02 , R = 4E1 -----
C
  214 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.2978387832240039867037588450227048042507D+00
      Omega( 2) = 1.9920021768207375049194313154643509733432D+00
C
      Alpha( 1) = 0.0931855955444948900315608152078450387990D+00
      Alpha( 2) = 1.0304678137783844420281159637298173947784D+00
C
      Xi( 1) = 1.093332292125060299864749979726D+00
      Xi( 2) = 2.024812554457079488527682098464D+00
      Xi( 3) = 5.148542084107910186442647049176D+00
      Xi( 4) = 18.048492691174313850774324485826D+00
      GOTO 9999
C
C     ----- K = 02 , R = 5E1 (Best : R = 41.54) -----
C
  299 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.2976555432327632915757986697968817679794D+00
      Omega( 2) = 1.9917443679019896428470520755027450832131D+00
C
      Alpha( 1) = 0.0931061725688367425719750222756765367649D+00
      Alpha( 2) = 1.0301539335400714347910217094650420222024D+00
C
      Xi( 1) = 1.093360513875665603511673495962D+00
      Xi( 2) = 2.025198654658210892275452263966D+00
      Xi( 3) = 5.150796128864857912049018962009D+00
      Xi( 4) = 18.070172281159798970476271229302D+00
      GOTO 9999
C
C     ===== K = 03 =====
C
  300 CONTINUE
C
      GOTO ( 302, 305, 311, 312, 313, 314, 315, 316, 317, 318, 319, 321,
     *       399, 399, 399, 399, 399, 399, 399, 399, 399, 399, 399, 399,
     *       399, 399, 399, 399, 399, 399, 399),InitR
C
C     ----- K = 03 , R = 2E0 -----
C
  302 CONTINUE
C
      RIni = 2.0D+00
C
      Omega( 1) = 0.7470594125158281184444220013318727069418D+00
      Omega( 2) = 1.9588903206974837864322303015995885289158D+00
      Omega( 3) = 4.1617883704995866428130724123946038162103D+00
C
      Alpha( 1) = 0.2874933084171688071331227098959004706558D+00
      Alpha( 2) = 1.6046370491507346881746715050454099582566D+00
      Alpha( 3) = 4.5064864376399242354084839767836001556134D+00
C
      Xi( 1) = 1.011367648358682691891487215319D+00
      Xi( 2) = 1.102107335879371621334757436195D+00
      Xi( 3) = 1.280275921324935759957296121936D+00
      Xi( 4) = 1.528180108975737024209172465472D+00
      Xi( 5) = 1.792807750855102859387051050799D+00
      Xi( 6) = 1.974208219701866289202789317425D+00
      GOTO 9999
C
C     ----- K = 03 , R = 5E0 -----
C
  305 CONTINUE
C
      RIni = 5.0D+00
C
      Omega( 1) = 0.4365792105033683860376356178356616055680D+00
      Omega( 2) = 1.2723305210744182858640521915916110629041D+00
      Omega( 3) = 3.2151540832501005597130611102230091091769D+00
C
      Alpha( 1) = 0.1660731375014149112742277078491603958810D+00
      Alpha( 2) = 0.9784372085453775896011531509888214941384D+00
      Alpha( 3) = 3.0530159808682457700784035337093769157946D+00
C
      Xi( 1) = 1.025951895784413650568669962304D+00
      Xi( 2) = 1.242824075650160353407464031150D+00
      Xi( 3) = 1.722934671753702002227302825244D+00
      Xi( 4) = 2.541326431637942081815523254207D+00
      Xi( 5) = 3.710191462368181307841102922751D+00
      Xi( 6) = 4.812712621252895713562514057315D+00
      GOTO 9999
C
C     ----- K = 03 , R = 1E1 -----
C
  311 CONTINUE
C
      RIni = 1.0D+01
C
      Omega( 1) = 0.2847348675836868480131403719157745513257D+00
      Omega( 2) = 0.9583115118002354952046577263313764660779D+00
      Omega( 3) = 2.8443772254878889997348895279927205592685D+00
C
      Alpha( 1) = 0.1064455484301103369894087730118670265256D+00
      Alpha( 2) = 0.6791933646735415630178896162494339705518D+00
      Alpha( 3) = 2.4024163101166760476718603500856374921568D+00
C
      Xi( 1) = 1.036540052009057201009949167325D+00
      Xi( 2) = 1.350512629433860964335220589128D+00
      Xi( 3) = 2.095440330945998347236119974113D+00
      Xi( 4) = 3.530895274669204600285002393356D+00
      Xi( 5) = 6.064676170659236835534106058176D+00
      Xi( 6) = 9.312377099278923659429751946703D+00
      GOTO 9999
C
C     ----- K = 03 , R = 2E1 -----
C
  312 CONTINUE
C
      RIni = 2.0D+01
C
      Omega( 1) = 0.1871865945351820821081342480218800972125D+00
      Omega( 2) = 0.7633857188437595157155454717035070189013D+00
      Omega( 3) = 2.6169711142660253494832134357750419439981D+00
C
      Alpha( 1) = 0.0679013689405892574945246102646745356424D+00
      Alpha( 2) = 0.4897948875089661041485125658745403143257D+00
      Alpha( 3) = 1.9962015809195739495960839060550995327503D+00
C
      Xi( 1) = 1.046004741949051884642926668612D+00
      Xi( 2) = 1.450848248703333635638811072166D+00
      Xi( 3) = 2.466950930523215088823255070061D+00
      Xi( 4) = 4.633421766226166740951925016034D+00
      Xi( 5) = 9.248118901369171863426077440096D+00
      Xi( 6) = 17.567914206128274501664265905276D+00
      GOTO 9999
C
C     ----- K = 03 , R = 3E1 -----
C
  313 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.1489350280707968614941037743815144267501D+00
      Omega( 2) = 0.6872993149781066612611854504955033462466D+00
      Omega( 3) = 2.5251745372914686719838239792323975052568D+00
C
      Alpha( 1) = 0.0526373510053126044740958670159480803363D+00
      Alpha( 2) = 0.4159550868841870631409125130817372451020D+00
      Alpha( 3) = 1.8359809178980240516269178963071340149327D+00
C
      Xi( 1) = 1.050731777552041640516958254103D+00
      Xi( 2) = 1.502520887461407482184674866410D+00
      Xi( 3) = 2.667783186472804712440626273562D+00
      Xi( 4) = 5.279908463448491815882962496786D+00
      Xi( 5) = 11.391839687100992576165858771731D+00
      Xi( 6) = 24.939047426082355123772016369799D+00
      GOTO 9999
C
C     ----- K = 03 , R = 4E1 -----
C
  314 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.1282763604743762873410797484274858959452D+00
      Omega( 2) = 0.6457800255491172428698280760794858679219D+00
      Omega( 3) = 2.4737466774457358498554016001946820324520D+00
C
      Alpha( 1) = 0.0443249082477236827920660192786650455332D+00
      Alpha( 2) = 0.3760267128377692993999176218622437772865D+00
      Alpha( 3) = 1.7481061682069727251757906838491862799856D+00
C
      Xi( 1) = 1.053649416445949383441815783158D+00
      Xi( 2) = 1.534965201688940375458411469189D+00
      Xi( 3) = 2.797338196085134732517263334017D+00
      Xi( 4) = 5.717225016873238558224451510981D+00
      Xi( 5) = 12.960177258090050030417406112804D+00
      Xi( 6) = 31.526153474782242738805315696027D+00
      GOTO 9999
C
C     ----- K = 03 , R = 5E1 -----
C
  315 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.1153310756696578369059902335891365510179D+00
      Omega( 2) = 0.6194232578183114542973164429184862456168D+00
      Omega( 3) = 2.4405036326787768433295433556651232720469D+00
C
      Alpha( 1) = 0.0390820984323069169736556555116990807619D+00
      Alpha( 2) = 0.3509230683778314732214603094417881834488D+00
      Alpha( 3) = 1.6921947262166477795545388440601186630374D+00
C
      Xi( 1) = 1.055646218481340204787058412172D+00
      Xi( 2) = 1.557420836116662476826533190177D+00
      Xi( 3) = 2.888617000023559338922271577665D+00
      Xi( 4) = 6.035444707901820501731726675843D+00
      Xi( 5) = 14.162824787784242307718540754280D+00
      Xi( 6) = 37.407934771317043306376737277219D+00
      GOTO 9999
C
C     ----- K = 03 , R = 6E1 -----
C
  316 CONTINUE
C
      RIni = 6.0D+01
C
      Omega( 1) = 0.1065007940761846950413838444604586186415D+00
      Omega( 2) = 0.6012169550303838670810181732395705012095D+00
      Omega( 3) = 2.4172446838218981427469367329052829518332D+00
C
      Alpha( 1) = 0.0354878575633928706640498294822005931337D+00
      Alpha( 2) = 0.3337331092077952527763084339396115751697D+00
      Alpha( 3) = 1.6535367452293752402354976016596310728346D+00
C
      Xi( 1) = 1.057097434208872948745855380182D+00
      Xi( 2) = 1.573870966645937349978265351602D+00
      Xi( 3) = 2.956339877497224052612170841492D+00
      Xi( 4) = 6.277178023394458090695319230079D+00
      Xi( 5) = 15.111869488781895343176842061439D+00
      Xi( 6) = 42.640384216822704587651626795264D+00
      GOTO 9999
C
C     ----- K = 03 , R = 7E1 -----
C
  317 CONTINUE
C
      RIni = 7.0D+01
C
      Omega( 1) = 0.1001553077368610132974404043626481808360D+00
      Omega( 2) = 0.5879845498502223689126118533199161220182D+00
      Omega( 3) = 2.4001809699912176228331350813505196128972D+00
C
      Alpha( 1) = 0.0328950169895424992609872082843791574192D+00
      Alpha( 2) = 0.3213333776732262143905564238943739496790D+00
      Alpha( 3) = 1.6254323321274150328501900686362091619230D+00
C
      Xi( 1) = 1.058191270863622694426570136628D+00
      Xi( 2) = 1.586343845420391796365890135778D+00
      Xi( 3) = 3.008179205772900836370065125180D+00
      Xi( 4) = 6.465564686699797597186495368149D+00
      Xi( 5) = 15.873238341608802483631568414069D+00
      Xi( 6) = 47.261670190788973886647506006398D+00
      GOTO 9999
C
C     ----- K = 03 , R = 8E1 -----
C
  318 CONTINUE
C
      RIni = 8.0D+01
C
      Omega( 1) = 0.0954485614946326069841066666588069722366D+00
      Omega( 2) = 0.5780719785823823125598015792991901662390D+00
      Omega( 3) = 2.3873083432732535235590992650500652416667D+00
C
      Alpha( 1) = 0.0309660405726075814911553646738284761852D+00
      Alpha( 2) = 0.3121038074526534121424636153419029938050D+00
      Alpha( 3) = 1.6043802276101923300543825035013867363887D+00
C
      Xi( 1) = 1.059033121789881161920778007701D+00
      Xi( 2) = 1.595987137144691836248243388408D+00
      Xi( 3) = 3.048551860121487124032427229459D+00
      Xi( 4) = 6.614345532157845423959496855559D+00
      Xi( 5) = 16.488307114705869405035798536119D+00
      Xi( 6) = 51.297243347978425765526511170833D+00
      GOTO 9999
C
C     ----- K = 03 , R = 9E1 -----
C
  319 CONTINUE
C
      RIni = 9.0D+01
C
      Omega( 1) = 0.0918992098575219402425365983200222785854D+00
      Omega( 2) = 0.5705336658200345004285022887380307565763D+00
      Omega( 3) = 2.3774665399131085228353471583417899637425D+00
C
      Alpha( 1) = 0.0295080586695789376959540937367387769541D+00
      Alpha( 2) = 0.3051223245003002779000573357892633907795D+00
      Alpha( 3) = 1.5883739921280894394003957437178087275242D+00
C
      Xi( 1) = 1.059686558350682067061211688586D+00
      Xi( 2) = 1.603499064055321644611459830365D+00
      Xi( 3) = 3.080180337433629595518189137060D+00
      Xi( 4) = 6.732188839664491609906388225326D+00
      Xi( 5) = 16.984271206669289473734085760270D+00
      Xi( 6) = 54.763342161663440026825311335301D+00
      GOTO 9999
C
C     ----- K = 03 , R = 1E2 -----
C
  321 CONTINUE
C
      RIni = 1.0D+02
C
      Omega( 1) = 0.0892157079089669919486071947289929084945D+00
      Omega( 2) = 0.5647942501087851651861426738321370066842D+00
      Omega( 3) = 2.3699428171677330734901795183944273048837D+00
C
      Alpha( 1) = 0.0284038048199532795564959406289795573741D+00
      Alpha( 2) = 0.2998300570947111501655338999805522348652D+00
      Alpha( 3) = 1.5761907592561199853995260178329829159338D+00
C
      Xi( 1) = 1.060191971776997436806982966218D+00
      Xi( 2) = 1.609324357152911759254176882195D+00
      Xi( 3) = 3.104816257355714646673852663561D+00
      Xi( 4) = 6.824779154407094683074119778610D+00
      Xi( 5) = 17.379498870857770060141533896569D+00
      Xi( 6) = 57.669325212093752022757264796837D+00
      GOTO 9999
C
C     ----- K = 03 , R = 2E2 (Best : R = 146.8) -----
C
  399 CONTINUE
C
      RIni = 2.0D+02
C
      Omega( 1) = 0.0842336964653581986314334444088114395299D+00
      Omega( 2) = 0.5540364331533339688207195905533808399923D+00
      Omega( 3) = 2.3557688429224689736704578812975796608953D+00
C
      Alpha( 1) = 0.0263492728481437144350885744964463341944D+00
      Alpha( 2) = 0.2899685548481513726706056827131519071372D+00
      Alpha( 3) = 1.5533661852788560075230156476422394007386D+00
C
      Xi( 1) = 1.061157926448131444429646974559D+00
      Xi( 2) = 1.620497451354158359860917204553D+00
      Xi( 3) = 3.152337144451245588925442087103D+00
      Xi( 4) = 7.005381504309560792426170294789D+00
      Xi( 5) = 18.164730430146035160265949315317D+00
      Xi( 6) = 63.847716086263413940576905147850D+00
      GOTO 9999
C
C     ===== K = 04 =====
C
  400 CONTINUE
C
      GOTO ( 402, 405, 411, 412, 413, 414, 415, 416, 417, 418, 419, 421,
     *       422, 423, 424, 499, 499, 499, 499, 499, 499, 499, 499, 499,
     *       499, 499, 499, 499, 499, 499, 499),InitR
C
C     ----- K = 04 , R = 2E0 -----
C
  402 CONTINUE
C
      RIni = 2.0D+00
C
      Omega( 1) = 0.5753086924118193929257938923349513515859D+00
      Omega( 2) = 1.4324382751398495198852031662362094266427D+00
      Omega( 3) = 2.6015042679469189200285039742510662108543D+00
      Omega( 4) = 4.8372822744559876655644214960005911052576D+00
C
      Alpha( 1) = 0.2225282101342608647137188276898811523097D+00
      Alpha( 2) = 1.2118271769428668733874221441482177397120D+00
      Alpha( 3) = 3.1874817920306483924348245340141261294775D+00
      Alpha( 4) = 6.7398873748331835607794371156842316850089D+00
C
      Xi( 1) = 1.006506167989540905796305714226D+00
      Xi( 2) = 1.058464707807605930571862029588D+00
      Xi( 3) = 1.161505681175087609235951902065D+00
      Xi( 4) = 1.311950847515082049137349895496D+00
      Xi( 5) = 1.499364570000543965826340919723D+00
      Xi( 6) = 1.701144502067884147606356692872D+00
      Xi( 7) = 1.879073519869262322767519690636D+00
      Xi( 8) = 1.985688675481389943249486973453D+00
      GOTO 9999
C
C     ----- K = 04 , R = 5E0 -----
C
  405 CONTINUE
C
      RIni = 5.0D+00
C
      Omega( 1) = 0.3311114029441486656963472556824967796274D+00
      Omega( 2) = 0.8719992880247880499789900343721171793732D+00
      Omega( 3) = 1.7621447067201108642887225075313040179026D+00
      Omega( 4) = 3.7722154835897864035877680155550706331269D+00
C
      Alpha( 1) = 0.1272851568278456049125666599275419343940D+00
      Alpha( 2) = 0.7128840158683421319844343566618505292354D+00
      Alpha( 3) = 1.9837003473579683075636928557550220375560D+00
      Alpha( 4) = 4.5827403151170012138813925073321797754033D+00
C
      Xi( 1) = 1.015209771693483679195178781551D+00
      Xi( 2) = 1.139795898096606982757665604833D+00
      Xi( 3) = 1.403933946114703158530509496948D+00
      Xi( 4) = 1.834881569408138940859356869062D+00
      Xi( 5) = 2.463725443706430423464887047302D+00
      Xi( 6) = 3.294920084559301091866462263269D+00
      Xi( 7) = 4.220215506742421560527206425562D+00
      Xi( 8) = 4.899608368220416046347892580215D+00
      GOTO 9999
C
C     ----- K = 04 , R = 1E1 -----
C
  411 CONTINUE
C
      RIni = 1.0D+01
C
      Omega( 1) = 0.2109714423067830878737302299774647451613D+00
      Omega( 2) = 0.6012085214120437117791585779347229845371D+00
      Omega( 3) = 1.3842061450697847088241870805447319980885D+00
      Omega( 4) = 3.3493765842990434143926908872046510623477D+00
C
      Alpha( 1) = 0.0804090244155528715704300650457536647764D+00
      Alpha( 2) = 0.4687930946688866533453993179492158560606D+00
      Alpha( 3) = 1.4098715378713611090033888673112016931555D+00
      Alpha( 4) = 3.6048552773416722889727320522368358979293D+00
C
      Xi( 1) = 1.021921225313027279247950207530D+00
      Xi( 2) = 1.204384941419539613869163496851D+00
      Xi( 3) = 1.607251451203153021457352422896D+00
      Xi( 4) = 2.309746723025455121362792265671D+00
      Xi( 5) = 3.442661796455891675284438635529D+00
      Xi( 6) = 5.174135212344695517513309956925D+00
      Xi( 7) = 7.518763336083220949045829994972D+00
      Xi( 8) = 9.646065333930355733746409008234D+00
      GOTO 9999
C
C     ----- K = 04 , R = 2E1 -----
C
  412 CONTINUE
C
      RIni = 2.0D+01
C
      Omega( 1) = 0.1329180070416147036311800902486091047194D+00
      Omega( 2) = 0.4264908534251436216986307081722173961680D+00
      Omega( 3) = 1.1416477627542353926511781159192082668596D+00
      Omega( 4) = 3.0780675346606618560107992976782043115236D+00
C
      Alpha( 1) = 0.0499698775458987140683406961289003689330D+00
      Alpha( 2) = 0.3105631707014663601691250444103431505027D+00
      Alpha( 3) = 1.0398249107852826248832647793030048433138D+00
      Alpha( 4) = 2.9750469432139245655032783188786993378017D+00
C
      Xi( 1) = 1.028384820056705543857462903379D+00
      Xi( 2) = 1.268165682588351670712946417385D+00
      Xi( 3) = 1.816842986610023565584304261478D+00
      Xi( 4) = 2.829868970539194235119459519545D+00
      Xi( 5) = 4.612352038396031943777442219101D+00
      Xi( 6) = 7.714494741864398525896251435086D+00
      Xi( 7) = 12.817017304467905825828000221378D+00
      Xi( 8) = 18.811438665072096257727118917558D+00
      GOTO 9999
C
C     ----- K = 04 , R = 3E1 -----
C
  413 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.1016013598614176051835515483812155679288D+00
      Omega( 2) = 0.3564500076585994777433278463130150726101D+00
      Omega( 3) = 1.0408682232432721238716069600549474216677D+00
      Omega( 4) = 2.9620670599932430282208439820124112884514D+00
C
      Alpha( 1) = 0.0377493790524789586313635092462703823912D+00
      Alpha( 2) = 0.2471731689001984523724369038899517647678D+00
      Alpha( 3) = 0.8901553582336263936887477132842150240322D+00
      Alpha( 4) = 2.7154179633949424115240550703376243291132D+00
C
      Xi( 1) = 1.031884400461319287699291091798D+00
      Xi( 2) = 1.303394819838255136359300712101D+00
      Xi( 3) = 1.936484594904135799484437563400D+00
      Xi( 4) = 3.140840870036364630224837157790D+00
      Xi( 5) = 5.359510014817230771043410353371D+00
      Xi( 6) = 9.492142930507176949023184775456D+00
      Xi( 7) = 17.064887801084204502904961842091D+00
      Xi( 8) = 27.603035761750134504874387708639D+00
      GOTO 9999
C
C     ----- K = 04 , R = 4E1 -----
C
  414 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.0842613451067411968134360580884312241778D+00
      Omega( 2) = 0.3175083683168243892547700363282459079528D+00
      Omega( 3) = 0.9827580985120555624170654573923400221247D+00
      Omega( 4) = 2.8938728508266506878315177386795653546869D+00
C
      Alpha( 1) = 0.0309718827633249730023221003502587045375D+00
      Alpha( 2) = 0.2120774570049154385080730084167655036254D+00
      Alpha( 3) = 0.8062000011279130441146055507939394146888D+00
      Alpha( 4) = 2.5672235706906126311704907649868800945114D+00
C
      Xi( 1) = 1.034189547957952938364925565828D+00
      Xi( 2) = 1.326882303593381459455753657029D+00
      Xi( 3) = 2.017833912902638116102643772720D+00
      Xi( 4) = 3.358339765728229320143963243517D+00
      Xi( 5) = 5.903755879081696503109938367615D+00
      Xi( 6) = 10.860721975215827649410837985045D+00
      Xi( 7) = 20.628245205359198052516567223336D+00
      Xi( 8) = 36.056476263170379424616296404338D+00
      GOTO 9999
C
C     ----- K = 04 , R = 5E1 -----
C
  415 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.0731190495973734073321850628068840904916D+00
      Omega( 2) = 0.2923269496573136512322394625629229381047D+00
      Omega( 3) = 0.9440309351700997790167620149492933023794D+00
      Omega( 4) = 2.8478366618531806178475640489722309212084D+00
C
      Alpha( 1) = 0.0266079437201842705628133858158923530368D+00
      Alpha( 2) = 0.1895065774790961192112310682023057495371D+00
      Alpha( 3) = 0.7515202985781507951474830764304613239801D+00
      Alpha( 4) = 2.4693480571818920657912316274362751755689D+00
C
      Xi( 1) = 1.035859420488966962437850372059D+00
      Xi( 2) = 1.344041617001740170165430265925D+00
      Xi( 3) = 2.078083732500034939115474141857D+00
      Xi( 4) = 3.522680772883364073815026795700D+00
      Xi( 5) = 6.327056844638300348981213572230D+00
      Xi( 6) = 11.967990442183917604648812904600D+00
      Xi( 7) = 23.694624806761000475330281034303D+00
      Xi( 8) = 44.192689038985971203971070409011D+00
      GOTO 9999
C
C     ----- K = 04 , R = 6E1 -----
C
  416 CONTINUE
C
      RIni = 6.0D+01
C
      Omega( 1) = 0.0653078213757228939496122036012515366110D+00
      Omega( 2) = 0.2745449432568480962601076972307012624697D+00
      Omega( 3) = 0.9160037860065624061296600533221834439246D+00
      Omega( 4) = 2.8142148838640842138485165246919450510177D+00
C
      Alpha( 1) = 0.0235419918657027563556896837884679918318D+00
      Alpha( 2) = 0.1736604768046437706328520220178113220300D+00
      Alpha( 3) = 0.7126891672717474266905354762791091616236D+00
      Alpha( 4) = 2.3990539647469760751227058070966791092360D+00
C
      Xi( 1) = 1.037140461619688747412632023881D+00
      Xi( 2) = 1.357289114402989183786468752047D+00
      Xi( 3) = 2.125078140520427659714702839366D+00
      Xi( 4) = 3.652834342225301128646158810298D+00
      Xi( 5) = 6.669801867650820922320958894502D+00
      Xi( 6) = 12.892172317412252485931134504682D+00
      Xi( 7) = 26.378389732187363631021881360539D+00
      Xi( 8) = 52.027795795803441894489260732826D+00
      GOTO 9999
C
C     ----- K = 04 , R = 7E1 -----
C
  417 CONTINUE
C
      RIni = 7.0D+01
C
      Omega( 1) = 0.0595075285913887735555084096406019966707D+00
      Omega( 2) = 0.2612399364166240053278938981762280491239D+00
      Omega( 3) = 0.8946057701502559431701484815580016629610D+00
      Omega( 4) = 2.7883700207922745009040538244349249907827D+00
C
      Alpha( 1) = 0.0212604165557323362225922592366744723336D+00
      Alpha( 2) = 0.1618724211569130696873998823126505897108D+00
      Alpha( 3) = 0.6835043049845140389537447023204919105410D+00
      Alpha( 4) = 2.3457307234433385698373769612246064752981D+00
C
      Xi( 1) = 1.038161965305633614828524635687D+00
      Xi( 2) = 1.367906084839922344650701979685D+00
      Xi( 3) = 2.163045651450423545025109062223D+00
      Xi( 4) = 3.759269913511129595096865729786D+00
      Xi( 5) = 6.955090352815260643353639791187D+00
      Xi( 6) = 13.680462783788660021712380032710D+00
      Xi( 7) = 28.756568839488641399840251899889D+00
      Xi( 8) = 59.575852601901405913853437823491D+00
      GOTO 9999
C
C     ----- K = 04 , R = 8E1 -----
C
  418 CONTINUE
C
      RIni = 8.0D+01
C
      Omega( 1) = 0.0550202496658656947137210676407326204185D+00
      Omega( 2) = 0.2508680510132605612151797391806695713967D+00
      Omega( 3) = 0.8776426020163860330729720116149650266379D+00
      Omega( 4) = 2.7677729356352664641209571083635410104762D+00
C
      Alpha( 1) = 0.0194916432536069991175730915095032358408D+00
      Alpha( 2) = 0.1527342674760213229605320187087791339309D+00
      Alpha( 3) = 0.6606717166460769025495593231855906424244D+00
      Alpha( 4) = 2.3036894119803122266132433537677570711821D+00
C
      Xi( 1) = 1.038999699185351887459859143714D+00
      Xi( 2) = 1.376648661571191919659150337818D+00
      Xi( 3) = 2.194515905695126582828496930588D+00
      Xi( 4) = 3.848374497746649226153151102103D+00
      Xi( 5) = 7.197426589624710588366518493331D+00
      Xi( 6) = 14.363811748530706862911343613121D+00
      Xi( 7) = 30.884378731911624308553721007797D+00
      Xi( 8) = 66.849598004065109602134242550164D+00
      GOTO 9999
C
C     ----- K = 04 , R = 9E1 -----
C
  419 CONTINUE
C
      RIni = 9.0D+01
C
      Omega( 1) = 0.0514404134505259737501683264815355300925D+00
      Omega( 2) = 0.2425316805672637418404054970855199258040D+00
      Omega( 3) = 0.8638140458215766392642900872722577787499D+00
      Omega( 4) = 2.7509105528330038540653357248544352842146D+00
C
      Alpha( 1) = 0.0180777829160191627153871996372402897180D+00
      Alpha( 2) = 0.1454283956184184608531519827812061862460D+00
      Alpha( 3) = 0.6422668197625267019735227636001440032487D+00
      Alpha( 4) = 2.2695770442654134364975082593929300855962D+00
C
      Xi( 1) = 1.039701559473601906805062533579D+00
      Xi( 2) = 1.383998181274938783931992281051D+00
      Xi( 3) = 2.221116333865750902369640140677D+00
      Xi( 4) = 3.924322959823527990578378998876D+00
      Xi( 5) = 7.406527255831039190710901598536D+00
      Xi( 6) = 14.963700097020325378849303632478D+00
      Xi( 7) = 32.803053350819499158624870105427D+00
      Xi( 8) = 73.860605145714235443554773752339D+00
      GOTO 9999
C
C     ----- K = 04 , R = 1E2 -----
C
  421 CONTINUE
C
      RIni = 1.0D+02
C
      Omega( 1) = 0.0485154483001083335682321676096684370805D+00
      Omega( 2) = 0.2356710456997001232359086747569776321143D+00
      Omega( 3) = 0.8522950664164703286885226007907334633273D+00
      Omega( 4) = 2.7368154329484132645643884584885086042050D+00
C
      Alpha( 1) = 0.0169204412159692850155435659906255629892D+00
      Alpha( 2) = 0.1394458611981022595169591934971542457333D+00
      Alpha( 3) = 0.6270840649394218112842348811231119043441D+00
      Alpha( 4) = 2.2412769022182530667848532335284517102991D+00
C
      Xi( 1) = 1.040299549550007821561471710314D+00
      Xi( 2) = 1.390278043830974921083593054405D+00
      Xi( 3) = 2.243950554331804970228242401831D+00
      Xi( 4) = 3.989985712671379214496955589730D+00
      Xi( 5) = 7.589215465942349175560571739041D+00
      Xi( 6) = 15.495677450862050677712900004224D+00
      Xi( 7) = 34.544307705980290587599590423906D+00
      Xi( 8) = 80.619317886311765174178756154788D+00
      GOTO 9999
C
C     ----- K = 04 , R = 2E2 -----
C
  422 CONTINUE
C
      RIni = 2.0D+02
C
      Omega( 1) = 0.0346528544291562165083103903839845116863D+00
      Omega( 2) = 0.2022620232531529587587451612940014911146D+00
      Omega( 3) = 0.7941993308902318025530031597725155734224D+00
      Omega( 4) = 2.6650476801253452636790430840818544311333D+00
C
      Alpha( 1) = 0.0114012537962333943318828338546466749648D+00
      Alpha( 2) = 0.1108289024285984015197803129182396730812D+00
      Alpha( 3) = 0.5527015496667354365950206029900471094152D+00
      Alpha( 4) = 2.1002718748225481968597977422419376125617D+00
C
      Xi( 1) = 1.043510941463284021849672322801D+00
      Xi( 2) = 1.424292069942840605165264300513D+00
      Xi( 3) = 2.369336520202418435061819668164D+00
      Xi( 4) = 4.358428194352952914601695422192D+00
      Xi( 5) = 8.647755843713323312300922385276D+00
      Xi( 6) = 18.726822746608632615228362539028D+00
      Xi( 7) = 45.936015344901233076885249673182D+00
      Xi( 8) = 135.930079665824886978042762564201D+00
      GOTO 9999
C
C     ----- K = 04 , R = 3E2 -----
C
  423 CONTINUE
C
      RIni = 3.0D+02
C
      Omega( 1) = 0.0300842254221815340984474453078778566351D+00
      Omega( 2) = 0.1907535375929733114611520222725360440563D+00
      Omega( 3) = 0.7733295903723639011994803182936664143199D+00
      Omega( 4) = 2.6389901708906205511726700274621748576465D+00
C
      Alpha( 1) = 0.0095667747961039157790762718983068602085D+00
      Alpha( 2) = 0.1012379805187263745680720586350709311318D+00
      Alpha( 3) = 0.5269302725083039626134684407787744930829D+00
      Alpha( 4) = 2.0503770061024033061029825386611946669291D+00
C
      Xi( 1) = 1.044749147786660410067334503470D+00
      Xi( 2) = 1.437539613008931760579182934645D+00
      Xi( 3) = 2.418961360192959209481494697869D+00
      Xi( 4) = 4.508037017931998435057300289586D+00
      Xi( 5) = 9.094182415437122085286048012165D+00
      Xi( 6) = 20.168709377634733440065595289070D+00
      Xi( 7) = 51.485939374184318526039838559427D+00
      Xi( 8) = 171.534260231812210592550194121486D+00
      GOTO 9999
C
C     ----- K = 04 , R = 4E2 -----
C
  424 CONTINUE
C
      RIni = 4.0D+02
C
      Omega( 1) = 0.0284661205929678002576150783187181048106D+00
      Omega( 2) = 0.1865878163370500782299293146548890831582D+00
      Omega( 3) = 0.7656572669069769046914022969652080519154D+00
      Omega( 4) = 2.6293741631156146041845000915770924621029D+00
C
      Alpha( 1) = 0.0089149584966167103430493114449295610768D+00
      Alpha( 2) = 0.0978104436659742490270375535332814109779D+00
      Alpha( 3) = 0.5175885566543470744449786935525992248586D+00
      Alpha( 4) = 2.0321413590981910171542196286154080553388D+00
C
      Xi( 1) = 1.045216293509242250106376637131D+00
      Xi( 2) = 1.442556485572168388509251002461D+00
      Xi( 3) = 2.437872235678494421511064427222D+00
      Xi( 4) = 4.565631989514494578404762981094D+00
      Xi( 5) = 9.268663264519454292669764505774D+00
      Xi( 6) = 20.745603718304171844341587949145D+00
      Xi( 7) = 53.790818525643893238674664303289D+00
      Xi( 8) = 188.385827462242781263346280695714D+00
      GOTO 9999
C
C     ----- K = 04 , R = 5E2 (Best : R = 436.1) -----
C
  499 CONTINUE
C
      RIni = 5.0D+02
C
      Omega( 1) = 0.0283509824145518462858693315570279835924D+00
      Omega( 2) = 0.1862893386105706895712131113129750303870D+00
      Omega( 3) = 0.7651050730792038023189878537255737001033D+00
      Omega( 4) = 2.6286812528458147286778562312647977705637D+00
C
      Alpha( 1) = 0.0088685367946663247640828832071835385875D+00
      Alpha( 2) = 0.0975658350270974381366175312568778110744D+00
      Alpha( 3) = 0.5169189990170087889041414253021855529369D+00
      Alpha( 4) = 2.0308311260118173089978671486122152600728D+00
C
      Xi( 1) = 1.045250181977026276611796029581D+00
      Xi( 2) = 1.442920714529975515167184441090D+00
      Xi( 3) = 2.439247862946276155723945855414D+00
      Xi( 4) = 4.569832780139895886221956811113D+00
      Xi( 5) = 9.281448986761759875618871262759D+00
      Xi( 6) = 20.788176902012078735987321564949D+00
      Xi( 7) = 53.962878608716110463011661479982D+00
      Xi( 8) = 189.696990422674594614638188261324D+00
      GOTO 9999
C
C     ===== K = 05 =====
C
  500 CONTINUE
C
      GOTO ( 502, 505, 511, 512, 513, 514, 515, 516, 517, 518, 519, 521,
     *       522, 523, 524, 525, 526, 527, 528, 529, 531, 599, 599, 599,
     *       599, 599, 599, 599, 599, 599, 599),InitR
C
C     ----- K = 05 , R = 2E0 -----
C
  502 CONTINUE
C
      RIni = 2.0D+00
C
      Omega( 1) = 0.4683556115061091129983267961067738838210D+00
      Omega( 2) = 1.1387420740801897163681222213682531219092D+00
      Omega( 3) = 1.9531818897084401308490708704113103522104D+00
      Omega( 4) = 3.1153038431409946034094521349899764572910D+00
      Omega( 5) = 5.3979418510235632052904342081944832898444D+00
C
      Alpha( 1) = 0.1816122344478981142344448801440037755128D+00
      Alpha( 2) = 0.9776830415195706051986919427054800735277D+00
      Alpha( 3) = 2.5060472629612186804687334884533811418805D+00
      Alpha( 4) = 4.9959238094351378477869785044163108977955D+00
      Alpha( 5) = 9.0782980472197038765772014734523054357851D+00
C
      Xi( 1) = 1.004205961424876742218471881163D+00
      Xi( 2) = 1.037808810177569494902563573824D+00
      Xi( 3) = 1.104673526155082756558187262463D+00
      Xi( 4) = 1.203620646526363327142271353321D+00
      Xi( 5) = 1.331606110593612308330355165076D+00
      Xi( 6) = 1.482173116895686719907306028166D+00
      Xi( 7) = 1.643636751837373089933300174259D+00
      Xi( 8) = 1.797882208171468448811188001191D+00
      Xi( 9) = 1.921494689550123476790011167736D+00
      Xi(10) = 1.990922307578263482060257349993D+00
      GOTO 9999
C
C     ----- K = 05 , R = 5E0 -----
C
  505 CONTINUE
C
      RIni = 5.0D+00
C
      Omega( 1) = 0.2675144938257100701254221492009222060915D+00
      Omega( 2) = 0.6731648185410648106067992013645806537170D+00
      Omega( 3) = 1.2341495375414908785378573230495646839699D+00
      Omega( 4) = 2.1730749987606481090328391925226014791406D+00
      Omega( 5) = 4.2400491045957718516841439271303215718945D+00
C
      Alpha( 1) = 0.1033351211759554590466098772383318760149D+00
      Alpha( 2) = 0.5658998554798738632692738093066253668439D+00
      Alpha( 3) = 1.5002594864446179758615818045086598431226D+00
      Alpha( 4) = 3.1563889823905420255363563075690080950153D+00
      Alpha( 5) = 6.1935841722085905663650051700130916287890D+00
C
      Xi( 1) = 1.009973918929146431255200466737D+00
      Xi( 2) = 1.090947626405043171474856800174D+00
      Xi( 3) = 1.259063365124301121355382404587D+00
      Xi( 4) = 1.526000210062745005099721362640D+00
      Xi( 5) = 1.907288936960188738592957002904D+00
      Xi( 6) = 2.417693978255027923308234649724D+00
      Xi( 7) = 3.058580838005749557607926281122D+00
      Xi( 8) = 3.789900418969845993441558640313D+00
      Xi( 9) = 4.488508901774702612979400795368D+00
      Xi(10) = 4.937710663142177220773926249464D+00
      GOTO 9999
C
C     ----- K = 05 , R = 1E1 -----
C
  511 CONTINUE
C
      RIni = 1.0D+01
C
      Omega( 1) = 0.1686100332421062595255559918294707699715D+00
      Omega( 2) = 0.4447462226391396488648245083341592476245D+00
      Omega( 3) = 0.8914300222454683441828879320034673128248D+00
      Omega( 4) = 1.7540776690871601983497465870875942073326D+00
      Omega( 5) = 3.7773593670209078805664187905577477977204D+00
C
      Alpha( 1) = 0.0647973162949741815670734457199841926922D+00
      Alpha( 2) = 0.3632942397911341266415318623783647922210D+00
      Alpha( 3) = 1.0098084995269309119183404055419828182494D+00
      Alpha( 4) = 2.2813077174805524267257178516743465479522D+00
      Alpha( 5) = 4.8771539519428148820243129613061228155857D+00
C
      Xi( 1) = 1.014582863653057650654004673285D+00
      Xi( 2) = 1.134203389307848182355163701995D+00
      Xi( 3) = 1.389246760143581714357606526900D+00
      Xi( 4) = 1.812037270091620028882387072322D+00
      Xi( 5) = 2.454117398216207501576366367679D+00
      Xi( 6) = 3.388587251028423504711772906717D+00
      Xi( 7) = 4.699016817830402586476934434501D+00
      Xi( 8) = 6.418823685933021436830075989732D+00
      Xi( 9) = 8.349475129996430378988281351127D+00
      Xi(10) = 9.786083246339030821889237721223D+00
      GOTO 9999
C
C     ----- K = 05 , R = 2E1 -----
C
  512 CONTINUE
C
      RIni = 2.0D+01
C
      Omega( 1) = 0.1043419845604544110779284687984258539473D+00
      Omega( 2) = 0.2955585665005043083473218468171239692310D+00
      Omega( 3) = 0.6678313388646436865592193043994484469295D+00
      Omega( 4) = 1.4788816616452081139634552875428141760494D+00
      Omega( 5) = 3.4728708916802090379740558523380400401948D+00
C
      Alpha( 1) = 0.0397910355186319709875524593817308627308D+00
      Alpha( 2) = 0.2313133225382769353847172214000416090585D+00
      Alpha( 3) = 0.6899245534043173482918569205946113243044D+00
      Alpha( 4) = 1.7098435082055959542040105003657401994133D+00
      Alpha( 5) = 4.0149225768154399888662520190507621009601D+00
C
      Xi( 1) = 1.019217117302840040836235668831D+00
      Xi( 2) = 1.178415662855246872275580272138D+00
      Xi( 3) = 1.526280432408703061791278854376D+00
      Xi( 4) = 2.125613075941608395673376730173D+00
      Xi( 5) = 3.086751187750071800464449212065D+00
      Xi( 6) = 4.593720184650979501857215314686D+00
      Xi( 7) = 6.928231715123500258392275075181D+00
      Xi( 8) = 10.429602304922355690899871127897D+00
      Xi( 9) = 15.109104672966703414999090071547D+00
      Xi(10) = 19.306177258864197996898748321826D+00
      GOTO 9999
C
C     ----- K = 05 , R = 3E1 -----
C
  513 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.0785202296725237321788569942737634477226D+00
      Omega( 2) = 0.2352291127246095037714793130056989411969D+00
      Omega( 3) = 0.5755235459038332647744261316713476617224D+00
      Omega( 4) = 1.3603613273128701396854495220267722288554D+00
      Omega( 5) = 3.3389033426820007048260730009303642873419D+00
C
      Alpha( 1) = 0.0297577867799910187276724591112775364365D+00
      Alpha( 2) = 0.1781464556286160730389639406112678443606D+00
      Alpha( 3) = 0.5601138852938733217684888487930550127203D+00
      Alpha( 4) = 1.4739028120126258845998951096945006611350D+00
      Alpha( 5) = 3.6519366545145374282690942102647113642888D+00
C
      Xi( 1) = 1.021836627027043814075306371603D+00
      Xi( 2) = 1.203748836653624470203506613419D+00
      Xi( 3) = 1.606670345753720860427701810913D+00
      Xi( 4) = 2.315548608377475355837513126822D+00
      Xi( 5) = 3.486487534924729528957795809951D+00
      Xi( 6) = 5.397491666427624752450792344760D+00
      Xi( 7) = 8.519390416720351725046733726288D+00
      Xi( 8) = 13.555879139999029356182524974628D+00
      Xi( 9) = 21.021496749683150869966730134308D+00
      Xi(10) = 28.634423521169729146343407855113D+00
      GOTO 9999
C
C     ----- K = 05 , R = 4E1 -----
C
  514 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.0641814838050678704967598464659062074134D+00
      Omega( 2) = 0.2015553177348803795505350136552102924270D+00
      Omega( 3) = 0.5226910879403149711537254762383497563860D+00
      Omega( 4) = 1.2901011954205592439665767345857716463797D+00
      Omega( 5) = 3.2584115181337518739324021765924044302665D+00
C
      Alpha( 1) = 0.0241901803500972818551558720864824714170D+00
      Alpha( 2) = 0.1485733749263374468200897549063910219047D+00
      Alpha( 3) = 0.4872759750389151346509908346860129313427D+00
      Alpha( 4) = 1.3391490990318978887216316842057040048530D+00
      Alpha( 5) = 3.4411575727997536413432788648947280307766D+00
C
      Xi( 1) = 1.023618696885570287786852705647D+00
      Xi( 2) = 1.221131920310683900633870468955D+00
      Xi( 3) = 1.662645702391932947322369940046D+00
      Xi( 4) = 2.450464176589166269372085937839D+00
      Xi( 5) = 3.778112484856030209908611672276D+00
      Xi( 6) = 6.004079808102726620484823616408D+00
      Xi( 7) = 9.771896848110523801922422393318D+00
      Xi( 8) = 16.155999486316149824777221866157D+00
      Xi( 9) = 26.333922981430985232539909723926D+00
      Xi(10) = 37.796347893531772609704288967691D+00
      GOTO 9999
C
C     ----- K = 05 , R = 5E1 -----
C
  515 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.0549289684719608573105023054117257430562D+00
      Omega( 2) = 0.1797226084604662145184595864466636783163D+00
      Omega( 3) = 0.4876092684357195764356749517753897293915D+00
      Omega( 4) = 1.2421875649935035342474529129752625067340D+00
      Omega( 5) = 3.2030368198904093346955634924455580403446D+00
C
      Alpha( 1) = 0.0205985169364180585578296594009839459716D+00
      Alpha( 2) = 0.1294650714429877070990940191030915684678D+00
      Alpha( 3) = 0.4397972396705089706309306507581169398691D+00
      Alpha( 4) = 1.2499372840249322914873975065574995824136D+00
      Alpha( 5) = 3.2997722744581938507293233930894871264172D+00
C
      Xi( 1) = 1.024945306274655407348384006383D+00
      Xi( 2) = 1.234152929071583455950845975568D+00
      Xi( 3) = 1.705016340573330133512271189478D+00
      Xi( 4) = 2.554066360145842692896955838933D+00
      Xi( 5) = 4.006449188957825125866757387527D+00
      Xi( 6) = 6.490851924497973947456724852501D+00
      Xi( 7) = 10.808236636812674757827934524101D+00
      Xi( 8) = 18.395005103541284728618787625010D+00
      Xi( 9) = 31.182470293965814951003823196629D+00
      Xi(10) = 46.806410907619687457315027501181D+00
      GOTO 9999
C
C     ----- K = 05 , R = 6E1 -----
C
  516 CONTINUE
C
      RIni = 6.0D+01
C
      Omega( 1) = 0.0484092394476768007366320866607933837145D+00
      Omega( 2) = 0.1642667164220850362451762857740611423196D+00
      Omega( 3) = 0.4622317543222329255900738403939342902049D+00
      Omega( 4) = 1.2068045801673655563132636636680672381772D+00
      Omega( 5) = 3.1618887677979012988931789562840890539519D+00
C
      Alpha( 1) = 0.0180677099136463318526386444967668509776D+00
      Alpha( 2) = 0.1159843500861629782565289520368523312754D+00
      Alpha( 3) = 0.4060170289500016420044040121872797044489D+00
      Alpha( 4) = 1.1856093034676985442871580134216458191077D+00
      Alpha( 5) = 3.1967392715623493253812581471606790728401D+00
C
      Xi( 1) = 1.025987986599939304102439840971D+00
      Xi( 2) = 1.244436379710849673897732348138D+00
      Xi( 3) = 1.738749585862048788209056893184D+00
      Xi( 4) = 2.637470542163933028435157668312D+00
      Xi( 5) = 4.193080245286467884740372324970D+00
      Xi( 6) = 6.896419867232555861707304911690D+00
      Xi( 7) = 11.692614370719187257517679423557D+00
      Xi( 8) = 20.366298558617970998821378181276D+00
      Xi( 9) = 35.654447253787359656262978546692D+00
      Xi(10) = 55.674065914067402998516298495701D+00
      GOTO 9999
C
C     ----- K = 05 , R = 7E1 -----
C
  517 CONTINUE
C
      RIni = 7.0D+01
C
      Omega( 1) = 0.0435404641423749599441785345516064964499D+00
      Omega( 2) = 0.1526714077865754897296796457895062815169D+00
      Omega( 3) = 0.4428218963958594054004542367453112206022D+00
      Omega( 4) = 1.1792923812761709411681826376039339265844D+00
      Omega( 5) = 3.1297433721720418944422376084091297343548D+00
C
      Alpha( 1) = 0.0161774474636488094969117472812780711422D+00
      Alpha( 2) = 0.1059055746618679526608545332644339964645D+00
      Alpha( 3) = 0.3805585874490105481766921702302397534368D+00
      Alpha( 4) = 1.1365624686784197416161701355008517566603D+00
      Alpha( 5) = 3.1174919552876026438738288337759740898036D+00
C
      Xi( 1) = 1.026838038680242269021919421235D+00
      Xi( 2) = 1.252852622804190629098204656788D+00
      Xi( 3) = 1.766537099233853070679757524264D+00
      Xi( 4) = 2.706794351109581448465435382644D+00
      Xi( 5) = 4.350130814145329771498033055721D+00
      Xi( 6) = 7.243079059817089169429765727060D+00
      Xi( 7) = 12.463487647364055717000286271912D+00
      Xi( 8) = 22.129085741641334232621129984864D+00
      Xi( 9) = 39.810779880025723560105221920935D+00
      Xi(10) = 64.406047341502177097649628478848D+00
      GOTO 9999
C
C     ----- K = 05 , R = 8E1 -----
C
  518 CONTINUE
C
      RIni = 8.0D+01
C
      Omega( 1) = 0.0397514331536537414319993754413529174485D+00
      Omega( 2) = 0.1436062765137879609363786778675198263500D+00
      Omega( 3) = 0.4273835064255168622604236922102316498240D+00
      Omega( 4) = 1.1571122334202288380652218213739956809150D+00
      Omega( 5) = 3.1037318963686114029875556807880343512807D+00
C
      Alpha( 1) = 0.0147059497531036244120876258813235271106D+00
      Alpha( 2) = 0.0980528681678504911183773609661962211703D+00
      Alpha( 3) = 0.3605733280656044851652641120010045483468D+00
      Alpha( 4) = 1.0976668403036145611100307672991505114624D+00
      Alpha( 5) = 3.0541811446566613800959050228911451085878D+00
C
      Xi( 1) = 1.027549586211971455510481854301D+00
      Xi( 2) = 1.259920380026032329847524016131D+00
      Xi( 3) = 1.789997919671034310416801726973D+00
      Xi( 4) = 2.765763801097122743686759771542D+00
      Xi( 5) = 4.485113281421210089296852663310D+00
      Xi( 6) = 7.544956076967540107257315096234D+00
      Xi( 7) = 13.145950170095980844529870434911D+00
      Xi( 8) = 23.723757443087620236030210740807D+00
      Xi( 9) = 43.696432384087426959579092411445D+00
      Xi(10) = 73.007490012850690647150386780595D+00
      GOTO 9999
C
C     ----- K = 05 , R = 9E1 -----
C
  519 CONTINUE
C
      RIni = 9.0D+01
C
      Omega( 1) = 0.0367103115933988785408103542456936807525D+00
      Omega( 2) = 0.1362975085252602825992016347123048092271D+00
      Omega( 3) = 0.4147426140491629501059079460079104251236D+00
      Omega( 4) = 1.1387451937085927495315468283010318373272D+00
      Omega( 5) = 3.0821271502266368218312525550928171469423D+00
C
      Alpha( 1) = 0.0135244470648898468016302571240161434218D+00
      Alpha( 2) = 0.0917428214601229520200201608681744858131D+00
      Alpha( 3) = 0.3444004900958717987143227479140961122539D+00
      Alpha( 4) = 1.0659073186356656392668670885193193953455D+00
      Alpha( 5) = 3.0021576019667892893435373613542083148786D+00
C
      Xi( 1) = 1.028157239821055244765463099199D+00
      Xi( 2) = 1.265972618547575588121394118968D+00
      Xi( 3) = 1.810179597639107708308310595857D+00
      Xi( 4) = 2.816815239727872043030629245486D+00
      Xi( 5) = 4.603010284514818944989755777897D+00
      Xi( 6) = 7.811600560233624539370478334632D+00
      Xi( 7) = 13.757390685449223435496413081580D+00
      Xi( 8) = 25.179356479133221431362943398824D+00
      Xi( 9) = 47.345971610642393925111326424826D+00
      Xi(10) = 81.482534713832055844884294515396D+00
      GOTO 9999
C
C     ----- K = 05 , R = 1E2 -----
C
  521 CONTINUE
C
      RIni = 1.0D+02
C
      Omega( 1) = 0.0342103034874631526696152957617291789916D+00
      Omega( 2) = 0.1302622960820672937630720217283553097332D+00
      Omega( 3) = 0.4041584156458790294866832704112624696791D+00
      Omega( 4) = 1.1232181675451095866731174299779638658947D+00
      Omega( 5) = 3.0638169728620764668738235547706949546409D+00
C
      Alpha( 1) = 0.0125527275404679892651334575069677068626D+00
      Alpha( 2) = 0.0865493948772892533749523380492973956279D+00
      Alpha( 3) = 0.3310012429371387315162403308654859301896D+00
      Alpha( 4) = 1.0393832416814095931769171277103680495202D+00
      Alpha( 5) = 2.9584699518773349188188737102223058172967D+00
C
      Xi( 1) = 1.028684362432210875628289803441D+00
      Xi( 2) = 1.271235368477093939892745566578D+00
      Xi( 3) = 1.827797553096407015267403639580D+00
      Xi( 4) = 2.861628476901701719934886791741D+00
      Xi( 5) = 4.707300182002275256310658280157D+00
      Xi( 6) = 8.049789021787946660321799718929D+00
      Xi( 7) = 14.310414647355005401084537819401D+00
      Xi( 8) = 26.517614005906436927539893133599D+00
      Xi( 9) = 50.786829027389754267346466676258D+00
      Xi(10) = 89.834677621284348428276622655630D+00
      GOTO 9999
C
C     ----- K = 05 , R = 2E2 -----
C
  522 CONTINUE
C
      RIni = 2.0D+02
C
      Omega( 1) = 0.0219924131992907643790133211808557334166D+00
      Omega( 2) = 0.1002064224819224335166702351263001702364D+00
      Omega( 3) = 0.3489637351854245363700929988270971193742D+00
      Omega( 4) = 1.0398862719837947781158921101152259325318D+00
      Omega( 5) = 2.9648211490348502911412048588246648250788D+00
C
      Alpha( 1) = 0.0077919805414365443251355311960609784094D+00
      Alpha( 2) = 0.0610302875027291444151751869523492288749D+00
      Alpha( 3) = 0.2635451761362904776547170376810313996430D+00
      Alpha( 4) = 0.9023059551184773100754483998731103611135D+00
      Alpha( 5) = 2.7287535886135676362583557530427924575633D+00
C
      Xi( 1) = 1.031738582081012241178225030680D+00
      Xi( 2) = 1.301959934154840365511107691532D+00
      Xi( 3) = 1.931949578030785384056416953413D+00
      Xi( 4) = 3.131318808130420078359584801220D+00
      Xi( 5) = 5.350957513524513323426717859022D+00
      Xi( 6) = 9.568335189349001663680627238762D+00
      Xi( 7) = 17.987957016817579344869493240822D+00
      Xi( 8) = 35.937509974005508950334331075283D+00
      Xi( 9) = 77.269299127612637412276797022059D+00
      Xi(10) = 167.164290422268234273528442201950D+00
      GOTO 9999
C
C     ----- K = 05 , R = 3E2 -----
C
  523 CONTINUE
C
      RIni = 3.0D+02
C
      Omega( 1) = 0.0174282312374317400131801909443640674624D+00
      Omega( 2) = 0.0885516446827289778186164705131577790098D+00
      Omega( 3) = 0.3261206213119612461344941356511384356054D+00
      Omega( 4) = 1.0041414563483471091443616218796819339332D+00
      Omega( 5) = 2.9219747649465219435215707077801994273614D+00
C
      Alpha( 1) = 0.0060033834494476276132450510373411711740D+00
      Alpha( 2) = 0.0513810282138952673963618999336544490575D+00
      Alpha( 3) = 0.2369999919944957109688371890698910249284D+00
      Alpha( 4) = 0.8463234388251620139622527494438486428407D+00
      Alpha( 5) = 2.6327559231103213198584012122438480218989D+00
C
      Xi( 1) = 1.033174749276737282299709619338D+00
      Xi( 2) = 1.316546435265119147366326413096D+00
      Xi( 3) = 1.982179306549416015082593078844D+00
      Xi( 4) = 3.264351293528295230471011900697D+00
      Xi( 5) = 5.678771289831076665276937687210D+00
      Xi( 6) = 10.374156421715174907459411368649D+00
      Xi( 7) = 20.046686141095781039794809785803D+00
      Xi( 8) = 41.607779583810231966573001116672D+00
      Xi( 9) = 95.084831518769201169627436343035D+00
      Xi(10) = 234.568627387691309854611532159652D+00
      GOTO 9999
C
C     ----- K = 05 , R = 4E2 -----
C
  524 CONTINUE
C
      RIni = 4.0D+02
C
      Omega( 1) = 0.0150187614602143699419223993318794896368D+00
      Omega( 2) = 0.0822228451037669558342933957828524427214D+00
      Omega( 3) = 0.3132830923567541781904564196503670814309D+00
      Omega( 4) = 0.9837032818773332431801573183527409582894D+00
      Omega( 5) = 2.8973695770360148279195383613782155407534D+00
C
      Alpha( 1) = 0.0050550447094746876118002336473328250932D+00
      Alpha( 2) = 0.0462342261395064916236290718243306940849D+00
      Alpha( 3) = 0.2224845241613079064560594411137373072052D+00
      Alpha( 4) = 0.8150922595119977967221928938368336048370D+00
      Alpha( 5) = 2.5785670834577289001546263058628483122448D+00
C
      Xi( 1) = 1.034032920030843131142458957150D+00
      Xi( 2) = 1.325306119866962213421808514102D+00
      Xi( 3) = 2.012588769819031008683277161886D+00
      Xi( 4) = 3.345847426858509376110598076970D+00
      Xi( 5) = 5.883019761153540387252358634385D+00
      Xi( 6) = 10.887461813180383892254998734472D+00
      Xi( 7) = 21.397302457233739180278431213367D+00
      Xi( 8) = 45.485650878612636598824492395465D+00
      Xi( 9) = 108.077596856357892887712246476895D+00
      Xi(10) = 293.377329734328583604741069734700D+00
      GOTO 9999
C
C     ----- K = 05 , R = 5E2 -----
C
  525 CONTINUE
C
      RIni = 5.0D+02
C
      Omega( 1) = 0.0135371869189180438887112238946497377867D+00
      Omega( 2) = 0.0782451713630320120227627615772103553127D+00
      Omega( 3) = 0.3050398442664156748805118213319076403423D+00
      Omega( 4) = 0.9704412982176837114303528941672283281150D+00
      Omega( 5) = 2.8813614470814399848557635364088014284789D+00
C
      Alpha( 1) = 0.0044700910206532437740403729785343411862D+00
      Alpha( 2) = 0.0430422335189548189549968524453271712105D+00
      Alpha( 3) = 0.2133256975578133257876378356399094116114D+00
      Alpha( 4) = 0.7951322401352172105932353263213485661254D+00
      Alpha( 5) = 2.5436793036816291951971585882930071420560D+00
C
      Xi( 1) = 1.034604842237904686209210014525D+00
      Xi( 2) = 1.331161827039902273081550809763D+00
      Xi( 3) = 2.033020735682391787017830875328D+00
      Xi( 4) = 3.401014082257426986632617427198D+00
      Xi( 5) = 6.022775779578891303719240246650D+00
      Xi( 6) = 11.243704457368707444117617022528D+00
      Xi( 7) = 22.352756423276303403280262926955D+00
      Xi( 8) = 48.306073236663569628324133020669D+00
      Xi( 9) = 117.945145859368349001983222734680D+00
      Xi(10) = 344.399657536421490683364865503790D+00
      GOTO 9999
C
C     ----- K = 05 , R = 6E2 -----
C
  526 CONTINUE
C
      RIni = 6.0D+02
C
      Omega( 1) = 0.0125497022369018914496793782285821716727D+00
      Omega( 2) = 0.0755480277998204573136121521559349645258D+00
      Omega( 3) = 0.2993684653512476888869656838076949156857D+00
      Omega( 4) = 0.9612526221486810956294750962758399737140D+00
      Omega( 5) = 2.8702502162736308354874792403421679409803D+00
C
      Alpha( 1) = 0.0040793511611987108165482531028228629921D+00
      Alpha( 2) = 0.0408997178514718597193682708690332106016D+00
      Alpha( 3) = 0.2071003853755610939370431985984133405054D+00
      Alpha( 4) = 0.7814442554474885124478691911242833612050D+00
      Alpha( 5) = 2.5196339666338295893831950023056265308696D+00
C
      Xi( 1) = 1.035008238780495264021869195048D+00
      Xi( 2) = 1.335300992017198816116940740528D+00
      Xi( 3) = 2.047513221463767489919846620516D+00
      Xi( 4) = 3.440345054969623298608480399530D+00
      Xi( 5) = 6.123156777924488606591746364671D+00
      Xi( 6) = 11.502110566433446770147341897683D+00
      Xi( 7) = 23.055158337444412440866980951171D+00
      Xi( 8) = 50.420991181853802340362413758612D+00
      Xi( 9) = 125.578805960313507290859735832100D+00
      Xi(10) = 388.108156802750548258762464115534D+00
      GOTO 9999
C
C     ----- K = 05 , R = 7E2 -----
C
  527 CONTINUE
C
      RIni = 7.0D+02
C
      Omega( 1) = 0.0118636637624195178692006451964813251720D+00
      Omega( 2) = 0.0736487157459496412766292508522703030849D+00
      Omega( 3) = 0.2953333596223997398124046231959738406658D+00
      Omega( 4) = 0.9546824601007417951544463396995610082740D+00
      Omega( 5) = 2.8622952932595032279547431208754915132886D+00
C
      Alpha( 1) = 0.0038074691411251674333998201932641181067D+00
      Alpha( 2) = 0.0394027284673490241780448913727630255721D+00
      Alpha( 3) = 0.2027096957655570047296704136452483169251D+00
      Alpha( 4) = 0.7717281393047814657369654067764486171654D+00
      Alpha( 5) = 2.5025046733436876863956171979097575786000D+00
C
      Xi( 1) = 1.035300325762522878733813691543D+00
      Xi( 2) = 1.338302645324408399912265166254D+00
      Xi( 3) = 2.058049151069528705498140608299D+00
      Xi( 4) = 3.469043393629417977889681234949D+00
      Xi( 5) = 6.196792486997452185926160694862D+00
      Xi( 6) = 11.693019319216821878121281841345D+00
      Xi( 7) = 23.579165309594961322942063830510D+00
      Xi( 8) = 52.021973375747542937125311723889D+00
      Xi( 9) = 131.493131827179792911475253447406D+00
      Xi(10) = 424.753380875778627501215822803715D+00
      GOTO 9999
C
C     ----- K = 05 , R = 8E2 -----
C
  528 CONTINUE
C
      RIni = 8.0D+02
C
      Omega( 1) = 0.0113813497171839778759997200904829917079D+00
      Omega( 2) = 0.0722993078519351827732693401007413314119D+00
      Omega( 3) = 0.2924451938945003215083486003411650244743D+00
      Omega( 4) = 0.9499630028765620042311909609900055784237D+00
      Omega( 5) = 2.8565759204045159773410755832401264342479D+00
C
      Alpha( 1) = 0.0036161175535254347276672282204476593925D+00
      Alpha( 2) = 0.0383455049806827012912025746255162417242D+00
      Alpha( 3) = 0.1995869766812679076551772378023130727343D+00
      Alpha( 4) = 0.7647855301606951202580964022104126343038D+00
      Alpha( 5) = 2.4902332388884395458772402776048693340272D+00
C
      Xi( 1) = 1.035512044375376673852225117844D+00
      Xi( 2) = 1.340480804514106308159872293917D+00
      Xi( 3) = 2.065708029999885967905107198028D+00
      Xi( 4) = 3.489961455007227071091480130605D+00
      Xi( 5) = 6.250673885791233096772406829800D+00
      Xi( 6) = 11.833441977084792261394607937319D+00
      Xi( 7) = 23.967358721928112862131765581353D+00
      Xi( 8) = 53.220943640742990547731761097339D+00
      Xi( 9) = 135.999940734977351364243247644481D+00
      Xi(10) = 454.425212167193654899666199753483D+00
      GOTO 9999
C
C     ----- K = 05 , R = 9E2 -----
C
  529 CONTINUE
C
      RIni = 9.0D+02
C
      Omega( 1) = 0.0110493996986944845713091178408815640211D+00
      Omega( 2) = 0.0713632054457008080852590571618865311621D+00
      Omega( 3) = 0.2904310341712171869663506457159840579152D+00
      Omega( 4) = 0.9466633544503644491496796575802363804542D+00
      Omega( 5) = 2.8525745409845265157262189648079697690264D+00
C
      Alpha( 1) = 0.0034843237838238312649514669648576070671D+00
      Alpha( 2) = 0.0376153534981393007195370117162758205609D+00
      Alpha( 3) = 0.1974191937931761697681051076491698381687D+00
      Alpha( 4) = 0.7599497569265250643288402143671333988095D+00
      Alpha( 5) = 2.4816698505783679781604156300289787395741D+00
C
      Xi( 1) = 1.035661029320602322481049972769D+00
      Xi( 2) = 1.342014708503715602716348243018D+00
      Xi( 3) = 2.071108717563173735399031927251D+00
      Xi( 4) = 3.504739759714249971306321285169D+00
      Xi( 5) = 6.288846628111508652372863004132D+00
      Xi( 6) = 11.933300415226327029563635484521D+00
      Xi( 7) = 24.244845393727907296652079871535D+00
      Xi( 8) = 54.084778335166424953484032300821D+00
      Xi( 9) = 139.288693645026466591541058903658D+00
      Xi(10) = 477.083147859152222031164924942459D+00
      GOTO 9999
C
C     ----- K = 05 , R = 1E3 -----
C
  531 CONTINUE
C
      RIni = 1.0D+03
C
      Omega( 1) = 0.0108382726570232800350219570101328869782D+00
      Omega( 2) = 0.0707644978086732066545881149333130366585D+00
      Omega( 3) = 0.2891382091047487843294035159136257107093D+00
      Omega( 4) = 0.9445417262916081805245935298920301192993D+00
      Omega( 5) = 2.8500005633404065035450825060792112708441D+00
C
      Alpha( 1) = 0.0034004600489073895373085156109901894261D+00
      Alpha( 2) = 0.0371498167966149987156249893255344218801D+00
      Alpha( 3) = 0.1960321090378673603580109152666466343362D+00
      Alpha( 4) = 0.7568483876518314330643366627526802403736D+00
      Alpha( 5) = 2.4761708329681671751556354399426140844298D+00
C
      Xi( 1) = 1.035757223385584084281510475201D+00
      Xi( 2) = 1.343005822739786468667487484563D+00
      Xi( 3) = 2.074601499880166638077258478567D+00
      Xi( 4) = 3.514309494963886897024238353815D+00
      Xi( 5) = 6.313612662509091697123075892506D+00
      Xi( 6) = 11.998250870322400191830825022521D+00
      Xi( 7) = 24.425972041287355445912665707908D+00
      Xi( 8) = 54.651711438412047962265294387407D+00
      Xi( 9) = 141.466224071874645137891768342797D+00
      Xi(10) = 492.570529270241514563366536094691D+00
      GOTO 9999
C
C     ----- K = 05 , R = 2E3 (Best : R = 1154) -----
C
  599 CONTINUE
C
      RIni = 2.0D+03
C
      Omega( 1) = 0.0107186010624974955309879299722441992060D+00
      Omega( 2) = 0.0704239357038686291254884867807373183268D+00
      Omega( 3) = 0.2884012053308492404702506606284373447124D+00
      Omega( 4) = 0.9433310104981166196980689464535174693083D+00
      Omega( 5) = 2.8485313369244249009568092922961568547180D+00
C
      Alpha( 1) = 0.0033529112764591385087846226072116895267D+00
      Alpha( 2) = 0.0368855238086302728956974734259022241645D+00
      Alpha( 3) = 0.1952428895765193203392729240364111831241D+00
      Alpha( 4) = 0.7550812995393414493814493448908820028009D+00
      Alpha( 5) = 2.4730352459564652644382481971874199189187D+00
C
      Xi( 1) = 1.035812293717382617536798550262D+00
      Xi( 2) = 1.343573108837654108403160402307D+00
      Xi( 3) = 2.076601508202342995451986085165D+00
      Xi( 4) = 3.519794398851761027102041934356D+00
      Xi( 5) = 6.327824256171820903934394086932D+00
      Xi( 6) = 12.035582649906387529316342810404D+00
      Xi( 7) = 24.530303291186305682247681403396D+00
      Xi( 8) = 54.979390713200053542736300826732D+00
      Xi( 9) = 142.731756929721936510979674039845D+00
      Xi(10) = 501.753271542624243922725568722853D+00
      GOTO 9999
C
C     ===== K = 06 =====
C
  600 CONTINUE
C
      GOTO ( 602, 605, 611, 612, 613, 614, 615, 616, 617, 618, 619, 621,
     *       622, 623, 624, 625, 626, 627, 628, 629, 631, 632, 699, 699,
     *       699, 699, 699, 699, 699, 699, 699),InitR
C
C     ----- K = 06 , R = 2E0 -----
C
  602 CONTINUE
C
      RIni = 2.0D+00
C
      Omega( 1) = 0.3951503586666182849006841854011540249303D+00
      Omega( 2) = 0.9483584803953558999261436168737304797105D+00
      Omega( 3) = 1.5813287669193051950466619159918479908811D+00
      Omega( 4) = 2.3797403876283135313977890046288621306303D+00
      Omega( 5) = 3.5481339355732610745496058424208740689210D+00
      Omega( 6) = 5.8817260741536767109684247234469012255431D+00
C
      Alpha( 1) = 0.1534423273269616295463391511533224331743D+00
      Alpha( 2) = 0.8208030144142543005673050282577918324023D+00
      Alpha( 3) = 2.0763266377595740032162285881867092030006D+00
      Alpha( 4) = 4.0375449448252483082913433243277268047677D+00
      Alpha( 5) = 6.9549672561485968930519718789184935303638D+00
      Alpha( 6) = 11.4891014499758782963292902401519768318394D+00
C
      Xi( 1) = 1.002939863002787006869727381986D+00
      Xi( 2) = 1.026434552149436049629967204755D+00
      Xi( 3) = 1.073260845596178922312612102896D+00
      Xi( 4) = 1.142927410284706687547580383768D+00
      Xi( 5) = 1.234275411331628588876806185137D+00
      Xi( 6) = 1.344957165532231756774686481926D+00
      Xi( 7) = 1.470755151352145535001252063534D+00
      Xi( 8) = 1.604854336886718575651451790254D+00
      Xi( 9) = 1.737351219132925049439680753594D+00
      Xi(10) = 1.855486880309094952531782929395D+00
      Xi(11) = 1.945142207156052932765302743157D+00
      Xi(12) = 1.993735287513832045501287193456D+00
      GOTO 9999
C
C     ----- K = 06 , R = 5E0 -----
C
  605 CONTINUE
C
      RIni = 5.0D+00
C
      Omega( 1) = 0.2246903767486849122763520325563035839878D+00
      Omega( 2) = 0.5519095844142891326106030447373740344119D+00
      Omega( 3) = 0.9622019394266135803054848518733876971964D+00
      Omega( 4) = 1.5504627855961601076272698973568253677513D+00
      Omega( 5) = 2.5290208569930264208974746154012791521382D+00
      Omega( 6) = 4.6466141585747064604410005994594712319667D+00
C
      Alpha( 1) = 0.0870216340647555737652257303038627611613D+00
      Alpha( 2) = 0.4709136752047922974001321200221426011012D+00
      Alpha( 3) = 1.2180676458000685170304441817812346471328D+00
      Alpha( 4) = 2.4535517473926842261719438731404352438403D+00
      Alpha( 5) = 4.4452949405703532036703007346289950874052D+00
      Alpha( 6) = 7.8614295814483422927473887931171248055762D+00
C
      Xi( 1) = 1.007037883811018135664719364009D+00
      Xi( 2) = 1.063908019632764205227894172001D+00
      Xi( 3) = 1.180628980475084192761948964634D+00
      Xi( 4) = 1.362952739862027088735621505045D+00
      Xi( 5) = 1.618921217783021904005648361480D+00
      Xi( 6) = 1.957758248114852028462506972062D+00
      Xi( 7) = 2.387184641951110330646002033461D+00
      Xi( 8) = 2.907461913380709053050798829076D+00
      Xi( 9) = 3.500166704810058573979145157828D+00
      Xi(10) = 4.111882968405614833130173879638D+00
      Xi(11) = 4.641862576083718421298546230602D+00
      Xi(12) = 4.957655499604545906288527445938D+00
      GOTO 9999
C
C     ----- K = 06 , R = 1E1 -----
C
  611 CONTINUE
C
      RIni = 1.0D+01
C
      Omega( 1) = 0.1407385838051433856106010866771072187476D+00
      Omega( 2) = 0.3565694200685690354226536902260136230325D+00
      Omega( 3) = 0.6603802616738012804843693803746873527416D+00
      Omega( 4) = 1.1604682915072341640571468301956770119432D+00
      Omega( 5) = 2.0803301182531367932628668659944537466799D+00
      Omega( 6) = 4.1513349861478130481580450794609760123421D+00
C
      Alpha( 1) = 0.0543212598500059367314274534621620915686D+00
      Alpha( 2) = 0.2985187647358117048385951763922463442213D+00
      Alpha( 3) = 0.7962225930216238800108709272596030359637D+00
      Alpha( 4) = 1.6836229524774230601608104862521031463984D+00
      Alpha( 5) = 3.2535534765211933906029623253530758120178D+00
      Alpha( 6) = 6.1993107601849360499893237808066714933375D+00
C
      Xi( 1) = 1.010390027445351043833277060457D+00
      Xi( 2) = 1.094952075027758760716531383217D+00
      Xi( 3) = 1.271782267388700225991052250585D+00
      Xi( 4) = 1.556518752320054307936025850267D+00
      Xi( 5) = 1.973568404102859693400283391806D+00
      Xi( 6) = 2.557355324903913973899269307388D+00
      Xi( 7) = 3.352030882203745386273968653867D+00
      Xi( 8) = 4.404595560535857340627580169823D+00
      Xi( 9) = 5.740477777935007439296527520511D+00
      Xi(10) = 7.301940614229004384433396879217D+00
      Xi(11) = 8.840257599451475464640004542360D+00
      Xi(12) = 9.857134216221037015942529890822D+00
      GOTO 9999
C
C     ----- K = 06 , R = 2E1 -----
C
  612 CONTINUE
C
      RIni = 2.0D+01
C
      Omega( 1) = 0.0862496741180035379067893470950156853405D+00
      Omega( 2) = 0.2287552313838759988283118665397530833161D+00
      Omega( 3) = 0.4619231653435139031303388601479298358754D+00
      Omega( 4) = 0.9029137915898263100529519409320755585213D+00
      Omega( 5) = 1.7807264424526320117014507871111561598809D+00
      Omega( 6) = 3.8200570280772443360669393674911020752916D+00
C
      Alpha( 1) = 0.0331253840720488536691890318203190801682D+00
      Alpha( 2) = 0.1862453189927838743364689119696997465780D+00
      Alpha( 3) = 0.5202262124793207121137110504616174466719D+00
      Alpha( 4) = 1.1783399761432671823966136792627423801605D+00
      Alpha( 5) = 2.4682032832613331014767754645333752705483D+00
      Alpha( 6) = 5.0989823083682598225604976072133922571084D+00
C
      Xi( 1) = 1.013856805828272228940135912811D+00
      Xi( 2) = 1.127427449637189030957028224034D+00
      Xi( 3) = 1.369180650217897420007316655610D+00
      Xi( 4) = 1.769539581176343221585733900270D+00
      Xi( 5) = 2.378972498614929419104985219846D+00
      Xi( 6) = 3.276246661733991947206001693615D+00
      Xi( 7) = 4.578788142114554839817719411599D+00
      Xi( 8) = 6.449052322730253097859964839245D+00
      Xi( 9) = 9.076047734421376751712262631600D+00
      Xi(10) = 12.560202401726437062418062495794D+00
      Xi(11) = 16.536631228932167728476465562437D+00
      Xi(12) = 19.547499375541998470340310234405D+00
      GOTO 9999
C
C     ----- K = 06 , R = 3E1 -----
C
  613 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.0643875392260774473662246139560316748884D+00
      Omega( 2) = 0.1769925533203732905728139748746663428847D+00
      Omega( 3) = 0.3803989314000021870147000280226734503231D+00
      Omega( 4) = 0.7936758834219182629486112612315906744698D+00
      Omega( 5) = 1.6486087888192316119276986663422235324106D+00
      Omega( 6) = 3.6716777532566438266292685455027822172269D+00
C
      Alpha( 1) = 0.0246338601868061877442589273357054224789D+00
      Alpha( 2) = 0.1410255763456897599880624991075528384954D+00
      Alpha( 3) = 0.4082545329312478092306272492439944699072D+00
      Alpha( 4) = 0.9707499275233410204527170761501508877700D+00
      Alpha( 5) = 2.1393343031375326331305214422329186163552D+00
      Alpha( 6) = 4.6296955440744570115831069134770814343938D+00
C
      Xi( 1) = 1.015871521114546384184813831997D+00
      Xi( 2) = 1.146484930769694886125431576573D+00
      Xi( 3) = 1.427344744422456389006462207369D+00
      Xi( 4) = 1.899818296204869565537744213746D+00
      Xi( 5) = 2.634570814368591317594647804867D+00
      Xi( 6) = 3.746933318713922312064718012081D+00
      Xi( 7) = 5.419224037294607949043212524742D+00
      Xi( 8) = 7.927168045245041083472770093721D+00
      Xi( 9) = 11.649874700288855315123692779622D+00
      Xi(10) = 16.955935530593848448166482967281D+00
      Xi(11) = 23.594132485605730422534298540427D+00
      Xi(12) = 29.124355626658757639074748269081D+00
      GOTO 9999
C
C     ----- K = 06 , R = 4E1 -----
C
  614 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.0522574742342978113803029626555085584982D+00
      Omega( 2) = 0.1480796993930270559432010521461187124714D+00
      Omega( 3) = 0.3341395292959524739658146164078544870790D+00
      Omega( 4) = 0.7296921875931181159023568294497152919575D+00
      Omega( 5) = 1.5689274489411984757044130067704656994465D+00
      Omega( 6) = 3.5813517206284184537547210513963591438369D+00
C
      Alpha( 1) = 0.0199272921588992815234223088260989698028D+00
      Alpha( 2) = 0.1158687609653954539857666911073508231311D+00
      Alpha( 3) = 0.3455473250255029791578372660998397236654D+00
      Alpha( 4) = 0.8529186964399603203962341169130922935437D+00
      Alpha( 5) = 1.9493007795284760018481723387040460693242D+00
      Alpha( 6) = 4.3543982689127892935984642797109245293541D+00
C
      Xi( 1) = 1.017270403178994494716724794614D+00
      Xi( 2) = 1.159800883046806106312261175351D+00
      Xi( 3) = 1.468442034643479504585870942801D+00
      Xi( 4) = 1.993273757141346267458999197419D+00
      Xi( 5) = 2.821531744056516985289309595863D+00
      Xi( 6) = 4.099651161191333565564365315481D+00
      Xi( 7) = 6.067370770734152255442006573816D+00
      Xi( 8) = 9.106291550278170173794711761417D+00
      Xi( 9) = 13.788581281739392109426944887929D+00
      Xi(10) = 20.801839405309055643783500144650D+00
      Xi(11) = 30.180838554184981833983236754193D+00
      Xi(12) = 38.605808888195780537860857251076D+00
      GOTO 9999
C
C     ----- K = 06 , R = 5E1 -----
C
  615 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.0444334182386757780728681402337976180661D+00
      Omega( 2) = 0.1293308348039938405240863519840566198127D+00
      Omega( 3) = 0.3036711754764852394777429472894070272559D+00
      Omega( 4) = 0.6863985328085999401193963154188537600930D+00
      Omega( 5) = 1.5138503087235758502553106819021877527121D+00
      Omega( 6) = 3.5185424210031163373864349086517222531256D+00
C
      Alpha( 1) = 0.0168938375522182825927815447500379697487D+00
      Alpha( 2) = 0.0996091777682384279030331809823284316963D+00
      Alpha( 3) = 0.3047652371799653853450158547122583740929D+00
      Alpha( 4) = 0.7753183769485387444244002541626770153016D+00
      Alpha( 5) = 1.8222505520282237026757216069583478201821D+00
      Alpha( 6) = 4.1681458423137244346731722366428130044369D+00
C
      Xi( 1) = 1.018329136868172673938484629197D+00
      Xi( 2) = 1.169925964521422761719336713693D+00
      Xi( 3) = 1.499946922777239079064073856706D+00
      Xi( 4) = 2.065711648691398642004729557797D+00
      Xi( 5) = 2.968537148823861617896041620313D+00
      Xi( 6) = 4.381973458167951779307358606275D+00
      Xi( 7) = 6.597215529572176407994976710114D+00
      Xi( 8) = 10.094351252177452381818267834745D+00
      Xi( 9) = 15.634955098062942901657179639585D+00
      Xi(10) = 24.250794196703078522969754260430D+00
      Xi(11) = 36.390287374282789891605238707939D+00
      Xi(12) = 48.002121522630162350070426313664D+00
      GOTO 9999
C
C     ----- K = 06 , R = 6E1 -----
C
  616 CONTINUE
C
      RIni = 6.0D+01
C
      Omega( 1) = 0.0389207539214614393984598920278594746947D+00
      Omega( 2) = 0.1160606055099792352353874243242604080706D+00
      Omega( 3) = 0.2817827958041292704650070122252003557151D+00
      Omega( 4) = 0.6545864705958214315481384637518402769274D+00
      Omega( 5) = 1.4727168297337244618882418079053309156734D+00
      Omega( 6) = 3.4714354715327135230719390879272623351426D+00
C
      Alpha( 1) = 0.0147577944494537626858251828237955649570D+00
      Alpha( 2) = 0.0881339680501746974836184238266678647733D+00
      Alpha( 3) = 0.2758135968716968830646233540315748200555D+00
      Alpha( 4) = 0.7195968816061252057566730022664103216812D+00
      Alpha( 5) = 1.7298527418488277009002082595934268738347D+00
      Alpha( 6) = 4.0313890355775024482327950625659696015646D+00
C
      Xi( 1) = 1.019173107753838788806267923182D+00
      Xi( 2) = 1.178026728564142044789787855397D+00
      Xi( 3) = 1.525313988635216473003890225169D+00
      Xi( 4) = 2.124542779770625053510635371623D+00
      Xi( 5) = 3.089282425702473901208897077986D+00
      Xi( 6) = 4.617141829792998202527248841065D+00
      Xi( 7) = 7.045958766593616910255820640074D+00
      Xi( 8) = 10.947651740140429289967205495060D+00
      Xi( 9) = 17.267385931674823157724874000962D+00
      Xi(10) = 27.393277391962152302745669985029D+00
      Xi(11) = 42.283685047310163161726759284420D+00
      Xi(12) = 57.320130973726621487901455864744D+00
      GOTO 9999
C
C     ----- K = 06 , R = 7E1 -----
C
  617 CONTINUE
C
      RIni = 7.0D+01
C
      Omega( 1) = 0.0348032175661934841130653456309218540810D+00
      Omega( 2) = 0.1061085857260316455183803488471205866972D+00
      Omega( 3) = 0.2651359722976122245839594032679187307622D+00
      Omega( 4) = 0.6299268945664218586569932378793623684032D+00
      Omega( 5) = 1.4404181621079914719625938990255065164092D+00
      Omega( 6) = 3.4343276793451510164097845168029721207859D+00
C
      Alpha( 1) = 0.0131630672750795466269084499355557227318D+00
      Alpha( 2) = 0.0795507056312876082931700126577023723939D+00
      Alpha( 3) = 0.2540370048152601382225254655899959743692D+00
      Alpha( 4) = 0.6772494438031879543122421027945279092819D+00
      Alpha( 5) = 1.6588625672007166433853317388802395271341D+00
      Alpha( 6) = 3.9254796374566991239932894908193361516169D+00
C
      Xi( 1) = 1.019869801124880972402822476575D+00
      Xi( 2) = 1.184733879548496024566514583931D+00
      Xi( 3) = 1.546426224341374471861997041966D+00
      Xi( 4) = 2.173852206399786536139898118769D+00
      Xi( 5) = 3.191425632925343441491025653356D+00
      Xi( 6) = 4.818387366773663990724557715239D+00
      Xi( 7) = 7.435254134314602479909755983201D+00
      Xi( 8) = 11.699895029075140608261496044662D+00
      Xi( 9) = 18.734606364728364477634237061920D+00
      Xi(10) = 30.288827417441373105952417965625D+00
      Xi(11) = 47.904785539258988878719147663787D+00
      Xi(12) = 66.564805255975383981026372026690D+00
      GOTO 9999
C
C     ----- K = 06 , R = 8E1 -----
C
  618 CONTINUE
C
      RIni = 8.0D+01
C
      Omega( 1) = 0.0315973344518909737188756565312575119719D+00
      Omega( 2) = 0.0983312504597541692870528440284516591419D+00
      Omega( 3) = 0.2519549389039057720135559442775630145661D+00
      Omega( 4) = 0.6100807113831754208426068342152603918294D+00
      Omega( 5) = 1.4141485422756007230568955912453077417013D+00
      Omega( 6) = 3.4040697463532191200091120419912726902112D+00
C
      Alpha( 1) = 0.0119218682970624811615680807216910430668D+00
      Alpha( 2) = 0.0728594690057079250061376415881220225401D+00
      Alpha( 3) = 0.2369701576901333403899060567332490734316D+00
      Alpha( 4) = 0.6437479496493271354895342184487816439287D+00
      Alpha( 5) = 1.6021676951562195459748821901868609529629D+00
      Alpha( 6) = 3.8403261825125701504728137347655092526111D+00
C
      Xi( 1) = 1.020459608471411983455362504358D+00
      Xi( 2) = 1.190426416844526231217198108947D+00
      Xi( 3) = 1.564422761724479715196578499814D+00
      Xi( 4) = 2.216133937261998612189631274738D+00
      Xi( 5) = 3.279697509222747613946355449421D+00
      Xi( 6) = 4.994006773068674855022702763208D+00
      Xi( 7) = 7.778930158075274756569045253585D+00
      Xi( 8) = 12.373095970732255066346083616224D+00
      Xi( 9) = 20.069427291837646079586399139316D+00
      Xi(10) = 32.979406739200383538029859309404D+00
      Xi(11) = 53.286665854063588545347363023552D+00
      Xi(12) = 75.739948898847376648757467876294D+00
      GOTO 9999
C
C     ----- K = 06 , R = 9E1 -----
C
  619 CONTINUE
C
      RIni = 9.0D+01
C
      Omega( 1) = 0.0290224356155021399123123456570416678346D+00
      Omega( 2) = 0.0920629962514437429516337904611411602218D+00
      Omega( 3) = 0.2412003862634107494074828595409520914927D+00
      Omega( 4) = 0.5936581793011396837047265184761357659227D+00
      Omega( 5) = 1.3922177274028809855705227094269105236890D+00
      Omega( 6) = 3.3787565044853692037326448049938676376769D+00
C
      Alpha( 1) = 0.0109252384908035184792304329338963331253D+00
      Alpha( 2) = 0.0674791066323320984753487798701865330031D+00
      Alpha( 3) = 0.2231770678139233854745358820448331016451D+00
      Alpha( 4) = 0.6164401736272702965815928344284202466952D+00
      Alpha( 5) = 1.5555685911402398983321981873473305313382D+00
      Alpha( 6) = 3.7699289128884461914145015404287164528796D+00
C
      Xi( 1) = 1.020968577308356571250493038949D+00
      Xi( 2) = 1.195349326545488175899470406627D+00
      Xi( 3) = 1.580044347500419270107982483431D+00
      Xi( 4) = 2.253022698082180482999706261360D+00
      Xi( 5) = 3.357230602733887736771958842930D+00
      Xi( 6) = 5.149563677934596625258151236793D+00
      Xi( 7) = 8.086405713116187474881479957389D+00
      Xi( 8) = 12.982536846305238532536807660733D+00
      Xi( 9) = 21.295206267189717373386481646946D+00
      Xi(10) = 35.496053516596015268286246069351D+00
      Xi(11) = 58.455305927451702747293937889594D+00
      Xi(12) = 84.848581717880271528542390768735D+00
      GOTO 9999
C
C     ----- K = 06 , R = 1E2 -----
C
  621 CONTINUE
C
      RIni = 1.0D+02
C
      Omega( 1) = 0.0269037503472777864172587897889643571148D+00
      Omega( 2) = 0.0868884034646989601314973940271446650740D+00
      Omega( 3) = 0.2322198468998249781331455129929075553719D+00
      Omega( 4) = 0.5797744666263794617393732833132702353396D+00
      Omega( 5) = 1.3735366550622797052541579354567602422321D+00
      Omega( 6) = 3.3571561544965228026781572934211794745352D+00
C
      Alpha( 1) = 0.0101053613446297862534198233434856728508D+00
      Alpha( 2) = 0.0630473793038213744022494298968606329936D+00
      Alpha( 3) = 0.2117607008763383316597879446607777254030D+00
      Alpha( 4) = 0.5936601207247772421017086019734421142857D+00
      Alpha( 5) = 1.5164074607971299795501282647869345510117D+00
      Alpha( 6) = 3.7104671222364464652429538737976599804824D+00
C
      Xi( 1) = 1.021414424800761986929578672978D+00
      Xi( 2) = 1.199669935580648808329251586713D+00
      Xi( 3) = 1.593799340204969835197747196620D+00
      Xi( 4) = 2.285647994232349912218857368451D+00
      Xi( 5) = 3.426207707105316589745250266397D+00
      Xi( 6) = 5.288978769298995855335093763117D+00
      Xi( 7) = 8.364407269306575645750889691499D+00
      Xi( 8) = 13.539315320810248635501060743280D+00
      Xi( 9) = 22.429276179440487052133001988352D+00
      Xi(10) = 37.862557259468652044037684234468D+00
      Xi(11) = 63.431700995949399989742856220687D+00
      Xi(12) = 93.893166381930095822083703893668D+00
      GOTO 9999
C
C     ----- K = 06 , R = 2E2 -----
C
  622 CONTINUE
C
      RIni = 2.0D+02
C
      Omega( 1) = 0.0164846314068179163341601152839577260067D+00
      Omega( 2) = 0.0611067628119328394039168686480145709083D+00
      Omega( 3) = 0.1855559934147548801965793563439888202993D+00
      Omega( 4) = 0.5047457845186150892030987891168081205251D+00
      Omega( 5) = 1.2702004011589922115310624661788097000681D+00
      Omega( 6) = 3.2370283689529729053650092263083593024930D+00
C
      Alpha( 1) = 0.0060745561605835059879668618321862938458D+00
      Alpha( 2) = 0.0411656892946592426047976772535141876119D+00
      Alpha( 3) = 0.1543149999726211490301509485223441231483D+00
      Alpha( 4) = 0.4758340265306222515633103459231634246862D+00
      Alpha( 5) = 1.3088413484383164279902661553123266457987D+00
      Alpha( 6) = 3.3901114378459397153287718795766636503686D+00
C
      Xi( 1) = 1.024101300602834063355715821242D+00
      Xi( 2) = 1.225872067300112482112275924528D+00
      Xi( 3) = 1.678117393118015734172938924207D+00
      Xi( 4) = 2.488617257603293679265113458321D+00
      Xi( 5) = 3.863957253547394488785160504563D+00
      Xi( 6) = 6.196242623729781210870082297504D+00
      Xi( 7) = 10.229266424355462202985811437372D+00
      Xi( 8) = 17.412774327048138035128888034819D+00
      Xi( 9) = 30.682210829818703335530871711967D+00
      Xi(10) = 56.130245727702952542037539984676D+00
      Xi(11) = 105.433492840975274018600060799145D+00
      Xi(12) = 181.154627443317280025003590537835D+00
      GOTO 9999
C
C     ----- K = 06 , R = 3E2 -----
C
  623 CONTINUE
C
      RIni = 3.0D+02
C
      Omega( 1) = 0.0125220873991649013673126546002112302602D+00
      Omega( 2) = 0.0510449651372189538865904463033063009902D+00
      Omega( 3) = 0.1660665774537122245792068742087366217675D+00
      Omega( 4) = 0.4717249694785838887698715959917450391004D+00
      Omega( 5) = 1.2233401903869242636395744949773245480174D+00
      Omega( 6) = 3.1821812879622630178787223709235831847764D+00
C
      Alpha( 1) = 0.0045403772662585912767843686170532446766D+00
      Alpha( 2) = 0.0327772446201638313597389196712983405746D+00
      Alpha( 3) = 0.1315230758019218048579827493038685304327D+00
      Alpha( 4) = 0.4270458936595012981913849908766067642318D+00
      Alpha( 5) = 1.2198581690530462475104134245640352673945D+00
      Alpha( 6) = 3.2496928841682374872364136297875347736408D+00
C
      Xi( 1) = 1.025454614360495746291576657505D+00
      Xi( 2) = 1.239178295945885844762078542569D+00
      Xi( 3) = 1.721535746309017900212978702257D+00
      Xi( 4) = 2.595149849331321221211632543913D+00
      Xi( 5) = 4.099740166302397086431352768443D+00
      Xi( 6) = 6.701036214670878926602526348688D+00
      Xi( 7) = 11.308385676268067748621859447411D+00
      Xi( 8) = 19.762232804543666981125782378115D+00
      Xi( 9) = 35.988217681413820079466825418990D+00
      Xi(10) = 68.797082696545387298092677497152D+00
      Xi(11) = 138.073633010752780642116377407547D+00
      Xi(12) = 263.244689306263335931701519143644D+00
      GOTO 9999
C
C     ----- K = 06 , R = 4E2 -----
C
  624 CONTINUE
C
      RIni = 4.0D+02
C
      Omega( 1) = 0.0103875843304171820269982020983057235952D+00
      Omega( 2) = 0.0455094300305711951095202511680115620152D+00
      Omega( 3) = 0.1548849634653172308842647431070815855492D+00
      Omega( 4) = 0.4522533544737275122324914045091048819813D+00
      Omega( 5) = 1.1952746785512347834409124791577028190659D+00
      Omega( 6) = 3.1492145201891648162006304145776880432095D+00
C
      Alpha( 1) = 0.0037126126629825554528679758519577047693D+00
      Alpha( 2) = 0.0282275965399766230694815353011173364450D+00
      Alpha( 3) = 0.1188555545634441366689162088154940732920D+00
      Alpha( 4) = 0.3992215234884476781219552615764811775989D+00
      Alpha( 5) = 1.1681141741170859201784035241722392584052D+00
      Alpha( 6) = 3.1670518413818134278395893810298389325908D+00
C
      Xi( 1) = 1.026309589054521672854027036781D+00
      Xi( 2) = 1.247623140493916250176598126131D+00
      Xi( 3) = 1.749302523234483264453001383210D+00
      Xi( 4) = 2.664006702747980768119714922726D+00
      Xi( 5) = 4.254371915215652049692407477011D+00
      Xi( 6) = 7.038230898778517470679855216886D+00
      Xi( 7) = 12.045687217398791613899577956914D+00
      Xi( 8) = 21.412379743666467133214603713043D+00
      Xi( 9) = 39.847404817516257326259765392251D+00
      Xi(10) = 78.447323801851049439315399780526D+00
      Xi(11) = 164.775747948596678463251130608569D+00
      Xi(12) = 340.704993926550859129065429442562D+00
      GOTO 9999
C
C     ----- K = 06 , R = 5E2 -----
C
  625 CONTINUE
C
      RIni = 5.0D+02
C
      Omega( 1) = 0.0090407515643733420463126391163066752910D+00
      Omega( 2) = 0.0419530265693814966586653539964979486854D+00
      Omega( 3) = 0.1474881550985332631474719264352835068621D+00
      Omega( 4) = 0.4391435240680694682146884694207145116707D+00
      Omega( 5) = 1.1761879348950756306226711478402080501837D+00
      Omega( 6) = 3.1267420244356869851905245116796550064464D+00
C
      Alpha( 1) = 0.0031893814523236114041891876456566445874D+00
      Alpha( 2) = 0.0253389996114558638162154828588068511408D+00
      Alpha( 3) = 0.1106604325079244116039264844708611690294D+00
      Alpha( 4) = 0.3808951402331243825578627887340488200607D+00
      Alpha( 5) = 1.1335878719950364830731243115380380004353D+00
      Alpha( 6) = 3.1114724532084870984530861059980111349432D+00
C
      Xi( 1) = 1.026911118547184017645718112632D+00
      Xi( 2) = 1.253582552323086940580104675647D+00
      Xi( 3) = 1.768996911522499009016863580346D+00
      Xi( 4) = 2.713191598845857231215097926658D+00
      Xi( 5) = 4.365907597138108712630372165719D+00
      Xi( 6) = 7.284472214720572345002247427459D+00
      Xi( 7) = 12.592436525112847781754377152197D+00
      Xi( 8) = 22.659479034882555902041811890513D+00
      Xi( 9) = 42.835993321550367429750849623815D+00
      Xi(10) = 86.171248709276722130279146938392D+00
      Xi(11) = 187.263674144445891420196481647054D+00
      Xi(12) = 413.913484907066588913693294671248D+00
      GOTO 9999
C
C     ----- K = 06 , R = 6E2 -----
C
  626 CONTINUE
C
      RIni = 6.0D+02
C
      Omega( 1) = 0.0081091385540135693656027429492005875744D+00
      Omega( 2) = 0.0394541019605209582625016985490962184713D+00
      Omega( 3) = 0.1421765733804508024330824980663301460027D+00
      Omega( 4) = 0.4296107974766591557622998076571008141400D+00
      Omega( 5) = 1.1622091695018335619563054628144982416416D+00
      Omega( 6) = 3.1102558315953168074571855150622923247283D+00
C
      Alpha( 1) = 0.0028268458406437786379807047690629340231D+00
      Alpha( 2) = 0.0233296022544698573519074196944145249688D+00
      Alpha( 3) = 0.1048731918356399272484742561661885673630D+00
      Alpha( 4) = 0.3677786713597671238969513685113099654700D+00
      Alpha( 5) = 1.1086419641112550304095848985319605617406D+00
      Alpha( 6) = 3.0710862053749614941113021959750994938076D+00
C
      Xi( 1) = 1.027362349375534096769004011040D+00
      Xi( 2) = 1.258062845629194640849519981618D+00
      Xi( 3) = 1.783857473947565594691103130209D+00
      Xi( 4) = 2.750495517298796569740856887698D+00
      Xi( 5) = 4.451106009008991984746139713991D+00
      Xi( 6) = 7.474277175325931010536201215544D+00
      Xi( 7) = 13.018669023256232651283847978618D+00
      Xi( 8) = 23.645500520094804292395296307916D+00
      Xi( 9) = 45.242679959784439926856469682548D+00
      Xi(10) = 92.550254538427181048854119893576D+00
      Xi(11) = 206.575346884772450523071185557455D+00
      Xi(12) = 483.170485098160643666620472913564D+00
      GOTO 9999
C
C     ----- K = 06 , R = 7E2 -----
C
  627 CONTINUE
C
      RIni = 7.0D+02
C
      Omega( 1) = 0.0074247323082149151767562551395424064182D+00
      Omega( 2) = 0.0375927111082512853721488907399184142832D+00
      Omega( 3) = 0.1381527530049599144834670053083591767518D+00
      Omega( 4) = 0.4223205987016474642354712382141812554437D+00
      Omega( 5) = 1.1514604701601748779592193638876551631256D+00
      Omega( 6) = 3.0975627662452193883998313728156404067704D+00
C
      Alpha( 1) = 0.0025601001960133667433892277467619158493D+00
      Alpha( 2) = 0.0218457448223638122831329923256848957180D+00
      Alpha( 3) = 0.1005461602165559508416316399082024446443D+00
      Alpha( 4) = 0.3578681828337668482410954123862367737274D+00
      Alpha( 5) = 1.0896558648067596280053018209521553671948D+00
      Alpha( 6) = 3.0402149545555030530838425129047664086102D+00
C
      Xi( 1) = 1.027715621821195243930369134233D+00
      Xi( 2) = 1.261576389565768080143308427221D+00
      Xi( 3) = 1.795544353439286316509275465148D+00
      Xi( 4) = 2.779948400327085405222410363635D+00
      Xi( 5) = 4.518743573786206716835084362671D+00
      Xi( 6) = 7.626014676975904829549918195752D+00
      Xi( 7) = 13.362419624020622798722668367333D+00
      Xi( 8) = 24.449525117501510517240004283224D+00
      Xi( 9) = 47.233724318332957828325824323201D+00
      Xi(10) = 97.935128153593729548320734323852D+00
      Xi(11) = 223.395485833209753306904943315203D+00
      Xi(12) = 548.719796409537993453486848238754D+00
      GOTO 9999
C
C     ----- K = 06 , R = 8E2 -----
C
  628 CONTINUE
C
      RIni = 8.0D+02
C
      Omega( 1) = 0.0069001546543077985549532039726006171776D+00
      Omega( 2) = 0.0361483829711729105784953110236390472210D+00
      Omega( 3) = 0.1349880105474074203292852297542125938890D+00
      Omega( 4) = 0.4165440392673443189099687783416570141526D+00
      Omega( 5) = 1.1429068035001628213156305013953328852949D+00
      Omega( 6) = 3.0874514747660918537620206025806623983954D+00
C
      Alpha( 1) = 0.0023553678339827302485676510657164328677D+00
      Alpha( 2) = 0.0207030056277595794300103474935847991389D+00
      Alpha( 3) = 0.0971788429385239001901753211343226723784D+00
      Alpha( 4) = 0.3500900854010225815668867072050218780532D+00
      Alpha( 5) = 1.0746683097025738517268228511269967384578D+00
      Alpha( 6) = 3.0157614053961606392657152087011240837455D+00
C
      Xi( 1) = 1.028000752156941029246198238134D+00
      Xi( 2) = 1.264415988943018486130066835305D+00
      Xi( 3) = 1.805010490262082701240151094702D+00
      Xi( 4) = 2.803879916986824682220008631006D+00
      Xi( 5) = 4.573942469778034288722251554660D+00
      Xi( 6) = 7.750540580820214526212852401699D+00
      Xi( 7) = 13.646518061899365275614626202838D+00
      Xi( 8) = 25.119963251522863285566944391292D+00
      Xi( 9) = 48.913594408691437643438293392251D+00
      Xi(10) = 102.554530536405437556035646906594D+00
      Xi(11) = 238.202187815886369665063782008474D+00
      Xi(12) = 610.759132377441485761693229505909D+00
      GOTO 9999
C
C     ----- K = 06 , R = 9E2 -----
C
  629 CONTINUE
C
      RIni = 9.0D+02
C
      Omega( 1) = 0.0064853094461266979653356052996915348530D+00
      Omega( 2) = 0.0349935860300550033064764278434299882292D+00
      Omega( 3) = 0.1324294705506935298291179906038372848798D+00
      Omega( 4) = 0.4118457658248256582898295941053135038601D+00
      Omega( 5) = 1.1359254669154330633345367784770019170537D+00
      Omega( 6) = 3.0791919318758857709427551041514448115777D+00
C
      Alpha( 1) = 0.0021932677980895974418018107244751635498D+00
      Alpha( 2) = 0.0197953716360736805643274864585645644866D+00
      Alpha( 3) = 0.0944803091081423382480261770333118676035D+00
      Alpha( 4) = 0.3438129085859721877115654370404840278752D+00
      Alpha( 5) = 1.0625153853630918895922796707509405678138D+00
      Alpha( 6) = 2.9958772123567106030524870297071515778953D+00
C
      Xi( 1) = 1.028236134913692806066090601469D+00
      Xi( 2) = 1.266762739177218750441338002144D+00
      Xi( 3) = 1.812848047722443427004664073809D+00
      Xi( 4) = 2.823744785906744068698301375342D+00
      Xi( 5) = 4.619926215583368029737182380856D+00
      Xi( 6) = 7.854753049594836031115085983600D+00
      Xi( 7) = 13.885655519336573675628909718682D+00
      Xi( 8) = 25.688468482267040883873510104252D+00
      Xi( 9) = 50.352049626865415805909709234811D+00
      Xi(10) = 106.565717689397061546108247398479D+00
      Xi(11) = 251.342760645662371368769250068453D+00
      Xi(12) = 669.448234681487855890491545096666D+00
      GOTO 9999
C
C     ----- K = 06 , R = 1E3 -----
C
  631 CONTINUE
C
      RIni = 1.0D+03
C
      Omega( 1) = 0.0061493505796972659139391947143854499913D+00
      Omega( 2) = 0.0340491288320566085672728810773701724202D+00
      Omega( 3) = 0.1303175043656297746362884848458918440883D+00
      Omega( 4) = 0.4079481557970084787390335073764546791608D+00
      Omega( 5) = 1.1301170634334537682492635712883100040926D+00
      Omega( 6) = 3.0723153218216233070487347633914509970054D+00
C
      Alpha( 1) = 0.0020618552752413614870631087313407325823D+00
      Alpha( 2) = 0.0190573955732373034512451125815912789108D+00
      Alpha( 3) = 0.0922691708416082665949044076814100634465D+00
      Alpha( 4) = 0.3386390121450568758694043419721353416207D+00
      Alpha( 5) = 1.0524587572217026424205646439880013076618D+00
      Alpha( 6) = 2.9793845872747349067324873495010706392350D+00
C
      Xi( 1) = 1.028433820474111892490261255251D+00
      Xi( 2) = 1.268735476408904108095582197180D+00
      Xi( 3) = 1.819446326348997650503361289687D+00
      Xi( 4) = 2.840504832213112635947577233608D+00
      Xi( 5) = 4.658839268464775823768569962269D+00
      Xi( 6) = 7.943279025846940839643284260063D+00
      Xi( 7) = 14.089786680061612422418249046530D+00
      Xi( 8) = 26.176766994187056540599467524544D+00
      Xi( 9) = 51.597819895252218635323115947244D+00
      Xi(10) = 110.081315607349564660022434026132D+00
      Xi(11) = 263.077018330371225407482427272043D+00
      Xi(12) = 724.915711823892740806041246059976D+00
      GOTO 9999
C
C     ----- K = 06 , R = 2E3 -----
C
  632 CONTINUE
C
      RIni = 2.0D+03
C
      Omega( 1) = 0.0046569405523143578406867516827905006949D+00
      Omega( 2) = 0.0297242860906955031989184080193355796951D+00
      Omega( 3) = 0.1204049483744157716992866265226513888820D+00
      Omega( 4) = 0.3894115976155329942069369120183353061293D+00
      Omega( 5) = 1.1022784500339817690328300670010719386482D+00
      Omega( 6) = 3.0392951653690305389380471590143883986457D+00
C
      Alpha( 1) = 0.0014764458510144259402872805191954397319D+00
      Alpha( 2) = 0.0157368046519649566180259299061350297677D+00
      Alpha( 3) = 0.0820975255837409421109419904727388939136D+00
      Alpha( 4) = 0.3144524324480677776084708990866900535366D+00
      Alpha( 5) = 1.0049446929502894142053129322711413351499D+00
      Alpha( 6) = 2.9009766057570200677769717279019801026152D+00
C
      Xi( 1) = 1.029405214778790805764049909321D+00
      Xi( 2) = 1.278453273819321666239338941562D+00
      Xi( 3) = 1.852083750419863797821024009593D+00
      Xi( 4) = 2.923888032492990372152030786879D+00
      Xi( 5) = 4.854025758190635494223469370567D+00
      Xi( 6) = 8.392005549262923054000540279418D+00
      Xi( 7) = 15.138500474576849719188231802036D+00
      Xi( 8) = 28.729264082951998496642809222124D+00
      Xi( 9) = 58.264641065642921217981875514624D+00
      Xi(10) = 129.562709698379522213484982273712D+00
      Xi(11) = 331.833089656248126891879834943211D+00
      Xi(12) = 1116.683301472208890703363692864514D+00
      GOTO 9999
C
C     ----- K = 06 , R = 3E3 (R = 2802.1) -----
C
  699 CONTINUE
C
      RIni = 3.0D+03
C
      Omega( 1) = 0.0044122353526579834677601589087297950798D+00
      Omega( 2) = 0.0289892203219406956332546360446955624468D+00
      Omega( 3) = 0.1186779113929924570088980488591356987627D+00
      Omega( 4) = 0.3861395125485350843393649034052117485771D+00
      Omega( 5) = 1.0973265611814315093758276375979221484158D+00
      Omega( 6) = 3.0334106111925839318731873417434030670847D+00
C
      Alpha( 1) = 0.0013802019452414498485521955422304640759D+00
      Alpha( 2) = 0.0151836143815727678219742818009618812169D+00
      Alpha( 3) = 0.0803615118552532432872931474066025003822D+00
      Alpha( 4) = 0.3102556086327740258025507241734786134657D+00
      Alpha( 5) = 0.9966114862688711845714432946419236714064D+00
      Alpha( 6) = 2.8871396539466742396824738836613732928527D+00
C
      Xi( 1) = 1.029582279599890563067006443720D+00
      Xi( 2) = 1.280228685956502250381169871751D+00
      Xi( 3) = 1.858070579019150707117576104555D+00
      Xi( 4) = 2.939270843402635112993903643286D+00
      Xi( 5) = 4.890324755598177758428996009243D+00
      Xi( 6) = 8.476319446291319014558096611722D+00
      Xi( 7) = 15.338163216545078493605047675885D+00
      Xi( 8) = 29.223576767850096037870688547855D+00
      Xi( 9) = 59.585839834692867395676829644913D+00
      Xi(10) = 133.559268625906095689925834335554D+00
      Xi(11) = 346.736477454457815289279309922676D+00
      Xi(12) = 1218.904339756573323549915244257136D+00
      GOTO 9999
C
C     ===== K = 07 =====
C
  700 CONTINUE
      GOTO ( 702, 705, 711, 712, 713, 714, 715, 716, 717, 718, 719, 721,
     *       722, 723, 724, 725, 726, 727, 728, 729, 731, 732, 733, 734,
     *       735, 736, 799, 799, 799, 799, 799),InitR
C
C     ----- K = 07 , R = 2E0 -----
C
  702 CONTINUE
C
      RIni = 2.0D+00
C
      Omega( 1) = 0.3418302741901420474621587503039066291421D+00
      Omega( 2) = 0.8139373975876501259149251032987137932651D+00
      Omega( 3) = 1.3352973461439932906149458125000251129677D+00
      Omega( 4) = 1.9505008993945933530068936878620888819569D+00
      Omega( 5) = 2.7444403923927759290143230108682814716303D+00
      Omega( 6) = 3.9247848246178932362336622130527530316613D+00
      Omega( 7) = 6.3099568403220470053441304436603331851074D+00
C
      Alpha( 1) = 0.1328537955998787967458766787198776171408D+00
      Alpha( 2) = 0.7079130219116617702703429804156343152499D+00
      Alpha( 3) = 1.7769359317465696517825135547496984145255D+00
      Alpha( 4) = 3.4093982762602818997483417051697074384720D+00
      Alpha( 5) = 5.7364147937170302084600170822170639439719D+00
      Alpha( 6) = 9.0227759271360481047041801616614975500852D+00
      Alpha( 7) = 13.9533183597427484952391285588646496762522D+00
C
      Xi( 1) = 1.002169736045136631320297271319D+00
      Xi( 2) = 1.019513465126059823515937419192D+00
      Xi( 3) = 1.054112462151100426720543024040D+00
      Xi( 4) = 1.105723906980923610765370213027D+00
      Xi( 5) = 1.173819232993989099852853719153D+00
      Xi( 6) = 1.257377121511579858600775316813D+00
      Xi( 7) = 1.354606744408410648819462984704D+00
      Xi( 8) = 1.462620903503482741050099880109D+00
      Xi( 9) = 1.577101049248088766060842125238D+00
      Xi(10) = 1.692071173794360988110359322878D+00
      Xi(11) = 1.799942384597369494621602048490D+00
      Xi(12) = 1.892020132924478879833316002923D+00
      Xi(13) = 1.959581528128465564039632773596D+00
      Xi(14) = 1.995418384645174889589248901078D+00
      GOTO 9999
C
C     ----- K = 07 , R = 5E0 -----
C
  705 CONTINUE
C
      RIni = 5.0D+00
C
      Omega( 1) = 0.1938010079679485214818220051302688489159D+00
      Omega( 2) = 0.4692435586300529009988209000470149590001D+00
      Omega( 3) = 0.7946376596686955935222411440133072346725D+00
      Omega( 4) = 1.2188241997775221336442266761856956236443D+00
      Omega( 5) = 1.8330663504453953155251891282695453355700D+00
      Omega( 6) = 2.8441417219183059963884957488389204627310D+00
      Omega( 7) = 5.0081428000286361285758873762574694410432D+00
C
      Alpha( 1) = 0.0751780120539355767539336714011533047142D+00
      Alpha( 2) = 0.4039405737710808214351466266522905357306D+00
      Alpha( 3) = 1.0300953839123784970457226028983654941840D+00
      Alpha( 4) = 2.0257208844822332206511705798845923709450D+00
      Alpha( 5) = 3.5300416178849783098783743362503173557343D+00
      Alpha( 6) = 5.8202600945046032879064135112656686033006D+00
      Alpha( 7) = 9.5718793022987917473720642647094791755080D+00
C
      Xi( 1) = 1.005229217356591901169279856809D+00
      Xi( 2) = 1.047367901529659821226875271449D+00
      Xi( 3) = 1.133255240513607687907401966321D+00
      Xi( 4) = 1.266031972392625048920260233398D+00
      Xi( 5) = 1.450194606485410606338919137226D+00
      Xi( 6) = 1.691237676145176444431235396326D+00
      Xi( 7) = 1.994883041225001147787529443001D+00
      Xi( 8) = 2.365467028461262406508619493550D+00
      Xi( 9) = 2.802861052159663947814377538137D+00
      Xi(10) = 3.297286131697760815399342315679D+00
      Xi(11) = 3.822053774311960143168082071341D+00
      Xi(12) = 4.326580054007320654766571843552D+00
      Xi(13) = 4.736373451605665986594712046909D+00
      Xi(14) = 4.969367264862313466472487633041D+00
      GOTO 9999
C
C     ----- K = 07 , R = 1E1 -----
C
  711 CONTINUE
C
      RIni = 1.0D+01
C
      Omega( 1) = 0.1209024409395209843302420454902446067535D+00
      Omega( 2) = 0.2992121471107249075992943337931606606617D+00
      Omega( 3) = 0.5287623608132956787406149523622644892384D+00
      Omega( 4) = 0.8652325406126138992264282268074282455927D+00
      Omega( 5) = 1.4096599453909605129337326290261955819005D+00
      Omega( 6) = 2.3721982300004752701287624727655156675610D+00
      Omega( 7) = 4.4850705608595406738946242963805843828595D+00
C
      Alpha( 1) = 0.0467849719196950606124379631878262841838D+00
      Alpha( 2) = 0.2541310917703071038133404851988572659138D+00
      Alpha( 3) = 0.6620206366128168775329887463243494494236D+00
      Alpha( 4) = 1.3468236945914095781815303376482972907979D+00
      Alpha( 5) = 2.4608175652259984735742426309457187016960D+00
      Alpha( 6) = 4.3018718689144129466335464417880984910880D+00
      Alpha( 7) = 7.5591271744403052404090437299544191773748D+00
C
      Xi( 1) = 1.007773548460501324597511640935D+00
      Xi( 2) = 1.070746162242091784034224433153D+00
      Xi( 3) = 1.200884006431706493194314089301D+00
      Xi( 4) = 1.406664527874591639599460846100D+00
      Xi( 5) = 1.701161285541335777644553206578D+00
      Xi( 6) = 2.102581382471403875604465394389D+00
      Xi( 7) = 2.634629620557902439500694291574D+00
      Xi( 8) = 3.325656924246834897970076516671D+00
      Xi( 9) = 4.204512745060396103607275319902D+00
      Xi(10) = 5.289117016413992876551486421732D+00
      Xi(11) = 6.561239747493596864529041834402D+00
      Xi(12) = 7.922448746536713549883795826290D+00
      Xi(13) = 9.146670152521306849935367466742D+00
      Xi(14) = 9.897948626231446764278665284564D+00
      GOTO 9999
C
C     ----- K = 07 , R = 2E1 -----
C
  712 CONTINUE
C
      RIni = 2.0D+01
C
      Omega( 1) = 0.0736409437937410912873993466820632392000D+00
      Omega( 2) = 0.1880960775887743137598211856653662721328D+00
      Omega( 3) = 0.3535856122925518073838498794847495787508D+00
      Omega( 4) = 0.6310263698265568733545768931314512428798D+00
      Omega( 5) = 1.1268723268357688314239561933050026709680D+00
      Omega( 6) = 2.0532076958089364882714678017805454146583D+00
      Omega( 7) = 4.1312146140908651273035423479029759619152D+00
C
      Alpha( 1) = 0.0283978750193578094886787902689784246490D+00
      Alpha( 2) = 0.1566946545197924593886091779171287896588D+00
      Alpha( 3) = 0.4212835366409847733014729098677264573780D+00
      Alpha( 4) = 0.9007103394850066366002054596417991660928D+00
      Alpha( 5) = 1.7551920884002582784293167161848714385997D+00
      Alpha( 6) = 3.2943879648814646311262566147348707090714D+00
      Alpha( 7) = 6.2167521169331160300723648770571116983774D+00
C
      Xi( 1) = 1.010457578611366577406131816996D+00
      Xi( 2) = 1.095616183002617286261666518232D+00
      Xi( 3) = 1.273980560723253206830904438185D+00
      Xi( 4) = 1.562112152636461765136538515364D+00
      Xi( 5) = 1.986628611083233840158349114979D+00
      Xi( 6) = 2.587264263840953612419784612975D+00
      Xi( 7) = 3.421112813296283916997958529649D+00
      Xi( 8) = 4.566870080787266725479839335478D+00
      Xi( 9) = 6.126331365088566714011131253415D+00
      Xi(10) = 8.214734908047492183309767899146D+00
      Xi(11) = 10.916644469830227914992903937019D+00
      Xi(12) = 14.157909730233225860385193328739D+00
      Xi(13) = 17.448466798972224843858325193935D+00
      Xi(14) = 19.682176912149916015240536104614D+00
      GOTO 9999
C
C     ----- K = 07 , R = 3E1 -----
C
  713 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.0547081236603999407457348414657305468722D+00
      Omega( 2) = 0.1431673361803440539338381706779834701138D+00
      Omega( 3) = 0.2818519061350474052586589923663495937944D+00
      Omega( 4) = 0.5330147207850684359367252063144348994683D+00
      Omega( 5) = 1.0042678371684878323210135531162734423560D+00
      Omega( 6) = 1.9102105077080432205633916953324558107852D+00
      Omega( 7) = 3.9707839278055398883095217232863660683506D+00
C
      Alpha( 1) = 0.0210415974965753205954983179326589493030D+00
      Alpha( 2) = 0.1175159106746453374671966020381663753369D+00
      Alpha( 3) = 0.3237851994719231859306673299014889266800D+00
      Alpha( 4) = 0.7182511074334368187353783152815367429866D+00
      Alpha( 5) = 1.4622083631638707608250971059504763616133D+00
      Alpha( 6) = 2.8681290607978677652690341348673541688186D+00
      Alpha( 7) = 5.6391282837140386504162781200477638776647D+00
C
      Xi( 1) = 1.012047889671287567302411514181D+00
      Xi( 2) = 1.110459031424489875165134700197D+00
      Xi( 3) = 1.318193740074288370246052837231D+00
      Xi( 4) = 1.657889794272495439489585600246D+00
      Xi( 5) = 2.166670828411587028600529536426D+00
      Xi( 6) = 2.901774920327571963274415622891D+00
      Xi( 7) = 3.948924790532304569335222965698D+00
      Xi( 8) = 5.432838744548646986443074524686D+00
      Xi( 9) = 7.528256253676515277663966463351D+00
      Xi(10) = 10.462117475713087632119280279142D+00
      Xi(11) = 14.471943652043618301628047362328D+00
      Xi(12) = 19.619191297398738436236209636832D+00
      Xi(13) = 25.270403304102207706419158661504D+00
      Xi(14) = 29.392338201320215115841527619978D+00
      GOTO 9999
C
C     ----- K = 07 , R = 4E1 -----
C
  714 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.0442163443975152241595058272977114555147D+00
      Omega( 2) = 0.1180995323059765671812955563357849797512D+00
      Omega( 3) = 0.2413652023935609491770218348882792724908D+00
      Omega( 4) = 0.4763952040553355030605073294758255997294D+00
      Omega( 5) = 0.9312065231173353566963835081793376957648D+00
      Omega( 6) = 1.8229592256705783305482590495039119105058D+00
      Omega( 7) = 3.8722630501939474274158969890891057730187D+00
C
      Alpha( 1) = 0.0169689443476473593619315970854044550720D+00
      Alpha( 2) = 0.0957454021439034121850595628056002084350D+00
      Alpha( 3) = 0.2692943329972380760823583850660511984643D+00
      Alpha( 4) = 0.6152724295093568839128649050618236060473D+00
      Alpha( 5) = 1.2943422432026802725664113569159496819339D+00
      Alpha( 6) = 2.6197801492130474551502800073521370904928D+00
      Alpha( 7) = 5.2979645334238242792256057267508140284917D+00
C
      Xi( 1) = 1.013167810074629683896214882299D+00
      Xi( 2) = 1.120961719456940277922681048484D+00
      Xi( 3) = 1.349752481091188788231947925933D+00
      Xi( 4) = 1.727074796288967656202398281984D+00
      Xi( 5) = 2.298702839347466489114402143912D+00
      Xi( 6) = 3.136713029630052694714248406882D+00
      Xi( 7) = 4.351843312869864519899787413948D+00
      Xi( 8) = 6.110484393461254443678798420336D+00
      Xi( 9) = 8.656634765265322829265459692039D+00
      Xi(10) = 12.330714532485533930193655205443D+00
      Xi(11) = 17.544287001608629954912954573842D+00
      Xi(12) = 24.563692529334879703292737751497D+00
      Xi(13) = 32.732631020313671822014045531546D+00
      Xi(14) = 39.041770580631060420795375165426D+00
      GOTO 9999
C
C     ----- K = 07 , R = 5E1 -----
C
  715 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.0374556861766770008238325914001753957905D+00
      Omega( 2) = 0.1018584053448467243574568005803904213735D+00
      Omega( 3) = 0.2148511025892223438697476584069434579760D+00
      Omega( 4) = 0.4385126249624096627874065418284388329084D+00
      Omega( 5) = 0.8811012601672848720498788799293521378786D+00
      Omega( 6) = 1.7621025260651733441042560146527762299229D+00
      Omega( 7) = 3.8032676122312763994978007664471419957408D+00
C
      Alpha( 1) = 0.0143466169541190409781331515856606095838D+00
      Alpha( 2) = 0.0816866369651240595850809518563284683523D+00
      Alpha( 3) = 0.2339291825665959799257151430218470977707D+00
      Alpha( 4) = 0.5478173696161150821597358995251880742217D+00
      Alpha( 5) = 1.1828967187208399169549496998676829662145D+00
      Alpha( 6) = 2.4526048384096859368563858794232146465220D+00
      Alpha( 7) = 5.0658409539581052498900815006521725081257D+00
C
      Xi( 1) = 1.014025144133170897075071903437D+00
      Xi( 2) = 1.129030581189734573849606236173D+00
      Xi( 3) = 1.374154955229312968139135375800D+00
      Xi( 4) = 1.781044048069493173345502479155D+00
      Xi( 5) = 2.402854810318760382702521494203D+00
      Xi( 6) = 3.324598199402773961120757806320D+00
      Xi( 7) = 4.679287478364376717723110976621D+00
      Xi( 8) = 6.671367587349018773990039221644D+00
      Xi( 9) = 9.610141382797516037064800009659D+00
      Xi(10) = 13.947791102397180343190619122851D+00
      Xi(11) = 20.279511536792605675208478466232D+00
      Xi(12) = 29.122085171822439941813343011745D+00
      Xi(13) = 39.901986900788082385238553229101D+00
      Xi(14) = 48.637992964763487789559004426110D+00
      GOTO 9999
C
C     ----- K = 07 , R = 6E1 -----
C
  716 CONTINUE
C
      RIni = 6.0D+01
C
      Omega( 1) = 0.0326960774209725745576341198600500703719D+00
      Omega( 2) = 0.0903720874058653475289029123607154048159D+00
      Omega( 3) = 0.1959082814526107491255330766977138523544D+00
      Omega( 4) = 0.4109239117501715364837340527248699117990D+00
      Omega( 5) = 0.8438800600320581920773567452531693788842D+00
      Omega( 6) = 1.7163114229596175923635986704240963263146D+00
      Omega( 7) = 3.7512051733025360371592132890938842137984D+00
C
      Alpha( 1) = 0.0125016247905536379320489532261662102286D+00
      Alpha( 2) = 0.0717712321052866914869341856408890834018D+00
      Alpha( 3) = 0.2088738414319771117188748177362356273079D+00
      Alpha( 4) = 0.4996135480465156085655258066946515782547D+00
      Alpha( 5) = 1.1023091213725187257656268347005834584706D+00
      Alpha( 6) = 2.3303116968910064584104280260845598604647D+00
      Alpha( 7) = 4.8945625765439792422169251739916262522456D+00
C
      Xi( 1) = 1.014715224361727711533710805103D+00
      Xi( 2) = 1.135543811863650034239478503473D+00
      Xi( 3) = 1.393953321295632137338978140395D+00
      Xi( 4) = 1.825135379699663323014614813022D+00
      Xi( 5) = 2.488699823601178217008858539749D+00
      Xi( 6) = 3.481151315361104700606478812297D+00
      Xi( 7) = 4.955626374217314921051535936414D+00
      Xi( 8) = 7.151627151292512403132906051439D+00
      Xi( 9) = 10.440090096247071853492738302549D+00
      Xi(10) = 15.382079466381195343056831337236D+00
      Xi(11) = 22.760534155163923900402900279971D+00
      Xi(12) = 33.373972944163259365946139212156D+00
      Xi(13) = 46.822227620804437678986298365658D+00
      Xi(14) = 58.185982218651912008244853069527D+00
      GOTO 9999
C
C     ----- K = 07 , R = 7E1 -----
C
  717 CONTINUE
C
      RIni = 7.0D+01
C
      Omega( 1) = 0.0291433464945640642622870161715376013944D+00
      Omega( 2) = 0.0817642335427500664324818209494072362986D+00
      Omega( 3) = 0.1815754004803848480574784279850675261514D+00
      Omega( 4) = 0.3896901972802879141066518392522155522784D+00
      Omega( 5) = 0.8147620352906383847462444547371518410728D+00
      Omega( 6) = 1.6801229596135672934549901147427419800806D+00
      Omega( 7) = 3.7099722173278536427321050439331884263083D+00
C
      Alpha( 1) = 0.0111252038160296457427713345475495110293D+00
      Alpha( 2) = 0.0643586014972104123637100525634480163717D+00
      Alpha( 3) = 0.1900644749502537916602696449475118356531D+00
      Alpha( 4) = 0.4631352947176304528224301260141970715267D+00
      Alpha( 5) = 1.0406826317548054407213786731745130964555D+00
      Alpha( 6) = 2.2358669464308131998551976149514075586922D+00
      Alpha( 7) = 4.7613343074885285197833661996469345467631D+00
C
      Xi( 1) = 1.015289737852678177056769959208D+00
      Xi( 2) = 1.140979005247219047109764888948D+00
      Xi( 3) = 1.410544090266070947148682734884D+00
      Xi( 4) = 1.862294490029174263436527458904D+00
      Xi( 5) = 2.561577244847484791058153663634D+00
      Xi( 6) = 3.615255468026502653393697284123D+00
      Xi( 7) = 5.194845325019531162482472330666D+00
      Xi( 8) = 7.572390344480325552467547822744D+00
      Xi( 9) = 11.177153038806722059467224639207D+00
      Xi(10) = 16.675834464013250280425837424048D+00
      Xi(11) = 25.040315400228195095999605968018D+00
      Xi(12) = 37.372988442251295473917060974145D+00
      Xi(13) = 53.524677150646856726295963468942D+00
      Xi(14) = 67.689337530528587344447188911545D+00
      GOTO 9999
C
C     ----- K = 07 , R = 8E1 -----
C
  718 CONTINUE
C
      RIni = 8.0D+01
C
      Omega( 1) = 0.0263787173200564479601888581011825962719D+00
      Omega( 2) = 0.0750421509086382211394091953926555049748D+00
      Omega( 3) = 0.1702797881866766488298753703944932169634D+00
      Omega( 4) = 0.3726996354512291501385416042646525625059D+00
      Omega( 5) = 0.7911419754631662469546680982457331765545D+00
      Omega( 6) = 1.6505212211957283195869206071648704892141D+00
      Omega( 7) = 3.6761865418821602685597177417875514038315D+00
C
      Alpha( 1) = 0.0100546153997157513998472223859526053502D+00
      Alpha( 2) = 0.0585824764836521650425210303025913560759D+00
      Alpha( 3) = 0.1753502950297512436001563448928486366185D+00
      Alpha( 4) = 0.4343856131368536278131945588265594437871D+00
      Alpha( 5) = 0.9916572813188190211371869642320575621852D+00
      Alpha( 6) = 2.1600909301499331841359130557123080507154D+00
      Alpha( 7) = 4.6537869270461901666235959851292136590928D+00
C
      Xi( 1) = 1.015779815962351235580224395605D+00
      Xi( 2) = 1.145624668844172087999247477974D+00
      Xi( 3) = 1.424775149344744374084632820843D+00
      Xi( 4) = 1.894322217755564204230873948109D+00
      Xi( 5) = 2.624780635964467335411592907590D+00
      Xi( 6) = 3.732449332910154737429769822299D+00
      Xi( 7) = 5.405778563515919540698184686534D+00
      Xi( 8) = 7.947205775677921247169416263390D+00
      Xi( 9) = 11.841357539307096187648771223877D+00
      Xi(10) = 17.857273402076895882362173040292D+00
      Xi(11) = 27.155320486195248888897091354977D+00
      Xi(12) = 41.157692449794122908218740874986D+00
      Xi(13) = 60.033087433123451526206793005258D+00
      Xi(14) = 77.150815062068650709847794644247D+00
      GOTO 9999
C
C     ----- K = 07 , R = 9E1 -----
C
  719 CONTINUE
C
      RIni = 9.0D+01
C
      Omega( 1) = 0.0241592217050219513695098142586070721904D+00
      Omega( 2) = 0.0696282194693923318982824166578637914427D+00
      Omega( 3) = 0.1611029161809211525906290404797083226640D+00
      Omega( 4) = 0.3587065401325918283790324808668614764429D+00
      Omega( 5) = 0.7714604530426776657575996254401218266139D+00
      Omega( 6) = 1.6256811668432707961582175948400674769800D+00
      Omega( 7) = 3.6477955931137628568360770087153355234477D+00
C
      Alpha( 1) = 0.0091954804054835872354338476876434160800D+00
      Alpha( 2) = 0.0539396604718906497494984066121137544769D+00
      Alpha( 3) = 0.1634793191158028323581221513682848467397D+00
      Alpha( 4) = 0.4110293908202165037984055634412428048563D+00
      Alpha( 5) = 0.9514935033869456469830805378506966007990D+00
      Alpha( 6) = 2.0975458645195081742486353792997988421121D+00
      Alpha( 7) = 4.5645482583921669532747711350850750022801D+00
C
      Xi( 1) = 1.016205645890911096708084104812D+00
      Xi( 2) = 1.149668216519620178562265111477D+00
      Xi( 3) = 1.437199773265788005512620661985D+00
      Xi( 4) = 1.922401009668877397938702156210D+00
      Xi( 5) = 2.680488469883554993608912320191D+00
      Xi( 6) = 3.836430600122321613997672384855D+00
      Xi( 7) = 5.594389492923190437909802819050D+00
      Xi( 8) = 8.285339363704412511332786372975D+00
      Xi( 9) = 12.446610927164962057470121159497D+00
      Xi(10) = 18.946365628555859900172819720865D+00
      Xi(11) = 29.132014537375160499396709035125D+00
      Xi(12) = 44.757116842193479560457758381631D+00
      Xi(13) = 66.366208781268083836957227816811D+00
      Xi(14) = 86.572612087553766122005605154754D+00
      GOTO 9999
C
C     ----- K = 07 , R = 1E2 -----
C
  721 CONTINUE
C
      RIni = 1.0D+02
C
      Omega( 1) = 0.0223336425511763742141036447053092928172D+00
      Omega( 2) = 0.0651620107304038612404959232571322047534D+00
      Omega( 3) = 0.1534694844777638628207424437155381724551D+00
      Omega( 4) = 0.3469229620284127962143942008887975703146D+00
      Omega( 5) = 0.7547179370389609007045185473661064179396D+00
      Omega( 6) = 1.6044221299260411336745288379823648483580D+00
      Omega( 7) = 3.6234684847661182520439648868659787694924D+00
C
      Alpha( 1) = 0.0084890821340208078627211844661981654525D+00
      Alpha( 2) = 0.0501166420149617110350252335154719318666D+00
      Alpha( 3) = 0.1536699828592926572490588374431119689234D+00
      Alpha( 4) = 0.3916035531845051195505135605001711951445D+00
      Alpha( 5) = 0.9178336466850517870669570741348763931455D+00
      Alpha( 6) = 2.0447798621093103792461381784661966776184D+00
      Alpha( 7) = 4.4889127432163770567950800138845579567715D+00
C
      Xi( 1) = 1.016581033751988272229027643334D+00
      Xi( 2) = 1.153238255900683739696432894561D+00
      Xi( 3) = 1.448198864476133218559539272974D+00
      Xi( 4) = 1.947348932807454913174183275260D+00
      Xi( 5) = 2.730218361459509690547520222026D+00
      Xi( 6) = 3.929796491821510695530150769628D+00
      Xi( 7) = 5.764907707920420065761391015258D+00
      Xi( 8) = 8.593437345984634490037024079356D+00
      Xi( 9) = 13.003022433206446351050544762984D+00
      Xi(10) = 19.957861788850703277375941979699D+00
      Xi(11) = 30.990380197874582535177978570573D+00
      Xi(12) = 48.193880797171424035241216188297D+00
      Xi(13) = 72.539294925666396807173441629857D+00
      Xi(14) = 95.956531905924009606678737327456D+00
      GOTO 9999
C
C     ----- K = 07 , R = 2E2 -----
C
  722 CONTINUE
C
      RIni = 2.0D+02
C
      Omega( 1) = 0.0133615547183825570028305575534521842940D+00
      Omega( 2) = 0.0429728469424360175410925952177443321034D+00
      Omega( 3) = 0.1143029399081515586560726591147663100401D+00
      Omega( 4) = 0.2838881266934189482611071431161775535656D+00
      Omega( 5) = 0.6622322841999484042811198458711174907876D+00
      Omega( 6) = 1.4847175320092703810050463464342840325116D+00
      Omega( 7) = 3.4859753729916252771962870138366952232900D+00
C
      Alpha( 1) = 0.0050213411684266507485648978019454613531D+00
      Alpha( 2) = 0.0312546410994290844202411500801774835168D+00
      Alpha( 3) = 0.1045970270084145620410366606112262388706D+00
      Alpha( 4) = 0.2920522758702768403556507270657505159761D+00
      Alpha( 5) = 0.7407504784499061527671195936939341208927D+00
      Alpha( 6) = 1.7609744335543204401530945069076494746696D+00
      Alpha( 7) = 4.0759036969145123916954953635638503328664D+00
C
      Xi( 1) = 1.018899180639725698428732503231D+00
      Xi( 2) = 1.175398537435454030561192417803D+00
      Xi( 3) = 1.517098188949008791422644326641D+00
      Xi( 4) = 2.105575536480536643453229106449D+00
      Xi( 5) = 3.050761571414857339359580357119D+00
      Xi( 6) = 4.543870834868153870452628373400D+00
      Xi( 7) = 6.913496021851973589392864605330D+00
      Xi( 8) = 10.726771563318656134249085010168D+00
      Xi( 9) = 16.979968567486211656475392217658D+00
      Xi(10) = 27.460101193699478661081059271964D+00
      Xi(11) = 45.403849726057555587915848249736D+00
      Xi(12) = 76.434664319683272064309687010564D+00
      Xi(13) = 127.557313717779402660601473229462D+00
      Xi(14) = 187.964656376434722406809640915526D+00
      GOTO 9999
C
C     ----- K = 07 , R = 3E2 -----
C
  723 CONTINUE
C
      RIni = 3.0D+02
C
      Omega( 1) = 0.0099434776685598156043417164204334390831D+00
      Omega( 2) = 0.0343511106592218869145501525946828635938D+00
      Omega( 3) = 0.0981869859630001312231998428226020791953D+00
      Omega( 4) = 0.2563375496711152665965313679485859665874D+00
      Omega( 5) = 0.6200522686047906112676193768340482392887D+00
      Omega( 6) = 1.4287348681193340442507899723523223656230D+00
      Omega( 7) = 3.4213578343787924831639890577150708850240D+00
C
      Alpha( 1) = 0.0037021290962675962693202485944560242714D+00
      Alpha( 2) = 0.0240203563378520316728596080066426843302D+00
      Alpha( 3) = 0.0852762937444175518725514133222542767498D+00
      Alpha( 4) = 0.2512427085458493746265504209524976886314D+00
      Alpha( 5) = 0.6652246780099562398647777827775939840649D+00
      Alpha( 6) = 1.6360585970149727252963625656612123293598D+00
      Alpha( 7) = 3.8902964899531037119752990838250639171747D+00
C
      Xi( 1) = 1.020114753497002453054565140889D+00
      Xi( 2) = 1.187099191216807305706247566146D+00
      Xi( 3) = 1.553915186630408594393656929178D+00
      Xi( 4) = 2.191512615183995796221821938943D+00
      Xi( 5) = 3.228604039450749353841463307191D+00
      Xi( 6) = 4.893713679900430141896799929313D+00
      Xi( 7) = 7.588583297277757381844170980756D+00
      Xi( 8) = 12.026587110560332955985773928376D+00
      Xi( 9) = 19.505469250220939298259836824911D+00
      Xi(10) = 32.459052347340309092882382202561D+00
      Xi(11) = 55.579138018720356601704013499443D+00
      Xi(12) = 97.897786866246901023302928734893D+00
      Xi(13) = 174.028024411478690586241313553728D+00
      Xi(14) = 277.096563357526212451054092866798D+00
      GOTO 9999
C
C     ----- K = 07 , R = 4E2 -----
C
  724 CONTINUE
C
      RIni = 4.0D+02
C
      Omega( 1) = 0.0080939178056842052675398224009795100642D+00
      Omega( 2) = 0.0296137712027515596681381555149714657205D+00
      Omega( 3) = 0.0889710554315036591940877956641298140994D+00
      Omega( 4) = 0.2400346223587189581734823926040256480974D+00
      Omega( 5) = 0.5945144839955654431193528275390036696990D+00
      Omega( 6) = 1.3943733291175232474326181608681451962184D+00
      Omega( 7) = 3.3815881302823882889520973327179831358080D+00
C
      Alpha( 1) = 0.0029885877621226805003736622535354117147D+00
      Alpha( 2) = 0.0200860577922968259766053076079139749766D+00
      Alpha( 3) = 0.0745569090055890414765402129937399422488D+00
      Alpha( 4) = 0.2279899926003866874278918491780743238451D+00
      Alpha( 5) = 0.6211585820580099310248868571537883553901D+00
      Alpha( 6) = 1.5618502570945419418314142889769868816074D+00
      Alpha( 7) = 3.7787455933745262991359237236821400074405D+00
C
      Xi( 1) = 1.020907027261065597428107976086D+00
      Xi( 2) = 1.194755421073434828059187107208D+00
      Xi( 3) = 1.578171835966652915761096465097D+00
      Xi( 4) = 2.248663631211762458468259029409D+00
      Xi( 5) = 3.348346364020586124520331705590D+00
      Xi( 6) = 5.132934571193718323138532877437D+00
      Xi( 7) = 8.058748380630870102037111024629D+00
      Xi( 8) = 12.951435148119643400721479054027D+00
      Xi( 9) = 21.347969131813360109395372177232D+00
      Xi(10) = 36.215466708309418209649210851353D+00
      Xi(11) = 63.506453953796701269313151883011D+00
      Xi(12) = 115.417602470391644095548677739771D+00
      Xi(13) = 214.691514081141909806027534557415D+00
      Xi(14) = 363.703742836510989511200619972442D+00
      GOTO 9999
C
C     ----- K = 07 , R = 5E2 -----
C
  725 CONTINUE
C
      RIni = 5.0D+02
C
      Omega( 1) = 0.0069203048135253428019736431434094647841D+00
      Omega( 2) = 0.0265678981798603566214964869646175671392D+00
      Omega( 3) = 0.0828637474252807121509402550618705873831D+00
      Omega( 4) = 0.2289794140563879391050928699402611243841D+00
      Omega( 5) = 0.5769324424498350087852326095738675348912D+00
      Omega( 6) = 1.3704988341020230761639620009617601681384D+00
      Omega( 7) = 3.3539045747733188186819053289156045138952D+00
C
      Alpha( 1) = 0.0025357978643651919453790228361350633790D+00
      Alpha( 2) = 0.0175787590913981830039277591895552355084D+00
      Alpha( 3) = 0.0676131686107736120728336354224197179974D+00
      Alpha( 4) = 0.2126283417185967508991617513913752191002D+00
      Alpha( 5) = 0.5915602300563041923654392095777154736425D+00
      Alpha( 6) = 1.5113889410817196977138837654663916509890D+00
      Alpha( 7) = 3.7022956650360443564638096569652248035709D+00
C
      Xi( 1) = 1.021479441200213300524474169251D+00
      Xi( 2) = 1.200302232522448429331840547007D+00
      Xi( 3) = 1.595827735605926870027064523594D+00
      Xi( 4) = 2.290529108175524496300579091468D+00
      Xi( 5) = 3.436812160845001003738052869352D+00
      Xi( 6) = 5.311563852341080136862383476171D+00
      Xi( 7) = 8.414316926197732417549990380223D+00
      Xi( 8) = 13.661402656813166562113637247933D+00
      Xi( 9) = 22.787609388017212220037266412476D+00
      Xi(10) = 39.213126744657628864471066165720D+00
      Xi(11) = 69.999978856728368538375750773639D+00
      Xi(12) = 130.265809858301510423816971240285D+00
      Xi(13) = 250.980726330417617891965775811514D+00
      Xi(14) = 447.983276787880737329627933718257D+00
      GOTO 9999
C
C     ----- K = 07 , R = 6E2 -----
C
  726 CONTINUE
C
      RIni = 6.0D+02
C
      Omega( 1) = 0.0061035796916500113814640741949357760632D+00
      Omega( 2) = 0.0244232192416007003895835926476909349958D+00
      Omega( 3) = 0.0784589868370574932828170715282123381940D+00
      Omega( 4) = 0.2208702419251439254743382689660791129427D+00
      Omega( 5) = 0.5638921330027344504579565886981384892351D+00
      Omega( 6) = 1.3526720019223379766525397482901382772980D+00
      Omega( 7) = 3.3332048728001092039677916156747983222886D+00
C
      Alpha( 1) = 0.0022206030882728451253574190622092077163D+00
      Alpha( 2) = 0.0158270515451072657162057133461985358736D+00
      Alpha( 3) = 0.0626946503162819286951099040372525905696D+00
      Alpha( 4) = 0.2015781143691273810301077540341196936424D+00
      Alpha( 5) = 0.5700007444812679664497730158689847712594D+00
      Alpha( 6) = 1.4742950212653216273198109043107706384035D+00
      Alpha( 7) = 3.6457725385269232113404080930507689117803D+00
C
      Xi( 1) = 1.021919091049763004728188897641D+00
      Xi( 2) = 1.204571103144114048617840939670D+00
      Xi( 3) = 1.609463268497709121816405253291D+00
      Xi( 4) = 2.323015621981505911654644624953D+00
      Xi( 5) = 3.505896984626772737523442424745D+00
      Xi( 6) = 5.452178620159553543850217138811D+00
      Xi( 7) = 8.696906558515406310272410905782D+00
      Xi( 8) = 14.232062660029646540896242079555D+00
      Xi( 9) = 23.960469947688607286750972491518D+00
      Xi(10) = 41.695231511953006958010048421670D+00
      Xi(11) = 75.486978336182950784227774221335D+00
      Xi(12) = 143.151997159489598895887318974474D+00
      Xi(13) = 283.787903272999503967444212548799D+00
      Xi(14) = 530.071360403332262234421534685680D+00
      GOTO 9999
C
C     ----- K = 07 , R = 7E2 -----
C
  727 CONTINUE
C
      RIni = 7.0D+02
C
      Omega( 1) = 0.0054997334786715528114468842270368131864D+00
      Omega( 2) = 0.0228205582831145280442084171540484405227D+00
      Omega( 3) = 0.0751020958906572246355681961305084115565D+00
      Omega( 4) = 0.2146086969873823602510034408430428243264D+00
      Omega( 5) = 0.5537362828757469355414784495383884177500D+00
      Omega( 6) = 1.3387155190228089959641674666634969526058D+00
      Omega( 7) = 3.3169815640530344305267340665466235805070D+00
C
      Alpha( 1) = 0.0019874597606895922284125043360408779947D+00
      Alpha( 2) = 0.0145272077069423383971261110568506946805D+00
      Alpha( 3) = 0.0590012024024202301324961653788969684342D+00
      Alpha( 4) = 0.1931754652858629902690071333815069465345D+00
      Alpha( 5) = 0.5534433535878024911706767052788080718528D+00
      Alpha( 6) = 1.4456024138634961035462106471172205601761D+00
      Alpha( 7) = 3.6018543623381220743304848008747853782552D+00
C
      Xi( 1) = 1.022270812436681717877591524246D+00
      Xi( 2) = 1.207991628120255596184028090700D+00
      Xi( 3) = 1.620418778730794495636022278795D+00
      Xi( 4) = 2.349214970193795601512143367273D+00
      Xi( 5) = 3.561890991607034621154598275083D+00
      Xi( 6) = 5.566867419566254913607106225015D+00
      Xi( 7) = 8.929140258862125879499305192155D+00
      Xi( 8) = 14.705262472528805760968451110671D+00
      Xi( 9) = 24.943550746590106043099055455059D+00
      Xi(10) = 43.803025607890732375654385677421D+00
      Xi(11) = 80.223932633082059941165287142439D+00
      Xi(12) = 154.522575945131116492703426956723D+00
      Xi(13) = 313.725737845372681794620106643379D+00
      Xi(14) = 610.074245598087845920964866763825D+00
      GOTO 9999
C
C     ----- K = 07 , R = 8E2 -----
C
  728 CONTINUE
C
      RIni = 8.0D+02
C
      Omega( 1) = 0.0050337173255694819542215847577720788664D+00
      Omega( 2) = 0.0215715572186778580597268017635115278097D+00
      Omega( 3) = 0.0724425268654193898633223726302166056712D+00
      Omega( 4) = 0.2095952536416926865374746324910404382535D+00
      Omega( 5) = 0.5455484720427803431832061709094716661639D+00
      Omega( 6) = 1.3274159182138670886588402431094380062859D+00
      Omega( 7) = 3.3038349430915769215123811708423318123096D+00
C
      Alpha( 1) = 0.0018074376308293873225900948837306658756D+00
      Alpha( 2) = 0.0135206342248096353230596162325882758637D+00
      Alpha( 3) = 0.0561110636131199456631791448041335357289D+00
      Alpha( 4) = 0.1865310578633183984223259041512754663472D+00
      Alpha( 5) = 0.5402437912418699069187419392612525825825D+00
      Alpha( 6) = 1.4225951349686722217403878065766775762313D+00
      Alpha( 7) = 3.5665103665179134825174989131824077048805D+00
C
      Xi( 1) = 1.022560533027442646698476624856D+00
      Xi( 2) = 1.210812880720948947802471928892D+00
      Xi( 3) = 1.629474819752517520191070765634D+00
      Xi( 4) = 2.370937131634333485058832580172D+00
      Xi( 5) = 3.608506609082108268493627178941D+00
      Xi( 6) = 5.662837814619105219777345228138D+00
      Xi( 7) = 9.124675675335249740854903777176D+00
      Xi( 8) = 15.106626870729963952461050169518D+00
      Xi( 9) = 25.784828768779630973895233836402D+00
      Xi(10) = 45.626411447423677481688431001317D+00
      Xi(11) = 84.378553476106636921372228243854D+00
      Xi(12) = 164.680771008311224751219370432409D+00
      Xi(13) = 341.242194277617065245600258549530D+00
      Xi(14) = 688.080995614329459397762178696212D+00
      GOTO 9999
C
C     ----- K = 07 , R = 9E2 -----
C
  729 CONTINUE
C
      RIni = 9.0D+02
C
      Omega( 1) = 0.0046623733840567663782794506806272938348D+00
      Omega( 2) = 0.0205672519359959761756968419851854257274D+00
      Omega( 3) = 0.0702737396603948621029152137029694102921D+00
      Omega( 4) = 0.2054711368999697145915550242079028464559D+00
      Omega( 5) = 0.5387744815015728043591965168435109490019D+00
      Omega( 6) = 1.3180345908931347003508610660205135900469D+00
      Omega( 7) = 3.2929120497385742784117457215842250661808D+00
C
      Alpha( 1) = 0.0016639061581903090382578440289268462138D+00
      Alpha( 2) = 0.0127159638686803215795124489574663417812D+00
      Alpha( 3) = 0.0537791314332657295004772469299014758803D+00
      Alpha( 4) = 0.1811218120543096948355443406852849363986D+00
      Alpha( 5) = 0.5294244944099308792118813160598733702500D+00
      Alpha( 6) = 1.4036450526560744079661996464025719433266D+00
      Alpha( 7) = 3.5373112546752497732747283487597655948775D+00
C
      Xi( 1) = 1.022804500014966087081264012326D+00
      Xi( 2) = 1.213191167955960221135763843492D+00
      Xi( 3) = 1.637123071334967660292285251966D+00
      Xi( 4) = 2.389329226512886018406883748000D+00
      Xi( 5) = 3.648110078430141154292795646619D+00
      Xi( 6) = 5.744722848990831974829579520403D+00
      Xi( 7) = 9.292377415951423910885076384858D+00
      Xi( 8) = 15.452991735382019830233324420021D+00
      Xi( 9) = 26.516284754150251320309328306735D+00
      Xi(10) = 47.226412734817914820112383367245D+00
      Xi(11) = 88.067292490806068507025461400417D+00
      Xi(12) = 173.844074280575919919100691402036D+00
      Xi(13) = 366.679168982654130809484271935617D+00
      Xi(14) = 764.169197413019728071237324229514D+00
      GOTO 9999
C
C     ----- K = 07 , R = 1E3 -----
C
  731 CONTINUE
C
      RIni = 1.0D+03
C
      Omega( 1) = 0.0043590321081121139760861777322220600439D+00
      Omega( 2) = 0.0197399376432508969001888149419343676527D+00
      Omega( 3) = 0.0684653140928303800778536788995687345505D+00
      Omega( 4) = 0.2020068374139948520101014300953679025952D+00
      Omega( 5) = 0.5330567089683808103500566033527974241224D+00
      Omega( 6) = 1.3100924544159886419789073919162092352053D+00
      Omega( 7) = 3.2836589740207806184697780782499876295333D+00
C
      Alpha( 1) = 0.0015465907100920743574390230079540486940D+00
      Alpha( 2) = 0.0120566431726508442136752479942329863150D+00
      Alpha( 3) = 0.0518524899332544268603262683364762963834D+00
      Alpha( 4) = 0.1766179546713693626684661339143467273516D+00
      Alpha( 5) = 0.5203635254756012724994956863966422133672D+00
      Alpha( 6) = 1.3877091798262134707134599076461256572657D+00
      Alpha( 7) = 3.5126938632453743125566569815987350011710D+00
C
      Xi( 1) = 1.023013513345144816463032355625D+00
      Xi( 2) = 1.215230582716563716714000542574D+00
      Xi( 3) = 1.643691936875600002583922010224D+00
      Xi( 4) = 2.405159331170766592216184176500D+00
      Xi( 5) = 3.682296258249919650379702873444D+00
      Xi( 6) = 5.815666045838884868832141528117D+00
      Xi( 7) = 9.438312805633033496927053462855D+00
      Xi( 8) = 15.755998129717939035959739513260D+00
      Xi( 9) = 27.160318797056292632746687942813D+00
      Xi(10) = 48.646378109863155934922129830511D+00
      Xi(11) = 91.374590421681239627016424265094D+00
      Xi(12) = 182.174592894364896014658405931641D+00
      Xi(13) = 390.305983622066960142538150080327D+00
      Xi(14) = 838.408036003843683214853399476851D+00
      GOTO 9999
C
C     ----- K = 07 , R = 2E3 -----
C
  732 CONTINUE
C
      RIni = 2.0D+03
C
      Omega( 1) = 0.0029076880267603933005398324359788708726D+00
      Omega( 2) = 0.0156597956896735144358535993941439201649D+00
      Omega( 3) = 0.0592095502043018142181824651354427047067D+00
      Omega( 4) = 0.1838875600867601159041754901868337412907D+00
      Omega( 5) = 0.5027200930136521302951831124961046270982D+00
      Omega( 6) = 1.2675787498436361208815903456326168452506D+00
      Omega( 7) = 3.2340325901955293901816196333598441015056D+00
C
      Alpha( 1) = 0.0009839513388777788122618579800399141178D+00
      Alpha( 2) = 0.0088654691035194740209818644628703143695D+00
      Alpha( 3) = 0.0422681213886127304387536549046266998175D+00
      Alpha( 4) = 0.1536724501306380062459018207010608136898D+00
      Alpha( 5) = 0.4733864179916387237624132711122371119927D+00
      Alpha( 6) = 1.3040672497163430146966886513659744650795D+00
      Alpha( 7) = 3.3825029684199556978714540367292329392512D+00
C
      Xi( 1) = 1.024172280366627781934829588550D+00
      Xi( 2) = 1.226568594740137924472590802427D+00
      Xi( 3) = 1.680383653786395965409883324071D+00
      Xi( 4) = 2.494162545601589924502972106524D+00
      Xi( 5) = 3.876214624517233026026324926860D+00
      Xi( 6) = 6.222616866269559330205468627994D+00
      Xi( 7) = 10.286950020887491977564232037423D+00
      Xi( 8) = 17.547328549187459604402627988406D+00
      Xi( 9) = 31.045965513023039257511914712495D+00
      Xi(10) = 57.434448623674729736410249358869D+00
      Xi(11) = 112.539312812692807910797121451196D+00
      Xi(12) = 238.042729684642031182106869380277D+00
      Xi(13) = 560.898336854662458939024816118035D+00
      Xi(14) = 1490.286275794594197607878527378489D+00
      GOTO 9999
C
C     ----- K = 07 , R = 3E3 -----
C
  733 CONTINUE
C
      RIni = 3.0D+03
C
      Omega( 1) = 0.0023891359913667532623401029484230817523D+00
      Omega( 2) = 0.0141270708183685676429170753609643185200D+00
      Omega( 3) = 0.0555607731872176977069453709251467543595D+00
      Omega( 4) = 0.1765507108276267910564353777469470685446D+00
      Omega( 5) = 0.4902176680956567550581301495604691353947D+00
      Omega( 6) = 1.2498655911588724229822205957773917361919D+00
      Omega( 7) = 3.2133062542046689007927545711496009062103D+00
C
      Alpha( 1) = 0.0007821171870305443173034760429886136368D+00
      Alpha( 2) = 0.0077017029989496356004449804759370845630D+00
      Alpha( 3) = 0.0386294252067045345620343969883503376650D+00
      Alpha( 4) = 0.1446819307132389343853841670584081668949D+00
      Alpha( 5) = 0.4545657667811826265153243853411879626947D+00
      Alpha( 6) = 1.2700397189863825259895785402264323238342D+00
      Alpha( 7) = 3.3290409499662179716207577317987897913554D+00
C
      Xi( 1) = 1.024675716539555973241988617861D+00
      Xi( 2) = 1.231511459124680835488033214808D+00
      Xi( 3) = 1.696471376869980616391465899628D+00
      Xi( 4) = 2.533496948444540662224652716183D+00
      Xi( 5) = 3.962846228259305424732397704624D+00
      Xi( 6) = 6.406910222305436564300962221097D+00
      Xi( 7) = 10.677689748845748881310446787296D+00
      Xi( 8) = 18.388813827747888991065439157069D+00
      Xi( 9) = 32.917116563310722682916820147625D+00
      Xi(10) = 61.800599438868931787688376644496D+00
      Xi(11) = 123.497094864236621504505198032575D+00
      Xi(12) = 268.713219652258063802063858815927D+00
      Xi(13) = 663.478795889117814399416062087766D+00
      Xi(14) = 1999.469575084823482424845053628815D+00
      GOTO 9999
C
C     ----- K = 07 , R = 4E3 -----
C
  734 CONTINUE
C
      RIni = 4.0D+03
C
      Omega( 1) = 0.0021345887454745103229132954271440869576D+00
      Omega( 2) = 0.0133516115008142197954869455529813837558D+00
      Omega( 3) = 0.0536724306078645814847285448051517064982D+00
      Omega( 4) = 0.1727066836549376267545205521081186361698D+00
      Omega( 5) = 0.4836135983756697330070324636341894120051D+00
      Omega( 6) = 1.2404615898649592524395288140048876357469D+00
      Omega( 7) = 3.2022901936660388647754027502401186211500D+00
C
      Alpha( 1) = 0.0006828418611695833685875723414321347704D+00
      Alpha( 2) = 0.0071228279773048582995505152888615629792D+00
      Alpha( 3) = 0.0367803190813663724561134293206121625985D+00
      Alpha( 4) = 0.1400427289025622261988942088284382236907D+00
      Alpha( 5) = 0.4447519660204219866823233120456393407949D+00
      Alpha( 6) = 1.2521695310166051146776380886826984806248D+00
      Alpha( 7) = 3.3008429720439399334547259101668714720290D+00
C
      Xi( 1) = 1.024948138888778894684851439312D+00
      Xi( 2) = 1.234190219000572436014451227493D+00
      Xi( 3) = 1.705213827210675740656334320455D+00
      Xi( 4) = 2.554952491376573394006588291205D+00
      Xi( 5) = 4.010342010249804854324362501927D+00
      Xi( 6) = 6.508599720176197863524825137915D+00
      Xi( 7) = 10.895005443340458234878498355869D+00
      Xi( 8) = 18.861337129879108520311148389226D+00
      Xi( 9) = 33.980614487849755446513544399068D+00
      Xi(10) = 64.320822143869004941518774387532D+00
      Xi(11) = 129.955360699156203016513266845777D+00
      Xi(12) = 287.355331978838669043652132017996D+00
      Xi(13) = 728.957088334483430391852465390912D+00
      Xi(14) = 2380.807754337805310029807515093125D+00
      GOTO 9999
C
C     ----- K = 07 , R = 5E3 -----
C
  735 CONTINUE
C
      RIni = 5.0D+03
C
      Omega( 1) = 0.0020013110880352670689172903334754338722D+00
      Omega( 2) = 0.0129378908695508702795281923136894164372D+00
      Omega( 3) = 0.0526524725928330593509372851979133400846D+00
      Omega( 4) = 0.1706164960435682915551678512366962081614D+00
      Omega( 5) = 0.4800066743351757202824987846856430451226D+00
      Omega( 6) = 1.2353112694406893889422990873150354218524D+00
      Omega( 7) = 3.1962532851668917213347476113938228081679D+00
C
      Alpha( 1) = 0.0006308136329875133503749580097738625106D+00
      Alpha( 2) = 0.0068171406204782821707966887665730570944D+00
      Alpha( 3) = 0.0357916075110693214643040830497122684051D+00
      Alpha( 4) = 0.1375409863779743612616484477184641832537D+00
      Alpha( 5) = 0.4394293305525136815502598719440641161782D+00
      Alpha( 6) = 1.2424396843361854057092782954718757082446D+00
      Alpha( 7) = 3.2854539018961136489019314588233555696206D+00
C
      Xi( 1) = 1.025098873577052448636398596093D+00
      Xi( 2) = 1.235673707244880951635658539178D+00
      Xi( 3) = 1.710062509020929812368801148370D+00
      Xi( 4) = 2.566876297161696589841164972690D+00
      Xi( 5) = 4.036811428959137008735946761107D+00
      Xi( 6) = 6.565473574729242831147374426237D+00
      Xi( 7) = 11.017072395598808961375558723716D+00
      Xi( 8) = 19.128174733489920446988241664599D+00
      Xi( 9) = 34.585218590953022612527467671839D+00
      Xi(10) = 65.766092894947603769706390153260D+00
      Xi(11) = 133.702927996013346795267651145878D+00
      Xi(12) = 298.367673115269793904236195203339D+00
      Xi(13) = 768.766113582342625665244639776574D+00
      Xi(14) = 2636.957864008649196518874191497162D+00
      GOTO 9999
C
C     ----- K = 07 , R = 6E3 -----
C
  736 CONTINUE
C
      RIni = 6.0D+03
C
      Omega( 1) = 0.0019443655262837805540929227176606686189D+00
      Omega( 2) = 0.0127592807278033828906170380271256625804D+00
      Omega( 3) = 0.0522093509652323378418623672780896072254D+00
      Omega( 4) = 0.1697052986393442418718751144024459165394D+00
      Omega( 5) = 0.4784306996286163934429570832040923278328D+00
      Omega( 6) = 1.2330577486624195179083438023148744377977D+00
      Omega( 7) = 3.1936109524714979715324936204723371702130D+00
C
      Alpha( 1) = 0.0006085749350024496484217942501512843556D+00
      Alpha( 2) = 0.0066859021591416628164316834654484766887D+00
      Alpha( 3) = 0.0353643071742100135552912699293814924317D+00
      Alpha( 4) = 0.1364550050371723066521120401595190685384D+00
      Alpha( 5) = 0.4371119961322447686495520219329691258281D+00
      Alpha( 6) = 1.2381951172423498714432119061079617949872D+00
      Alpha( 7) = 3.2787324516562921345678299767101293582527D+00
C
      Xi( 1) = 1.025165174452603894690583263838D+00
      Xi( 2) = 1.236326517914779124572392798864D+00
      Xi( 3) = 1.712197846342166541056678541466D+00
      Xi( 4) = 2.572132749436840402955731232915D+00
      Xi( 5) = 4.048496441962004332472402357013D+00
      Xi( 6) = 6.590626762178348379064840578678D+00
      Xi( 7) = 11.071180942174621612827944971968D+00
      Xi( 8) = 19.246778242553667320410837326961D+00
      Xi( 9) = 34.854893906749322175786520716656D+00
      Xi(10) = 66.413622087835886102780147410840D+00
      Xi(11) = 135.392437669718992576672889072142D+00
      Xi(12) = 303.379269717156431346305467400271D+00
      Xi(13) = 787.164708503534546246704906025116D+00
      Xi(14) = 2762.028952486066240679107863797981D+00
      GOTO 9999
C
C     ----- K = 07 , R = 7E3 (Best : R = 6373) -----
C
  799 CONTINUE
C
      RIni = 7.0D+03
C
      Omega( 1) = 0.0019398222440315960905672225964360699813D+00
      Omega( 2) = 0.0127449810742930615739735666740173325451D+00
      Omega( 3) = 0.0521738008587097122689798991274079398295D+00
      Omega( 4) = 0.1696321211629053736074648414677845664755D+00
      Omega( 5) = 0.4783040507603607306979110608269323279274D+00
      Omega( 6) = 1.2328765453712284030614657059921057680185D+00
      Omega( 7) = 3.1933984156189349494676943397308832572890D+00
C
      Alpha( 1) = 0.0006068004605425493834683378191301172944D+00
      Alpha( 2) = 0.0066754145503086015556908419665838039236D+00
      Alpha( 3) = 0.0353300853116583099772316650666326154351D+00
      Alpha( 4) = 0.1363679064445000981291473973744388104024D+00
      Alpha( 5) = 0.4369259718550706375703868156179510151560D+00
      Alpha( 6) = 1.2378541669818243195532250777723959345167D+00
      Alpha( 7) = 3.2781922981409610852194386199087716704526D+00
C
      Xi( 1) = 1.025170524052779595907541887456D+00
      Xi( 2) = 1.236379131356275595647088716422D+00
      Xi( 3) = 1.712369980532019143231250990489D+00
      Xi( 4) = 2.572556450402184307106254190778D+00
      Xi( 5) = 4.049438838950574166830609357248D+00
      Xi( 6) = 6.592655571675112157956077929555D+00
      Xi( 7) = 11.075549856600157824781938131764D+00
      Xi( 8) = 19.256364499899104640590064008165D+00
      Xi( 9) = 34.876717865489795487865531953275D+00
      Xi(10) = 66.466108155508556797064212418036D+00
      Xi(11) = 135.529643829296833873465644160206D+00
      Xi(12) = 303.787584679777685398027742280647D+00
      Xi(13) = 788.671587835625583551291128969751D+00
      Xi(14) = 2772.466801786170626620275925233727D+00
      GOTO 9999
C
C     ===== K = 08 =====
C
  800 CONTINUE
      GOTO ( 811, 811, 811, 812, 813, 814, 815, 816, 817, 818, 819, 821,
     *       822, 823, 824, 825, 826, 827, 828, 829, 831, 832, 833, 834,
     *       835, 836, 837, 838, 839, 841, 899),InitR
C
C     ----- K = 08 , R = 1E1 -----
C
  811 CONTINUE
C
      RIni = 1.0D+01
C
      Omega( 1) = 0.1060264691056738337355419034280235024426D+00
      Omega( 2) = 0.2585290566968715258728672007348947659011D+00
      Omega( 3) = 0.4435891050076717719745974383149800246429D+00
      Omega( 4) = 0.6929754149010114514916497396868066971365D+00
      Omega( 5) = 1.0604512558549935925984730733517835687962D+00
      Omega( 6) = 1.6406263617959378568180936475151554532204D+00
      Omega( 7) = 2.6364875094068331538854632611901251948439D+00
      Omega( 8) = 4.7875207637643221363424328451685596519383D+00
C
      Alpha( 1) = 0.0410961677249340287632430913922210180544D+00
      Alpha( 2) = 0.2215921003939924626704397608856211832062D+00
      Alpha( 3) = 0.5688643520073686326546553992944410538257D+00
      Alpha( 4) = 1.1299764765876945176933943137953519908478D+00
      Alpha( 5) = 1.9937655342879093120283737206044349932199D+00
      Alpha( 6) = 3.3208105201716792307872760936504619166953D+00
      Alpha( 7) = 5.4098231210814420621448006532006047564209D+00
      Alpha( 8) = 8.9485547791067179199120396049238479463384D+00
C
      Xi( 1) = 1.006032718863063354138714933228D+00
      Xi( 2) = 1.054756906560646363473984776782D+00
      Xi( 3) = 1.154677820077761327588646012021D+00
      Xi( 4) = 1.310785006622422077461782852215D+00
      Xi( 5) = 1.530719648521820868422929629027D+00
      Xi( 6) = 1.825017031800220838346458795964D+00
      Xi( 7) = 2.207357900716805350046015310461D+00
      Xi( 8) = 2.694552124162489622127489297121D+00
      Xi( 9) = 3.305727151173377462937269521959D+00
      Xi(10) = 4.059779988418272830916905258292D+00
      Xi(11) = 4.969437980590544347292425442220D+00
      Xi(12) = 6.029429134977404547077184382431D+00
      Xi(13) = 7.196510027041024338753660849477D+00
      Xi(14) = 8.364013108639125808861647382031D+00
      Xi(15) = 9.348328430558432060662465978851D+00
      Xi(16) = 9.923503335592442870655149889103D+00
      GOTO 9999
C
C     ----- K = 08 , R = 2E1 -----
C
  812 CONTINUE
C
      RIni = 2.0D+01
C
      Omega( 1) = 0.0643091646223874215151082057995157725827D+00
      Omega( 2) = 0.1604657999272340102763292901677516510972D+00
      Omega( 3) = 0.2880804447393661645070508936505859765020D+00
      Omega( 4) = 0.4816504189697576558540966645827197112339D+00
      Omega( 5) = 0.7991696608085075711298481127009551983065D+00
      Omega( 6) = 1.3381266056699614649949658939931396162137D+00
      Omega( 7) = 2.3014174412873261213723780072015756559267D+00
      Omega( 8) = 4.4140627077983801873801994464230347148259D+00
C
      Alpha( 1) = 0.0248626641496695616997454235039266556839D+00
      Alpha( 2) = 0.1356048286078502581323201522933707963148D+00
      Alpha( 3) = 0.3561108775384803358553861135638030077644D+00
      Alpha( 4) = 0.7334936098496531953704578865593788350452D+00
      Alpha( 5) = 1.3603355122458833637542896521921420571744D+00
      Alpha( 6) = 2.4047037240593269399879794434582436224446D+00
      Alpha( 7) = 4.1743696392540901437748224189761003799504D+00
      Alpha( 8) = 7.3612293134220797129890767607207635592204D+00
C
      Xi( 1) = 1.008169052232099786869934376909D+00
      Xi( 2) = 1.074413773288232067234723721150D+00
      Xi( 3) = 1.211692604293373886530441052578D+00
      Xi( 4) = 1.429797599193880932363226454740D+00
      Xi( 5) = 1.744176486994071265720224195306D+00
      Xi( 6) = 2.177223636374015249038763042400D+00
      Xi( 7) = 2.760158261984511549134296837238D+00
      Xi( 8) = 3.535169509939452733978879073184D+00
      Xi( 9) = 4.557231078278236897829284179906D+00
      Xi(10) = 5.894169856949790875164096881633D+00
      Xi(11) = 7.621200059709250569910804484763D+00
      Xi(12) = 9.801003964955703424109800803166D+00
      Xi(13) = 12.432003598376196459342202249587D+00
      Xi(14) = 15.343339384741938492481838629544D+00
      Xi(15) = 18.054131542215802208942165840710D+00
      Xi(16) = 19.764723853693879527040899724000D+00
      GOTO 9999
C
C     ----- K = 08 , R = 3E1 -----
C
  813 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.0476193831333583450525004984038579536332D+00
      Omega( 2) = 0.1209066723887145041542424094005436785437D+00
      Omega( 3) = 0.2245974893577788025821138223614426010499D+00
      Omega( 4) = 0.3939685643048650259248535249190226181781D+00
      Omega( 5) = 0.6878199173590028391955635000964974778981D+00
      Omega( 6) = 1.2046845144004612635597536995923917402251D+00
      Omega( 7) = 2.1494044055658569707632776779604455441586D+00
      Omega( 8) = 4.2432459544114843929055180815623771195533D+00
C
      Alpha( 1) = 0.0183751523863023500047613476363062012808D+00
      Alpha( 2) = 0.1010910396253187920628411819445879871182D+00
      Alpha( 3) = 0.2701415651193927010753279005506044541107D+00
      Alpha( 4) = 0.5719241549530381922713216658049617535653D+00
      Alpha( 5) = 1.0991792961155158183869651189468186203158D+00
      Alpha( 6) = 2.0209412748621891147641355113151462319365D+00
      Alpha( 7) = 3.6475369117856662377636506322531317891844D+00
      Alpha( 8) = 6.6737616671756857531297435492945169244194D+00
C
      Xi( 1) = 1.009453069531725518575074329597D+00
      Xi( 2) = 1.086294465453709110424294903563D+00
      Xi( 3) = 1.246517015355910598770235175881D+00
      Xi( 4) = 1.503568330084838722401560651143D+00
      Xi( 5) = 1.878971864677563542462418455514D+00
      Xi( 6) = 2.404690851105510255355385274001D+00
      Xi( 7) = 3.126710352324097630730137531430D+00
      Xi( 8) = 4.109621855206218431471459906490D+00
      Xi( 9) = 5.441960414658121477672614307863D+00
      Xi(10) = 7.241175263151872247384399639003D+00
      Xi(11) = 9.653682038183337798045724476026D+00
      Xi(12) = 12.836502406889897921413168369043D+00
      Xi(13) = 16.886690073653670472395016410871D+00
      Xi(14) = 21.653659797237005603387505203727D+00
      Xi(15) = 26.393779632799863620745695058645D+00
      Xi(16) = 29.554146068708933157292984006403D+00
      GOTO 9999
C
C     ----- K = 08 , R = 4E1 -----
C
  814 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.0383807437367189014720077113590468442794D+00
      Omega( 2) = 0.0988727723491150179601283029540148561409D+00
      Omega( 3) = 0.1888951020665660617106661536834799619555D+00
      Omega( 4) = 0.3438325244242168562481057336732703788584D+00
      Omega( 5) = 0.6224537875740374240698948860028849594528D+00
      Omega( 6) = 1.1241351163372292887559361629357113088190D+00
      Omega( 7) = 2.0558777824169633277693464146196333786065D+00
      Omega( 8) = 4.1376874677588956052270985352947718638461D+00
C
      Alpha( 1) = 0.0147868844915852609137710857271919007161D+00
      Alpha( 2) = 0.0819390877546529709812448902250814342096D+00
      Alpha( 3) = 0.2221894844955850291406572113994677408755D+00
      Alpha( 4) = 0.4811176249313395669314919605996649920598D+00
      Alpha( 5) = 0.9506792305778743405048070158613882085774D+00
      Alpha( 6) = 1.7993973054918034820215747005001105662814D+00
      Alpha( 7) = 3.3386937535433484879054955030142082250677D+00
      Alpha( 8) = 6.2657281268480889825471524812172674501198D+00
C
      Xi( 1) = 1.010366740445417035334195465790D+00
      Xi( 2) = 1.094780045401361425697756013520D+00
      Xi( 3) = 1.271563130120029259440862789976D+00
      Xi( 4) = 1.557138585674537680143304785307D+00
      Xi( 5) = 1.978051847176626979450154986129D+00
      Xi( 6) = 2.574362080522365413834742775556D+00
      Xi( 7) = 3.404866893413130592279877695283D+00
      Xi( 8) = 4.554125907078944354971355990003D+00
      Xi( 9) = 6.141583216364771941834371782321D+00
      Xi(10) = 8.332339656152746481913851095413D+00
      Xi(11) = 11.345203586933031587521969463239D+00
      Xi(12) = 15.441954294911313443246225762806D+00
      Xi(13) = 20.849989559360751762220886895705D+00
      Xi(14) = 27.503663248036671212098935868084D+00
      Xi(15) = 34.457132831177792381405211585843D+00
      Xi(16) = 39.301898615146465043590495724857D+00
      GOTO 9999
C
C     ----- K = 08 , R = 5E1 -----
C
  815 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.0324331424753547827576646931518888905543D+00
      Omega( 2) = 0.0846168152459831565353576103560850896201D+00
      Omega( 3) = 0.1656001458179835813551818504696377942764D+00
      Omega( 4) = 0.3106025687325338813151581751625940341910D+00
      Omega( 5) = 0.5781375786016099380419316910995064517920D+00
      Omega( 6) = 1.0683503216704084423172363416121299906081D+00
      Omega( 7) = 1.9902248502356169701336952027048710078816D+00
      Omega( 8) = 4.0633908158171135622388492336654053360689D+00
C
      Alpha( 1) = 0.0124783628250833338653203201037977798649D+00
      Alpha( 2) = 0.0695851236918722079302017206881014743658D+00
      Alpha( 3) = 0.1911241780349972787915197730557181898803D+00
      Alpha( 4) = 0.4218794617526348468170786093489610379947D+00
      Alpha( 5) = 0.8527570534377367394876338912990121343682D+00
      Alpha( 6) = 1.6513816100170211501139277965322094132716D+00
      Alpha( 7) = 3.1297492657062539102991172157430810329970D+00
      Alpha( 8) = 5.9869906376636193414200737272778951592045D+00
C
      Xi( 1) = 1.011072047923903786492520751050D+00
      Xi( 2) = 1.101348923582572341653591785349D+00
      Xi( 3) = 1.291052883774184086604179533264D+00
      Xi( 4) = 1.599124392338631489031940124956D+00
      Xi( 5) = 2.056410952563171109673187397782D+00
      Xi( 6) = 2.710029243561897579692376858418D+00
      Xi( 7) = 3.630148625993221634918398099678D+00
      Xi( 8) = 4.919394629851516541427930429009D+00
      Xi( 9) = 6.725824433335095013999915813940D+00
      Xi(10) = 9.259909167783606280532904264646D+00
      Xi(11) = 12.811960610130180616451911479459D+00
      Xi(12) = 17.752979007771977057461576521291D+00
      Xi(13) = 24.460071329031597824402166629909D+00
      Xi(14) = 33.001617176073367095068267218494D+00
      Xi(15) = 42.293955496997097776024077120383D+00
      Xi(16) = 49.013661838087606636288340311580D+00
      GOTO 9999
C
C     ----- K = 08 , R = 6E1 -----
C
  816 CONTINUE
C
      RIni = 6.0D+01
C
      Omega( 1) = 0.0282493396917258093250667396606479186971D+00
      Omega( 2) = 0.0745460391371821543519829780799490492882D+00
      Omega( 3) = 0.1490178202155422004619722553542260357062D+00
      Omega( 4) = 0.2866012944928036406517200818111490434603D+00
      Omega( 5) = 0.5455078206488269725515580232944046201737D+00
      Omega( 6) = 1.0265829920889139290117800240587087046151D+00
      Omega( 7) = 1.9405634730900531873887607248185815933539D+00
      Omega( 8) = 4.0070879826910095447729809325920768969809D+00
C
      Alpha( 1) = 0.0108553510974204830253448880728051051392D+00
      Alpha( 2) = 0.0608802488511118427343626023640066335929D+00
      Alpha( 3) = 0.1691520259232484054237079240667895874140D+00
      Alpha( 4) = 0.3797103414106192527887294330124490215894D+00
      Alpha( 5) = 0.7823660354159745778388282411164311724860D+00
      Alpha( 6) = 1.5437686299849278366395910477848474329221D+00
      Alpha( 7) = 2.9762479203500894029289086661904661923472D+00
      Alpha( 8) = 5.7806062770457643908497691320746980636613D+00
C
      Xi( 1) = 1.011643782347894183711901894274D+00
      Xi( 2) = 1.106685789782311103214042236065D+00
      Xi( 3) = 1.306952921678010662580489420481D+00
      Xi( 4) = 1.633572383294769909912666494556D+00
      Xi( 5) = 2.121165649437425079499447422293D+00
      Xi( 6) = 2.823126617453443450850292251175D+00
      Xi( 7) = 3.819881754177334538249630191409D+00
      Xi( 8) = 5.230598170176873665335876006921D+00
      Xi( 9) = 7.230001575562373054145703932249D+00
      Xi(10) = 10.071751936538247290155168034431D+00
      Xi(11) = 14.116078363430355207576438747452D+00
      Xi(12) = 19.844920052593694769979548730987D+00
      Xi(13) = 27.797619290786595202050190955845D+00
      Xi(14) = 38.214659335640939443606844960755D+00
      Xi(15) = 49.936766005697233127696987864397D+00
      Xi(16) = 58.693184247146195215566466529822D+00
      GOTO 9999
C
C     ----- K = 08 , R = 7E1 -----
C
  817 CONTINUE
C
      RIni = 7.0D+01
C
      Omega( 1) = 0.0251286437058089860943941662788092550329D+00
      Omega( 2) = 0.0670064982182152067786084191636231111033D+00
      Omega( 3) = 0.1365152448171116218549048237917986625689D+00
      Omega( 4) = 0.2682605973514779726699867185013559378604D+00
      Omega( 5) = 0.5201604752923396644327148907738234129283D+00
      Omega( 6) = 0.9936935209922501315103066099609918637725D+00
      Omega( 7) = 1.9011380241938253567429012447220770809508D+00
      Omega( 8) = 3.9623285503691931192307706943722678261111D+00
C
      Alpha( 1) = 0.0096453351395891204763473249922389740618D+00
      Alpha( 2) = 0.0543778231378707960550324476689532104956D+00
      Alpha( 3) = 0.1526831785356098382909484367786490111030D+00
      Alpha( 4) = 0.3479126051459739386872722227694865182457D+00
      Alpha( 5) = 0.7288141658794775911475033458586381129862D+00
      Alpha( 6) = 1.4610854083998511351470336872004907036171D+00
      Alpha( 7) = 2.8572596946046264153838689647812998373411D+00
      Alpha( 8) = 5.6195795452967103279437455221057007292984D+00
C
      Xi( 1) = 1.012122710671435961058382824174D+00
      Xi( 2) = 1.111164724011816449752466928746D+00
      Xi( 3) = 1.320342732736776704904023416187D+00
      Xi( 4) = 1.662718595953324588160131269898D+00
      Xi( 5) = 2.176281095847154413739851785969D+00
      Xi( 6) = 2.920088909828480301655861950394D+00
      Xi( 7) = 3.983932103718470257424505431842D+00
      Xi( 8) = 5.502265284245295174150869321750D+00
      Xi( 9) = 7.674822924632206803171458053825D+00
      Xi(10) = 10.796457643348444377373596125835D+00
      Xi(11) = 15.295496764972239377849971830869D+00
      Xi(12) = 21.765117983615827092214178861695D+00
      Xi(13) = 30.915267742492391381606275935212D+00
      Xi(14) = 43.188723957921283339023155178893D+00
      Xi(15) = 57.408826043273802532812544185958D+00
      Xi(16) = 68.343169229978600588693016248953D+00
      GOTO 9999
C
C     ----- K = 08 , R = 8E1 -----
C
  818 CONTINUE
C
      RIni = 8.0D+01
C
      Omega( 1) = 0.0227017559634129721878547086139077393341D+00
      Omega( 2) = 0.0611239009072996963401575801905218909837D+00
      Omega( 3) = 0.1266953565686958380293123532789323348879D+00
      Omega( 4) = 0.2536760883721414111286341502848884488230D+00
      Omega( 5) = 0.4997158187726234234262486649802426086353D+00
      Omega( 6) = 0.9668633389980441755740502018934279249152D+00
      Omega( 7) = 1.8687591398468625841821147415267034830322D+00
      Omega( 8) = 3.9255289517027899676149260699276055675000D+00
C
      Alpha( 1) = 0.0087047417233341318281537391313151807992D+00
      Alpha( 2) = 0.0493145028931534314920142984817186260216D+00
      Alpha( 3) = 0.1398190631110847891754086352866792708483D+00
      Alpha( 4) = 0.3229342398078029359443068962987055670055D+00
      Alpha( 5) = 0.6864049880172785394680246007226287474623D+00
      Alpha( 6) = 1.3950320723685946309967562117826389567199D+00
      Alpha( 7) = 2.7614744811077356516967190591671510446758D+00
      Alpha( 8) = 5.4892323987691283215459292499360799411079D+00
C
      Xi( 1) = 1.012533502490892778542287688825D+00
      Xi( 2) = 1.115012603425935855958398001953D+00
      Xi( 3) = 1.331879579513728790161365689304D+00
      Xi( 4) = 1.687931917130258218902663647221D+00
      Xi( 5) = 2.224201391792045462945937028820D+00
      Xi( 6) = 3.004915982444246826772379921344D+00
      Xi( 7) = 4.128492671460314285096504516304D+00
      Xi( 8) = 5.743620554542309518050724470584D+00
      Xi( 9) = 8.073607998075502993617913460866D+00
      Xi(10) = 11.452690616934524448812526831176D+00
      Xi(11) = 16.375430771618280048618210287259D+00
      Xi(12) = 23.545756770140535349411514687290D+00
      Xi(13) = 33.849897118730002693737235475169D+00
      Xi(14) = 47.957291188064149884545672719582D+00
      Xi(15) = 64.727769266696378293313163254652D+00
      Xi(16) = 77.965680890470509610545857270836D+00
      GOTO 9999
C
C     ----- K = 08 , R = 9E1 -----
C
  819 CONTINUE
C
      RIni = 9.0D+01
C
      Omega( 1) = 0.0207545161695637046829612587170443838147D+00
      Omega( 2) = 0.0563898985798704578030699288704319371845D+00
      Omega( 3) = 0.1187430252961514762452565653616520791047D+00
      Omega( 4) = 0.2417297705831963939448634934303861143690D+00
      Omega( 5) = 0.4827599837096196503359330637339397185315D+00
      Omega( 6) = 0.9443959761885041913733254692608909408591D+00
      Omega( 7) = 1.8414905485795193190246346603977656286588D+00
      Omega( 8) = 3.8945094081684952511731440116804492390656D+00
C
      Alpha( 1) = 0.0079503396990795410480889018368932941172D+00
      Alpha( 2) = 0.0452471530544417981952029159109507361336D+00
      Alpha( 3) = 0.1294552708085345075057727079470559772290D+00
      Alpha( 4) = 0.3027034338865617590615467169534724689584D+00
      Alpha( 5) = 0.6518003973409469030885973750422124339821D+00
      Alpha( 6) = 1.3407130798147101330810085362266192987590D+00
      Alpha( 7) = 2.6821760146617595517177373709571952531405D+00
      Alpha( 8) = 5.3808006207281632595450238110146301551140D+00
C
      Xi( 1) = 1.012892217775423017332367403576D+00
      Xi( 2) = 1.118377380194879730447750365574D+00
      Xi( 3) = 1.341993546337162129746152505749D+00
      Xi( 4) = 1.710112273452325181224975558791D+00
      Xi( 5) = 2.266542972591529318338379739295D+00
      Xi( 6) = 3.080271879844941867532606694091D+00
      Xi( 7) = 4.257723429170776089855432466891D+00
      Xi( 8) = 5.960920914601750644681160684613D+00
      Xi( 9) = 8.435492033735638570633719179881D+00
      Xi(10) = 12.053415046951222512963686206433D+00
      Xi(11) = 17.373660061890040428450587839393D+00
      Xi(12) = 25.209960095336718662734076801257D+00
      Xi(13) = 36.628681045053516098647383714138D+00
      Xi(14) = 52.545874290794805278742574117246D+00
      Xi(15) = 71.907509186043684815126120213336D+00
      Xi(16) = 87.562358890709824886522394393751D+00
      GOTO 9999
C
C     ----- K = 08 , R = 1E2 -----
C
  821 CONTINUE
C
      RIni = 1.0D+02
C
      Omega( 1) = 0.0191536972157081944805691103503320071866D+00
      Omega( 2) = 0.0524874991701344780364337210681879852814D+00
      Omega( 3) = 0.1121483176832860124096298039264585355568D+00
      Omega( 4) = 0.2317178192093658137794016244059847053904D+00
      Omega( 5) = 0.4683927366669551456933306976537778609782D+00
      Omega( 6) = 0.9251988674368323040423864322967517637153D+00
      Omega( 7) = 1.8180761380444023894405528030127072725008D+00
      Omega( 8) = 3.8678537676116576005547892380675989443262D+00
C
      Alpha( 1) = 0.0073303678892273275250013949040639449350D+00
      Alpha( 2) = 0.0418998419489961924599388982948244830595D+00
      Alpha( 3) = 0.1209028306256034907491844591909124773110D+00
      Alpha( 4) = 0.2859242916917437047036147645950876494680D+00
      Alpha( 5) = 0.6229028130797220469665881337206769785553D+00
      Alpha( 6) = 1.2950331295281306881374966399889103740861D+00
      Alpha( 7) = 2.6150915210491833956529816873626259621233D+00
      Alpha( 8) = 5.2886811792799689560008735700336046647863D+00
C
      Xi( 1) = 1.013209889188098188333303495412D+00
      Xi( 2) = 1.121360848910737284967603644148D+00
      Xi( 3) = 1.350981373984641096446315811086D+00
      Xi( 4) = 1.729883192048452760121446392105D+00
      Xi( 5) = 2.304431601657352488878027774177D+00
      Xi( 6) = 3.148024141512270462732459019861D+00
      Xi( 7) = 4.374562374757779956418318745293D+00
      Xi( 8) = 6.158623105037639874209665080684D+00
      Xi( 9) = 8.767045170776013054275699332152D+00
      Xi(10) = 12.608057826417582214517665306630D+00
      Xi(11) = 18.303279924216017092442054803314D+00
      Xi(12) = 26.775037004656948386172032883223D+00
      Xi(13) = 39.272402299385986434304696146569D+00
      Xi(14) = 56.974567404222613137970698815593D+00
      Xi(15) = 78.959365725034720430441925742571D+00
      Xi(16) = 97.134544711358374660670200739787D+00
      GOTO 9999
C
C     ----- K = 08 , R = 2E2 -----
C
  822 CONTINUE
C
      RIni = 2.0D+02
C
      Omega( 1) = 0.0112997080726952972012938495108280712032D+00
      Omega( 2) = 0.0331538459933014433800801089302634494516D+00
      Omega( 3) = 0.0786965757234113262867832017111169307100D+00
      Omega( 4) = 0.1789636493609061990991233523362602397810D+00
      Omega( 5) = 0.3899113717033011928972483645106095195842D+00
      Omega( 6) = 0.8174921432572277218628342687622989615193D+00
      Omega( 7) = 1.6846079041638185670769559121318081906793D+00
      Omega( 8) = 3.7155370264539169717667166370489439941593D+00
C
      Alpha( 1) = 0.0042923690505157349716684159375568174966D+00
      Alpha( 2) = 0.0254154224653902906883522856348878171673D+00
      Alpha( 3) = 0.0783393362576383196028756081275012945753D+00
      Alpha( 4) = 0.2007925992125950392100863436950497487032D+00
      Alpha( 5) = 0.4726395102313897322930495714654597350091D+00
      Alpha( 6) = 1.0517208224751955088736854460051972637302D+00
      Alpha( 7) = 2.2505781328076893770505362279976679928950D+00
      Alpha( 8) = 4.7811047016943063726022999926357215372263D+00
C
      Xi( 1) = 1.015206143343586549059043289134D+00
      Xi( 2) = 1.140189268684429005460030659425D+00
      Xi( 3) = 1.408141644285239425229624976588D+00
      Xi( 4) = 1.856952607098087805991004639594D+00
      Xi( 5) = 2.551250894168072993421075844722D+00
      Xi( 6) = 3.596788903236659127758342857639D+00
      Xi( 7) = 5.163755900602811575207662375675D+00
      Xi( 8) = 7.524095349419460813152754630195D+00
      Xi( 9) = 11.114903730582899794333828769055D+00
      Xi(10) = 16.646935196204675398712335976370D+00
      Xi(11) = 25.288912071546789949588740498143D+00
      Xi(12) = 38.968652370577076770458146981468D+00
      Xi(13) = 60.779458366592280044171658914820D+00
      Xi(14) = 95.031428293897582691707182078744D+00
      Xi(15) = 143.996271130142976066301763182764D+00
      Xi(16) = 191.696182955146930640411007118473D+00
      GOTO 9999
C
C     ----- K = 08 , R = 3E2 -----
C
  823 CONTINUE
C
      RIni = 3.0D+02
C
      Omega( 1) = 0.0083147700732372714089635679264589374071D+00
      Omega( 2) = 0.0256810205584535582997796412940694210647D+00
      Omega( 3) = 0.0651854465211138051674568880122784264586D+00
      Omega( 4) = 0.1563410800780707693744897915366642848767D+00
      Omega( 5) = 0.3545343631542285954061973635154103590139D+00
      Omega( 6) = 0.7671776007943373896016958646271177713061D+00
      Omega( 7) = 1.6209236463502489251084850985229479647387D+00
      Omega( 8) = 3.6426182431755443347221046224149176850915D+00
C
      Alpha( 1) = 0.0031400004160420679633702276687193255533D+00
      Alpha( 2) = 0.0191108173471338873873208484902019854701D+00
      Alpha( 3) = 0.0617360444099322960463064463576543516865D+00
      Alpha( 4) = 0.1664281836158163361329706889102375555467D+00
      Alpha( 5) = 0.4095542480247392515540095364601214100730D+00
      Alpha( 6) = 0.9458455055554492088413544303993063522285D+00
      Alpha( 7) = 2.0873834230907897340206308411936220181815D+00
      Alpha( 8) = 4.5494170928592982520478893437854139847332D+00
C
      Xi( 1) = 1.016281976238305648414274318103D+00
      Xi( 2) = 1.150394802721959229985931871809D+00
      Xi( 3) = 1.439444124816771421095790151057D+00
      Xi( 4) = 1.927518194641683521887640107018D+00
      Xi( 5) = 2.690794755644557892425641765222D+00
      Xi( 6) = 3.856167060340970678287628170544D+00
      Xi( 7) = 5.631827990273911388723754312124D+00
      Xi( 8) = 8.358023927568324704627522603317D+00
      Xi( 9) = 12.596532565686797791246198663373D+00
      Xi(10) = 19.290531183175411522254250051844D+00
      Xi(11) = 30.052574545636373200968582963810D+00
      Xi(12) = 47.682345416800974432364190747080D+00
      Xi(13) = 77.024725923281626246974518323896D+00
      Xi(14) = 125.858082461702271405423747552277D+00
      Xi(15) = 201.808154923244766287826479356227D+00
      Xi(16) = 284.486122795369502369577929812294D+00
      GOTO 9999
C
C     ----- K = 08 , R = 4E2 -----
C
  824 CONTINUE
C
      RIni = 4.0D+02
C
      Omega( 1) = 0.0067008875889055753375121758999566345949D+00
      Omega( 2) = 0.0215908496440356932643997141599290223724D+00
      Omega( 3) = 0.0575423155787227755035507983233378226373D+00
      Omega( 4) = 0.1430624602634638181891755387420417378053D+00
      Omega( 5) = 0.3331842059183278898014217428436012369275D+00
      Omega( 6) = 0.7362076092535837336188440926942888609119D+00
      Omega( 7) = 1.5812540713021125379821388290402239817922D+00
      Omega( 8) = 3.5971080384364662310345273077416550222551D+00
C
      Alpha( 1) = 0.0025176525567014111136737313000402704866D+00
      Alpha( 2) = 0.0156870855266922170677206492539479398829D+00
      Alpha( 3) = 0.0525805310064843255993228491457225359795D+00
      Alpha( 4) = 0.1470151831911213632464539152744364969294D+00
      Alpha( 5) = 0.3730161746691587561534312472710794850173D+00
      Alpha( 6) = 0.8831960386953907907940479471431416413907D+00
      Alpha( 7) = 1.9892072955102488654183928717955609499768D+00
      Alpha( 8) = 4.4084930703152622123892900862784927085158D+00
C
      Xi( 1) = 1.016997180607474292844683017023D+00
      Xi( 2) = 1.157202679675031171914120764477D+00
      Xi( 3) = 1.460451278101122088329093251513D+00
      Xi( 4) = 1.975265489965839091498177315653D+00
      Xi( 5) = 2.786220940640250699543364887312D+00
      Xi( 6) = 4.035887149943814944733078142747D+00
      Xi( 7) = 5.961190076668444562304344236381D+00
      Xi( 8) = 8.955253699668341887431266545860D+00
      Xi( 9) = 13.678943432092563078222857697597D+00
      Xi(10) = 21.265716999283591331568188387280D+00
      Xi(11) = 33.703880926196548187961932185885D+00
      Xi(12) = 54.561382722672526549012594898613D+00
      Xi(13) = 90.310496138781121207295132791160D+00
      Xi(14) = 152.230375307364352008288044260098D+00
      Xi(15) = 254.493575469728197963048543783771D+00
      Xi(16) = 375.770376438034457905468244121039D+00
      GOTO 9999
C
C     ----- K = 08 , R = 5E2 -----
C
  825 CONTINUE
C
      RIni = 5.0D+02
C
      Omega( 1) = 0.0056765954727616951858215415486347277252D+00
      Omega( 2) = 0.0189684285531276371907465202668041648337D+00
      Omega( 3) = 0.0525102584623810031392329537858199017819D+00
      Omega( 4) = 0.1340894633572563902608812402839078004035D+00
      Omega( 5) = 0.3184836575206165450461961302641711313299D+00
      Omega( 6) = 0.7145967548360081613746663808939274531440D+00
      Omega( 7) = 1.5533459981559690892558359132102907551598D+00
      Omega( 8) = 3.5650469439391760267422937236148072770447D+00
C
      Alpha( 1) = 0.0021229535274735288473389613556410737694D+00
      Alpha( 2) = 0.0135064199433631837297302771827567369201D+00
      Alpha( 3) = 0.0466737186745842354107353684017356343361D+00
      Alpha( 4) = 0.1342539621944193918437818191513599686004D+00
      Alpha( 5) = 0.3485587576491086894984929717145050176441D+00
      Alpha( 6) = 0.8406244096896624128523872743201650337141D+00
      Alpha( 7) = 1.9217275963995139784722121500948333050474D+00
      Alpha( 8) = 4.3108989561714496580382971480105425143847D+00
C
      Xi( 1) = 1.017522457222688383104633458309D+00
      Xi( 2) = 1.162214480881127272250859239211D+00
      Xi( 3) = 1.475981656325268065888672852637D+00
      Xi( 4) = 2.010766171804123481721068844585D+00
      Xi( 5) = 2.857699018743528042614532314047D+00
      Xi( 6) = 4.171748635930965416057647487946D+00
      Xi( 7) = 6.212887708931987642158728535335D+00
      Xi( 8) = 9.417362631740984755743251177762D+00
      Xi( 9) = 14.528380793754103720953618772427D+00
      Xi(10) = 22.840921073627481002743522253695D+00
      Xi(11) = 36.670066045180380594897817658762D+00
      Xi(12) = 60.271227894864704381638897956464D+00
      Xi(13) = 101.628326632819418605391259546877D+00
      Xi(14) = 175.457508957763793452921952109591D+00
      Xi(15) = 303.185783856004199443567870275729D+00
      Xi(16) = 465.696952975465335022864366010253D+00
      GOTO 9999
C
C     ----- K = 08 , R = 6E2 -----
C
  826 CONTINUE
C
      RIni = 6.0D+02
C
      Omega( 1) = 0.0049631227560872719340512640700396884696D+00
      Omega( 2) = 0.0171254524636490031161900138637066426384D+00
      Omega( 3) = 0.0488948450803400766644427576090858877933D+00
      Omega( 4) = 0.1275139287006728620876838675712861004286D+00
      Omega( 5) = 0.3075600247602412360640317112725483639224D+00
      Omega( 6) = 0.6983780043252394544375794005475199810462D+00
      Omega( 7) = 1.5322730134232326378723731963304999226239D+00
      Omega( 8) = 3.5408125259521600184699219315831442145281D+00
C
      Alpha( 1) = 0.0018481578123667529228029920789454221275D+00
      Alpha( 2) = 0.0119828198121361834915281287423927114588D+00
      Alpha( 3) = 0.0425003466075095890485270166275766356989D+00
      Alpha( 4) = 0.1250994074781486592218325906711307027308D+00
      Alpha( 5) = 0.3307649289127311088907872071995797114141D+00
      Alpha( 6) = 0.8092949962970130067330468137676291462412D+00
      Alpha( 7) = 1.8716385878240684829953988099759953911416D+00
      Alpha( 8) = 4.2380476718029878352607398017681816781987D+00
C
      Xi( 1) = 1.017931737489983994677730794187D+00
      Xi( 2) = 1.166126687088432865944259697599D+00
      Xi( 3) = 1.488142831743968411421959185947D+00
      Xi( 4) = 2.038685505878753312109658013540D+00
      Xi( 5) = 2.914229161373346585223870208736D+00
      Xi( 6) = 4.279952233877284060482859562313D+00
      Xi( 7) = 6.415003888346393525442257210756D+00
      Xi( 8) = 9.791979041278517329784103484513D+00
      Xi( 9) = 15.224499415759320633187434701483D+00
      Xi(10) = 24.147941963990778684034932410896D+00
      Xi(11) = 39.166863288595609133430075488036D+00
      Xi(12) = 65.159262885039493602701288210710D+00
      Xi(13) = 111.517879335856661342230022881949D+00
      Xi(14) = 196.296994258655902676480309310136D+00
      Xi(15) = 348.607063705276868759153074961432D+00
      Xi(16) = 554.362393276937058528286428327192D+00
      GOTO 9999
C
C     ----- K = 08 , R = 7E2 -----
C
  827 CONTINUE
C
      RIni = 7.0D+02
C
      Omega( 1) = 0.0044348751043990017105410794857889289489D+00
      Omega( 2) = 0.0157499318410671428728766184368392799797D+00
      Omega( 3) = 0.0461450866606818998462370608887317402491D+00
      Omega( 4) = 0.1224336447000104455669561645536624894248D+00
      Omega( 5) = 0.2990278393358638452592061091062092259563D+00
      Omega( 6) = 0.6856106320477458669995253492945863627028D+00
      Omega( 7) = 1.5156041931287733760033015206758477688709D+00
      Omega( 8) = 3.5216267135009914567532474105604478609166D+00
C
      Alpha( 1) = 0.0016447651044251732014945396810785827046D+00
      Alpha( 2) = 0.0108516614681113453641541537544878970323D+00
      Alpha( 3) = 0.0393710657262846921716368999571988851471D+00
      Alpha( 4) = 0.1181468832597833753259367051136052850779D+00
      Alpha( 5) = 0.3170960230793336978930008207555601984495D+00
      Alpha( 6) = 0.7850072328236101285546533201653573996737D+00
      Alpha( 7) = 1.8325419240849563618477036097687005167245D+00
      Alpha( 8) = 4.1809312247825503000391023089576947313617D+00
C
      Xi( 1) = 1.018263452788968317116839734648D+00
      Xi( 2) = 1.169302024710275848297380552054D+00
      Xi( 3) = 1.498038435178118336628221018980D+00
      Xi( 4) = 2.061481544340214850383544153978D+00
      Xi( 5) = 2.960592752162711259640570649054D+00
      Xi( 6) = 4.369192178959472600517643936513D+00
      Xi( 7) = 6.582798084819636397473779920730D+00
      Xi( 8) = 10.105353697117153155722590884835D+00
      Xi( 9) = 15.811915705435575898833711061542D+00
      Xi(10) = 25.261995455184047860638929527966D+00
      Xi(11) = 41.320061386100109580149508659019D+00
      Xi(12) = 69.433193047246356169321934714844D+00
      Xi(13) = 120.312329102978800024481564889811D+00
      Xi(14) = 215.238606077263635044105605231834D+00
      Xi(15) = 391.261408758174212108027489875894D+00
      Xi(16) = 641.835825018210841652610554319836D+00
      GOTO 9999
C
C     ----- K = 08 , R = 8E2 -----
C
  828 CONTINUE
C
      RIni = 8.0D+02
C
      Omega( 1) = 0.0040264999918426540221549566906517059195D+00
      Omega( 2) = 0.0146786769119747113790630218058130829206D+00
      Omega( 3) = 0.0439682178014438697386619039203514702763D+00
      Omega( 4) = 0.1183595924121215987276671197103716792753D+00
      Omega( 5) = 0.2921245499269274579492387117207741198399D+00
      Omega( 6) = 0.6752146704956449841349142015900497426628D+00
      Omega( 7) = 1.5019777460020226257262840330319875192799D+00
      Omega( 8) = 3.5059316086191087472306165961910551231995D+00
C
      Alpha( 1) = 0.0014875565232259687857860485271817296837D+00
      Alpha( 2) = 0.0099749829069579409847811583139653990315D+00
      Alpha( 3) = 0.0369240609001513219396858715704023978788D+00
      Alpha( 4) = 0.1126502931174692987160968447868381048238D+00
      Alpha( 5) = 0.3061859528751703799301104452768029773324D+00
      Alpha( 6) = 0.7654746988259726943760324902044089867559D+00
      Alpha( 7) = 1.8009234408139323254478478264495322491712D+00
      Alpha( 8) = 4.1345719764847616165340249416715323604876D+00
C
      Xi( 1) = 1.018540004076203246089621368142D+00
      Xi( 2) = 1.171952366106674925231613537679D+00
      Xi( 3) = 1.506315092686870306631745364268D+00
      Xi( 4) = 2.080601695298627345403000288293D+00
      Xi( 5) = 2.999624216167487557359663807866D+00
      Xi( 6) = 4.444665879532511173176378971306D+00
      Xi( 7) = 6.725481320767191097381026976265D+00
      Xi( 8) = 10.373512238455663971720133176291D+00
      Xi( 9) = 16.318228897540168238419755475377D+00
      Xi(10) = 26.230309144769280512918263603694D+00
      Xi(11) = 43.210003744964286895996474768822D+00
      Xi(12) = 73.228488043359466513892996886170D+00
      Xi(13) = 128.234621569760617182387463230953D+00
      Xi(14) = 232.622632419814380777545181899768D+00
      Xi(15) = 431.521930984642145351370956518622D+00
      Xi(16) = 728.170044478462002657259688476188D+00
      GOTO 9999
C
C     ----- K = 08 , R = 9E2 -----
C
  829 CONTINUE
C
      RIni = 9.0D+02
C
      Omega( 1) = 0.0037004598585148988313034407312759888065D+00
      Omega( 2) = 0.0138175172883332139873169817391451719146D+00
      Omega( 3) = 0.0421928680892260580071807026664165718444D+00
      Omega( 4) = 0.1150007485811967754960245292894516211391D+00
      Omega( 5) = 0.2863906977665336270528519124756172686830D+00
      Omega( 6) = 0.6665335644303945399156280382513273252698D+00
      Omega( 7) = 1.4905611878984845606243778148858325494075D+00
      Omega( 8) = 3.4927739456075436869086914271065325010568D+00
C
      Alpha( 1) = 0.0013620554429901594556538322548094877007D+00
      Alpha( 2) = 0.0092734079691412496733424783261490187414D+00
      Alpha( 3) = 0.0349498666613168215373320165534343217928D+00
      Alpha( 4) = 0.1081731334469284820388448528638569712257D+00
      Alpha( 5) = 0.2972266643940104389803522066948460178537D+00
      Alpha( 6) = 0.7493320377090901402957270893701036129642D+00
      Alpha( 7) = 1.7746689959691647210096240860899285962660D+00
      Alpha( 8) = 4.0959603100621581276558069717452781333122D+00
C
      Xi( 1) = 1.018775508491156873958592699037D+00
      Xi( 2) = 1.174211612730809165200743238167D+00
      Xi( 3) = 1.513382632847937119707250330780D+00
      Xi( 4) = 2.096967615958879766228581398302D+00
      Xi( 5) = 3.033136975610514185800956066252D+00
      Xi( 6) = 4.509721095769690817067765209458D+00
      Xi( 7) = 6.849034886445259618162340053971D+00
      Xi( 8) = 10.606960214946192893638310383864D+00
      Xi( 9) = 16.761727614323374801683086410264D+00
      Xi(10) = 27.084571457613766028507673588877D+00
      Xi(11) = 44.891417230264682947105114863007D+00
      Xi(12) = 76.639019392290796128808239018326D+00
      Xi(13) = 135.442688929812149994935666086349D+00
      Xi(14) = 248.697760513201823903695064643671D+00
      Xi(15) = 469.676669712967440134532282058899D+00
      Xi(16) = 813.407470316859758374672395575544D+00
      GOTO 9999
C
C     ----- K = 08 , R = 1E3 -----
C
  831 CONTINUE
C
      RIni = 1.0D+03
C
      Omega( 1) = 0.0034335760503910689907179870465625537435D+00
      Omega( 2) = 0.0131080577194026246206680157667390229648D+00
      Omega( 3) = 0.0407113742175260309220409526975714697983D+00
      Omega( 4) = 0.1121716653092656144037673852931202844729D+00
      Omega( 5) = 0.2815304358415968095431282380847193280715D+00
      Omega( 6) = 0.6591413831361904632319577423960055284624D+00
      Omega( 7) = 1.4808119960150753685384703017824392645707D+00
      Omega( 8) = 3.4815321032916017986516554749698570958572D+00
C
      Alpha( 1) = 0.0012593274957499998663477814415077860044D+00
      Alpha( 2) = 0.0086978504539733445175800526094622711071D+00
      Alpha( 3) = 0.0333181980142722798886561596382049899034D+00
      Alpha( 4) = 0.1044413334133183721434603302702370797306D+00
      Alpha( 5) = 0.2897059260448647628944686405194630651749D+00
      Alpha( 6) = 0.7357066637534231497890305218056994362996D+00
      Alpha( 7) = 1.7524189550363361240819734532259133175103D+00
      Alpha( 8) = 4.0631526124948221172054896310754656951758D+00
C
      Xi( 1) = 1.018979407447829179018429501813D+00
      Xi( 2) = 1.176169329208864889084115712414D+00
      Xi( 3) = 1.519516377580273219349143998258D+00
      Xi( 4) = 2.111199937117118880566118965803D+00
      Xi( 5) = 3.062359396681441598125211767112D+00
      Xi( 6) = 4.566636996217601827171422845808D+00
      Xi( 7) = 6.957560060020085674660494934329D+00
      Xi( 8) = 10.812960773503130699688123961266D+00
      Xi( 9) = 17.155169793462104483455776815504D+00
      Xi(10) = 27.847134518765816187796380276609D+00
      Xi(11) = 46.403372070943336071685569876877D+00
      Xi(12) = 79.732837463404732533667207405870D+00
      Xi(13) = 142.053194706536473054803160209758D+00
      Xi(14) = 263.652774992788203028970528407626D+00
      Xi(15) = 505.955334415113799101382241474312D+00
      Xi(16) = 897.583498348762517882182265793745D+00
      GOTO 9999
C
C     ----- K = 08 , R = 2E3 -----
C
  832 CONTINUE
C
      RIni = 2.0D+03
C
      Omega( 1) = 0.0021430462268822023309589085496296156919D+00
      Omega( 2) = 0.0095934513264009831643493869668404272488D+00
      Omega( 3) = 0.0330567547313259784287067452601682759905D+00
      Omega( 4) = 0.0971319794098423857755587353113302384600D+00
      Omega( 5) = 0.2551846773649667515662751299254651371484D+00
      Omega( 6) = 0.6185015005689335353296354202967677338165D+00
      Omega( 7) = 1.4267377848347798636021235840765086777537D+00
      Omega( 8) = 3.4190751095195143275089905143815371957317D+00
C
      Alpha( 1) = 0.0007623416133137365415047053069166893557D+00
      Alpha( 2) = 0.0058908109757378770075222831209549312348D+00
      Alpha( 3) = 0.0251495451866708728604304879869490996214D+00
      Alpha( 4) = 0.0852376686243077630886558204714731346030D+00
      Alpha( 5) = 0.2501340726547855392654547135311560168702D+00
      Alpha( 6) = 0.6627794284196205648477043248512075024337D+00
      Alpha( 7) = 1.6318321257986886528481423885850176702661D+00
      Alpha( 8) = 3.8839165956082179677767013403766327428457D+00
C
      Xi( 1) = 1.020159001593107734868863534228D+00
      Xi( 2) = 1.187526285869552161852599758696D+00
      Xi( 3) = 1.555265978199967590446355092393D+00
      Xi( 4) = 2.194689107792471268213937984903D+00
      Xi( 5) = 3.235249322883711310368595737152D+00
      Xi( 6) = 4.906996158365866961097168097439D+00
      Xi( 7) = 7.614850924736838379876940896551D+00
      Xi( 8) = 12.079316566765041890754717179846D+00
      Xi( 9) = 19.616275207518496383196637822266D+00
      Xi(10) = 32.715915456623407751163279399975D+00
      Xi(11) = 56.297129941781333100497830557174D+00
      Xi(12) = 100.593458953313436066157038339952D+00
      Xi(13) = 188.357478859669821333322481393679D+00
      Xi(14) = 373.907275932987620992165034294885D+00
      Xi(15) = 795.634013655747957927122371302175D+00
      Xi(16) = 1686.377993447082192846053771972947D+00
      GOTO 9999
C
C     ----- K = 08 , R = 3E3 -----
C
  833 CONTINUE
C
      RIni = 3.0D+03
C
      Omega( 1) = 0.0016651967973811464223829755079871184353D+00
      Omega( 2) = 0.0082344958996268037770835195553187357831D+00
      Omega( 3) = 0.0299132583617116829806614709472145641200D+00
      Omega( 4) = 0.0907228736405908252229783605979918092999D+00
      Omega( 5) = 0.2436730155314357148480658432632139920315D+00
      Omega( 6) = 0.6004207194957538095076224338164649907412D+00
      Omega( 7) = 1.4024065780531140042045393223624216716416D+00
      Omega( 8) = 3.3909109012617059081100279183118573200773D+00
C
      Alpha( 1) = 0.0005779647722766572749244552188729873166D+00
      Alpha( 2) = 0.0048345080412755526751872900501766361359D+00
      Alpha( 3) = 0.0219435969317702805491434843790377051675D+00
      Alpha( 4) = 0.0773991677369858432824351046619426597317D+00
      Alpha( 5) = 0.2334900109680693839101731784646531764338D+00
      Alpha( 6) = 0.6314114180649208010482481556735478989140D+00
      Alpha( 7) = 1.5791206489394280554393670823465356534143D+00
      Alpha( 8) = 3.8047625323126017483563177101757446507690D+00
C
      Xi( 1) = 1.020717974317504311582843756767D+00
      Xi( 2) = 1.192926375345656953897745256032D+00
      Xi( 3) = 1.572365758565759568505214305301D+00
      Xi( 4) = 2.234949192792201978991287192322D+00
      Xi( 5) = 3.319519259832642186667725492377D+00
      Xi( 6) = 5.075134836263528399172978433640D+00
      Xi( 7) = 7.944772461828507571605928472280D+00
      Xi( 8) = 12.726911915245122794947130184795D+00
      Xi( 9) = 20.902626152924975227662129917405D+00
      Xi(10) = 35.327057585343022875351559619617D+00
      Xi(11) = 61.770511965615394315426422977566D+00
      Xi(12) = 112.580787376352591942774150624018D+00
      Xi(13) = 216.291139842679021737525957291837D+00
      Xi(14) = 444.932477510570292433822459088333D+00
      Xi(15) = 1002.025485610155657867359479951119D+00
      Xi(16) = 2390.559013053741090448767181442236D+00
      GOTO 9999
C
C     ----- K = 08 , R = 4E3 -----
C
  834 CONTINUE
C
      RIni = 4.0D+03
C
      Omega( 1) = 0.0014125661829260472694481433082042132199D+00
      Omega( 2) = 0.0074937692680186526238192543016708446757D+00
      Omega( 3) = 0.0281426446017923193173519008647431860481D+00
      Omega( 4) = 0.0870442922664944302666443012861385852830D+00
      Omega( 5) = 0.2369808061228626872355123439417567965393D+00
      Omega( 6) = 0.5898121473669076002743424935736982206436D+00
      Omega( 7) = 1.3880475721468792164826552837375572835299D+00
      Omega( 8) = 3.3742707589985145493014645490248426540347D+00
C
      Alpha( 1) = 0.0004803056376499482843359545048171274639D+00
      Alpha( 2) = 0.0042692087887265292081189805867974529363D+00
      Alpha( 3) = 0.0201825342118137213973306414230290961598D+00
      Alpha( 4) = 0.0729992859041493389140632663814045599793D+00
      Alpha( 5) = 0.2239987472474404483303639032398812958036D+00
      Alpha( 6) = 0.6133156517874289083034830549756577511289D+00
      Alpha( 7) = 1.5484597079648389776791728156268845850718D+00
      Alpha( 8) = 3.7584802839867816838439334992116869216261D+00
C
      Xi( 1) = 1.021056476561818582402159372169D+00
      Xi( 2) = 1.196202442602344475633055487229D+00
      Xi( 3) = 1.582771745869115058219164793307D+00
      Xi( 4) = 2.259551932099742987264390237101D+00
      Xi( 5) = 3.371305780872138385369576885076D+00
      Xi( 6) = 5.179188258767591076636022107138D+00
      Xi( 7) = 8.150662048084573192002766361952D+00
      Xi( 8) = 13.135051401130087644388011458574D+00
      Xi( 9) = 21.722841000299467618256388234954D+00
      Xi(10) = 37.015307537045970150829665712422D+00
      Xi(11) = 65.370220308190230787404040313504D+00
      Xi(12) = 120.634159819952608179438602320488D+00
      Xi(13) = 235.588202886921525197605475909768D+00
      Xi(14) = 495.947551325105920855262908730765D+00
      Xi(15) = 1159.523072215187794564705825450801D+00
      Xi(16) = 3021.329814481966200334639438551676D+00
      GOTO 9999
C
C     ----- K = 08 , R = 5E3 -----
C
  835 CONTINUE
C
      RIni = 5.0D+03
C
      Omega( 1) = 0.0012560128074441001169983661285393938822D+00
      Omega( 2) = 0.0070239046575416292430630292066562336828D+00
      Omega( 3) = 0.0269953790374489263546685510405942487466D+00
      Omega( 4) = 0.0846323403990839879843205255383065122032D+00
      Omega( 5) = 0.2325573562750163136630607296773121106526D+00
      Omega( 6) = 0.5827590540484507476183539953495937879779D+00
      Omega( 7) = 1.3784658861922865615099592639936076921003D+00
      Omega( 8) = 3.3631587036004961274553159977784844159032D+00
C
      Alpha( 1) = 0.0004196991604112789323126704230364908410D+00
      Alpha( 2) = 0.0039154579369630628607557650162906970337D+00
      Alpha( 3) = 0.0190601015881723848084195252483574734015D+00
      Alpha( 4) = 0.0701548744514945170271547764118125201094D+00
      Alpha( 5) = 0.2178004795859835697544992089147442015928D+00
      Alpha( 6) = 0.6014111142968013818213036947302185808439D+00
      Alpha( 7) = 1.5281833987584513796309862443401073051064D+00
      Alpha( 8) = 3.7277730360685648396904673385066075752547D+00
C
      Xi( 1) = 1.021286012308503826965210004030D+00
      Xi( 2) = 1.198426449965374330683919035234D+00
      Xi( 3) = 1.589849845991704522228139817575D+00
      Xi( 4) = 2.276331635832671055957518446711D+00
      Xi( 5) = 3.406750436624172942434718436289D+00
      Xi( 6) = 5.250725632191437263970867910245D+00
      Xi( 7) = 8.292971227012546952921112630719D+00
      Xi( 8) = 13.418939586929014879009003369248D+00
      Xi( 9) = 22.297650276840474209416598228017D+00
      Xi(10) = 38.209159167489258602384838425792D+00
      Xi(11) = 67.944318550299602946229704514280D+00
      Xi(12) = 126.474664437222455208587046371349D+00
      Xi(13) = 249.846997526762718458415157840591D+00
      Xi(14) = 534.665541448239452937407634181000D+00
      Xi(15) = 1284.180367206866845375401453566155D+00
      Xi(16) = 3585.705870058596230975922480865847D+00
      GOTO 9999
C
C     ----- K = 08 , R = 6E3 -----
C
  836 CONTINUE
C
      RIni = 6.0D+03
C
      Omega( 1) = 0.0011500481004688015065847933138080960713D+00
      Omega( 2) = 0.0066999330935280467300949091072337360231D+00
      Omega( 3) = 0.0261924342913283282166704438523319176113D+00
      Omega( 4) = 0.0829303042986406247909529285133878317993D+00
      Omega( 5) = 0.2294182983616035149894473235687719636644D+00
      Omega( 6) = 0.5777335833512779340176424014430267561693D+00
      Omega( 7) = 1.3716213004158607933948704760140913094801D+00
      Omega( 8) = 3.3552168348270158923681016904794205402141D+00
C
      Alpha( 1) = 0.0003786335074948956111030892008294246320D+00
      Alpha( 2) = 0.0036740934531544308959442678803528070652D+00
      Alpha( 3) = 0.0182837045590645175482968388946947158047D+00
      Alpha( 4) = 0.0681673475707708599458773367218977767834D+00
      Alpha( 5) = 0.2134385607409870259064399444781656711712D+00
      Alpha( 6) = 0.5929904812716678494540455124361955085988D+00
      Alpha( 7) = 1.5137888616481399616442091171641948221804D+00
      Alpha( 8) = 3.7059238120115058068553687276036612274766D+00
C
      Xi( 1) = 1.021451800449782277646973427387D+00
      Xi( 2) = 1.200034072905669535404234171683D+00
      Xi( 3) = 1.594973221740001397320717069217D+00
      Xi( 4) = 2.288500077504828713508347459715D+00
      Xi( 5) = 3.432518210254069016863776031556D+00
      Xi( 6) = 5.302895618439775006925512856881D+00
      Xi( 7) = 8.397140788545276907615944050889D+00
      Xi( 8) = 13.627664939714634267410586776492D+00
      Xi( 9) = 22.722519618169650721176133778556D+00
      Xi(10) = 39.097212007429638102457003512313D+00
      Xi(11) = 69.874316127822566215022082047881D+00
      Xi(12) = 130.898155782560052906582370724209D+00
      Xi(13) = 260.793276038229136143398179115138D+00
      Xi(14) = 564.980498947044184665511323828468D+00
      Xi(15) = 1384.874279477222157042071160049090D+00
      Xi(16) = 4088.104869330484655742807831302343D+00
      GOTO 9999
C
C     ----- K = 08 , R = 7E3 -----
C
  837 CONTINUE
C
      RIni = 7.0D+03
C
      Omega( 1) = 0.0010744332166565012706056802573029163028D+00
      Omega( 2) = 0.0064652733490262209884080399618228041092D+00
      Omega( 3) = 0.0256044015068254107644749777497605869314D+00
      Omega( 4) = 0.0816762846342895764326064687865791569266D+00
      Omega( 5) = 0.2270959749376227446724177405634748083685D+00
      Omega( 6) = 0.5740045808031631187308072017394522390532D+00
      Omega( 7) = 1.3665329240577759860059831042988776061975D+00
      Omega( 8) = 3.3493104419351631469582014632990762947884D+00
C
      Alpha( 1) = 0.0003493074543081077457771562854316887980D+00
      Alpha( 2) = 0.0035007144790761561605025670379856550518D+00
      Alpha( 3) = 0.0177200698247082988648451766164337861653D+00
      Alpha( 4) = 0.0667135377029118788362463746133279585138D+00
      Alpha( 5) = 0.2102311912125660162451541365657980975357D+00
      Alpha( 6) = 0.5867753361711184133904170567852531803510D+00
      Alpha( 7) = 1.5031361158199717780598048122442378371488D+00
      Alpha( 8) = 3.6897272261912295819846069333536320300482D+00
C
      Xi( 1) = 1.021576047090998875534716139324D+00
      Xi( 2) = 1.201239584206932473216776491309D+00
      Xi( 3) = 1.598818959507872945031113631753D+00
      Xi( 4) = 2.297646168074588161629040472533D+00
      Xi( 5) = 3.451922062136160265820652992552D+00
      Xi( 6) = 5.342269967332008041328850200813D+00
      Xi( 7) = 8.475977879712730296210554392644D+00
      Xi( 8) = 13.786149094289558133218098756601D+00
      Xi( 9) = 23.046374048145767864104183075824D+00
      Xi(10) = 39.777345296839743717109971754553D+00
      Xi(11) = 71.361214735857919258532611195278D+00
      Xi(12) = 134.331918860858340325492576994293D+00
      Xi(13) = 269.377559056215929556010735268501D+00
      Xi(14) = 589.114902585990282690708852442185D+00
      Xi(15) = 1466.989064506464314763256595597340D+00
      Xi(16) = 4531.215030129676081482870131367235D+00
      GOTO 9999
C
C     ----- K = 08 , R = 8E3 -----
C
  838 CONTINUE
C
      RIni = 8.0D+03
C
      Omega( 1) = 0.0010187920024821150578778111207324530429D+00
      Omega( 2) = 0.0062904921216730113587560122793279626663D+00
      Omega( 3) = 0.0251627516939175241693924314767549788030D+00
      Omega( 4) = 0.0807301242767707417244079683149582393753D+00
      Omega( 5) = 0.2253383067673434065665170542014372756512D+00
      Omega( 6) = 0.5711759383022554021399480594922692944237D+00
      Omega( 7) = 1.3626676872789057304129201053655151554267D+00
      Omega( 8) = 3.3448225248609738599614438214757683454081D+00
C
      Alpha( 1) = 0.0003277166785761017410492756111355452942D+00
      Alpha( 2) = 0.0033724318575932676636188492459364485221D+00
      Alpha( 3) = 0.0172995667533701635395262377267733278075D+00
      Alpha( 4) = 0.0656226259651636144700284636321896414302D+00
      Alpha( 5) = 0.2078148268124083186713432075398699794277D+00
      Alpha( 6) = 0.5820796197507841215966310766116720287755D+00
      Alpha( 7) = 1.4950714931789085518788057216355014134024D+00
      Alpha( 8) = 3.6774503027702362826026338815665894799167D+00
C
      Xi( 1) = 1.021671002042388205223975850000D+00
      Xi( 2) = 1.202161314562530556150979355046D+00
      Xi( 3) = 1.601761466027823694051497593893D+00
      Xi( 4) = 2.304651398713548862695610353057D+00
      Xi( 5) = 3.466804726006286386684115963241D+00
      Xi( 6) = 5.372522613929845453073497196783D+00
      Xi( 7) = 8.536677921424592142309040188053D+00
      Xi( 8) = 13.908476252112584528250471826993D+00
      Xi( 9) = 23.297092629209026914266500440931D+00
      Xi(10) = 40.305769961082365378246539222573D+00
      Xi(11) = 72.521641272860303741032428348490D+00
      Xi(12) = 137.027405284593694581585232583620D+00
      Xi(13) = 276.169416913870757579463344200121D+00
      Xi(14) = 608.436098962609436813231411633751D+00
      Xi(15) = 1533.982633106747890172627535321226D+00
      Xi(16) = 4916.479141479043034834717218473088D+00
      GOTO 9999
C
C     ----- K = 08 , R = 9E3 -----
C
  839 CONTINUE
C
      RIni = 9.0D+03
C
      Omega( 1) = 0.0009772948930853391284634293194322207654D+00
      Omega( 2) = 0.0061588545859857373910286541373454438641D+00
      Omega( 3) = 0.0248279792632893715773956862941096446207D+00
      Omega( 4) = 0.0800104155964547076993918436893515888642D+00
      Omega( 5) = 0.2239981341538947426952855732085190254566D+00
      Omega( 6) = 0.5690154793075477250444485499691893437557D+00
      Omega( 7) = 1.3597123047826839243668387013919129913120D+00
      Omega( 8) = 3.3413902799279070789106871330886860960163D+00
C
      Alpha( 1) = 0.0003116085515013381649146047259156533382D+00
      Alpha( 2) = 0.0032763270270228551517249077334320572774D+00
      Alpha( 3) = 0.0169824721145694412746424573762427101542D+00
      Alpha( 4) = 0.0647962903315198952947204409646975165060D+00
      Alpha( 5) = 0.2059788824482929637675653403161568633095D+00
      Alpha( 6) = 0.5785040412974863724415719445914874086156D+00
      Alpha( 7) = 1.4889211961369413103574085854852171451057D+00
      Alpha( 8) = 3.6680786204192270931207647821636896878772D+00
C
      Xi( 1) = 1.021743940791255173482741935853D+00
      Xi( 2) = 1.202869572669394368350827273506D+00
      Xi( 3) = 1.604023797288146306822334552500D+00
      Xi( 4) = 2.310041602167562402878400740214D+00
      Xi( 5) = 3.478268425986425059753121002970D+00
      Xi( 6) = 5.395856515936656412537469895696D+00
      Xi( 7) = 8.583571074372407718779121044150D+00
      Xi( 8) = 14.003158945111380723029526063073D+00
      Xi( 9) = 23.491596446149704716793715775225D+00
      Xi(10) = 40.716854890305933873523303034858D+00
      Xi(11) = 73.427573061585111519056212614487D+00
      Xi(12) = 139.141130298612416882342301960307D+00
      Xi(13) = 281.528042976889119886063639341955D+00
      Xi(14) = 623.821571913905158934809946913447D+00
      Xi(15) = 1588.132591724796786158968586732954D+00
      Xi(16) = 5244.360558124685915970530913909897D+00
      GOTO 9999
C
C     ----- K = 08 , R = 1E4 -----
C
  841 CONTINUE
C
      RIni = 1.0D+04
C
      Omega( 1) = 0.0009464736592531918982037029551990930942D+00
      Omega( 2) = 0.0060603142125795529876743582796405629409D+00
      Omega( 3) = 0.0245761508893177345417288895412788463091D+00
      Omega( 4) = 0.0794675791825153871774235596336932729855D+00
      Omega( 5) = 0.2229854619672547855061591620673411284770D+00
      Omega( 6) = 0.5673808117367401690706377181960817779327D+00
      Omega( 7) = 1.3574743205521424079971690157542241195188D+00
      Omega( 8) = 3.3387907492178366939718225925304295742535D+00
C
      Alpha( 1) = 0.0002996417442897515258706464888341716724D+00
      Alpha( 2) = 0.0032046869025454530459728701673183248744D+00
      Alpha( 3) = 0.0167448876770150786602602997407696072685D+00
      Alpha( 4) = 0.0641750239049883675144635847287233332281D+00
      Alpha( 5) = 0.2045953172880951720781321501174687682578D+00
      Alpha( 6) = 0.5758049688355264809263081571266695846134D+00
      Alpha( 7) = 1.4842730758101718564713131232046805507707D+00
      Alpha( 8) = 3.6609907150717963281606054382066872676660D+00
C
      Xi( 1) = 1.021799368551998172280174947968D+00
      Xi( 2) = 1.203407933153845310266828738666D+00
      Xi( 3) = 1.605744200736071323793070775299D+00
      Xi( 4) = 2.314143118448317799169464126230D+00
      Xi( 5) = 3.486998390044023478622631229129D+00
      Xi( 6) = 5.413644324178145450510157843382D+00
      Xi( 7) = 8.619363150613084765573612688883D+00
      Xi( 8) = 14.075530346751954986067589814525D+00
      Xi( 9) = 23.640522751564629041001741605044D+00
      Xi(10) = 41.032293548734703098751497307717D+00
      Xi(11) = 74.124568878431914588367224183685D+00
      Xi(12) = 140.773073238400151588334274777026D+00
      Xi(13) = 285.685022748084026883441666200270D+00
      Xi(14) = 635.842924952911650204878668546371D+00
      Xi(15) = 1630.941436333689831705839878850384D+00
      Xi(16) = 5514.459053591922825354743054049322D+00
      GOTO 9999
C
C     ----- K = 08 , R = 2E4 (Best : R = 13750) -----
C
  899 CONTINUE
C
      RIni = 2.0D+04
C
      Omega( 1) = 0.0008991679845521025146213091349452900536D+00
      Omega( 2) = 0.0059076940528835928267816846645027473173D+00
      Omega( 3) = 0.0241839922144492679877196507007286463420D+00
      Omega( 4) = 0.0786197299999429678366664672739538843871D+00
      Omega( 5) = 0.2214005640066087043894230840246351021960D+00
      Omega( 6) = 0.5648187066649001726484184693344303695994D+00
      Omega( 7) = 1.3539633386811781257065209760348523104767D+00
      Omega( 8) = 3.3347117690388683390167318121299899758014D+00
C
      Alpha( 1) = 0.0002812708994976841560705178499606632370D+00
      Alpha( 2) = 0.0030942621869177655448272020594080444766D+00
      Alpha( 3) = 0.0163765529725487554336287900069724088326D+00
      Alpha( 4) = 0.0632081354614848777052774422835490497619D+00
      Alpha( 5) = 0.2024364017926240634856352787473987575595D+00
      Alpha( 6) = 0.5715854881795667998684029598166489449795D+00
      Alpha( 7) = 1.4769970948293096661392900803555505717668D+00
      Alpha( 8) = 3.6498865034832642553590253964301837186213D+00
C
      Xi( 1) = 1.021886671863076921249954831517D+00
      Xi( 2) = 1.204256055576628144164301470465D+00
      Xi( 3) = 1.608455843152621252459787137834D+00
      Xi( 4) = 2.320612438458095723274599020414D+00
      Xi( 5) = 3.500779688469234837353005751481D+00
      Xi( 6) = 5.441757327447499788513479712648D+00
      Xi( 7) = 8.676006252118678827370268358266D+00
      Xi( 8) = 14.190251215366537212471398365921D+00
      Xi( 9) = 23.877064381777278509466633238389D+00
      Xi(10) = 41.534481087341492515363494675285D+00
      Xi(11) = 75.237562784543763741429422964302D+00
      Xi(12) = 143.389208759031316908760800288292D+00
      Xi(13) = 292.384601788253456366994953441463D+00
      Xi(14) = 655.376702446691242420762080200802D+00
      Xi(15) = 1701.442526845445518124400052784040D+00
      Xi(16) = 5981.188006841379713574013976540300D+00
      GOTO 9999
C
C     ===== K = 09 =====
C
  900 CONTINUE
      GOTO ( 911, 911, 911, 912, 913, 914, 915, 916, 917, 918, 919, 921,
     *       922, 923, 924, 925, 926, 927, 928, 929, 931, 932, 933, 934,
     *       935, 936, 937, 938, 939, 941, 999),InitR
C
C     ----- K = 09 , R = 1E1 -----
C
  911 CONTINUE
C
      RIni = 1.0D+01
C
      Omega( 1) = 0.0944408179910022914211827939345855753572D+00
      Omega( 2) = 0.2279868254700224364391568376841945564593D+00
      Omega( 3) = 0.3835833446278580757457781630348492285520D+00
      Omega( 4) = 0.5808769624236879772758747331184991935515D+00
      Omega( 5) = 0.8513259963968657339497200786659902860265D+00
      Omega( 6) = 1.2461698737081927510677095694013871707284D+00
      Omega( 7) = 1.8552456496972611524982552100304644682183D+00
      Omega( 8) = 2.8782542842146350474483856851648511110398D+00
      Omega( 9) = 5.0648647749669027666626075490086122954381D+00
C
      Alpha( 1) = 0.0366466399628543326388667965287648797812D+00
      Alpha( 2) = 0.1966210450881259960855375681920698127669D+00
      Alpha( 3) = 0.4998872452506653510785443461639232509697D+00
      Alpha( 4) = 0.9775594233646502536597307220223029844419D+00
      Alpha( 5) = 1.6858162243103592955540195874597486636048D+00
      Alpha( 6) = 2.7212944265320903013127828984352163388394D+00
      Alpha( 7) = 4.2485871297667448256117628258010654462851D+00
      Alpha( 8) = 6.5659021657465178678850237226072295015911D+00
      Alpha( 9) = 10.3619750572942871608486670176318966696272D+00
C
      Xi( 1) = 1.004816667567330690951678262568D+00
      Xi( 2) = 1.043640164615983558925489671498D+00
      Xi( 3) = 1.122838292110773394764933197365D+00
      Xi( 4) = 1.245537281040695091961952034598D+00
      Xi( 5) = 1.416503870858663658945965357905D+00
      Xi( 6) = 1.642263421803972880713869131331D+00
      Xi( 7) = 1.931242857291945032001748150385D+00
      Xi( 8) = 2.293851689305635893449675233313D+00
      Xi( 9) = 2.742336706136967312526578344212D+00
      Xi(10) = 3.290135009447957957322283828461D+00
      Xi(11) = 3.950260267941195376894397695011D+00
      Xi(12) = 4.731964228677754497661561705613D+00
      Xi(13) = 5.634602288469151520206024130744D+00
      Xi(14) = 6.637671877203011474612159847908D+00
      Xi(15) = 7.687408987534133705983585560517D+00
      Xi(16) = 8.684756878114089741378667230975D+00
      Xi(17) = 9.487188575731532130461132190646D+00
      Xi(18) = 9.940546114292697808638432643846D+00
      GOTO 9999
C
C     ----- K = 09 , R = 2E1 -----
C
  912 CONTINUE
C
      RIni = 2.0D+01
C
      Omega( 1) = 0.0571068968380650208214283861801741082331D+00
      Omega( 2) = 0.1403046002401256655449986870953793527406D+00
      Omega( 3) = 0.2442888425192801570354217410852015746059D+00
      Omega( 4) = 0.3900028093978512700292296533355340670823D+00
      Omega( 5) = 0.6117598936983984954656075327950048858838D+00
      Omega( 6) = 0.9636093286970012084923647988077277659613D+00
      Omega( 7) = 1.5367428589142293419555346245530813575897D+00
      Omega( 8) = 2.5294131477214145515240717454474861369818D+00
      Omega( 9) = 4.6740157375771901292205168232385403825901D+00
C
      Alpha( 1) = 0.0221160161671775412491568451461843025641D+00
      Alpha( 2) = 0.1197005617264399827869773666744190876443D+00
      Alpha( 3) = 0.3095577652107071025687663745618039001783D+00
      Alpha( 4) = 0.6220398529933125745974668852067424040797D+00
      Alpha( 5) = 1.1147022114834890278841847410795651285298D+00
      Alpha( 6) = 1.8885600218294077823734736232275110978662D+00
      Alpha( 7) = 3.1147249042730755213409521431344728625845D+00
      Alpha( 8) = 5.0981371610313976662498536285283989855088D+00
      Alpha( 9) = 8.5274716594293656964781424889565641933586D+00
C
      Xi( 1) = 1.006555872841583084060547437222D+00
      Xi( 2) = 1.059567009314663162903441540053D+00
      Xi( 3) = 1.168615284932517771484226931644D+00
      Xi( 4) = 1.339862039200077573975269928930D+00
      Xi( 5) = 1.582914108413751619482655319526D+00
      Xi( 6) = 1.911428455969247629849666614277D+00
      Xi( 7) = 2.344012560312972394752084603908D+00
      Xi( 8) = 2.905321623723958540490286406310D+00
      Xi( 9) = 3.627168437279999827415818969278D+00
      Xi(10) = 4.549312527505255060976396830341D+00
      Xi(11) = 5.719136926761125761485560214936D+00
      Xi(12) = 7.188365599284229816648417177305D+00
      Xi(13) = 9.002970775840758896682580747495D+00
      Xi(14) = 11.179329521976524121525120580856D+00
      Xi(15) = 13.657279910840638271866953612488D+00
      Xi(16) = 16.227987378030576691434472280662D+00
      Xi(17) = 18.472407258722503753042842866705D+00
      Xi(18) = 19.818889952580328229006512152921D+00
      GOTO 9999
C
C     ----- K = 09 , R = 3E1 -----
C
  913 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.0421868132593801466490870087644449171194D+00
      Omega( 2) = 0.1050116169989357095166672938224561306697D+00
      Omega( 3) = 0.1876064252395335822036319395333947568361D+00
      Omega( 4) = 0.3112629685874389948304578196180969484885D+00
      Omega( 5) = 0.5109461867562471753438993471441165183933D+00
      Omega( 6) = 0.8412092785410361046654834082936247341422D+00
      Omega( 7) = 1.3943134568732370930999389657678477760783D+00
      Omega( 8) = 2.3697175189404877360642431849413469535648D+00
      Omega( 9) = 4.4940473157040011671636836609877718728967D+00
C
      Alpha( 1) = 0.0163142165710595383079330655470839950283D+00
      Alpha( 2) = 0.0888739577095855045956677292773306930940D+00
      Alpha( 3) = 0.2328241238721762329164349097920805320427D+00
      Alpha( 4) = 0.4776215751713829359528015044977777847635D+00
      Alpha( 5) = 0.8804879707385602953333657294532343939863D+00
      Alpha( 6) = 1.5426690213493910481135779066974578199734D+00
      Alpha( 7) = 2.6362216580612873393762302098863870014611D+00
      Alpha( 8) = 4.4683557670936419926877025687872446724214D+00
      Alpha( 9) = 7.7290029467926875373327832985381746766507D+00
C
      Xi( 1) = 1.007612743465411295597580765282D+00
      Xi( 2) = 1.069288322758420971270165722444D+00
      Xi( 3) = 1.196793605127912162879554036632D+00
      Xi( 4) = 1.398623977355137069597014465305D+00
      Xi( 5) = 1.688169721369932256018440985823D+00
      Xi( 6) = 2.084819481200011412769940166712D+00
      Xi( 7) = 2.615652456460386988132751917568D+00
      Xi( 8) = 3.317638945726810187353272851851D+00
      Xi( 9) = 4.240261748474029180732552957167D+00
      Xi(10) = 5.448388874927914446010673854559D+00
      Xi(11) = 7.024638234516258417268474989470D+00
      Xi(12) = 9.068900307006999057628238158912D+00
      Xi(13) = 11.689140847001693195028604321806D+00
      Xi(14) = 14.970329231989185316507862122393D+00
      Xi(15) = 18.896983906883134981050109679757D+00
      Xi(16) = 23.201684329870220000954783579061D+00
      Xi(17) = 27.172322404323542907947364177801D+00
      Xi(18) = 29.659116605127599357663581436384D+00
      GOTO 9999
C
C     ----- K = 09 , R = 4E1 -----
C
  914 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.0339352597594549956029142193103531610632D+00
      Omega( 2) = 0.0853875404125285589817064869910812063836D+00
      Omega( 3) = 0.1558201128416026146370843355470903190962D+00
      Omega( 4) = 0.2665467533840560541123516219430555906911D+00
      Omega( 5) = 0.4525182570791364093112062183532628978355D+00
      Omega( 6) = 0.7683860176045937915851984090220838652385D+00
      Omega( 7) = 1.3074948146510294393323609041246413653425D+00
      Omega( 8) = 2.2708533075573004104269481118194562441204D+00
      Omega( 9) = 4.3823089011892681450197062353879573493032D+00
C
      Alpha( 1) = 0.0131076927865854293291480287507455582130D+00
      Alpha( 2) = 0.0717893279105617697030163049110029760413D+00
      Alpha( 3) = 0.1901024737453172946342319291068712061588D+00
      Alpha( 4) = 0.3967095145193068460472401309813150760419D+00
      Alpha( 5) = 0.7480739001355727603405007530845693963784D+00
      Alpha( 6) = 1.3446599602615252448418320385670199357264D+00
      Alpha( 7) = 2.3582852576121468594558261422378109273268D+00
      Alpha( 8) = 4.0973891923794437705357018852936334951664D+00
      Alpha( 9) = 7.2533276338963031727204155973964816439548D+00
C
      Xi( 1) = 1.008370802207068996059723120506D+00
      Xi( 2) = 1.076281916485633244188503676586D+00
      Xi( 3) = 1.217180087021721421173571509033D+00
      Xi( 4) = 1.441475326386836438912961155800D+00
      Xi( 5) = 1.765697188344513895216086885043D+00
      Xi( 6) = 2.214079990555948211682427140268D+00
      Xi( 7) = 2.821020282627014928592337761337D+00
      Xi( 8) = 3.634363517681795004588690689573D+00
      Xi( 9) = 4.719590811785275055820904066550D+00
      Xi(10) = 6.164995925690650534038705909978D+00
      Xi(11) = 8.087321179937766064092785622819D+00
      Xi(12) = 10.635473568815829112539639300650D+00
      Xi(13) = 13.985214370229430863069863821835D+00
      Xi(14) = 18.306471237533177499196002102622D+00
      Xi(15) = 23.662468079847879753874195785102D+00
      Xi(16) = 29.775621021240134683738531862218D+00
      Xi(17) = 35.656631518931774505604126446201D+00
      Xi(18) = 39.469135361457544960983589987791D+00
      GOTO 9999
C
C     ----- K = 09 , R = 5E1 -----
C
  915 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.0286272689853100972254734734104286708600D+00
      Omega( 2) = 0.0727085059279048214692987329998974388445D+00
      Omega( 3) = 0.1351357560506628231771679773953742653703D+00
      Omega( 4) = 0.2371065680463565181662665207196205585660D+00
      Omega( 5) = 0.4133327297773413407479866504834475904318D+00
      Omega( 6) = 0.7184889744300452265571461929472008023367D+00
      Omega( 7) = 1.2469241956836626880611440526536171091720D+00
      Omega( 8) = 2.2011224300594862497357229180749982333509D+00
      Omega( 9) = 4.3033668024855845596170211830866492164205D+00
C
      Alpha( 1) = 0.0110461620270614760939376585401061614533D+00
      Alpha( 2) = 0.0607803585908880953167640268097748901255D+00
      Alpha( 3) = 0.1624695706747093375434251236177907173897D+00
      Alpha( 4) = 0.3440842840849093881819449614045680618801D+00
      Alpha( 5) = 0.6612286910567070099512161640564755771265D+00
      Alpha( 6) = 1.2133287214430294528736201109531123165652D+00
      Alpha( 7) = 2.1716507047182063833406368003053898974031D+00
      Alpha( 8) = 3.8454420774551983344853589574796615124797D+00
      Alpha( 9) = 6.9274020285996225332857123557772638378083D+00
C
      Xi( 1) = 1.008959735058196170785597967079D+00
      Xi( 2) = 1.081727547054457924820790315490D+00
      Xi( 3) = 1.233121852783218921423627478795D+00
      Xi( 4) = 1.475183691902814312737687862942D+00
      Xi( 5) = 1.827141305274111254060460940440D+00
      Xi( 6) = 2.317454167204209215281951461662D+00
      Xi( 7) = 2.986999140777004608073366687115D+00
      Xi( 8) = 3.893408271618215431243356650093D+00
      Xi( 9) = 5.116816594572857777867280759665D+00
      Xi(10) = 6.767428367328306293339984511803D+00
      Xi(11) = 8.994713612110818832037772985188D+00
      Xi(12) = 11.996105690184234046635758375032D+00
      Xi(13) = 16.017531079781982929557959671740D+00
      Xi(14) = 21.323363339601217870261651299302D+00
      Xi(15) = 28.079299750913681289060663814894D+00
      Xi(16) = 36.038988461443780951520832189772D+00
      Xi(17) = 43.963748828900448661904931668687D+00
      Xi(18) = 49.253357175193826684306275254244D+00
      GOTO 9999
C
C     ----- K = 09 , R = 6E1 -----
C
  916 CONTINUE
C
      RIni = 6.0D+01
C
      Omega( 1) = 0.0248959721764766027179375390554483260530D+00
      Omega( 2) = 0.0637623571241729140057427627219510668510D+00
      Omega( 3) = 0.1204491051105075692019704368396215699022D+00
      Omega( 4) = 0.2159745478825277155882121984942223491544D+00
      Omega( 5) = 0.3847391931442429500694665106541592081157D+00
      Omega( 6) = 0.6814342485253308534780963823251909161627D+00
      Omega( 7) = 1.2013063905514669937966282242669535662571D+00
      Omega( 8) = 2.1481696096724234316868024086133459604753D+00
      Omega( 9) = 4.2433533131659190401445413964864883382688D+00
C
      Alpha( 1) = 0.0095976729315090804611749044153445886707D+00
      Alpha( 2) = 0.0530301581506099356459857548096170631879D+00
      Alpha( 3) = 0.1429528058023013083942210840748820999124D+00
      Alpha( 4) = 0.3067286838593251691443934692227202276626D+00
      Alpha( 5) = 0.5991056928552873535320788533908853423782D+00
      Alpha( 6) = 1.1184415035302334036869387201562631162233D+00
      Alpha( 7) = 2.0353775012644672000924200849780731914507D+00
      Alpha( 8) = 3.6597441611647041148983050540266503958264D+00
      Alpha( 9) = 6.6854573090516597900362372186577886168379D+00
C
      Xi( 1) = 1.009439712975342740008954933728D+00
      Xi( 2) = 1.086173843961676821479045429442D+00
      Xi( 3) = 1.246182533203064046745112136705D+00
      Xi( 4) = 1.502931295016856072521704845713D+00
      Xi( 5) = 1.878023333089494872893965293770D+00
      Xi( 6) = 2.403678824259185109468717200798D+00
      Xi( 7) = 3.126614448205453200557712989394D+00
      Xi( 8) = 4.113390795506283836857619462535D+00
      Xi( 9) = 5.457701372264594397165637928993D+00
      Xi(10) = 7.290351469819014162965231662383D+00
      Xi(11) = 9.792165891831702495551326137502D+00
      Xi(12) = 13.208156738942577135276223998517D+00
      Xi(13) = 17.855245368191081212216886342503D+00
      Xi(14) = 24.098110732540835243753463146277D+00
      Xi(15) = 32.222282275814727370288892416283D+00
      Xi(16) = 42.047677291122653286947752349079D+00
      Xi(17) = 52.118697621441455680724175181240D+00
      Xi(18) = 59.014681121470907498860469786450D+00
      GOTO 9999
C
C     ----- K = 09 , R = 7E1 -----
C
  917 CONTINUE
C
      RIni = 7.0D+01
C
      Omega( 1) = 0.0221145238599984708285472578132013765639D+00
      Omega( 2) = 0.0570717255426385474206558284793189272932D+00
      Omega( 3) = 0.1094026520368098471004138748374767686755D+00
      Omega( 4) = 0.1999182472556715514091715824029904524650D+00
      Omega( 5) = 0.3626940710347193269039341168014090044380D+00
      Omega( 6) = 0.6524437931310793758327148950115770276170D+00
      Omega( 7) = 1.1652087065303460656573789067103064098774D+00
      Omega( 8) = 2.1059899831229173835893397903618051714147D+00
      Omega( 9) = 4.1955111714606977495710393721850550718955D+00
C
      Alpha( 1) = 0.0085183608716480930129208784529204478986D+00
      Alpha( 2) = 0.0472454375504523413084134817602199873932D+00
      Alpha( 3) = 0.1283432881500915206924142222821849657066D+00
      Alpha( 4) = 0.2786356835219511445694319518029402615866D+00
      Alpha( 5) = 0.5520531959128402951265945919345767833875D+00
      Alpha( 6) = 1.0459297104212418192065517374977900999511D+00
      Alpha( 7) = 1.9302841753638598767603418160376804735279D+00
      Alpha( 8) = 3.5153907951084480490184375200968247554556D+00
      Alpha( 9) = 6.4962582918876183627442111312433326020255D+00
C
      Xi( 1) = 1.009843670644150163007851661590D+00
      Xi( 2) = 1.089921584463474284599875108182D+00
      Xi( 3) = 1.257222614712810082149112289596D+00
      Xi( 4) = 1.526478640844834332927597297935D+00
      Xi( 5) = 1.921418080233361886256099326786D+00
      Xi( 6) = 2.477658685557917726512491829460D+00
      Xi( 7) = 3.247246099484508796142664954765D+00
      Xi( 8) = 4.304973006142101080943179747251D+00
      Xi( 9) = 5.757178149665511641443205181901D+00
      Xi(10) = 7.754134190414868861747355044400D+00
      Xi(11) = 10.506743335083980274598236537287D+00
      Xi(12) = 14.306497198812701568577943067595D+00
      Xi(13) = 19.541352837094909644308216023489D+00
      Xi(14) = 26.680033709516938091391158316412D+00
      Xi(15) = 36.141002166271460668139914318431D+00
      Xi(16) = 47.840286390914921566680506614944D+00
      Xi(17) = 60.139334259777297103655158139190D+00
      Xi(18) = 68.755191896088548135501650193646D+00
      GOTO 9999
C
C     ----- K = 09 , R = 8E1 -----
C
  918 CONTINUE
C
      RIni = 8.0D+01
C
      Omega( 1) = 0.0199526784301063577451595563647113174000D+00
      Omega( 2) = 0.0518563003933426289609489017590204795738D+00
      Omega( 3) = 0.1007466086972549770961059441287677174159D+00
      Omega( 4) = 0.1872164941269185330232445105269611929089D+00
      Omega( 5) = 0.3450262461891316848462511746031822212899D+00
      Omega( 6) = 0.6289183401844695380346686452988080873183D+00
      Omega( 7) = 1.1356374020567791251418512943338612330990D+00
      Omega( 8) = 2.0712467277560673076502312239810521532490D+00
      Omega( 9) = 4.1560789671987043254405835845943784079282D+00
C
      Alpha( 1) = 0.0076797894839484662018849520006419373530D+00
      Alpha( 2) = 0.0427441668967045518829467841415148399165D+00
      Alpha( 3) = 0.1169452389354414968946618488820909931292D+00
      Alpha( 4) = 0.2566228998618464373596571165281332582708D+00
      Alpha( 5) = 0.5149406599350990045905508696311159155812D+00
      Alpha( 6) = 0.9882757618211388782867611590976508750828D+00
      Alpha( 7) = 1.8460526692834495029365074469929197675810D+00
      Alpha( 8) = 3.3988951192899249120239757449013495715917D+00
      Alpha( 9) = 6.3427950576266634284162970747189547182643D+00
C
      Xi( 1) = 1.010191606375282269425497361226D+00
      Xi( 2) = 1.093153804208305990286476117035D+00
      Xi( 3) = 1.266767168213191796510841413692D+00
      Xi( 4) = 1.546904808150974197644700325327D+00
      Xi( 5) = 1.959220781480741635505182784982D+00
      Xi( 6) = 2.542436945678284931915413835668D+00
      Xi( 7) = 3.353509288478506267239254157175D+00
      Xi( 8) = 4.474882125703746105216396244941D+00
      Xi( 9) = 6.024763681615760368970313720638D+00
      Xi(10) = 8.171904787613485154906811658293D+00
      Xi(11) = 11.156115140955559894564430578612D+00
      Xi(12) = 15.314228304191129126365922274999D+00
      Xi(13) = 21.104851155624359648754762019962D+00
      Xi(14) = 29.103157066959354892787814605981D+00
      Xi(15) = 39.870672818322979935601324541494D+00
      Xi(16) = 53.445175507062735675845033256337D+00
      Xi(17) = 68.039170680158321147246169857681D+00
      Xi(18) = 78.476477920522839326622488442808D+00
      GOTO 9999
C
C     ----- K = 09 , R = 9E1 -----
C
  919 CONTINUE
C
      RIni = 9.0D+01
C
      Omega( 1) = 0.0182189939054983629658344577648132478487D+00
      Omega( 2) = 0.0476627110113685720830481885174068068523D+00
      Omega( 3) = 0.0937524260468592182812212071363333620866D+00
      Omega( 4) = 0.1768615462829543274748652151306949065201D+00
      Omega( 5) = 0.3304536625757883221111369659306333801396D+00
      Omega( 6) = 0.6093043202241387087383581167010504486825D+00
      Omega( 7) = 1.1107832014454115858535093730452558702382D+00
      Omega( 8) = 2.0419093040247857171864270808825381209317D+00
      Omega( 9) = 4.1227649447721858360846325375348442321410D+00
C
      Alpha( 1) = 0.0070075258403555020703036993606760418629D+00
      Alpha( 2) = 0.0391306245919652705714470233844259894340D+00
      Alpha( 3) = 0.1077729099627231377348294226459213263070D+00
      Alpha( 4) = 0.2388361570386172796128659845393826799409D+00
      Alpha( 5) = 0.4847687047212494723745068031517391204943D+00
      Alpha( 6) = 0.9410620539784709467651807601740188147232D+00
      Alpha( 7) = 1.7765815928488142974106880900286853375292D+00
      Alpha( 8) = 3.3022331320416157920179939244675892950909D+00
      Alpha( 9) = 6.2148972696211912656628906237443743520998D+00
C
      Xi( 1) = 1.010496581083280220880737942935D+00
      Xi( 2) = 1.095990183864951506009788317897D+00
      Xi( 3) = 1.275160572005126224937043843966D+00
      Xi( 4) = 1.564919881246217947534660197562D+00
      Xi( 5) = 1.992684549902902002571636330686D+00
      Xi( 6) = 2.600037625374358896124249440618D+00
      Xi( 7) = 3.448494194251035427356555373990D+00
      Xi( 8) = 4.627657101784136806088554294547D+00
      Xi( 9) = 6.266935365306330218970742862439D+00
      Xi(10) = 8.552684926856656488780572544783D+00
      Xi(11) = 11.752554209802655194039289199281D+00
      Xi(12) = 16.247597185827061139207216911018D+00
      Xi(13) = 22.566441385940585462321905652061D+00
      Xi(14) = 31.392232420985656915490835672244D+00
      Xi(15) = 43.437546052548135833148990059271D+00
      Xi(16) = 58.884123231982044899268657900393D+00
      Xi(17) = 75.828863188570153397449757903814D+00
      Xi(18) = 88.179798689784689713633270002902D+00
      GOTO 9999
C
C     ----- K = 09 , R = 1E2 -----
C
  921 CONTINUE
C
      RIni = 1.0D+02
C
      Omega( 1) = 0.0167944177457944369643606796760337829255D+00
      Omega( 2) = 0.0442084363213146724151648113421142483048D+00
      Omega( 3) = 0.0879647289700077565909605917249258766333D+00
      Omega( 4) = 0.1682207519727353216011092248982627950227D+00
      Omega( 5) = 0.3181646277188681652041499919869593782096D+00
      Omega( 6) = 0.5926071686898262811282918349764514687195D+00
      Omega( 7) = 1.0894766158992761576757013486371761246119D+00
      Omega( 8) = 2.0166575312935179065381569030179775836586D+00
      Omega( 9) = 4.0940777439727191787735560701833037455799D+00
C
      Alpha( 1) = 0.0064552940709981691840570444240049452489D+00
      Alpha( 2) = 0.0361585229138805650089794584134406107978D+00
      Alpha( 3) = 0.1002117521668118395314415851327094486578D+00
      Alpha( 4) = 0.2241169715425050551775489754424697252944D+00
      Alpha( 5) = 0.4596569999787066861996788491584453595351D+00
      Alpha( 6) = 0.9015052273114635050170299901406423259687D+00
      Alpha( 7) = 1.7180040401346316646016137985597538317961D+00
      Alpha( 8) = 3.2202910181231106476629905133179931908671D+00
      Alpha( 9) = 6.1060543748685659035287109919210024600034D+00
C
      Xi( 1) = 1.010767590706046830870135089242D+00
      Xi( 2) = 1.098513253256041984550969714540D+00
      Xi( 3) = 1.282640843522771026668366500800D+00
      Xi( 4) = 1.581016658720702851423020596222D+00
      Xi( 5) = 2.022682780613193844919048269837D+00
      Xi( 6) = 2.651877998803424621053134724313D+00
      Xi( 7) = 3.534377760344112353891460331923D+00
      Xi( 8) = 4.766517190711064898105131515393D+00
      Xi( 9) = 6.488322663266226617267279364043D+00
      Xi(10) = 8.902976339778169532525320395422D+00
      Xi(11) = 12.304988817151161595124519987365D+00
      Xi(12) = 17.118551029943605555280750074587D+00
      Xi(13) = 23.941540277556665305969119117435D+00
      Xi(14) = 33.565990820897510290427012691339D+00
      Xi(15) = 46.861900712050953334703029540265D+00
      Xi(16) = 64.174408587402146174338835038498D+00
      Xi(17) = 83.517069627790160295244259458514D+00
      Xi(18) = 97.866183331819870887258172587053D+00
      GOTO 9999
C
C     ----- K = 09 , R = 2E2 -----
C
  922 CONTINUE
C
      RIni = 2.0D+02
C
      Omega( 1) = 0.0098170280410918193187669420103624862151D+00
      Omega( 2) = 0.0271418312236300197459824140588935392771D+00
      Omega( 3) = 0.0588555405499913494831571141724380069604D+00
      Omega( 4) = 0.1233686937014436572622217043460057794846D+00
      Omega( 5) = 0.2520334255785888517701395089565252760622D+00
      Omega( 6) = 0.4999564200366363836923726954308833114737D+00
      Omega( 7) = 0.9685402483680957653956918040893953047998D+00
      Omega( 8) = 1.8714190637073344861563556351669035393570D+00
      Omega( 9) = 3.9288430703647423862870380339984421880217D+00
C
      Alpha( 1) = 0.0037535059072584840494327812487895812410D+00
      Alpha( 2) = 0.0215518207542155084664736459237888155371D+00
      Alpha( 3) = 0.0627363115303287840557539337127401779526D+00
      Alpha( 4) = 0.1500627747875535932740561143039492719709D+00
      Alpha( 5) = 0.3306054926762187856322579987411458546376D+00
      Alpha( 6) = 0.6934050066019539150150539918993075616527D+00
      Alpha( 7) = 1.4030303812801839501980469271380513873737D+00
      Alpha( 8) = 2.7716770734119855206130378366680133694899D+00
      Alpha( 9) = 5.5024333947554721585375403503093139079283D+00
C
      Xi( 1) = 1.012493178977090445302446097831D+00
      Xi( 2) = 1.114635600388191899131652695676D+00
      Xi( 3) = 1.330753639161060988111394409739D+00
      Xi( 4) = 1.685489635820716217791057078834D+00
      Xi( 5) = 2.219623428939024205059499239567D+00
      Xi( 6) = 2.997013539656331337869207831481D+00
      Xi( 7) = 4.115637502885650771990155849522D+00
      Xi( 8) = 5.723997886549447128637368997772D+00
      Xi( 9) = 8.046725109427186248094165943456D+00
      Xi(10) = 11.425345310044984696411229929680D+00
      Xi(11) = 16.382984293605163969229088216650D+00
      Xi(12) = 23.725841312075762377931109625706D+00
      Xi(13) = 34.695755476516356146515773772876D+00
      Xi(14) = 51.167672329122699286657605100004D+00
      Xi(15) = 75.757973024436673144255127709812D+00
      Xi(16) = 111.141141004968368143079882059965D+00
      Xi(17) = 156.003502086901291792675472436258D+00
      Xi(18) = 193.941202758701799216400907255320D+00
      GOTO 9999
C
C     ----- K = 09 , R = 3E2 -----
C
  923 CONTINUE
C
      RIni = 3.0D+02
C
      Omega( 1) = 0.0071726984278288066010701145430727976304D+00
      Omega( 2) = 0.0205765888728709433145848363075347542406D+00
      Omega( 3) = 0.0472784447759684887763415115508047747994D+00
      Omega( 4) = 0.1045511950832360717830596744115645435613D+00
      Omega( 5) = 0.2227845285766869881552247611988804010252D+00
      Omega( 6) = 0.4572258464535360821489034383047211917983D+00
      Omega( 7) = 0.9110357207216383647677138135634322679834D+00
      Omega( 8) = 1.8011087998289808716906312824335145705845D+00
      Omega( 9) = 3.8486906070706142239581243336132843069208D+00
C
      Alpha( 1) = 0.0027314605081480598387174042610339719239D+00
      Alpha( 2) = 0.0159839368530351834281333627679755338136D+00
      Alpha( 3) = 0.0482285546093564691738587914965341596485D+00
      Alpha( 4) = 0.1205956314577905223286492276957293512396D+00
      Alpha( 5) = 0.2773862359659811593578550292793494236321D+00
      Alpha( 6) = 0.6044051058271434355060290621874941052738D+00
      Alpha( 7) = 1.2638987330530245045212903165499085389456D+00
      Alpha( 8) = 2.5683342984551687556826066183290890876378D+00
      Alpha( 9) = 5.2238613500063365430170492231809475924820D+00
C
      Xi( 1) = 1.013442243452601123016379447428D+00
      Xi( 2) = 1.123545736564435914097738467010D+00
      Xi( 3) = 1.357579144166563902564569266129D+00
      Xi( 4) = 1.744447270564926417198842178991D+00
      Xi( 5) = 2.332479735856684662865037638291D+00
      Xi( 6) = 3.198527988114039162514945502380D+00
      Xi( 7) = 4.462522014311318948893286817636D+00
      Xi( 8) = 6.309665798547319267089439343010D+00
      Xi( 9) = 9.026287646419280932472140044354D+00
      Xi(10) = 13.058806753397421812276646413409D+00
      Xi(11) = 19.111213672692908200831385379104D+00
      Xi(12) = 28.306646857575006329121869441678D+00
      Xi(13) = 42.451836058384489782507875688111D+00
      Xi(14) = 64.442554456834715372048894010959D+00
      Xi(15) = 98.726758168282922258074396637539D+00
      Xi(16) = 150.976388946992133244240363865174D+00
      Xi(17) = 222.590737029752850958952414828218D+00
      Xi(18) = 288.838153884362069473423773047216D+00
      GOTO 9999
C
C     ----- K = 09 , R = 4E2 -----
C
  924 CONTINUE
C
      RIni = 4.0D+02
C
      Omega( 1) = 0.0057455219493550986707511589370728222192D+00
      Omega( 2) = 0.0169957484843711026029143378035524136749D+00
      Omega( 3) = 0.0407999991108632573202387710392669006865D+00
      Omega( 4) = 0.0936390097268970759833905601055015210932D+00
      Omega( 5) = 0.2052932850347774952756360611338948274351D+00
      Omega( 6) = 0.4310646867374589646330212966418571340910D+00
      Omega( 7) = 0.8752159980258892398767914255763855635450D+00
      Omega( 8) = 1.7568568252994703761090192206673066266376D+00
      Omega( 9) = 3.7981800919499917636168451195999296032824D+00
C
      Alpha( 1) = 0.0021805301099834817776713274169869682950D+00
      Alpha( 2) = 0.0129668297960735722897319044041863911332D+00
      Alpha( 3) = 0.0402738064105043472570085068273915140935D+00
      Alpha( 4) = 0.1041067480547515475663013055640515602818D+00
      Alpha( 5) = 0.2468847433368982753301061963679963184859D+00
      Alpha( 6) = 0.5522250628274896434768276298754585695860D+00
      Alpha( 7) = 1.1807304388190987321817504440701895873644D+00
      Alpha( 8) = 2.4449277513093071833964742278766379968147D+00
      Alpha( 9) = 5.0530353436462703249978933861541463556932D+00
C
      Xi( 1) = 1.014082320008305114359195564511D+00
      Xi( 2) = 1.129572660367690155216839620334D+00
      Xi( 3) = 1.375820336165024656551944537419D+00
      Xi( 4) = 1.784828830160060559066256824501D+00
      Xi( 5) = 2.410491847853145446194805812112D+00
      Xi( 6) = 3.339407540750610744540241570100D+00
      Xi( 7) = 4.708260852050882275168092183648D+00
      Xi( 8) = 6.730819490899820774741646456363D+00
      Xi( 9) = 9.742509792408877671368583328615D+00
      Xi(10) = 14.275279067561807030267706508653D+00
      Xi(11) = 21.184531719626648712390126760852D+00
      Xi(12) = 31.866723607990150701069209659977D+00
      Xi(13) = 48.632487294712915149030862949076D+00
      Xi(14) = 75.327499865931313749456421646755D+00
      Xi(15) = 118.207952411799593513130890443108D+00
      Xi(16) = 186.230305258828872594367709325525D+00
      Xi(17) = 284.882646179396948948880918806026D+00
      Xi(18) = 382.757902531871198242852116777613D+00
      GOTO 9999
C
C     ----- K = 09 , R = 5E2 -----
C
  925 CONTINUE
C
      RIni = 5.0D+02
C
      Omega( 1) = 0.0048408422622110301172535852011480361767D+00
      Omega( 2) = 0.0147065498776455584619584477344689243949D+00
      Omega( 3) = 0.0365691795723620899713360646310356116828D+00
      Omega( 4) = 0.0863219618381844940111901481882838993442D+00
      Omega( 5) = 0.1933129417331675636105908430706357137296D+00
      Omega( 6) = 0.4128560803248047374820924038107960996058D+00
      Omega( 7) = 0.8499869026869296652283887472112411387570D+00
      Omega( 8) = 1.7254635478924916411856693709836463312968D+00
      Omega( 9) = 3.7623132069881963044505335247436050849501D+00
C
      Alpha( 1) = 0.0018316193162704232806072771749805738750D+00
      Alpha( 2) = 0.0110482476764524436467582862997360848567D+00
      Alpha( 3) = 0.0351648885678759096644785831636825790270D+00
      Alpha( 4) = 0.0933425211230552068260714164427138683777D+00
      Alpha( 5) = 0.2266113854977841110267377383591380635153D+00
      Alpha( 6) = 0.5169700044462441223419059410648657149068D+00
      Alpha( 7) = 1.1237641613889137305690862911333738338726D+00
      Alpha( 8) = 2.3595017117435727388094490741821118717780D+00
      Alpha( 9) = 4.9339327011643232204912701277521591691766D+00
C
      Xi( 1) = 1.014557844970328790747127190031D+00
      Xi( 2) = 1.134059522948674922230086203889D+00
      Xi( 3) = 1.389451040287651769223470354930D+00
      Xi( 4) = 1.815157261209636782063603366311D+00
      Xi( 5) = 2.469464621121473972293475229733D+00
      Xi( 6) = 3.446760036341784853809386279799D+00
      Xi( 7) = 4.897283105684400378483644988847D+00
      Xi( 8) = 7.058237777334755448741671068902D+00
      Xi( 9) = 10.305980129936124188583246308326D+00
      Xi(10) = 15.245008648988065745426279118391D+00
      Xi(11) = 22.861666274799535308881459627628D+00
      Xi(12) = 34.793888874816124672645356596945D+00
      Xi(13) = 53.808524630705320771634214693790D+00
      Xi(14) = 84.637269347141066234974093873689D+00
      Xi(15) = 135.292823009987981955348779194992D+00
      Xi(16) = 218.145883117094211575692774829349D+00
      Xi(17) = 343.752713393957227056008107979324D+00
      Xi(18) = 475.812907782750572627250562618428D+00
      GOTO 9999
C
C     ----- K = 09 , R = 6E2 -----
C
  926 CONTINUE
C
      RIni = 6.0D+02
C
      Omega( 1) = 0.0042112164997987732153974691721304290937D+00
      Omega( 2) = 0.0131017443419484174921678409399261600043D+00
      Omega( 3) = 0.0335483667275851311346905863491896226947D+00
      Omega( 4) = 0.0809880866937857405520083745043002920738D+00
      Omega( 5) = 0.1844390570715671000409045104939664838639D+00
      Omega( 6) = 0.3992056659940321296274959306238372391817D+00
      Omega( 7) = 0.8309034361257299758688442403720131324008D+00
      Omega( 8) = 1.7015875252701152502013762224386539401166D+00
      Omega( 9) = 3.7350145975046860522764818890451010702236D+00
C
      Alpha( 1) = 0.0015889645189828853134388664020311410230D+00
      Alpha( 2) = 0.0097094235867913892325478815957529032943D+00
      Alpha( 3) = 0.0315686052432080812238802524499758916932D+00
      Alpha( 4) = 0.0856610590267349970372903755766458289145D+00
      Alpha( 5) = 0.2119351219922484596478980470313224770962D+00
      Alpha( 6) = 0.4911220300723124560155537915440149276947D+00
      Alpha( 7) = 1.0815587767859619615666219583083318411809D+00
      Alpha( 8) = 2.2957009271064489539480868440790573004051D+00
      Alpha( 9) = 4.8444980562273129774501778666007112406078D+00
C
      Xi( 1) = 1.014931989558186863813540878887D+00
      Xi( 2) = 1.137595427276255482867614587938D+00
      Xi( 3) = 1.400223484478566062872220698843D+00
      Xi( 4) = 1.839219415469965509591677987888D+00
      Xi( 5) = 2.516486149705543408925181858571D+00
      Xi( 6) = 3.532883133055617747323925992298D+00
      Xi( 7) = 5.050020403199381573491888453020D+00
      Xi( 8) = 7.324977296251430986407521261938D+00
      Xi( 9) = 10.769247260418148980261332514630D+00
      Xi(10) = 16.050472350587626915036087993371D+00
      Xi(11) = 24.270655484484105677417398183593D+00
      Xi(12) = 37.284660631758730655466171910462D+00
      Xi(13) = 58.277068934025140340554616003033D+00
      Xi(14) = 92.809724042426474277522885358849D+00
      Xi(15) = 150.592468673002287851692138076487D+00
      Xi(16) = 247.461463866105427042429987238847D+00
      Xi(17) = 399.765074289033366328194318839451D+00
      Xi(18) = 568.077352225634511329044329386306D+00
      GOTO 9999
C
C     ----- K = 09 , R = 7E2 -----
C
  927 CONTINUE
C
      RIni = 7.0D+02
C
      Omega( 1) = 0.0037452979413056402919975806280541985593D+00
      Omega( 2) = 0.0119065364988764364785814981427536363157D+00
      Omega( 3) = 0.0312620574896743206805217914212580154754D+00
      Omega( 4) = 0.0768822495241898944451949642064914058892D+00
      Omega( 5) = 0.1775211901125933941356831516356074018859D+00
      Omega( 6) = 0.3884622797054659326697941684747306112513D+00
      Omega( 7) = 0.8157768943467678142010719721799461012779D+00
      Omega( 8) = 1.6825795696161266758704433410542833371437D+00
      Omega( 9) = 3.7132685721630956357432878922608665561711D+00
C
      Alpha( 1) = 0.0014095059934701930788443244670950227349D+00
      Alpha( 2) = 0.0087163949435635453861511518988680613518D+00
      Alpha( 3) = 0.0288802601170618445065656990783375945853D+00
      Alpha( 4) = 0.0798510207914467600695583929248932975042D+00
      Alpha( 5) = 0.2007018612184257304212157968503760230305D+00
      Alpha( 6) = 0.4711331220524638425386446810572138588213D+00
      Alpha( 7) = 1.0486449545129898443035151855973197143612D+00
      Alpha( 8) = 2.2456262013296283809680314869083872508781D+00
      Alpha( 9) = 4.7740022239383786352602545566270464405534D+00
C
      Xi( 1) = 1.015237855473021364284894052510D+00
      Xi( 2) = 1.140489695132720713915616683742D+00
      Xi( 3) = 1.409061234383698115726554767058D+00
      Xi( 4) = 1.859021466956424458653700237942D+00
      Xi( 5) = 2.555337142598838082825293827050D+00
      Xi( 6) = 3.604394232696263654803364717694D+00
      Xi( 7) = 5.177580777145438844111940968062D+00
      Xi( 8) = 7.549221981124213971727637817466D+00
      Xi( 9) = 11.161601790690508574296901933565D+00
      Xi(10) = 16.738314212686065972357507547486D+00
      Xi(11) = 25.485098092299234071381186517158D+00
      Xi(12) = 39.454079013578476423096397240897D+00
      Xi(13) = 62.215656701418109827267377553994D+00
      Xi(14) = 100.112967315529583313404859978846D+00
      Xi(15) = 164.492889647548323692194394141097D+00
      Xi(16) = 274.665312278384626365479448395490D+00
      Xi(17) = 453.319129539005994583567016320558D+00
      Xi(18) = 659.604619900343409066856281697255D+00
      GOTO 9999
C
C     ----- K = 09 , R = 8E2 -----
C
  928 CONTINUE
C
      RIni = 8.0D+02
C
      Omega( 1) = 0.0033852098812278768834514551403294668752D+00
      Omega( 2) = 0.0109774080736766307342131628217718475327D+00
      Omega( 3) = 0.0294590554580619398994140601538704871132D+00
      Omega( 4) = 0.0735981065963514141640045181091966952636D+00
      Omega( 5) = 0.1719299109739136701433068893374311869593D+00
      Omega( 6) = 0.3797108500746642855382118053597650941811D+00
      Omega( 7) = 0.8033826246878963353733292540592714203740D+00
      Omega( 8) = 1.6669487573286433863047756309683222752938D+00
      Omega( 9) = 3.6953768800115142395979611622536253889848D+00
C
      Alpha( 1) = 0.0012708763612674633672627278152371388398D+00
      Alpha( 2) = 0.0079473213793494062269487572015835841199D+00
      Alpha( 3) = 0.0267833433812802426014028269601885412499D+00
      Alpha( 4) = 0.0752722660784924992309792568523363343047D+00
      Alpha( 5) = 0.1917594605878600712647614329187106818608D+00
      Alpha( 6) = 0.4550830556615172917860404067091906199494D+00
      Alpha( 7) = 1.0220320315021205132346993060821205290267D+00
      Alpha( 8) = 2.2049227056952739652734285513702161551919D+00
      Alpha( 9) = 4.7164964053301330262207846644173514505383D+00
C
      Xi( 1) = 1.015494846734757918623598405006D+00
      Xi( 2) = 1.142924036879366089781116799795D+00
      Xi( 3) = 1.416508618588864020534749676994D+00
      Xi( 4) = 1.875751259452837171729823462218D+00
      Xi( 5) = 2.588270227246376346558892622696D+00
      Xi( 6) = 3.665260371207954169384471421189D+00
      Xi( 7) = 5.286676676004319297592914939088D+00
      Xi( 8) = 7.742061977409237162172511714786D+00
      Xi( 9) = 11.501099651012934720235858510051D+00
      Xi(10) = 17.337630763701917932170792902014D+00
      Xi(11) = 26.551512591897243413183171689695D+00
      Xi(12) = 41.375925788490504571293104874030D+00
      Xi(13) = 65.740140071894493468962483717632D+00
      Xi(14) = 106.725407085254902904425389209564D+00
      Xi(15) = 177.257642594410686792683051749009D+00
      Xi(16) = 300.103328267503049187059360747298D+00
      Xi(17) = 504.714364302674262402481630829243D+00
      Xi(18) = 750.435276622015730974002423181446D+00
      GOTO 9999
C
C     ----- K = 09 , R = 9E2 -----
C
  929 CONTINUE
C
      RIni = 9.0D+02
C
      Omega( 1) = 0.0030977424691614607340063927698459167459D+00
      Omega( 2) = 0.0102316638905545772966616227712954856344D+00
      Omega( 3) = 0.0279930901026207251671101914475281446926D+00
      Omega( 4) = 0.0708952689284168618234554734569563549940D+00
      Omega( 5) = 0.1672877491668182620230403978567323974858D+00
      Omega( 6) = 0.3723968344810377277521139249838455498320D+00
      Omega( 7) = 0.7929727209927074070040797781899755136692D+00
      Omega( 8) = 1.6537803753979151391746704735652429008042D+00
      Omega( 9) = 3.6802970160533538548051912808745100846863D+00
C
      Alpha( 1) = 0.0011602484211060985717217235472717429268D+00
      Alpha( 2) = 0.0073321764357024223340904170269360928813D+00
      Alpha( 3) = 0.0250950982980019906870734758275176545794D+00
      Alpha( 4) = 0.0715519741026311104838022840446409844617D+00
      Alpha( 5) = 0.1844298926633792353379712181116190095054D+00
      Alpha( 6) = 0.4418305063609447410808393763570478540714D+00
      Alpha( 7) = 0.9999272003534077926873227526449738888914D+00
      Alpha( 8) = 2.1709623019090606504347823468847877848020D+00
      Alpha( 9) = 4.6683738668041492225410649119510253512999D+00
C
      Xi( 1) = 1.015715269904661042610176380219D+00
      Xi( 2) = 1.145013877896202366670971861673D+00
      Xi( 3) = 1.422912376064371018724932782540D+00
      Xi( 4) = 1.890168336484572671955632638952D+00
      Xi( 5) = 2.616730005799423755237856359379D+00
      Xi( 6) = 3.718043628867526893388290032405D+00
      Xi( 7) = 5.381674116079378136345140459795D+00
      Xi( 8) = 7.910768261877659578238525250882D+00
      Xi( 9) = 11.799680026576712552240089326716D+00
      Xi(10) = 17.867848238659794161270477630410D+00
      Xi(11) = 27.501306142315619481658606204633D+00
      Xi(12) = 43.100648356764131651636118292004D+00
      Xi(13) = 68.930828688179012590786509662166D+00
      Xi(14) = 112.772932024889469160289223736982D+00
      Xi(15) = 189.076604838093731836279332014783D+00
      Xi(16) = 324.032486664164790096664958696238D+00
      Xi(17) = 554.184538502184091124380671544714D+00
      Xi(18) = 840.601406203501388902932234259424D+00
      GOTO 9999
C
C     ----- K = 09 , R = 1E3 -----
C
  931 CONTINUE
C
      RIni = 1.0D+03
C
      Omega( 1) = 0.0028624090304025796460983327093585693035D+00
      Omega( 2) = 0.0096180997032853235358422147593948636768D+00
      Omega( 3) = 0.0267727106569027455297690373803298768962D+00
      Omega( 4) = 0.0686213572099158147516839018065226696308D+00
      Omega( 5) = 0.1633526775549459982288420800689010548012D+00
      Omega( 6) = 0.3661615356237251258893824340523437399497D+00
      Omega( 7) = 0.7840601806864651043913611028202126362885D+00
      Omega( 8) = 1.6424763479445014240521577564280164551747D+00
      Omega( 9) = 3.6673470190152673215452233668187886905798D+00
C
      Alpha( 1) = 0.0010697132697980306932461928089324321167D+00
      Alpha( 2) = 0.0068276991894412194727555210325975476593D+00
      Alpha( 3) = 0.0237021379736091671702849807343294052941D+00
      Alpha( 4) = 0.0684570314572635938460301412156194800218D+00
      Alpha( 5) = 0.1782853335724231008961636682497164940742D+00
      Alpha( 6) = 0.4306491453842748466022787023899098812763D+00
      Alpha( 7) = 0.9811811804065308630713911075549304996457D+00
      Alpha( 8) = 2.1420504811752469024200423719506147790526D+00
      Alpha( 9) = 4.6272998769299419703944498127867746006814D+00
C
      Xi( 1) = 1.015907400071548715553802766376D+00
      Xi( 2) = 1.146836899869768950548445940463D+00
      Xi( 3) = 1.428506324684849970426195331363D+00
      Xi( 4) = 1.902786202134254691319646679570D+00
      Xi( 5) = 2.641698869403860285140148156380D+00
      Xi( 6) = 3.764492866971857517720961849861D+00
      Xi( 7) = 5.465568235491984114878677436522D+00
      Xi( 8) = 8.060363154364896828493747271693D+00
      Xi( 9) = 12.065651317312179709000108385197D+00
      Xi(10) = 18.342605257009002163448574407312D+00
      Xi(11) = 28.356729072394385633010260860942D+00
      Xi(12) = 44.664383316634532559985526489754D+00
      Xi(13) = 71.845980142920193092792580102923D+00
      Xi(14) = 118.348121088373596204723625646693D+00
      Xi(15) = 200.092162779904987002232985560113D+00
      Xi(16) = 346.650998970642015284404635622195D+00
      Xi(17) = 601.917721942256615852429035840032D+00
      Xi(18) = 930.129063527018485946218362414584D+00
      GOTO 9999
C
C     ----- K = 09 , R = 2E3 -----
C
  932 CONTINUE
C
      RIni = 2.0D+03
C
      Omega( 1) = 0.0017219984012311614023473319012857984944D+00
      Omega( 2) = 0.0065884426254116958339659862291248781929D+00
      Omega( 3) = 0.0204962747939638545263652761457828788139D+00
      Omega( 4) = 0.0565294509352286292890293201152118740538D+00
      Omega( 5) = 0.1419270750951637811022431473473304208710D+00
      Omega( 6) = 0.3315985863927448277859409697665427074753D+00
      Omega( 7) = 0.7339849281745116267323264525046511153050D+00
      Omega( 8) = 1.5784277751334237942845012403481064211519D+00
      Omega( 9) = 3.5938760022078233764050020360159010124335D+00
C
      Alpha( 1) = 0.0006313454889490319018397847304711195671D+00
      Alpha( 2) = 0.0043668469857874161270711748684958397604D+00
      Alpha( 3) = 0.0167545678178722372817405520881162139801D+00
      Alpha( 4) = 0.0525840159291544383493056946665689110887D+00
      Alpha( 5) = 0.1459750048379582043313806502538021447890D+00
      Alpha( 6) = 0.3706381471395138468101146389521716173476D+00
      Alpha( 7) = 0.8789214132697631501897964079184077945683D+00
      Alpha( 8) = 1.9823948779811632675204591258477648807457D+00
      Alpha( 9) = 4.3986398388485846411809221390143420649110D+00
C
      Xi( 1) = 1.017049176702895268119275495877D+00
      Xi( 2) = 1.157698409290087672310690103394D+00
      Xi( 3) = 1.461985362290557343041552673668D+00
      Xi( 4) = 1.978766373725918028452300245768D+00
      Xi( 5) = 2.793256426785199025971881403585D+00
      Xi( 6) = 4.049236578360383199005206877885D+00
      Xi( 7) = 5.985905338379571236865489547441D+00
      Xi( 8) = 9.000728837074756394191765718205D+00
      Xi( 9) = 13.763279935388228015363809575433D+00
      Xi(10) = 21.425935302543523047758000643626D+00
      Xi(11) = 34.023698458379380317462281624330D+00
      Xi(12) = 55.263913287691619299901724104984D+00
      Xi(13) = 92.143612152471854152835373241714D+00
      Xi(14) = 158.428881638126718631176537144256D+00
      Xi(15) = 282.483090323835139001662497548750D+00
      Xi(16) = 524.843340620022861309212913738520D+00
      Xi(17) = 1008.768050953667418712100811717391D+00
      Xi(18) = 1793.900715959898712004161325239693D+00
      GOTO 9999
C
C     ----- K = 09 , R = 3E3 -----
C
  933 CONTINUE
C
      RIni = 3.0D+03
C
      Omega( 1) = 0.0012962831308852621829681126030056614873D+00
      Omega( 2) = 0.0054180644503889189698053654303810944270D+00
      Omega( 3) = 0.0179136340878167857434044487469071427199D+00
      Omega( 4) = 0.0513232221538696522669480409167830359252D+00
      Omega( 5) = 0.1324094929296019472031008951118380423395D+00
      Omega( 6) = 0.3158801151655616752330930224745841883305D+00
      Omega( 7) = 0.7108062282245082569428692997881569226593D+00
      Omega( 8) = 1.5484532018565422108527518396670075162547D+00
      Omega( 9) = 3.5594302119364723932699867692264206198161D+00
C
      Alpha( 1) = 0.0004677826770463522486164677222216998209D+00
      Alpha( 2) = 0.0034369131730817087311227362670523755028D+00
      Alpha( 3) = 0.0140274848449730914044667265394572552850D+00
      Alpha( 4) = 0.0460846626837133622854461925677505007570D+00
      Alpha( 5) = 0.1322703437013994585806381937276299254336D+00
      Alpha( 6) = 0.3444646922297356469088933095257232253061D+00
      Alpha( 7) = 0.8333436224059946878105382961354763438067D+00
      Alpha( 8) = 1.9100798947374523063889167895368359495478D+00
      Alpha( 9) = 4.2939725970702426905561965853053152386565D+00
C
      Xi( 1) = 1.017616265156594057691638588459D+00
      Xi( 2) = 1.163110781763274042535302732482D+00
      Xi( 3) = 1.478765111309849790942327896204D+00
      Xi( 4) = 2.017148114569818525805902642567D+00
      Xi( 5) = 2.870600924325366836960607419549D+00
      Xi( 6) = 4.196401930766708810328219358965D+00
      Xi( 7) = 6.258870094387393507272265846808D+00
      Xi( 8) = 9.502540637572148627190693481737D+00
      Xi( 9) = 14.686968839704956300319660389420D+00
      Xi(10) = 23.141109520577700826848843718153D+00
      Xi(11) = 37.257002672376713498653133527228D+00
      Xi(12) = 61.491445268627430022212454474584D+00
      Xi(13) = 104.487770788361471641092492745884D+00
      Xi(14) = 183.830182742659324748868421295356D+00
      Xi(15) = 337.441768310797018143309955462428D+00
      Xi(16) = 651.933533159956624059017116223913D+00
      Xi(17) = 1329.806380711968678443035685177165D+00
      Xi(18) = 2607.084823743725834788165229838341D+00
      GOTO 9999
C
C     ----- K = 09 , R = 4E3 -----
C
  934 CONTINUE
C
      RIni = 4.0D+03
C
      Omega( 1) = 0.0010688850352942277345185288013780511296D+00
      Omega( 2) = 0.0047768063467026780887263930162195402218D+00
      Omega( 3) = 0.0164437571556601970830269751969177072226D+00
      Omega( 4) = 0.0482872060920564158153134217549284912252D+00
      Omega( 5) = 0.1267659789113546653697116306513947847634D+00
      Omega( 6) = 0.3064418543767718427985220774889540962249D+00
      Omega( 7) = 0.6967562347704147392309613484595587351578D+00
      Omega( 8) = 1.5301758899530688042261300507718146945990D+00
      Omega( 9) = 3.5384053951731580640004126703246356555610D+00
C
      Alpha( 1) = 0.0003803762514503546170662433837566032691D+00
      Alpha( 2) = 0.0029354867547009724771518262816452637942D+00
      Alpha( 3) = 0.0125188405641583705003004340500227797861D+00
      Alpha( 4) = 0.0423977620853282027837060589453566095131D+00
      Alpha( 5) = 0.1243410987679817497871871158354739606011D+00
      Alpha( 6) = 0.3290894034731464541068853002503402649381D+00
      Alpha( 7) = 0.8062550457285698337446186445554019428528D+00
      Alpha( 8) = 1.8667285745423568974398972897255077896261D+00
      Alpha( 9) = 4.2308745838119774058602717836663487105398D+00
C
      Xi( 1) = 1.017972913409440976432313030742D+00
      Xi( 2) = 1.166520663629237118219354740756D+00
      Xi( 3) = 1.489369630213643176512051946592D+00
      Xi( 4) = 2.041508815880282935858666681916D+00
      Xi( 5) = 2.919964474799958625535548084606D+00
      Xi( 6) = 4.290980070192719836568578939406D+00
      Xi( 7) = 6.435732295665031909258357867287D+00
      Xi( 8) = 9.830756490223758554437838430573D+00
      Xi( 9) = 15.297664105121758036827384064082D+00
      Xi(10) = 24.289225229657723223453125172000D+00
      Xi(11) = 39.452631126727291607098058179304D+00
      Xi(12) = 65.792145778324712890317016444897D+00
      Xi(13) = 113.186204073241912321234892147004D+00
      Xi(14) = 202.174048148890269122057716799645D+00
      Xi(15) = 378.387710310245006184448612884808D+00
      Xi(16) = 750.653484216962727126887244821773D+00
      Xi(17) = 1595.695540737967720867906962212146D+00
      Xi(18) = 3375.521036011437759905007283123268D+00
      GOTO 9999
C
C     ----- K = 09 , R = 5E3 -----
C
  935 CONTINUE
C
      RIni = 5.0D+03
C
      Omega( 1) = 0.0009260627770325222336527758290516260109D+00
      Omega( 2) = 0.0043656639391461567718954344709259007828D+00
      Omega( 3) = 0.0154761374940805984633387460194797746738D+00
      Omega( 4) = 0.0462564321296459549713955307359558588587D+00
      Omega( 5) = 0.1229495139573777406216811489136908619457D+00
      Omega( 6) = 0.3000063118609164515841920828354005834626D+00
      Omega( 7) = 0.6871166160166532573721001608824110462592D+00
      Omega( 8) = 1.5175870789580829839544862114308898526360D+00
      Omega( 9) = 3.5239144130064043321758082694117319988436D+00
C
      Alpha( 1) = 0.0003254404360101518881557299786881284898D+00
      Alpha( 2) = 0.0026180514674484443450536228902708613120D+00
      Alpha( 3) = 0.0115451773269337696752609781492004970005D+00
      Alpha( 4) = 0.0399764124320777946288384390588488415119D+00
      Alpha( 5) = 0.1190640832195332738838947778114984998865D+00
      Alpha( 6) = 0.3187534379658245386546591704224340446672D+00
      Alpha( 7) = 0.7879045136257066413686123151904894257314D+00
      Alpha( 8) = 1.8371950908189229402555156522858226253447D+00
      Alpha( 9) = 4.1877312846909169937598127475553155818488D+00
C
      Xi( 1) = 1.018223491055936285850161415745D+00
      Xi( 2) = 1.168919254817437955126270543449D+00
      Xi( 3) = 1.496844550309035365475221479592D+00
      Xi( 4) = 2.058728171771944849208213490499D+00
      Xi( 5) = 2.954985959115944692569485918732D+00
      Xi( 6) = 4.358387310953027374196139254181D+00
      Xi( 7) = 6.562468157680101060388877520779D+00
      Xi( 8) = 10.067425096070548431147206880354D+00
      Xi( 9) = 15.741193797883095929421304326468D+00
      Xi(10) = 25.130010411199288482622660190202D+00
      Xi(11) = 41.076166787470114780123076059226D+00
      Xi(12) = 69.008812964846139359487153797090D+00
      Xi(13) = 119.782851976330407949011824797481D+00
      Xi(14) = 216.324815207572608483088671960104D+00
      Xi(15) = 410.676067869934812465393392244550D+00
      Xi(16) = 830.866962130459026758533269685358D+00
      Xi(17) = 1821.993947312338340438486739003565D+00
      Xi(18) = 4103.125862949434461413034114229959D+00
      GOTO 9999
C
C     ----- K = 09 , R = 6E3 -----
C
  936 CONTINUE
C
      RIni = 6.0D+03
C
      Omega( 1) = 0.0008275446967731432786926443147170612269D+00
      Omega( 2) = 0.0040770928209128811744981358677203386875D+00
      Omega( 3) = 0.0147834768067865480155659491968977992116D+00
      Omega( 4) = 0.0447858538699645466399227363568646964609D+00
      Omega( 5) = 0.1201638766104322943376720399244472758937D+00
      Omega( 6) = 0.2952809718970280440132614163672997165122D+00
      Omega( 7) = 0.6800069561584170269724625157437003508676D+00
      Omega( 8) = 1.5082761105851120459874631363206276546407D+00
      Omega( 9) = 3.5131912499637148285854676421635645056085D+00
C
      Alpha( 1) = 0.0002875179253957972909773293786282050144D+00
      Alpha( 2) = 0.0023975729163532282753073832408038934538D+00
      Alpha( 3) = 0.0108584821309771691825234673188516065778D+00
      Alpha( 4) = 0.0382462256420431207152676837679461385733D+00
      Alpha( 5) = 0.1152564953442573633031497694345635096624D+00
      Alpha( 6) = 0.3112408729002721530937109611913982121223D+00
      Alpha( 7) = 0.7744924306712252477856266164035758947648D+00
      Alpha( 8) = 1.8155216145202244967761170957309957429970D+00
      Alpha( 9) = 4.1559868654318949401059446469020031145192D+00
C
      Xi( 1) = 1.018411464449650201402608473700D+00
      Xi( 2) = 1.170720123158882433568248049305D+00
      Xi( 3) = 1.502465139934904708317493782221D+00
      Xi( 4) = 2.071702288831712671264562164453D+00
      Xi( 5) = 2.981443669360396931788254049067D+00
      Xi( 6) = 4.409480226859837182894902296582D+00
      Xi( 7) = 6.658908402585880750845781506086D+00
      Xi( 8) = 10.248336902673067018779140546769D+00
      Xi( 9) = 16.082016847958395967116418390219D+00
      Xi(10) = 25.780032612769895003848596992668D+00
      Xi(11) = 42.340290267741560462783079188398D+00
      Xi(12) = 71.534627945986173996884005532593D+00
      Xi(13) = 125.016349036148568474313691467614D+00
      Xi(14) = 227.695969317093756867365073048859D+00
      Xi(15) = 437.058285546168342500239489822889D+00
      Xi(16) = 897.938755526446165500242813095610D+00
      Xi(17) = 2018.127927928557927450547992975771D+00
      Xi(18) = 4792.881465381862065910922865441535D+00
      GOTO 9999
C
C     ----- K = 09 , R = 7E3 -----
C
  937 CONTINUE
C
      RIni = 7.0D+03
C
      Omega( 1) = 0.0007552936323448430041703487061287985993D+00
      Omega( 2) = 0.0038622717168615726822444232354050730471D+00
      Omega( 3) = 0.0142598450027955562363895375410649535297D+00
      Omega( 4) = 0.0436642852027035399011651271850009692344D+00
      Omega( 5) = 0.1180264413253419070705331657411552015446D+00
      Omega( 6) = 0.2916386422069088975826678389324797535664D+00
      Omega( 7) = 0.6745080482776006745187537261276844446911D+00
      Omega( 8) = 1.5010591549296192516709663733465163204528D+00
      Omega( 9) = 3.5048765058635483835708379629636510799173D+00
C
      Alpha( 1) = 0.0002596868790846979322642372917410644173D+00
      Alpha( 2) = 0.0022348903483392374692937683679574978335D+00
      Alpha( 3) = 0.0103453966022495648471076889335362736233D+00
      Alpha( 4) = 0.0369401084153355743395249870253405077847D+00
      Alpha( 5) = 0.1123604405679129684885266850236096303206D+00
      Alpha( 6) = 0.3054946294132511537728919687184614417674D+00
      Alpha( 7) = 0.7641900750527943561455149201755432386562D+00
      Alpha( 8) = 1.7988216034442439343740346990863088194601D+00
      Alpha( 9) = 4.1314779669129050722158058572830441335100D+00
C
      Xi( 1) = 1.018558731978950639162663505122D+00
      Xi( 2) = 1.172131936927446008915414121798D+00
      Xi( 3) = 1.506876523430133237350068975413D+00
      Xi( 4) = 2.081901165189758657518906415085D+00
      Xi( 5) = 3.002283519752387932252174485015D+00
      Xi( 6) = 4.449827618791080872072407625950D+00
      Xi( 7) = 6.735296143463850805557097123355D+00
      Xi( 8) = 10.392136042058479286284700116028D+00
      Xi( 9) = 16.354011451083167446446386961156D+00
      Xi(10) = 26.301210973404164810904304339090D+00
      Xi(11) = 43.359455691751780596898679931428D+00
      Xi(12) = 73.584391609781571862158866537129D+00
      Xi(13) = 129.297798148595386447112787209335D+00
      Xi(14) = 237.092743920405970126097372485674D+00
      Xi(15) = 459.150124798627372768589083307234D+00
      Xi(16) = 955.154559206768593582204118774825D+00
      Xi(17) = 2190.358162303570911166872292596963D+00
      Xi(18) = 5447.138394667314925001733172393870D+00
      GOTO 9999
C
C     ----- K = 09 , R = 8E3 -----
C
  938 CONTINUE
C
      RIni = 8.0D+03
C
      Omega( 1) = 0.0006999750349259222250828226387191211000D+00
      Omega( 2) = 0.0036956292786038210907891552511189692254D+00
      Omega( 3) = 0.0138485816729835371586079546118996930204D+00
      Omega( 4) = 0.0427771976018919282652545887923878353831D+00
      Omega( 5) = 0.1163277118663094138832354677848268664775D+00
      Omega( 6) = 0.2887334228096178135637445844041693021609D+00
      Omega( 7) = 0.6701100499979678797664479905638046375316D+00
      Omega( 8) = 1.4952771916809653082172312799080771128502D+00
      Omega( 9) = 3.4982130048175731091425477714551561803091D+00
C
      Alpha( 1) = 0.0002383647610636543066935782918757368021D+00
      Alpha( 2) = 0.0021096524885159010384065933355404731664D+00
      Alpha( 3) = 0.0099462230796899336260355408041722302670D+00
      Alpha( 4) = 0.0359154574427805942811160831024119843846D+00
      Alpha( 5) = 0.1100747420247687572523597084039881366380D+00
      Alpha( 6) = 0.3009391008073506232193584930545426914250D+00
      Alpha( 7) = 0.7559949292957711996566484702331223388683D+00
      Alpha( 8) = 1.7855045682887562425713787184200498359132D+00
      Alpha( 9) = 4.1119029053730469726486707049417645976064D+00
C
      Xi( 1) = 1.018677716815463165224257824359D+00
      Xi( 2) = 1.173273207018096588190834983489D+00
      Xi( 3) = 1.510445803105953123802833448774D+00
      Xi( 4) = 2.090163365282045846837294389431D+00
      Xi( 5) = 3.019193541099971872087442958943D+00
      Xi( 6) = 4.482633558072083487452963446174D+00
      Xi( 7) = 6.797553391683034926338408121538D+00
      Xi( 8) = 10.509657889464081687372343232312D+00
      Xi( 9) = 16.577024137863919182184213596543D+00
      Xi(10) = 26.730143399664715267438630519337D+00
      Xi(11) = 44.201935389617642624404680873340D+00
      Xi(12) = 75.287795912764834453667628366702D+00
      Xi(13) = 132.878975976798816921453649797513D+00
      Xi(14) = 245.017212574918897008413587457198D+00
      Xi(15) = 477.983292761807701531928671556670D+00
      Xi(16) = 1004.685293736425898469821760272680D+00
      Xi(17) = 2343.085202243852703141158144717338D+00
      Xi(18) = 6067.768060278267428397924732053070D+00
      GOTO 9999
C
C     ----- K = 09 , R = 9E3 -----
C
  939 CONTINUE
C
      RIni = 9.0D+03
C
      Omega( 1) = 0.0006562555949134862097456894652110909183D+00
      Omega( 2) = 0.0035623967101711022665845959205357940114D+00
      Omega( 3) = 0.0135163916656435436341326689763109314413D+00
      Omega( 4) = 0.0420565524273398218171177154189877001045D+00
      Omega( 5) = 0.1149422647181856013597826653149791553687D+00
      Omega( 6) = 0.2863569692532348407516991362076197447095D+00
      Omega( 7) = 0.6665045419456762991429130127407631789538D+00
      Omega( 8) = 1.4905304766796539943287530638471594102157D+00
      Omega( 9) = 3.4927411942424347814258045463731150448439D+00
C
      Alpha( 1) = 0.0002215041451664834586830238490322980693D+00
      Alpha( 2) = 0.0020101870345634199486422168199478699435D+00
      Alpha( 3) = 0.0096263287145913500154076488345145090619D+00
      Alpha( 4) = 0.0350886056219235608138952604850846483941D+00
      Alpha( 5) = 0.1082211098172572829307475604510013766912D+00
      Alpha( 6) = 0.2972311371099197367467124875073380962931D+00
      Alpha( 7) = 0.7493060755473707331363562855752746827420D+00
      Alpha( 8) = 1.7746133568189385439030011926142549327778D+00
      Alpha( 9) = 4.0958729118449167887726358738831322625629D+00
C
      Xi( 1) = 1.018776069157263152419046203789D+00
      Xi( 2) = 1.174216979156755687858058878614D+00
      Xi( 3) = 1.513399629431814344404649830267D+00
      Xi( 4) = 2.097007699319862089040386599059D+00
      Xi( 5) = 3.033220597771280707200328152773D+00
      Xi( 6) = 4.509890311786309907964714893680D+00
      Xi( 7) = 6.849382021323657984603755499720D+00
      Xi( 8) = 10.607718344733282039848987210462D+00
      Xi( 9) = 16.763597330244027922355187776304D+00
      Xi(10) = 27.090082887005839542768348415791D+00
      Xi(11) = 44.911495157432774393002627988380D+00
      Xi(12) = 76.728678917868063699692093848626D+00
      Xi(13) = 135.924664766140963156892951246846D+00
      Xi(14) = 251.802780753667186458555349304334D+00
      Xi(15) = 494.256277307559753447785411140103D+00
      Xi(16) = 1048.041669105673348250107324020064D+00
      Xi(17) = 2479.529732791401696267996612732532D+00
      Xi(18) = 6656.264013706788183810658665606752D+00
      GOTO 9999
C
C     ----- K = 09 , R = 1E4 -----
C
  941 CONTINUE
C
      RIni = 1.0D+04
C
      Omega( 1) = 0.0006208594515187709415282896912614829521D+00
      Omega( 2) = 0.0034534126161566192116306856473018083165D+00
      Omega( 3) = 0.0132423168362171444399849953854619855065D+00
      Omega( 4) = 0.0414591277165024014841659873858992213513D+00
      Omega( 5) = 0.1137899306668965527603314298599190834693D+00
      Omega( 6) = 0.2843755027087346596648478880275945357425D+00
      Omega( 7) = 0.6634927640743865600138698401799786097399D+00
      Omega( 8) = 1.4865608254662938828103563304594558758254D+00
      Omega( 9) = 3.4881641795154761623472261522493909069453D+00
C
      Alpha( 1) = 0.0002078469652528859301345068560495227272D+00
      Alpha( 2) = 0.0019292989209037795216477166365849627017D+00
      Alpha( 3) = 0.0093641492864657089188382653455189341685D+00
      Alpha( 4) = 0.0344069545580813640534196835157887406353D+00
      Alpha( 5) = 0.1066866208021499817933889359133914354061D+00
      Alpha( 6) = 0.2941521685207762780499100851994143113188D+00
      Alpha( 7) = 0.7437390948258375387058989602895309189989D+00
      Alpha( 8) = 1.7655336768516124521660737300976506958250D+00
      Alpha( 9) = 4.0824948352969070938280060634184565060423D+00
C
      Xi( 1) = 1.018858789375025899808999141349D+00
      Xi( 2) = 1.175011031484274259969156906092D+00
      Xi( 3) = 1.515886425956611331731897773789D+00
      Xi( 4) = 2.102774530086370540936033246560D+00
      Xi( 5) = 3.045052594955341328242237608492D+00
      Xi( 6) = 4.532913593446208804383507073865D+00
      Xi( 7) = 6.893232572595227878112233987906D+00
      Xi( 8) = 10.690843024689531597613834090765D+00
      Xi( 9) = 16.922099277879264508581136361443D+00
      Xi(10) = 27.396661436507848682436749854219D+00
      Xi(11) = 45.517687762936599963486949960156D+00
      Xi(12) = 77.964215296603784070839093800487D+00
      Xi(13) = 138.548135610549745161401702375770D+00
      Xi(14) = 257.681395599821175701471531738207D+00
      Xi(15) = 508.463092995186691336328976831282D+00
      Xi(16) = 1086.316738391781835049165749751410D+00
      Xi(17) = 2602.119680662271022031362122106657D+00
      Xi(18) = 7213.819733283016149893995816455572D+00
      GOTO 9999
C
C     ----- K = 09 , R = 3E4 (Best : R = 28393) -----
C
  999 CONTINUE
C
      RIni = 3.0D+04
C
      Omega( 1) = 0.0004354299056155451189326690961860569229D+00
      Omega( 2) = 0.0028608520316584030092596681186037366196D+00
      Omega( 3) = 0.0117112913825153417034176557554037145792D+00
      Omega( 4) = 0.0380713875095065937734207205112024752225D+00
      Omega( 5) = 0.1071875873349240602293774501929668474531D+00
      Omega( 6) = 0.2729343308454240926079308049434501981523D+00
      Omega( 7) = 0.6460013499115237401077861678633240671843D+00
      Omega( 8) = 1.4634218815677372793051733146363346804719D+00
      Omega( 9) = 3.4614670015318758573060614613936536443362D+00
C
      Alpha( 1) = 0.0001362078758772922180869218623951687197D+00
      Alpha( 2) = 0.0014984233172990983195404520551471949652D+00
      Alpha( 3) = 0.0079304843136686811123963131500702594678D+00
      Alpha( 4) = 0.0306088525260754423692717909877281101672D+00
      Alpha( 5) = 0.0980233175125910484790222806927850740522D+00
      Alpha( 6) = 0.2766007114067537993448361866022366939433D+00
      Alpha( 7) = 0.7117747229082216156863423262013412795568D+00
      Alpha( 8) = 1.7131253759530024614919987979710924719257D+00
      Alpha( 9) = 4.0050154343014464928054985115579711418832D+00
C
      Xi( 1) = 1.019349605390273929466163349566D+00
      Xi( 2) = 1.179727839421699866128898093187D+00
      Xi( 3) = 1.530686767100153370941943709038D+00
      Xi( 4) = 2.137188978850325381123390067195D+00
      Xi( 5) = 3.115909405473552945846013595421D+00
      Xi( 6) = 4.671400470121846554196615164400D+00
      Xi( 7) = 7.158377525361323866562313078532D+00
      Xi( 8) = 11.196526468244599649727588630554D+00
      Xi( 9) = 17.893206861623897827029416873756D+00
      Xi(10) = 29.290661647905315439349704931260D+00
      Xi(11) = 49.300006109312959838575363136215D+00
      Xi(12) = 85.766036314244301040632389998564D+00
      Xi(13) = 155.365014625779427434593138457330D+00
      Xi(14) = 296.099822100602221353060627961895D+00
      Xi(15) = 603.777482619712320743854405691309D+00
      Xi(16) = 1353.360691141729413522476477282908D+00
      Xi(17) = 3513.499234321727268337554050958715D+00
      Xi(18) = 12351.225064419754213318469737714622D+00
      GOTO 9999
C
C     ===== K = 10 =====
C
 1000 CONTINUE
      GOTO (1011,1011,1011,1012,1013,1014,1015,1016,1017,1018,1019,1021,
     *      1022,1023,1024,1025,1026,1027,1028,1029,1031,1032,1033,1034,
     *      1035,1036,1037,1038,1039,1041,1099),InitR
C
C     ----- K = 10 , R = 1E1 -----
C
 1011 CONTINUE
C
      RIni = 1.0D+01
C
      Omega( 1) = 0.0851548884512817157106711121739017755772D+00
      Omega( 2) = 0.2041221377375185690611911551450852186917D+00
      Omega( 3) = 0.3387624704041514381172919834206425093726D+00
      Omega( 4) = 0.5020218012767448245844623122557948136091D+00
      Omega( 5) = 0.7136389791481699146086931095922523127228D+00
      Omega( 6) = 1.0042283383685008889794898490066543672583D+00
      Omega( 7) = 1.4225951252649079001484644368957788174157D+00
      Omega( 8) = 2.0553775439606230667627384534057455311995D+00
      Omega( 9) = 3.1013168392332142260427868540872964331356D+00
      Omega(10) = 5.3215604240635308100348943582957872422412D+00
C
      Alpha( 1) = 0.0330697624272925114554972508829067123770D+00
      Alpha( 2) = 0.1768078437824842392430508761225205205392D+00
      Alpha( 3) = 0.4464853448677116219906260696781430397095D+00
      Alpha( 4) = 0.8637994095087873673501251137185619199954D+00
      Alpha( 5) = 1.4665304201175292089440183040949250425911D+00
      Alpha( 6) = 2.3172114001372251723340212059554232837399D+00
      Alpha( 7) = 3.5172089429912433802046661801554705562012D+00
      Alpha( 8) = 5.2328732180584053819763501458339760574745D+00
      Alpha( 9) = 7.7617287680176058257443261201302675544866D+00
      Alpha(10) = 11.7952949791468539881417187942247437604237D+00
C
      Xi( 1) = 1.003934043651302852277057420238D+00
      Xi( 2) = 1.035597560330198496896664250855D+00
      Xi( 3) = 1.099946515856572986790003210178D+00
      Xi( 4) = 1.199038529418928672736106844088D+00
      Xi( 5) = 1.336001895201351444384015032529D+00
      Xi( 6) = 1.515097369569101782649581111606D+00
      Xi( 7) = 1.741798718468895162004433929681D+00
      Xi( 8) = 2.022861086752043427612612891409D+00
      Xi( 9) = 2.366318054679312713438343873662D+00
      Xi(10) = 2.781311568499321894504597296738D+00
      Xi(11) = 3.277602366254750317336313925054D+00
      Xi(12) = 3.864515159405216115267103771380D+00
      Xi(13) = 4.548945094352016048008802151514D+00
      Xi(14) = 5.331920668070182002442475299731D+00
      Xi(15) = 6.203227682007736412846121698550D+00
      Xi(16) = 7.134066726905839409336729239897D+00
      Xi(17) = 8.069204892498868964857694496828D+00
      Xi(18) = 8.922965548049153544293843332014D+00
      Xi(19) = 9.586495357975749287068878423668D+00
      Xi(20) = 9.952473091634462124335791521901D+00
      GOTO 9999
C
C     ----- K = 10 , R = 2E1 -----
C
 1012 CONTINUE
C
      RIni = 2.0D+01
C
      Omega( 1) = 0.0513720958134025681927881633315191578504D+00
      Omega( 2) = 0.1248594580087193300568662937544139879265D+00
      Omega( 3) = 0.2128344130226482852554382302345814537148D+00
      Omega( 4) = 0.3287926283808711603196678341376468779345D+00
      Omega( 5) = 0.4942264205009505737215338982970536108041D+00
      Omega( 6) = 0.7420669272415499498969822245175720354382D+00
      Omega( 7) = 1.1227892497020226877544810117903750779078D+00
      Omega( 8) = 1.7234803772519538559099619434711314625019D+00
      Omega( 9) = 2.7403911047158000403173305770110346202273D+00
      Omega(10) = 4.9150273553728394682060809373069787397981D+00
C
      Alpha( 1) = 0.0199190782019326982112128465729483650648D+00
      Alpha( 2) = 0.1072334417383124932971112143464953447847D+00
      Alpha( 3) = 0.2744085978174513274971662440959718765043D+00
      Alpha( 4) = 0.5421416040798437781158408688142458231596D+00
      Alpha( 5) = 0.9483605618497397696719727888670092852408D+00
      Alpha( 6) = 1.5578821538499603195637721597677227691747D+00
      Alpha( 7) = 2.4764307129347023839812530487947128676751D+00
      Alpha( 8) = 3.8758067491779373346227205443526031558577D+00
      Alpha( 9) = 6.0582950800444962447845187458739246721962D+00
      Alpha(10) = 9.7118480399069814258672383999737576232292D+00
C
      Xi( 1) = 1.005376604124994693195173367517D+00
      Xi( 2) = 1.048763421907059670040891774523D+00
      Xi( 3) = 1.137541925740631291530738367079D+00
      Xi( 4) = 1.275782535377121582531242949266D+00
      Xi( 5) = 1.469783188682559988444235876681D+00
      Xi( 6) = 1.728376440612933768624023822547D+00
      Xi( 7) = 2.063393850121227532639495150235D+00
      Xi( 8) = 2.490252281796120580742437411459D+00
      Xi( 9) = 3.028594043714212232952173731171D+00
      Xi(10) = 3.702877357639770333199921714673D+00
      Xi(11) = 4.542709189548428958763809371035D+00
      Xi(12) = 5.582457295094456607795901925684D+00
      Xi(13) = 6.859182461836509940925066253214D+00
      Xi(14) = 8.407072372916607403726951819323D+00
      Xi(15) = 10.245285079537717073602798301124D+00
      Xi(16) = 12.354965727196505731813602757541D+00
      Xi(17) = 14.642575027180748620941885818070D+00
      Xi(18) = 16.896736001875174576492000255712D+00
      Xi(19) = 18.771523113469569298342176644212D+00
      Xi(20) = 19.856318263968247872161554390402D+00
      GOTO 9999
C
C     ----- K = 10 , R = 3E1 -----
C
 1013 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.0378829891316186436572269104899879366144D+00
      Omega( 2) = 0.0930163245037145861547622782694588750019D+00
      Omega( 3) = 0.1617466336454914180433906426004320167067D+00
      Omega( 4) = 0.2576551952740756494650200408846885125058D+00
      Omega( 5) = 0.4026941949745052461324956727395374400658D+00
      Omega( 6) = 0.6301925993044387373483508629767868569616D+00
      Omega( 7) = 0.9910799607619160759013239581083354323710D+00
      Omega( 8) = 1.5734470164829399977312815450680716367060D+00
      Omega( 9) = 2.5740012849105933854902683588505851730588D+00
      Omega(10) = 4.7268570117366334642008962596548826695653D+00
C
      Alpha( 1) = 0.0146720893474109634543403883795431585924D+00
      Alpha( 2) = 0.0793869117946712017544041301020385503762D+00
      Alpha( 3) = 0.2051749781339107614426065309687352566925D+00
      Alpha( 4) = 0.4118459242697055932040314796260815910500D+00
      Alpha( 5) = 0.7367388171009950032040031198121710076521D+00
      Alpha( 6) = 1.2445399693315783262552443044768324398319D+00
      Alpha( 7) = 2.0412209987653119546650165583301372862479D+00
      Alpha( 8) = 3.2995507853921115116319073834461050864775D+00
      Alpha( 9) = 5.3237296721066788398116220726308256416814D+00
      Alpha(10) = 8.8014613611571692972784797248664290236775D+00
C
      Xi( 1) = 1.006260876037949373641190720097D+00
      Xi( 2) = 1.056862976860428054949256337913D+00
      Xi( 3) = 1.160831214850607703750057231318D+00
      Xi( 4) = 1.323797135747777582617627778472D+00
      Xi( 5) = 1.554547200491692039627661392487D+00
      Xi( 6) = 1.865583007695159876906249030171D+00
      Xi( 7) = 2.273985700647530612452523746114D+00
      Xi( 8) = 2.802556054918465656879827951897D+00
      Xi( 9) = 3.481186170552955607380984925925D+00
      Xi(10) = 4.348427674928144312429395546005D+00
      Xi(11) = 5.453102342622122880935742195163D+00
      Xi(12) = 6.855472150034190314399351606767D+00
      Xi(13) = 8.626697477281858152456384125628D+00
      Xi(14) = 10.843790080905853050186049557624D+00
      Xi(15) = 13.574302827800757838619394313184D+00
      Xi(16) = 16.840507670545135452538909603959D+00
      Xi(17) = 20.549543164941929868083647647836D+00
      Xi(18) = 24.387232452140358875125603077549D+00
      Xi(19) = 27.729559759629049972215475428072D+00
      Xi(20) = 29.731006308352625576787997552231D+00
      GOTO 9999
C
C     ----- K = 10 , R = 4E1 -----
C
 1014 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.0304283542679826195679411611483278221613D+00
      Omega( 2) = 0.0753370830029148128009893956025244676766D+00
      Omega( 3) = 0.1331690520655589057678631209613584474027D+00
      Omega( 4) = 0.2174481193685754137342536554400318493663D+00
      Omega( 5) = 0.3501479372604658767023470822277175784620D+00
      Omega( 6) = 0.5645330070305193969379224661064853307835D+00
      Omega( 7) = 0.9118556551925829048335474247277687709357D+00
      Omega( 8) = 1.4812981052443766873333644285004595531063D+00
      Omega( 9) = 2.4704999769201472898348670437229657181888D+00
      Omega(10) = 4.6095978515266311219961137268796846910845D+00
C
      Alpha( 1) = 0.0117740251702337547500765916989873272058D+00
      Alpha( 2) = 0.0639700951282584003517481079092998896840D+00
      Alpha( 3) = 0.1666917616889322231515588312866782416677D+00
      Alpha( 4) = 0.3390289700500114554242939851613769519645D+00
      Alpha( 5) = 0.6176072358757308952085379738505821478611D+00
      Alpha( 6) = 1.0663680454451372958072502550308513491473D+00
      Alpha( 7) = 1.7906093738966222766832714352602806684445D+00
      Alpha( 8) = 2.9631424352797224503787421046041572481045D+00
      Alpha( 9) = 4.8893793130761661856753530486230374663137D+00
      Alpha(10) = 8.2575316823540801510969422416508223250275D+00
C
      Xi( 1) = 1.006899144780989947909471959875D+00
      Xi( 2) = 1.062723543495707518853447826057D+00
      Xi( 3) = 1.177761340350092555961936113818D+00
      Xi( 4) = 1.358932941180709253370557931806D+00
      Xi( 5) = 1.617097610798962527950926870712D+00
      Xi( 6) = 1.967858618396710338057341294071D+00
      Xi( 7) = 2.432817040973496266770964568948D+00
      Xi( 8) = 3.041253416943164042526501211761D+00
      Xi( 9) = 3.832241560710106011329589836834D+00
      Xi(10) = 4.857251484427018241307738577461D+00
      Xi(11) = 6.183217811492185554851641615670D+00
      Xi(12) = 7.895646709574505373897823634621D+00
      Xi(13) = 10.100429441955376910766695175425D+00
      Xi(14) = 12.920925598076232635423588046564D+00
      Xi(15) = 16.482361549301608256296036714161D+00
      Xi(16) = 20.867101081829256971345021298703D+00
      Xi(17) = 26.013304987079913349265480881911D+00
      Xi(18) = 31.534033822387357665073537305034D+00
      Xi(19) = 36.516222241626955218735117725259D+00
      Xi(20) = 39.582863294995880239918184528847D+00
      GOTO 9999
C
C     ----- K = 10 , R = 5E1 -----
C
 1015 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.0256360764998129773236843712697075048368D+00
      Omega( 2) = 0.0639288721978045180823380273793521411108D+00
      Omega( 3) = 0.1146131231768703616989810330217824940746D+00
      Omega( 4) = 0.1910990220813975077864205545652431794679D+00
      Omega( 5) = 0.3152112267330760065394551494577513039985D+00
      Omega( 6) = 0.5200333749196062467891679692577611149318D+00
      Omega( 7) = 0.8571142202739547632855607317559076818725D+00
      Omega( 8) = 1.4166433883580459902494161350539059185394D+00
      Omega( 9) = 2.3972309879960180553724186092168224604393D+00
      Omega(10) = 4.5265143423641330715570529896751850174041D+00
C
      Alpha( 1) = 0.0099118398530477141314073454003708008031D+00
      Alpha( 2) = 0.0540446717644748341431780273846907647339D+00
      Alpha( 3) = 0.1418346518220869974035791907795989175156D+00
      Alpha( 4) = 0.2917769967006175313559059730783573627377D+00
      Alpha( 5) = 0.5397886064780781045194420209032415414185D+00
      Alpha( 6) = 0.9489117072037067764827059090926297812985D+00
      Alpha( 7) = 1.6235559881076618259088117923738536774181D+00
      Alpha( 8) = 2.7363110644426438824126923687884982427931D+00
      Alpha( 9) = 4.5934743313426938901046503449521196671412D+00
      Alpha(10) = 7.8839620327921848055852116488040337571874D+00
C
      Xi( 1) = 1.007397524921629083536830928658D+00
      Xi( 2) = 1.067308174745297444981767531758D+00
      Xi( 3) = 1.191052363596173063612151943325D+00
      Xi( 4) = 1.386654455349809759155156607680D+00
      Xi( 5) = 1.666761037061785107179709419789D+00
      Xi( 6) = 2.049681437360226300995955095452D+00
      Xi( 7) = 2.561013762777291654683387012170D+00
      Xi( 8) = 3.235848668315498850378900130842D+00
      Xi( 9) = 4.121612128283250076499905079075D+00
      Xi(10) = 5.281726961622831101501873707349D+00
      Xi(11) = 6.800205251162270153695438290598D+00
      Xi(12) = 8.786900637551380829903813207693D+00
      Xi(13) = 11.382148526181198749246403778557D+00
      Xi(14) = 14.757044191321142191826443212932D+00
      Xi(15) = 19.099655989294517563492625811250D+00
      Xi(16) = 24.564947661652084267072670975551D+00
      Xi(17) = 31.145527462279573546397037908662D+00
      Xi(18) = 38.410994668580101859406417119658D+00
      Xi(19) = 45.162035093038253464464704123316D+00
      Xi(20) = 49.415398342449165687562118165488D+00
      GOTO 9999
C
C     ----- K = 10 , R = 6E1 -----
C
 1016 CONTINUE
C
      RIni = 6.0D+01
C
      Omega( 1) = 0.0222692249040388280981964813642903777691D+00
      Omega( 2) = 0.0558880663046408164463683867356769141566D+00
      Omega( 3) = 0.1014636556552278832694230503719268199347D+00
      Omega( 4) = 0.1722689305543891316927331350261631826015D+00
      Omega( 5) = 0.2899119020042486984762783452929113536811D+00
      Omega( 6) = 0.4872765569924375810494879962542214002497D+00
      Omega( 7) = 0.8161879928937111060597772715929920650524D+00
      Omega( 8) = 1.3677291102282897400713168600816516118357D+00
      Omega( 9) = 2.3414239803231609166056514936826715711504D+00
      Omega(10) = 4.4631975323559974601697608598271926894085D+00
C
      Alpha( 1) = 0.0086040646080336160600617306098980563434D+00
      Alpha( 2) = 0.0470627087261883153550430774342272499666D+00
      Alpha( 3) = 0.1242996631171070397297857661189635081200D+00
      Alpha( 4) = 0.2583066428451812494902859601575428882825D+00
      Alpha( 5) = 0.4843311800515679819837452230180119272518D+00
      Alpha( 6) = 0.8645073678078467050963154538756327838200D+00
      Alpha( 7) = 1.5023398624500846272903850642244094615307D+00
      Alpha( 8) = 2.5701128719246770133483731557433316083916D+00
      Alpha( 9) = 4.3748135182289043707513920278273644726141D+00
      Alpha(10) = 7.6060959473532058099966290942717250800342D+00
C
      Xi( 1) = 1.007805439364083889943260463617D+00
      Xi( 2) = 1.071066197625785616364020580438D+00
      Xi( 3) = 1.201978120630536587599458686615D+00
      Xi( 4) = 1.409534003621606906863673869257D+00
      Xi( 5) = 1.707957933545320009394857807550D+00
      Xi( 6) = 2.117969744747726989131039743164D+00
      Xi( 7) = 2.668766664277351642348828919538D+00
      Xi( 8) = 3.400728088857348286634663114825D+00
      Xi( 9) = 4.368971642621038846063252769625D+00
      Xi(10) = 5.648069605065367200773945866032D+00
      Xi(11) = 7.338201033536233721787239714374D+00
      Xi(12) = 9.572659536432813427719867860333D+00
      Xi(13) = 12.525630984383066084814073448683D+00
      Xi(14) = 16.416380028637400057336948044906D+00
      Xi(15) = 21.498893484353131022590588372623D+00
      Xi(16) = 28.009491746673664291575334139850D+00
      Xi(17) = 36.013714095210152738024689256235D+00
      Xi(18) = 45.064676508108325404811100867164D+00
      Xi(19) = 53.686820000316714349142266371473D+00
      Xi(20) = 59.230909829919811071530233448357D+00
      GOTO 9999
C
C     ----- K = 10 , R = 7E1 -----
C
 1017 CONTINUE
C
      RIni = 7.0D+01
C
      Omega( 1) = 0.0197607384753493559913451122281302652084D+00
      Omega( 2) = 0.0498802316575754729679071514114596652689D+00
      Omega( 3) = 0.0915913970615895810505752150376679310284D+00
      Omega( 4) = 0.1580208841785472934673680359130010231183D+00
      Omega( 5) = 0.2705366518430575556019169330967955033884D+00
      Omega( 6) = 0.4618333919085841772738786764618268421145D+00
      Omega( 7) = 0.7839904120304717415165969562895753597331D+00
      Omega( 8) = 1.3288768434592462668606197007115099495422D+00
      Omega( 9) = 2.2968563750443367990107523812781664673821D+00
      Omega(10) = 4.4126139217534362389934454729001345185679D+00
C
      Alpha( 1) = 0.0076300405792137536050681514282478801192D+00
      Alpha( 2) = 0.0418550058743523850566224404207149589752D+00
      Alpha( 3) = 0.1111879397803506057252460013928807569528D+00
      Alpha( 4) = 0.2331854554385689059001597958631357698778D+00
      Alpha( 5) = 0.4424728513495298915230314468205463640516D+00
      Alpha( 6) = 0.8003165994032521342056024638011990646191D+00
      Alpha( 7) = 1.4093598104639633360277845008923236491682D+00
      Alpha( 8) = 2.4415582485058356129563328362763741097297D+00
      Alpha( 9) = 4.2044568362569176688675609554479706275743D+00
      Alpha(10) = 7.3884269231485020785858697234260716868448D+00
C
      Xi( 1) = 1.008150014524303845044397864861D+00
      Xi( 2) = 1.074244696826699676161717689205D+00
      Xi( 3) = 1.211241091789887549981345749028D+00
      Xi( 4) = 1.428996356505864060186181130341D+00
      Xi( 5) = 1.743149796067134401082905637015D+00
      Xi( 6) = 2.176601195846597373894162896768D+00
      Xi( 7) = 2.761831104162296402501977632404D+00
      Xi( 8) = 3.544087200266895613682410770551D+00
      Xi( 9) = 4.585635727511376983755314595825D+00
      Xi(10) = 5.971522290488774669780375203842D+00
      Xi(11) = 7.817294699650083355125379513950D+00
      Xi(12) = 10.278832317284462163561875591711D+00
      Xi(13) = 13.563459418203007213546418263661D+00
      Xi(14) = 17.938583992132021224449878182838D+00
      Xi(15) = 23.725978660722990391113151886771D+00
      Xi(16) = 31.249706910884162639613359502277D+00
      Xi(17) = 40.663158133347690439268751205759D+00
      Xi(18) = 51.527416320312516485879772432099D+00
      Xi(19) = 62.104689349878719990122322158754D+00
      Xi(20) = 69.031045631153914177158270604195D+00
      GOTO 9999
C
C     ----- K = 10 , R = 8E1 -----
C
 1018 CONTINUE
C
      RIni = 8.0D+01
C
      Omega( 1) = 0.0178119645779939205493930774737432187749D+00
      Omega( 2) = 0.0452010429992078838318486470881918393161D+00
      Omega( 3) = 0.0838686860067898385246972295381517881196D+00
      Omega( 4) = 0.1467933491672222748726343855008025229836D+00
      Omega( 5) = 0.2550997772518251667029417356635789815300D+00
      Omega( 6) = 0.4413114002778232506502258808422922697901D+00
      Omega( 7) = 0.7577390458093131011879303726708201338624D+00
      Omega( 8) = 1.2969458549711413536094720133640123549412D+00
      Omega( 9) = 2.2600630049093749383750523440639312866551D+00
      Omega(10) = 4.3708423865441739823364808437133888219250D+00
C
      Alpha( 1) = 0.0068735846308498890855135895257971290206D+00
      Alpha( 2) = 0.0378052651777488056238627801841634124003D+00
      Alpha( 3) = 0.1009686477595863478192708337366489956821D+00
      Alpha( 4) = 0.2135380826836232647045893395798543679120D+00
      Alpha( 5) = 0.4095631635599569478529708571601375410864D+00
      Alpha( 6) = 0.7494985795520889645558379466017129288957D+00
      Alpha( 7) = 1.3351857641416567897864331460056064315722D+00
      Alpha( 8) = 2.3382521089266642320102229657408088314696D+00
      Alpha( 9) = 4.0667052456180286103097631933422917427379D+00
      Alpha(10) = 7.2115952445320296061714859803259969339706D+00
C
      Xi( 1) = 1.008447777913406502949857090901D+00
      Xi( 2) = 1.076994369922411397700462432070D+00
      Xi( 3) = 1.219270759862891835112290450649D+00
      Xi( 4) = 1.445915848636912761325523657518D+00
      Xi( 5) = 1.773854178956438908403188403629D+00
      Xi( 6) = 2.227978804389550490780078084718D+00
      Xi( 7) = 2.843796420245597696835337941934D+00
      Xi( 8) = 3.671075196579713069769401734277D+00
      Xi( 9) = 4.778771840255771170651033497023D+00
      Xi(10) = 6.261827428427214680371759580879D+00
      Xi(11) = 8.250454285898860409299848672759D+00
      Xi(12) = 10.922326929336661857465606839668D+00
      Xi(13) = 14.517172348889335643795650554466D+00
      Xi(14) = 19.350245010161896179468055123607D+00
      Xi(15) = 25.812262812914589486454342015520D+00
      Xi(16) = 34.319794162066901945656871575352D+00
      Xi(17) = 45.126350830342776623838041416548D+00
      Xi(18) = 57.823101718997251711323892564565D+00
      Xi(19) = 70.426307830426652024924383255211D+00
      Xi(20) = 78.817056210728345402860561819125D+00
      GOTO 9999
C
C     ----- K = 10 , R = 9E1 -----
C
 1019 CONTINUE
C
      RIni = 9.0D+01
C
      Omega( 1) = 0.0162498298873089111255316576223894919906D+00
      Omega( 2) = 0.0414415425972865049722947128341132838614D+00
      Omega( 3) = 0.0776387647291330781353872053651699047805D+00
      Omega( 4) = 0.1376735143936578371714031868155991844560D+00
      Omega( 5) = 0.2424335958616652716941320191978981313241D+00
      Omega( 6) = 0.4242897948094856199201425905886964073943D+00
      Omega( 7) = 0.7357627394488079689547667394666774498546D+00
      Omega( 8) = 1.2700319870608946222576701723383507669496D+00
      Omega( 9) = 2.2289313489910051518591571628746805799892D+00
      Omega(10) = 4.3354907748034642064709209829942437863792D+00
C
      Alpha( 1) = 0.0062673829680089525943469932686702428271D+00
      Alpha( 2) = 0.0345560641043288385510564176411607562045D+00
      Alpha( 3) = 0.0927525124197384534383741330754169496231D+00
      Alpha( 4) = 0.1976907281888568097740627171887695112673D+00
      Alpha( 5) = 0.3828877720426687085257793502490741843758D+00
      Alpha( 6) = 0.7080447357505527855082473609460436136942D+00
      Alpha( 7) = 1.2742625091008633055836979375285977766907D+00
      Alpha( 8) = 2.2528493904765100892383794528583962346602D+00
      Alpha( 9) = 3.9522041796110818614919807378882410375809D+00
      Alpha(10) = 7.0640128564798895837231751393403555994155D+00
C
      Xi( 1) = 1.008709550799474295615203167653D+00
      Xi( 2) = 1.079414015510788574540509965338D+00
      Xi( 3) = 1.226349205084010790772590382414D+00
      Xi( 4) = 1.460868137240908749417862833031D+00
      Xi( 5) = 1.801074201916648020462517754137D+00
      Xi( 6) = 2.273700068589444158980231169842D+00
      Xi( 7) = 2.917062160450168891876951415831D+00
      Xi( 8) = 3.785154229053986666645587044755D+00
      Xi( 9) = 4.953234030557328548306783577715D+00
      Xi(10) = 6.525632508391738927349540189127D+00
      Xi(11) = 8.646604435005628950389189313341D+00
      Xi(12) = 11.514889775118924520463936644887D+00
      Xi(13) = 15.401887752234283438142192768083D+00
      Xi(14) = 20.670270248651332094941279748923D+00
      Xi(15) = 27.780358779898185700094059846776D+00
      Xi(16) = 37.244886866785996006046222106534D+00
      Xi(17) = 49.427722187554696847699986506086D+00
      Xi(18) = 63.970150072867874503634855187784D+00
      Xi(19) = 78.660058441117475559745120250454D+00
      Xi(20) = 88.589934153812753173107275017628D+00
      GOTO 9999
C
C     ----- K = 10 , R = 1E2 -----
C
 1021 CONTINUE
C
      RIni = 1.0D+02
C
      Omega( 1) = 0.0149667333303671774160669197782647099615D+00
      Omega( 2) = 0.0383470287811951085504445432827758111216D+00
      Omega( 3) = 0.0724914643973222100692215506823057324937D+00
      Omega( 4) = 0.1300892818709850394054503702756520056028D+00
      Omega( 5) = 0.2318018689487819192392159539650720034842D+00
      Omega( 6) = 0.4098643132422220099664224995938965889763D+00
      Omega( 7) = 0.7169870732466540024906825889772221671592D+00
      Omega( 8) = 1.2469010750155313586558583627272867033753D+00
      Omega( 9) = 2.2020857411608898814361912066317472636001D+00
      Omega(10) = 4.3050004878976240942252406984636081688222D+00
C
      Alpha( 1) = 0.0057695950967900246555054735806550292310D+00
      Alpha( 2) = 0.0318850373523618157524438330302496424906D+00
      Alpha( 3) = 0.0859854511862230953773761008007703310341D+00
      Alpha( 4) = 0.1845984841571087922094127412031561163985D+00
      Alpha( 5) = 0.3607477727164535910564773651953984767715D+00
      Alpha( 6) = 0.6734363415369500948455712407980655598294D+00
      Alpha( 7) = 1.2230817507171462061232522500198172110686D+00
      Alpha( 8) = 2.1806860194693055696787270525405233456695D+00
      Alpha( 9) = 3.8549835889359444251241887391401519380452D+00
      Alpha(10) = 6.9382543120618274915152590143208044537460D+00
C
      Xi( 1) = 1.008942800663505860595328678109D+00
      Xi( 2) = 1.081571842562595240776375515868D+00
      Xi( 3) = 1.232671760475624759558453680253D+00
      Xi( 4) = 1.474253298620731967671741713399D+00
      Xi( 5) = 1.825509665946051017482686107396D+00
      Xi( 6) = 2.314882219289935853844333846219D+00
      Xi( 7) = 2.983314428037550270107758354499D+00
      Xi( 8) = 3.888772807784422466627002878958D+00
      Xi( 9) = 5.112476240397841844437243352672D+00
      Xi(10) = 6.767702546743285968357944915041D+00
      Xi(11) = 9.012185726495320496447216651603D+00
      Xi(12) = 12.065068387750407239052130581314D+00
      Xi(13) = 16.228708919855096235243530067294D+00
      Xi(14) = 21.912686087031235649541516252725D+00
      Xi(15) = 29.647268804459919185750882419939D+00
      Xi(16) = 40.044179128258341238827400232481D+00
      Xi(17) = 53.586313012247669108900627321646D+00
      Xi(18) = 69.983253115838594211883805229490D+00
      Xi(19) = 86.812756991639693997242499534650D+00
      Xi(20) = 98.350485614697928909988533519027D+00
      GOTO 9999
C
C     ----- K = 10 , R = 2E2 -----
C
 1022 CONTINUE
C
      RIni = 2.0D+02
C
      Omega( 1) = 0.0086915765856702453875327267322770108393D+00
      Omega( 2) = 0.0230969157217989523684319695673305972150D+00
      Omega( 3) = 0.0467619442529359648299560042045142083111D+00
      Omega( 4) = 0.0912152273160531483554608507546834772484D+00
      Omega( 5) = 0.1754696414861878131220785867472322649974D+00
      Omega( 6) = 0.3309435153894120798729380078517792185266D+00
      Omega( 7) = 0.6115252378444786909069547098383168304281D+00
      Omega( 8) = 1.1144275238224498491721933812748090986133D+00
      Omega( 9) = 2.0466210507551132822667144806061401141051D+00
      Omega(10) = 4.1283173287555058033476695911190290644299D+00
C
      Alpha( 1) = 0.0033373898429685276293693579766422296906D+00
      Alpha( 2) = 0.0187829201025006288636507280689902366078D+00
      Alpha( 3) = 0.0525560175455382537018861680011561077208D+00
      Alpha( 4) = 0.1191576576823400992268279716124190770188D+00
      Alpha( 5) = 0.2481220755782996451618924865778481603229D+00
      Alpha( 6) = 0.4935977722085605890313484561682244589065D+00
      Alpha( 7) = 0.9512698505332244500913921325668809458875D+00
      Alpha( 8) = 1.7897496715257404193907819767161981872050D+00
      Alpha( 9) = 3.3196100790475945492303405659484383249946D+00
      Alpha(10) = 6.2374001617059706020253229130645422628731D+00
C
      Xi( 1) = 1.010443336959742160639438846026D+00
      Xi( 2) = 1.095495276973907675674771433627D+00
      Xi( 3) = 1.273697812479731929485166075366D+00
      Xi( 4) = 1.561788189507271183698033922838D+00
      Xi( 5) = 1.986893817861121977886955380921D+00
      Xi( 6) = 2.590149845576304480417734188435D+00
      Xi( 7) = 3.432417010925453394352621638852D+00
      Xi( 8) = 4.602431687100476793116088369828D+00
      Xi( 9) = 6.228688173480221028539827177184D+00
      Xi(10) = 8.497306434165050245503614245735D+00
      Xi(11) = 11.678924840204528626599889129523D+00
      Xi(12) = 16.168921976974195827900238064956D+00
      Xi(13) = 22.546738410295669833433018780511D+00
      Xi(14) = 31.659698516862932566109933851806D+00
      Xi(15) = 44.727628056492924835119140425377D+00
      Xi(16) = 63.420391313750978063562557807131D+00
      Xi(17) = 89.695670287561551518862312804004D+00
      Xi(18) = 124.670261729711519121943652521622D+00
      Xi(19) = 164.812157020091428330510474609127D+00
      Xi(20) = 195.389707710008929869571758786151D+00
      GOTO 9999
C
C     ----- K = 10 , R = 3E2 -----
C
 1023 CONTINUE
C
      RIni = 3.0D+02
C
      Omega( 1) = 0.0063194806278197052362891983183501576349D+00
      Omega( 2) = 0.0172560047833030945709404139695730684423D+00
      Omega( 3) = 0.0366472048428242284041141552802800518407D+00
      Omega( 4) = 0.0752354171213032440356958605032211551134D+00
      Omega( 5) = 0.1510870545398002325309652280060923601468D+00
      Omega( 6) = 0.2952035614099551131639213863433468532094D+00
      Omega( 7) = 0.5620186886646714090960859355750045551758D+00
      Omega( 8) = 1.0505752343277138402760986779505003596569D+00
      Omega( 9) = 1.9705361918601552558482845056708754327701D+00
      Omega(10) = 4.0417630245914110847550715366338636158616D+00
C
      Alpha( 1) = 0.0024194723389312347069082440292192570652D+00
      Alpha( 2) = 0.0138046334311983167548093303853007363102D+00
      Alpha( 3) = 0.0396925971499723762111360488707978433354D+00
      Alpha( 4) = 0.0934222984881429462702398178186591337635D+00
      Alpha( 5) = 0.2024464290319135188056650850874085278974D+00
      Alpha( 6) = 0.4181018464748283449924134286268539995035D+00
      Alpha( 7) = 0.8332962399975895222744431967143441397639D+00
      Alpha( 8) = 1.6150345745196448490757848426291332089022D+00
      Alpha( 9) = 3.0746551660908783176924530256002299211104D+00
      Alpha(10) = 5.9113092020828519575562087773334951634752D+00
C
      Xi( 1) = 1.011281696362410611089895084991D+00
      Xi( 2) = 1.103306398274419302838952883228D+00
      Xi( 3) = 1.296889679570163099865502676078D+00
      Xi( 4) = 1.611794637885313709367920975701D+00
      Xi( 5) = 2.080320702954512547024068513934D+00
      Xi( 6) = 2.752091846962221399200063109980D+00
      Xi( 7) = 3.701643236381298505792425412864D+00
      Xi( 8) = 5.039415138300416063723424775489D+00
      Xi( 9) = 6.928276783724347475095645121357D+00
      Xi(10) = 9.609094668843139655858842651526D+00
      Xi(11) = 13.440420023547740535976857678691D+00
      Xi(12) = 18.960193433730371682238713404445D+00
      Xi(13) = 26.981585834209268605007880914570D+00
      Xi(14) = 38.739364903289162822552782472485D+00
      Xi(15) = 56.100879741652382719230285346157D+00
      Xi(16) = 81.813805024382479445799187800503D+00
      Xi(17) = 119.549667143381229877929783356194D+00
      Xi(18) = 172.649731371664969442614179229167D+00
      Xi(19) = 238.035976555240286464676380262517D+00
      Xi(20) = 291.598770626314468118467360113755D+00
      GOTO 9999
C
C     ----- K = 10 , R = 4E2 -----
C
 1024 CONTINUE
C
      RIni = 4.0D+02
C
      Omega( 1) = 0.0050414887113661361037624832320130918717D+00
      Omega( 2) = 0.0140800054500074398739956657404526207955D+00
      Omega( 3) = 0.0310367263079246550432213147888149684661D+00
      Omega( 4) = 0.0660863849445842620891427253437822031401D+00
      Omega( 5) = 0.1366765837125404271286009325048205198527D+00
      Omega( 6) = 0.2735256663780587675692621690837214032399D+00
      Omega( 7) = 0.5313705135370472408692064936186483237179D+00
      Omega( 8) = 1.0104385988466745368999713527990991224215D+00
      Omega( 9) = 1.9222796166689597670328495948588454211858D+00
      Omega(10) = 3.9868277184968530305053668882919737370685D+00
C
      Alpha( 1) = 0.0019254909170795910657038692770023913248D+00
      Alpha( 2) = 0.0111128914345121174053033314896299366126D+00
      Alpha( 3) = 0.0326715796695796239417029088736299513585D+00
      Alpha( 4) = 0.0791423112744457488463385372301495124248D+00
      Alpha( 5) = 0.1765489795507469377686430786478233301295D+00
      Alpha( 6) = 0.3743260077419205072524286542767768537487D+00
      Alpha( 7) = 0.7634651418561960165075179485238976440087D+00
      Alpha( 8) = 1.5097794912507614131184224826043305256462D+00
      Alpha( 9) = 2.9250174225165053239379464589120516393450D+00
      Alpha(10) = 5.7101524172963629357065162839290906049428D+00
C
      Xi( 1) = 1.011853390856991647133063250141D+00
      Xi( 2) = 1.108646476666356957751710665683D+00
      Xi( 3) = 1.312818245875861967899948801186D+00
      Xi( 4) = 1.646359283871812145328729359850D+00
      Xi( 5) = 2.145420700521415087068041738227D+00
      Xi( 6) = 2.866047574432412507836748227774D+00
      Xi( 7) = 3.893283466834573220164902873641D+00
      Xi( 8) = 5.354529532742844356429240404438D+00
      Xi( 9) = 7.440034709028623380582673219052D+00
      Xi(10) = 10.435182827798015401526976475566D+00
      Xi(11) = 14.771615580191396934449377109644D+00
      Xi(12) = 21.108666358851921747777358184806D+00
      Xi(13) = 30.463805408628496824638420825693D+00
      Xi(14) = 44.420598469816782581653935579880D+00
      Xi(15) = 65.450058006619073823484367125047D+00
      Xi(16) = 97.351899309879051984395470498157D+00
      Xi(17) = 145.591442086941418629364441983398D+00
      Xi(18) = 216.187417830411720928718288803339D+00
      Xi(19) = 307.737175259318556874710282045271D+00
      Xi(20) = 387.133073207267179111079613562652D+00
      GOTO 9999
C
C     ----- K = 10 , R = 5E2 -----
C
 1025 CONTINUE
C
      RIni = 5.0D+02
C
      Omega( 1) = 0.0042324685609204748920790582593365325437D+00
      Omega( 2) = 0.0120546370748870361851306522681104294037D+00
      Omega( 3) = 0.0273986415737904902802037250542088742122D+00
      Omega( 4) = 0.0600063698043882148383293580196085059697D+00
      Omega( 5) = 0.1268815804111528272178806286962782223782D+00
      Omega( 6) = 0.2585247942583658795339274921731842482586D+00
      Omega( 7) = 0.5098606010576789215687186496150928860516D+00
      Omega( 8) = 0.9819685015827939782379410460499968849035D+00
      Omega( 9) = 1.8878328906613232435716953627746761412709D+00
      Omega(10) = 3.9475916782101656337874984181723903020611D+00
C
      Alpha( 1) = 0.0016130554450903338090927353529711874014D+00
      Alpha( 2) = 0.0094040709862812426859680339438773977179D+00
      Alpha( 3) = 0.0281792896456298726902417331810490885857D+00
      Alpha( 4) = 0.0698808951661137112344084412751676893549D+00
      Alpha( 5) = 0.1594696974511107449390227841501399552726D+00
      Alpha( 6) = 0.3449733568032465537314505688026855523276D+00
      Alpha( 7) = 0.7159420334350728644823985691303391831752D+00
      Alpha( 8) = 1.4372483830045267496483551994401750562247D+00
      Alpha( 9) = 2.8208885751505575071110265561458163574571D+00
      Alpha(10) = 5.5692173765763346026010094380609416475636D+00
C
      Xi( 1) = 1.012281851750328150402538640584D+00
      Xi( 2) = 1.112655829018262414881190736082D+00
      Xi( 3) = 1.324817075799919270065636844880D+00
      Xi( 4) = 1.672514848849586383385326660012D+00
      Xi( 5) = 2.194966480380893188899607548770D+00
      Xi( 6) = 2.953386524695425789611633038412D+00
      Xi( 7) = 4.041374209184313866585758212580D+00
      Xi( 8) = 5.600307202630111662756268176011D+00
      Xi( 9) = 7.843307547048567834879762017408D+00
      Xi(10) = 11.093509658933041624705173600063D+00
      Xi(11) = 15.845547108056645307753995677302D+00
      Xi(12) = 22.865211568856501675450765276310D+00
      Xi(13) = 33.352580843064771526362832787527D+00
      Xi(14) = 49.209705705642749529876711633847D+00
      Xi(15) = 73.472708373403977248627949592219D+00
      Xi(16) = 110.957993990505837049453941034471D+00
      Xi(17) = 168.946615691431218028717431423047D+00
      Xi(18) = 256.413208477791467743145403801464D+00
      Xi(19) = 374.596819945121224632034184764962D+00
      Xi(20) = 482.078964309526471265776592645125D+00
      GOTO 9999
C
C     ----- K = 10 , R = 6E2 -----
C
 1026 CONTINUE
C
      RIni = 6.0D+02
C
      Omega( 1) = 0.0036700303900818465139500916755502246680D+00
      Omega( 2) = 0.0106377997883372633690940585980688748435D+00
      Omega( 3) = 0.0248163213140422915740365217632246164214D+00
      Omega( 4) = 0.0556039406929946345510464705830244724893D+00
      Omega( 5) = 0.1196656466589770147608768127711620365972D+00
      Omega( 6) = 0.2473234458266989885580865171643694111481D+00
      Omega( 7) = 0.4936263432805592477157097947015884642497D+00
      Omega( 8) = 0.9603068597394762664258907602921055968181D+00
      Omega( 9) = 1.8614967394761155464926252744994883414620D+00
      Omega(10) = 3.9175799473154924604332421944619113673980D+00
C
      Alpha( 1) = 0.0013960043685149540151937456685694860514D+00
      Alpha( 2) = 0.0082132548876076855998078587485500268173D+00
      Alpha( 3) = 0.0250272674132974799887917669374837181806D+00
      Alpha( 4) = 0.0633069328322906192752840028037386588267D+00
      Alpha( 5) = 0.1471799227081377646676090334421083127836D+00
      Alpha( 6) = 0.3235737232994003323774401259527166985208D+00
      Alpha( 7) = 0.6808942948679821898861144746639695313206D+00
      Alpha( 8) = 1.3832418428333822428642069235493750056776D+00
      Alpha( 9) = 2.7427730207536506401230780305411371955415D+00
      Alpha(10) = 5.4629433264669168906178808864382290266803D+00
C
      Xi( 1) = 1.012621454662819640043895652592D+00
      Xi( 2) = 1.115838165453791844586826920160D+00
      Xi( 3) = 1.334365050983588460919646023761D+00
      Xi( 4) = 1.693400697466688524192059139839D+00
      Xi( 5) = 2.234705295692805331313895988998D+00
      Xi( 6) = 3.023818437237688030744947598194D+00
      Xi( 7) = 4.161557714823225141535789628477D+00
      Xi( 8) = 5.801204286139057726603279752098D+00
      Xi( 9) = 8.175571683591240763032403915389D+00
      Xi(10) = 11.640672377038550208520895079545D+00
      Xi(11) = 16.746696098108507603022254706104D+00
      Xi(12) = 24.354621648984547767347041968833D+00
      Xi(13) = 35.830254270563893500417540494141D+00
      Xi(14) = 53.369584665590483127384224815160D+00
      Xi(15) = 80.540436987329664747514712530574D+00
      Xi(16) = 123.139323981935882947047478097602D+00
      Xi(17) = 190.258853088247873563965129051212D+00
      Xi(18) = 294.009996425292684962826683658932D+00
      Xi(19) = 439.055243665943953190744508674470D+00
      Xi(20) = 576.493245747046950178127389108340D+00
      GOTO 9999
C
C     ----- K = 10 , R = 7E2 -----
C
 1027 CONTINUE
C
      RIni = 7.0D+02
C
      Omega( 1) = 0.0032542049889380949915407986467591472390D+00
      Omega( 2) = 0.0095845715277716938660811010070039728248D+00
      Omega( 3) = 0.0228716368797110218745124478323393901746D+00
      Omega( 4) = 0.0522328218464833619212042085250269352059D+00
      Omega( 5) = 0.1140629704981057233851454192630203721137D+00
      Omega( 6) = 0.2385322997992477096290903429420815484718D+00
      Omega( 7) = 0.4807761161686558414333960559394753886409D+00
      Omega( 8) = 0.9430492834876176160593778952101473578296D+00
      Omega( 9) = 1.8404331172312120606026838998126038404735D+00
      Omega(10) = 3.8935670915039185988829745221906364349707D+00
C
      Alpha( 1) = 0.0012356310955025927425262089495543171846D+00
      Alpha( 2) = 0.0073310324723258707728122175219243050748D+00
      Alpha( 3) = 0.0226776888586415631861574904865053170511D+00
      Alpha( 4) = 0.0583566335293559620051072681090609250987D+00
      Alpha( 5) = 0.1378184890156283799739081810664664828892D+00
      Alpha( 6) = 0.3070966214157072800034365550603254746420D+00
      Alpha( 7) = 0.6536553453439855522629463313588615847038D+00
      Alpha( 8) = 1.3409422950604044433081316056721732365986D+00
      Alpha( 9) = 2.6812225946438997526127118176120234238624D+00
      Alpha(10) = 5.3788596769382154365833625941917262025527D+00
C
      Xi( 1) = 1.012900857627860401314646987547D+00
      Xi( 2) = 1.118459326721759505883849183938D+00
      Xi( 3) = 1.342245589900514743842037390653D+00
      Xi( 4) = 1.710687375875769244211628550278D+00
      Xi( 5) = 2.267714201987363026357183426462D+00
      Xi( 6) = 3.082579276498763421257565120470D+00
      Xi( 7) = 4.262342355741563575902836058518D+00
      Xi( 8) = 5.970655739022618686915461205800D+00
      Xi( 9) = 8.457640888002688175223431343852D+00
      Xi(10) = 12.108483930024105924838639936070D+00
      Xi(11) = 17.523177212053987253023712433730D+00
      Xi(12) = 25.648992276300671777286566133114D+00
      Xi(13) = 38.003832043215744079106555020076D+00
      Xi(14) = 57.057203179280471496553905907945D+00
      Xi(15) = 86.879552209645019580919189650103D+00
      Xi(16) = 134.212137799403177715329427144297D+00
      Xi(17) = 209.941935034556916031545092948818D+00
      Xi(18) = 329.436073714199028183990947127313D+00
      Xi(19) = 501.424828964952372711838535224160D+00
      Xi(20) = 670.416864260303763611847926995324D+00
      GOTO 9999
C
C     ----- K = 10 , R = 8E2 -----
C
 1028 CONTINUE
C
      RIni = 8.0D+02
C
      Omega( 1) = 0.0029330749052116580108893871157646482395D+00
      Omega( 2) = 0.0087672082685003208973357905278711577068D+00
      Omega( 3) = 0.0213446274367968236463239664266189343778D+00
      Omega( 4) = 0.0495476801027003352572385379976038777272D+00
      Omega( 5) = 0.1095487420615464476506841461078689548003D+00
      Omega( 6) = 0.2313858405917663885528875664099324183098D+00
      Omega( 7) = 0.4702560460707860016330971592024567939916D+00
      Omega( 8) = 0.9288450627750289506981418163888974959264D+00
      Omega( 9) = 1.8230399046731256502756543524945698209194D+00
      Omega(10) = 3.8737316934718114202432920345486877522490D+00
C
      Alpha( 1) = 0.0011118450661519535918240580681451423750D+00
      Alpha( 2) = 0.0066484639002268138156620752663578111097D+00
      Alpha( 3) = 0.0208496197653120780107457218205402327271D+00
      Alpha( 4) = 0.0544701624352677744922478306893731669902D+00
      Alpha( 5) = 0.1303955767535651947050924850945996524842D+00
      Alpha( 6) = 0.2939119642783786198780068138480814354807D+00
      Alpha( 7) = 0.6316881490532415380266886839599749237095D+00
      Alpha( 8) = 1.3066088455436533010637495388905904292187D+00
      Alpha( 9) = 2.6310144253057876860806690855199008183263D+00
      Alpha(10) = 5.3100363236689658024032578698125917071593D+00
C
      Xi( 1) = 1.013136955479273742511056588533D+00
      Xi( 2) = 1.120676286477326305339972944974D+00
      Xi( 3) = 1.348922273258674577652710402376D+00
      Xi( 4) = 1.725367830002335719076327069654D+00
      Xi( 5) = 2.295830009766047126668411793027D+00
      Xi( 6) = 3.132813074819947013862145301921D+00
      Xi( 7) = 4.348872548296264884358508506779D+00
      Xi( 8) = 6.116849595751287185351613429063D+00
      Xi( 9) = 8.702312553797890090807876095624D+00
      Xi(10) = 12.516694687068216307324064473505D+00
      Xi(11) = 18.205181659072122620987577157337D+00
      Xi(12) = 26.794097712685306787727679278532D+00
      Xi(13) = 39.942137325331399523575814924925D+00
      Xi(14) = 60.374968778913925184642641497135D+00
      Xi(15) = 92.640055205432798930698101713688D+00
      Xi(16) = 144.390157487883737968958897113225D+00
      Xi(17) = 228.282573541115812940094365046662D+00
      Xi(18) = 363.020362127012237968282093447669D+00
      Xi(19) = 561.941036587875227037436332011566D+00
      Xi(20) = 763.881072280655437523044781755743D+00
      GOTO 9999
C
C     ----- K = 10 , R = 9E2 -----
C
 1029 CONTINUE
C
      RIni = 9.0D+02
C
      Omega( 1) = 0.0026768722975028171728839701841093368095D+00
      Omega( 2) = 0.0081121894164151209087417122335639252384D+00
      Omega( 3) = 0.0201076998512748532391228870400134987051D+00
      Omega( 4) = 0.0473454129573102471413006617217966720546D+00
      Omega( 5) = 0.1058099448966131957098756218027535425108D+00
      Omega( 6) = 0.2254222361170528210309611100159443708435D+00
      Omega( 7) = 0.4614245274188429461227622940810988438898D+00
      Omega( 8) = 0.9168662209932350510600829185392512954422D+00
      Omega( 9) = 1.8083310522241474101482958114495147583511D+00
      Omega(10) = 3.8569524630728154492020909316352117457427D+00
C
      Alpha( 1) = 0.0010131325949005472269497042832331334949D+00
      Alpha( 2) = 0.0061029931115648970293762832009520913346D+00
      Alpha( 3) = 0.0193811446162218188716495079551571656395D+00
      Alpha( 4) = 0.0513226634278318588158773229379594482680D+00
      Alpha( 5) = 0.1243314878775564602441587418180102275755D+00
      Alpha( 6) = 0.2830557631852255335571603295230502794766D+00
      Alpha( 7) = 0.6134788317812593461749066936494756419052D+00
      Alpha( 8) = 1.2779919332611943687434685745785145627451D+00
      Alpha( 9) = 2.5889883766039190108286527358671946785762D+00
      Alpha(10) = 5.2522618822502933752169318459834812529152D+00
C
      Xi( 1) = 1.013340506069180610372766626615D+00
      Xi( 2) = 1.122589180203960424787867911078D+00
      Xi( 3) = 1.354691599735056251529660698019D+00
      Xi( 4) = 1.738078713455105556834855151660D+00
      Xi( 5) = 2.320235898215292067326559477536D+00
      Xi( 6) = 3.176555398741145603983340728327D+00
      Xi( 7) = 4.424497164492826078736192885188D+00
      Xi( 8) = 6.245151350824582261057199561449D+00
      Xi( 9) = 8.918035302609499784948221545733D+00
      Xi(10) = 12.878450829603987428523259595181D+00
      Xi(11) = 18.812984656048486938398345280632D+00
      Xi(12) = 27.820980821978816168241199768829D+00
      Xi(13) = 41.692348392723356135386447718361D+00
      Xi(14) = 63.393913446195932721860533121117D+00
      Xi(15) = 97.927487715505802341742214167652D+00
      Xi(16) = 153.826154826311490259049818973835D+00
      Xi(17) = 245.489989054823343678224389918796D+00
      Xi(18) = 395.010353992267319478415288358519D+00
      Xi(19) = 620.788700367442668193174171165083D+00
      Xi(20) = 856.910708122485446525118391036813D+00
      GOTO 9999
C
C     ----- K = 10 , R = 1E3 -----
C
 1031 CONTINUE
C
      RIni = 1.0D+03
C
      Omega( 1) = 0.0024672464939504652393320415079344676279D+00
      Omega( 2) = 0.0075740393679528136391924269567310634699D+00
      Omega( 3) = 0.0190813545239317744910730999946535391132D+00
      Omega( 4) = 0.0454979068375473446648820340099117487398D+00
      Omega( 5) = 0.1026467684099852501974654145344700850728D+00
      Omega( 6) = 0.2203438450309990335784825557507993210038D+00
      Omega( 7) = 0.4538648781925274396217753064552979935797D+00
      Omega( 8) = 0.9065719038631264092235190199531302823743D+00
      Omega( 9) = 1.7956602125219735557838315886947100352700D+00
      Omega(10) = 3.8424941566456491484106428035971703138785D+00
C
      Alpha( 1) = 0.0009323990731474346103138267389132815000D+00
      Alpha( 2) = 0.0056560052513406161738321840937171458563D+00
      Alpha( 3) = 0.0181719764811097294023625172906810654183D+00
      Alpha( 4) = 0.0487116706076877454791506857900706961573D+00
      Alpha( 5) = 0.1192619458576450608360106260541488865101D+00
      Alpha( 6) = 0.2739171355714497416503072635407534107799D+00
      Alpha( 7) = 0.5980606335541609821517504419485788957900D+00
      Alpha( 8) = 1.2536454845296414365536305512982551135792D+00
      Alpha( 9) = 2.5531023689261667071817574203862477588700D+00
      Alpha(10) = 5.2028048639403972550065979874034383101389D+00
C
      Xi( 1) = 1.013518771925512285039111537444D+00
      Xi( 2) = 1.124265631526049065186563524765D+00
      Xi( 3) = 1.359754378828921279800251431524D+00
      Xi( 4) = 1.749252064327726823608592787718D+00
      Xi( 5) = 2.341737351042612545618318287133D+00
      Xi( 6) = 3.215197438739649914238999395266D+00
      Xi( 7) = 4.491518208540886354651622269785D+00
      Xi( 8) = 6.359268176451460747537064444046D+00
      Xi( 9) = 9.110684492029050861316985532312D+00
      Xi(10) = 13.202955683170126036096725385960D+00
      Xi(11) = 19.360892828771091835060591535012D+00
      Xi(12) = 28.751732205868852112334854709275D+00
      Xi(13) = 43.288356603745349986123169827579D+00
      Xi(14) = 66.165637340636030923335386688677D+00
      Xi(15) = 102.819359616610749581788741835453D+00
      Xi(16) = 162.634010646464310198555658359965D+00
      Xi(17) = 261.723013102670544638428395955998D+00
      Xi(18) = 425.598861167297845137236222967658D+00
      Xi(19) = 678.117414638977752505688556539098D+00
      Xi(20) = 949.526067440132668984187347405168D+00
      GOTO 9999
C
C     ----- K = 10 , R = 2E3 -----
C
 1032 CONTINUE
C
      RIni = 2.0D+03
C
      Omega( 1) = 0.0014526807816713179251917649882457073129D+00
      Omega( 2) = 0.0049297471955163571493195942292003808305D+00
      Omega( 3) = 0.0138551513028217050564071840916546829447D+00
      Omega( 4) = 0.0357468212429620454283945088502782994055D+00
      Omega( 5) = 0.0854970858922482741348684853743833400586D+00
      Omega( 6) = 0.1922337452291708614644100139301485796750D+00
      Omega( 7) = 0.4113201462352773910328106177214380068108D+00
      Omega( 8) = 0.8478938111702770348337629413570937231270D+00
      Omega( 9) = 1.7228690042716290663563083462861413863720D+00
      Omega(10) = 3.7593552654949775578344106641281996417092D+00
C
      Alpha( 1) = 0.0005421696478208119686995916880565410390D+00
      Alpha( 2) = 0.0034806730732080148226928955441580870733D+00
      Alpha( 3) = 0.0121808446627809141984838674395430491160D+00
      Alpha( 4) = 0.0354347079465288640331293874147222666693D+00
      Alpha( 5) = 0.0928087866746072898794335179206704822263D+00
      Alpha( 6) = 0.2251455517645407656819427077987683105675D+00
      Alpha( 7) = 0.5142100271408126002073597782260350186334D+00
      Alpha( 8) = 1.1191922258880422136112162134757852527400D+00
      Alpha( 9) = 2.3525743224761141579766399178907931855065D+00
      Alpha(10) = 4.9242248867504468650552196251624081924092D+00
C
      Xi( 1) = 1.014597792749656095289204071896D+00
      Xi( 2) = 1.134436845099320007208831828649D+00
      Xi( 3) = 1.390599483503960958718885521090D+00
      Xi( 4) = 1.817719358871181273176569070937D+00
      Xi( 5) = 2.474464282290303179497395480446D+00
      Xi( 6) = 3.455904036003200488188016881175D+00
      Xi( 7) = 4.913482237359631337163951014801D+00
      Xi( 8) = 7.086528454205757836855544828936D+00
      Xi( 9) = 10.355243055451960874392691724921D+00
      Xi(10) = 15.331357867549850092919738742125D+00
      Xi(11) = 23.015738839648443813529210544289D+00
      Xi(12) = 35.078615893121463394088710430196D+00
      Xi(13) = 54.370071151073339515980009650775D+00
      Xi(14) = 85.879927506270057162296449604355D+00
      Xi(15) = 138.589860380804692760459140288276D+00
      Xi(16) = 229.159680799216293481390671615827D+00
      Xi(17) = 389.249173017283270181732035553068D+00
      Xi(18) = 678.447113000007039851357859561176D+00
      Xi(19) = 1188.115962211603321430075652642699D+00
      Xi(20) = 1855.702942249214593473105594512162D+00
      GOTO 9999
C
C     ----- K = 10 , R = 3E3 -----
C
 1033 CONTINUE
C
      RIni = 3.0D+03
C
      Omega( 1) = 0.0010738804618956168932148597825945723905D+00
      Omega( 2) = 0.0039152313311347269926616563272270266083D+00
      Omega( 3) = 0.0117284581530090255200150881759180165886D+00
      Omega( 4) = 0.0315715502645784553185058507462912302799D+00
      Omega( 5) = 0.0778819587605281879913693247885486670157D+00
      Omega( 6) = 0.1794002209408897666648207216977262135060D+00
      Omega( 7) = 0.3914628342219864378054381554017382427446D+00
      Omega( 8) = 0.8200399624470611673163374100159472845917D+00
      Omega( 9) = 1.6879552117417538126308354518023691070994D+00
      Omega(10) = 3.7194242684483121124793675305042484069418D+00
C
      Alpha( 1) = 0.0003967302317420766487669641541446638455D+00
      Alpha( 2) = 0.0026604835814450934334069928926397308189D+00
      Alpha( 3) = 0.0098489286120088280087307578206101688068D+00
      Alpha( 4) = 0.0300488167483034730690792842022762698662D+00
      Alpha( 5) = 0.0816621938621181960049101589138587087291D+00
      Alpha( 6) = 0.2039328886821691243984126795663058828723D+00
      Alpha( 7) = 0.4767849144334121988867119384947201865543D+00
      Alpha( 8) = 1.0579265306400313928492096238187514245510D+00
      Alpha( 9) = 2.2597542577639734308597024847742318343080D+00
      Alpha(10) = 4.7939093124023811104579551045645757767488D+00
C
      Xi( 1) = 1.015150512195101957795169422827D+00
      Xi( 2) = 1.139662893625249794190765206814D+00
      Xi( 3) = 1.406534849554869738802280465517D+00
      Xi( 4) = 1.853355656995012800520271356586D+00
      Xi( 5) = 2.544208360205743575969614123089D+00
      Xi( 6) = 3.583883042504509402461851386157D+00
      Xi( 7) = 5.140944517623982631688434041806D+00
      Xi( 8) = 7.484744461305179324701192244973D+00
      Xi( 9) = 11.048733490328873131820774933232D+00
      Xi(10) = 16.540679503715717895209746934881D+00
      Xi(11) = 25.137923544129997780252616657037D+00
      Xi(12) = 38.842667988550012486692608604244D+00
      Xi(13) = 61.145845410447943829579386942896D+00
      Xi(14) = 98.315172188442081627202107085850D+00
      Xi(15) = 161.976990552996012631692401839700D+00
      Xi(16) = 274.521366554113617375332268011334D+00
      Xi(17) = 480.753414990375269988387429975774D+00
      Xi(18) = 872.152150704680091064524560806603D+00
      Xi(19) = 1617.989737536047364030444839499978D+00
      Xi(20) = 2730.560653699768991486962477210909D+00
      GOTO 9999
C
C     ----- K = 10 , R = 4E3 -----
C
 1034 CONTINUE
C
      RIni = 4.0D+03
C
      Omega( 1) = 0.0008710258738679722909390542499950133015D+00
      Omega( 2) = 0.0033608319216372058163594838554183441559D+00
      Omega( 3) = 0.0105209015614479294216962543956350728536D+00
      Omega( 4) = 0.0291318792682550152024377118631370819912D+00
      Omega( 5) = 0.0733425279782349960833393379133759992783D+00
      Omega( 6) = 0.1716321958691351225108649367068203162034D+00
      Omega( 7) = 0.3792963222960632354121331827334628883364D+00
      Omega( 8) = 0.8028142821879096244306467378137881496514D+00
      Omega( 9) = 1.6662385945818383603143561511039649758459D+00
      Omega(10) = 3.6945668905108900519854847610545789393655D+00
C
      Alpha( 1) = 0.0003189013513191434614517324352316388070D+00
      Alpha( 2) = 0.0022180408379182630683261988335797099126D+00
      Alpha( 3) = 0.0085622083802234129730279006340043501666D+00
      Alpha( 4) = 0.0269984516983066408202083684073069314024D+00
      Alpha( 5) = 0.0752068311142086706595180807533029820888D+00
      Alpha( 6) = 0.1914251600871076061422156044944387076612D+00
      Alpha( 7) = 0.4543973763628831388783346767468174220994D+00
      Alpha( 8) = 1.0208561605370438546977820970518280319084D+00
      Alpha( 9) = 2.2031043684761797731008758916004808270372D+00
      Alpha(10) = 4.7139161935991886124945604574776325534913D+00
C
      Xi( 1) = 1.015506565858864634942573812459D+00
      Xi( 2) = 1.143035114280607693749806930139D+00
      Xi( 3) = 1.416848829360455259648333969924D+00
      Xi( 4) = 1.876516805143232096515434892403D+00
      Xi( 5) = 2.589780887553195976198694427950D+00
      Xi( 6) = 3.668062082444905891431830013971D+00
      Xi( 7) = 5.291724324827579940475064113059D+00
      Xi( 8) = 7.751056290437090507978307174497D+00
      Xi( 9) = 11.517149648139810796595994268898D+00
      Xi(10) = 17.366666817265994425187325411741D+00
      Xi(11) = 26.605671471427667244824544212634D+00
      Xi(12) = 41.483096270017238978849993813469D+00
      Xi(13) = 65.976340454713680826193122896939D+00
      Xi(14) = 107.346974824950474040885506354925D+00
      Xi(15) = 179.337929725681243811608034377514D+00
      Xi(16) = 309.080899972430674593759647450497D+00
      Xi(17) = 552.743379803463856347889304743148D+00
      Xi(18) = 1031.064031042961785633593763122917D+00
      Xi(19) = 1993.591017126431915174045172989281D+00
      Xi(20) = 3578.011512315118128224611382393050D+00
      GOTO 9999
C
C     ----- K = 10 , R = 5E3 -----
C
 1035 CONTINUE
C
      RIni = 5.0D+03
C
      Omega( 1) = 0.0007431794401095743669437292025069935475D+00
      Omega( 2) = 0.0030055420983453206422728796977025811366D+00
      Omega( 3) = 0.0097249569761531400332958945814665341345D+00
      Omega( 4) = 0.0274923139468489620731419999927203079437D+00
      Omega( 5) = 0.0702505967829803888220112709217524127325D+00
      Omega( 6) = 0.1662865450907947724451399831857578703875D+00
      Omega( 7) = 0.3708552468418009445650599525956536695048D+00
      Omega( 8) = 0.7907883037645155538744590195676664734492D+00
      Omega( 9) = 1.6510183817369136722663994221171890330879D+00
      Omega(10) = 3.6771356408478858242890735530394863417314D+00
C
      Alpha( 1) = 0.0002698635605074361434595552571144225218D+00
      Alpha( 2) = 0.0019374881805125896826202009119374802815D+00
      Alpha( 3) = 0.0077316581226501765259868437837192933770D+00
      Alpha( 4) = 0.0249919540890399625044510090177141137247D+00
      Alpha( 5) = 0.0708942348818701222867822396567216713947D+00
      Alpha( 6) = 0.1829661537692895389973189723686886054566D+00
      Alpha( 7) = 0.4391085307768620177998331044477708928753D+00
      Alpha( 8) = 0.9953449891370984821750700355380558903562D+00
      Alpha( 9) = 2.1638931456033836848735357749617946865328D+00
      Alpha(10) = 4.6583347845772465202311696241821437070030D+00
C
      Xi( 1) = 1.015761905522425848232524747683D+00
      Xi( 2) = 1.145456265549093649732358135562D+00
      Xi( 3) = 1.424269242260172810038594548576D+00
      Xi( 4) = 1.893227150506627856434670176267D+00
      Xi( 5) = 2.622778685669105851706317533711D+00
      Xi( 6) = 3.729287436722984407219336211448D+00
      Xi( 7) = 5.401967884976654192541456467325D+00
      Xi( 8) = 7.946941855860498587181550700365D+00
      Xi( 9) = 11.864030887071025224810405518383D+00
      Xi(10) = 17.983017475809047037646482003481D+00
      Xi(11) = 27.710351355899081444988940603480D+00
      Xi(12) = 43.489879502946803307289469486818D+00
      Xi(13) = 69.689071791289023578841455019983D+00
      Xi(14) = 114.380016461326117188646023237197D+00
      Xi(15) = 193.067490232188291657267065204451D+00
      Xi(16) = 336.924967640876272423344772732889D+00
      Xi(17) = 612.112609457937406176597505691461D+00
      Xi(18) = 1166.208947569273627342667509765306D+00
      Xi(19) = 2328.357007035189155885746004059911D+00
      Xi(20) = 4400.318794246789840141076410873211D+00
      GOTO 9999
C
C     ----- K = 10 , R = 6E3 -----
C
 1036 CONTINUE
C
      RIni = 6.0D+03
C
      Omega( 1) = 0.0006546460868159010347083298598722111983D+00
      Omega( 2) = 0.0027559343285720026859823156024226209482D+00
      Omega( 3) = 0.0091533655076862242054661902498913228499D+00
      Omega( 4) = 0.0262978801544948501235979525930686051538D+00
      Omega( 5) = 0.0679756840176229069013544119293168677132D+00
      Omega( 6) = 0.1623235380963315463529340193504779676914D+00
      Omega( 7) = 0.3645597507129552442266956502336938683584D+00
      Omega( 8) = 0.7817777963990977949321550810601877401496D+00
      Omega( 9) = 1.6395819121798266358291246147516062592331D+00
      Omega(10) = 3.6640322363638034132773957374240580975311D+00
C
      Alpha( 1) = 0.0002359054779922201880353551623295017825D+00
      Alpha( 2) = 0.0017421610274382283694814347045805302816D+00
      Alpha( 3) = 0.0071448724389958750840038618351579557952D+00
      Alpha( 4) = 0.0235534337944883915382909399943711648007D+00
      Alpha( 5) = 0.0677660658354808540797969453051141641708D+00
      Alpha( 6) = 0.1767741391135355176440291727346654226949D+00
      Alpha( 7) = 0.4278362266663702313991297226936083575310D+00
      Alpha( 8) = 0.9764291736433913701498875370443641941165D+00
      Alpha( 9) = 2.1346954872130372662421271545696299654082D+00
      Alpha(10) = 4.6168309999158093559143423156854169064900D+00
C
      Xi( 1) = 1.015956987048296602019724355070D+00
      Xi( 2) = 1.147307628546274119174519512399D+00
      Xi( 3) = 1.429951990212024972920143439659D+00
      Xi( 4) = 1.906050941262453609389808384389D+00
      Xi( 5) = 2.648169446245551599677936494004D+00
      Xi( 6) = 3.776553877910799129945587648827D+00
      Xi( 7) = 5.487407583017900029268704154006D+00
      Xi( 8) = 8.099432497604121565168422414160D+00
      Xi( 9) = 12.135421059281113738935464763102D+00
      Xi(10) = 18.467966908870437714104695281492D+00
      Xi(11) = 28.585123074481174617472722943035D+00
      Xi(12) = 45.090685285555316397926306493105D+00
      Xi(13) = 72.675858909161066459536648665107D+00
      Xi(14) = 120.094158812983521085471227962671D+00
      Xi(15) = 204.354962067814596077264255313821D+00
      Xi(16) = 360.146971185465807924197534362065D+00
      Xi(17) = 662.533546906581793267676516734355D+00
      Xi(18) = 1283.793870989625666045874652354541D+00
      Xi(19) = 2630.654446316201495292830259131733D+00
      Xi(20) = 5199.087330578308051798330780002289D+00
      GOTO 9999
C
C     ----- K = 10 , R = 7E3 -----
C
 1037 CONTINUE
C
      RIni = 7.0D+03
C
      Omega( 1) = 0.0005894337117817897871925262719573074266D+00
      Omega( 2) = 0.0025697219209575387010189239343771472335D+00
      Omega( 3) = 0.0087192982240147648271186265299392825767D+00
      Omega( 4) = 0.0253806036987043458811138434486065484919D+00
      Omega( 5) = 0.0662150923425636726442128007774634568250D+00
      Omega( 6) = 0.1592383500598885868158611361922183391471D+00
      Omega( 7) = 0.3596358039121044572461721039458026893954D+00
      Omega( 8) = 0.7747051418395835614347233466503439558437D+00
      Omega( 9) = 1.6305850484217997494341273467810538022604D+00
      Omega(10) = 3.6537205359673349575229395203024296279182D+00
C
      Alpha( 1) = 0.0002108892791676530489045705884024184407D+00
      Alpha( 2) = 0.0015975896285734081348582562667227924269D+00
      Alpha( 3) = 0.0067051290145370598709271604299930746862D+00
      Alpha( 4) = 0.0224625370876326169567013726208193347134D+00
      Alpha( 5) = 0.0653717748314939917488129023093179625903D+00
      Alpha( 6) = 0.1720007540745488256789990158535097464210D+00
      Alpha( 7) = 0.4190975398123096649504078226033954024388D+00
      Alpha( 8) = 0.9617003167186997781125737394525287982106D+00
      Alpha( 9) = 2.1118856540803401934495997505081277267891D+00
      Alpha(10) = 4.5843367189279305151661247919037123210728D+00
C
      Xi( 1) = 1.016112431848799921166662252325D+00
      Xi( 2) = 1.148783819519298703311360743129D+00
      Xi( 3) = 1.434488586048307485597969046509D+00
      Xi( 4) = 1.916304352019664957286829154448D+00
      Xi( 5) = 2.668513636604800630652140069898D+00
      Xi( 6) = 3.814523150955451240512808497307D+00
      Xi( 7) = 5.556251095647487081834875466413D+00
      Xi( 8) = 8.222727966325468763365835744850D+00
      Xi( 9) = 12.355717871798836730856674392953D+00
      Xi(10) = 18.863374909541553076097741836747D+00
      Xi(11) = 29.301965962163520094257984638375D+00
      Xi(12) = 46.410094597696795501451383358926D+00
      Xi(13) = 75.154152734836061185152988883829D+00
      Xi(14) = 124.872938892471809987916309125922D+00
      Xi(15) = 213.884402134001678094654685935438D+00
      Xi(16) = 379.979871483282035271589194280750D+00
      Xi(17) = 706.235618502009305785804826882668D+00
      Xi(18) = 1387.751699638245933554614452987153D+00
      Xi(19) = 2906.228803626507716018068094854243D+00
      Xi(20) = 5975.583040521315707938043715330423D+00
      GOTO 9999
C
C     ----- K = 10 , R = 8E3 -----
C
 1038 CONTINUE
C
      RIni = 8.0D+03
C
      Omega( 1) = 0.0005392552127602062085177379348061715847D+00
      Omega( 2) = 0.0024247926212958021032482688579892671221D+00
      Omega( 3) = 0.0083764286534713166290939804527193590644D+00
      Omega( 4) = 0.0246494419210904408322975562770085211639D+00
      Omega( 5) = 0.0648029218064254492278784477998776480945D+00
      Omega( 6) = 0.1567518884914159084978087579931305128866D+00
      Omega( 7) = 0.3556524160150506188183411326875216218468D+00
      Omega( 8) = 0.7689669348259188687961206143395287426756D+00
      Omega( 9) = 1.6232725366290187618209528430668342480203D+00
      Omega(10) = 3.6453370534672855701654320403903852820804D+00
C
      Alpha( 1) = 0.0001916362889846043065853641168007868989D+00
      Alpha( 2) = 0.0014858556424114500268540757402062346682D+00
      Alpha( 3) = 0.0063615903805791639624878260322188294751D+00
      Alpha( 4) = 0.0216018516951527043170173903938180437478D+00
      Alpha( 5) = 0.0634683844611125438155640153536918290911D+00
      Alpha( 6) = 0.1681839281196474854062706053015041618437D+00
      Alpha( 7) = 0.4120781965585719024689533712679789800859D+00
      Alpha( 8) = 0.9498272260709862341906267735325286594161D+00
      Alpha( 9) = 2.0934494387115830259151610936996235068364D+00
      Alpha(10) = 4.5580268727262413749369562854241166860447D+00
C
      Xi( 1) = 1.016240074424546586972541162286D+00
      Xi( 2) = 1.149996640003310312294106110276D+00
      Xi( 3) = 1.438219369245532938900514197123D+00
      Xi( 4) = 1.924747481555941825532261846288D+00
      Xi( 5) = 2.685294247801549651988653466539D+00
      Xi( 6) = 3.845907382537659943049293542749D+00
      Xi( 7) = 5.613293949872002155075667584327D+00
      Xi( 8) = 8.325176650420320748069435978778D+00
      Xi( 9) = 12.539349560191624636737439679735D+00
      Xi(10) = 19.194156819094702701861132609906D+00
      Xi(11) = 29.904126804560926226536632377417D+00
      Xi(12) = 47.523652979244679680354002115905D+00
      Xi(13) = 77.257305678500895959603944618266D+00
      Xi(14) = 128.954673044338415363596617169151D+00
      Xi(15) = 222.087597862233788223451647070306D+00
      Xi(16) = 397.217217380373791707492969749183D+00
      Xi(17) = 744.689647863577452946426404878366D+00
      Xi(18) = 1480.770111530073871342061408995505D+00
      Xi(19) = 3159.281593163338799579165083741827D+00
      Xi(20) = 6730.860033210824243887770990113495D+00
      GOTO 9999
C
C     ----- K = 10 , R = 9E3 -----
C
 1039 CONTINUE
C
      RIni = 9.0D+03
C
      Omega( 1) = 0.0004993672899010212969344114431521214348D+00
      Omega( 2) = 0.0023083851082641087179457485990125009678D+00
      Omega( 3) = 0.0080975548740404425835813419926045853003D+00
      Omega( 4) = 0.0240502434224015340449989242346817697182D+00
      Omega( 5) = 0.0636395853625634186795942751069254939011D+00
      Omega( 6) = 0.1546954156882665470701572474343521435003D+00
      Omega( 7) = 0.3523475646338251464513829230851271745451D+00
      Omega( 8) = 0.7641947978821455716427410442825873815309D+00
      Omega( 9) = 1.6171820561365553377188714057055562989262D+00
      Omega(10) = 3.6383529487866477763330363970339931256603D+00
C
      Alpha( 1) = 0.0001763278642964950008481394511735025621D+00
      Alpha( 2) = 0.0013966738489163576343957964224357093386D+00
      Alpha( 3) = 0.0060847863738081463643708594359049701694D+00
      Alpha( 4) = 0.0209025184439057381995591194598360651469D+00
      Alpha( 5) = 0.0619119562592124370580817186576760846606D+00
      Alpha( 6) = 0.1650476738341768858053760712922830578009D+00
      Alpha( 7) = 0.4062886092368531241320050995113888347987D+00
      Alpha( 8) = 0.9400053232670322994851512532488158058186D+00
      Alpha( 9) = 2.0781646409684503076526623788211622922972D+00
      Alpha(10) = 4.5361826641383027796175508861153957695933D+00
C
      Xi( 1) = 1.016347286163623462804106445390D+00
      Xi( 2) = 1.151015790821104004676422860598D+00
      Xi( 3) = 1.441356899587345926781918603599D+00
      Xi( 4) = 1.931855777473684393171805828704D+00
      Xi( 5) = 2.699441181486060780964769678647D+00
      Xi( 6) = 3.872412335802562820177147306921D+00
      Xi( 7) = 5.661566270315299894810151526414D+00
      Xi( 8) = 8.412077454143638951568340278442D+00
      Xi( 9) = 12.695525341059824706892167345273D+00
      Xi(10) = 19.476333699099736636653479315839D+00
      Xi(11) = 30.419560793419684807323277730973D+00
      Xi(12) = 48.480602311172952819523507272947D+00
      Xi(13) = 79.073001800190617187336084015215D+00
      Xi(14) = 132.497705706562297706119224471877D+00
      Xi(15) = 229.255331266340414531068603309905D+00
      Xi(16) = 412.402010579539868545539960109636D+00
      Xi(17) = 778.923674562525614584185262856408D+00
      Xi(18) = 1564.783963043916336643768261183141D+00
      Xi(19) = 3393.010515303080970150872985868773D+00
      Xi(20) = 7465.825669118156909132011378460447D+00
      GOTO 9999
C
C     ----- K = 10 , R = 1E4 -----
C
 1041 CONTINUE
C
      RIni = 1.0D+04
C
      Omega( 1) = 0.0004668500190071136037058161189026300741D+00
      Omega( 2) = 0.0022125830949226374988479443175343197581D+00
      Omega( 3) = 0.0078655446011830261521978946079314276574D+00
      Omega( 4) = 0.0235485349216305695394885416985020221148D+00
      Omega( 5) = 0.0626612189845160618488733965392167846176D+00
      Omega( 6) = 0.1529600992483176926246520380137106087659D+00
      Omega( 7) = 0.3495514360883487606104867012613013343980D+00
      Omega( 8) = 0.7601490813086587507488821324752592545337D+00
      Omega( 9) = 1.6120121524724761282376314652786675196694D+00
      Omega(10) = 3.6324233181727452956458895849678469858191D+00
C
      Alpha( 1) = 0.0001638448168262916940862950973285122602D+00
      Alpha( 2) = 0.0013236958800674285022613046645077738983D+00
      Alpha( 3) = 0.0058563634806415089364502364350367757950D+00
      Alpha( 4) = 0.0203212175461642492981981754363030034938D+00
      Alpha( 5) = 0.0606111723824622499480734971050166493001D+00
      Alpha( 6) = 0.1624157193093745306253445140343139030392D+00
      Alpha( 7) = 0.4014143723529730106919840432189161560927D+00
      Alpha( 8) = 0.9317156300794014760126987573229939698649D+00
      Alpha( 9) = 2.0652402323623246008753306934835336505785D+00
      Alpha(10) = 4.5176891627407891488012181557820667876513D+00
C
      Xi( 1) = 1.016438941992472549339515675015D+00
      Xi( 2) = 1.151887410843916417150893660715D+00
      Xi( 3) = 1.444041938087961746682054420265D+00
      Xi( 4) = 1.937944601327246360090474086579D+00
      Xi( 5) = 2.711573469892460428809846662013D+00
      Xi( 6) = 3.895176051201984182168966408533D+00
      Xi( 7) = 5.703096914696195845247833622338D+00
      Xi( 8) = 8.486990466098886826129543692332D+00
      Xi( 9) = 12.830460806532057196210905303246D+00
      Xi(10) = 19.720759648459364998460130458113D+00
      Xi(11) = 30.867341651348655255648889550457D+00
      Xi(12) = 49.314738074799366471939743306052D+00
      Xi(13) = 80.661929483772102499050671298164D+00
      Xi(14) = 135.612742585992632399216617500315D+00
      Xi(15) = 235.593109019815892993254813347903D+00
      Xi(16) = 425.923429412591868059578814609267D+00
      Xi(17) = 809.687804313004781064933013112750D+00
      Xi(18) = 1641.240732349594282579730020188435D+00
      Xi(19) = 3609.930576169334359937224121495092D+00
      Xi(20) = 8181.271691613231036743769664099091D+00
      GOTO 9999
C
C     ----- K = 10 , R = 1E5 (Best : R = 56503) -----
C
 1099 CONTINUE
C
      RIni = 1.0D+05
C
      Omega( 1) = 0.0002188017493762318239417294425826730764D+00
      Omega( 2) = 0.0014375664500890244128473657598404900071D+00
      Omega( 3) = 0.0058848743519528664629849115403487092024D+00
      Omega( 4) = 0.0191306240822222516468974857703488900995D+00
      Omega( 5) = 0.0538584874422464578561152524077693115601D+00
      Omega( 6) = 0.1370884682117943122853752357936940597938D+00
      Omega( 7) = 0.3236444502032899059047309914260992513846D+00
      Omega( 8) = 0.7222915374704975218400800074380896376169D+00
      Omega( 9) = 1.5633339391232095614897232294104867378337D+00
      Omega(10) = 3.5765361929512171882024340607841850214754D+00
C
      Alpha( 1) = 0.0000684439013883050839929452714076145436D+00
      Alpha( 2) = 0.0007529515955878675215771287905844322275D+00
      Alpha( 3) = 0.0039850355998364926202256629163972467822D+00
      Alpha( 4) = 0.0153807990762694658962519717732009283395D+00
      Alpha( 5) = 0.0492554652626580150358520746728396844105D+00
      Alpha( 6) = 0.1389707382811857045096584628574021280656D+00
      Alpha( 7) = 0.3573119289310187454190587824642832970312D+00
      Alpha( 8) = 0.8557928463574533264955450373445700051889D+00
      Alpha( 9) = 1.9457922045681717026794493863306456660212D+00
      Alpha(10) = 4.3457510815955149355814535194753034375026D+00
C
      Xi( 1) = 1.017332022553130616709937683595D+00
      Xi( 2) = 1.160396353209442369671874906167D+00
      Xi( 3) = 1.470341637704521359350171016978D+00
      Xi( 4) = 1.997855369163787341403940023277D+00
      Xi( 5) = 2.831657524964081751707792311556D+00
      Xi( 6) = 4.122147772655313542090882439339D+00
      Xi( 7) = 6.120802466554915003110054483315D+00
      Xi( 8) = 9.248005859104586343436005257956D+00
      Xi( 9) = 14.216952520405455036127395640477D+00
      Xi(10) = 22.265232010172739753744997059925D+00
      Xi(11) = 35.599346873582316214817966582018D+00
      Xi(12) = 58.285299504795977050541910813308D+00
      Xi(13) = 98.107643930079989784409644215657D+00
      Xi(14) = 170.678774856488366634676978605967D+00
      Xi(15) = 309.186027541697107778873032657430D+00
      Xi(16) = 589.257915242668196764697086109663D+00
      Xi(17) = 1201.556899697256635350939291129180D+00
      Xi(18) = 2693.276966514136449903915604409121D+00
      Xi(19) = 6992.095093521639840528081322190701D+00
      Xi(20) = 24579.752122908689996094722118868958D+00
      GOTO 9999
C
C     ===== K = 11 =====
C
 1100 CONTINUE
      GOTO (1111,1111,1111,1112,1113,1114,1115,1116,1117,1118,1119,1121,
     *      1122,1123,1124,1125,1126,1127,1128,1129,1131,1132,1133,1134,
     *      1135,1136,1137,1138,1139,1141,1199),InitR
C
C     ----- K = 11 , R = 1E1 -----
C
 1111 CONTINUE
C
      RIni = 1.0D+01
C
      Omega( 1) = 0.0775418031116303837967595156965838754104D+00
      Omega( 2) = 0.1849123891997807165503506926507171215235D+00
      Omega( 3) = 0.3038503048558095734777567686890620279883D+00
      Omega( 4) = 0.4433256105757104311240485855494242173336D+00
      Omega( 5) = 0.6164377958322638620695596567422569478367D+00
      Omega( 6) = 0.8427130345849869985067233379982809537978D+00
      Omega( 7) = 1.1516837447426178303111835021788067479065D+00
      Omega( 8) = 1.5901310218673417025501276578225429148006D+00
      Omega( 9) = 2.2427211748928490614354153143139569692721D+00
      Omega(10) = 3.3086113852215574260545033480340748610615D+00
      Omega(11) = 5.5609371683070527290528028974136987017118D+00
C
      Alpha( 1) = 0.0301309885271363365100741023641650606635D+00
      Alpha( 2) = 0.1606809945496051955719481532658399203228D+00
      Alpha( 3) = 0.4037750551486719004321696013803943969833D+00
      Alpha( 4) = 0.7751832496328281628091352861709140142921D+00
      Alpha( 5) = 1.3015684469973897338353555075407541608001D+00
      Alpha( 6) = 2.0256723922926537207432617915259243090986D+00
      Alpha( 7) = 3.0143817180020071619470783952365877667035D+00
      Alpha( 8) = 4.3718289090112480007807582715884109347826D+00
      Alpha( 9) = 6.2650017998353700757543927046810949832434D+00
      Alpha(10) = 8.9909992515811430506197932110978854325367D+00
      Alpha(11) = 13.2454300125589543215415266885770506632980D+00
C
      Xi( 1) = 1.003273313513527648056442020952D+00
      Xi( 2) = 1.029590869049646218238991401961D+00
      Xi( 3) = 1.082926499973093964869844119114D+00
      Xi( 4) = 1.164689536100922592743410477478D+00
      Xi( 5) = 1.277018303784519149537621252044D+00
      Xi( 6) = 1.422814405817666808320987037462D+00
      Xi( 7) = 1.605789617280226246032702464017D+00
      Xi( 8) = 1.830512445012425343715198322592D+00
      Xi( 9) = 2.102430837927628287827019892831D+00
      Xi(10) = 2.427834409249072842359312707217D+00
      Xi(11) = 2.813696452389779884776382923306D+00
      Xi(12) = 3.267308650574832364944788398731D+00
      Xi(13) = 3.795571461478587432228012921165D+00
      Xi(14) = 4.403743151478856171499032257088D+00
      Xi(15) = 5.093394497119079525251222895221D+00
      Xi(16) = 5.859312922846066839718814911464D+00
      Xi(17) = 6.685280709353001782780517858740D+00
      Xi(18) = 7.539203178158188994872901389499D+00
      Xi(19) = 8.369184439007884772354639579817D+00
      Xi(20) = 9.103695956024776096693340576849D+00
      Xi(21) = 9.659796125678525681619579490444D+00
      Xi(22) = 9.961142800362639472544490359951D+00
      GOTO 9999
C
C     ----- K = 11 , R = 2E1 -----
C
 1112 CONTINUE
C
      RIni = 2.0D+01
C
      Omega( 1) = 0.0466937996822860070988069850633772617243D+00
      Omega( 2) = 0.1126033802262315067067399536326899323058D+00
      Omega( 3) = 0.1890391440443657539844084930491519003226D+00
      Omega( 4) = 0.2851482966031922318276073391452030136861D+00
      Omega( 5) = 0.4150403333396616663882993519818498384666D+00
      Omega( 6) = 0.5999483249149235138055098748299798216976D+00
      Omega( 7) = 0.8710338480743368342988641739310651246342D+00
      Omega( 8) = 1.2759883242857371767964638764603080289817D+00
      Omega( 9) = 1.8993330622314406922693935664270270535781D+00
      Omega(10) = 2.9368796146680767613322549092202962128795D+00
      Omega(11) = 5.1400782864171481395879070852572567673633D+00
C
      Alpha( 1) = 0.0181210797127243208924727430053636112461D+00
      Alpha( 2) = 0.0971750380905392633493276692557838458697D+00
      Alpha( 3) = 0.2467985172992078721931506410380841032293D+00
      Alpha( 4) = 0.4817566785490354173688946903819640965594D+00
      Alpha( 5) = 0.8282771166045613832600719161902702580846D+00
      Alpha( 6) = 1.3300696805608714204426437643924430176412D+00
      Alpha( 7) = 2.0567305766295164660886068963563388933835D+00
      Alpha( 8) = 3.1164004963694246137274890262602866641828D+00
      Alpha( 9) = 4.6805076135644237333641171083797871688148D+00
      Alpha(10) = 7.0492221261381141847457421079070627456531D+00
      Alpha(11) = 10.9116044227192424539638060387858331523603D+00
C
      Xi( 1) = 1.004488687272064465568106872428D+00
      Xi( 2) = 1.040655880503802272613045332150D+00
      Xi( 3) = 1.114370866869753685913201068924D+00
      Xi( 4) = 1.228430737034884367444263764124D+00
      Xi( 5) = 1.387139795240555471041860036152D+00
      Xi( 6) = 1.596477373296791676471687027039D+00
      Xi( 7) = 1.864351775240325198123962635588D+00
      Xi( 8) = 2.200928373803418052097324864036D+00
      Xi( 9) = 2.619005200711196410575526405395D+00
      Xi(10) = 3.134392695918814298466914536334D+00
      Xi(11) = 3.766234046800330914411478189230D+00
      Xi(12) = 4.537127178375117797764354277135D+00
      Xi(13) = 5.472770514940351053306821782485D+00
      Xi(14) = 6.600612423468904096848092200034D+00
      Xi(15) = 7.946569514961566195465625384031D+00
      Xi(16) = 9.528324635071206902270368743757D+00
      Xi(17) = 11.343159668603970342652675906692D+00
      Xi(18) = 13.348482476294802900190772376732D+00
      Xi(19) = 15.436126389431109655510365730180D+00
      Xi(20) = 17.410130365170752750683669241027D+00
      Xi(21) = 18.991985340400373383912269176221D+00
      Xi(22) = 19.883248872480938938822614403534D+00
      GOTO 9999
C
C     ----- K = 11 , R = 3E1 -----
C
 1113 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.0343853828194110494539958728515971486672D+00
      Omega( 2) = 0.0836001053696773024971632994051251586143D+00
      Omega( 3) = 0.1425876773323265062450764320556295672304D+00
      Omega( 4) = 0.2204440782713079492494438402228063011989D+00
      Omega( 5) = 0.3315518760784969383925094993381676999888D+00
      Omega( 6) = 0.4974967407138457998094956125090249088316D+00
      Omega( 7) = 0.7496946753853811701642784193211355159292D+00
      Omega( 8) = 1.1363279712558223806875834505092370818602D+00
      Omega( 9) = 1.7427500950293133717607790833525882590038D+00
      Omega(10) = 2.7645508616331733189007413553639480596757D+00
      Omega(11) = 4.9444585797781260795222268988879932294367D+00
C
      Alpha( 1) = 0.0133321421683787708297889767691812856754D+00
      Alpha( 2) = 0.0717844674758855171126212431353952325708D+00
      Alpha( 3) = 0.1837505079175499636242110956807493948872D+00
      Alpha( 4) = 0.3631906488145699331954695254776055435286D+00
      Alpha( 5) = 0.6356426510512829221245974242737730719455D+00
      Alpha( 6) = 1.0444692700724251522258201330117799443542D+00
      Alpha( 7) = 1.6592152251340612241882824240413185634679D+00
      Alpha( 8) = 2.5883048394719798256536497094160154119891D+00
      Alpha( 9) = 4.0041452021372941614575646784857099191868D+00
      Alpha(10) = 6.2084129877437377363971759969274444301846D+00
      Alpha(11) = 9.8885546801249998207644642533864498545881D+00
C
      Xi( 1) = 1.005238982501608224241762545859D+00
      Xi( 2) = 1.047506997148030853444708454525D+00
      Xi( 3) = 1.133953004614822940255766670070D+00
      Xi( 4) = 1.268458006813831051920460346594D+00
      Xi( 5) = 1.457034379328764806763724104499D+00
      Xi( 6) = 1.708130758025110139787863805516D+00
      Xi( 7) = 2.033102713009531136269594786548D+00
      Xi( 8) = 2.446840637749924481612884274995D+00
      Xi( 9) = 2.968531831765823256327405110788D+00
      Xi(10) = 3.622542791092627238215551077616D+00
      Xi(11) = 4.439389335501653323281384677657D+00
      Xi(12) = 5.456670502894996907169988809017D+00
      Xi(13) = 6.719657910243575584235270481415D+00
      Xi(14) = 8.280823895686300801372015101265D+00
      Xi(15) = 10.196868876272738388237548701198D+00
      Xi(16) = 12.520501353935608300840809370413D+00
      Xi(17) = 15.282277337325394486224061640911D+00
      Xi(18) = 18.455974983409653485122436933352D+00
      Xi(19) = 21.902872540956904594960930054981D+00
      Xi(20) = 25.305408442239061496750740509754D+00
      Xi(21) = 28.140092899406508115262015579106D+00
      Xi(22) = 29.782364787011379817990852636456D+00
      GOTO 9999
C
C     ----- K = 11 , R = 4E1 -----
C
 1114 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.0275873899402055327081950686363143177005D+00
      Omega( 2) = 0.0675179313170176000007236156996714981915D+00
      Omega( 3) = 0.1166597880630956601271546770048903596262D+00
      Omega( 4) = 0.1840049234146371288326372167576394645039D+00
      Omega( 5) = 0.2839493541216279612023672823672981735399D+00
      Omega( 6) = 0.4380335127182523924686211730383789131338D+00
      Omega( 7) = 0.6776838734799814061167920475892145759644D+00
      Omega( 8) = 1.0515697426467314990958246134766795876203D+00
      Omega( 9) = 1.6460013694707742813033712891979121195618D+00
      Omega(10) = 2.6569503323667347158536927720362541549548D+00
      Omega(11) = 4.8222004072349348761071519131604645735933D+00
C
      Alpha( 1) = 0.0106884539456373632020807530607076074602D+00
      Alpha( 2) = 0.0577396059898886379488326323072833190508D+00
      Alpha( 3) = 0.1487543597431635365247251656661831020756D+00
      Alpha( 4) = 0.2970663897990963662326929090040650294213D+00
      Alpha( 5) = 0.5275519950057114952103598515265048263245D+00
      Alpha( 6) = 0.8829075520464834679916152349044011771184D+00
      Alpha( 7) = 1.4319631560494335048485531336481813013961D+00
      Alpha( 8) = 2.2826484117443071109650881700403601826110D+00
      Alpha( 9) = 3.6076327856256673880415430932799836227787D+00
      Alpha(10) = 5.7096768822342786236026523916109454148682D+00
      Alpha(11) = 9.2758730438427994067357440499677068146411D+00
C
      Xi( 1) = 1.005783326006271300085727349316D+00
      Xi( 2) = 1.052487601408567548114461609465D+00
      Xi( 3) = 1.148244607841531594546492789366D+00
      Xi( 4) = 1.297835051320845218083589733826D+00
      Xi( 5) = 1.508698769283583550923459604132D+00
      Xi( 6) = 1.791373445471515501786416435515D+00
      Xi( 7) = 2.160176288700313776587613734392D+00
      Xi( 8) = 2.634125137047316175245223468515D+00
      Xi( 9) = 3.238087431930898431667320558347D+00
      Xi(10) = 4.004184086936066402830775023958D+00
      Xi(11) = 4.973462649229208518229100599584D+00
      Xi(12) = 6.197780262115873076405464514593D+00
      Xi(13) = 7.741592558633213706081876992471D+00
      Xi(14) = 9.682877078701319828804572331649D+00
      Xi(15) = 12.111404184952905989207383452566D+00
      Xi(16) = 15.120567916287438012430677236608D+00
      Xi(17) = 18.785452511544115491298745368454D+00
      Xi(18) = 23.114867093063940418906643259334D+00
      Xi(19) = 27.963223265502905141507983266536D+00
      Xi(20) = 32.905737217259679330827459864395D+00
      Xi(21) = 37.149516720183049747761039327543D+00
      Xi(22) = 39.663654492691002611398642940799D+00
      GOTO 9999
C
C     ----- K = 11 , R = 5E1 -----
C
 1115 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.0232195566793358917775057497639989989580D+00
      Omega( 2) = 0.0571513181735215659321164369949253014624D+00
      Omega( 3) = 0.0998552854507156438591580945030035110221D+00
      Omega( 4) = 0.1602055619046999012056132702386257449234D+00
      Omega( 5) = 0.2525035438948522461502944197464515241336D+00
      Omega( 6) = 0.3981171244943776991954877825241609912155D+00
      Omega( 7) = 0.6284424002596707273856496689390382925922D+00
      Omega( 8) = 0.9926088267135148024964720614438107304522D+00
      Omega( 9) = 1.5778155724565829864328853071597791313252D+00
      Omega(10) = 2.5805592790528422068689617940862035538885D+00
      Omega(11) = 4.7353741399348671268706989501140469656093D+00
C
      Alpha( 1) = 0.0089905001583754167007777116649158699602D+00
      Alpha( 2) = 0.0487041770923081894940215308409392491740D+00
      Alpha( 3) = 0.1261762122605637203871103979579970655323D+00
      Alpha( 4) = 0.2542357904458434050744969345814183725452D+00
      Alpha( 5) = 0.4571594292572615855362212766710428013539D+00
      Alpha( 6) = 0.7769092949171712697564728955512691754848D+00
      Alpha( 7) = 1.2814468596702209815443168627879799714719D+00
      Alpha( 8) = 2.0780301498130218532198881797867784371192D+00
      Alpha( 9) = 3.3393616918465426032010950896378176366852D+00
      Alpha(10) = 5.3690541246141367530582411138340148681891D+00
      Alpha(11) = 8.8542885137191006629173162245649564283667D+00
C
      Xi( 1) = 1.006210109757504921426142507901D+00
      Xi( 2) = 1.056398646441784029150444401690D+00
      Xi( 3) = 1.159500533992196700345742033100D+00
      Xi( 4) = 1.321070476658640847097796189313D+00
      Xi( 5) = 1.549783157108070042362775453615D+00
      Xi( 6) = 1.857999745604813185938242181550D+00
      Xi( 7) = 2.262653987453266012618013913738D+00
      Xi( 8) = 2.786454618005919255802060874316D+00
      Xi( 9) = 3.459413085995143127306106212160D+00
      Xi(10) = 4.320768676526033707455815013887D+00
      Xi(11) = 5.421392081002391185717143029521D+00
      Xi(12) = 6.826671040936758972240278042776D+00
      Xi(13) = 8.619645170878245339640688627014D+00
      Xi(14) = 10.903625654862894262496497699289D+00
      Xi(15) = 13.802322586481959816595566081787D+00
      Xi(16) = 17.452867307040970041784322397405D+00
      Xi(17) = 21.981989252163744585563676103845D+00
      Xi(18) = 27.447010430433434611391385615775D+00
      Xi(19) = 33.715502395166521192038100451782D+00
      Xi(20) = 40.272848099043959756160315777151D+00
      Xi(21) = 46.045079807836921025054888190198D+00
      Xi(22) = 49.529969194038888841147283415012D+00
      GOTO 9999
C
C     ----- K = 11 , R = 6E1 -----
C
 1116 CONTINUE
C
      RIni = 6.0D+01
C
      Omega( 1) = 0.0201523470175281204055153230422781085451D+00
      Omega( 2) = 0.0498513967434951159696883329214278646191D+00
      Omega( 3) = 0.0879662646198283677617061476761151084247D+00
      Omega( 4) = 0.1432515275765687200420876734607844582570D+00
      Omega( 5) = 0.2298666555952093790837894790590745230929D+00
      Omega( 6) = 0.3689707967994120758393507275574307868737D+00
      Omega( 7) = 0.5919302613549222069535410417628895629605D+00
      Omega( 8) = 0.9482919217093114909998032013849211807610D+00
      Omega( 9) = 1.5260458052374904870185715055619368740736D+00
      Omega(10) = 2.5222361903145047716968268902704153333616D+00
      Omega(11) = 4.6690760135659036588265391465313314256491D+00
C
      Alpha( 1) = 0.0077985503072843401969374522390149317985D+00
      Alpha( 2) = 0.0423524201109992350250692174501443432177D+00
      Alpha( 3) = 0.1102652551111842159071952870763588805403D+00
      Alpha( 4) = 0.2239475061574449626645945052305641809198D+00
      Alpha( 5) = 0.4071358003831677819096650228214429034779D+00
      Alpha( 6) = 0.7010697302865118014881737340537881664204D+00
      Alpha( 7) = 1.1728396207014495131832829133422535505815D+00
      Alpha( 8) = 1.9290152408304663420252414440803079287434D+00
      Alpha( 9) = 3.1422391785388204986026461362769168772502D+00
      Alpha(10) = 5.1168202530289228812179769256829331425251D+00
      Alpha(11) = 8.5402070139792344233634069361471574666211D+00
C
      Xi( 1) = 1.006560623915288661504863210183D+00
      Xi( 2) = 1.059614804442524975619868699006D+00
      Xi( 3) = 1.168778918256115415875463769968D+00
      Xi( 4) = 1.340289416578526023580454629158D+00
      Xi( 5) = 1.583913090945669306501032702617D+00
      Xi( 6) = 1.913637316052279297102015065413D+00
      Xi( 7) = 2.348750746943249423370758677265D+00
      Xi( 8) = 2.915315924424118237002351294862D+00
      Xi( 9) = 3.648065668999952153956842226812D+00
      Xi(10) = 4.592844553199314636213884677218D+00
      Xi(11) = 5.809747422460572900682063135491D+00
      Xi(12) = 7.377041094945992064872003490805D+00
      Xi(13) = 9.395727948083074930127334134511D+00
      Xi(14) = 11.994057570426518195480514172857D+00
      Xi(15) = 15.329926577711685170384015020062D+00
      Xi(16) = 19.585963444057799367689631786504D+00
      Xi(17) = 24.945356665340155425183987780713D+00
      Xi(18) = 31.524101840103794855765806381953D+00
      Xi(19) = 39.219801161246302452501044655264D+00
      Xi(20) = 47.445806203028849497294050507890D+00
      Xi(21) = 54.842855007443132461103507679923D+00
      Xi(22) = 59.383169485026616251505116039766D+00
      GOTO 9999
C
C     ----- K = 11 , R = 7E1 -----
C
 1117 CONTINUE
C
      RIni = 7.0D+01
C
      Omega( 1) = 0.0178680837427477367867698077041893611749D+00
      Omega( 2) = 0.0444015974630119514095422973698479296445D+00
      Omega( 3) = 0.0790534342226611162648920380635608751163D+00
      Omega( 4) = 0.1304614133244965352645275760701260026053D+00
      Omega( 5) = 0.2126234626501610114779772936488555501455D+00
      Omega( 6) = 0.3464873698648142582049921922582313982275D+00
      Omega( 7) = 0.5633974185429598534346519456139645853909D+00
      Omega( 8) = 0.9132730201360275555286759308071253826711D+00
      Omega( 9) = 1.4848031156704543130154042018808979719324D+00
      Omega(10) = 2.4755646905106925954008489920710189835518D+00
      Omega(11) = 4.6160207288556499025458657037290777225280D+00
C
      Alpha( 1) = 0.0069111231519738776855104537936991260239D+00
      Alpha( 2) = 0.0376175443778602281864995200699741673134D+00
      Alpha( 3) = 0.0983788554316176733961925825189265282233D+00
      Alpha( 4) = 0.2012493936056623861145754503243487931741D+00
      Alpha( 5) = 0.3694783097483234116608068892118055259743D+00
      Alpha( 6) = 0.6436206330275095796708724471546503309582D+00
      Alpha( 7) = 1.0899407695335288343398486010649151012331D+00
      Alpha( 8) = 1.8143508270266151124420345164267587279028D+00
      Alpha( 9) = 2.9893905748368822881521483791900095638994D+00
      Alpha(10) = 4.9199519381062117805679390070139334056876D+00
      Alpha(11) = 8.2938255368030769556358938743301223439630D+00
C
      Xi( 1) = 1.006857599192093716614082588023D+00
      Xi( 2) = 1.062342605357837705478026013584D+00
      Xi( 3) = 1.176664399235137160783931653540D+00
      Xi( 4) = 1.356669684735823086670601644865D+00
      Xi( 5) = 1.613107790238305128604698324146D+00
      Xi( 6) = 1.961437595963733477992009535118D+00
      Xi( 7) = 2.423095507358677820532871782788D+00
      Xi( 8) = 3.027228083368775823826929416427D+00
      Xi( 9) = 3.812945912708375304943017591874D+00
      Xi(10) = 4.832271776130559904922218139589D+00
      Xi(11) = 6.154016021838048258307124482869D+00
      Xi(12) = 7.868747300855569675945289098706D+00
      Xi(13) = 10.094835069599521508096340305372D+00
      Xi(14) = 12.984989087355007315351118624136D+00
      Xi(15) = 16.731241509380213740298959379516D+00
      Xi(16) = 21.562745035293026557229878470423D+00
      Xi(17) = 27.722554796770373059661141112997D+00
      Xi(18) = 35.393240225111605889996013729615D+00
      Xi(19) = 44.516783567365161258105876740387D+00
      Xi(20) = 54.451824494240437701114698754168D+00
      Xi(21) = 63.554269878117245724108430593446D+00
      Xi(22) = 69.224585099912302292579635576431D+00
      GOTO 9999
C
C     ----- K = 11 , R = 8E1 -----
C
 1118 CONTINUE
C
      RIni = 8.0D+01
C
      Omega( 1) = 0.0160941951431383953638438273278654655485D+00
      Omega( 2) = 0.0401601741924849507256662425186410914080D+00
      Omega( 3) = 0.0720906972120942371015021692126634178521D+00
      Omega( 4) = 0.1204112625174464150731726622134409865339D+00
      Omega( 5) = 0.1989522181966777404301966597488515731129D+00
      Omega( 6) = 0.3284599168601711009763104848202530661183D+00
      Omega( 7) = 0.5402638660330096453542243761791752376666D+00
      Omega( 8) = 0.8846145384436353546904995004407368242028D+00
      Omega( 9) = 1.4508208171080411968533602218300870845269D+00
      Omega(10) = 2.4369656699328211739517119349685003726336D+00
      Omega(11) = 4.5721418768127472043519277100642739242176D+00
C
      Alpha( 1) = 0.0062221568462019169464015996146022402513D+00
      Alpha( 2) = 0.0339374511936524437029913999624985976311D+00
      Alpha( 3) = 0.0891223549172805987108204189839355535696D+00
      Alpha( 4) = 0.1835224902391902816436086362528179449782D+00
      Alpha( 5) = 0.3399444112314101982587084133102806049465D+00
      Alpha( 6) = 0.5983039637414345515687395804516057751243D+00
      Alpha( 7) = 1.0240978685121112761726921980276472368132D+00
      Alpha( 8) = 1.7226238644716444793061546625345670236129D+00
      Alpha( 9) = 2.8662979670373295178718853204458127947873D+00
      Alpha(10) = 4.7605096085039891777926368554574310110183D+00
      Alpha(11) = 8.0934197853927114085087479722346870403271D+00
C
      Xi( 1) = 1.007114911323129302742236668156D+00
      Xi( 2) = 1.064708254106896957421127258847D+00
      Xi( 3) = 1.183514884140128698439622945138D+00
      Xi( 4) = 1.370935080634906710244161343049D+00
      Xi( 5) = 1.638612033508315807034567967992D+00
      Xi( 6) = 2.003351900306294064001838561850D+00
      Xi( 7) = 2.488570524563886765144135604366D+00
      Xi( 8) = 3.126275779556943167037871833536D+00
      Xi( 9) = 3.959667397510368679969597338619D+00
      Xi(10) = 5.046586336691558532119988855769D+00
      Xi(11) = 6.464118428179701520124716962457D+00
      Xi(12) = 8.314620795618188192448172468829D+00
      Xi(13) = 10.733276889443528670682603820552D+00
      Xi(14) = 13.896753202124445932853780139027D+00
      Xi(15) = 18.031001346196642312613711567337D+00
      Xi(16) = 23.412291370075801024211448186918D+00
      Xi(17) = 30.346026098270278199831029297684D+00
      Xi(18) = 39.087647695362775564220658708336D+00
      Xi(19) = 49.635855874445203150119310109289D+00
      Xi(20) = 61.311065980470514071021259638883D+00
      Xi(21) = 72.187931301061125745344160264949D+00
      Xi(22) = 79.055224610952105873673545488600D+00
      GOTO 9999
C
C     ----- K = 11 , R = 9E1 -----
C
 1119 CONTINUE
C
      RIni = 9.0D+01
C
      Omega( 1) = 0.0146727601814103851884249431376805450356D+00
      Omega( 2) = 0.0367546982817821544715951916776353058935D+00
      Omega( 3) = 0.0664809481604872105456089000086183915528D+00
      Omega( 4) = 0.1122696684291449136910414838275151794278D+00
      Omega( 5) = 0.1877844975152047632752094763475270156050D+00
      Omega( 6) = 0.3135844709409314501086674803476128658986D+00
      Omega( 7) = 0.5209897018134802104249309051553495919507D+00
      Omega( 8) = 0.8605452139568929573664943233346491524571D+00
      Omega( 9) = 1.4221134659936936344174981949883829202008D+00
      Omega(10) = 2.4042539239717671687227018573196346551413D+00
      Omega(11) = 4.5349558958883104250667250845197031594580D+00
C
      Alpha( 1) = 0.0056702141892506984487967117841700215308D+00
      Alpha( 2) = 0.0309862655699217783785727724915615155510D+00
      Alpha( 3) = 0.0816860387399204145020530781762246874678D+00
      Alpha( 4) = 0.1692434066400484682065122332694784290652D+00
      Alpha( 5) = 0.3160607081649595111595522911596134463252D+00
      Alpha( 6) = 0.5614600806021925772119549047367570437927D+00
      Alpha( 7) = 0.9702293534113280694072302401398388838061D+00
      Alpha( 8) = 1.6470963177289165131709364198187017791497D+00
      Alpha( 9) = 2.7643431184907410428242685584976356949483D+00
      Alpha(10) = 4.6277899426647656983338507963310348714003D+00
      Alpha(11) = 7.9259744054270826574097597028156769738416D+00
C
      Xi( 1) = 1.007341663573080700199054160482D+00
      Xi( 2) = 1.066794619685856562035786154752D+00
      Xi( 3) = 1.189565740566661882176950626455D+00
      Xi( 4) = 1.383562565681869578225124395310D+00
      Xi( 5) = 1.661249550979290727135105620871D+00
      Xi( 6) = 2.040676979969849535361328674021D+00
      Xi( 7) = 2.547099296104168583611965237523D+00
      Xi( 8) = 3.215197927414357416658202692972D+00
      Xi( 9) = 4.092017650220201428149652267408D+00
      Xi(10) = 5.240906303037136232478993003170D+00
      Xi(11) = 6.746840074156372169585788034496D+00
      Xi(12) = 8.723505522613407395064732963164D+00
      Xi(13) = 11.322386074322037554512609958124D+00
      Xi(14) = 14.743607175741540703424459213267D+00
      Xi(15) = 19.246733999116489780348282501166D+00
      Xi(16) = 25.155461057089254839347347569856D+00
      Xi(17) = 32.839384846893302876552622748818D+00
      Xi(18) = 42.631954001327561701306656516408D+00
      Xi(19) = 54.599273429896236945607634893918D+00
      Xi(20) = 68.039133671991829224734438241740D+00
      Xi(21) = 80.750605961361787560326419566081D+00
      Xi(22) = 88.875884517700640997939220966373D+00
      GOTO 9999
C
C     ----- K = 11 , R = 1E2 -----
C
 1121 CONTINUE
C
      RIni = 1.0D+02
C
      Omega( 1) = 0.0135056228913994452729794171902399924079D+00
      Omega( 2) = 0.0339533364876027449583584683773240797677D+00
      Omega( 3) = 0.0618515718361504429331830610255815727783D+00
      Omega( 4) = 0.1055161868702422083980988973261894159350D+00
      Omega( 5) = 0.1784485164513013136943940170620059859630D+00
      Omega( 6) = 0.3010351533892806076249740176997704566020D+00
      Omega( 7) = 0.5045904598592531176809959647666659066090D+00
      Omega( 8) = 0.8399224594117821994791912854072535310479D+00
      Omega( 9) = 1.3973915223155492387231702000605082503171D+00
      Omega(10) = 2.3760047579964309286560536493837503257964D+00
      Omega(11) = 4.5028432521787169753521573589694071415579D+00
C
      Alpha( 1) = 0.0052171157900164305121616745847257234914D+00
      Alpha( 2) = 0.0285613209731876607790204474976447635015D+00
      Alpha( 3) = 0.0755656714563188581487750486442855901714D+00
      Alpha( 4) = 0.1574618406901295399800922922284307503560D+00
      Alpha( 5) = 0.2962809494169539860049852192380726023657D+00
      Alpha( 6) = 0.5307942425023796681840017319675695262049D+00
      Alpha( 7) = 0.9251356226523106539728208419504085213703D+00
      Alpha( 8) = 1.5835045879490817360719240314281819337339D+00
      Alpha( 9) = 2.6780450591320426989138531181922076029878D+00
      Alpha(10) = 4.5149547171879260030677438741975038283272D+00
      Alpha(11) = 7.7831431411206166304933562472001540299971D+00
C
      Xi( 1) = 1.007544151737991758224083360673D+00
      Xi( 2) = 1.068659033452348684181849869379D+00
      Xi( 3) = 1.194980270598195126806155552668D+00
      Xi( 4) = 1.394883709874343532099730258444D+00
      Xi( 5) = 1.681594294760566527142942039319D+00
      Xi( 6) = 2.074319473268862619712987749310D+00
      Xi( 7) = 2.600032748539986276288191180406D+00
      Xi( 8) = 3.295928120544334691916985025095D+00
      Xi( 9) = 4.212683432651104258569174643156D+00
      Xi(10) = 5.418881494632919341557925063668D+00
      Xi(11) = 7.007049298105176418204748101104D+00
      Xi(12) = 9.101789195740375220181062054792D+00
      Xi(13) = 11.870409633437578280322488577525D+00
      Xi(14) = 15.536009144308547636099959721179D+00
      Xi(15) = 20.391410969809726090942736398404D+00
      Xi(16) = 26.807857506229526480673319710490D+00
      Xi(17) = 35.220527963930422440941292450134D+00
      Xi(18) = 46.045097106276776009747830897822D+00
      Xi(19) = 59.424485080890632523326067016001D+00
      Xi(20) = 74.648490737217016943727543321074D+00
      Xi(21) = 89.247781400980597130145888229436D+00
      Xi(22) = 98.687213059750639010392347927336D+00
      GOTO 9999
C
C     ----- K = 11 , R = 2E2 -----
C
 1122 CONTINUE
C
      RIni = 2.0D+02
C
      Omega( 1) = 0.0078046543400422635145728894948136122878D+00
      Omega( 2) = 0.0201790394732283336855016087421632597909D+00
      Omega( 3) = 0.0388177376499393517197641391425211310207D+00
      Omega( 4) = 0.0712382820959274406944156457499595092031D+00
      Omega( 5) = 0.1296799649960446518400739718268610545238D+00
      Omega( 6) = 0.2333959079982976153372267041907761608854D+00
      Omega( 7) = 0.4136818764549498482409762428746446971672D+00
      Omega( 8) = 0.7229529177297216195260953774948120553745D+00
      Omega( 9) = 1.2547944496604302322016397108761509571195D+00
      Omega(10) = 2.2115303605695885785179660709687254893652D+00
      Omega(11) = 4.3158752146954668738390581417263547336916D+00
C
      Alpha( 1) = 0.0030056959919188562807105142291388966136D+00
      Alpha( 2) = 0.0166858950303621396647486396086490501034D+00
      Alpha( 3) = 0.0454121585139150468891477540472889629086D+00
      Alpha( 4) = 0.0988641643548232611750045152609889242967D+00
      Alpha( 5) = 0.1964885269077900034113580082784089597681D+00
      Alpha( 6) = 0.3731861006919959337360323536891826279316D+00
      Alpha( 7) = 0.6885800478141612706770545437873209948521D+00
      Alpha( 8) = 1.2431291028181766957069406820401979985036D+00
      Alpha( 9) = 2.2076885241065623104173010204931415501051D+00
      Alpha(10) = 3.8906895639880266766531813349416779601597D+00
      Alpha(11) = 6.9840797596545937373814505466640412123525D+00
C
      Xi( 1) = 1.008857660934746233814124360961D+00
      Xi( 2) = 1.080784273809255970695726578601D+00
      Xi( 3) = 1.230364761635588597733781401899D+00
      Xi( 4) = 1.469372260898910083869173259785D+00
      Xi( 5) = 1.816609709388716484041066678845D+00
      Xi( 6) = 2.299915446269172837048463953202D+00
      Xi( 7) = 2.959328043960797540801524463383D+00
      Xi( 8) = 3.851503823852511351625085844752D+00
      Xi( 9) = 5.055838276384788621594518209790D+00
      Xi(10) = 6.683232597826647388220000500780D+00
      Xi(11) = 8.888742086174264016990020920161D+00
      Xi(12) = 11.889757156156335888251562860063D+00
      Xi(13) = 15.991943396208497255722713603632D+00
      Xi(14) = 21.625657965206819011508287964318D+00
      Xi(15) = 29.395022467630873463234619791251D+00
      Xi(16) = 40.137368984784588022557416664426D+00
      Xi(17) = 54.973256143442616388633537027886D+00
      Xi(18) = 75.270314164360716667478978791905D+00
      Xi(19) = 102.281561292680015257117176474821D+00
      Xi(20) = 135.856224216334324300659019968407D+00
      Xi(21) = 171.364500755064211867662038457638D+00
      Xi(22) = 196.376475465801705613810845818534D+00
      GOTO 9999
C
C     ----- K = 11 , R = 3E2 -----
C
 1123 CONTINUE
C
      RIni = 3.0D+02
C
      Omega( 1) = 0.0056543295301813358414011055803893857075D+00
      Omega( 2) = 0.0149238456038072123943028965929735729645D+00
      Omega( 3) = 0.0298410617506146968957271093172978204677D+00
      Omega( 4) = 0.0573864042420610532420062552955153378775D+00
      Omega( 5) = 0.1090165155437165930358200602312557236928D+00
      Omega( 6) = 0.2033835540589411647522700976598031274989D+00
      Omega( 7) = 0.3717369601140614015079115250994590269329D+00
      Omega( 8) = 0.6672646318758590398508880048300540011041D+00
      Omega( 9) = 1.1853197469033224647214433411512857219350D+00
      Omega(10) = 2.1303494112152294552315484610360840633803D+00
      Omega(11) = 4.2235788700889298776484925834751038564718D+00
C
      Alpha( 1) = 0.0021727378031523375944273458000230436760D+00
      Alpha( 2) = 0.0121865964288996270824291085453638938674D+00
      Alpha( 3) = 0.0338656294644117483577108133202537842976D+00
      Alpha( 4) = 0.0760307668100809659503204127051212424249D+00
      Alpha( 5) = 0.1565903606314177170491692969722663519860D+00
      Alpha( 6) = 0.3081737637671490040143854050036509306665D+00
      Alpha( 7) = 0.5877869036139700349209193108723781051594D+00
      Alpha( 8) = 1.0936113751271432683849840761247662612732D+00
      Alpha( 9) = 1.9954974609163636708455621948665736908879D+00
      Alpha(10) = 3.6029405911917065715322633367989624275651D+00
      Alpha(11) = 6.6099436566276548320371386768812271839124D+00
C
      Xi( 1) = 1.009600799991315490770904528262D+00
      Xi( 2) = 1.087668626770462309769071929111D+00
      Xi( 3) = 1.250588229153527876618551273324D+00
      Xi( 4) = 1.512339286585170538852046506761D+00
      Xi( 5) = 1.895400881497233290832668539672D+00
      Xi( 6) = 2.433427157673583628508251486267D+00
      Xi( 7) = 3.175475198888437448592816925164D+00
      Xi( 8) = 4.191958353626640123874153021077D+00
      Xi( 9) = 5.583113562397886250193712598389D+00
      Xi(10) = 7.491445875865887293193479123943D+00
      Xi(11) = 10.120153278409915897743942547748D+00
      Xi(12) = 13.760407643010427368446269014157D+00
      Xi(13) = 18.831750256110034424592014623556D+00
      Xi(14) = 25.941631829303968593408713516624D+00
      Xi(15) = 35.971610122283145787730473941224D+00
      Xi(16) = 50.195840854900154449219629171353D+00
      Xi(17) = 70.420611278164901382370377547204D+00
      Xi(18) = 99.061839343598339889529480473129D+00
      Xi(19) = 138.826312319355694380185184400034D+00
      Xi(20) = 190.933104513611828043084805983653D+00
      Xi(21) = 249.601758117843898668319901901214D+00
      Xi(22) = 293.453465815889291107287917270696D+00
      GOTO 9999
C
C     ----- K = 11 , R = 4E2 -----
C
 1124 CONTINUE
C
      RIni = 4.0D+02
C
      Omega( 1) = 0.0044976260521517319245515960889070239404D+00
      Omega( 2) = 0.0120741168056918430547008215525384144939D+00
      Omega( 3) = 0.0248952788581032363513678056302080854323D+00
      Omega( 4) = 0.0495477165718859348641492021392274125446D+00
      Omega( 5) = 0.0969568792731942327023245478678781239523D+00
      Omega( 6) = 0.1853828690454563002214249944898938338156D+00
      Omega( 7) = 0.3460084439065048523631942800626859479962D+00
      Omega( 8) = 0.6324824500366259017207146153616292849620D+00
      Omega( 9) = 1.1413356299301324647040706272704824186803D+00
      Omega(10) = 2.0785533474380979603989699766586340956565D+00
      Omega(11) = 4.1646774569138222208547484015639383869711D+00
C
      Alpha( 1) = 0.0017251108842001406274932291868597078466D+00
      Alpha( 2) = 0.0097587036631565804408185399280994953841D+00
      Alpha( 3) = 0.0275863189794949365422823414975561640006D+00
      Alpha( 4) = 0.0634477708115134323541657243583635761297D+00
      Alpha( 5) = 0.1341890869411384295140782230149056886148D+00
      Alpha( 6) = 0.2708946251364908589561852209071091124315D+00
      Alpha( 7) = 0.5287828806322352404511652479435213081160D+00
      Alpha( 8) = 1.0044247033581155623691363865113146403019D+00
      Alpha( 9) = 1.8668735332583040877974314697418378727889D+00
      Alpha(10) = 3.4262637969634559642444215921130989954690D+00
      Alpha(11) = 6.3781033361686373488186330149574132519774D+00
C
      Xi( 1) = 1.010112021550399436756358106937D+00
      Xi( 2) = 1.092414881681432928250963243499D+00
      Xi( 3) = 1.264587592050053307447289585586D+00
      Xi( 4) = 1.542250938775270479035202353391D+00
      Xi( 5) = 1.950643916300210820025107238962D+00
      Xi( 6) = 2.527848505185540314626688185307D+00
      Xi( 7) = 3.329887321827251532753433327372D+00
      Xi( 8) = 4.437960282291271765179457853279D+00
      Xi( 9) = 5.968915119305805363266598240202D+00
      Xi(10) = 8.090901694363919286097364125254D+00
      Xi(11) = 11.046947046654886710054244147727D+00
      Xi(12) = 15.190517045401885963193389095238D+00
      Xi(13) = 21.039401899348980558637767224184D+00
      Xi(14) = 29.357464499375479619405759912354D+00
      Xi(15) = 41.277759212052887664856637428556D+00
      Xi(16) = 58.482446156036501781760872376026D+00
      Xi(17) = 83.442682564773062829643901494592D+00
      Xi(18) = 119.645028020573583910080017744804D+00
      Xi(19) = 171.414887890938018963349342982383D+00
      Xi(20) = 241.858695941404226781301645132771D+00
      Xi(21) = 324.959591817497274235382320739518D+00
      Xi(22) = 390.041181302548662090945441605072D+00
      GOTO 9999
C
C     ----- K = 11 , R = 5E2 -----
C
 1125 CONTINUE
C
      RIni = 5.0D+02
C
      Omega( 1) = 0.0037662887410819571916741684437096582272D+00
      Omega( 2) = 0.0102607532707418736217397522113292662738D+00
      Omega( 3) = 0.0217061831502438882615679295595700892818D+00
      Omega( 4) = 0.0443837939780572404802430099002297225752D+00
      Omega( 5) = 0.0888304726922509532856774006059907833333D+00
      Omega( 6) = 0.1730191077007708891810806861588645233496D+00
      Omega( 7) = 0.3280594190240620734114879164522449173091D+00
      Omega( 8) = 0.6079089945236058984898153134679432696430D+00
      Omega( 9) = 1.1099636389059035398432437324878208073642D+00
      Omega(10) = 2.0414044190001449170963898627206845048931D+00
      Omega(11) = 4.1224225984546086082316307397377386223525D+00
C
      Alpha( 1) = 0.0014423138924763025197926043403928719755D+00
      Alpha( 2) = 0.0082197985891394618109204455065619487897D+00
      Alpha( 3) = 0.0235807105099614116229154395926181919663D+00
      Alpha( 4) = 0.0553321513921753326662486376086302897193D+00
      Alpha( 5) = 0.1195250489508127474734913188447738008335D+00
      Alpha( 6) = 0.2460990065813065139977474510279975561389D+00
      Alpha( 7) = 0.4889378463987340393164578861151525757123D+00
      Alpha( 8) = 0.9433780372993746744293176720308480298627D+00
      Alpha( 9) = 1.7778182811830074268539050863502382071601D+00
      Alpha(10) = 3.3028236886633358783491726251568820771354D+00
      Alpha(11) = 6.2150744684586401466765137602266122485162D+00
C
      Xi( 1) = 1.010497815491529190363995116542D+00
      Xi( 2) = 1.096002240606849505622118201575D+00
      Xi( 3) = 1.275199737243853945439613084556D+00
      Xi( 4) = 1.565017259187337682140549188770D+00
      Xi( 5) = 1.992906227574651268578259655229D+00
      Xi( 6) = 2.600532897616855750478476183574D+00
      Xi( 7) = 3.449617410857001624513440418696D+00
      Xi( 8) = 4.630278431259522230623720817633D+00
      Xi( 9) = 6.273261560193874468698699287117D+00
      Xi(10) = 8.568459619920009329337162728990D+00
      Xi(11) = 11.793138496154818108119233244224D+00
      Xi(12) = 16.355117405512885283419555193518D+00
      Xi(13) = 22.859290892700708325291358846698D+00
      Xi(14) = 32.210581239743349135640215763488D+00
      Xi(15) = 45.773156189807511291106623119163D+00
      Xi(16) = 65.612262749568161603053173536182D+00
      Xi(17) = 94.839737306547094809749953725486D+00
      Xi(18) = 138.010810624332375340372891514562D+00
      Xi(19) = 201.158564261307254705801739191884D+00
      Xi(20) = 289.635851036345949777173203187886D+00
      Xi(21) = 397.980376028435770652258440804871D+00
      Xi(22) = 486.207578566478576365117447721786D+00
      GOTO 9999
C
C     ----- K = 11 , R = 6E2 -----
C
 1126 CONTINUE
C
      RIni = 6.0D+02
C
      Omega( 1) = 0.0032583762595687854266888154727496718088D+00
      Omega( 2) = 0.0089945425144352905224046871440757300320D+00
      Omega( 3) = 0.0194534746653866053971984259596261779279D+00
      Omega( 4) = 0.0406703100825870640662289007966068155042D+00
      Omega( 5) = 0.0828822448445453864693428791543539091435D+00
      Omega( 6) = 0.1638367468622252963076453211033189916179D+00
      Omega( 7) = 0.3145704196522117309037583354758282894181D+00
      Omega( 8) = 0.5892630713497090854053120367872509177687D+00
      Omega( 9) = 1.0859847078241911882360432040073305870465D+00
      Omega(10) = 2.0128882604504029184278035558008923544548D+00
      Omega(11) = 4.0899797614649650006474201902051390788984D+00
C
      Alpha( 1) = 0.0012460394897290092060417434141617043508D+00
      Alpha( 2) = 0.0071487801396430483380248590598304847532D+00
      Alpha( 3) = 0.0207774988093894703261279221714885601102D+00
      Alpha( 4) = 0.0495983068314865137162330329528847627785D+00
      Alpha( 5) = 0.1090359089997250805799270181872806517731D+00
      Alpha( 6) = 0.2281340299783249546611892805136889705864D+00
      Alpha( 7) = 0.4597232649902577561722843979552521886944D+00
      Alpha( 8) = 0.8981462859609064504958636510068004099594D+00
      Alpha( 9) = 1.7112485746366723854362465462308762198518D+00
      Alpha(10) = 3.2099066335151224178302842293675212204107D+00
      Alpha(11) = 6.0917531992763160613064965520635496432078D+00
C
      Xi( 1) = 1.010805372459477708096779735580D+00
      Xi( 2) = 1.098865619000078383549201532787D+00
      Xi( 3) = 1.283689514573291367335178270181D+00
      Xi( 4) = 1.583287498127780903219713470875D+00
      Xi( 5) = 2.026956412345776807702041333314D+00
      Xi( 6) = 2.659376188697842394994749759540D+00
      Xi( 7) = 3.547095792989746083955296396262D+00
      Xi( 8) = 4.787854381541439793053965834346D+00
      Xi( 9) = 6.524384052416819790298774250914D+00
      Xi(10) = 8.965525477268821448635638393654D+00
      Xi(11) = 12.418707643050627490062531865789D+00
      Xi(12) = 17.340195095127145944249869025100D+00
      Xi(13) = 24.413492412775800864677133716896D+00
      Xi(14) = 34.672557359549868657988547226978D+00
      Xi(15) = 49.696120687816508886686106549746D+00
      Xi(16) = 71.911084122990895563243274324350D+00
      Xi(17) = 105.046282059082216490952799148317D+00
      Xi(18) = 154.713563805560001346295351254412D+00
      Xi(19) = 228.703813614806019172132067751591D+00
      Xi(20) = 334.878404555376608997541154622013D+00
      Xi(21) = 469.014081344866218648004263513940D+00
      Xi(22) = 581.997135078390529783209927927601D+00
      GOTO 9999
C
C     ----- K = 11 , R = 7E2 -----
C
 1127 CONTINUE
C
      RIni = 7.0D+02
C
      Omega( 1) = 0.0028831912480045087212277655835598877410D+00
      Omega( 2) = 0.0080547896960305267873784628579469924148D+00
      Omega( 3) = 0.0177641918678628270847818723665589857319D+00
      Omega( 4) = 0.0378426247694864904115932564138136129372D+00
      Omega( 5) = 0.0782870521182277218151350506314400945485D+00
      Omega( 6) = 0.1566599587223711810546594030979505873802D+00
      Omega( 7) = 0.3039273296837429993424908175603249560481D+00
      Omega( 8) = 0.5744369494539767600225013688408637335669D+00
      Omega( 9) = 1.0668055886728782982774202392484141910245D+00
      Omega(10) = 1.9900009359991648603179642540439431286359D+00
      Omega(11) = 4.0639355181090332552913679808881397548248D+00
C
      Alpha( 1) = 0.0011011368472607193091320547883270319911D+00
      Alpha( 2) = 0.0063561904304476650575254676085734661939D+00
      Alpha( 3) = 0.0186927643171768198647865553280178474438D+00
      Alpha( 4) = 0.0452979299110936900636930485530839973762D+00
      Alpha( 5) = 0.1010854056292192224152820013982623237325D+00
      Alpha( 6) = 0.2143707645737023147116534493861195187492D+00
      Alpha( 7) = 0.4371216892824491518049039029403246559013D+00
      Alpha( 8) = 0.8628534639195982823404486306007044049693D+00
      Alpha( 9) = 1.6589340653420582790723847299574345015571D+00
      Alpha(10) = 3.1364760958078274867208423692943597416161D+00
      Alpha(11) = 5.9939098333794121674193700055610634080949D+00
C
      Xi( 1) = 1.011059676264312853452044527192D+00
      Xi( 2) = 1.101235574587819892418069644968D+00
      Xi( 3) = 1.290729288973113345214381586512D+00
      Xi( 4) = 1.598475864989992153552406106698D+00
      Xi( 5) = 2.055354171858966342038496644129D+00
      Xi( 6) = 2.708643847270304997037343452959D+00
      Xi( 7) = 3.629086150599954327302526513321D+00
      Xi( 8) = 4.921080606943753329909213567461D+00
      Xi( 9) = 6.737918074576284085779043220299D+00
      Xi(10) = 9.305267659806523622423313213403D+00
      Xi(11) = 12.957587106833398172926752334178D+00
      Xi(12) = 18.194973643455962528450720050444D+00
      Xi(13) = 25.772788989918090972117137305020D+00
      Xi(14) = 36.844257238445314590891133121886D+00
      Xi(15) = 53.188858434996097189484398626291D+00
      Xi(16) = 77.576545348772702706541259942696D+00
      Xi(17) = 114.330947293240342728382064052539D+00
      Xi(18) = 170.104003977413448703281950713517D+00
      Xi(19) = 254.471866625116703020115238587096D+00
      Xi(20) = 378.002620022668104127205879194662D+00
      Xi(21) = 538.308905487260507150004684717715D+00
      Xi(22) = 677.441769430224394354933536988028D+00
      GOTO 9999
C
C     ----- K = 11 , R = 8E2 -----
C
 1128 CONTINUE
C
      RIni = 8.0D+02
C
      Omega( 1) = 0.0025936679815209045092815352611878565448D+00
      Omega( 2) = 0.0073265355590078712569803592348505194565D+00
      Omega( 3) = 0.0164427238548874570983523426756511831570D+00
      Omega( 4) = 0.0356008498361247634063874150517348127210D+00
      Omega( 5) = 0.0745996200705102757499498702942197780885D+00
      Omega( 6) = 0.1508449808442598329518476857336040097834D+00
      Omega( 7) = 0.2952357646445974116104077333844202257751D+00
      Omega( 8) = 0.5622513167694076248589660216303798279114D+00
      Omega( 9) = 1.0509646969145498192083759692039279798337D+00
      Omega(10) = 1.9710423046491469765148854542680112444941D+00
      Omega(11) = 4.0423577589097630046420517313521258984110D+00
C
      Alpha( 1) = 0.0009893735940206601844169881235582786339D+00
      Alpha( 2) = 0.0057435672506857186733056418008469545811D+00
      Alpha( 3) = 0.0170741427403647941375252738288592624372D+00
      Alpha( 4) = 0.0419335759461701156943912605773450152924D+00
      Alpha( 5) = 0.0948076122813876336905495462170634368704D+00
      Alpha( 6) = 0.2034033714513136402901767596507198732070D+00
      Alpha( 7) = 0.4189621037780286376243660956264491801448D+00
      Alpha( 8) = 0.8342932954015259300350505766719066969017D+00
      Alpha( 9) = 1.6163458927661897599837095840236145249946D+00
      Alpha(10) = 3.0764177556541122311468267058742753761180D+00
      Alpha(11) = 5.9136220183418721921611571623600411840016D+00
C
      Xi( 1) = 1.011275517891623727307660873809D+00
      Xi( 2) = 1.103248794090624789756248380623D+00
      Xi( 3) = 1.296718497999818083574012639314D+00
      Xi( 4) = 1.611425340338749509342709809179D+00
      Xi( 5) = 2.079631241991189730513103772758D+00
      Xi( 6) = 2.750900718173535138087043572241D+00
      Xi( 7) = 3.699679890254345520413797876458D+00
      Xi( 8) = 5.036287746054576338738256735539D+00
      Xi( 9) = 6.923459855102234069331584853657D+00
      Xi(10) = 9.602020876295634602658191081304D+00
      Xi(11) = 13.430962838312866545163737352908D+00
      Xi(12) = 18.950483820982974941110610345163D+00
      Xi(13) = 26.982267075158729401074264586669D+00
      Xi(14) = 38.790660332355037492879645810717D+00
      Xi(15) = 56.344113096241490102472626588792D+00
      Xi(16) = 82.739291247479588242708103962286D+00
      Xi(17) = 122.874004490899161136374662106618D+00
      Xi(18) = 184.421970762005453678411548423810D+00
      Xi(19) = 278.758005143637350276009101435193D+00
      Xi(20) = 419.309994116410967468500103905171D+00
      Xi(21) = 606.051488249862153390168373334745D+00
      Xi(22) = 772.565802767476019097259154477797D+00
      GOTO 9999
C
C     ----- K = 11 , R = 9E2 -----
C
 1129 CONTINUE
C
      RIni = 9.0D+02
C
      Omega( 1) = 0.0023628382210883959192670945772402069096D+00
      Omega( 2) = 0.0067436900257026460985495023257862978738D+00
      Omega( 3) = 0.0153759124610560547947820026794263359449D+00
      Omega( 4) = 0.0337695236021534751132415659580221500846D+00
      Omega( 5) = 0.0715558676321224931195122236082806210788D+00
      Omega( 6) = 0.1460054304956700081414427971893132962578D+00
      Omega( 7) = 0.2879536312425977612856290571707873482410D+00
      Omega( 8) = 0.5519856851543989045881868915621737414767D+00
      Omega( 9) = 1.0375637395027244186960016514298388301540D+00
      Omega(10) = 1.9549638627492123649177432898049744380842D+00
      Omega(11) = 4.0240548291653549630234043554821710131364D+00
C
      Alpha( 1) = 0.0009003070544349668756826245250447088297D+00
      Alpha( 2) = 0.0052544185410532049085930220416046365628D+00
      Alpha( 3) = 0.0157763948086487224303656271701123969819D+00
      Alpha( 4) = 0.0392174231619657099484101137054636865287D+00
      Alpha( 5) = 0.0896975387361955071486457370666389721237D+00
      Alpha( 6) = 0.1944045560423672174753207883557415414089D+00
      Alpha( 7) = 0.4039555012406253642776726708873979987402D+00
      Alpha( 8) = 0.8105465859229441190673091932605842657722D+00
      Alpha( 9) = 1.5807541895705499766051557730683896352275D+00
      Alpha(10) = 3.0260253637311591221048523436820687493309D+00
      Alpha(11) = 5.8460680636665212756529486082257562884479D+00
C
      Xi( 1) = 1.011462352317341486618133827857D+00
      Xi( 2) = 1.104992721003111662845044826931D+00
      Xi( 3) = 1.301913484870989259748787214566D+00
      Xi( 4) = 1.622677887976778621421415638171D+00
      Xi( 5) = 2.100775626259635315727300475253D+00
      Xi( 6) = 2.787808747240925408067066526563D+00
      Xi( 7) = 3.761541879546799615032307007034D+00
      Xi( 8) = 5.137621988706277868005217746017D+00
      Xi( 9) = 7.087332602485865488961547242752D+00
      Xi(10) = 9.865302075733673028978820696011D+00
      Xi(11) = 13.853006800327013128305841949128D+00
      Xi(12) = 19.627657504269309289915423910600D+00
      Xi(13) = 28.072607486447898102882581028439D+00
      Xi(14) = 40.556398044558649679941497723235D+00
      Xi(15) = 59.226255018542725845620555702453D+00
      Xi(16) = 87.491043814091919862685475806074D+00
      Xi(17) = 130.803769290163998964549030290527D+00
      Xi(18) = 197.840738696161377085935839659214D+00
      Xi(19) = 301.779756698642040280144982489219D+00
      Xi(20) = 459.029228277024053728183616840397D+00
      Xi(21) = 672.388667147153316072660800273297D+00
      Xi(22) = 867.388489463458042361132527275913D+00
      GOTO 9999
C
C     ----- K = 11 , R = 1E3 -----
C
 1131 CONTINUE
C
      RIni = 1.0D+03
C
      Omega( 1) = 0.0021740872264102924137251822292796199498D+00
      Omega( 2) = 0.0062654140734613558082574018628929657382D+00
      Omega( 3) = 0.0144934344125449496880266454096330974721D+00
      Omega( 4) = 0.0322384802162817058210429310011191361696D+00
      Omega( 5) = 0.0689880349533423857129039828695038494288D+00
      Omega( 6) = 0.1418933622404710467179581381835795639290D+00
      Omega( 7) = 0.2817301334836233389050282049925755245567D+00
      Omega( 8) = 0.5431705675223087211446297561945328880029D+00
      Omega( 9) = 1.0260142895533087236108887863039740295790D+00
      Omega(10) = 1.9410767529159068538234869660819015280140D+00
      Omega(11) = 4.0082438733806685813784720373575964913471D+00
C
      Alpha( 1) = 0.0008275060405979828267253562920375414835D+00
      Alpha( 2) = 0.0048538992490044590716160904843995416158D+00
      Alpha( 3) = 0.0147096837575472349300117818823863125033D+00
      Alpha( 4) = 0.0369705659049513704169829218793008607236D+00
      Alpha( 5) = 0.0854390615716348159294660051521486110460D+00
      Alpha( 6) = 0.1868523116342620025531577609978484133535D+00
      Alpha( 7) = 0.3912822366416272664075594323040974131800D+00
      Alpha( 8) = 0.7903843504629096339752711286585906691471D+00
      Alpha( 9) = 1.5504001921788530224703347992765145590965D+00
      Alpha(10) = 2.9828994649313298984528303181562591817055D+00
      Alpha(11) = 5.7881154950766118133520521382706647273153D+00
C
      Xi( 1) = 1.011626577944182569612600886000D+00
      Xi( 2) = 1.106526584008231192429901057483D+00
      Xi( 3) = 1.306488004306895040059974255620D+00
      Xi( 4) = 1.632602251589605259554649485665D+00
      Xi( 5) = 2.119461908127519490958878822795D+00
      Xi( 6) = 2.820506515514542811209203199141D+00
      Xi( 7) = 3.816505442352685583328950058579D+00
      Xi( 8) = 5.227950254525374241664631425053D+00
      Xi( 9) = 7.233934065422368251980361764097D+00
      Xi(10) = 10.101762872371427856230119424197D+00
      Xi(11) = 14.233684181896668193186006323714D+00
      Xi(12) = 20.241312228725278458962844041480D+00
      Xi(13) = 29.065709461086927115069933336144D+00
      Xi(14) = 42.173589286440474402373190798698D+00
      Xi(15) = 61.882031162953867478582203887072D+00
      Xi(16) = 91.899086735024803594684694729722D+00
      Xi(17) = 138.215380021578030739215492417316D+00
      Xi(18) = 210.490640457978115673842722799236D+00
      Xi(19) = 323.703942267917895547757467511474D+00
      Xi(20) = 497.339829521204033480064055083858D+00
      Xi(21) = 737.439254546925192723705322350725D+00
      Xi(22) = 961.925573460294248806512484861742D+00
      GOTO 9999
C
C     ----- K = 11 , R = 2E3 -----
C
 1132 CONTINUE
C
      RIni = 2.0D+03
C
      Omega( 1) = 0.0012623682047825584772750883767213070374D+00
      Omega( 2) = 0.0039256526753984839104248943247865710759D+00
      Omega( 3) = 0.0100465837533007697967448834490411524456D+00
      Omega( 4) = 0.0242425014592881167561005473568003232288D+00
      Omega( 5) = 0.0551805151273874833863831729924287472500D+00
      Omega( 6) = 0.1192702380940837676869437262994200210642D+00
      Omega( 7) = 0.2468429160149552121882162922239523084045D+00
      Omega( 8) = 0.4929860865537508678060722722147346530619D+00
      Omega( 9) = 0.9594751280677348143619866116527106214562D+00
      Omega(10) = 1.8604943398396530794981079681704727590841D+00
      Omega(11) = 3.9164423268801717532231615459537010792701D+00
C
      Alpha( 1) = 0.0004763386763845254530552357019948517269D+00
      Alpha( 2) = 0.0029099646551678040284737028833839111996D+00
      Alpha( 3) = 0.0094574935145634442861732431306887036726D+00
      Alpha( 4) = 0.0256521647921702371382255856643861413602D+00
      Alpha( 5) = 0.0634401600811532823473307476924709646937D+00
      Alpha( 6) = 0.1469147278490494683213414550682074377619D+00
      Alpha( 7) = 0.3228783614455336845267072881648928728282D+00
      Alpha( 8) = 0.6796431373233775072176601761420045022533D+00
      Alpha( 9) = 1.3812552178932358922435960635333174195694D+00
      Alpha(10) = 2.7398655433520379364181057679772379742644D+00
      Alpha(11) = 5.4589664557360760804675947621689147126745D+00
C
      Xi( 1) = 1.012634485295069944484590285239D+00
      Xi( 2) = 1.115960417122124936450781873543D+00
      Xi( 3) = 1.334732406200042509596820350382D+00
      Xi( 4) = 1.694205723556559923170458659136D+00
      Xi( 5) = 2.236241696464598578296054109948D+00
      Xi( 6) = 3.026551793939026981138454286402D+00
      Xi( 7) = 4.166247455693754736527623094844D+00
      Xi( 8) = 5.809102145284259851795716222966D+00
      Xi( 9) = 8.188781606285571878015017155406D+00
      Xi(10) = 11.662817491591790737895384388878D+00
      Xi(11) = 16.784270306190127018675095627032D+00
      Xi(12) = 24.420053833854110511211721323832D+00
      Xi(13) = 35.949725216187675389672984138656D+00
      Xi(14) = 53.606023489636841609234929606487D+00
      Xi(15) = 81.069835718520333420200607577044D+00
      Xi(16) = 124.532655436403525327226926577850D+00
      Xi(17) = 194.619603518130789329698693279624D+00
      Xi(18) = 309.879071731328180172582165141648D+00
      Xi(19) = 502.674500056099144235099984712178D+00
      Xi(20) = 825.737819569139915920619188227647D+00
      Xi(21) = 1333.340188188764327881230542516278D+00
      Xi(22) = 1893.776396022996856172149193753285D+00
      GOTO 9999
C
C     ----- K = 11 , R = 3E3 -----
C
 1133 CONTINUE
C
      RIni = 3.0D+03
C
      Omega( 1) = 0.0009228838994159140362278747997393235636D+00
      Omega( 2) = 0.0030346810581029129714051889750436874493D+00
      Omega( 3) = 0.0082645523392666875460888138704707017723D+00
      Omega( 4) = 0.0208629228147157623313135647175053932756D+00
      Omega( 5) = 0.0491040452568306093184924247807554564815D+00
      Omega( 6) = 0.1089988086210330150874030230578881628389D+00
      Omega( 7) = 0.2305969093077353990705029990193608568916D+00
      Omega( 8) = 0.4691265413835654474552540327270122588743D+00
      Omega( 9) = 0.9273310285024501460247013662208104278761D+00
      Omega(10) = 1.8211892895434077157407193325866501254495D+00
      Omega(11) = 3.8716236620173660497919859713178425408842D+00
C
      Alpha( 1) = 0.0003458600493587631532409070893091182497D+00
      Alpha( 2) = 0.0021800684483446950346975147256439742094D+00
      Alpha( 3) = 0.0074338002448057351524994453700889707193D+00
      Alpha( 4) = 0.0211226719585344040053666130052345550894D+00
      Alpha( 5) = 0.0542911478650846585830623634155722712080D+00
      Alpha( 6) = 0.1297318847429075853197556408846757847186D+00
      Alpha( 7) = 0.2925888019277061919461851990931222644576D+00
      Alpha( 8) = 0.6294124318896298635617886363391448867333D+00
      Alpha( 9) = 1.3030115311584663801824229834025459240365D+00
      Alpha(10) = 2.6257268942687263856092944047659898387792D+00
      Alpha(11) = 5.3027692366205613138871455269907073670765D+00
C
      Xi( 1) = 1.013162307629898129250796412482D+00
      Xi( 2) = 1.120914469518859819590836379710D+00
      Xi( 3) = 1.349640187754250304752977329681D+00
      Xi( 4) = 1.726948755314575472503621500486D+00
      Xi( 5) = 2.298863022302009189454588544699D+00
      Xi( 6) = 3.138244439814425477573364697825D+00
      Xi( 7) = 4.358256474678705915358489786371D+00
      Xi( 8) = 6.132764794237344253401839067052D+00
      Xi( 9) = 8.729089111644693055672150627800D+00
      Xi(10) = 12.561715766963198824546221743503D+00
      Xi(11) = 18.281335377765054883356099502123D+00
      Xi(12) = 26.924710149403320686226082081305D+00
      Xi(13) = 40.172007476294265639060743211530D+00
      Xi(14) = 60.798697236948366733749749002413D+00
      Xi(15) = 93.487711881290384327569675804170D+00
      Xi(16) = 146.331520562003058835709445872908D+00
      Xi(17) = 233.672616773833143694050207272994D+00
      Xi(18) = 381.600754617379850003322516727167D+00
      Xi(19) = 638.368852255887035962444997494458D+00
      Xi(20) = 1090.853238414367184838660307377722D+00
      Xi(21) = 1858.189718002260019669158452870761D+00
      Xi(22) = 2804.971625837513768741970920927997D+00
      GOTO 9999
C
C     ----- K = 11 , R = 4E3 -----
C
 1134 CONTINUE
C
      RIni = 4.0D+03
C
      Omega( 1) = 0.0007412355914717758371035817772411180560D+00
      Omega( 2) = 0.0025500918289050421850566258358608551227D+00
      Omega( 3) = 0.0072606523430177817511419347939333190922D+00
      Omega( 4) = 0.0188983051896847964339808600628400903076D+00
      Omega( 5) = 0.0454905661909004588254576591233035998130D+00
      Omega( 6) = 0.1027828884911571632042206862756028584727D+00
      Omega( 7) = 0.2206244947739454568354146996034259586850D+00
      Omega( 8) = 0.4543089394242018984706229373093933077143D+00
      Omega( 9) = 0.9071881653619918974224348162671560658055D+00
      Omega(10) = 1.7964239050734691755233790599888266115158D+00
      Omega(11) = 3.8433675617984309610605797580262787960237D+00
C
      Alpha( 1) = 0.0002761320530126240699661397732427373097D+00
      Alpha( 2) = 0.0017871801758091304107081822910227386636D+00
      Alpha( 3) = 0.0063237062667975813191577881024196816639D+00
      Alpha( 4) = 0.0185749572332312158613005712015642245660D+00
      Alpha( 5) = 0.0490227902621349213037446525959017762375D+00
      Alpha( 6) = 0.1196387864534846704318805585276486169732D+00
      Alpha( 7) = 0.2745011177460962377883287033331072279907D+00
      Alpha( 8) = 0.5990053817796525529085736871959255722686D+00
      Alpha( 9) = 1.2551208194212885533918785951890129126696D+00
      Alpha(10) = 2.5552713219979284167606564670549573747849D+00
      Alpha(11) = 5.2057929169179171491853297393248567459523D+00
C
      Xi( 1) = 1.013507909133304014655078961749D+00
      Xi( 2) = 1.124163449694987797081516256448D+00
      Xi( 3) = 1.359445665438874723592727056243D+00
      Xi( 4) = 1.748570385623043480015186001975D+00
      Xi( 5) = 2.340424869223935786747420317866D+00
      Xi( 6) = 3.212837585309808281167770482867D+00
      Xi( 7) = 4.487424836087381988577321711986D+00
      Xi( 8) = 6.352303794614191673195691123510D+00
      Xi( 9) = 9.098958048284406790497114148053D+00
      Xi(10) = 13.183328635620850772906809122986D+00
      Xi(11) = 19.328223806317943620977639618275D+00
      Xi(12) = 28.697951916635848879089110141649D+00
      Xi(13) = 43.202468477442194458998381279002D+00
      Xi(14) = 66.040542432351887432162573787764D+00
      Xi(15) = 102.694192864624376272664196818596D+00
      Xi(16) = 162.810990503060741912966236100146D+00
      Xi(17) = 263.864668019880545618560674370201D+00
      Xi(18) = 438.521112893176594121902311940175D+00
      Xi(19) = 749.532175553113846067088132940626D+00
      Xi(20) = 1317.076815743827621907335867490474D+00
      Xi(21) = 2333.268953940222266929538363910979D+00
      Xi(22) = 3698.566202684434597314222514796711D+00
      GOTO 9999
C
C     ----- K = 11 , R = 5E3 -----
C
 1135 CONTINUE
C
      RIni = 5.0D+03
C
      Omega( 1) = 0.0006267332758852317722579485079117261881D+00
      Omega( 2) = 0.0022404994313095141259883842947339682716D+00
      Omega( 3) = 0.0066017708522960727291522728919781481949D+00
      Omega( 4) = 0.0175803778127506669060281605659890047377D+00
      Omega( 5) = 0.0430286752670226813875548489002342833487D+00
      Omega( 6) = 0.0984970458414388567585122033487543546926D+00
      Omega( 7) = 0.2136815596076955863706452187766871020358D+00
      Omega( 8) = 0.4439105540875763014380551291226950638702D+00
      Omega( 9) = 0.8929659232509779923656415001609332193766D+00
      Omega(10) = 1.7788722459640630017873100254810481146706D+00
      Omega(11) = 3.8233333460474906823848423309541999515204D+00
C
      Alpha( 1) = 0.0002322146395909955911310504293710264834D+00
      Alpha( 2) = 0.0015383078777110473123499812322934054976D+00
      Alpha( 3) = 0.0056097155499255455726426665679501537554D+00
      Alpha( 4) = 0.0169052401931734672522236176059329260113D+00
      Alpha( 5) = 0.0455116261284354011137681834953117032683D+00
      Alpha( 6) = 0.1128182709493343310874899007578697762710D+00
      Alpha( 7) = 0.2621385842803852858769967509866560817500D+00
      Alpha( 8) = 0.5780283644600492836741235935793525868576D+00
      Alpha( 9) = 1.2218328763756418393763084062619839187391D+00
      Alpha(10) = 2.5060164008391585472104723386621571989963D+00
      Alpha(11) = 5.1377321548383988246248843534402794830385D+00
C
      Xi( 1) = 1.013759180858394429625320143362D+00
      Xi( 2) = 1.126528252383917901827299623552D+00
      Xi( 3) = 1.366596914335719902358103061690D+00
      Xi( 4) = 1.764382443729622758224820366912D+00
      Xi( 5) = 2.370924666432016908023533430239D+00
      Xi( 6) = 3.267810182254802050258804535510D+00
      Xi( 7) = 4.583096387201138603457312159506D+00
      Xi( 8) = 6.515835113413934717285086462013D+00
      Xi( 9) = 9.376216422625220800468603954414D+00
      Xi(10) = 13.652573425612661578085194058474D+00
      Xi(11) = 20.124650489238526481286517544333D+00
      Xi(12) = 30.058601331363003265895184057399D+00
      Xi(13) = 45.550203837520550555206799714369D+00
      Xi(14) = 70.145402727739678563900760366323D+00
      Xi(15) = 109.992188118941259349081729368436D+00
      Xi(16) = 176.057872207031846437108590919252D+00
      Xi(17) = 288.530732333027965463845276872235D+00
      Xi(18) = 485.921466125530878754634755978259D+00
      Xi(19) = 844.296017988520174801791995378153D+00
      Xi(20) = 1515.858811846124858280226987972128D+00
      Xi(21) = 2769.870549675472343631454918977397D+00
      Xi(22) = 4576.245877897764160469051830659737D+00
      GOTO 9999
C
C     ----- K = 11 , R = 6E3 -----
C
 1136 CONTINUE
C
      RIni = 6.0D+03
C
      Omega( 1) = 0.0005473782929676091589050948712450589406D+00
      Omega( 2) = 0.0020234317344929258391381305705325477362D+00
      Omega( 3) = 0.0061296388124654699446628358314170803567D+00
      Omega( 4) = 0.0166202678647742327646509097674454302762D+00
      Omega( 5) = 0.0412142813057742613662979332816993505162D+00
      Omega( 6) = 0.0953100824256778086240740957557626700236D+00
      Omega( 7) = 0.2084812698209122131751949366651466277744D+00
      Omega( 8) = 0.4360759297357788821509864962688851619532D+00
      Omega( 9) = 0.8822012112666169261873845564014828823929D+00
      Omega(10) = 1.7655503222027704023511934794399280690413D+00
      Omega(11) = 3.8081221072884642157331536171938068946474D+00
C
      Alpha( 1) = 0.0002017946813401587113227188651355496762D+00
      Alpha( 2) = 0.0013650957954943520156917383286121481234D+00
      Alpha( 3) = 0.0051063430539639230514999362724182230266D+00
      Alpha( 4) = 0.0157103155764604293100853565912089493395D+00
      Alpha( 5) = 0.0429662099418053528877836644284116118797D+00
      Alpha( 6) = 0.1078215546982082305435611362109948885291D+00
      Alpha( 7) = 0.2530041706297298811326624418116537640344D+00
      Alpha( 8) = 0.5624205865907779740850772065563489832130D+00
      Alpha( 9) = 1.1969260235286672264861523862222725256288D+00
      Alpha(10) = 2.4690046401427888621022549431316406298720D+00
      Alpha(11) = 5.0864407388447149165403016191078222618671D+00
C
      Xi( 1) = 1.013953504011853172452235771761D+00
      Xi( 2) = 1.128358627594889074187596567089D+00
      Xi( 3) = 1.372140064760185083512708525966D+00
      Xi( 4) = 1.776663964374088323537256850493D+00
      Xi( 5) = 2.394676147487823382551536122165D+00
      Xi( 6) = 3.310757150500711268456197067245D+00
      Xi( 7) = 4.658118960797535882626535519435D+00
      Xi( 8) = 6.644618065181252250690768779151D+00
      Xi( 9) = 9.595598690279814975065408866328D+00
      Xi(10) = 14.025829661216053410692505831747D+00
      Xi(11) = 20.761870967492430821443871735710D+00
      Xi(12) = 31.154365044786872313703485204428D+00
      Xi(13) = 47.454701976670969162941382535337D+00
      Xi(14) = 73.502772137602551523472005357007D+00
      Xi(15) = 116.017300440218688671600855144561D+00
      Xi(16) = 187.112745425990175437980234107727D+00
      Xi(17) = 309.376229073982038336732003358520D+00
      Xi(18) = 526.584806978543925104485623478467D+00
      Xi(19) = 927.109321573275227124089070684931D+00
      Xi(20) = 1693.810124284694755392166598539916D+00
      Xi(21) = 3175.115660441574632955052948091179D+00
      Xi(22) = 5439.118719873584542945366138155805D+00
      GOTO 9999
C
C     ----- K = 11 , R = 7E3 -----
C
 1137 CONTINUE
C
      RIni = 7.0D+03
C
      Omega( 1) = 0.0004888589654993823415061000240821459384D+00
      Omega( 2) = 0.0018616953743137684792181802762370268489D+00
      Omega( 3) = 0.0057714059692291130183748788535363916097D+00
      Omega( 4) = 0.0158821609122009120908280196265982553427D+00
      Omega( 5) = 0.0398065979662624536909287104036181403899D+00
      Omega( 6) = 0.0928200129151813189718280873441447198502D+00
      Omega( 7) = 0.2043948992766658286535270960815324770010D+00
      Omega( 8) = 0.4298908173319880503910825342295254358760D+00
      Omega( 9) = 0.8736723641609264178224322772159382566315D+00
      Omega(10) = 1.7549721269953933305840720890955708455294D+00
      Omega(11) = 3.7960404785939114290661128281456626609724D+00
C
      Alpha( 1) = 0.0001793701948587827513269946831881221372D+00
      Alpha( 2) = 0.0012368771763232082847653708337562483699D+00
      Alpha( 3) = 0.0047295378501987948447370308565490537589D+00
      Alpha( 4) = 0.0148046949969108685156272769110896714295D+00
      Alpha( 5) = 0.0410168641143887723388475850805345856998D+00
      Alpha( 6) = 0.1039627887046145387749140377720014072338D+00
      Alpha( 7) = 0.2459022093684296517352490946128895643596D+00
      Alpha( 8) = 0.5502188036604130099869666703504833549232D+00
      Alpha( 9) = 1.1773684055533485784709665789016241888021D+00
      Alpha(10) = 2.4398440817448355183785779765059942292282D+00
      Alpha(11) = 5.0459378421517919626783366116029583281488D+00
C
      Xi( 1) = 1.014110063993542450080237449228D+00
      Xi( 2) = 1.129834266023652116221927255157D+00
      Xi( 3) = 1.376614175376544919296613322945D+00
      Xi( 4) = 1.786592754684194123463401859819D+00
      Xi( 5) = 2.413917056239944505672059826829D+00
      Xi( 6) = 3.345636074011955358098110990817D+00
      Xi( 7) = 4.719228876055178750932583842115D+00
      Xi( 8) = 6.749870651720982652962965708809D+00
      Xi( 9) = 9.775572634850614675387781016269D+00
      Xi(10) = 14.333314418138938924063285895727D+00
      Xi(11) = 21.289252861263578554817943455646D+00
      Xi(12) = 32.065959035318667681713211692340D+00
      Xi(13) = 49.048341237057613624383822781283D+00
      Xi(14) = 76.330652603958345600587698243089D+00
      Xi(15) = 121.130517723825521826663820235126D+00
      Xi(16) = 196.576460619874407456664222593190D+00
      Xi(17) = 327.405204366704011637034810178193D+00
      Xi(18) = 562.187777745814405094826327058399D+00
      Xi(19) = 1000.730898143708903658133380076833D+00
      Xi(20) = 1855.202325197516517896190180181293D+00
      Xi(21) = 3553.944782316665923271870042299270D+00
      Xi(22) = 6287.992938677223824228690318705048D+00
      GOTO 9999
C
C     ----- K = 11 , R = 8E3 -----
C
 1138 CONTINUE
C
      RIni = 8.0D+03
C
      Omega( 1) = 0.0004437681784262299374822296159619899880D+00
      Omega( 2) = 0.0017359012756834154712376973761674137098D+00
      Omega( 3) = 0.0054884264315541892356410051245080294890D+00
      Omega( 4) = 0.0152927906015183914024177080220878721661D+00
      Omega( 5) = 0.0386741261492350979778198378361553366744D+00
      Omega( 6) = 0.0908051858607855987034324606044233263447D+00
      Omega( 7) = 0.2010729870218403514886074470724430796054D+00
      Omega( 8) = 0.4248436695705187065837202947671258357332D+00
      Omega( 9) = 0.8666922993395973932656203841951025879098D+00
      Omega(10) = 1.7462992361734780941443997970274892850284D+00
      Omega(11) = 3.7861326498591262976374516302513484333758D+00
C
      Alpha( 1) = 0.0001620957580603920404350047751169776322D+00
      Alpha( 2) = 0.0011377383639357606060877822502092060475D+00
      Alpha( 3) = 0.0044352997606686039615651190055195529638D+00
      Alpha( 4) = 0.0140900437920721600546582500876335863893D+00
      Alpha( 5) = 0.0394651913802239503377433139930108296767D+00
      Alpha( 6) = 0.1008699769730821261219796611263710417461D+00
      Alpha( 7) = 0.2401783505512109114215374416900772303052D+00
      Alpha( 8) = 0.5403404704563012579177951055786621736843D+00
      Alpha( 9) = 1.1614778624872791111619779735875113146903D+00
      Alpha(10) = 2.4160862126637673468528694220580632645579D+00
      Alpha(11) = 5.0128780625618468073760591874332703810069D+00
C
      Xi( 1) = 1.014239942221573816377028220970D+00
      Xi( 2) = 1.131059076026157462959570443317D+00
      Xi( 3) = 1.380331395988050338543944028924D+00
      Xi( 4) = 1.794852498112521126615546973593D+00
      Xi( 5) = 2.429950200208132349722484066845D+00
      Xi( 6) = 3.374759709016794407112141418459D+00
      Xi( 7) = 4.770378813680489580011795380798D+00
      Xi( 8) = 6.838212765075277659379549000107D+00
      Xi( 9) = 9.927096669501406712987634506362D+00
      Xi(10) = 14.593081887978388286493391934329D+00
      Xi(11) = 21.736488130016903172828102164260D+00
      Xi(12) = 32.842314094798293326593618957077D+00
      Xi(13) = 50.412076426022924838243888245870D+00
      Xi(14) = 78.763800701877293090891196669645D+00
      Xi(15) = 125.557583029224057967476735342416D+00
      Xi(16) = 204.829876278781896445368992942804D+00
      Xi(17) = 343.264066898434656999183900438766D+00
      Xi(18) = 593.830362490899102856367619551747D+00
      Xi(19) = 1067.014697285996970399502004056558D+00
      Xi(20) = 2003.005390344890530140986584228813D+00
      Xi(21) = 3910.001707684114783170059581607347D+00
      Xi(22) = 7123.500668713473708226757707961951D+00
      GOTO 9999
C
C     ----- K = 11 , R = 9E3 -----
C
 1139 CONTINUE
C
      RIni = 9.0D+03
C
      Omega( 1) = 0.0004078691872318507710981469102911461944D+00
      Omega( 2) = 0.0016348875607537805668889520794434533357D+00
      Omega( 3) = 0.0052581081186506578951953894737053607500D+00
      Omega( 4) = 0.0148087257998724855491222808859252957348D+00
      Omega( 5) = 0.0377381223895507049116086317058771371080D+00
      Omega( 6) = 0.0891318209355365257209619783695941919177D+00
      Omega( 7) = 0.1983032656250016002305114684589604223675D+00
      Omega( 8) = 0.4206221039756080757492703176003701059926D+00
      Omega( 9) = 0.8608396444681421005233439014414642542761D+00
      Omega(10) = 1.7390161743430621089662874156900329580822D+00
      Omega(11) = 3.7778109552097616187157913536509568075417D+00
C
      Alpha( 1) = 0.0001483448807384251019126060000628136715D+00
      Alpha( 2) = 0.0010585558771085235814493063512785331337D+00
      Alpha( 3) = 0.0041982030737258507678037679385074643079D+00
      Alpha( 4) = 0.0135089149642617613732045560304140252583D+00
      Alpha( 5) = 0.0381940850164093613008117842483324366754D+00
      Alpha( 6) = 0.0983215967127150371909316377283527543796D+00
      Alpha( 7) = 0.2354400512436616309838548211086362016431D+00
      Alpha( 8) = 0.5321321943865780585629192411811771989960D+00
      Alpha( 9) = 1.1482339764255602875724429634551881918014D+00
      Alpha(10) = 2.3962399342344530084672471481610500632087D+00
      Alpha(11) = 4.9852187457262847206443390213337352179224D+00
C
      Xi( 1) = 1.014350075600351303397629598724D+00
      Xi( 2) = 1.132098149109364029003994833733D+00
      Xi( 3) = 1.383487455589817327443469263670D+00
      Xi( 4) = 1.801873039568553124738052584330D+00
      Xi( 5) = 2.443597017918493595782167271402D+00
      Xi( 6) = 3.399591860500832806975604616362D+00
      Xi( 7) = 4.814079903885744832637422208066D+00
      Xi( 8) = 6.913864059193249945838022352262D+00
      Xi( 9) = 10.057187199621232224708289315629D+00
      Xi(10) = 14.816750600359321647300769875244D+00
      Xi(11) = 22.122812980254363109111626073400D+00
      Xi(12) = 33.515353358244164720669555634913D+00
      Xi(13) = 51.599133570115167723296734081373D+00
      Xi(14) = 80.891508404224807551874398825476D+00
      Xi(15) = 129.449498973704297075104641123744D+00
      Xi(16) = 212.130668776373502368293166853164D+00
      Xi(17) = 357.396008057185506134878494322038D+00
      Xi(18) = 622.279099349365191984340128783515D+00
      Xi(19) = 1127.278119364376255506776658421586D+00
      Xi(20) = 2139.390923877392672647346216763253D+00
      Xi(21) = 4246.099010265820874021613917648210D+00
      Xi(22) = 7946.161467249157825065708493639249D+00
      GOTO 9999
C
C     ----- K = 11 , R = 1E4 -----
C
 1141 CONTINUE
C
      RIni = 1.0D+04
C
      Omega( 1) = 0.0003785550695185704905331018800670669910D+00
      Omega( 2) = 0.0015517451432116935316520689837685442747D+00
      Omega( 3) = 0.0050662753610968026142610148073591735951D+00
      Omega( 4) = 0.0144023942219292172832968896602212627300D+00
      Omega( 5) = 0.0369481737121121147226652983025507737125D+00
      Omega( 6) = 0.0877136968671946349224198893890225292580D+00
      Omega( 7) = 0.1959481538043601072551129840815242744156D+00
      Omega( 8) = 0.4170227140801808644366994310859197980790D+00
      Omega( 9) = 0.8558390892074689600002926348487619634398D+00
      Omega(10) = 1.7327854324885193371000649542601479424775D+00
      Omega(11) = 3.7706904785971858274761653317952436736959D+00
C
      Alpha( 1) = 0.0001371172950178501341687661597154926363D+00
      Alpha( 2) = 0.0009937039401902874349137946216603545224D+00
      Alpha( 3) = 0.0040024563087556733581913622193465540633D+00
      Alpha( 4) = 0.0130252872174303745869984312404088466764D+00
      Alpha( 5) = 0.0371294655008602045503181190294617053382D+00
      Alpha( 6) = 0.0961764698255383165804120737529814277877D+00
      Alpha( 7) = 0.2314355666053775903228621052543090286235D+00
      Alpha( 8) = 0.5251727289213194890793966196351760800098D+00
      Alpha( 9) = 1.1369760622900190238709558498975127349695D+00
      Alpha(10) = 2.3793366605690113944199248763311516086105D+00
      Alpha(11) = 4.9616300421987320260451548303848312571063D+00
C
      Xi( 1) = 1.014445078090098826813052201867D+00
      Xi( 2) = 1.132994811083856317285821491136D+00
      Xi( 3) = 1.386212832419826497727145808714D+00
      Xi( 4) = 1.807941230908676616189953434422D+00
      Xi( 5) = 2.455406801772977036730019606559D+00
      Xi( 6) = 3.421113194397476642890504328065D+00
      Xi( 7) = 4.852020621598441331347867366830D+00
      Xi( 8) = 6.979672539648318155876965951379D+00
      Xi( 9) = 10.170604636139685588228087631535D+00
      Xi(10) = 15.012230965278301672398419963628D+00
      Xi(11) = 22.461386446764169332393490385869D+00
      Xi(12) = 34.107029147621249549893196473249D+00
      Xi(13) = 52.646324094495085033768022597656D+00
      Xi(14) = 82.776011367184239880812057776893D+00
      Xi(15) = 132.912412796370639544996095082752D+00
      Xi(16) = 218.661691303756028137428302215994D+00
      Xi(17) = 370.119212246490729811831954521040D+00
      Xi(18) = 648.091812080745762469913273662314D+00
      Xi(19) = 1182.497918253986295522572902427783D+00
      Xi(20) = 2266.010502890212877868947316528647D+00
      Xi(21) = 4564.472467287974115635051930439658D+00
      Xi(22) = 8756.418598288730433587545576301636D+00
      GOTO 9999
C
C     ----- K = 11 , R = 2E5 (Best : R = 109000) -----
C
 1199 CONTINUE
C
      RIni = 2.0D+05
C
      Omega( 1) = 0.0001134976878119174444673697307829282610D+00
      Omega( 2) = 0.0007457000167406750592463748688546593124D+00
      Omega( 3) = 0.0030526245415056300336642704359654820756D+00
      Omega( 4) = 0.0099235004487624246363231851619790546692D+00
      Omega( 5) = 0.0279373537885593099774266066109840167542D+00
      Omega( 6) = 0.0711041222257731397104010308274357754499D+00
      Omega( 7) = 0.1677705024084778823205898676262926016989D+00
      Omega( 8) = 0.3732056083739767542541750783291476523118D+00
      Omega( 9) = 0.7941436626469130445247178451406711019445D+00
      Omega(10) = 1.6552701124330565916236215895374073170387D+00
      Omega(11) = 3.6820059233978352326507876846051203756360D+00
C
      Alpha( 1) = 0.0000355034846592468107132831054925398845D+00
      Alpha( 2) = 0.0003905739575885531102960276960463920082D+00
      Alpha( 3) = 0.0020671330057150484115067632599066493881D+00
      Alpha( 4) = 0.0079783846601041099849384525058426653743D+00
      Alpha( 5) = 0.0255498864500882378069962930916547705351D+00
      Alpha( 6) = 0.0720851227548212716324249615951202940778D+00
      Alpha( 7) = 0.1853055788144508502635196198204958761835D+00
      Alpha( 8) = 0.4433472301960455170871319707792679309932D+00
      Alpha( 9) = 1.0024332764988156885743383894826763480523D+00
      Alpha(10) = 2.1748065663060481430245152179026035810239D+00
      Alpha(11) = 4.6738220578828596602459499731452297055512D+00
C
      Xi( 1) = 1.015690089382477667647071417623D+00
      Xi( 2) = 1.144775026302406757206311938013D+00
      Xi( 3) = 1.422180113927765727826688091362D+00
      Xi( 4) = 1.888518573347652048733183449691D+00
      Xi( 5) = 2.613470594874711714662718353708D+00
      Xi( 6) = 3.711993684861224562093751089265D+00
      Xi( 7) = 5.370779642078565153506114482340D+00
      Xi( 8) = 7.891427121501472941120930437719D+00
      Xi( 9) = 11.765526964896047277912838335823D+00
      Xi(10) = 17.807606050638013578391039004600D+00
      Xi(11) = 27.395214005522875326537057816267D+00
      Xi(12) = 42.915949154636157513581506961486D+00
      Xi(13) = 68.624694514972746979541984302386D+00
      Xi(14) = 112.360691745346326406163051103704D+00
      Xi(15) = 189.131571697762011746712929038949D+00
      Xi(16) = 329.035348174625998651787739390784D+00
      Xi(17) = 596.051036400452403984928650970687D+00
      Xi(18) = 1135.976006601578366161930944144842D+00
      Xi(19) = 2316.370944283527541607270450185752D+00
      Xi(20) = 5192.120834502648774044786250669858D+00
      Xi(21) = 13479.416793377781868201736870105378D+00
      Xi(22) = 47385.042527437597300377092324197292D+00
      GOTO 9999
C
C     ===== K = 12 =====
C
 1200 CONTINUE
C
      GOTO (1211,1211,1211,1212,1213,1214,1215,1216,1217,1218,1219,1221,
     *      1222,1223,1224,1225,1226,1227,1228,1229,1231,1232,1233,1234,
     *      1235,1236,1237,1238,1239,1241,1299),InitR
C
C     ----- K = 12 , R = 1E1 -----
C
 1211 CONTINUE
C
      RIni = 1.0D+01
C
      Omega( 1) = 0.0711846766353695929835615312242236285556D+00
      Omega( 2) = 0.1690895322362734570071068213037435157275D+00
      Omega( 3) = 0.2757926387649254820061406334730236267205D+00
      Omega( 4) = 0.3977636094386997021865592927269972278737D+00
      Omega( 5) = 0.5441057434880599478588085460284418104493D+00
      Omega( 6) = 0.7279147631051287236844671291180475236615D+00
      Omega( 7) = 0.9682939484624312369894644225709612328501D+00
      Omega( 8) = 1.2936786640404163718212279809982589995343D+00
      Omega( 9) = 1.7493131133110281919592482458014615076536D+00
      Omega(10) = 2.4187704026228852782157835443754834159336D+00
      Omega(11) = 3.5024366140575561975791657465251205394452D+00
      Omega(12) = 5.7855524079892674525239792959752094247960D+00
C
      Alpha( 1) = 0.0276731292622081597226468612990557538467D+00
      Alpha( 2) = 0.1472866131015313441898935409302140442378D+00
      Alpha( 3) = 0.3687581752417785056644226188149460199384D+00
      Alpha( 4) = 0.7039332404419826932288721532682274073522D+00
      Alpha( 5) = 1.1723654412494890637288710855123952114809D+00
      Alpha( 6) = 1.8045683154882165159836951429106477462483D+00
      Alpha( 7) = 2.6469524304892961248745347768718261249887D+00
      Alpha( 8) = 3.7693192226293995227785066370884692332766D+00
      Alpha( 9) = 5.2773524398830704345772502072975385090103D+00
      Alpha(10) = 7.3381749735963551352184441878279130833107D+00
      Alpha(11) = 10.2488461667396625765641404726125074375886D+00
      Alpha(12) = 14.7099922873718894451508520937466073519317D+00
C
      Xi( 1) = 1.002765923971119948229606400059D+00
      Xi( 2) = 1.024986275526004813121727643477D+00
      Xi( 3) = 1.069923061571954508281422080440D+00
      Xi( 4) = 1.138574075278952991139604089543D+00
      Xi( 5) = 1.232450417327966717367764004631D+00
      Xi( 6) = 1.353597504629638669632583913849D+00
      Xi( 7) = 1.504622215175725945452735177188D+00
      Xi( 8) = 1.688724456972242326537568535283D+00
      Xi( 9) = 1.909718822962894374246738127265D+00
      Xi(10) = 2.172034255460995098938742353667D+00
      Xi(11) = 2.480662653811127114304030394365D+00
      Xi(12) = 2.841026290314171571073711963784D+00
      Xi(13) = 3.258702872965208679426757365505D+00
      Xi(14) = 3.738936981267856123891876740650D+00
      Xi(15) = 4.285822352545366410898897147064D+00
      Xi(16) = 4.901021586284760217771400148123D+00
      Xi(17) = 5.581889846714328228590579206170D+00
      Xi(18) = 6.318929117702670764707834072382D+00
      Xi(19) = 7.092747329803447991613402079381D+00
      Xi(20) = 7.871138577292408604162143870298D+00
      Xi(21) = 8.607632829144091446416098278860D+00
      Xi(22) = 9.243536184503822437062703354371D+00
      Xi(23) = 9.715357748524472440189436728275D+00
      Xi(24) = 9.967640980357458236130185902368D+00
      GOTO 9999
C
C     ----- K = 12 , R = 2E1 -----
C
 1212 CONTINUE
C
      RIni = 2.0D+01
C
      Omega( 1) = 0.0428025346628120542797864885725678263384D+00
      Omega( 2) = 0.1026156437369573046664321044896261980739D+00
      Omega( 3) = 0.1703341135440457080398339237414795377390D+00
      Omega( 4) = 0.2524250366464257813270237462699441266523D+00
      Omega( 5) = 0.3585406407629632903914448927018554513779D+00
      Omega( 6) = 0.5029449307365428544934135568489352863253D+00
      Omega( 7) = 0.7061343335450083705143002610338243130172D+00
      Omega( 8) = 0.9975940043005185541252592684724476157498D+00
      Omega( 9) = 1.4229943025732463503154473505318833304045D+00
      Omega(10) = 2.0653230769823831772941113982255956216250D+00
      Omega(11) = 3.1208980779060891430111424060811486924649D+00
      Omega(12) = 5.3514751896835863285289969226710127259139D+00
C
      Alpha( 1) = 0.0166219868333402929036517929953231664797D+00
      Alpha( 2) = 0.0888764190139657434596505598822790972235D+00
      Alpha( 3) = 0.2244621641849088582339188935321772078169D+00
      Alpha( 4) = 0.4343008089118843558320594289146043820438D+00
      Alpha( 5) = 0.7372692134906314825708306681661952097784D+00
      Alpha( 6) = 1.1640767884289861903166288237265746374760D+00
      Alpha( 7) = 1.7626592744339100066490108598138419893075D+00
      Alpha( 8) = 2.6056037700656303733332680439005457628809D+00
      Alpha( 9) = 3.8021293786557150470167831812773329147603D+00
      Alpha(10) = 5.5228868604047138988685783367316162184579D+00
      Alpha(11) = 8.0665421480846943892706324419350494281389D+00
      Alpha(12) = 12.1245972500037078271178936361707201285753D+00
C
      Xi( 1) = 1.003803596370103570420417860021D+00
      Xi( 2) = 1.034415659365927397353619732190D+00
      Xi( 3) = 1.096621417100223667496008972400D+00
      Xi( 4) = 1.192406516394687885436638741243D+00
      Xi( 5) = 1.324814247094212635584738191330D+00
      Xi( 6) = 1.498042874009141811977342817475D+00
      Xi( 7) = 1.717591997493018617679076676108D+00
      Xi( 8) = 1.990454740061872790672301003490D+00
      Xi( 9) = 2.325343203762643022602940590104D+00
      Xi(10) = 2.732929767195396435815885061160D+00
      Xi(11) = 3.226077701749926812105004336750D+00
      Xi(12) = 3.820012877579278808134674583208D+00
      Xi(13) = 4.532351036147622822067987380379D+00
      Xi(14) = 5.382800296171293342756963840046D+00
      Xi(15) = 6.392248841049624770004750073937D+00
      Xi(16) = 7.580727023437338037929050793551D+00
      Xi(17) = 8.963474656843955691402914265353D+00
      Xi(18) = 10.544065759858001988386044800450D+00
      Xi(19) = 12.303519685154400178407618771104D+00
      Xi(20) = 14.185251542330838449464114336962D+00
      Xi(21) = 16.078706928215402250509358239583D+00
      Xi(22) = 17.810527882264631542238997319316D+00
      Xi(23) = 19.158744744777995581280771730803D+00
      Xi(24) = 19.903266289867562605087059068865D+00
      GOTO 9999
C
C     ----- K = 12 , R = 3E1 -----
C
 1213 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.0314847850052882651287426002406633074315D+00
      Omega( 2) = 0.0759880787044356675646785553857043282733D+00
      Omega( 3) = 0.1277666945808081383489799090491523259061D+00
      Omega( 4) = 0.1931751032420674228553669066199560688801D+00
      Omega( 5) = 0.2819959712025565720780375528198291590343D+00
      Omega( 6) = 0.4087743922620433270642172074449405272389D+00
      Omega( 7) = 0.5942255934432832301756929882508018181397D+00
      Omega( 8) = 0.8680605358456985755463870824133465475825D+00
      Omega( 9) = 1.2764587208952323604902595599597248110513D+00
      Omega(10) = 1.9030088361974479215735917003726740404090D+00
      Omega(11) = 2.9432226543582369869284792729402511213266D+00
      Omega(12) = 5.1490168368308802796830259129734486123198D+00
C
      Alpha( 1) = 0.0122175904856221182198541905933319995370D+00
      Alpha( 2) = 0.0655438352296026405680593254765575750298D+00
      Alpha( 3) = 0.1665923720397374805578206632961091315792D+00
      Alpha( 4) = 0.3255827057148261954432760800770552123140D+00
      Alpha( 5) = 0.5606935393689712887610454572495655156672D+00
      Alpha( 6) = 0.9021601691438875129176906686456760553483D+00
      Alpha( 7) = 1.3977054457316340485822231398671533497691D+00
      Alpha( 8) = 2.1199036671113539323854646490019604243571D+00
      Alpha( 9) = 3.1783136457891825786634815154130251357856D+00
      Alpha(10) = 4.7445349934637615109400687884999570087530D+00
      Alpha(11) = 7.1183009631362091041914541555968298780499D+00
      Alpha(12) = 10.9882665995376886869036581195757662499091D+00
C
      Xi( 1) = 1.004447934429248711295755425166D+00
      Xi( 2) = 1.040285530941391825554553918209D+00
      Xi( 3) = 1.113322679833070314174732484691D+00
      Xi( 4) = 1.226321105027237134163335674497D+00
      Xi( 5) = 1.383537201353583936665019027412D+00
      Xi( 6) = 1.590897874604819522731564973572D+00
      Xi( 7) = 1.856271132776980358988251629260D+00
      Xi( 8) = 2.189828918204150003242944300119D+00
      Xi( 9) = 2.604493478509141737766569013601D+00
      Xi(10) = 3.116455392284870871922805579679D+00
      Xi(11) = 3.745757447736137540211798224732D+00
      Xi(12) = 4.516908253769600511181442348274D+00
      Xi(13) = 5.459441990165533564044464398535D+00
      Xi(14) = 6.608214497704099591336018848509D+00
      Xi(15) = 8.003018535318867998987202305727D+00
      Xi(16) = 9.686740724612728241707526599669D+00
      Xi(17) = 11.700636306985161984479948404925D+00
      Xi(18) = 14.074413061266897988048352186041D+00
      Xi(19) = 16.807854304494409820869282867761D+00
      Xi(20) = 19.840822097852637208303949734045D+00
      Xi(21) = 23.012711666017984169646526382280D+00
      Xi(22) = 26.025801437599993062160796952753D+00
      Xi(23) = 28.450293676293621054906868295120D+00
      Xi(24) = 29.820317503622383885700242167616D+00
      GOTO 9999
C
C     ----- K = 12 , R = 4E1 -----
C
 1214 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.0252371170645091488656224908843084264731D+00
      Omega( 2) = 0.0612388028802558352680415060254048853494D+00
      Omega( 3) = 0.1040508555604487832515667698751737901830D+00
      Omega( 4) = 0.1599052433098304811453452961567123224995D+00
      Omega( 5) = 0.2385711867493230491088507250196926179342D+00
      Omega( 6) = 0.3545821249596775316106564424067926211137D+00
      Omega( 7) = 0.5285932206532508617451149302013391206856D+00
      Omega( 8) = 0.7904471850597390465286047611659370204507D+00
      Omega( 9) = 1.1868782571026183060916570077836240670877D+00
      Omega(10) = 1.8022353684717613577469241814554834490991D+00
      Omega(11) = 2.8319457784449118414610985228208051012189D+00
      Omega(12) = 5.0221823554408107077688672159609950540471D+00
C
      Alpha( 1) = 0.0097872502132220878271146122934220912271D+00
      Alpha( 2) = 0.0526466975146047576613141973089771141758D+00
      Alpha( 3) = 0.1345082558306065039079946546451971300939D+00
      Alpha( 4) = 0.2650565360294568865312265365519195370325D+00
      Alpha( 5) = 0.4618694245718520602968660410114765113576D+00
      Alpha( 6) = 0.7545813463702144361223804469585019205624D+00
      Alpha( 7) = 1.1902704560564051072812929987776442430913D+00
      Alpha( 8) = 1.8408744447358315109251197805306787813606D+00
      Alpha( 9) = 2.8156533824155635659103324996976880356669D+00
      Alpha(10) = 4.2866097725398282212473655139461925500655D+00
      Alpha(11) = 6.5543641021523252536244708021939686659607D+00
      Alpha(12) = 10.3064204128060784605713795514247976825573D+00
C
      Xi( 1) = 1.004917393904752588975129445892D+00
      Xi( 2) = 1.044569558409484608335962563253D+00
      Xi( 3) = 1.125552323994750328188009513841D+00
      Xi( 4) = 1.251274713961436201286464564575D+00
      Xi( 5) = 1.427010101392351663299878916646D+00
      Xi( 6) = 1.660149052252017745504378276600D+00
      Xi( 7) = 1.960591483574498210826556054975D+00
      Xi( 8) = 2.341277156839835152726814193613D+00
      Xi( 9) = 2.818850155076643253281368228791D+00
      Xi(10) = 3.414460705474536696179058137268D+00
      Xi(11) = 4.154720695961834792900996271570D+00
      Xi(12) = 5.072811401955001301983944062002D+00
      Xi(13) = 6.209671722714000902283715266350D+00
      Xi(14) = 7.615067047492341609747634745275D+00
      Xi(15) = 9.348072787550938980011272505521D+00
      Xi(16) = 11.475990199432322107046811066056D+00
      Xi(17) = 14.069749403353831805618678885139D+00
      Xi(18) = 17.192254390006447204580775078853D+00
      Xi(19) = 20.873827331033322897077542101840D+00
      Xi(20) = 25.067204801490871780497715626268D+00
      Xi(21) = 29.578089799215747926139363777054D+00
      Xi(22) = 33.987322349454254028638944262752D+00
      Xi(23) = 37.627748199051418528937595908701D+00
      Xi(24) = 39.723083912498277824043579897761D+00
      GOTO 9999
C
C     ----- K = 12 , R = 5E1 -----
C
 1215 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.0212246513027600224522744220764791656109D+00
      Omega( 2) = 0.0517398368864022841059815277509059683325D+00
      Omega( 3) = 0.0887040340301526822420282065995156273175D+00
      Omega( 4) = 0.1382327771396707316017620978398916520291D+00
      Omega( 5) = 0.2100216299707748596143968775051824593447D+00
      Omega( 6) = 0.3184838878625700616721051761093264076408D+00
      Omega( 7) = 0.4841478354476556527426385490020521729093D+00
      Omega( 8) = 0.7369760980033169930946204684296674258803D+00
      Omega( 9) = 1.1242221404269488538654742093569893768290D+00
      Omega(10) = 1.7309580289551620669533799068418034039496D+00
      Omega(11) = 2.7527605274972598524679118225932938912592D+00
      Omega(12) = 4.9319359578631724503293420891480991485878D+00
C
      Alpha( 1) = 0.0082269231802614559072336203847386482124D+00
      Alpha( 2) = 0.0443547657196888649133151207720748843144D+00
      Alpha( 3) = 0.1138293811452207303662378240494135184235D+00
      Alpha( 4) = 0.2259109878434789831300058896013105425027D+00
      Alpha( 5) = 0.3976625203871757075271831105567343911389D+00
      Alpha( 6) = 0.6581133163982475239948272127676887066627D+00
      Alpha( 7) = 1.0535979897201334513131415482334318767244D+00
      Alpha( 8) = 1.6552845367932827516915111898399004530802D+00
      Alpha( 9) = 2.5719827479339771442428536074586986615031D+00
      Alpha(10) = 3.9759001427243164585468432337123090292152D+00
      Alpha(11) = 6.1684054473566169670729897323013801724301D+00
      Alpha(12) = 9.8365133085386865204394202244486677955138D+00
C
      Xi( 1) = 1.005286713345526050981081078728D+00
      Xi( 2) = 1.047944183621078399456030827430D+00
      Xi( 3) = 1.135210433843728024933504572314D+00
      Xi( 4) = 1.271053249752736233921072972475D+00
      Xi( 5) = 1.461628112483563147485331190634D+00
      Xi( 6) = 1.715604595616679065570898965465D+00
      Xi( 7) = 2.044676138922400543103743975060D+00
      Xi( 8) = 2.464253002358599976655273433579D+00
      Xi( 9) = 2.994337852047697069290260740715D+00
      Xi(10) = 3.660609812792010766017633538461D+00
      Xi(11) = 4.495759111326378142103887824987D+00
      Xi(12) = 5.541107071300827739690486595237D+00
      Xi(13) = 6.848473528743382710698428539331D+00
      Xi(14) = 8.482121150474734669159893041268D+00
      Xi(15) = 10.520303395708345418034512785965D+00
      Xi(16) = 13.055308816817881779892562832401D+00
      Xi(17) = 16.189663296095421140055603537178D+00
      Xi(18) = 20.023783993073709482762856382010D+00
      Xi(19) = 24.626643041042886221950181102613D+00
      Xi(20) = 29.976661609065542288612982169127D+00
      Xi(21) = 35.861161955219889561213841844278D+00
      Xi(22) = 41.747243713964529397697500456843D+00
      Xi(23) = 46.711681987399490362938836440776D+00
      Xi(24) = 49.613923452080444308903706129854D+00
      GOTO 9999
C
C     ----- K = 12 , R = 6E1 -----
C
 1216 CONTINUE
C
      RIni = 6.0D+01
C
      Omega( 1) = 0.0184080983839379267157781994157517146959D+00
      Omega( 2) = 0.0450560812760502163758973091095549889928D+00
      Omega( 3) = 0.0778611314389885509642088082249422598125D+00
      Omega( 4) = 0.1228310593868079889846747602966114953915D+00
      Omega( 5) = 0.1895609344573431226393814622022127025502D+00
      Omega( 6) = 0.2923043033618055340429765093235303652364D+00
      Omega( 7) = 0.4514544295919487180892420985500734786910D+00
      Omega( 8) = 0.6970886541216345195170166204601258641560D+00
      Omega( 9) = 1.0769243506632811455646370202288153450354D+00
      Omega(10) = 1.6766859343620682565615884196397189498384D+00
      Omega(11) = 2.6921885124990737729079592899239514736109D+00
      Omega(12) = 4.8629171815900725591060027586109981712070D+00
C
      Alpha( 1) = 0.0071319631843329120966111414180871319601D+00
      Alpha( 2) = 0.0385288520026886962092113695504957959770D+00
      Alpha( 3) = 0.0992694979607395232008675012291831762923D+00
      Alpha( 4) = 0.1982658028996117775436891852991472262602D+00
      Alpha( 5) = 0.3521334070004843228962176232554881494252D+00
      Alpha( 6) = 0.5893267643752341043814242660392466177655D+00
      Alpha( 7) = 0.9554417205575109880359534197946658196088D+00
      Alpha( 8) = 1.5208802148863778551423145413146187365783D+00
      Alpha( 9) = 2.3939769277419587804271033926184486517741D+00
      Alpha(10) = 3.7470487790881047123620062322402191057336D+00
      Alpha(11) = 5.8821023096538253248366512337952372035943D+00
      Alpha(12) = 9.4859726615850346702341067128827489796095D+00
C
      Xi( 1) = 1.005590897534543594229843044285D+00
      Xi( 2) = 1.050726610075362197922049789511D+00
      Xi( 3) = 1.143190120587738044482910282351D+00
      Xi( 4) = 1.287442936812643280264194245444D+00
      Xi( 5) = 1.490422409980629544698493837984D+00
      Xi( 6) = 1.761939900711092939341803620401D+00
      Xi( 7) = 2.115302543429060096007165991949D+00
      Xi( 8) = 2.568162320285456149153088811765D+00
      Xi( 9) = 3.143597672735349753516190252078D+00
      Xi(10) = 3.871472651209925308103676488969D+00
      Xi(11) = 4.790153673573017386547873153191D+00
      Xi(12) = 5.948648593343000728125152809511D+00
      Xi(13) = 7.409180553411453687331217343726D+00
      Xi(14) = 9.250068242662710976849382848997D+00
      Xi(15) = 11.568467124960551785546603031918D+00
      Xi(16) = 14.481811087291866183002220225262D+00
      Xi(17) = 18.125290118076745983499709069520D+00
      Xi(18) = 22.639673152777500846760672104097D+00
      Xi(19) = 28.138486427565130274658367071083D+00
      Xi(20) = 34.636063633484886807589742119262D+00
      Xi(21) = 41.915097966903564364465895053513D+00
      Xi(22) = 49.338638589697195634825588328454D+00
      Xi(23) = 55.715382113648056268406794089287D+00
      Xi(24) = 59.494370263074479988907317107305D+00
      GOTO 9999
C
C     ----- K = 12 , R = 7E1 -----
C
 1217 CONTINUE
C
      RIni = 7.0D+01
C
      Omega( 1) = 0.0163112546177454600382497129987258155381D+00
      Omega( 2) = 0.0400697356456504853207400973502672059112D+00
      Omega( 3) = 0.0697424933664369035060106791212852783701D+00
      Omega( 4) = 0.1112380896871291995238462124717671031249D+00
      Omega( 5) = 0.1740396434997305038887283057391641705181D+00
      Omega( 6) = 0.2722299633243371610688034201164242631421D+00
      Omega( 7) = 0.4260759427480407430805908058912834235343D+00
      Omega( 8) = 0.6657626611697179455900269529067259099975D+00
      Omega( 9) = 1.0394169828252716300445029906640570516174D+00
      Omega(10) = 1.6333462645016366815691605252069962261885D+00
      Omega(11) = 2.6436380689380596005228635236861123303242D+00
      Omega(12) = 4.8076086410744625688519493511563496213057D+00
C
      Alpha( 1) = 0.0063170013560629504831786822007660964218D+00
      Alpha( 2) = 0.0341880877391648044947766987150528095185D+00
      Alpha( 3) = 0.0884008694023858860409491484955157858394D+00
      Alpha( 4) = 0.1775737848854596828896594898350436153578D+00
      Alpha( 5) = 0.3179284457802149525178312533535418538122D+00
      Alpha( 6) = 0.5373835811949758285970685234289589971013D+00
      Alpha( 7) = 0.8808348910539448565386669298682420503610D+00
      Alpha( 8) = 1.4179620136198704534284675826683042032528D+00
      Alpha( 9) = 2.2566378506322052439284769898009130884020D+00
      Alpha(10) = 3.5692344888793875149098078214393581220065D+00
      Alpha(11) = 5.6583087139239065569383535514447203240707D+00
      Alpha(12) = 9.2106775903010455952210633334686917805811D+00
C
      Xi( 1) = 1.005849257632735276350374742371D+00
      Xi( 2) = 1.053092012352318902301644143460D+00
      Xi( 3) = 1.149985578372599481257465559025D+00
      Xi( 4) = 1.301434763600383661069984087000D+00
      Xi( 5) = 1.515081610792431805029810421015D+00
      Xi( 6) = 1.801771543866024114345994466824D+00
      Xi( 7) = 2.176283703217635774997476549864D+00
      Xi( 8) = 2.658329448751052618334783650411D+00
      Xi( 9) = 3.273833265818200818257979745063D+00
      Xi(10) = 4.056563155826173982469035061982D+00
      Xi(11) = 5.050223975979483648397760919835D+00
      Xi(12) = 6.311120439927974264202742427088D+00
      Xi(13) = 7.911451037221845648556711738664D+00
      Xi(14) = 9.943163719482029876302331405924D+00
      Xi(15) = 12.521980802474876036736728401877D+00
      Xi(16) = 15.790426333215222919406639512374D+00
      Xi(17) = 19.916953022029388532809779377430D+00
      Xi(18) = 25.084653646156171050846106496124D+00
      Xi(19) = 31.456063148345611419492984239810D+00
      Xi(20) = 39.089771616338169758081422244800D+00
      Xi(21) = 47.776116506157187392711538365120D+00
      Xi(22) = 56.784586987926429016426688178854D+00
      Xi(23) = 64.648275342701393565703060772876D+00
      Xi(24) = 69.365518178125603089567707826291D+00
      GOTO 9999
C
C     ----- K = 12 , R = 8E1 -----
C
 1218 CONTINUE
C
      RIni = 8.0D+01
C
      Omega( 1) = 0.0146834403786877239651904761101597340911D+00
      Omega( 2) = 0.0361914353823823021965373943065191753021D+00
      Omega( 3) = 0.0634072655822392595469159763676980645641D+00
      Omega( 4) = 0.1021478552799039412541976142023703033601D+00
      Omega( 5) = 0.1617806182160848998003712870263903766954D+00
      Omega( 6) = 0.2562191165902423766050858416898350355950D+00
      Omega( 7) = 0.4056164627770592757797802480057569596283D+00
      Omega( 8) = 0.6402570726581067900870861131501499130536D+00
      Omega( 9) = 1.0086292613465813007090146613542458453594D+00
      Omega(10) = 1.5975625195441151351087236132109126174328D+00
      Omega(11) = 2.6034273539351463129000452401129450663575D+00
      Omega(12) = 4.7618102577925454188360887552988742754678D+00
C
      Alpha( 1) = 0.0056844758581019654055792875504971028988D+00
      Alpha( 2) = 0.0308158126566199542090130052834906493331D+00
      Alpha( 3) = 0.0799429238320965633604023477254418850180D+00
      Alpha( 4) = 0.1614319011604487479982859149396645648267D+00
      Alpha( 5) = 0.2911529715886109996993234144690987363902D+00
      Alpha( 6) = 0.4965287577366228496630103789177823614409D+00
      Alpha( 7) = 0.8218028957748642600398408797346405663120D+00
      Alpha( 8) = 1.3359859877786959548836079325617731683451D+00
      Alpha( 9) = 2.1465138972910534464599158210873497409921D+00
      Alpha(10) = 3.4257800602112389596123898094504056643927D+00
      Alpha(11) = 5.4768215137600342588310164959608528079116D+00
      Alpha(12) = 8.9865269269342936426375745284644835919607D+00
C
      Xi( 1) = 1.006073602738567478002973398787D+00
      Xi( 2) = 1.055147598456383129352301186721D+00
      Xi( 3) = 1.155899772802454390091542180130D+00
      Xi( 4) = 1.313638049284265273022702380512D+00
      Xi( 5) = 1.536647076774095812840698060953D+00
      Xi( 6) = 1.836719343566482923993561648945D+00
      Xi( 7) = 2.229990798652839172240558429650D+00
      Xi( 8) = 2.738082525376893865435776187489D+00
      Xi( 9) = 3.389574080726167901959217099161D+00
      Xi(10) = 4.221900745674179864213443336141D+00
      Xi(11) = 5.283817592672097366093830750344D+00
      Xi(12) = 6.638587627006000431613419454635D+00
      Xi(13) = 8.368001849960878511976158922181D+00
      Xi(14) = 10.577235893869860045013453486717D+00
      Xi(15) = 13.400224985438609952292932092632D+00
      Xi(16) = 17.004416677055524467765401652741D+00
      Xi(17) = 21.591855315333097315236288160634D+00
      Xi(18) = 27.389345058557642623858763286648D+00
      Xi(19) = 34.611908204340102266499679117828D+00
      Xi(20) = 43.369330502997703969542131474668D+00
      Xi(21) = 53.470561241119019674217227588997D+00
      Xi(22) = 64.102226806568861176838236559661D+00
      Xi(23) = 73.517454101237934040169452742930D+00
      Xi(24) = 79.228194435685670563018767609265D+00
      GOTO 9999
C
C     ----- K = 12 , R = 9E1 -----
C
 1219 CONTINUE
C
      RIni = 9.0D+01
C
      Omega( 1) = 0.0133794494576575845756776948422550610474D+00
      Omega( 2) = 0.0330793026375962093940706967853948583524D+00
      Omega( 3) = 0.0583083622425075984217843899637401250402D+00
      Omega( 4) = 0.0947986056263172510938591966422883672294D+00
      Omega( 5) = 0.1518020216293596127748923713118855971516D+00
      Omega( 6) = 0.2430698353940594271123919190014106561648D+00
      Omega( 7) = 0.3886538069394757499014358970157445583027D+00
      Omega( 8) = 0.6189288727611131342518785425355076768028D+00
      Omega( 9) = 0.9827038031723337512478820554573388790232D+00
      Omega(10) = 1.5672784492918067736659179511882200586115D+00
      Omega(11) = 2.5693057474657722766622769405842063861201D+00
      Omega(12) = 4.7229543998458347066858598051197759559727D+00
C
      Alpha( 1) = 0.0051778836534097643948459314521687302246D+00
      Alpha( 2) = 0.0281125858476668863738048582262551988720D+00
      Alpha( 3) = 0.0731525476894912956787680145820385035904D+00
      Alpha( 4) = 0.1484433217602887751442475999796855035129D+00
      Alpha( 5) = 0.2695385351817635452911781079876618605340D+00
      Alpha( 6) = 0.4634018779090883400240564321537206637913D+00
      Alpha( 7) = 0.7736733807857686019845110114712127824532D+00
      Alpha( 8) = 1.2687470578926418802480358127837689607986D+00
      Alpha( 9) = 2.0556487983990030488871264546801853612124D+00
      Alpha(10) = 3.3067709916274913978500443745289771868556D+00
      Alpha(11) = 5.3255742673944999232733821248331196329673D+00
      Alpha(12) = 8.7990705510743595623573964914498901634943D+00
C
      Xi( 1) = 1.006271692633365518279066341822D+00
      Xi( 2) = 1.056963864704113705179910898124D+00
      Xi( 3) = 1.161132291472661585084689972458D+00
      Xi( 4) = 1.324455033512620702180855036811D+00
      Xi( 5) = 1.555807813919135473845756223454D+00
      Xi( 6) = 1.867859209744250026559442279250D+00
      Xi( 7) = 2.278005022350426610330473287114D+00
      Xi( 8) = 2.809650170707710874684268076429D+00
      Xi( 9) = 3.493868344334070932940758402729D+00
      Xi(10) = 4.371557818077144736985256701089D+00
      Xi(11) = 5.496275775368103647111617249266D+00
      Xi(12) = 6.937944306517945176546180441335D+00
      Xi(13) = 8.787602244058466935232309946713D+00
      Xi(14) = 11.163280732117624457523652470314D+00
      Xi(15) = 14.216772640700797801678600951192D+00
      Xi(16) = 18.140225091983027677980389036350D+00
      Xi(17) = 23.169423359036196138904806218761D+00
      Xi(18) = 29.575886628057655853424323666445D+00
      Xi(19) = 37.629897665946181042201379440826D+00
      Xi(20) = 47.498288763048672721234222393605D+00
      Xi(21) = 59.018463345375563759709347522175D+00
      Xi(22) = 71.304831430491172651608966504000D+00
      Xi(23) = 82.328487884719088740759218580934D+00
      Xi(24) = 89.083051291139891222914393154042D+00
      GOTO 9999
C
C     ----- K = 12 , R = 1E2 -----
C
 1221 CONTINUE
C
      RIni = 1.0D+02
C
      Omega( 1) = 0.0123090459207837410891199329820988594975D+00
      Omega( 2) = 0.0305206065457638403504104975677568845072D+00
      Omega( 3) = 0.0541046151353604127178941081421914560678D+00
      Omega( 4) = 0.0887139631277540403485239867142908565256D+00
      Omega( 5) = 0.1434876391615258704037702205957849344031D+00
      Omega( 6) = 0.2320235858589071699863929515439409101418D+00
      Omega( 7) = 0.3742832847705676747721882657859993059901D+00
      Omega( 8) = 0.6007236370303555422225059989838058527312D+00
      Omega( 9) = 0.9604392175073996686953357870830672027296D+00
      Omega(10) = 1.5411563493400488027585543138187063050282D+00
      Omega(11) = 2.5398045743466134197257461080354801197245D+00
      Omega(12) = 4.6893656358078771848908739006134283044958D+00
C
      Alpha( 1) = 0.0047621173572975827785759539954868468925D+00
      Alpha( 2) = 0.0258922311893780472792723611891174684274D+00
      Alpha( 3) = 0.0675671840652446778192983384281333769650D+00
      Alpha( 4) = 0.1377372357479949906979791590666550860078D+00
      Alpha( 5) = 0.2516682772723749776536215222444425876347D+00
      Alpha( 6) = 0.4358989325752495758051034691060721115718D+00
      Alpha( 7) = 0.7335111188980595020819701346237451389243D+00
      Alpha( 8) = 1.2123305594792852413633249542002090493042D+00
      Alpha( 9) = 1.9789993733384250202030074650139113145997D+00
      Alpha(10) = 3.2058935052153564042483335772004693353665D+00
      Alpha(11) = 5.1968507098720845721086247781528300038190D+00
      Alpha(12) = 8.6390362946276927349695262670081774558639D+00
C
      Xi( 1) = 1.006448902079904277376797006038D+00
      Xi( 2) = 1.058589676293567137109648268467D+00
      Xi( 3) = 1.165821620001373944228856283090D+00
      Xi( 4) = 1.334165259123700095602108040005D+00
      Xi( 5) = 1.573044468555178850683283486855D+00
      Xi( 6) = 1.895942984412371630866846194952D+00
      Xi( 7) = 2.321435517447334613821280369095D+00
      Xi( 8) = 2.874602436329419588838945953668D+00
      Xi( 9) = 3.588872098063118456492612273756D+00
      Xi(10) = 4.508430174224262648730032765343D+00
      Xi(11) = 5.691416210890235599043768255356D+00
      Xi(12) = 7.214146415187596048188695130321D+00
      Xi(13) = 9.176595487619671475364169221933D+00
      Xi(14) = 11.709302775310305939730770585783D+00
      Xi(15) = 14.981573410594507048121393211204D+00
      Xi(16) = 19.209998861148907067439295381917D+00
      Xi(17) = 24.664136319335358502394250912459D+00
      Xi(18) = 31.660955242341479728279685090797D+00
      Xi(19) = 40.528232960375789543233659983912D+00
      Xi(20) = 51.494869298111737847267743717339D+00
      Xi(21) = 64.435607046864416112830209470985D+00
      Xi(22) = 78.403035185486382022124196566892D+00
      Xi(23) = 91.085884791545486179054069708627D+00
      Xi(24) = 98.930619049291083751118680567060D+00
      GOTO 9999
C
C     ----- K = 12 , R = 2E2 -----
C
 1222 CONTINUE
C
      RIni = 2.0D+02
C
      Omega( 1) = 0.0070860134344731771125858264120944696174D+00
      Omega( 2) = 0.0179638474505374610739656280819498412882D+00
      Omega( 3) = 0.0332648386084722072207381544928228578328D+00
      Omega( 4) = 0.0580597117881842633590919015851294915365D+00
      Omega( 5) = 0.1005771448110310824520792320757589521918D+00
      Omega( 6) = 0.1733333929652051708802809951159851209468D+00
      Omega( 7) = 0.2957310433985090605238550687250409509943D+00
      Omega( 8) = 0.4987160340350490686360009073840870996719D+00
      Omega( 9) = 0.8331516142667499745348702600011847607675D+00
      Omega(10) = 1.3896105342194977049505646982119344556850D+00
      Omega(11) = 2.3672949507125710425729098185243515217735D+00
      Omega(12) = 4.4930548591707977260610584746558515689685D+00
C
      Alpha( 1) = 0.0027347725482160792943084998921771111569D+00
      Alpha( 2) = 0.0150339160352873828102875271295335579680D+00
      Alpha( 3) = 0.0401109195501032844837787337544110499721D+00
      Alpha( 4) = 0.0846931137323594645930410068634941467280D+00
      Alpha( 5) = 0.1620925487078466827506242203815745028805D+00
      Alpha( 6) = 0.2958529950954934552818931248197031891323D+00
      Alpha( 7) = 0.5251822397506850208519398048867543593587D+00
      Alpha( 8) = 0.9139677614786228681837548792898928695649D+00
      Alpha( 9) = 1.5660334617940182692138323394814847233647D+00
      Alpha(10) = 2.6532976247909677335393152652898152155103D+00
      Alpha(11) = 4.4819518632296585563742907787343483505538D+00
      Alpha(12) = 7.7409496329154443465883628228141333238455D+00
C
      Xi( 1) = 1.007606335947693436188785098828D+00
      Xi( 2) = 1.069232017552700409715633889096D+00
      Xi( 3) = 1.196646752951617210668396051876D+00
      Xi( 4) = 1.398375867761524281394219793029D+00
      Xi( 5) = 1.687889530462865563822071057754D+00
      Xi( 6) = 2.084773905860931427803939275378D+00
      Xi( 7) = 2.616577766581499571571164941552D+00
      Xi( 8) = 3.321363400960225862375979621888D+00
      Xi( 9) = 4.251126249734825880005617548818D+00
      Xi(10) = 5.476484773152846827985046340714D+00
      Xi(11) = 7.093214112109250613340954982888D+00
      Xi(12) = 9.231335345132891039704459146265D+00
      Xi(13) = 12.067697332434482497542926893175D+00
      Xi(14) = 15.843238509328470788560527804378D+00
      Xi(15) = 20.886295899388681220759655943198D+00
      Xi(16) = 27.642822895472207384362506310538D+00
      Xi(17) = 36.712144547343591401145079800017D+00
      Xi(18) = 48.879107075080135788591739753883D+00
      Xi(19) = 65.110967290470128980561792531034D+00
      Xi(20) = 86.428747757007699487052931885955D+00
      Xi(21) = 113.430340398014508088153640130713D+00
      Xi(22) = 145.049438578274902361675557926901D+00
      Xi(23) = 176.322305548129662736145384371866D+00
      Xi(24) = 197.078080187922438468173247372306D+00
      GOTO 9999
C
C     ----- K = 12 , R = 3E2 -----
C
 1223 CONTINUE
C
      RIni = 3.0D+02
C
      Omega( 1) = 0.0051195983342860145902886647665402941598D+00
      Omega( 2) = 0.0131892853745070899709111679129480698691D+00
      Omega( 3) = 0.0251976096018116801238985458244457049659D+00
      Omega( 4) = 0.0458385770750394970663051368714446809349D+00
      Omega( 5) = 0.0827442424008601229071637253587034877000D+00
      Omega( 6) = 0.1478233441962730683889295710264910610476D+00
      Omega( 7) = 0.2601727262941121533506780294509042050777D+00
      Omega( 8) = 0.4509304574403051471076595696496625009786D+00
      Omega( 9) = 0.7718498864664846378143975291763467794226D+00
      Omega(10) = 1.3151307967790183471416820548682835578802D+00
      Omega(11) = 2.2815665390532840575178730180283537265495D+00
      Omega(12) = 4.3955515211933913343085589264802592879278D+00
C
      Alpha( 1) = 0.0019724053061639556097514082173144389643D+00
      Alpha( 2) = 0.0109300683169841590296897317242941216620D+00
      Alpha( 3) = 0.0296391592056178483410768670358592657976D+00
      Alpha( 4) = 0.0641708378072586352788820272807779332425D+00
      Alpha( 5) = 0.1266927676410249966438507224220089142364D+00
      Alpha( 6) = 0.2389728369268007877960482629453586866930D+00
      Alpha( 7) = 0.4379665566520968119018988873314413012849D+00
      Alpha( 8) = 0.7852367571901352752612203433901783000692D+00
      Alpha( 9) = 1.3828078771058658968462831695411807686469D+00
      Alpha(10) = 2.4020735504281130311826641943184768024366D+00
      Alpha(11) = 4.1504387185145925299664670937715982290683D+00
      Alpha(12) = 7.3183269826043906326751287139131818548776D+00
C
      Xi( 1) = 1.008267959695940689372230247844D+00
      Xi( 2) = 1.075334014570839409954960963489D+00
      Xi( 3) = 1.214423438364915396893949084411D+00
      Xi( 4) = 1.435707676780266528926595370752D+00
      Xi( 5) = 1.755348731706775427694952607105D+00
      Xi( 6) = 2.197070763825094897496795209868D+00
      Xi( 7) = 2.794641811616995056167384037238D+00
      Xi( 8) = 3.595319636244884395435278401365D+00
      Xi( 9) = 4.664555667667056663028973817475D+00
      Xi(10) = 6.092613434832621599392815658902D+00
      Xi(11) = 8.004000654849777432638113783270D+00
      Xi(12) = 10.570906296406753927148713945172D+00
      Xi(13) = 14.032353148255792877306669108606D+00
      Xi(14) = 18.721414303265493240049921475077D+00
      Xi(15) = 25.103699919348321019374470353114D+00
      Xi(16) = 33.830718480144582167856714960408D+00
      Xi(17) = 45.810545997389025451446142866985D+00
      Xi(18) = 62.290592020751041651971391388543D+00
      Xi(19) = 84.919463929991559805743328581684D+00
      Xi(20) = 115.667867926532999825472991517472D+00
      Xi(21) = 156.249955261180685589517125322345D+00
      Xi(22) = 206.183554648235459069716135616090D+00
      Xi(23) = 258.379990278291688771128420398782D+00
      Xi(24) = 294.757469626102057425054425721100D+00
      GOTO 9999
C
C     ----- K = 12 , R = 4E2 -----
C
 1224 CONTINUE
C
      RIni = 4.0D+02
C
      Omega( 1) = 0.0040632299910078106759216186403410597805D+00
      Omega( 2) = 0.0106063465272962994428892742962422524045D+00
      Omega( 3) = 0.0207757312971020101062205208061750560944D+00
      Omega( 4) = 0.0389902821279679968334054302597779795292D+00
      Omega( 5) = 0.0724629194976112335485748591756838266065D+00
      Omega( 6) = 0.1327041092315043892269077474566074670292D+00
      Omega( 7) = 0.2385928737227557325901398238565676024336D+00
      Omega( 8) = 0.4213494413144101220281868835915162208039D+00
      Omega( 9) = 0.7332827906426043373835747807287788191388D+00
      Omega(10) = 1.2677049097445011802262268552077273398027D+00
      Omega(11) = 2.2266091411905947496617519387740458114422D+00
      Omega(12) = 4.3330561116531063439206994392094429713325D+00
C
      Alpha( 1) = 0.0015632017689609898157798153692232823175D+00
      Alpha( 2) = 0.0087194292824458665820168347566909705648D+00
      Alpha( 3) = 0.0239611098766580661549173598967588105779D+00
      Alpha( 4) = 0.0529226964629682000570097804431624055610D+00
      Alpha( 5) = 0.1069822237653646031718332465310083989607D+00
      Alpha( 6) = 0.2066916845880041533352724017480461782270D+00
      Alpha( 7) = 0.3874734399418142787572573448873214374544D+00
      Alpha( 8) = 0.7092770597551052706611178499951364528897D+00
      Alpha( 9) = 1.2728176104506660636008635867355565096659D+00
      Alpha(10) = 2.2490192304692428984947555337114977191959D+00
      Alpha(11) = 3.9460569038339200339735035605315260909265D+00
      Alpha(12) = 7.0555049835035526799471850001310713196290D+00
C
      Xi( 1) = 1.008726364096767681775672353961D+00
      Xi( 2) = 1.079569865239148009376443648755D+00
      Xi( 3) = 1.226807850253259740951755629723D+00
      Xi( 4) = 1.461846526871927482967364331223D+00
      Xi( 5) = 1.802882368342192318986204335740D+00
      Xi( 6) = 2.276806611874737694542134014419D+00
      Xi( 7) = 2.922209999747150059590250492647D+00
      Xi( 8) = 3.793578015320768035290727171471D+00
      Xi( 9) = 4.967093522904663892118815038401D+00
      Xi(10) = 6.548938380413871829244243372159D+00
      Xi(11) = 8.687296182516121927606744801409D+00
      Xi(12) = 11.589738085971347570507505864157D+00
      Xi(13) = 15.548427601878168852547745792947D+00
      Xi(14) = 20.976776646222467454308557854858D+00
      Xi(15) = 28.462646901987146044743504091912D+00
      Xi(16) = 38.844884001942801776685598724725D+00
      Xi(17) = 53.320433416294026763432301052603D+00
      Xi(18) = 73.583309351268848384763909820094D+00
      Xi(19) = 101.967629986026167489276961219957D+00
      Xi(20) = 141.459971877096662859485398655579D+00
      Xi(21) = 195.111381434438942938935923621102D+00
      Xi(22) = 263.524373433416917222871234116610D+00
      Xi(23) = 338.072865592173264936981524897419D+00
      Xi(24) = 392.068013409484955378303538964246D+00
      GOTO 9999
C
C     ----- K = 12 , R = 5E2 -----
C
 1225 CONTINUE
C
      RIni = 5.0D+02
C
      Omega( 1) = 0.0033960422874595711797197125202812939548D+00
      Omega( 2) = 0.0089658613194300815344673044431089703465D+00
      Omega( 3) = 0.0179367737694282584733686822589526776994D+00
      Omega( 4) = 0.0345134149083668092621013639281768448086D+00
      Omega( 5) = 0.0655952830806395807914104152669398928310D+00
      Omega( 6) = 0.1224040530863569261612548016993073929370D+00
      Omega( 7) = 0.2236461872101807314401709231299619773381D+00
      Omega( 8) = 0.4005751196598171588627960992834431408482D+00
      Omega( 9) = 0.7058874382446316435367299202408020164512D+00
      Omega(10) = 1.2337273932034403278214240717858274365426D+00
      Omega(11) = 2.1870446405759955243574416128282678073447D+00
      Omega(12) = 4.2880666798091348198770922373057601362234D+00
C
      Alpha( 1) = 0.0013049277071097104255392994673692919072D+00
      Alpha( 2) = 0.0073201722780452361153633597870560301146D+00
      Alpha( 3) = 0.0203478395613008850024403017034413387876D+00
      Alpha( 4) = 0.0457004677472745743920673217736139193335D+00
      Alpha( 5) = 0.0941638306719849646012347917090057336509D+00
      Alpha( 6) = 0.1853856908569083935172999363927281990527D+00
      Alpha( 7) = 0.3536480269813660527473270223097756570496D+00
      Alpha( 8) = 0.6576800518814915979225188547019342877320D+00
      Alpha( 9) = 1.1971736850908161353288991501919724669278D+00
      Alpha(10) = 2.1426416636948536359203593137934262813360D+00
      Alpha(11) = 3.8028020862221158767194073568873591284500D+00
      Alpha(12) = 6.8701597226829623989326212285533301837859D+00
C
      Xi( 1) = 1.009074237506387751147143849728D+00
      Xi( 2) = 1.082788834124289475795144266357D+00
      Xi( 3) = 1.236243635746990285275044574309D+00
      Xi( 4) = 1.481834199212673094112405203227D+00
      Xi( 5) = 1.839397126819089084794923982802D+00
      Xi( 6) = 2.338397763170883258791732384196D+00
      Xi( 7) = 3.021387942382548134076861168218D+00
      Xi( 8) = 3.948843286226582439827140214028D+00
      Xi( 9) = 5.205937621410271708057709849626D+00
      Xi(10) = 6.912343228558149717862674199509D+00
      Xi(11) = 9.236561551875578097992036674668D+00
      Xi(12) = 12.416918625646503052638303099542D+00
      Xi(13) = 16.792417588408565404592409642248D+00
      Xi(14) = 22.848305341171556122684127565492D+00
      Xi(15) = 31.283472283025091970412590036688D+00
      Xi(16) = 43.109804876488710723841935035239D+00
      Xi(17) = 59.796093856544525677043644407149D+00
      Xi(18) = 83.466063442960214159571652459135D+00
      Xi(19) = 117.132188040261245025319691848154D+00
      Xi(20) = 164.827560596713928939283100305602D+00
      Xi(21) = 231.079740044725963465932494500521D+00
      Xi(22) = 317.951780134916631187502744637641D+00
      Xi(23) = 415.840851811601635301673951516932D+00
      Xi(24) = 489.064245362743104550418138387613D+00
      GOTO 9999
C
C     ----- K = 12 , R = 6E2 -----
C
 1226 CONTINUE
C
      RIni = 6.0D+02
C
      Omega( 1) = 0.0029330978418186225229015098728627131131D+00
      Omega( 2) = 0.0078221933123818168768042456175104693017D+00
      Omega( 3) = 0.0159389660701572009094708053805744540909D+00
      Omega( 4) = 0.0313142278421377730498000903358590818470D+00
      Omega( 5) = 0.0606021829072125219372614308294133778077D+00
      Omega( 6) = 0.1148010491641904454824738013307694473042D+00
      Omega( 7) = 0.2124732222970532045155239719846917978430D+00
      Omega( 8) = 0.3848807027475980752659421441075338066184D+00
      Omega( 9) = 0.6850094387100437870745557600304920242706D+00
      Omega(10) = 1.2076616461896024742102978555280401451455D+00
      Omega(11) = 2.1565782421799108423739882445246962561214D+00
      Omega(12) = 4.2534223060838955649334836017771976912627D+00
C
      Alpha( 1) = 0.0011258192253550014499241239919122281599D+00
      Alpha( 2) = 0.0063474759189304904574433830767576925069D+00
      Alpha( 3) = 0.0178245049430911581278964535626774257082D+00
      Alpha( 4) = 0.0406174118666135568191338876126161849811D+00
      Alpha( 5) = 0.0850442268691548172035248563671583355017D+00
      Alpha( 6) = 0.1700436027524185550685803872128509794948D+00
      Alpha( 7) = 0.3290007972164096816420231911282101577854D+00
      Alpha( 8) = 0.6196717314799680028356877736772645448582D+00
      Alpha( 9) = 1.1409124547374645714274341012917091120471D+00
      Alpha(10) = 2.0628733271752983560298061460613894269045D+00
      Alpha(11) = 3.6946810222503385231444111846954569955415D+00
      Alpha(12) = 6.7296153572336118308012953281860291099292D+00
C
      Xi( 1) = 1.009352858684480677242505108904D+00
      Xi( 2) = 1.085369791743565834438341177748D+00
      Xi( 3) = 1.243824575125177332463738411050D+00
      Xi( 4) = 1.497938282004267463191557463453D+00
      Xi( 5) = 1.868921519321079305125596348169D+00
      Xi( 6) = 2.388413777333400952454164900551D+00
      Xi( 7) = 3.102332381575931267341308950058D+00
      Xi( 8) = 4.076287495485813287040416375007D+00
      Xi( 9) = 5.403217826091134451654135473220D+00
      Xi(10) = 7.214554549252716551463460170623D+00
      Xi(11) = 9.696684339680174313270966113265D+00
      Xi(12) = 13.115277401711676166611020288855D+00
      Xi(13) = 17.851442912697049459486153821786D+00
      Xi(14) = 24.455742830752111850040053653288D+00
      Xi(15) = 33.729262324603299694897318161679D+00
      Xi(16) = 46.845267992600511424566844809192D+00
      Xi(17) = 65.529878294449921419251303689180D+00
      Xi(18) = 92.320485313405437011180509898622D+00
      Xi(19) = 130.896657375562962877246775406093D+00
      Xi(20) = 186.351426250791749744672998190254D+00
      Xi(21) = 264.783104229940797202935698351212D+00
      Xi(22) = 370.008871873476376246347996357144D+00
      Xi(23) = 491.967307881478889547421928796211D+00
      Xi(24) = 585.781742831649248459680023870533D+00
      GOTO 9999
C
C     ----- K = 12 , R = 7E2 -----
C
 1227 CONTINUE
C
      RIni = 7.0D+02
C
      Omega( 1) = 0.0025913964430986229023616985306377458187D+00
      Omega( 2) = 0.0069745687472657970477579281003144284767D+00
      Omega( 3) = 0.0144458782481699179102050626527509180619D+00
      Omega( 4) = 0.0288910389174395462479787762694050101686D+00
      Omega( 5) = 0.0567656733522657164232709265522247576996D+00
      Omega( 6) = 0.1088872602145011116267676876123715601352D+00
      Omega( 7) = 0.2036941460477392089637675015589834970342D+00
      Omega( 8) = 0.3724435069704642134089632443672002182211D+00
      Omega( 9) = 0.6683474147663148126770858714618128715301D+00
      Omega(10) = 1.1867479034725879530722328358827155625477D+00
      Omega(11) = 2.1320583937567418447971839823296136273711D+00
      Omega(12) = 4.2255387852958599535953265746712759209913D+00
C
      Alpha( 1) = 0.0009936834721502659399096799518712186394D+00
      Alpha( 2) = 0.0056283719522631914286091411783047311346D+00
      Alpha( 3) = 0.0159514188285338891451384073227881543744D+00
      Alpha( 4) = 0.0368179787624991746786736607210954730363D+00
      Alpha( 5) = 0.0781633783526307162708946993379122147871D+00
      Alpha( 6) = 0.1583492582236514436698666952185732270664D+00
      Alpha( 7) = 0.3100284584585823945850936722568036429948D+00
      Alpha( 8) = 0.5901523820112894313941534485401518850267D+00
      Alpha( 9) = 1.0968730441192009743452129288954211006057D+00
      Alpha(10) = 2.0000185378949157531582730973340744640154D+00
      Alpha(11) = 3.6090366250071584601055024732474407755944D+00
      Alpha(12) = 6.6178680810696371172342777899899601834477D+00
C
      Xi( 1) = 1.009584167126639444580712967614D+00
      Xi( 2) = 1.087514380980905951638165740647D+00
      Xi( 3) = 1.250134264708169984717245348804D+00
      Xi( 4) = 1.511372453379046270531896323241D+00
      Xi( 5) = 1.893623009760089895042907415235D+00
      Xi( 6) = 2.430406020502554801428901631866D+00
      Xi( 7) = 3.170572493340563957403208106278D+00
      Xi( 8) = 4.184228167584695687233192762378D+00
      Xi( 9) = 5.571163680861715695753072496998D+00
      Xi(10) = 7.473262964238113993751394525944D+00
      Xi(11) = 10.092930148516707195872899038847D+00
      Xi(12) = 13.720542848569237331414483183067D+00
      Xi(13) = 18.775586499438879755849440833515D+00
      Xi(14) = 25.868720497977782796922285868391D+00
      Xi(15) = 35.896015764496048767312474936375D+00
      Xi(16) = 50.182383787779369900550197058919D+00
      Xi(17) = 70.698658082982278487438332348347D+00
      Xi(18) = 100.381169430693210645189061835936D+00
      Xi(19) = 143.563697999821433773370848996365D+00
      Xi(20) = 206.403391174944324415463370314683D+00
      Xi(21) = 296.635299425856340693741231007152D+00
      Xi(22) = 420.066826894266429737179180392559D+00
      Xi(23) = 566.652856082373333757029598700683D+00
      Xi(24) = 682.245952844929716807342856554897D+00
      GOTO 9999
C
C     ----- K = 12 , R = 8E2 -----
C
 1228 CONTINUE
C
      RIni = 8.0D+02
C
      Omega( 1) = 0.0023278936489411329762620967223640811028D+00
      Omega( 2) = 0.0063185218754966287458375373900509863745D+00
      Omega( 3) = 0.0132814453404489143366560004351495383190D+00
      Omega( 4) = 0.0269786605441758097134123239174696706755D+00
      Omega( 5) = 0.0537007996946042293508595619894174788556D+00
      Omega( 6) = 0.1041144250073139989408552326388690545400D+00
      Omega( 7) = 0.1965488527774496979729857218388566764133D+00
      Omega( 8) = 0.3622488239008863353252110073299263603985D+00
      Omega( 9) = 0.6546090522727519354201884682620260491603D+00
      Omega(10) = 1.1694264755675532816000736224815170771762D+00
      Omega(11) = 2.1116976718956828160036337527216687703913D+00
      Omega(12) = 4.2023836966363481912334343348902621073648D+00
C
      Alpha( 1) = 0.0008918316487520093297028874758901328956D+00
      Alpha( 2) = 0.0050730424161482702214215540804931325880D+00
      Alpha( 3) = 0.0144995827015344837641555196180997988620D+00
      Alpha( 4) = 0.0338544679809992165817891188878130392936D+00
      Alpha( 5) = 0.0727516916962798764045698175773058125060D+00
      Alpha( 6) = 0.1490704377177037137896641744161385645384D+00
      Alpha( 7) = 0.2948485991588116589788570731656136558740D+00
      Alpha( 8) = 0.5663551610182622583116243686340851581917D+00
      Alpha( 9) = 1.0611357875910290857277701315197759868170D+00
      Alpha(10) = 1.9487292245922343926012657022717178278981D+00
      Alpha(11) = 3.5388436890305506690203307673669996802346D+00
      Alpha(12) = 6.5259940885676365634505968760237237802357D+00
C
      Xi( 1) = 1.009781189422376461309674433586D+00
      Xi( 2) = 1.089342453375589435697265439096D+00
      Xi( 3) = 1.255520216420012828435065488364D+00
      Xi( 4) = 1.522861997141996169818217721215D+00
      Xi( 5) = 1.914800539616287154351392807428D+00
      Xi( 6) = 2.466514573148087141173795400206D+00
      Xi( 7) = 3.229453337277839216311325909459D+00
      Xi( 8) = 4.277730525222177454178207334223D+00
      Xi( 9) = 5.717273132463409214109456568664D+00
      Xi(10) = 7.699387389298664483711087669349D+00
      Xi(11) = 10.441018635400417615723134279193D+00
      Xi(12) = 14.255128687173797280215992167030D+00
      Xi(13) = 19.596544917191017438906630587780D+00
      Xi(14) = 27.131721338938801776213827210427D+00
      Xi(15) = 37.845675962046086616974438499028D+00
      Xi(16) = 53.206649423236871805642511645829D+00
      Xi(17) = 75.419151614884124459892156266960D+00
      Xi(18) = 107.804895712109601112194123828658D+00
      Xi(19) = 155.338725466247233222838453059467D+00
      Xi(20) = 225.240339361376605026299202449991D+00
      Xi(21) = 326.928991424194883730347527261983D+00
      Xi(22) = 468.396188718868363570768664772004D+00
      Xi(23) = 640.048365516021274701508048110554D+00
      Xi(24) = 778.476211396508393225435185058814D+00
      GOTO 9999
C
C     ----- K = 12 , R = 9E2 -----
C
 1229 CONTINUE
C
      RIni = 9.0D+02
C
      Omega( 1) = 0.0021179393643737881895871855717286558196D+00
      Omega( 2) = 0.0057940548940096001734946810565740626942D+00
      Omega( 3) = 0.0123440249542606461606643352725554141358D+00
      Omega( 4) = 0.0254226054514351960271001992092143595414D+00
      Omega( 5) = 0.0511805111153068904816223951698139771338D+00
      Omega( 6) = 0.1001552558913137193962535238800359849165D+00
      Omega( 7) = 0.1905788931856397850245514111189315542561D+00
      Omega( 8) = 0.3536793060741086684685072216716150705906D+00
      Omega( 9) = 0.6430024712651347888314051770564105936501D+00
      Omega(10) = 1.1547365191625175191844285271791648028739D+00
      Omega(11) = 2.0943916499248837598517319902313715829223D+00
      Omega(12) = 4.1827013514862671538299676576855290477397D+00
C
      Alpha( 1) = 0.0008107099970118842197798566422736049608D+00
      Alpha( 2) = 0.0046299931135256944031852331580129122912D+00
      Alpha( 3) = 0.0133373609699492468702364039376262461190D+00
      Alpha( 4) = 0.0314684237600592341910720438347137672963D+00
      Alpha( 5) = 0.0683619730740917786696465158202684619937D+00
      Alpha( 6) = 0.1414853034908993946401666377821193520958D+00
      Alpha( 7) = 0.2823490730713291792155356096927221187798D+00
      Alpha( 8) = 0.5466319853569378457995749098774496133046D+00
      Alpha( 9) = 1.0313485189672434379258916803401291417686D+00
      Alpha(10) = 1.9057753817936694537083300304836086525029D+00
      Alpha(11) = 3.4798372032584188881867981457673977274681D+00
      Alpha(12) = 6.4485549634885074733042786832015735853929D+00
C
      Xi( 1) = 1.009952281380811104868379379784D+00
      Xi( 2) = 1.090930967917693870028114644377D+00
      Xi( 3) = 1.260205860792128242748796895789D+00
      Xi( 4) = 1.532874411119366155498849879635D+00
      Xi( 5) = 1.933293810743679655775317804967D+00
      Xi( 6) = 2.498127091446271916480928454440D+00
      Xi( 7) = 3.281156979832339873684454722991D+00
      Xi( 8) = 4.360110951557405069897027649617D+00
      Xi( 9) = 5.846481208596198264094978558347D+00
      Xi(10) = 7.900164152090885672646836113131D+00
      Xi(11) = 10.751433973032685802694030652304D+00
      Xi(12) = 14.734082922574942209857640218473D+00
      Xi(13) = 20.335756299552270846484058886006D+00
      Xi(14) = 28.275075403277745666325015960751D+00
      Xi(15) = 39.620843458520804584305219364637D+00
      Xi(16) = 55.977406661451159352455020368211D+00
      Xi(17) = 79.773154247089361597311540919009D+00
      Xi(18) = 114.702775670644112206153497623973D+00
      Xi(19) = 166.369080825248077573630389736081D+00
      Xi(20) = 243.049485971892347824230284913938D+00
      Xi(21) = 355.881817085389865445987922498716D+00
      Xi(22) = 515.203501803473302822489188201871D+00
      Xi(23) = 712.272365609997750501847235682362D+00
      Xi(24) = 874.487760295694255829879182329023D+00
      GOTO 9999
C
C     ----- K = 12 , R = 1E3 -----
C
 1231 CONTINUE
C
      RIni = 1.0D+03
C
      Omega( 1) = 0.0019463548977259609882261675835660175160D+00
      Omega( 2) = 0.0053641265236251452123256936582729181850D+00
      Omega( 3) = 0.0115705678700995050980931564896247998320D+00
      Omega( 4) = 0.0241262389665478289319807498822245683812D+00
      Omega( 5) = 0.0490612219221091198371250286652811567478D+00
      Omega( 6) = 0.0968006411727045761107563089098437814073D+00
      Omega( 7) = 0.1854888010464084995205548411778551098905D+00
      Omega( 8) = 0.3463341660098125813485245294565118001628D+00
      Omega( 9) = 0.6330104554036071431879877102755216355945D+00
      Omega(10) = 1.1420476000340242452198158407128403268871D+00
      Omega(11) = 2.0794138334222343020719209194524523809378D+00
      Omega(12) = 4.1656658270972997198076936964383776285104D+00
C
      Alpha( 1) = 0.0007444374958604834432202152752402340985D+00
      Alpha( 2) = 0.0042674831635267779901862078822255552879D+00
      Alpha( 3) = 0.0123834190491907699616523272877210981591D+00
      Alpha( 4) = 0.0294995182851700339077928103917458635408D+00
      Alpha( 5) = 0.0647151525530052908758284029022700423184D+00
      Alpha( 6) = 0.1351401304661587269689273713146882016645D+00
      Alpha( 7) = 0.2718256002117980587326571267903219109030D+00
      Alpha( 8) = 0.5299318074134546454104742474466860358007D+00
      Alpha( 9) = 1.0060016003098821085271277753214747008315D+00
      Alpha(10) = 1.8690725806928342152615774018897809582995D+00
      Alpha(11) = 3.4292526283354700571224388250257675281318D+00
      Alpha(12) = 6.3820141367267550843063450560777027931181D+00
C
      Xi( 1) = 1.010103112092257265333845872579D+00
      Xi( 2) = 1.092332114783997979996603422403D+00
      Xi( 3) = 1.264343204757084850888525229262D+00
      Xi( 4) = 1.541728095174602886172771087914D+00
      Xi( 5) = 1.949677080149472249855600869406D+00
      Xi( 6) = 2.526194557119787677647651524815D+00
      Xi( 7) = 3.327182911423273878282966253828D+00
      Xi( 8) = 4.433661683368288571136656139871D+00
      Xi( 9) = 5.962217011861906646418007849064D+00
      Xi(10) = 8.080640332344100968536837470424D+00
      Xi(11) = 11.031528435262277587289303504292D+00
      Xi(12) = 15.168030557469529144391795494329D+00
      Xi(13) = 21.008459485960092431866552686870D+00
      Xi(14) = 29.320491489101491479341010659709D+00
      Xi(15) = 41.252235313912665522206868473631D+00
      Xi(16) = 58.537788685829933197496410812732D+00
      Xi(17) = 83.820647941553667853442632917904D+00
      Xi(18) = 121.157060388289714747411807138633D+00
      Xi(19) = 176.765049793261642255282062308197D+00
      Xi(20) = 259.972767569124886100073723582682D+00
      Xi(21) = 383.661620605873604800128973124629D+00
      Xi(22) = 560.651672202403613232135626276431D+00
      Xi(23) = 783.420520368272808864329448397257D+00
      Xi(24) = 970.293091739903134307354548582225D+00
      GOTO 9999
C
C     ----- K = 12 , R = 2E3 -----
C
 1232 CONTINUE
C
      RIni = 2.0D+03
C
      Omega( 1) = 0.0011191579213804319974210639944758871600D+00
      Omega( 2) = 0.0032686761628313806416415020390797696592D+00
      Omega( 3) = 0.0077087399912689356829477058900140784559D+00
      Omega( 4) = 0.0174324625651019033129170924395490160919D+00
      Omega( 5) = 0.0377804662828506910933804539831637470115D+00
      Omega( 6) = 0.0785010102683294701204951351880545473705D+00
      Omega( 7) = 0.1571527226341211399098570650201267540069D+00
      Omega( 8) = 0.3047343816013479167481049769694578799317D+00
      Omega( 9) = 0.5755987681276236801785131658171223989484D+00
      Omega(10) = 1.0683278448896231092217837255198276125157D+00
      Omega(11) = 1.9918273202232110990345104184484625875484D+00
      Omega(12) = 4.0660174427838312121406827426284280591062D+00
C
      Alpha( 1) = 0.0004253461425896496173415485612641709356D+00
      Alpha( 2) = 0.0025124573159036588932423487378203796894D+00
      Alpha( 3) = 0.0077110331233492505094457441800620545180D+00
      Alpha( 4) = 0.0196667688917383451300792884461343668079D+00
      Alpha( 5) = 0.0460699664597410117796848650489893728377D+00
      Alpha( 6) = 0.1019345617495361915912218883639717148526D+00
      Alpha( 7) = 0.2155714528451965541220106126196753848490D+00
      Alpha( 8) = 0.4389685497524493766878950912202483891633D+00
      Alpha( 9) = 0.8656822115642424942121187225740897019932D+00
      Alpha(10) = 1.6631068381675097116959710641381775531045D+00
      Alpha(11) = 3.1423301713480807483697132553857045422774D+00
      Alpha(12) = 6.0017142252759861143808495231866118047037D+00
C
      Xi( 1) = 1.011039072006112649049214269326D+00
      Xi( 2) = 1.101043496799447574119312776020D+00
      Xi( 3) = 1.290158227396799084374379162909D+00
      Xi( 4) = 1.597242793944516847659542690963D+00
      Xi( 5) = 2.053046323150903518239829281811D+00
      Xi( 6) = 2.704635222047308662496334363823D+00
      Xi( 7) = 3.622407478466833641306266144966D+00
      Xi( 8) = 4.910219350075307076508579484653D+00
      Xi( 9) = 6.720507534150138795325801766012D+00
      Xi(10) = 9.277601521601525291771395131946D+00
      Xi(11) = 12.913884274682290609348456200678D+00
      Xi(12) = 18.126309733916155408905979129486D+00
      Xi(13) = 25.665795866861291329996674903668D+00
      Xi(14) = 36.680441203648435077866452047601D+00
      Xi(15) = 52.948441202086392820547189330682D+00
      Xi(16) = 77.262366058087110332053271122277D+00
      Xi(17) = 114.072385534359611369836784433573D+00
      Xi(18) = 170.573730887064442640621564351022D+00
      Xi(19) = 258.538057694210294812364736571908D+00
      Xi(20) = 397.270057505410022713476791977882D+00
      Xi(21) = 617.487588820508449316548649221659D+00
      Xi(22) = 961.300852052352183818584308028221D+00
      Xi(23) = 1448.751627700116159758181311190128D+00
      Xi(24) = 1918.726179869732959559769369661808D+00
      GOTO 9999
C
C     ----- K = 12 , R = 3E3 -----
C
 1233 CONTINUE
C
      RIni = 3.0D+03
C
      Omega( 1) = 0.0008121033978678005086831820597079861779D+00
      Omega( 2) = 0.0024758914491879611614147562023355297889D+00
      Omega( 3) = 0.0061840387301675452017574066525262832528D+00
      Omega( 4) = 0.0146476159436097251154813240342500435531D+00
      Omega( 5) = 0.0328799244389716317279591818040218598185D+00
      Omega( 6) = 0.0702772393108532951038229172335825012397D+00
      Omega( 7) = 0.1440596376824298634336052674087547131876D+00
      Omega( 8) = 0.2850571929983484794485226743177008756902D+00
      Omega( 9) = 0.5479078906024392761034587373547566357956D+00
      Omega(10) = 1.0322349251234491782818272764821188047790D+00
      Omega(11) = 1.9485640056433970451884793440022747290641D+00
      Omega(12) = 4.0167705581257705758913001403698217472993D+00
C
      Alpha( 1) = 0.0003071487155038260791730359526704660644D+00
      Alpha( 2) = 0.0018562367167545763176936772639522010842D+00
      Alpha( 3) = 0.0059269627132180731568844903565940196444D+00
      Alpha( 4) = 0.0157853122542906193461669048933027070802D+00
      Alpha( 5) = 0.0384313670603751672376346970594340746175D+00
      Alpha( 6) = 0.0878495253001757317210342924929111774190D+00
      Alpha( 7) = 0.1909682470184396789957773782342442814297D+00
      Alpha( 8) = 0.3981208018572957993072569671255678258603D+00
      Alpha( 9) = 0.8012392220997077544892422951150479093485D+00
      Alpha(10) = 1.5667393798490310214202581029496741393814D+00
      Alpha(11) = 3.0061217791432274326710732959000438313524D+00
      Alpha(12) = 5.8193331245322036022506184682612229153165D+00
C
      Xi( 1) = 1.011537667270811324251797436169D+00
      Xi( 2) = 1.105696052958149435524980164214D+00
      Xi( 3) = 1.304010485718364174031851099311D+00
      Xi( 4) = 1.627225596146438388257988993946D+00
      Xi( 5) = 2.109334385478168750116187424304D+00
      Xi( 6) = 2.802776896249266310725753448274D+00
      Xi( 7) = 3.786687504015108891473537222616D+00
      Xi( 8) = 5.178921891244562241123777779350D+00
      Xi( 9) = 7.154325833633077038210129616758D+00
      Xi(10) = 9.973323332104322347033131812566D+00
      Xi(11) = 14.026931720695269942451866018729D+00
      Xi(12) = 19.908322302108640984921739125468D+00
      Xi(13) = 28.528155044368060091683481083180D+00
      Xi(14) = 41.303228107103747646089431100336D+00
      Xi(15) = 60.470502289662372164685821829266D+00
      Xi(16) = 89.619228828159433752897911062973D+00
      Xi(17) = 134.609726544418902852151553872773D+00
      Xi(18) = 205.189443803730656584738412107072D+00
      Xi(19) = 317.862303811160296501681088443547D+00
      Xi(20) = 500.942847350866574590577684489290D+00
      Xi(21) = 802.607963614237230498549280355292D+00
      Xi(22) = 1297.890370986454122492759211127122D+00
      Xi(23) = 2052.941393499940959888050429071882D+00
      Xi(24) = 2852.754713529029453988883346937655D+00
      GOTO 9999
C
C     ----- K = 12 , R = 4E3 -----
C
 1234 CONTINUE
C
      RIni = 4.0D+03
C
      Omega( 1) = 0.0006481129780097903614989966194440823566D+00
      Omega( 2) = 0.0020466770735909012241228119039632149523D+00
      Omega( 3) = 0.0053330587438768426823312146772132491535D+00
      Omega( 4) = 0.0130421841198512183248274650879117153579D+00
      Omega( 5) = 0.0299838018313827793310448553004698979407D+00
      Omega( 6) = 0.0653224926526793809930114716177929068408D+00
      Omega( 7) = 0.1360452092785600851604192740307297526670D+00
      Omega( 8) = 0.2728507539383111943560086126714381293823D+00
      Omega( 9) = 0.5305383007341930598575047151044259408081D+00
      Omega(10) = 1.0094001089269477237746197584300489324960D+00
      Omega(11) = 1.9210522466051994238019073879542020222289D+00
      Omega(12) = 3.9854419726506038249534691342290670945658D+00
C
      Alpha( 1) = 0.0002441084588471243577005574969165241361D+00
      Alpha( 2) = 0.0015039659540854241270781306571979740738D+00
      Alpha( 3) = 0.0049543174187313296201879949179347484289D+00
      Alpha( 4) = 0.0136203241766619226683682122164276151466D+00
      Alpha( 5) = 0.0340694986532491067115742747195561435092D+00
      Alpha( 6) = 0.0796364298725844749237464531566033087984D+00
      Alpha( 7) = 0.1763621932285932128648851466290459200081D+00
      Alpha( 8) = 0.3734982400670228677372342040996500145411D+00
      Alpha( 9) = 0.7618906435501059938000985305439627381929D+00
      Alpha(10) = 1.5072708912101296354008397870849478294986D+00
      Alpha(11) = 2.9213720831136537564674121592034339300881D+00
      Alpha(12) = 5.7052025799370194676304879344286291598110D+00
C
      Xi( 1) = 1.011868124782982881842106903658D+00
      Xi( 2) = 1.108784281261581288781370435714D+00
      Xi( 3) = 1.313230305611500008285130736141D+00
      Xi( 4) = 1.647256661318397008477582010855D+00
      Xi( 5) = 2.147119232044813253107787276441D+00
      Xi( 6) = 2.869040857831678182406770494239D+00
      Xi( 7) = 3.898364032497771645589465516757D+00
      Xi( 8) = 5.362992949926492766265367917455D+00
      Xi( 9) = 7.454045205298210262367941680850D+00
      Xi(10) = 10.458479260954887335444141172047D+00
      Xi(11) = 14.811003284264805978862056257128D+00
      Xi(12) = 21.177532366231453608493295881487D+00
      Xi(13) = 30.591410896226637256001268383443D+00
      Xi(14) = 44.679355356495013291279194689309D+00
      Xi(15) = 66.043498670043934588302914079350D+00
      Xi(16) = 98.920467704516685955584698319853D+00
      Xi(17) = 150.342748961565701254494697991504D+00
      Xi(18) = 232.234614185135523775627142128997D+00
      Xi(19) = 365.256778760870198602406233590045D+00
      Xi(20) = 585.925054037377420645960057754564D+00
      Xi(21) = 959.087740800150882758412507200774D+00
      Xi(22) = 1593.727443434642945141455072644021D+00
      Xi(23) = 2613.271812605958078101764385792194D+00
      Xi(24) = 3774.791876342242130659698773342825D+00
      GOTO 9999
C
C     ----- K = 12 , R = 5E3 -----
C
 1235 CONTINUE
C
      RIni = 5.0D+03
C
      Omega( 1) = 0.0005448622032481286372600947547420457973D+00
      Omega( 2) = 0.0017734323890971321100885741729735745054D+00
      Omega( 3) = 0.0047781227439880182135425426732944220731D+00
      Omega( 4) = 0.0119707005213017274373966600253815784072D+00
      Omega( 5) = 0.0280174152450522154481120590610587761660D+00
      Omega( 6) = 0.0619132750049512343984360780713149807752D+00
      Omega( 7) = 0.1304701295911892246838807360442835658887D+00
      Omega( 8) = 0.2642812084313892949610867028065186445929D+00
      Omega( 9) = 0.5182503659100501419080934961680640071791D+00
      Omega(10) = 0.9931501845600177424120726110690071664067D+00
      Omega(11) = 1.9014045488724224262011217034817889270926D+00
      Omega(12) = 3.9630619605274942168163937461500268000236D+00
C
      Alpha( 1) = 0.0002044570226688318376350054358871373428D+00
      Alpha( 2) = 0.0012812569582845620733450595963610896266D+00
      Alpha( 3) = 0.0043315761250334632171618022816045323609D+00
      Alpha( 4) = 0.0122095813310535008168426361792324730260D+00
      Alpha( 5) = 0.0311780207666068839313360666620766892265D+00
      Alpha( 6) = 0.0741104015575603983884914877089400420118D+00
      Alpha( 7) = 0.1664105913111946309087619058630913571051D+00
      Alpha( 8) = 0.3565435846098175170560719282608630464892D+00
      Alpha( 9) = 0.7345542696373138491765936164412664766132D+00
      Alpha(10) = 1.4656545188883580361268593739154653121659D+00
      Alpha(11) = 2.8617277624721939989580615781861183677393D+00
      Alpha(12) = 5.6245660855886989468004066861084311312879D+00
C
      Xi( 1) = 1.012110763042167530293378996387D+00
      Xi( 2) = 1.111054165000070571411167064113D+00
      Xi( 3) = 1.320019842610480564752163668008D+00
      Xi( 4) = 1.662046220801892847302762279416D+00
      Xi( 5) = 2.175108770476030362875027868164D+00
      Xi( 6) = 2.918324544001868842309221907527D+00
      Xi( 7) = 3.981816971017502796662684838580D+00
      Xi( 8) = 5.501281333895242910136946790089D+00
      Xi( 9) = 7.680552305253643530832258212726D+00
      Xi(10) = 10.827509524563446416910161840264D+00
      Xi(11) = 15.411638503587643841789833754063D+00
      Xi(12) = 22.157344588140915005863851305890D+00
      Xi(13) = 32.197716288236957044716080034163D+00
      Xi(14) = 47.332251889737306068534072878151D+00
      Xi(15) = 70.467652321923360765609789524433D+00
      Xi(16) = 106.388566350058342439843794835497D+00
      Xi(17) = 163.136249882802584793006772656554D+00
      Xi(18) = 254.543858205361176835768155513051D+00
      Xi(19) = 404.995912311389834215002991868460D+00
      Xi(20) = 658.547479369785563552142093612929D+00
      Xi(21) = 1095.902157239230646057492890577123D+00
      Xi(22) = 1860.061478154625132841459844712517D+00
      Xi(23) = 3139.042099606381820198208743022406D+00
      Xi(24) = 4686.177830762634208827677184672211D+00
      GOTO 9999
C
C     ----- K = 12 , R = 6E3 -----
C
 1236 CONTINUE
C
      RIni = 6.0D+03
C
      Omega( 1) = 0.0004733547189603010971959035112829189407D+00
      Omega( 2) = 0.0015823881371287818824851266865175181131D+00
      Omega( 3) = 0.0043823111403624851655164498809291728776D+00
      Omega( 4) = 0.0111926736918029304569891370195466873128D+00
      Omega( 5) = 0.0265709338401388661733454506219676005685D+00
      Omega( 6) = 0.0593801198165422312119589211970582809386D+00
      Omega( 7) = 0.1262934109001463779865413392111683776875D+00
      Omega( 8) = 0.2578165632814459010948779371652506142709D+00
      Omega( 9) = 0.5089272241359717529425241155927039926610D+00
      Omega(10) = 0.9807660789125268947286941101371127160746D+00
      Omega(11) = 1.8863908554847012133794462163116634201288D+00
      Omega(12) = 3.9459563742834082044654009369111236082972D+00
C
      Alpha( 1) = 0.0001770171299286721908510122436850378613D+00
      Alpha( 2) = 0.0011264749626048635599989632106142367629D+00
      Alpha( 3) = 0.0038940614761975710174005089287893599526D+00
      Alpha( 4) = 0.0112041479869499801548030884675022367247D+00
      Alpha( 5) = 0.0290893432383205473994880084151470889253D+00
      Alpha( 6) = 0.0700727250077648080423507737091171065913D+00
      Alpha( 7) = 0.1590694792498797117502552442291907652816D+00
      Alpha( 8) = 0.3439359505070394979462690410176861632863D+00
      Alpha( 9) = 0.7140902598402039924132753134777829018276D+00
      Alpha(10) = 1.4343293737380057116916412529583624291263D+00
      Alpha(11) = 2.8166420926084831063453128852103191093192D+00
      Alpha(12) = 5.5634336743866897334077903369831119562150D+00
C
      Xi( 1) = 1.012300011019211134870310342393D+00
      Xi( 2) = 1.112825973018472957513230747750D+00
      Xi( 3) = 1.325327181573794575004596185686D+00
      Xi( 4) = 1.673629716108549977298183830676D+00
      Xi( 5) = 2.197085318013530684825782013370D+00
      Xi( 6) = 2.957139278999038539425675553574D+00
      Xi( 7) = 4.047776890306395496457736449258D+00
      Xi( 8) = 5.611023815110407759413485573319D+00
      Xi( 9) = 7.861109698646963799169296027713D+00
      Xi(10) = 11.123127573663211476763523766209D+00
      Xi(11) = 15.895376757966858063779447274300D+00
      Xi(12) = 22.951120792761833038983509958086D+00
      Xi(13) = 33.507452204478437359641551651634D+00
      Xi(14) = 49.510773333937274864879896085768D+00
      Xi(15) = 74.129402489529204321738742322623D+00
      Xi(16) = 112.624182707111735367300209276209D+00
      Xi(17) = 173.924249695524910375166172116224D+00
      Xi(18) = 273.567504803601884016961420797998D+00
      Xi(19) = 439.321187585954119836362252726758D+00
      Xi(20) = 722.226988948892116848554678654182D+00
      Xi(21) = 1218.075212719420962192984347893798D+00
      Xi(22) = 2103.529011125181807262407573944074D+00
      Xi(23) = 3636.192316312189927618447882196051D+00
      Xi(24) = 5587.788891480815288037575783164357D+00
      GOTO 9999
C
C     ----- K = 12 , R = 7E3 -----
C
 1237 CONTINUE
C
      RIni = 7.0D+03
C
      Omega( 1) = 0.0004206406183074642751176014425712815226D+00
      Omega( 2) = 0.0014403628889290248813555614772544122282D+00
      Omega( 3) = 0.0040830026632685866912733294292053853880D+00
      Omega( 4) = 0.0105957928830076622490262384587711430584D+00
      Omega( 5) = 0.0254497062877291561974345181912360303045D+00
      Omega( 6) = 0.0574007983725829418998815641822508482051D+00
      Omega( 7) = 0.1230084215018497156453043499257926640666D+00
      Omega( 8) = 0.2527041493193299904671238892417406418645D+00
      Omega( 9) = 0.5015205762704294671140183015101143837455D+00
      Omega(10) = 0.9708929815731902137373836303435581385202D+00
      Omega(11) = 1.8743958599473489759707281465672679132695D+00
      Omega(12) = 3.9322874092045371702611178399422442453215D+00
C
      Alpha( 1) = 0.0001568012924109277522218339664798227029D+00
      Alpha( 2) = 0.0010120177099265387919519793491138215380D+00
      Alpha( 3) = 0.0035674354694836556680850582446914986967D+00
      Alpha( 4) = 0.0104444295668740813643152079252807151732D+00
      Alpha( 5) = 0.0274936281542208234292105645951709291808D+00
      Alpha( 6) = 0.0669594194082923951550517273345608515456D+00
      Alpha( 7) = 0.1533655507263689535599584037872666897329D+00
      Alpha( 8) = 0.3340773637667244942630207643619044688421D+00
      Alpha( 9) = 0.6980031233317447962984439446820772445790D+00
      Alpha(10) = 1.4095969802818261701988708511912307130842D+00
      Alpha(11) = 2.7809257068145880041100392876529667773866D+00
      Alpha(12) = 5.5148932957723285465737039778844064130681D+00
C
      Xi( 1) = 1.012453649742454550326115070291D+00
      Xi( 2) = 1.114265297694975322129185002495D+00
      Xi( 3) = 1.329643492732267751835900648771D+00
      Xi( 4) = 1.683064785667951284288228197550D+00
      Xi( 5) = 2.215021562050113414785515453254D+00
      Xi( 6) = 2.988894290072117407557755752734D+00
      Xi( 7) = 4.101893533078582910252757320890D+00
      Xi( 8) = 5.701352487159417659328547456354D+00
      Xi( 9) = 8.010255908784789502692647022997D+00
      Xi(10) = 11.368275428512243724159247459937D+00
      Xi(11) = 16.298258484248156412882435617462D+00
      Xi(12) = 23.615328103170042141345508213135D+00
      Xi(13) = 34.609093419200483088965514255619D+00
      Xi(14) = 51.353658086930686718640792776114D+00
      Xi(15) = 77.246707733481081742388241195840D+00
      Xi(16) = 117.970525286899102543336503856608D+00
      Xi(17) = 183.248207119697166425642542719743D+00
      Xi(18) = 290.160519779868272233214199218310D+00
      Xi(19) = 469.578766911519154769738548793612D+00
      Xi(20) = 779.062326962213241576815647704279D+00
      Xi(21) = 1328.780547703719858487758642695553D+00
      Xi(22) = 2328.495107797294902951534822932445D+00
      Xi(23) = 4108.885896644536002586534095826210D+00
      Xi(24) = 6480.251750339536623624070443838718D+00
      GOTO 9999
C
C     ----- K = 12 , R = 8E3 -----
C
 1238 CONTINUE
C
      RIni = 8.0D+03
C
      Omega( 1) = 0.0003800261919516156242255321248864341488D+00
      Omega( 2) = 0.0013301003573762626409333374786873871454D+00
      Omega( 3) = 0.0038471561499707714797889309202294505496D+00
      Omega( 4) = 0.0101197893626686987452541113124970806325D+00
      Omega( 5) = 0.0245478874191187860814600566958243454962D+00
      Omega( 6) = 0.0557982667134647215201250381594322291789D+00
      Omega( 7) = 0.1203343865818571066086191574673325277445D+00
      Omega( 8) = 0.2485237367430795088545049040251733174500D+00
      Omega( 9) = 0.4954414608392572636700722926716977667638D+00
      Omega(10) = 0.9627660079132293382284460414588522780832D+00
      Omega(11) = 1.8645049667248534153727104745179588007886D+00
      Omega(12) = 3.9210143330472333653667627029193454291089D+00
C
      Alpha( 1) = 0.0001412333942297918437753135947085141033D+00
      Alpha( 2) = 0.0009235844571854584796527123490960525798D+00
      Alpha( 3) = 0.0033129136408568407550036179773714217767D+00
      Alpha( 4) = 0.0098462274094449256249853789230153111589D+00
      Alpha( 5) = 0.0262254439890183024227846659767737769187D+00
      Alpha( 6) = 0.0644660543575470885120713530891389808630D+00
      Alpha( 7) = 0.1487684047413587857232059735723694871012D+00
      Alpha( 8) = 0.3260898308896926300697559353292120931656D+00
      Alpha( 9) = 0.6849120562819440391250265520728390811200D+00
      Alpha(10) = 1.3893988560757597640468383182188460978068D+00
      Alpha(11) = 2.7516769901723567494800787303077527212736D+00
      Alpha(12) = 5.4750676123472046341947305769792819774011D+00
C
      Xi( 1) = 1.012581999202422624196257006091D+00
      Xi( 2) = 1.115468316488554929139485893952D+00
      Xi( 3) = 1.333254441146869702481021768836D+00
      Xi( 4) = 1.690968210962891987714229458106D+00
      Xi( 5) = 2.230070707413103697032072392226D+00
      Xi( 6) = 3.015590954585679176564935732863D+00
      Xi( 7) = 4.147496842777866264424108111442D+00
      Xi( 8) = 5.777671787565585226429526466418D+00
      Xi( 9) = 8.136642872935691993795115362076D+00
      Xi(10) = 11.576689096721222486247038219176D+00
      Xi(11) = 16.641981465350747019377175561772D+00
      Xi(12) = 24.184223921319856268757897410637D+00
      Xi(13) = 35.556695069646932102352954174762D+00
      Xi(14) = 52.946410986665998519434817026763D+00
      Xi(15) = 79.955208154598413738955997587254D+00
      Xi(16) = 122.643381238686981979379986285039D+00
      Xi(17) = 191.452633443719034231356168618277D+00
      Xi(18) = 304.874091538582232818743733560041D+00
      Xi(19) = 496.650186156782545099375170138956D+00
      Xi(20) = 830.452702523549992408291586798441D+00
      Xi(21) = 1430.183700202695451841883311772108D+00
      Xi(22) = 2538.038937035036850309310807460861D+00
      Xi(23) = 4560.218079011349314288992218280328D+00
      Xi(24) = 7364.042186608088992993970123279723D+00
      GOTO 9999
C
C     ----- K = 12 , R = 9E3 -----
C
 1239 CONTINUE
C
      RIni = 9.0D+03
C
      Omega( 1) = 0.0003476872036346924650056719599067536763D+00
      Omega( 2) = 0.0012416884882636269877667056760715824648D+00
      Omega( 3) = 0.0036555466665538696706320260457895310324D+00
      Omega( 4) = 0.0097290950938871556673918894984909311319D+00
      Omega( 5) = 0.0238023296274520649006825254168084882167D+00
      Omega( 6) = 0.0544659975598365621249363505845808575145D+00
      Omega( 7) = 0.1181011691502000846592970555171131508132D+00
      Omega( 8) = 0.2450191564644082776303459295097475489911D+00
      Omega( 9) = 0.4903290170640579521727127343400098880011D+00
      Omega(10) = 0.9559146236107875842192768689464799081179D+00
      Omega(11) = 1.8561541768321525229350210661749542850885D+00
      Omega(12) = 3.9114952214587968457818606404785555241688D+00
C
      Alpha( 1) = 0.0001288425823778130519868665891051323635D+00
      Alpha( 2) = 0.0008529882715494492601216538791176341450D+00
      Alpha( 3) = 0.0031081525804036526562012583045982450081D+00
      Alpha( 4) = 0.0093605603121938307665186201379781127940D+00
      Alpha( 5) = 0.0251875715678334414904921377542645899439D+00
      Alpha( 6) = 0.0624120873998579979934596406732083551105D+00
      Alpha( 7) = 0.1449610016318818002872348751908226915930D+00
      Alpha( 8) = 0.3194449964601693630022842540494565355402D+00
      Alpha( 9) = 0.6739813444718496708379061599369919122182D+00
      Alpha(10) = 1.3724831527625502694328271657830953245139D+00
      Alpha(11) = 2.7271246942905251098472618753731921970029D+00
      Alpha(12) = 5.4415834861582763355009051409183484793175D+00
C
      Xi( 1) = 1.012691548896615802415749341137D+00
      Xi( 2) = 1.116495545573975540099703751551D+00
      Xi( 3) = 1.336340203906648158049298713568D+00
      Xi( 4) = 1.697729473523824816959086614343D+00
      Xi( 5) = 2.242962729719132134592607386026D+00
      Xi( 6) = 3.038499696573863620820479813922D+00
      Xi( 7) = 4.186707020552541595823786702013D+00
      Xi( 8) = 5.843438921887700056127801229167D+00
      Xi( 9) = 8.245824891351123597446615143269D+00
      Xi(10) = 11.757222759267835202404151040412D+00
      Xi(11) = 16.940623929352614435162038830640D+00
      Xi(12) = 24.680139770350485697164577558738D+00
      Xi(13) = 36.385760954748655245094557386665D+00
      Xi(14) = 54.345565135131664386697325497266D+00
      Xi(15) = 82.345246801335715218506106083396D+00
      Xi(16) = 126.787846301940243534667196456667D+00
      Xi(17) = 198.771391179855170680546017081269D+00
      Xi(18) = 318.086545475709757263560462092755D+00
      Xi(19) = 521.148445853394496951604963896898D+00
      Xi(20) = 877.386738279595312817171759434132D+00
      Xi(21) = 1523.843537108099895016621871945972D+00
      Xi(22) = 2734.439606912585181364150344052177D+00
      Xi(23) = 4992.591587474125793377766058256384D+00
      Xi(24) = 8239.537097791827611636961137264734D+00
      GOTO 9999
C
C     ----- K = 12 , R = 1E4 -----
C
 1241 CONTINUE
C
      RIni = 1.0D+04
C
      Omega( 1) = 0.0003212728404662958500820070891672644575D+00
      Omega( 2) = 0.0011690041207126430353211517156747367707D+00
      Omega( 3) = 0.0034961596915281700555119354057939662361D+00
      Omega( 4) = 0.0094012115265047685735779500575126643724D+00
      Omega( 5) = 0.0231727232916882891201838454153794089052D+00
      Omega( 6) = 0.0533354965287454047289837037043591649876D+00
      Omega( 7) = 0.1161987098546097791707323865217915681569D+00
      Omega( 8) = 0.2420238401722279862924658636047503534883D+00
      Omega( 9) = 0.4859476215181996613078074798153238589293D+00
      Omega(10) = 0.9500306179525972926844246124744586268207D+00
      Omega(11) = 1.8489733269096180122382719335227818646672D+00
      Omega(12) = 3.9033086697822239601474852976892293554556D+00
C
      Alpha( 1) = 0.0001187252048309153856080831843274048243D+00
      Alpha( 2) = 0.0007951872127706306181007148516960703066D+00
      Alpha( 3) = 0.0029393125552287440714112630489943112089D+00
      Alpha( 4) = 0.0089568258624270747326619169191574965083D+00
      Alpha( 5) = 0.0243187365497662530606538811017713896945D+00
      Alpha( 6) = 0.0606828397924930648457404573636875255715D+00
      Alpha( 7) = 0.1417406039031726232155705344495366659885D+00
      Alpha( 8) = 0.3138030384443519017788944841518272710346D+00
      Alpha( 9) = 0.6646708541365320590111991083670517355131D+00
      Alpha(10) = 1.3580375489597360417906926621789409637131D+00
      Alpha(11) = 2.7061158915034748547682325048313600746042D+00
      Alpha(12) = 5.4128928432449124911671201054730317991925D+00
C
      Xi( 1) = 1.012786628273634303011412061135D+00
      Xi( 2) = 1.117387423175520109368610843781D+00
      Xi( 3) = 1.339021202886511108134763581301D+00
      Xi( 4) = 1.703609387488905072861877787460D+00
      Xi( 5) = 2.254187193152664353025527166352D+00
      Xi( 6) = 3.058474594772011529754396352843D+00
      Xi( 7) = 4.220953208805569629568504863570D+00
      Xi( 8) = 5.900991421794251700228689516692D+00
      Xi( 9) = 8.341574313890958056530444686416D+00
      Xi(10) = 11.915920290298694703308435283162D+00
      Xi(11) = 17.203828725510412148841377089070D+00
      Xi(12) = 25.118455231060140097135158221420D+00
      Xi(13) = 37.120842368979526635502930176358D+00
      Xi(14) = 55.590473138931917336524879402759D+00
      Xi(15) = 84.480151568398610562038442139965D+00
      Xi(16) = 130.506260232299767520824929079026D+00
      Xi(17) = 205.370941600141838523363446711301D+00
      Xi(18) = 330.069941392460730222868647842915D+00
      Xi(19) = 543.519262139611156137153358258729D+00
      Xi(20) = 920.592835018539883462640460720650D+00
      Xi(21) = 1610.929814205873337407126655307366D+00
      Xi(22) = 2919.449441247667238696905656070157D+00
      Xi(23) = 5407.926930663512710584228670995799D+00
      Xi(24) = 9107.045750956664901920589727524202D+00
      GOTO 9999
C
C     ----- K = 12 , R = 3E5 (Best : R = 204500) -----
C
 1299 CONTINUE
C
      RIni = 3.0D+05
C
      Omega( 1) = 0.0000605283299992814375306170671597275762D+00
      Omega( 2) = 0.0003976819351504732883886811907123148835D+00
      Omega( 3) = 0.0016279651361197932355453018996374603411D+00
      Omega( 4) = 0.0052922029903677376701148283612285982969D+00
      Omega( 5) = 0.0148989531146119075073506001806455945058D+00
      Omega( 6) = 0.0379189506065104615268343633133207859487D+00
      Omega( 7) = 0.0894577452373522931324033649613802765543D+00
      Omega( 8) = 0.1988442747750825461078641454015780709597D+00
      Omega( 9) = 0.4214479650252676439002932906596754492057D+00
      Omega(10) = 0.8619857428372385712666940704718143706486D+00
      Omega(11) = 1.7404432216810900461772745884125868087722D+00
      Omega(12) = 3.7794416490913082928154720363522756088059D+00
C
      Alpha( 1) = 0.0000189340122962563570449109308913818346D+00
      Alpha( 2) = 0.0002082931461534274049703011728266327518D+00
      Alpha( 3) = 0.0011024023520154727638740112445094043725D+00
      Alpha( 4) = 0.0042548734477736867134704129306960185275D+00
      Alpha( 5) = 0.0136257458735436130452432714095512000796D+00
      Alpha( 6) = 0.0384427107656654528163194233653099374237D+00
      Alpha( 7) = 0.0988182876456668784028759822124499123674D+00
      Alpha( 8) = 0.2363640171964210898842249919393765367204D+00
      Alpha( 9) = 0.5337345457964074140876781227049718836497D+00
      Alpha(10) = 1.1508219943995056622990796890704245925008D+00
      Alpha(11) = 2.4001213099764144435262908094941280978674D+00
      Alpha(12) = 4.9906311509941725908581777737538232031511D+00
C
      Xi( 1) = 1.014328418137978840909974032058D+00
      Xi( 2) = 1.131893784203605395478724193747D+00
      Xi( 3) = 1.382866538366302002361490908200D+00
      Xi( 4) = 1.800491275910822536274885630192D+00
      Xi( 5) = 2.440909708391232602314041555402D+00
      Xi( 6) = 3.394698867579993121351700224864D+00
      Xi( 7) = 4.805462676224806516100951370163D+00
      Xi( 8) = 6.898933578374459715028960138383D+00
      Xi( 9) = 10.031489549901425317561531880273D+00
      Xi(10) = 14.772521964267412364173182037774D+00
      Xi(11) = 22.046339415046315845375657360705D+00
      Xi(12) = 33.381979261537414117172462013627D+00
      Xi(13) = 51.363658254921380274388154063558D+00
      Xi(14) = 80.469189010275652261527934427932D+00
      Xi(15) = 128.677497542476539496569820641980D+00
      Xi(16) = 210.688423455113419333062374505516D+00
      Xi(17) = 354.643244517801801363177105486102D+00
      Xi(18) = 616.979417705080263045758215412206D+00
      Xi(19) = 1117.665086180729539400147132255370D+00
      Xi(20) = 2130.087451853435267956626830709865D+00
      Xi(21) = 4343.465540693967070318137757567456D+00
      Xi(22) = 9735.832337984589317514405593101401D+00
      Xi(23) = 25275.481162932050445135701011167839D+00
      Xi(24) = 88852.490738383218115359341027215123D+00
      GOTO 9999
C
C     ===== K = 13 =====
C
 1300 CONTINUE
C
      GOTO (1311,1311,1311,1312,1313,1314,1315,1316,1317,1318,1319,1321,
     *      1322,1323,1324,1325,1326,1327,1328,1329,1331,1332,1333,1334,
     *      1335,1336,1337,1338,1339,1341,1399),InitR
C
C     ----- K = 13 , R = 1E1 -----
C
 1311 CONTINUE
C
      RIni = 1.0D+01
C
      Omega( 1) = 0.0657951119706244487439494861291677807458D+00
      Omega( 2) = 0.1558144970704423388032679176351535943468D+00
      Omega( 3) = 0.2526931220910247708876655847731740323070D+00
      Omega( 4) = 0.3612508995957037221270075066925642204296D+00
      Omega( 5) = 0.4880601557364697465053824526615144918651D+00
      Omega( 6) = 0.6422740638610490109374276390674651793233D+00
      Omega( 7) = 0.8368561747820077732274747506213685710463D+00
      Omega( 8) = 1.0903726151683377960672355344584616432257D+00
      Omega( 9) = 1.4302814690796487430771155580089271097677D+00
      Omega(10) = 1.9007339128043295431360476532844927533006D+00
      Omega(11) = 2.5848157519580364366307279055945400614291D+00
      Omega(12) = 3.6846225483494229582717011117765082417463D+00
      Omega(13) = 5.9974171438876001808196414266660667635733D+00
C
      Alpha( 1) = 0.0255868207120509142801567182071154071821D+00
      Alpha( 2) = 0.1359771426930953219612682389416669082038D+00
      Alpha( 3) = 0.3394815030426081711657955441996392664805D+00
      Alpha( 4) = 0.6452380388667339172997609864079748831500D+00
      Alpha( 5) = 1.0680388682879816996448146815268387399556D+00
      Alpha( 6) = 1.6304515219187842809409408340037828111235D+00
      Alpha( 7) = 2.3659824436194987933631400656153687123151D+00
      Alpha( 8) = 3.3237148050771310988163453892241250287043D+00
      Alpha( 9) = 4.5753547547140072191546034119369323889259D+00
      Alpha(10) = 6.2274123577515104557313085908987204675213D+00
      Alpha(11) = 8.4469671933229768618842103755639527662424D+00
      Alpha(12) = 11.5314290237107121188733782624069590383442D+00
      Alpha(13) = 16.1870932176644028131773378476054858765565D+00
C
      Xi( 1) = 1.002367931474044091184373572734D+00
      Xi( 2) = 1.021378676057969264582593604818D+00
      Xi( 3) = 1.059761545282307757800607794785D+00
      Xi( 4) = 1.118243092103653798814050235322D+00
      Xi( 5) = 1.197920048122829643055115012018D+00
      Xi( 6) = 1.300278657513724719159593778972D+00
      Xi( 7) = 1.427202658643465212518104678896D+00
      Xi( 8) = 1.580999499896081555939605922667D+00
      Xi( 9) = 1.764419390174211692192546285796D+00
      Xi(10) = 1.980657586740019645990887320330D+00
      Xi(11) = 2.233343647407644670145610010792D+00
      Xi(12) = 2.526493258764244939900550646072D+00
      Xi(13) = 2.864395873153529338030198747589D+00
      Xi(14) = 3.251400783128782871148759747726D+00
      Xi(15) = 3.691587414387096180781733245624D+00
      Xi(16) = 4.188202780444439509138082611273D+00
      Xi(17) = 4.742824286606524062173684264110D+00
      Xi(18) = 5.354174409921217387461206405064D+00
      Xi(19) = 6.016530271216728066509954464536D+00
      Xi(20) = 6.717765304992344130297077242631D+00
      Xi(21) = 7.437326236737328808389102530185D+00
      Xi(22) = 8.144712410672528372056200307583D+00
      Xi(23) = 8.799430684699659993910914046467D+00
      Xi(24) = 9.353670018510573258216367342754D+00
      Xi(25) = 9.758432049866919944305920786576D+00
      Xi(26) = 9.972636514077775164548167463607D+00
      GOTO 9999
C
C     ----- K = 13 , R = 2E1 -----
C
 1312 CONTINUE
C
      RIni = 2.0D+01
C
      Omega( 1) = 0.0395138683042062947122689906681580751524D+00
      Omega( 2) = 0.0943049708717428432632281544434338371730D+00
      Omega( 3) = 0.1551950063737451642088696970867012225881D+00
      Omega( 4) = 0.2269192918890779815579289424576359124330D+00
      Omega( 5) = 0.3163249089900490444748123314289145469047D+00
      Omega( 6) = 0.4333263112160413281254579126455794835238D+00
      Omega( 7) = 0.5919767118483865042275968160190302569390D+00
      Omega( 8) = 0.8118655465128660861807291526659469127480D+00
      Omega( 9) = 1.1210827091354272629997526622425141340500D+00
      Omega(10) = 1.5638845436284244867254644706378030605265D+00
      Omega(11) = 2.2224139656716745058750994701313175028190D+00
      Omega(12) = 3.2940777647231293996394480005918126153119D+00
      Omega(13) = 5.5510419006212233681771262006066081085010D+00
C
      Alpha( 1) = 0.0153527424497649253380107594452175234778D+00
      Alpha( 2) = 0.0819058298666787536822611083542344800890D+00
      Alpha( 3) = 0.2059765874406550866432307270170731783310D+00
      Alpha( 4) = 0.3958844735513120887239815748337434797577D+00
      Alpha( 5) = 0.6656749056064381429983697890939708941005D+00
      Alpha( 6) = 1.0376871217418016666092506206098278198624D+00
      Alpha( 7) = 1.5461507729488070815755673681479720471543D+00
      Alpha( 8) = 2.2419642860926098973904652122968172989204D+00
      Alpha( 9) = 3.1994930736548128349591713082133992429590D+00
      Alpha(10) = 4.5282874945454924214653658065543595512281D+00
      Alpha(11) = 6.3981387299032817330725342319652781952755D+00
      Alpha(12) = 9.1067784968735133151532257222982025268720D+00
      Alpha(13) = 13.3491215794234994217529188809123752434971D+00
C
      Xi( 1) = 1.003264004819202082074980353354D+00
      Xi( 2) = 1.029509967775600109973765938776D+00
      Xi( 3) = 1.082718976902608210011180744914D+00
      Xi( 4) = 1.164339482501544065229381885729D+00
      Xi( 5) = 1.276584994157686558935438370099D+00
      Xi( 6) = 1.422492898985620035645638381538D+00
      Xi( 7) = 1.606012924664866670099172552000D+00
      Xi( 8) = 1.832124226907446029599668690935D+00
      Xi( 9) = 2.106975165590976750706886377884D+00
      Xi(10) = 2.438037647512079963973974372493D+00
      Xi(11) = 2.834264169129905308608385894331D+00
      Xi(12) = 3.306229242905461607193728923448D+00
      Xi(13) = 3.866220413454159565702639156370D+00
      Xi(14) = 4.528220762043572378274802758824D+00
      Xi(15) = 5.307669478395806272885693033459D+00
      Xi(16) = 6.220827499454482101270968996687D+00
      Xi(17) = 7.283455882798073766082325164462D+00
      Xi(18) = 8.508388718716954457477186224423D+00
      Xi(19) = 9.901432017101639004863500304054D+00
      Xi(20) = 11.454970491614840710781209054403D+00
      Xi(21) = 13.138975478166617705255703185685D+00
      Xi(22) = 14.890210861745023507296536990907D+00
      Xi(23) = 16.602876967717271838864689925686D+00
      Xi(24) = 18.127565187513955180520497378893D+00
      Xi(25) = 19.287724426470788330334471538663D+00
      Xi(26) = 19.918547471186226260897456086241D+00
      GOTO 9999
C
C     ----- K = 13 , R = 3E1 -----
C
 1313 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.0290391751465786714628172348467982644138D+00
      Omega( 2) = 0.0696930970758023124331454259405838058683D+00
      Omega( 3) = 0.1159194578409333255356603378793955982928D+00
      Omega( 4) = 0.1723198518541611062252633113089927974215D+00
      Omega( 5) = 0.2457778053882376194231619559449342204971D+00
      Omega( 6) = 0.3464046083311216512354117902372507842301D+00
      Omega( 7) = 0.4884592767890422960661272328009019361161D+00
      Omega( 8) = 0.6916744519623734678386017815654440710205D+00
      Omega( 9) = 0.9843875484296040770599081626279769352550D+00
      Omega(10) = 1.4113359869286716225889391496117752922146D+00
      Omega(11) = 2.0550145974924654234099880012465177969716D+00
      Omega(12) = 3.1115307366236419309591526571878716822539D+00
      Omega(13) = 5.3422507011769150369294434366196355767897D+00
C
      Alpha( 1) = 0.0112757434961027098823437898357513464731D+00
      Alpha( 2) = 0.0603223978722132603997598768869048413421D+00
      Alpha( 3) = 0.1525007237397825909968251609227785081657D+00
      Alpha( 4) = 0.2955277282819508820495540274420065429695D+00
      Alpha( 5) = 0.5027902095349923103644214361640507604534D+00
      Alpha( 6) = 0.7960938616773461903365106950847263078686D+00
      Alpha( 7) = 1.2093501035922478453353209881804275482864D+00
      Alpha( 8) = 1.7933089972016881873204813269140345255437D+00
      Alpha( 9) = 2.6223634743044171358829219631303431015112D+00
      Alpha(10) = 3.8064590061885735020313220600129966442182D+00
      Alpha(11) = 5.5162604961162127548482403405216700775782D+00
      Alpha(12) = 8.0501192457232339100281492960675677750260D+00
      Alpha(13) = 12.0989895223483451391216503267855841841083D+00
C
      Xi( 1) = 1.003823181499404824175598893721D+00
      Xi( 2) = 1.034594777838683702080729842621D+00
      Xi( 3) = 1.097135219703572158616612652882D+00
      Xi( 4) = 1.193463503379344423376180739371D+00
      Xi( 5) = 1.326679074476580561317562434454D+00
      Xi( 6) = 1.501068447036459740429915898918D+00
      Xi( 7) = 1.722267936530475874101417388395D+00
      Xi( 8) = 1.997483259315246300019501546341D+00
      Xi( 9) = 2.335760526037198537771533146667D+00
      Xi(10) = 2.748304829870090944731950144408D+00
      Xi(11) = 3.248839260751835120036648540953D+00
      Xi(12) = 3.854000274324109914661842801742D+00
      Xi(13) = 4.583734970851368934700170720831D+00
      Xi(14) = 5.461642641717418074007173167672D+00
      Xi(15) = 6.515120127022255510935294653407D+00
      Xi(16) = 7.775058708641121807837032209321D+00
      Xi(17) = 9.274648381319773452965971038964D+00
      Xi(18) = 11.046507851525320275352937127167D+00
      Xi(19) = 13.116937904918466130041454498212D+00
      Xi(20) = 15.495562270687938125762750374292D+00
      Xi(21) = 18.158489810829828919186423874521D+00
      Xi(22) = 21.024310467838543146912311243568D+00
      Xi(23) = 23.926867775241861960658362740162D+00
      Xi(24) = 26.598563146118983995316864543668D+00
      Xi(25) = 28.689894066111251145698157749564D+00
      Xi(26) = 29.849150430210857226978604650469D+00
      GOTO 9999
C
C     ----- K = 13 , R = 4E1 -----
C
 1314 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.0232593715227870833565752828331341106605D+00
      Omega( 2) = 0.0560723340674415472326076650272375090367D+00
      Omega( 3) = 0.0940721603347912478917188135174587415577D+00
      Omega( 4) = 0.1417366035997742708055094698615938852981D+00
      Omega( 5) = 0.2059105637639777391587059263144965370884D+00
      Omega( 6) = 0.2967053050336383149450120849666134859035D+00
      Omega( 7) = 0.4283329410906126902584215715430460136304D+00
      Omega( 8) = 0.6205113431853984286962720717806263337479D+00
      Omega( 9) = 0.9018119630642785586219049853706763997252D+00
      Omega(10) = 1.3175182943399094910369093169855148062197D+00
      Omega(11) = 1.9506750470698549813431191268797704196913D+00
      Omega(12) = 2.9969169133068959927964497458319215184019D+00
      Omega(13) = 5.2111890672789227571364001345699534795131D+00
C
      Alpha( 1) = 0.0090268981545618437966382375560459028208D+00
      Alpha( 2) = 0.0483993303389877087190048664949726386908D+00
      Alpha( 3) = 0.1228819816015637557480699032508297818822D+00
      Alpha( 4) = 0.2397391142273433974437717511318801655307D+00
      Alpha( 5) = 0.4118217589198438482317454756298502616119D+00
      Alpha( 6) = 0.6603888791946662452933894393858338389691D+00
      Alpha( 7) = 1.0187846262288229165753755678380798599392D+00
      Alpha( 8) = 1.5371450840589502646795633244991563515214D+00
      Alpha( 9) = 2.2893425823845693278454915819963844114682D+00
      Alpha(10) = 3.3851999418594697212163413535535028131562D+00
      Alpha(11) = 4.9958531488654190407666666473573968687560D+00
      Alpha(12) = 7.4203059140681452122111083635758177479147D+00
      Alpha(13) = 11.3476308264797006841034798796385985042434D+00
C
      Xi( 1) = 1.004232043627023784893026459031D+00
      Xi( 2) = 1.038318156231480049456195047242D+00
      Xi( 3) = 1.107721820878299290313862002222D+00
      Xi( 4) = 1.214939458996826759928922001563D+00
      Xi( 5) = 1.363815600562108043547482338909D+00
      Xi( 6) = 1.559695619851743542128019726078D+00
      Xi( 7) = 1.809661579665602149091796491120D+00
      Xi( 8) = 2.122852714864137180228453871678D+00
      Xi( 9) = 2.510866375264567018632533290656D+00
      Xi(10) = 2.988241273848355028519563347800D+00
      Xi(11) = 3.573027963140888108029002978583D+00
      Xi(12) = 4.287456156285559316643229976762D+00
      Xi(13) = 5.158681354306017041722043359187D+00
      Xi(14) = 6.219557003438503706149731442565D+00
      Xi(15) = 7.509292114577087673410460411105D+00
      Xi(16) = 9.073709985610532768873781606089D+00
      Xi(17) = 10.964534802859470019598120416049D+00
      Xi(18) = 13.236652315357856579680628783535D+00
      Xi(19) = 15.941500228811626982405869057402D+00
      Xi(20) = 19.113641092040605684587717405520D+00
      Xi(21) = 22.746480276188918878688127733767D+00
      Xi(22) = 26.753645194581302746428264072165D+00
      Xi(23) = 30.918352582718743093437296920456D+00
      Xi(24) = 34.850218727394619122605945449322D+00
      Xi(25) = 37.996806449500844848898850614205D+00
      Xi(26) = 39.768061025330197821858746465296D+00
      GOTO 9999
C
C     ----- K = 13 , R = 5E1 -----
C
 1315 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.0195487683014131804904302096292934010080D+00
      Omega( 2) = 0.0473066906441472042096202648558955772273D+00
      Omega( 3) = 0.0799532151988612554909083458831720037097D+00
      Omega( 4) = 0.1218567775112953490950352779254028945388D+00
      Omega( 5) = 0.1797943671268491339445844828803799941852D+00
      Omega( 6) = 0.2637979780802783293352630628048061112167D+00
      Omega( 7) = 0.3879558647919771652051897092849941373061D+00
      Omega( 8) = 0.5719461085923810345142508093640998367846D+00
      Omega( 9) = 0.8445649615904518039380573235508364859925D+00
      Omega(10) = 1.2516047076139490799731937564764905346237D+00
      Omega(11) = 1.8766592674405786687853378569279527710023D+00
      Omega(12) = 2.9152010854831847893892798895265627834306D+00
      Omega(13) = 5.1177889314922459049889136295519165287260D+00
C
      Alpha( 1) = 0.0075835662242047441891812278435408556376D+00
      Alpha( 2) = 0.0407376307940227880337270491284407114563D+00
      Alpha( 3) = 0.1038080361905194638215996043306699192499D+00
      Alpha( 4) = 0.2037030441964128006529597542106913010684D+00
      Alpha( 5) = 0.3528297904687777034678404508571958331231D+00
      Alpha( 6) = 0.5719362889920455448114489094280088465894D+00
      Alpha( 7) = 0.8937509529926758017857879590195580021827D+00
      Alpha( 8) = 1.3676956728177227598248349993426131732122D+00
      Alpha( 9) = 2.0670104178735341665142100087138032904477D+00
      Alpha(10) = 3.1012606244403970821160720916509490052704D+00
      Alpha(11) = 4.6418880421602061927405236119170695019420D+00
      Alpha(12) = 6.9884992950388464334927907106731481690076D+00
      Alpha(13) = 10.8291419643019349878382273644206179596949D+00
C
      Xi( 1) = 1.004554608672238489620875956376D+00
      Xi( 2) = 1.041258964374473610075416685206D+00
      Xi( 3) = 1.116101627887072712374399996449D+00
      Xi( 4) = 1.231992672938907364978654290777D+00
      Xi( 5) = 1.393424205008656586654569586869D+00
      Xi( 6) = 1.606667965054614480367549400608D+00
      Xi( 7) = 1.880081747489247432562724238192D+00
      Xi( 8) = 2.224527584018628217563118276701D+00
      Xi( 9) = 2.653899139995383925361149834998D+00
      Xi(10) = 3.185767125771570655850400966003D+00
      Xi(11) = 3.842162053775606351787089143279D+00
      Xi(12) = 4.650522429160159222378406740361D+00
      Xi(13) = 5.644807060262674138981231486412D+00
      Xi(14) = 6.866737597482240773594242000399D+00
      Xi(15) = 8.367044610287674544224501005374D+00
      Xi(16) = 10.206415511463699370003910260607D+00
      Xi(17) = 12.455507430207186489803938833631D+00
      Xi(18) = 15.192739213854361472570697610962D+00
      Xi(19) = 18.497452835697486160301816759954D+00
      Xi(20) = 22.434251096592815839445012038311D+00
      Xi(21) = 27.022081642493139871236129678778D+00
      Xi(22) = 32.180707418363440258657348636007D+00
      Xi(23) = 37.653189638622882779578349143890D+00
      Xi(24) = 42.927061612284193800603837587460D+00
      Xi(25) = 47.225778138736842323014419875449D+00
      Xi(26) = 49.677259754806401468446575009352D+00
      GOTO 9999
C
C     ----- K = 13 , R = 6E1 -----
C
 1316 CONTINUE
C
      RIni = 6.0D+01
C
      Omega( 1) = 0.0169449723784150246818773792242818210241D+00
      Omega( 2) = 0.0411429334765565038551575309522245760263D+00
      Omega( 3) = 0.0699893468845359939195321062610499751599D+00
      Omega( 4) = 0.1077561446651243788464702259843885912005D+00
      Omega( 5) = 0.1611404671865263836121094989661806096137D+00
      Omega( 6) = 0.2400622452866041391230994203609405701627D+00
      Omega( 7) = 0.3584665626662518737204684432517964864928D+00
      Omega( 8) = 0.5359938990470181285289547057359982318303D+00
      Omega( 9) = 0.8016474287450003043944621317073995214741D+00
      Omega(10) = 1.2016720578835738999752894007322367997403D+00
      Omega(11) = 1.8201701757324859294378746610298946961848D+00
      Omega(12) = 2.8525952723313601027115699038105844920210D+00
      Omega(13) = 5.0462646056798155956221729745436732628150D+00
C
      Alpha( 1) = 0.0065710035402700543248571778392336240060D+00
      Alpha( 2) = 0.0353569789480877304717082888069201729309D+00
      Alpha( 3) = 0.0903879851002289691225960455156496209383D+00
      Alpha( 4) = 0.1782821245318206675314421164046940759818D+00
      Alpha( 5) = 0.3110700601878565301492210459777254527580D+00
      Alpha( 6) = 0.5090319817929682756391281117647196197140D+00
      Alpha( 7) = 0.8042946625299619084416453962393944721043D+00
      Alpha( 8) = 1.2455733860384253659228312449691600249935D+00
      Alpha( 9) = 1.9054792787505055919744512138436220993754D+00
      Alpha(10) = 2.8932856501032743546438646964347185530642D+00
      Alpha(11) = 4.3806489535904541733624117938461495214142D+00
      Alpha(12) = 6.6677158397230641838533682363987509233993D+00
      Alpha(13) = 10.4419368034881749653097759988895631977357D+00
C
      Xi( 1) = 1.004820923451609358378194825079D+00
      Xi( 2) = 1.043689146683935296410113136023D+00
      Xi( 3) = 1.123038762755039575327691003803D+00
      Xi( 4) = 1.246146262067672114007624295517D+00
      Xi( 5) = 1.418079083392706159476998051261D+00
      Xi( 6) = 1.645936438125727314767121622907D+00
      Xi( 7) = 1.939224014751121743745966574668D+00
      Xi( 8) = 2.310365837291107429926281802146D+00
      Xi( 9) = 2.775354009902637464389865695757D+00
      Xi(10) = 3.354552724089791685031791690363D+00
      Xi(11) = 4.073693486997367368268596621306D+00
      Xi(12) = 4.965104191638338092164989268085D+00
      Xi(13) = 6.069197598346911939198866647871D+00
      Xi(14) = 7.436205820433130344714739434764D+00
      Xi(15) = 9.128058479981291190030745497097D+00
      Xi(16) = 11.220113416560385063112936787633D+00
      Xi(17) = 13.802057535277674121809710072739D+00
      Xi(18) = 16.976532089272953164257873304166D+00
      Xi(19) = 20.852555032717676044423660286498D+00
      Xi(20) = 25.528387830544335267146660584103D+00
      Xi(21) = 31.054840784752701937296293088764D+00
      Xi(22) = 37.367150273092381997097621848525D+00
      Xi(23) = 44.178361632359741350412063454201D+00
      Xi(24) = 50.857358679482774822117052693216D+00
      Xi(25) = 56.387973898118182392025232729793D+00
      Xi(26) = 59.578031283605855136786821901751D+00
      GOTO 9999
C
C     ----- K = 13 , R = 7E1 -----
C
 1317 CONTINUE
C
      RIni = 7.0D+01
C
      Omega( 1) = 0.0150071017032093006110648873187296548082D+00
      Omega( 2) = 0.0365471975379521750101896593321271211607D+00
      Omega( 3) = 0.0625365727088906125669814928258460895449D+00
      Omega( 4) = 0.0971612207871765812269653460331308991726D+00
      Omega( 5) = 0.1470341958518947886704400507262846531376D+00
      Omega( 6) = 0.2219512087350919740772218699542683850723D+00
      Omega( 7) = 0.3357153160293236112755085182035585944504D+00
      Omega( 8) = 0.5079360945962475216075224238032603807369D+00
      Omega( 9) = 0.7678029510712538914778829690632733218081D+00
      Omega(10) = 1.1619597933934012856226763998890305629175D+00
      Omega(11) = 1.7749717076250380731619951668598389460385D+00
      Omega(12) = 2.8023472237986139813867292325255675677909D+00
      Omega(13) = 4.9888828936432936915490454765631511691026D+00
C
      Alpha( 1) = 0.0058175684464509904005360128084051174469D+00
      Alpha( 2) = 0.0313496158111821703283461897776573934493D+00
      Alpha( 3) = 0.0803767583469204337375695924861052077404D+00
      Alpha( 4) = 0.1592740538724147044731408243145587277922D+00
      Alpha( 5) = 0.2797464647733758327156992212758623850277D+00
      Alpha( 6) = 0.4616472946686746037258698704963677528212D+00
      Alpha( 7) = 0.7365358864903887340064118915972102286105D+00
      Alpha( 8) = 1.1524615159229960063749093057694494746102D+00
      Alpha( 9) = 1.7814423255914706925454385477713969976321D+00
      Alpha(10) = 2.7324567429500320847428340709228677951614D+00
      Alpha(11) = 4.1773180520687124769874110086220753146335D+00
      Alpha(12) = 6.4166540692766925774344344279853658008506D+00
      Alpha(13) = 10.1375583316785151648947715941062597266864D+00
C
      Xi( 1) = 1.005047588223062141120182910736D+00
      Xi( 2) = 1.045759117208935584848218669496D+00
      Xi( 3) = 1.128956539525765419827198210445D+00
      Xi( 4) = 1.258246110899336475371160237735D+00
      Xi( 5) = 1.439214623863955571877494554744D+00
      Xi( 6) = 1.679711690973732572435168919345D+00
      Xi( 7) = 1.990289708987905371731855219686D+00
      Xi( 8) = 2.384806754208886699812966480927D+00
      Xi( 9) = 2.881194346521214070047656186091D+00
      Xi(10) = 3.502416205017879272543696034248D+00
      Xi(11) = 4.277673082152316418481224902237D+00
      Xi(12) = 5.243912828705896697362859315561D+00
      Xi(13) = 6.447699270215790543717832017379D+00
      Xi(14) = 7.947451355749782035773481903007D+00
      Xi(15) = 9.815983103290664850242019934257D+00
      Xi(16) = 12.143078404884346526738936322687D+00
      Xi(17) = 15.037406339684316336491320953428D+00
      Xi(18) = 18.626199237743337598638415641972D+00
      Xi(19) = 23.049352030501556292882425758251D+00
      Xi(20) = 28.441451052179243930498331316414D+00
      Xi(21) = 34.890186440161893889622124476091D+00
      Xi(22) = 42.354196800755224697893064345067D+00
      Xi(23) = 50.526044781888757088245123227921D+00
      Xi(24) = 58.660862952684484434434653010726D+00
      Xi(25) = 65.491295607339020265480478144582D+00
      Xi(26) = 69.471289540743901530506221320138D+00
      GOTO 9999
C
C     ----- K = 13 , R = 8E1 -----
C
 1318 CONTINUE
C
      RIni = 8.0D+01
C
      Omega( 1) = 0.0135031151724071027421650087116389293840D+00
      Omega( 2) = 0.0329745931435352286415689602311340777874D+00
      Omega( 3) = 0.0567263877008931249511724474038354415484D+00
      Omega( 4) = 0.0888672098152087184074854157855849479120D+00
      Omega( 5) = 0.1359255258845317027428887396434120660160D+00
      Omega( 6) = 0.2075704507292119542759831554712413037578D+00
      Omega( 7) = 0.3174713221754208590150679591967985970768D+00
      Omega( 8) = 0.4852129397558113599786457775797643421356D+00
      Omega( 9) = 0.7401508114959611988659983483618276522975D+00
      Omega(10) = 1.1292813870580639911116865858886626483581D+00
      Omega(11) = 1.7375901390625920424392106267141855369118D+00
      Omega(12) = 2.7606812801987436849487228540667160814337D+00
      Omega(13) = 4.9413195910534541549496978518618561793119D+00
C
      Alpha( 1) = 0.0052329389711535952884931147208874557819D+00
      Alpha( 2) = 0.0282375221936975021197891843466321049050D+00
      Alpha( 3) = 0.0725906701828313244301304520134099362849D+00
      Alpha( 4) = 0.1444595107707575670628235078862822149404D+00
      Alpha( 5) = 0.2552626614849805273025577095902249880055D+00
      Alpha( 6) = 0.4244631647399713661938588926680893109733D+00
      Alpha( 7) = 0.6830922302920382192543274912921447139524D+00
      Alpha( 8) = 1.0785818503932039662046235561909668376757D+00
      Alpha( 9) = 1.6824002197305862953916749669147634449473D+00
      Alpha(10) = 2.6032393267837956702618479321031941253750D+00
      Alpha(11) = 4.0130294793175877959660879934489230436157D+00
      Alpha(12) = 6.2128284447499835686096192333138787944335D+00
      Alpha(13) = 9.8895184069447695093466044191643504746025D+00
C
      Xi( 1) = 1.005244773103949057686291868041D+00
      Xi( 2) = 1.047561077168103439152466171969D+00
      Xi( 3) = 1.134114802087177959052585740629D+00
      Xi( 4) = 1.268812673247660698181565885889D+00
      Xi( 5) = 1.457715843211539361165204886639D+00
      Xi( 6) = 1.709361714395951217966562740713D+00
      Xi( 7) = 2.035267679186395596352063264156D+00
      Xi( 8) = 2.450621129468917855538656613845D+00
      Xi( 9) = 2.975160676224555998062998307674D+00
      Xi(10) = 3.634287845813459411677820432018D+00
      Xi(11) = 4.460475691095960163769229556152D+00
      Xi(12) = 5.495061529171468354872803763200D+00
      Xi(13) = 6.790494975217389227173381271996D+00
      Xi(14) = 8.413089781164437445069859133895D+00
      Xi(15) = 10.446246556745676386905585797393D+00
      Xi(16) = 12.993916949523328499549457148987D+00
      Xi(17) = 16.183625001847018949271972410031D+00
      Xi(18) = 20.167380860878859104648452849062D+00
      Xi(19) = 25.116783059549724644102242088195D+00
      Xi(20) = 31.204786827358898191817071499088D+00
      Xi(21) = 38.560022053792734751664905701318D+00
      Xi(22) = 47.171348864346439016398404575625D+00
      Xi(23) = 56.719719279352597291615767360895D+00
      Xi(24) = 66.352244188342391371937534216841D+00
      Xi(25) = 74.541686169997492819594420154772D+00
      Xi(26) = 79.357724794226199571678392885588D+00
      GOTO 9999
C
C     ----- K = 13 , R = 9E1 -----
C
 1319 CONTINUE
C
      RIni = 9.0D+01
C
      Omega( 1) = 0.0122986244453973046956789770545181994521D+00
      Omega( 2) = 0.0301091511469223380805432287513068168039D+00
      Omega( 3) = 0.0520540922468294390643302714527518304521D+00
      Omega( 4) = 0.0821720281454225061763515028379067928199D+00
      Omega( 5) = 0.1269082267776730492159524207873566581384D+00
      Omega( 6) = 0.1958075833731895737561956583050459812512D+00
      Omega( 7) = 0.3024160839681164305876165387942933193699D+00
      Omega( 8) = 0.4662984385462345585224210736674521626810D+00
      Omega( 9) = 0.7169579916993148207258503001870764137493D+00
      Omega(10) = 1.1017045394902961527984833201010417269572D+00
      Omega(11) = 1.7059067569607391338937230074179751682095D+00
      Omega(12) = 2.7252874685729666863042569557862293549988D+00
      Omega(13) = 4.9009295252479360156111498625364220060874D+00
C
      Alpha( 1) = 0.0047648122069902020234673299496630516359D+00
      Alpha( 2) = 0.0257437170231617231684525517673711725308D+00
      Alpha( 3) = 0.0663430976309872055498477808055390880781D+00
      Alpha( 4) = 0.1325491890132242400892432707126822322152D+00
      Alpha( 5) = 0.2355255628264578346566611097290255827374D+00
      Alpha( 6) = 0.3943771928109263805216702647449267260527D+00
      Alpha( 7) = 0.6396459009904628332772794085947509756807D+00
      Alpha( 8) = 1.0181944860684312073725518721367677699163D+00
      Alpha( 9) = 1.6009832616680563971466380590591427335312D+00
      Alpha(10) = 2.4964300537331274747795267776595551367791D+00
      Alpha(11) = 3.8765534322194465295143006589029255337664D+00
      Alpha(12) = 6.0427974354698543641740970944198352299281D+00
      Alpha(13) = 9.6819261346061663729353763407914357230766D+00
C
      Xi( 1) = 1.005419169394431866989929325040D+00
      Xi( 2) = 1.049155728700803947629240087114D+00
      Xi( 3) = 1.138684850126388985029103617652D+00
      Xi( 4) = 1.278189650421907566299473124616D+00
      Xi( 5) = 1.474168408304818242253453330903D+00
      Xi( 6) = 1.735794802552608655623125888390D+00
      Xi( 7) = 2.075483220568718073896843923087D+00
      Xi( 8) = 2.509661483903695123297283853248D+00
      Xi( 9) = 3.059764774506369659854559950318D+00
      Xi(10) = 3.753493148167162929938756033010D+00
      Xi(11) = 4.626424342029386261102019695102D+00
      Xi(12) = 5.724083056224131806945243372553D+00
      Xi(13) = 7.104569892337821482384413451783D+00
      Xi(14) = 8.841829722144629405584936865381D+00
      Xi(15) = 11.029571080202116001474366147050D+00
      Xi(16) = 13.785653703813367473331175094842D+00
      Xi(17) = 17.256289034577141021375989993203D+00
      Xi(18) = 21.618330610373903860263467002767D+00
      Xi(19) = 27.075670471000502757125127928362D+00
      Xi(20) = 33.841219752308959727027382058395D+00
      Xi(21) = 42.087911423474211412149026401153D+00
      Xi(22) = 51.840769452913119781645168160367D+00
      Xi(23) = 62.777335074704843181220281778110D+00
      Xi(24) = 73.942898924447816359772911454229D+00
      Xi(25) = 83.543802300131321615650481504645D+00
      Xi(26) = 89.237879240503543407103670404013D+00
      GOTO 9999
C
C     ----- K = 13 , R = 1E2 -----
C
 1321 CONTINUE
C
      RIni = 1.0D+02
C
      Omega( 1) = 0.0113101318768680369468646322757277022930D+00
      Omega( 2) = 0.0277543331291073884246130461239676634477D+00
      Omega( 3) = 0.0482051453560221898190271249545757470401D+00
      Omega( 4) = 0.0766369992937893382286911568568310570981D+00
      Omega( 5) = 0.1194143136132964391524333321037421029587D+00
      Omega( 6) = 0.1859624116085836987492894284423883277668D+00
      Omega( 7) = 0.2897143390284020331713107226923042958333D+00
      Omega( 8) = 0.4502180845978160193137946470942623022893D+00
      Omega( 9) = 0.6971087365781358037076544109655884540189D+00
      Omega(10) = 1.0779765276442093285824697090902191121131D+00
      Omega(11) = 1.6785411291166155571845039484735195856047D+00
      Omega(12) = 2.6946568538459302127267142035194069649151D+00
      Omega(13) = 4.8659851887432857154430765334041097958107D+00
C
      Alpha( 1) = 0.0043806958779880428850311965480179843269D+00
      Alpha( 2) = 0.0236960325297795352306781235113275485560D+00
      Alpha( 3) = 0.0612068179585451655720922548714613142806D+00
      Alpha( 4) = 0.1227397677409143183610193937604915248585D+00
      Alpha( 5) = 0.2192288472522569340688843378406325257401D+00
      Alpha( 6) = 0.3694491542866578709294195226320312031021D+00
      Alpha( 7) = 0.6034892573939584233628716236630395997054D+00
      Alpha( 8) = 0.9676872342224387200763924266055937550846D+00
      Alpha( 9) = 1.5325340755817475489782009123729267230374D+00
      Alpha(10) = 2.4061866587862055992357024347683136511478D+00
      Alpha(11) = 3.7607303110864559931615647148106518216082D+00
      Alpha(12) = 5.8979582101501310427427160476554490742274D+00
      Alpha(13) = 9.5045784620299077691263356371109694009647D+00
C
      Xi( 1) = 1.005575418829604012718910432955D+00
      Xi( 2) = 1.050585204860276456232358321952D+00
      Xi( 3) = 1.142785715858074474440136958364D+00
      Xi( 4) = 1.286616261298543977507902180690D+00
      Xi( 5) = 1.488980867544841086343859748453D+00
      Xi( 6) = 1.759646306988846534602628779442D+00
      Xi( 7) = 2.111865308076226843518596876237D+00
      Xi( 8) = 2.563231938172798349100112158538D+00
      Xi( 9) = 3.136781443005848844362420946297D+00
      Xi(10) = 3.862392628341886536289320752680D+00
      Xi(11) = 4.778600044854552687540599764660D+00
      Xi(12) = 5.934938395222085082310325976351D+00
      Xi(13) = 7.394949794463160231382248266385D+00
      Xi(14) = 9.239970870682225946059207277727D+00
      Xi(15) = 11.573758634168001793035562840828D+00
      Xi(16) = 14.527827017141322474454157021029D+00
      Xi(17) = 18.266882671339681290176293337879D+00
      Xi(18) = 22.992617176036376688483264807417D+00
      Xi(19) = 28.941626422796236008799786176837D+00
      Xi(20) = 36.368046303409890446234342675780D+00
      Xi(21) = 45.491932781370753489935676583400D+00
      Xi(22) = 56.379688628733474167059602777385D+00
      Xi(23) = 68.713059125646557309674555824586D+00
      Xi(24) = 81.441938730602989139628622439204D+00
      Xi(25) = 92.501408562411169524131171471026D+00
      Xi(26) = 99.112194153775850354037846301480D+00
      GOTO 9999
C
C     ----- K = 13 , R = 2E2 -----
C
 1322 CONTINUE
C
      RIni = 2.0D+02
C
      Omega( 1) = 0.0064910169387615788873851173888807064571D+00
      Omega( 2) = 0.0162170689188738721855351275838441615917D+00
      Omega( 3) = 0.0291812342620532922049194799948845790993D+00
      Omega( 4) = 0.0489100379253506474367969164562808259689D+00
      Omega( 5) = 0.0811158535922647771682666810455497596877D+00
      Omega( 6) = 0.1343271900794126477908128555749378563178D+00
      Omega( 7) = 0.2212386426239065575236140276382812430711D+00
      Omega( 8) = 0.3612893986938155437710091383296528988467D+00
      Omega( 9) = 0.5848980597832797884171908420203322975794D+00
      Omega(10) = 0.9414285557062445442618077440233292918492D+00
      Omega(11) = 1.5190239900002864945234570814847074871068D+00
      Omega(12) = 2.5149049645269635100842259278763890506525D+00
      Omega(13) = 4.6611089868946715738390329519802435243037D+00
C
      Alpha( 1) = 0.0025091447649231517130936230551216392648D+00
      Alpha( 2) = 0.0136939543239405979498931687290963488124D+00
      Alpha( 3) = 0.0360053496395654930710438813845630789956D+00
      Alpha( 4) = 0.0742866820971900976034460808983261870253D+00
      Alpha( 5) = 0.1379542477076401043700314835716014272293D+00
      Alpha( 6) = 0.2434756239225348904388823365052552105681D+00
      Alpha( 7) = 0.4177689435720809181128067993915564670715D+00
      Alpha( 8) = 0.7035481042365874107340738274896096982047D+00
      Alpha( 9) = 1.1680019135509342026717910512090270458430D+00
      Alpha(10) = 1.9172656578193901303871007901769019099447D+00
      Alpha(11) = 3.1235654399781725293078898930687614665658D+00
      Alpha(12) = 5.0909794073166716866889192072420655676979D+00
      Alpha(13) = 8.5067809666083261223648581683676184184151D+00
C
      Xi( 1) = 1.006601834937339819044804789527D+00
      Xi( 2) = 1.059993630307594630695321169522D+00
      Xi( 3) = 1.169875904016283995516881089038D+00
      Xi( 4) = 1.342575070422474490475922559440D+00
      Xi( 5) = 1.588006775801091085263162805408D+00
      Xi( 6) = 1.920390729415649058134558935240D+00
      Xi( 7) = 2.359375501626099070565795168264D+00
      Xi( 8) = 2.931586599079488560494824445080D+00
      Xi( 9) = 3.672655956040545165500188495855D+00
      Xi(10) = 4.629914626029690094630858476776D+00
      Xi(11) = 5.866023886463970311508364652298D+00
      Xi(12) = 7.463891992454524606723231805461D+00
      Xi(13) = 9.533304684701023087867177707722D+00
      Xi(14) = 12.219814936428096325583825265682D+00
      Xi(15) = 15.716566939691546548295286722663D+00
      Xi(16) = 20.279731535473521229079318928257D+00
      Xi(17) = 26.247922766759155376731227349296D+00
      Xi(18) = 34.064729238468355203284243515327D+00
      Xi(19) = 44.299760950361843418049412335336D+00
      Xi(20) = 57.653781313149099905374095342836D+00
      Xi(21) = 74.909832932261869960222089304125D+00
      Xi(22) = 96.740265623925491222934169854852D+00
      Xi(23) = 123.190236423527929972288053050988D+00
      Xi(24) = 152.603923740088436064588073293180D+00
      Xi(25) = 180.140007970985797022822616497706D+00
      Xi(26) = 197.594393511301768248666910210432D+00
      GOTO 9999
C
C     ----- K = 13 , R = 3E2 -----
C
 1323 CONTINUE
C
      RIni = 3.0D+02
C
      Omega( 1) = 0.0046795081858433881002059097440476387320D+00
      Omega( 2) = 0.0118427641824228194680434741482832894377D+00
      Omega( 3) = 0.0218565387549393473972666286295196691469D+00
      Omega( 4) = 0.0379717272342794108123476340410640261780D+00
      Omega( 5) = 0.0654616436269458626932447528612435405648D+00
      Omega( 6) = 0.1123190471227664696216583598009641775661D+00
      Omega( 7) = 0.1908397923835510966958676701277219933672D+00
      Omega( 8) = 0.3203673716733569136678235617088361664173D+00
      Omega( 9) = 0.5316673394828265334440160760909321879808D+00
      Omega(10) = 0.8750363462747959466658115301740394897934D+00
      Omega(11) = 1.4400626713409013551943360642582092623343D+00
      Omega(12) = 2.4250786861151264221445056357140401814831D+00
      Omega(13) = 4.5588403252510033966324320964247363008326D+00
C
      Alpha( 1) = 0.0018063447618946470180019119185742260125D+00
      Alpha( 2) = 0.0099215980419140012612502451706808503928D+00
      Alpha( 3) = 0.0264254867986646808014411166865009761295D+00
      Alpha( 4) = 0.0556459478438246073363606172823558893015D+00
      Alpha( 5) = 0.1061317262233611031827071498334635890615D+00
      Alpha( 6) = 0.1929787217317038158727240254597390389790D+00
      Alpha( 7) = 0.3412478358546339905418842697315540135605D+00
      Alpha( 8) = 0.5915378195802623962707214588796489351807D+00
      Alpha( 9) = 1.0090348395240912329979057759921090564603D+00
      Alpha(10) = 1.6985030889793068443376425347146607691684D+00
      Alpha(11) = 2.8320126081133543201502622554954768929747D+00
      Alpha(12) = 4.7148939360311128031130745164034578920109D+00
      Alpha(13) = 8.0352971415653418711666500229284793022089D+00
C
      Xi( 1) = 1.007193626623011663040636876065D+00
      Xi( 2) = 1.065432637443806315681767149517D+00
      Xi( 3) = 1.185616548254826569481698572694D+00
      Xi( 4) = 1.375324414382733756770892108445D+00
      Xi( 5) = 1.646491689482819092775998937039D+00
      Xi( 6) = 2.016373982421509058375955203424D+00
      Xi( 7) = 2.509066895179997598586130180820D+00
      Xi( 8) = 3.157602591261189096385683661161D+00
      Xi( 9) = 4.006736484589320281858582095680D+00
      Xi(10) = 5.116720794204446622500254138188D+00
      Xi(11) = 6.568512853202026385932704721782D+00
      Xi(12) = 8.470969485648279558714923709317D+00
      Xi(13) = 10.970782909024342312298561097350D+00
      Xi(14) = 14.266175813728105064778772792522D+00
      Xi(15) = 18.625738347419756483352992759706D+00
      Xi(16) = 24.414140711996516648618626987854D+00
      Xi(17) = 32.126557836826970551008519905167D+00
      Xi(18) = 42.432935152670238524580392791563D+00
      Xi(19) = 56.229388980517890157112548266127D+00
      Xi(20) = 74.682125611916942660439833190367D+00
      Xi(21) = 99.215116053828261460934001547685D+00
      Xi(22) = 131.305011547351796738491991334286D+00
      Xi(23) = 171.754940608600189771992816645252D+00
      Xi(24) = 218.852558605650953330035157762268D+00
      Xi(25) = 265.146181392494579703855706043214D+00
      Xi(26) = 295.708296475482044995963804012717D+00
      GOTO 9999
C
C     ----- K = 13 , R = 4E2 -----
C
 1324 CONTINUE
C
      RIni = 4.0D+02
C
      Omega( 1) = 0.0037074547827871263050682546077649326399D+00
      Omega( 2) = 0.0094812003352843043722770295309021371111D+00
      Omega( 3) = 0.0178578875997744910917176385661830906315D+00
      Omega( 4) = 0.0318905602746983744899078681150905723030D+00
      Omega( 5) = 0.0565361202845229826009982362217121121262D+00
      Omega( 6) = 0.0994286622560874676419909005964381520926D+00
      Omega( 7) = 0.1725966329403311167291559802272615797847D+00
      Omega( 8) = 0.2952899874667847870675189969613683160787D+00
      Omega( 9) = 0.4984624791364017406245824626243035027073D+00
      Omega(10) = 0.8330134160758220932072731779172158894653D+00
      Omega(11) = 1.3895433034020467885942193286830104170804D+00
      Omega(12) = 2.3672713492607711707055306860070231778082D+00
      Omega(13) = 4.4930584436212646576649931695612849580357D+00
C
      Alpha( 1) = 0.0014294954818313334632518094187425550246D+00
      Alpha( 2) = 0.0078925616602339314342899416997167971743D+00
      Alpha( 3) = 0.0212436561183783878532213605996870064985D+00
      Alpha( 4) = 0.0454727229384414977013764182223187759746D+00
      Alpha( 5) = 0.0885341543352852344698003528300489506364D+00
      Alpha( 6) = 0.1645798034473368497496029508320170009483D+00
      Alpha( 7) = 0.2974049919255520110382331328002791792642D+00
      Alpha( 8) = 0.5261556981667115560842747090664062170617D+00
      Alpha( 9) = 0.9146013961828088281594620334580270082370D+00
      Alpha(10) = 1.5664758893112251574367133066800761298509D+00
      Alpha(11) = 2.6536401907715668653520157960201686364599D+00
      Alpha(12) = 4.4822507877281657917567514459733502008021D+00
      Alpha(13) = 7.7412394984528697886722037324602752050851D+00
C
      Xi( 1) = 1.007606096810241348856428122982D+00
      Xi( 2) = 1.069229859498175030452574163498D+00
      Xi( 3) = 1.196640759740549974678158751118D+00
      Xi( 4) = 1.398364305325412257721175157688D+00
      Xi( 5) = 1.687871479767406181107629314919D+00
      Xi( 6) = 2.084750944590363017300607695326D+00
      Xi( 7) = 2.616558091060674186983936451156D+00
      Xi( 8) = 3.321371593068426360815040276897D+00
      Xi( 9) = 4.251226063961023316807585903376D+00
      Xi(10) = 5.476831814846624884880049721758D+00
      Xi(11) = 7.094177762738691004999747047322D+00
      Xi(12) = 9.233784044797457095897497136860D+00
      Xi(13) = 12.073668324863166596665198238014D+00
      Xi(14) = 15.857533441545550142830200979205D+00
      Xi(15) = 20.920324540554137085143771024320D+00
      Xi(16) = 27.723976175274394985470483021572D+00
      Xi(17) = 36.906997911226415293173275244953D+00
      Xi(18) = 49.351401945981475661434645374470D+00
      Xi(19) = 66.267664988518538801698376516924D+00
      Xi(20) = 89.285622974993283866573801077493D+00
      Xi(21) = 120.498395074273122885843978480125D+00
      Xi(22) = 162.287654409046177672593103125109D+00
      Xi(23) = 216.460512662666428068680701812809D+00
      Xi(24) = 281.680285519143629169303011394732D+00
      Xi(25) = 348.193887706741630805318266084214D+00
      Xi(26) = 393.535359982477614654294839624527D+00
      GOTO 9999
C
C     ----- K = 13 , R = 5E2 -----
C
 1325 CONTINUE
C
      RIni = 5.0D+02
C
      Omega( 1) = 0.0030940770041673703710907771327609516732D+00
      Omega( 2) = 0.0079837694611341389841908510599521875406D+00
      Omega( 3) = 0.0152993134475317661556109508948220199187D+00
      Omega( 4) = 0.0279405761711479469595415187424536362926D+00
      Omega( 5) = 0.0506232403794067759958039652568029254098D+00
      Omega( 6) = 0.0907199318854559077520121473936498546209D+00
      Omega( 7) = 0.1600583567295715970698949046080450386853D+00
      Omega( 8) = 0.2778002177833187805590202390515486285949D+00
      Omega( 9) = 0.4750130531377467067613400153458513841542D+00
      Omega(10) = 0.8030283835320917704462227726391887472346D+00
      Omega(11) = 1.3532170717002870102092446091823774168006D+00
      Omega(12) = 2.3255281638367188130987056604581653118657D+00
      Omega(13) = 4.4455695860020917236157811380792281852337D+00
C
      Alpha( 1) = 0.0011918361572482997446954992404148654384D+00
      Alpha( 2) = 0.0066097900285615277493251359768045016807D+00
      Alpha( 3) = 0.0179527056750687483633514927167107444461D+00
      Alpha( 4) = 0.0389639196029066920288412394224319967861D+00
      Alpha( 5) = 0.0771529416735844446117260561623929504549D+00
      Alpha( 6) = 0.1459668488904268170570487778148205393336D+00
      Alpha( 7) = 0.2682605794022303309740318938425573946915D+00
      Alpha( 8) = 0.4820900184432984925141964238637370954166D+00
      Alpha( 9) = 0.8501377564276385293733431791629584495240D+00
      Alpha(10) = 1.4753151598618290542925743369373492441810D+00
      Alpha(11) = 2.5292704379654323263749982064751975485706D+00
      Alpha(12) = 4.3187598299444165074120749459041235240875D+00
      Alpha(13) = 7.5333877438232410093249535876225309039000D+00
C
      Xi( 1) = 1.007920571368887845890675858396D+00
      Xi( 2) = 1.072128463819201193074279188089D+00
      Xi( 3) = 1.205075672577440925236844093682D+00
      Xi( 4) = 1.416050087437708750507346644465D+00
      Xi( 5) = 1.719766148884946971526002656994D+00
      Xi( 6) = 2.137716588041077865525035894834D+00
      Xi( 7) = 2.700305475636761196045476651761D+00
      Xi( 8) = 3.449804264385363344186385958778D+00
      Xi( 9) = 4.444350907576345230919068196584D+00
      Xi(10) = 5.763517444790797477835453399564D+00
      Xi(11) = 7.516173032776385768555238442445D+00
      Xi(12) = 9.851612969935577901643697362744D+00
      Xi(13) = 12.975339413006578889904496509899D+00
      Xi(14) = 17.171471500771985724265178419046D+00
      Xi(15) = 22.834704158258824147759158407212D+00
      Xi(16) = 30.515822412028053598614940611355D+00
      Xi(17) = 40.986238310313833299591834702369D+00
      Xi(18) = 55.327968709077348798131001927914D+00
      Xi(19) = 75.053627355245311279785536129339D+00
      Xi(20) = 102.249274562275927230592387218167D+00
      Xi(21) = 139.687387242808906465119100914762D+00
      Xi(22) = 190.713438761254829259539356201003D+00
      Xi(23) = 258.307893451777710713823665855671D+00
      Xi(24) = 341.856235332755066941290422732891D+00
      Xi(25) = 429.646147143148018854308389791186D+00
      Xi(26) = 491.120109199436753161238655707166D+00
      GOTO 9999
C
C     ----- K = 13 , R = 6E2 -----
C
 1326 CONTINUE
C
      RIni = 6.0D+02
C
      Omega( 1) = 0.0026687973590316281074251490799259523001D+00
      Omega( 2) = 0.0069412729280144503670891957015920414875D+00
      Omega( 3) = 0.0135041197046104785608436511801233592678D+00
      Omega( 4) = 0.0251330777276360453349312445991969866554D+00
      Omega( 5) = 0.0463522796265119676730132383668525619669D+00
      Omega( 6) = 0.0843323236642424205105413211047693522460D+00
      Omega( 7) = 0.1507401942263835080632397578348502520384D+00
      Omega( 8) = 0.2646556816965149062361936649345750538487D+00
      Omega( 9) = 0.4572195938687172712098800292368494524453D+00
      Omega(10) = 0.7800936606602578585276802636272464042122D+00
      Omega(11) = 1.3252656634606378833161899333248356924742D+00
      Omega(12) = 2.2933018738760703215950886857399382279254D+00
      Omega(13) = 4.4089135030774377404724773121813541365555D+00
C
      Alpha( 1) = 0.0010271376750990705541017466865053031633D+00
      Alpha( 2) = 0.0057189693059005456997162498370551597660D+00
      Alpha( 3) = 0.0156583921950699245940867316423394939306D+00
      Alpha( 4) = 0.0343971045134781304859457676248446489353D+00
      Alpha( 5) = 0.0690933050712323722092615858880737533809D+00
      Alpha( 6) = 0.1326398987674319438683032817971607641994D+00
      Alpha( 7) = 0.2471536181328989768683258854142792415587D+00
      Alpha( 8) = 0.4498264599858223900072807382999329206541D+00
      Alpha( 9) = 0.8024646171256123415582717228744513704441D+00
      Alpha(10) = 1.4072970218638971147363456171142104267346D+00
      Alpha(11) = 2.4357683331621138168431722759166291325528D+00
      Alpha(12) = 4.1950968917859868274194967696644198440481D+00
      Alpha(13) = 7.3754691998131300008367805531861449708231D+00
C
      Xi( 1) = 1.008173416081267910449602298595D+00
      Xi( 2) = 1.074461248734802413097415285659D+00
      Xi( 3) = 1.211876394948510017070264499495D+00
      Xi( 4) = 1.430345761103000806172165504382D+00
      Xi( 5) = 1.745630193286268445722331232872D+00
      Xi( 6) = 2.180834635848444269783791282791D+00
      Xi( 7) = 2.768791552967651401059509530533D+00
      Xi( 8) = 3.555371145354761311125485478968D+00
      Xi( 9) = 4.603991437296975227332074265263D+00
      Xi(10) = 6.001949714536108498267458211117D+00
      Xi(11) = 7.869442974468143793195795687723D+00
      Xi(12) = 10.372432825988690112753443095528D+00
      Xi(13) = 13.741047839939295708774902315952D+00
      Xi(14) = 18.295992158406104599599473203853D+00
      Xi(15) = 24.486618912601472877396679983519D+00
      Xi(16) = 32.945895338515881759638981307603D+00
      Xi(17) = 44.569663490244743889873157982606D+00
      Xi(18) = 60.629619299190485648409776331391D+00
      Xi(19) = 82.929263074030677947712231201649D+00
      Xi(20) = 114.001709620792057747751258034441D+00
      Xi(21) = 157.300713576312651373334539073312D+00
      Xi(22) = 217.172180125454940727292019175820D+00
      Xi(23) = 297.893750716248039644984402229966D+00
      Xi(24) = 399.857819251229574419870260726384D+00
      Xi(25) = 509.736103384614762573656676636347D+00
      Xi(26) = 588.491495358906647139551893133103D+00
      GOTO 9999
C
C     ----- K = 13 , R = 7E2 -----
C
 1327 CONTINUE
C
      RIni = 7.0D+02
C
      Omega( 1) = 0.0023551074366808127160831978471125514041D+00
      Omega( 2) = 0.0061695608640397718077784828399900973750D+00
      Omega( 3) = 0.0121659998421845802122178217818948553486D+00
      Omega( 4) = 0.0230164260498364710901680885157816192077D+00
      Omega( 5) = 0.0430881697404923999683470491744996522243D+00
      Omega( 6) = 0.0793891480732823163042827588342653655218D+00
      Omega( 7) = 0.1434523233695349266059171355891521670856D+00
      Omega( 8) = 0.2542818368570532217324501461641617083842D+00
      Omega( 9) = 0.4430674773326116878773266866131663732631D+00
      Omega(10) = 0.7617342359687817124900069409854097557400D+00
      Omega(11) = 1.3027813128385978228441360871059373494063D+00
      Omega(12) = 2.2673081489216234296786567359305308855255D+00
      Omega(13) = 4.3793496679197264718347293310785062203649D+00
C
      Alpha( 1) = 0.0009057063557368288924064534393173347970D+00
      Alpha( 2) = 0.0050609742799303003855303185673508714615D+00
      Alpha( 3) = 0.0139578927153902635739524530460176077185D+00
      Alpha( 4) = 0.0309928908733394336304705967907735519873D+00
      Alpha( 5) = 0.0630365269501992592267726293886465072092D+00
      Alpha( 6) = 0.1225299401529001124219763786682424466790D+00
      Alpha( 7) = 0.2309880248499910385210362809282891305429D+00
      Alpha( 8) = 0.4248923236469510487814480598212085737941D+00
      Alpha( 9) = 0.7653181374523062194497984023211500925754D+00
      Alpha(10) = 1.3539125860336706885615754014651201941888D+00
      Alpha(11) = 2.3619293562501908043029658079703381190484D+00
      Alpha(12) = 4.0969574805805687592666353147308200277621D+00
      Alpha(13) = 7.2496932242178477282142090576400050849770D+00
C
      Xi( 1) = 1.008384022836860419267308863756D+00
      Xi( 2) = 1.076405874089302964037931364949D+00
      Xi( 3) = 1.217553969104368558756812390698D+00
      Xi( 4) = 1.442305417051125772213981957037D+00
      Xi( 5) = 1.767324775011994340790863478041D+00
      Xi( 6) = 2.217116345408709288502049905567D+00
      Xi( 7) = 2.826633150094971313939090640233D+00
      Xi( 8) = 3.644905456456473356267183882906D+00
      Xi( 9) = 4.740013615456810919853575247984D+00
      Xi(10) = 6.206126350713595783009646211781D+00
      Xi(11) = 8.173582957816336732985440516330D+00
      Xi(12) = 10.823385901403868655443196722388D+00
      Xi(13) = 14.408058428103994915922547725984D+00
      Xi(14) = 19.281856961649636606273561234559D+00
      Xi(15) = 25.944689943726666691084514759780D+00
      Xi(16) = 35.106255469330143126338583670076D+00
      Xi(17) = 47.779719367432811779478063485271D+00
      Xi(18) = 65.417561260579817718208150978398D+00
      Xi(19) = 90.103882605797534721681074643129D+00
      Xi(20) = 124.809231064963922691601894143787D+00
      Xi(21) = 173.666481733179374452591403610313D+00
      Xi(22) = 242.045296652620456165361240152833D+00
      Xi(23) = 335.614657594662747414071546359082D+00
      Xi(24) = 456.012862937083508219071603662087D+00
      Xi(25) = 588.628705898983448308037225160660D+00
      Xi(26) = 685.670160951395615878123379616227D+00
      GOTO 9999
C
C     ----- K = 13 , R = 8E2 -----
C
 1328 CONTINUE
C
      RIni = 8.0D+02
C
      Omega( 1) = 0.0021133505242918429081988530501834411268D+00
      Omega( 2) = 0.0055729095031150907330734793534293691764D+00
      Omega( 3) = 0.0111249328298707799984366777733268971673D+00
      Omega( 4) = 0.0213527604993536956686666673788099757303D+00
      Omega( 5) = 0.0404923224591177310863340191998682726648D+00
      Omega( 6) = 0.0754164751756804483472204810035055544404D+00
      Omega( 7) = 0.1375432795043878238857443241371925068961D+00
      Omega( 8) = 0.2458070076870906891651925169073322763325D+00
      Omega( 9) = 0.4314308537768518283660353074271398554629D+00
      Omega(10) = 0.7465562294807239240076318642636010736169D+00
      Omega(11) = 1.2841170511265791665455812675666891209403D+00
      Omega(12) = 2.2456810726269699539456997650077596517804D+00
      Omega(13) = 4.3547537938385463066048863822743442142382D+00
C
      Alpha( 1) = 0.0008121562645573330148213797650535300532D+00
      Alpha( 2) = 0.0045532336041415750128387625057096999015D+00
      Alpha( 3) = 0.0126416380474307741316834503668420364164D+00
      Alpha( 4) = 0.0283442039758663218032650604749633771462D+00
      Alpha( 5) = 0.0582897240185137544883240898196952173294D+00
      Alpha( 6) = 0.1145411279601758402303711761871518604039D+00
      Alpha( 7) = 0.2181089137979318667159714731362818440630D+00
      Alpha( 8) = 0.4048745086967141709516273401225916472868D+00
      Alpha( 9) = 0.7352886271457931505819878570395786709923D+00
      Alpha(10) = 1.3104923188652284860803570731846434682666D+00
      Alpha(11) = 2.3015610953500072652887220581163774113520D+00
      Alpha(12) = 4.0163901817262514979366139211691688615247D+00
      Alpha(13) = 7.1461275739184389006394826360235583706526D+00
C
      Xi( 1) = 1.008563939331447717375617179059D+00
      Xi( 2) = 1.078068243250622171725688902555D+00
      Xi( 3) = 1.222413496521834490792998706432D+00
      Xi( 4) = 1.452559832886258677549923268302D+00
      Xi( 5) = 1.785967260910051081072326406574D+00
      Xi( 6) = 2.248377622169493433336440180170D+00
      Xi( 7) = 2.876627602813417167301399124391D+00
      Xi( 8) = 3.722566318997987185697093326375D+00
      Xi( 9) = 4.858458525145758260567474540181D+00
      Xi(10) = 6.384669240241603082545301939277D+00
      Xi(11) = 8.440747879163621662616801000212D+00
      Xi(12) = 11.221427886655754447252364425225D+00
      Xi(13) = 14.999833696404702469522074448349D+00
      Xi(14) = 20.161279869226432774620416665812D+00
      Xi(15) = 27.252842878380206744631797377565D+00
      Xi(16) = 37.056368544093142066570001347259D+00
      Xi(17) = 50.696266365555572926127059218970D+00
      Xi(18) = 69.797994065457069884361729350530D+00
      Xi(19) = 96.716915210622292145559164566748D+00
      Xi(20) = 134.851540808634282886391275724236D+00
      Xi(21) = 189.009172050851384269676280780459D+00
      Xi(22) = 265.598537457188825450238311987050D+00
      Xi(23) = 371.753223499904204146782049633657D+00
      Xi(24) = 510.561540063745275486173369472453D+00
      Xi(25) = 666.447772441221866601690493325805D+00
      Xi(26) = 782.671756056125553668856298372702D+00
      GOTO 9999
C
C     ----- K = 13 , R = 9E2 -----
C
 1329 CONTINUE
C
      RIni = 9.0D+02
C
      Omega( 1) = 0.0019208272844967494482146201499085658160D+00
      Omega( 2) = 0.0050963883462757398966150865823182547487D+00
      Omega( 3) = 0.0102886814614984720077820270524304735460D+00
      Omega( 4) = 0.0200039648510271715401272413573097175288D+00
      Omega( 5) = 0.0383659899437067586735312062218294926197D+00
      Omega( 6) = 0.0721328305621485951787006871138707708724D+00
      Omega( 7) = 0.1326220401624135987212350223929391290767D+00
      Omega( 8) = 0.2387032183270266254342074560557129814242D+00
      Omega( 9) = 0.4216224758355891454573733528876644527372D+00
      Omega(10) = 0.7337033242537522846355574346333128232800D+00
      Omega(11) = 1.2682562982661819158238317162279429339833D+00
      Omega(12) = 2.2272659740951444588743374231043503641558D+00
      Omega(13) = 4.3338116476561864979025195943052040092880D+00
C
      Alpha( 1) = 0.0007376832054758342712174149782226179051D+00
      Alpha( 2) = 0.0041484372456243542216096397475671997412D+00
      Alpha( 3) = 0.0115892775607377264963210837522833429603D+00
      Alpha( 4) = 0.0262164386240689977399393731025134535173D+00
      Alpha( 5) = 0.0544514124476461614242665462692905009590D+00
      Alpha( 6) = 0.1080339617046875448831168287089354862474D+00
      Alpha( 7) = 0.2075428669744146560887255309468812924933D+00
      Alpha( 8) = 0.3883422725388195523353682819811538706745D+00
      Alpha( 9) = 0.7103391909477947375952007480481853463061D+00
      Alpha(10) = 1.2742276377577958282587972105304174874618D+00
      Alpha(11) = 2.2509169842175276083674761995467861197540D+00
      Alpha(12) = 3.9485610579775714584684770391831420965900D+00
      Alpha(13) = 7.0587123560753475440092408188519357281621D+00
C
      Xi( 1) = 1.008720586782332077381563184559D+00
      Xi( 2) = 1.079516454195575416081555319980D+00
      Xi( 3) = 1.226651562695841871639991504939D+00
      Xi( 4) = 1.461516346230133188595357029982D+00
      Xi( 5) = 1.802281412551816476932432797842D+00
      Xi( 6) = 2.275798082616062384441260602053D+00
      Xi( 7) = 2.920597345898910810000487225935D+00
      Xi( 8) = 3.791077854083673642901150535600D+00
      Xi( 9) = 4.963300876517859849985925269067D+00
      Xi(10) = 6.543286329812640416889180094984D+00
      Xi(11) = 8.679023991891225893804961977196D+00
      Xi(12) = 11.577915343461111109119032747117D+00
      Xi(13) = 15.532178628785418812371932340000D+00
      Xi(14) = 20.956117309346031146732869387961D+00
      Xi(15) = 28.441103191377953986196391866770D+00
      Xi(16) = 38.837191986032072562456951558829D+00
      Xi(17) = 53.374743426128173573452784239635D+00
      Xi(18) = 73.845296896195637199877825906924D+00
      Xi(19) = 102.866954405769408388560126610400D+00
      Xi(20) = 144.257018853382552717179443391160D+00
      Xi(21) = 203.491298142388639003108607994363D+00
      Xi(22) = 288.027035957418353795400633998725D+00
      Xi(23) = 406.520921201055780158650776456852D+00
      Xi(24) = 563.687833666227591566943999623618D+00
      Xi(25) = 743.290311601880343383985660921098D+00
      Xi(26) = 879.508623151254246841368455989141D+00
      GOTO 9999
C
C     ----- K = 13 , R = 1E3 -----
C
 1331 CONTINUE
C
      RIni = 1.0D+03
C
      Omega( 1) = 0.0017635661455619365334288679760671447738D+00
      Omega( 2) = 0.0047061107563201549891152464417942713482D+00
      Omega( 3) = 0.0096001161973894923613035215503863639697D+00
      Omega( 4) = 0.0188839158491304791702029624786418615656D+00
      Omega( 5) = 0.0365840348403184362808691822838147444941D+00
      Omega( 6) = 0.0693591882562437035396351626970190551447D+00
      Omega( 7) = 0.1284376651802928805053442481987069356819D+00
      Omega( 8) = 0.2326290168465691475616768935097411485913D+00
      Omega( 9) = 0.4131949440169984410803783597687655060327D+00
      Omega(10) = 0.7226149903814321418991060974601481348145D+00
      Omega(11) = 1.2545308979027300954126045162695390899898D+00
      Omega(12) = 2.2113022645733460638285683419468341526226D+00
      Omega(13) = 4.3156577180022858523933404484296261216514D+00
C
      Alpha( 1) = 0.0006768696090223793169950347299677201329D+00
      Alpha( 2) = 0.0038174398927510270668774617448468222669D+00
      Alpha( 3) = 0.0107265164258034905514282509111922969680D+00
      Alpha( 4) = 0.0244642995291056325264016178128168021999D+00
      Alpha( 5) = 0.0512717184013090078658432448971415951178D+00
      Alpha( 6) = 0.1026078910917859007657991325668245252700D+00
      Alpha( 7) = 0.1986759226082185785823824158979888920840D+00
      Alpha( 8) = 0.3743869444419926817869005014616234916502D+00
      Alpha( 9) = 0.6891677591849794330789054364050372214479D+00
      Alpha(10) = 1.2433126215196539637170003445021393417846D+00
      Alpha(11) = 2.2075757403418671217555646624219889417873D+00
      Alpha(12) = 3.8903332120321473751832497267599819679162D+00
      Alpha(13) = 6.9835030321592778258625588971852948816377D+00
C
      Xi( 1) = 1.008859011660317063041429619386D+00
      Xi( 2) = 1.080796847166073275554876165216D+00
      Xi( 3) = 1.230402093767569798930631985723D+00
      Xi( 4) = 1.469453070518195829537687335797D+00
      Xi( 5) = 1.816762394796934359185967833294D+00
      Xi( 6) = 2.300186340854779247673544961827D+00
      Xi( 7) = 2.959797888000143212565082317944D+00
      Xi( 8) = 3.852322283237996409458597790731D+00
      Xi( 9) = 5.057300123566479988900662645435D+00
      Xi(10) = 6.685953173104684348244941549666D+00
      Xi(11) = 8.894077959709410160192566419113D+00
      Xi(12) = 11.900838381148493603373106353160D+00
      Xi(13) = 16.016291320121754701566452183670D+00
      Xi(14) = 21.681943816052676662908815785613D+00
      Xi(15) = 29.530984248548985635421382234256D+00
      Xi(16) = 40.478264345147939216873478329717D+00
      Xi(17) = 55.855439018915906790319070296391D+00
      Xi(18) = 77.613940237170021503587680200553D+00
      Xi(19) = 108.626801119276814200365066653831D+00
      Xi(20) = 153.121296855100346712141856642120D+00
      Xi(21) = 217.235117672761294776995200095371D+00
      Xi(22) = 309.479592371193537730933442730930D+00
      Xi(23) = 440.081918394029455110505466564064D+00
      Xi(24) = 615.537513789520825013834581795891D+00
      Xi(25) = 819.234884579074029875567219960431D+00
      Xi(26) = 976.190808663600207428778787743795D+00
      GOTO 9999
C
C     ----- K = 13 , R = 2E3 -----
C
 1332 CONTINUE
C
      RIni = 2.0D+03
C
      Omega( 1) = 0.0010067371248319556046560592499350406115D+00
      Omega( 2) = 0.0028099546056502035573047821150408087654D+00
      Omega( 3) = 0.0061880278742393199293739909291889311760D+00
      Omega( 4) = 0.0131638091493266705629816147767041645267D+00
      Omega( 5) = 0.0272016646696397412172476566391554619884D+00
      Omega( 6) = 0.0543757528682417203673980007211374854137D+00
      Omega( 7) = 0.1053434646459840612667778481092994979917D+00
      Omega( 8) = 0.1984814477535834189621125403513168805603D+00
      Omega( 9) = 0.3650556257206909886226832429034772076193D+00
      Omega(10) = 0.6584198791878357653255086912036375679236D+00
      Omega(11) = 1.1742485462233790930331595037117153879080D+00
      Omega(12) = 2.1173756701945022221447723076437341660494D+00
      Omega(13) = 4.2088436157456099954293771148172709217761D+00
C
      Alpha( 1) = 0.0003845271473448446583365740753402556296D+00
      Alpha( 2) = 0.0022186008736336012455931992939234254081D+00
      Alpha( 3) = 0.0065187824835520443497958998062963287268D+00
      Alpha( 4) = 0.0157790140744405842979894601108004081880D+00
      Alpha( 5) = 0.0351723648243899866533271943852545504683D+00
      Alpha( 6) = 0.0745116209124147938107417081521344215389D+00
      Alpha( 7) = 0.1517718419174551602483314374780754008043D+00
      Alpha( 8) = 0.2991167658475697826970206294472376384874D+00
      Alpha( 9) = 0.5729793346936465232291033067291152747202D+00
      Alpha(10) = 1.0710618053186080256191509141316942077538D+00
      Alpha(11) = 1.9629781296839430280477822865492498749518D+00
      Alpha(12) = 3.5583604501642892266669027101499978016363D+00
      Alpha(13) = 6.5515589465778985072894058383496940223267D+00
C
      Xi( 1) = 1.009725744646025334498584680176D+00
      Xi( 2) = 1.088827885747662834402781084631D+00
      Xi( 3) = 1.254003513448803964580417769081D+00
      Xi( 4) = 1.519624546493318705529286527511D+00
      Xi( 5) = 1.908828870204436245743701006106D+00
      Xi( 6) = 2.456323651603291157536831934571D+00
      Xi( 7) = 3.212819016710725915581328693271D+00
      Xi( 8) = 4.251287847349923359335788219937D+00
      Xi( 9) = 5.675912111606291277510899506709D+00
      Xi(10) = 7.635320513387641534232275208893D+00
      Xi(11) = 10.342348180933737219686086028503D+00
      Xi(12) = 14.103626293658320066393141445360D+00
      Xi(13) = 19.364251300863817621678020231002D+00
      Xi(14) = 26.775834048268365186057371474249D+00
      Xi(15) = 37.301270145058441431856754988416D+00
      Xi(16) = 52.377800508079400344102483089159D+00
      Xi(17) = 74.173699586342912999648024907628D+00
      Xi(18) = 105.996097954768730919905550535987D+00
      Xi(19) = 152.943427648277461672332577791167D+00
      Xi(20) = 222.944161912956156118559292167447D+00
      Xi(21) = 328.359385158048549313569708374416D+00
      Xi(22) = 488.176055827352966737242567774047D+00
      Xi(23) = 729.657320211492764039373781770337D+00
      Xi(24) = 1082.705180999423388521485378532816D+00
      Xi(25) = 1539.941298467155270324191462805175D+00
      Xi(26) = 1935.888468434480756519278088489955D+00
      GOTO 9999
C
C     ----- K = 13 , R = 3E3 -----
C
 1333 CONTINUE
C
      RIni = 3.0D+03
C
      Omega( 1) = 0.0007266173762481528841575934788560298117D+00
      Omega( 2) = 0.0020964974323044222425239772877869626200D+00
      Omega( 3) = 0.0048581767156706806801208444347023318599D+00
      Omega( 4) = 0.0108225994274088841993870465686689019691D+00
      Omega( 5) = 0.0231860545777698102652144510754661510532D+00
      Omega( 6) = 0.0477278172991908051166795416697663512196D+00
      Omega( 7) = 0.0947879329870736345213739007520281631969D+00
      Omega( 8) = 0.1824733193047515683886308085481786633864D+00
      Omega( 9) = 0.3419906840673959335069175685672959730255D+00
      Omega(10) = 0.6270943201002195372512591775571166863301D+00
      Omega(11) = 1.1345214719922042758727676048380317297415D+00
      Omega(12) = 2.0705197705712899576818014635293252467818D+00
      Omega(13) = 4.1555506786128121082986297540884379486670D+00
C
      Alpha( 1) = 0.0002765305534557809014453940609423385211D+00
      Alpha( 2) = 0.0016230480146192057838232835936982667580D+00
      Alpha( 3) = 0.0049241941747125575466193772491280133963D+00
      Alpha( 4) = 0.0123926268756996454932121443440196095764D+00
      Alpha( 5) = 0.0286746047845924961946768947523844772718D+00
      Alpha( 6) = 0.0627750595034835794506881956333899097444D+00
      Alpha( 7) = 0.1315527145078088977931144928912310376745D+00
      Alpha( 8) = 0.2657519373445610193456949132961852910739D+00
      Alpha( 9) = 0.5202041093762642991671016345911482403608D+00
      Alpha(10) = 0.9911579072113548517721122455004234552689D+00
      Alpha(11) = 1.8474989877676289366098538735627698770259D+00
      Alpha(12) = 3.3994398160829219373863657738255028561980D+00
      Alpha(13) = 6.3427254429993991389005736358797093998874D+00
C
      Xi( 1) = 1.010193884925419554091617635461D+00
      Xi( 2) = 1.093175713659357331281225811370D+00
      Xi( 3) = 1.266836191359841957363580045026D+00
      Xi( 4) = 1.547068851020137687183458874429D+00
      Xi( 5) = 1.959573740784946239198299755557D+00
      Xi( 6) = 2.543178477080369750832136710450D+00
      Xi( 7) = 3.355089312592423162655633217533D+00
      Xi( 8) = 4.478360657916579081611052615131D+00
      Xi( 9) = 6.032735215475828557344512015348D+00
      Xi(10) = 8.190928315066882576221674039374D+00
      Xi(11) = 11.203262287884983346847311835859D+00
      Xi(12) = 15.435129731786143888333517892875D+00
      Xi(13) = 21.424480376846679417074392759446D+00
      Xi(14) = 29.970988825066483688189244993794D+00
      Xi(15) = 42.276101777386887150406158397686D+00
      Xi(16) = 60.165807576485730891191483493685D+00
      Xi(17) = 86.450085911160924119489568084873D+00
      Xi(18) = 125.510614369933223281872169962980D+00
      Xi(19) = 184.274284425331046091689479737852D+00
      Xi(20) = 273.836327285173797618789137686690D+00
      Xi(21) = 412.142291932858564174058102480558D+00
      Xi(22) = 628.174180857900458452292014044360D+00
      Xi(23) = 966.981853512719977949974037301217D+00
      Xi(24) = 1487.459781101271937719943139200041D+00
      Xi(25) = 2208.800660325778965731302605490782D+00
      Xi(26) = 2885.097496793069043574675447416666D+00
      GOTO 9999
C
C     ----- K = 13 , R = 4E3 -----
C
 1334 CONTINUE
C
      RIni = 4.0D+03
C
      Omega( 1) = 0.0005772981384290036624670431752038429751D+00
      Omega( 2) = 0.0017117321655301405442927299010991148975D+00
      Omega( 3) = 0.0041224097445417860749515386042940434663D+00
      Omega( 4) = 0.0094856813604299347655451077679367166695D+00
      Omega( 5) = 0.0208318985608728884816644990815581728327D+00
      Omega( 6) = 0.0437488442607272109449820674670350584279D+00
      Omega( 7) = 0.0883611409328008595820522393105012071146D+00
      Omega( 8) = 0.1725833026893584265797250421048403268287D+00
      Omega( 9) = 0.3275606215791396407324437972663488949365D+00
      Omega(10) = 0.6072881207626289627515489555342753646983D+00
      Omega(11) = 1.1091987143511127322055218047403002401552D+00
      Omega(12) = 2.0405109967818031381416726821598217611609D+00
      Omega(13) = 4.1214134076962477036890075687836088036420D+00
C
      Alpha( 1) = 0.0002190369434955462190035078753869678125D+00
      Alpha( 2) = 0.0013041638214468583454602062556004771210D+00
      Alpha( 3) = 0.0040594961584673383144200413397140181360D+00
      Alpha( 4) = 0.0105190663597531257523630052186369709943D+00
      Alpha( 5) = 0.0249976696275790752605172303016378698715D+00
      Alpha( 6) = 0.0559909911522194053871775108743857884974D+00
      Alpha( 7) = 0.1196433236803296841193190551411174027407D+00
      Alpha( 8) = 0.2457744585688548566063766839806525865697D+00
      Alpha( 9) = 0.4881524088078709740592432686501211946961D+00
      Alpha(10) = 0.9420361988247535020048195608666219413863D+00
      Alpha(11) = 1.7757837240382841738437630363378616493719D+00
      Alpha(12) = 3.2999574598283268581262739349568846591865D+00
      Alpha(13) = 6.2112597218473327084775625905876950128004D+00
C
      Xi( 1) = 1.010507171655304307824468912358D+00
      Xi( 2) = 1.096089317567576409528234959723D+00
      Xi( 3) = 1.275457771083344649999148645581D+00
      Xi( 4) = 1.565572209208788321383419572808D+00
      Xi( 5) = 1.993939949609717315263772430267D+00
      Xi( 6) = 2.602319097392235636677371157788D+00
      Xi( 7) = 3.452578553195592656619231086701D+00
      Xi( 8) = 4.635076890885197571732373456044D+00
      Xi( 9) = 6.280951878486133464589424857394D+00
      Xi(10) = 8.580757164513207110694004153828D+00
      Xi(11) = 11.812928826767320134721872548766D+00
      Xi(12) = 16.387505908343608500635224523023D+00
      Xi(13) = 22.913986916926745118078256879812D+00
      Xi(14) = 32.307870360351605198978575117508D+00
      Xi(15) = 45.960257704975180259587830278178D+00
      Xi(16) = 66.011654195450042745263452559357D+00
      Xi(17) = 95.801090043883571842342483648736D+00
      Xi(18) = 140.614788668700952972701045950998D+00
      Xi(19) = 208.954038513489973927228504635423D+00
      Xi(20) = 314.711822968892776908322872486679D+00
      Xi(21) = 480.915486201436502888828528057275D+00
      Xi(22) = 745.985517155986802229072907266527D+00
      Xi(23) = 1172.664620532656124862747049064637D+00
      Xi(24) = 1851.445761013825762719697820557485D+00
      Xi(25) = 2839.909796921835096306097057095030D+00
      Xi(26) = 3825.735247723769567373963695899874D+00
      GOTO 9999
C
C     ----- K = 13 , R = 5E3 -----
C
 1335 CONTINUE
C
      RIni = 5.0D+03
C
      Omega( 1) = 0.0004834183698374707752262757922247574105D+00
      Omega( 2) = 0.0014675492248497519941626034676619705355D+00
      Omega( 3) = 0.0036457480254511605207782252470460893079D+00
      Omega( 4) = 0.0085991534896893102134834345300073987772D+00
      Omega( 5) = 0.0192417810195062726691189129019088355221D+00
      Omega( 6) = 0.0410222078190040722932101485004308116800D+00
      Omega( 7) = 0.0839043742330625100714195686857355127586D+00
      Omega( 8) = 0.1656549505050655346304446305372159997660D+00
      Omega( 9) = 0.3173632327682835929877694369771745641629D+00
      Omega(10) = 0.5931886289713927987946734399216097699536D+00
      Omega(11) = 1.0910702058875274777192954100968336206279D+00
      Omega(12) = 2.0189563667345757479870682615796795289498D+00
      Omega(13) = 4.0968895055939679610869375014914339772076D+00
C
      Alpha( 1) = 0.0001829255424868854810194156694383785244D+00
      Alpha( 2) = 0.0011029602809106096935085972524093200775D+00
      Alpha( 3) = 0.0035082040776387238560814207932736508155D+00
      Alpha( 4) = 0.0093055582700188832209547366153268299627D+00
      Alpha( 5) = 0.0225756890794564490266432572190824501490D+00
      Alpha( 6) = 0.0514532208102668484849516201268260218171D+00
      Alpha( 7) = 0.1115701158044498115593629559982868926227D+00
      Alpha( 8) = 0.2320750581708474017403678879567507920001D+00
      Alpha( 9) = 0.4659542555149446310105640356447054273303D+00
      Alpha(10) = 0.9077269511701410156984982935401973236367D+00
      Alpha(11) = 1.7253408354432977452660308292919921768771D+00
      Alpha(12) = 3.2295976939344447805002591422862678882666D+00
      Alpha(13) = 6.1179182378765278489674006578979970072396D+00
C
      Xi( 1) = 1.010738974857806489449606190956D+00
      Xi( 2) = 1.098247200440898018098702382250D+00
      Xi( 3) = 1.281854567869962779197434699086D+00
      Xi( 4) = 1.579334652848051773958541754705D+00
      Xi( 5) = 2.019580321691430035979664125989D+00
      Xi( 6) = 2.646610736278654934594892211841D+00
      Xi( 7) = 3.525915057422893787502857221305D+00
      Xi( 8) = 4.753560317982870825001540104893D+00
      Xi( 9) = 6.469652372559513124877089706999D+00
      Xi(10) = 8.878906761062757563816860706396D+00
      Xi(11) = 12.282266232036800817327415469293D+00
      Xi(12) = 17.125848188178688999080967469979D+00
      Xi(13) = 24.077550125482101208915231005392D+00
      Xi(14) = 34.148424906016304918032755466584D+00
      Xi(15) = 48.887862478507531315691236883936D+00
      Xi(16) = 70.702197987010334083446405628592D+00
      Xi(17) = 103.383810322159083130988399545913D+00
      Xi(18) = 153.005728623073744595894929432234D+00
      Xi(19) = 229.461785443632748524023234892866D+00
      Xi(20) = 349.166407012105063090157131000524D+00
      Xi(21) = 539.826624204983897770127043713728D+00
      Xi(22) = 848.788293460028481829393598445677D+00
      Xi(23) = 1356.143188866657152247263695699075D+00
      Xi(24) = 2185.311920916363763067735703771177D+00
      Xi(25) = 3440.872267032228991823217256751377D+00
      Xi(26) = 4758.866855376139633815313345621689D+00
      GOTO 9999
C
C     ----- K = 13 , R = 6E3 -----
C
 1336 CONTINUE
C
      RIni = 6.0D+03
C
      Omega( 1) = 0.0004184727948542464259221414018489332287D+00
      Omega( 2) = 0.0012972732263054492916603540686795725811D+00
      Omega( 3) = 0.0033075235899535997919634227638907564284D+00
      Omega( 4) = 0.0079584618577688491727290605488864816408D+00
      Omega( 5) = 0.0180763356026030094760514380786142751845D+00
      Omega( 6) = 0.0390017527131794468388801579161739851997D+00
      Omega( 7) = 0.0805718846011344139821795835854079115279D+00
      Omega( 8) = 0.1604343330103549718852664479618042037146D+00
      Omega( 9) = 0.3096284138665091020304904328996897788784D+00
      Omega(10) = 0.5824345405064120297111175672544192138957D+00
      Omega(11) = 1.0771838414761236184984463659652931255550D+00
      Omega(12) = 2.0024038613880578551390554409294963988941D+00
      Omega(13) = 4.0780542604105885617221671779475400398951D+00
C
      Alpha( 1) = 0.0001579639475325221557856034346471914287D+00
      Alpha( 2) = 0.0009633488427175769010574386298773053028D+00
      Alpha( 3) = 0.0031222338920011592445869299822930331700D+00
      Alpha( 4) = 0.0084447683196101985288653372398182117564D+00
      Alpha( 5) = 0.0208344862398892867293701729528021537874D+00
      Alpha( 6) = 0.0481516957484454969102964129457067343765D+00
      Alpha( 7) = 0.1056356213489921477369248327404793741380D+00
      Alpha( 8) = 0.2219158164417327056515405470094037809758D+00
      Alpha( 9) = 0.4493679596948363069274911402173344754374D+00
      Alpha(10) = 0.8819265141437007679081974687029799042648D+00
      Alpha(11) = 1.6872058594615989730869376472810472478159D+00
      Alpha(12) = 3.1761840167015348132453084195603310035949D+00
      Alpha(13) = 6.0468509031626904615293327527325573100825D+00
C
      Xi( 1) = 1.010920946649155907691268463378D+00
      Xi( 2) = 1.099942443889555764068369225761D+00
      Xi( 3) = 1.286886753704524820416527086664D+00
      Xi( 4) = 1.590181497923492496236741455817D+00
      Xi( 5) = 2.039836727934589126953399129683D+00
      Xi( 6) = 2.681703103067590013806886783954D+00
      Xi( 7) = 3.584216057170782720412566257906D+00
      Xi( 8) = 4.848110980743221974827161790955D+00
      Xi( 9) = 6.620874502625305116267767635563D+00
      Xi(10) = 9.118938881467079838986922180055D+00
      Xi(11) = 12.662000733272805380776049322122D+00
      Xi(12) = 17.726453775662492189091334893192D+00
      Xi(13) = 25.029577491848180788650002348561D+00
      Xi(14) = 35.663907097702922704279826504603D+00
      Xi(15) = 51.315002423162425583547108232096D+00
      Xi(16) = 74.620138063550451049121825519705D+00
      Xi(17) = 109.769787631884041319285660875948D+00
      Xi(18) = 163.536101042984578043948395986718D+00
      Xi(19) = 247.066473507943357573735099208534D+00
      Xi(20) = 379.078988115023749611021131045163D+00
      Xi(21) = 591.629447221732486350553870124713D+00
      Xi(22) = 940.530548868877992685355593494023D+00
      Xi(23) = 1522.796122253376686295212039112812D+00
      Xi(24) = 2495.439757517349457005551016663958D+00
      Xi(25) = 4016.574435451630723736826666936395D+00
      Xi(26) = 5685.191791248119010493411451534485D+00
      GOTO 9999
C
C     ----- K = 13 , R = 7E3 -----
C
 1337 CONTINUE
C
      RIni = 7.0D+03
C
      Omega( 1) = 0.0003706377943815102771234427431927896390D+00
      Omega( 2) = 0.0011709747407894105638609296825974892542D+00
      Omega( 3) = 0.0030528344477778092929221660702804319598D+00
      Omega( 4) = 0.0074687037676975781414815910332072323996D+00
      Omega( 5) = 0.0171753370381771652815008260065110512471D+00
      Omega( 6) = 0.0374259815736839124110510099163739994310D+00
      Omega( 7) = 0.0779539684179911716450906387920749551768D+00
      Omega( 8) = 0.1563078510967816640077907366923604115527D+00
      Omega( 9) = 0.3034823620318470197650042446957385777750D+00
      Omega(10) = 0.5738515410841367624301435912403945849292D+00
      Omega(11) = 1.0660630760791806437943682328572947426437D+00
      Omega(12) = 1.9891211408960819676710121695073496539408D+00
      Omega(13) = 4.0629379599812607197321934204836679782602D+00
C
      Alpha( 1) = 0.0001395911103949207554129102029963870046D+00
      Alpha( 2) = 0.0008602478231696831677067798698540362912D+00
      Alpha( 3) = 0.0028349335336120280063744556083016323811D+00
      Alpha( 4) = 0.0077968208961486511058163503384149661812D+00
      Alpha( 5) = 0.0195091641305399062398872678594785590178D+00
      Alpha( 6) = 0.0456141150523087043678921882056132020011D+00
      Alpha( 7) = 0.1010362835063562390359743342616027916847D+00
      Alpha( 8) = 0.2139863759050597113292851297883245820231D+00
      Alpha( 9) = 0.4363438111546551760502443001454864202060D+00
      Alpha(10) = 0.8615632167206826681543282986375231757847D+00
      Alpha(11) = 1.6569797098202836513770180859594916000788D+00
      Alpha(12) = 3.1337076649686691487417533608095254749060D+00
      Alpha(13) = 5.9902047769526498423135751902890433484572D+00
C
      Xi( 1) = 1.011069519030782586544066348200D+00
      Xi( 2) = 1.101327355506823330727472765034D+00
      Xi( 3) = 1.291002214027701274585632762459D+00
      Xi( 4) = 1.599065635925675331612157148431D+00
      Xi( 5) = 2.056459251051908898632117339567D+00
      Xi( 6) = 2.710566638540833956814643523892D+00
      Xi( 7) = 3.632298569107429986758256434953D+00
      Xi( 8) = 4.926328628237788356120097077451D+00
      Xi( 9) = 6.746394341811706720727498387546D+00
      Xi(10) = 9.318910594218486549526747619865D+00
      Xi(11) = 12.979626574112289267258490976786D+00
      Xi(12) = 18.231006536335286163583169205538D+00
      Xi(13) = 25.833110196498794945349497531595D+00
      Xi(14) = 36.949525543467812566111874161834D+00
      Xi(15) = 53.385468217286557508771682734050D+00
      Xi(16) = 77.982689877503673842040754493610D+00
      Xi(17) = 115.287289250844873882140273479990D+00
      Xi(18) = 172.701987284767743080493573870626D+00
      Xi(19) = 262.517016506791535490616951165066D+00
      Xi(20) = 405.576384631176343198433187353658D+00
      Xi(21) = 638.005573073923045068767123666476D+00
      Xi(22) = 1023.675579114166468042146362904532D+00
      Xi(23) = 1676.071938308362667457984684915573D+00
      Xi(24) = 2786.083260147426660102709661259723D+00
      Xi(25) = 4570.464376150974832757611920897034D+00
      Xi(26) = 6605.212628370586382064999497742974D+00
      GOTO 9999
C
C     ----- K = 13 , R = 8E3 -----
C
 1338 CONTINUE
C
      RIni = 8.0D+03
C
      Omega( 1) = 0.0003338078810012542241301752767718147297D+00
      Omega( 2) = 0.0010731162787732948328650156013738814487D+00
      Omega( 3) = 0.0028528419239137359014562672328632619845D+00
      Omega( 4) = 0.0070792216098078710114682905469446483337D+00
      Omega( 5) = 0.0164520713096193200454373959870646437764D+00
      Omega( 6) = 0.0361518092567141136475328353633296174507D+00
      Omega( 7) = 0.0758243845754235437842788979112551572825D+00
      Omega( 8) = 0.1529339780982985849710521775168814428980D+00
      Omega( 9) = 0.2984353231132846150487972214371268364630D+00
      Omega(10) = 0.5667775223001799543716926199632410998674D+00
      Omega(11) = 1.0568715828439863723851518950702654819906D+00
      Omega(12) = 1.9781243638999431445598189971413205512363D+00
      Omega(13) = 4.0504218514964760687602218780511975637637D+00
C
      Alpha( 1) = 0.0001254533213413407806652386661250275779D+00
      Alpha( 2) = 0.0007806785450082557440315181957954382419D+00
      Alpha( 3) = 0.0026116176695613647849582669062742934507D+00
      Alpha( 4) = 0.0072882286878179625655993782891439725091D+00
      Alpha( 5) = 0.0184589798304450120202980243147572458895D+00
      Alpha( 6) = 0.0435867985980898777018804901572934795695D+00
      Alpha( 7) = 0.0973362507445847350491291601060073190865D+00
      Alpha( 8) = 0.2075698167314758098343872166902102094355D+00
      Alpha( 9) = 0.4257518170980607442161446074546660156557D+00
      Alpha(10) = 0.8449324268358560348162432374063968154587D+00
      Alpha(11) = 1.6322074462041672078278092827829937050410D+00
      Alpha(12) = 3.0988005415161087707093257037627154204529D+00
      Alpha(13) = 5.9435640977043874238865128756259537112783D+00
C
      Xi( 1) = 1.011194271189963009511050828770D+00
      Xi( 2) = 1.102490805240993707340632634661D+00
      Xi( 3) = 1.294462592735715565834576923354D+00
      Xi( 4) = 1.606544960882145392944836403970D+00
      Xi( 5) = 2.070475367190860802339355539736D+00
      Xi( 6) = 2.734950809647740910864749386278D+00
      Xi( 7) = 3.673009910906543813149602306645D+00
      Xi( 8) = 4.992723021119104396658289202193D+00
      Xi( 9) = 6.853239939403569421123851856947D+00
      Xi(10) = 9.489649798672527634089968184838D+00
      Xi(11) = 13.251721282740746303031864616884D+00
      Xi(12) = 18.664785660215381921100963324989D+00
      Xi(13) = 26.526631345696071123327342000664D+00
      Xi(14) = 38.063867020101055941433410012564D+00
      Xi(15) = 55.188416375678922345948684835548D+00
      Xi(16) = 80.925743847941144552826653324473D+00
      Xi(17) = 120.143624167390607002103841693952D+00
      Xi(18) = 180.819948808945976589934723222086D+00
      Xi(19) = 276.296939379611788206725364602789D+00
      Xi(20) = 429.395624268682294860655090928958D+00
      Xi(21) = 680.071425164502705251745595660395D+00
      Xi(22) = 1099.888095936666530638703420663660D+00
      Xi(23) = 1818.354039085222022542609465745045D+00
      Xi(24) = 3060.280942425657703997998737577291D+00
      Xi(25) = 5105.124648034188095113705685434979D+00
      Xi(26) = 7519.310923709670903658519591772347D+00
      GOTO 9999
C
C     ----- K = 13 , R = 9E3 -----
C
 1339 CONTINUE
C
      RIni = 9.0D+03
C
      Omega( 1) = 0.0003044979986515639643271668334694249336D+00
      Omega( 2) = 0.0009947872650128499371167273019472832019D+00
      Omega( 3) = 0.0026908314022329150792039797546531532113D+00
      Omega( 4) = 0.0067602481509608796317302924997610924507D+00
      Omega( 5) = 0.0158549983554854806457170895636621388292D+00
      Omega( 6) = 0.0350934163362851217193838764679680863878D+00
      Omega( 7) = 0.0740464043548204659236423343104949168492D+00
      Omega( 8) = 0.1501049592454127755924118969854674787712D+00
      Omega( 9) = 0.2941877084622041017272428553575025489408D+00
      Omega(10) = 0.5608055691529981708954612218409607748981D+00
      Omega(11) = 1.0490934946574624696659719380598119187198D+00
      Omega(12) = 1.9688053441501122312702262706629596777930D+00
      Omega(13) = 4.0398142712789524011451902651970158331096D+00
C
      Alpha( 1) = 0.0001142078282379464811224502862054778346D+00
      Alpha( 2) = 0.0007172191780088991689513650538458089390D+00
      Alpha( 3) = 0.0024323501465652189386784643148481377395D+00
      Alpha( 4) = 0.0068763947394231850905671140933989038402D+00
      Alpha( 5) = 0.0176015596004510661499911205324364615876D+00
      Alpha( 6) = 0.0419199056873468953450891961998170387460D+00
      Alpha( 7) = 0.0942759616337734391531286495813368730978D+00
      Alpha( 8) = 0.2022361057922115201056700514725683603956D+00
      Alpha( 9) = 0.4169099038558705398640601841320219023146D+00
      Alpha(10) = 0.8309996676277631378801889761032128944862D+00
      Alpha(11) = 1.6113925491144701489546961492571597318602D+00
      Alpha(12) = 3.0694021532196554213033079516748102832935D+00
      Alpha(13) = 5.9042205649169941920374748978161960621946D+00
C
      Xi( 1) = 1.011301249609793416798196596940D+00
      Xi( 2) = 1.103488913440093922109161428402D+00
      Xi( 3) = 1.297433455503520915194463902775D+00
      Xi( 4) = 1.612973024208289563020089774081D+00
      Xi( 5) = 2.082537319964158808397597377748D+00
      Xi( 6) = 2.755969232365370307673638206047D+00
      Xi( 7) = 3.708168547919199563040609790576D+00
      Xi( 8) = 5.050184365461531847338316758567D+00
      Xi( 9) = 6.945930525094032291413131963154D+00
      Xi(10) = 9.638154665146328028101085916290D+00
      Xi(11) = 13.489050539430503977342268306927D+00
      Xi(12) = 19.044302644665340680518728078141D+00
      Xi(13) = 27.135426454652572650361186390455D+00
      Xi(14) = 39.045608645104961920840880296169D+00
      Xi(15) = 56.783157011132844863704471904953D+00
      Xi(16) = 83.540273286663584033362450753657D+00
      Xi(17) = 124.478716923536892970114653422797D+00
      Xi(18) = 188.105625063055856172056223840627D+00
      Xi(19) = 288.738723310025514223742248987037D+00
      Xi(20) = 451.049222218557127439542142610662D+00
      Xi(21) = 718.613432280905455484809607469288D+00
      Xi(22) = 1170.358028656877776629841036992730D+00
      Xi(23) = 1951.379160595497792130004199862015D+00
      Xi(24) = 3320.304950255201992259657117756433D+00
      Xi(25) = 5622.572944123117865355965250273584D+00
      Xi(26) = 8427.788308099427949215964872564655D+00
      GOTO 9999
C
C     ----- K = 13 , R = 1E4 -----
C
 1341 CONTINUE
C
      RIni = 1.0D+04
C
      Omega( 1) = 0.0002805676165532984518155677907628314793D+00
      Omega( 2) = 0.0009304915861452454389846289861329409199D+00
      Omega( 3) = 0.0025563936765912246068727965708777816189D+00
      Omega( 4) = 0.0064930225459359853644045396710987594346D+00
      Omega( 5) = 0.0153513195022753170445123046573832059636D+00
      Omega( 6) = 0.0341957745875822269461215881930504778552D+00
      Omega( 7) = 0.0725318011194240101511339531559219295787D+00
      Omega( 8) = 0.1476859942928149881200428766847565498210D+00
      Omega( 9) = 0.2905441729112577614005819698350663315978D+00
      Omega(10) = 0.5556692375929922942615762915252730635984D+00
      Omega(11) = 1.0423899529883427146929528173124879231182D+00
      Omega(12) = 1.9607638827744524425474720308315568217949D+00
      Omega(13) = 4.0306601210273864796182485825681851565605D+00
C
      Alpha( 1) = 0.0001050303259562613398418802709865550737D+00
      Alpha( 2) = 0.0006653037355264198210729591502401308878D+00
      Alpha( 3) = 0.0022848087471415481963090890601617011235D+00
      Alpha( 4) = 0.0065347902968938500829803239441928575104D+00
      Alpha( 5) = 0.0168851672754504879700796068572876684755D+00
      Alpha( 6) = 0.0405185835722869734366836588779214878286D+00
      Alpha( 7) = 0.0916899642061235559744062847964407581003D+00
      Alpha( 8) = 0.1977094597844340329815268977875497569130D+00
      Alpha( 9) = 0.4093783172749800212085469208744825664326D+00
      Alpha(10) = 0.8190948779971189729672692569906189419271D+00
      Alpha(11) = 1.5935618858617478106045814434388319114078D+00
      Alpha(12) = 3.0441685261321626066507300967245441825071D+00
      Alpha(13) = 5.8704038773647556391857560420532990974607D+00
C
      Xi( 1) = 1.011394502750584783925397180937D+00
      Xi( 2) = 1.104359278856419517966108712059D+00
      Xi( 3) = 1.300025824648590792929729598182D+00
      Xi( 4) = 1.618587101892998377492861872273D+00
      Xi( 5) = 2.093083986917413600670548334826D+00
      Xi( 6) = 2.774373089345218839317508408016D+00
      Xi( 7) = 3.739004345991395365962628449807D+00
      Xi( 8) = 5.100674537215651497233659394936D+00
      Xi( 9) = 7.027542888986806526429446106974D+00
      Xi(10) = 9.769204956923797807805120996250D+00
      Xi(11) = 13.698998182122122726882273990867D+00
      Xi(12) = 19.380928351820645237635343249849D+00
      Xi(13) = 27.676970013468871121414816016681D+00
      Xi(14) = 39.921665490508533677416203033772D+00
      Xi(15) = 58.211130279764656602486949665831D+00
      Xi(16) = 85.890337944122038482674952319940D+00
      Xi(17) = 128.391748565267824450097933208781D+00
      Xi(18) = 194.712981765983032325650903260339D+00
      Xi(19) = 300.082023274835592047127263981565D+00
      Xi(20) = 470.910027685282373155484236804114D+00
      Xi(21) = 754.209698745703329469947107099870D+00
      Xi(22) = 1235.971801574550370084182304708520D+00
      Xi(23) = 2076.462566934585095124177200887061D+00
      Xi(24) = 3567.919536174050756160980313325126D+00
      Xi(25) = 6124.428667863334515342188524300582D+00
      Xi(26) = 9330.891138495153310117302680737339D+00
      GOTO 9999
C
C     ----- K = 13 , R = 4E5 (Best : R = 373800) -----
C
 1399 CONTINUE
C
      RIni = 4.0D+05
C
      Omega( 1) = 0.0000330785848121602154387056886249976126D+00
      Omega( 2) = 0.0002173321916008653986926517817884589590D+00
      Omega( 3) = 0.0008896788976872138107598869801936891655D+00
      Omega( 4) = 0.0028921756053065667819714998094254854166D+00
      Omega( 5) = 0.0081422349627770869181416228850534100303D+00
      Omega( 6) = 0.0207224876586965230132556534461962804983D+00
      Omega( 7) = 0.0488864342146971603968207738949436347298D+00
      Omega( 8) = 0.1086418495611125869620928671710480273305D+00
      Omega( 9) = 0.2300332300492383233344846973911224097264D+00
      Omega(10) = 0.4682950733039983514044771184536841701629D+00
      Omega(11) = 0.9262055359666433552523825967206505538343D+00
      Omega(12) = 1.8198087510110939301862312267310528568487D+00
      Omega(13) = 3.8700490617962436596440761737980551515648D+00
C
      Alpha( 1) = 0.0000103473916611944270948840590255708282D+00
      Alpha( 2) = 0.0001138316914487578993839338792673759188D+00
      Alpha( 3) = 0.0006024601498553105686272644218436783659D+00
      Alpha( 4) = 0.0023252777266436541401638786073433062818D+00
      Alpha( 5) = 0.0074464345169693261373645629380008115206D+00
      Alpha( 6) = 0.0210088043449285925962484672738228574929D+00
      Alpha( 7) = 0.0540032192609735148525727538998397747605D+00
      Alpha( 8) = 0.1291621153315104471087301774012701116590D+00
      Alpha( 9) = 0.2915643750527972055479851432036042524487D+00
      Alpha(10) = 0.6276946181536831419200961335036481614225D+00
      Alpha(11) = 1.3003150067055842798151993333632958638191D+00
      Alpha(12) = 2.6217712688276107583541796319792638314539D+00
      Alpha(13) = 5.2973357440769386740922475276249770104187D+00
C
      Xi( 1) = 1.013181302065140132755323432168D+00
      Xi( 2) = 1.121092899838553857147852022003D+00
      Xi( 3) = 1.350178171338921877860789200199D+00
      Xi( 4) = 1.728133062090680845384466080006D+00
      Xi( 5) = 2.301135413611220813115756933520D+00
      Xi( 6) = 3.142313650650617387529578428129D+00
      Xi( 7) = 4.365283442146903240734662210443D+00
      Xi( 8) = 6.144672947757623488775069464651D+00
      Xi( 9) = 8.749086754792229351194998709929D+00
      Xi(10) = 12.595210582443065289517947213938D+00
      Xi(11) = 18.337551132071687531172043961192D+00
      Xi(12) = 27.019638911679229103732224892553D+00
      Xi(13) = 40.333934960276501911075053286027D+00
      Xi(14) = 61.079107103866465810354524634818D+00
      Xi(15) = 93.984385911458973639209801831385D+00
      Xi(16) = 147.243788330935086616779727819448D+00
      Xi(17) = 235.457674684953069135784353704821D+00
      Xi(18) = 385.524479321568202178083240028172D+00
      Xi(19) = 648.938204148598712983631031647747D+00
      Xi(20) = 1128.970068389656095986239847661636D+00
      Xi(21) = 2045.142077976131562699890764633892D+00
      Xi(22) = 3897.707449713617865683090712991543D+00
      Xi(23) = 7947.823233342667208045639881675015D+00
      Xi(24) = 17814.961325541782590775596872845199D+00
      Xi(25) = 46249.942302022233448610677442047745D+00
      Xi(26) = 162585.334114312755389164522057399154D+00
      GOTO 9999
C
C     ===== K = 14 =====
C
 1400 CONTINUE
C
      GOTO (1411,1411,1411,1412,1413,1414,1415,1416,1417,1418,1419,1421,
     *      1422,1423,1424,1425,1426,1427,1428,1429,1431,1432,1433,1434,
     *      1435,1436,1437,1438,1439,1441,1499),InitR
C
C     ----- K = 14 , R = 1E1 -----
C
 1411 CONTINUE
C
      RIni = 1.0D+01
C
      Omega( 1) = 0.0611672710980397459221895393316481559509D+00
      Omega( 2) = 0.1445085337560793091400733814677614930133D+00
      Omega( 3) = 0.2333087043102459606655702320510137326437D+00
      Omega( 4) = 0.3312553013320872669935073656644064499233D+00
      Omega( 5) = 0.4432467435331534960001129835882593965835D+00
      Omega( 6) = 0.5759071772555116923647880455838787838729D+00
      Omega( 7) = 0.7383746794403462614239316841313609529607D+00
      Omega( 8) = 0.9433758854644169632359430865964355916731D+00
      Omega( 9) = 1.2089176341155743880147882540754267211014D+00
      Omega(10) = 1.5616549458497541315966580799390328593290D+00
      Omega(11) = 2.0450074532252133611975875027155780117027D+00
      Omega(12) = 2.7419850943350311993709778057848325261148D+00
      Omega(13) = 3.8566755067690577972808196882681386341574D+00
      Omega(14) = 6.1981813314547232786344554966717623756267D+00
C
      Alpha( 1) = 0.0237936689536894477217231907983929772854D+00
      Alpha( 2) = 0.1262969464732095495004306207054156629965D+00
      Alpha( 3) = 0.3146136033888996736719578795060670017847D+00
      Alpha( 4) = 0.5959506165914683579803642898831128604797D+00
      Alpha( 5) = 0.9817869160077900206459386445168036061659D+00
      Alpha( 6) = 1.4893061716231274493058359120745137715858D+00
      Alpha( 7) = 2.1434977755714617149093714765228924079565D+00
      Alpha( 8) = 2.9801766485096160056150055295276501965418D+00
      Alpha( 9) = 4.0503389263368282892027738562035210634349D+00
      Alpha(10) = 5.4269062114355195174282553427502762133372D+00
      Alpha(11) = 7.2168007000185059830396627766901929135201D+00
      Alpha(12) = 9.5870452772388343133311283672526315058349D+00
      Alpha(13) = 12.8357494149077693088717433411716228874866D+00
      Alpha(14) = 17.6753312545577532001461751320903204032220D+00
C
      Xi( 1) = 1.002048743182274802569937943719D+00
      Xi( 2) = 1.018505136379617066234859468210D+00
      Xi( 3) = 1.051666186927454282981804489783D+00
      Xi( 4) = 1.102090275357523667431117186943D+00
      Xi( 5) = 1.170601129974442013636712345281D+00
      Xi( 6) = 1.258281223804727011797953417638D+00
      Xi( 7) = 1.366533237763279598291597805026D+00
      Xi( 8) = 1.497085087079080443811691936151D+00
      Xi( 9) = 1.651984139160994388261655130279D+00
      Xi(10) = 1.833520904503394571990158856689D+00
      Xi(11) = 2.044409233122751011176368440836D+00
      Xi(12) = 2.287701715895157765848222375382D+00
      Xi(13) = 2.566570076646193390842651260719D+00
      Xi(14) = 2.884590301423110433052046652769D+00
      Xi(15) = 3.245079970134787466628373264754D+00
      Xi(16) = 3.651382264167655078293886483642D+00
      Xi(17) = 4.106056771269647577473488331279D+00
      Xi(18) = 4.610542740487232127050837293680D+00
      Xi(19) = 5.164333945273492986003521787453D+00
      Xi(20) = 5.763862439972125912591993657230D+00
      Xi(21) = 6.401371093462892352017601371728D+00
      Xi(22) = 7.063340611288946490654849785784D+00
      Xi(23) = 7.729450216803472793313661837189D+00
      Xi(24) = 8.371749672479516976228430813478D+00
      Xi(25) = 8.955490792852686543143447828186D+00
      Xi(26) = 9.441787161827626983709449426385D+00
      Xi(27) = 9.792471414911000586467348938413D+00
      Xi(28) = 9.976558470699145049660760964372D+00
      GOTO 9999
C
C     ----- K = 14 , R = 2E1 -----
C
 1412 CONTINUE
C
      RIni = 2.0D+01
C
      Omega( 1) = 0.0366971334946315666002201842654795171939D+00
      Omega( 2) = 0.0872726090345815495787507726221932102817D+00
      Omega( 3) = 0.1426593255181061152699582839087355523588D+00
      Omega( 4) = 0.2064271217737803232469221553779803457473D+00
      Omega( 5) = 0.2835897386182783272239298771877358262827D+00
      Omega( 6) = 0.3812489108142730704557703258483769559462D+00
      Omega( 7) = 0.5093454281471228856223096292943353091687D+00
      Omega( 8) = 0.6815036880705534611782532572465953535357D+00
      Omega( 9) = 0.9164311767479403860099142209438838335700D+00
      Omega(10) = 1.2411293792614432520554634820086903346237D+00
      Omega(11) = 1.6988916309365062128847873035297766364238D+00
      Omega(12) = 2.3714793690419589983642972841337837053288D+00
      Omega(13) = 3.4577518634250880304702169265596012337483D+00
      Omega(14) = 5.7402472869474302982761848568316054297611D+00
C
      Alpha( 1) = 0.0142641020572255696012418284615574926022D+00
      Alpha( 2) = 0.0759638522130805713449886763799501210315D+00
      Alpha( 3) = 0.1903985303505704626254955297293136595727D+00
      Alpha( 4) = 0.3640607520646051586846007885878151455472D+00
      Alpha( 5) = 0.6076945939410506078003776364582222413446D+00
      Alpha( 6) = 0.9380409019477502496897614037152379751205D+00
      Alpha( 7) = 1.3802978464353806456422699144859222997184D+00
      Alpha( 8) = 1.9713646649512173690283606264372906480276D+00
      Alpha( 9) = 2.7641415826304564767185167895036101981532D+00
      Alpha(10) = 3.8339843611569064936537426513396553673374D+00
      Alpha(11) = 5.2903788510434682675065420909277236205526D+00
      Alpha(12) = 7.3023268339139556773118022725554965290939D+00
      Alpha(13) = 10.1671211920942247573448891095893031888409D+00
      Alpha(14) = 14.5837970745518835380405420210081501863897D+00
C
      Xi( 1) = 1.002831485574532416013039670677D+00
      Xi( 2) = 1.025583475486455251675550759405D+00
      Xi( 3) = 1.071623521123393540607250662422D+00
      Xi( 4) = 1.142033044953567116550074977965D+00
      Xi( 5) = 1.238460861552316845781221421596D+00
      Xi( 6) = 1.363160575873700177534919575351D+00
      Xi( 7) = 1.519045447931094941019028454754D+00
      Xi( 8) = 1.709763123619948054702881445888D+00
      Xi( 9) = 1.939784035727543322910962353411D+00
      Xi(10) = 2.214503147865410618948256438188D+00
      Xi(11) = 2.540345053631744031948438977686D+00
      Xi(12) = 2.924868399997331741167402863191D+00
      Xi(13) = 3.376852039801018171699675818331D+00
      Xi(14) = 3.906342613885526484241578204504D+00
      Xi(15) = 4.524615371611106158771659080386D+00
      Xi(16) = 5.243987583175885720549597168727D+00
      Xi(17) = 6.077362373687328661337608837201D+00
      Xi(18) = 7.037344840674978575442710670984D+00
      Xi(19) = 8.134677080171074696306188211992D+00
      Xi(20) = 9.375679283081342884828557338039D+00
      Xi(21) = 10.758327198877061139925537247564D+00
      Xi(22) = 12.266718311791904568418265508711D+00
      Xi(23) = 13.864098226244888211779182629613D+00
      Xi(24) = 15.485650623782400917526069772379D+00
      Xi(25) = 17.033982474508729805171003413022D+00
      Xi(26) = 18.382163434717883752217426618358D+00
      Xi(27) = 19.389421462003029458104341742342D+00
      Xi(28) = 19.930475888186686651490364674544D+00
      GOTO 9999
C
C     ----- K = 14 , R = 3E1 -----
C
 1413 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.0269485719324102415012518437278932204038D+00
      Omega( 2) = 0.0643920828356638264082208514271776778060D+00
      Omega( 3) = 0.1062051071686407342615728922885498519690D+00
      Omega( 4) = 0.1558226242684818538188507713870833981673D+00
      Omega( 5) = 0.2182387103483475439470078496850824478770D+00
      Omega( 6) = 0.3006760473398981716450836104703725482068D+00
      Omega( 7) = 0.4132531737378125141432489569881525426354D+00
      Omega( 8) = 0.5697250681720736762347553905350139302755D+00
      Omega( 9) = 0.7889239435759886847178770641342282488040D+00
      Omega(10) = 1.0981226631535513495272621264930990037101D+00
      Omega(11) = 1.5410283271380334405411052789425241371646D+00
      Omega(12) = 2.1995131318883793108964752605061221402138D+00
      Omega(13) = 3.2707215533727314348867720728009089725674D+00
      Omega(14) = 5.5255489337075930690069969219990753117600D+00
C
      Alpha( 1) = 0.0104692054336327858570128424009482159818D+00
      Alpha( 2) = 0.0558852823007056333488396047265434063434D+00
      Alpha( 3) = 0.1406960623012082067513454919360782469084D+00
      Alpha( 4) = 0.2708801030466686922220081318313233964545D+00
      Alpha( 5) = 0.4565826558074826535668297444736651868880D+00
      Alpha( 6) = 0.7139990688724309940101954430335950974040D+00
      Alpha( 7) = 1.0679417676762392550460917250987336046819D+00
      Alpha( 8) = 1.5550829406200727021095803204175922473951D+00
      Alpha( 9) = 2.2282123272354860115274793885831172701728D+00
      Alpha(10) = 3.1627623992935905163602361422903186394251D+00
      Alpha(11) = 4.4686527799694548118009107540160584903788D+00
      Alpha(12) = 6.3156455909790721985064276822186002391391D+00
      Alpha(13) = 9.0012119220790479849972354209342029207619D+00
      Alpha(14) = 13.2194183434275530303683710009465812618146D+00
C
      Xi( 1) = 1.003321224832383769774712956036D+00
      Xi( 2) = 1.030030492626788905276280505774D+00
      Xi( 3) = 1.084196012045892534110189120433D+00
      Xi( 4) = 1.167327983796548967749921732473D+00
      Xi( 5) = 1.281737891987097182266638895065D+00
      Xi( 6) = 1.430605741312938948063028521407D+00
      Xi( 7) = 1.618081766492557762378463881436D+00
      Xi( 8) = 1.849424363655329838873435377433D+00
      Xi( 9) = 2.131171039029589531687766679191D+00
      Xi(10) = 2.471340346121211647804905009451D+00
      Xi(11) = 2.879661595395037482474770440533D+00
      Xi(12) = 3.367828706396157146000722160117D+00
      Xi(13) = 3.949772365114953853433912844828D+00
      Xi(14) = 4.641923498521994057713913051089D+00
      Xi(15) = 5.463422478780356666104578433618D+00
      Xi(16) = 6.436187086040242192729737835677D+00
      Xi(17) = 7.584677428480115245635478915531D+00
      Xi(18) = 8.935085516080642547599721670082D+00
      Xi(19) = 10.513520727690377768587248752397D+00
      Xi(20) = 12.342501004752789365084830741637D+00
      Xi(21) = 14.434812903450735910131086381369D+00
      Xi(22) = 16.783634779342583071351513446245D+00
      Xi(23) = 19.348159783798424478026833916999D+00
      Xi(24) = 22.035594549402970817805758052543D+00
      Xi(25) = 24.684372986728322205488006524376D+00
      Xi(26) = 27.059826381799241903219899185729D+00
      Xi(27) = 28.878537944622666046659564820231D+00
      Xi(28) = 29.871565401147938460107988944969D+00
      GOTO 9999
C
C     ----- K = 14 , R = 4E1 -----
C
 1414 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.0215714048457843105687122258612431480174D+00
      Omega( 2) = 0.0517388459374235217105636105983101913353D+00
      Omega( 3) = 0.0859543366519704896800508839294074903137D+00
      Omega( 4) = 0.1275363375090655946496537645473701161336D+00
      Omega( 5) = 0.1814236824781505632518110657236398708392D+00
      Omega( 6) = 0.2548345086504729486455694126290083545427D+00
      Omega( 7) = 0.3578645442824938114104692254979767085388D+00
      Omega( 8) = 0.5042194995002234180077847758294495861264D+00
      Omega( 9) = 0.7127864027157676590968661545222317954540D+00
      Omega(10) = 1.0111326193554143029794747943306276738440D+00
      Omega(11) = 1.4434560698218156258925901069822828048927D+00
      Omega(12) = 2.0919781635689984765260479582238417606277D+00
      Omega(13) = 3.1530482964361891962123962684927391819656D+00
      Omega(14) = 5.3905535884065372137508020955465326551348D+00
C
      Alpha( 1) = 0.0083766354672433427827797441066193240289D+00
      Alpha( 2) = 0.0447991964410272469703747522801196012665D+00
      Alpha( 3) = 0.1131902344655730421101795350746765933536D+00
      Alpha( 4) = 0.2191456758318374283477873026848570248148D+00
      Alpha( 5) = 0.3723391762342706555802534368337930459347D+00
      Alpha( 6) = 0.5884689484069488065202599424985407949862D+00
      Alpha( 7) = 0.8918422272547547164332770730510446810513D+00
      Alpha( 8) = 1.3185844672473211036797455242286503107607D+00
      Alpha( 9) = 1.9209270418492403716473337915715546841966D+00
      Alpha(10) = 2.7739127897433361309795923910215265095758D+00
      Alpha(11) = 3.9874784350916229929209788629407285043271D+00
      Alpha(12) = 5.7318680938431347069578203345940892177168D+00
      Alpha(13) = 8.3049530509144087178294291184954545315122D+00
      Alpha(14) = 12.3982499699347220156048221539890619169455D+00
C
      Xi( 1) = 1.003680405687570364558873492467D+00
      Xi( 2) = 1.033296098396227069050878899503D+00
      Xi( 3) = 1.093451374628653129691784051669D+00
      Xi( 4) = 1.186016540882130367202951237715D+00
      Xi( 5) = 1.313861639271574527181030145329D+00
      Xi( 6) = 1.480953050500121769743680499865D+00
      Xi( 7) = 1.692500867408396253975012835102D+00
      Xi( 8) = 1.955159670970068617155968038190D+00
      Xi( 9) = 2.277280556392905020443931385365D+00
      Xi(10) = 2.669213858930995028877902752740D+00
      Xi(11) = 3.143665520773058335776695138009D+00
      Xi(12) = 3.716111334016190307109536727914D+00
      Xi(13) = 4.405269811350842221794182895778D+00
      Xi(14) = 5.233616831145244719519782350403D+00
      Xi(15) = 6.227898500017066587240710529016D+00
      Xi(16) = 7.419548529613404719285085775482D+00
      Xi(17) = 8.844822521574932413200575975765D+00
      Xi(18) = 10.544315793569040863066066382459D+00
      Xi(19) = 12.561248743920816494691605758049D+00
      Xi(20) = 14.937519478743447998332627424389D+00
      Xi(21) = 17.705931018766468342320874462814D+00
      Xi(22) = 20.876409995041283034974965993058D+00
      Xi(23) = 24.413903521863484842591152101932D+00
      Xi(24) = 28.207393563142253014447424774858D+00
      Xi(25) = 32.035512194776539631607681357650D+00
      Xi(26) = 35.546799003083486633014498323746D+00
      Xi(27) = 38.287076400773418147910343023455D+00
      Xi(28) = 39.802914803067470653902271138236D+00
      GOTO 9999
C
C     ----- K = 14 , R = 5E1 -----
C
 1415 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.0181203903997484738327760711920633163174D+00
      Omega( 2) = 0.0436009472996198242532490691597546472735D+00
      Omega( 3) = 0.0728818458472537549459944854202131381271D+00
      Omega( 4) = 0.1091824331810118182446581527877960127171D+00
      Omega( 5) = 0.1573753385008842204718405854979579316932D+00
      Omega( 6) = 0.2246218843896930941441336324060351614662D+00
      Omega( 7) = 0.3209240269289367617541047783547014660144D+00
      Omega( 8) = 0.4598968526369810663615485951671857378642D+00
      Omega( 9) = 0.6604755354010544363412178736627566877360D+00
      Omega(10) = 0.9505074044318574636065755090186257802998D+00
      Omega(11) = 1.3746503676746198249368144383986134471343D+00
      Omega(12) = 2.0155108372394360859505835215088609402301D+00
      Omega(13) = 3.0690175985474151918331459087951884612266D+00
      Omega(14) = 5.2942233398720185285311889966664011808462D+00
C
      Alpha( 1) = 0.0070339795370923024902860712457425274025D+00
      Alpha( 2) = 0.0376784577605089209654284747308405911781D+00
      Alpha( 3) = 0.0954894503195428311095546182463156270614D+00
      Alpha( 4) = 0.1857638802321160366508258007800513667007D+00
      Alpha( 5) = 0.3177927469496878407117027021122623864358D+00
      Alpha( 6) = 0.5068344028592784806185565782410407109637D+00
      Alpha( 7) = 0.7766828152420551294394038266410973392340D+00
      Alpha( 8) = 1.1628491895241590386436345005982673228573D+00
      Alpha( 9) = 1.7169175230748922364020642161719365503814D+00
      Alpha(10) = 2.5134448279597254472406825387409412542183D+00
      Alpha(11) = 3.6622634753022948194984415026809188020707D+00
      Alpha(12) = 5.3339687858463377414207362559039893312729D+00
      Alpha(13) = 7.8268675578464850916410311398863086651545D+00
      Alpha(14) = 11.8309580846968924782228183012477984448196D+00
C
      Xi( 1) = 1.003964463629397876258224353574D+00
      Xi( 2) = 1.035881220231276358544232618630D+00
      Xi( 3) = 1.100792113768499243535610443434D+00
      Xi( 4) = 1.200880066059467387315842779039D+00
      Xi( 5) = 1.339501843116432876668216167548D+00
      Xi( 6) = 1.521312512319487506799037257466D+00
      Xi( 7) = 1.752457273050489124110486172103D+00
      Xi( 8) = 2.040833248721928070649328512687D+00
      Xi( 9) = 2.396420482118169978597368996454D+00
      Xi(10) = 2.831683015171900737443760887047D+00
      Xi(11) = 3.362052647165408762221228133882D+00
      Xi(12) = 4.006504479687130676219403158100D+00
      Xi(13) = 4.788240572094590569770383714499D+00
      Xi(14) = 5.735469840795347543169896420423D+00
      Xi(15) = 6.882254158139940615219326547347D+00
      Xi(16) = 8.269329729885666711332314893212D+00
      Xi(17) = 9.944709440199711880832567079125D+00
      Xi(18) = 11.963673873112535328555594560385D+00
      Xi(19) = 14.387430710146393157342070701077D+00
      Xi(20) = 17.279103242847792755790270469163D+00
      Xi(21) = 20.694861929304298055198874095595D+00
      Xi(22) = 24.666812271173920334543794297133D+00
      Xi(23) = 29.173367242468087497109241468252D+00
      Xi(24) = 34.094177311204777226089079888993D+00
      Xi(25) = 39.154260818376700605675244659665D+00
      Xi(26) = 43.881478345259495987135123229450D+00
      Xi(27) = 47.629809930857014223931633623010D+00
      Xi(28) = 49.726209480431178468901043387973D+00
      GOTO 9999
C
C     ----- K = 14 , R = 6E1 -----
C
 1416 CONTINUE
C
      RIni = 6.0D+01
C
      Omega( 1) = 0.0156994377913550977882427587759006648582D+00
      Omega( 2) = 0.0378817333353865612529927462542866933859D+00
      Omega( 3) = 0.0636654818251124950969320470672840883708D+00
      Omega( 4) = 0.0961847375948002209744796923984733183488D+00
      Omega( 5) = 0.1402432889797341093583065221861438942597D+00
      Omega( 6) = 0.2029227810849076473230035225214784588843D+00
      Omega( 7) = 0.2941076299401057839134291521654418488652D+00
      Omega( 8) = 0.4273186187383292571376983831621387821542D+00
      Omega( 9) = 0.6215386798463879908845791666394120511541D+00
      Omega(10) = 0.9048675733944338747389950217137055687999D+00
      Omega(11) = 1.3223740652817814597227985551342044345802D+00
      Omega(12) = 1.9570374697495082172253588748134234265308D+00
      Omega(13) = 3.0045539951549261944042240601859816706565D+00
      Omega(14) = 5.2203736700979702428301876260263725271216D+00
C
      Alpha( 1) = 0.0060922817824678279624866941797250996515D+00
      Alpha( 2) = 0.0326796427354352025450941101464286120404D+00
      Alpha( 3) = 0.0830432040368440788216186990156053582268D+00
      Alpha( 4) = 0.1622373410799410406180889529514921321152D+00
      Alpha( 5) = 0.2792335371332518192781513288736050526495D+00
      Alpha( 6) = 0.4489001056044332305485552309098906675899D+00
      Alpha( 7) = 0.6945418267921845624675303287887118131039D+00
      Alpha( 8) = 1.0510655723020722144027058742032920690690D+00
      Alpha( 9) = 1.5694187175490266713574760704119626097963D+00
      Alpha(10) = 2.3236707816357991059248722098828920934466D+00
      Alpha(11) = 3.4235072283402712242823817678782916118507D+00
      Alpha(12) = 5.0397939065273830204841343238797435333254D+00
      Alpha(13) = 7.4712575541170140043349678915518552457797D+00
      Alpha(14) = 11.4069124477270112142865410298497863550438D+00
C
      Xi( 1) = 1.004199463362554230845199998434D+00
      Xi( 2) = 1.038021580056463033818123231633D+00
      Xi( 3) = 1.106879359284786768771388643096D+00
      Xi( 4) = 1.213233295898756222762175649432D+00
      Xi( 5) = 1.360873325556787383238711852673D+00
      Xi( 6) = 1.555070342390879033667334951918D+00
      Xi( 7) = 1.802810809432306055760664165177D+00
      Xi( 8) = 2.113117747631799232766944274253D+00
      Xi( 9) = 2.497457179134984294995372455794D+00
      Xi(10) = 2.970237139378840199180689674030D+00
      Xi(11) = 3.549413976207429963476841505710D+00
      Xi(12) = 4.257233936341095928250488622524D+00
      Xi(13) = 5.121126723365594167296233063524D+00
      Xi(14) = 6.174761118797634141912300531629D+00
      Xi(15) = 7.459241468024081396503977003931D+00
      Xi(16) = 9.024367421522351983935450459029D+00
      Xi(17) = 10.929764591262393068696323616962D+00
      Xi(18) = 13.245466038343728465073723121748D+00
      Xi(19) = 16.051122814181819527246108947338D+00
      Xi(20) = 19.432246835564534259005453620262D+00
      Xi(21) = 23.470700729134137326251097199759D+00
      Xi(22) = 28.224802961012600523854620959696D+00
      Xi(23) = 33.692641973277278354126362813759D+00
      Xi(24) = 39.752409716617594977877692130619D+00
      Xi(25) = 46.082213176208074587730800786289D+00
      Xi(26) = 52.088252836165963022274638660747D+00
      Xi(27) = 56.916242705365995533522971427942D+00
      Xi(28) = 59.642539771599628304482587637381D+00
      GOTO 9999
C
C     ----- K = 14 , R = 7E1 -----
C
 1417 CONTINUE
C
      RIni = 7.0D+01
C
      Omega( 1) = 0.0138981094100516390688979258792590254501D+00
      Omega( 2) = 0.0336195382439525929142991496345604929274D+00
      Omega( 3) = 0.0567778323204290500227434113078572863742D+00
      Omega( 4) = 0.0864325676208157085079661872961676039040D+00
      Omega( 5) = 0.1273193360592727064610528778243470071629D+00
      Omega( 6) = 0.1864307073569864624219220156087217787899D+00
      Omega( 7) = 0.2735287270934270563058857833604875509081D+00
      Omega( 8) = 0.4020466010199495706441564363275986693225D+00
      Omega( 9) = 0.5910134830489486492820401519754369701332D+00
      Omega(10) = 0.8687528181682690328266338575247118569678D+00
      Omega(11) = 1.2806972849123447405830789058711616235087D+00
      Omega(12) = 1.9101752773557870700724473089771038303297D+00
      Omega(13) = 2.9527569215812303133391475418179084044823D+00
      Omega(14) = 5.1610696744092252451902025356389458465856D+00
C
      Alpha( 1) = 0.0053917358661889226640618331173188115812D+00
      Alpha( 2) = 0.0289579546036361528682647826019036507716D+00
      Alpha( 3) = 0.0737635463848466191102309398586189814750D+00
      Alpha( 4) = 0.1446605225882026802256994854611882317386D+00
      Alpha( 5) = 0.2503473594396809384600086079153413720633D+00
      Alpha( 6) = 0.4053437838072211329130435653045338995071D+00
      Alpha( 7) = 0.6324982596023326382591797645282838402636D+00
      Alpha( 8) = 0.9661477674983821334483112541757776625673D+00
      Alpha( 9) = 1.4566423523519519287123050177434890883887D+00
      Alpha(10) = 2.1775887642399564558294555727258057231666D+00
      Alpha(11) = 3.2385103860715660794612757999999530511559D+00
      Alpha(12) = 4.8104879927941818903644466409730284794932D+00
      Alpha(13) = 7.1926391193233954741807767430117337426054D+00
      Alpha(14) = 11.0733054471847641789250538124633749248460D+00
C
      Xi( 1) = 1.004399828902913362942486696916D+00
      Xi( 2) = 1.039847728421465022181566228987D+00
      Xi( 3) = 1.112079776788726031958899620466D+00
      Xi( 4) = 1.223806877783323803594514589399D+00
      Xi( 5) = 1.379210529911305202163153416084D+00
      Xi( 6) = 1.584120511386782501994478777885D+00
      Xi( 7) = 1.846290820870384225254462462473D+00
      Xi( 8) = 2.175777180328766481588395609315D+00
      Xi( 9) = 2.585417910500366631696214647640D+00
      Xi(10) = 3.091425783935278747945399335784D+00
      Xi(11) = 3.714118769055528738315397063730D+00
      Xi(12) = 4.478822257191948924461705194844D+00
      Xi(13) = 5.416976525759794772445915311820D+00
      Xi(14) = 6.567471281122377282490204519760D+00
      Xi(15) = 7.978203918950735546670310904815D+00
      Xi(16) = 9.707800066685734122554074509992D+00
      Xi(17) = 11.827316057772773032728819142534D+00
      Xi(18) = 14.421495083963127516527014115866D+00
      Xi(19) = 17.588671250208150659971595430875D+00
      Xi(20) = 21.437511045352986821521668048263D+00
      Xi(21) = 26.077244143316060470683126482072D+00
      Xi(22) = 31.595543488792589763985985129580D+00
      Xi(23) = 38.015251733818948200877008147813D+00
      Xi(24) = 45.220081559448178532001350049541D+00
      Xi(25) = 52.847944494740012696953890269924D+00
      Xi(26) = 60.184168246765109535484361558133D+00
      Xi(27) = 66.153095322911794506914873181813D+00
      Xi(28) = 69.552680159615619150115239932575D+00
      GOTO 9999
C
C     ----- K = 14 , R = 8E1 -----
C
 1418 CONTINUE
C
      RIni = 8.0D+01
C
      Omega( 1) = 0.0125004212646736390650997489820683838957D+00
      Omega( 2) = 0.0303076914188914629644759765256245565013D+00
      Omega( 3) = 0.0514124614377946843495937895990310906313D+00
      Omega( 4) = 0.0788083970102124296733085706545551829549D+00
      Omega( 5) = 0.1171649099103936408845837505166009862023D+00
      Omega( 6) = 0.1733827405509416185201481894195829625005D+00
      Omega( 7) = 0.2571046068008916202663350497603644839728D+00
      Omega( 8) = 0.3816850457961544672508616415917970243754D+00
      Omega( 9) = 0.5661965543161639033875771886972216861977D+00
      Omega(10) = 0.8391601113814530699314998829496659027427D+00
      Omega(11) = 1.2463316648187273651864417911561133678333D+00
      Omega(12) = 1.8713633171352733962876369044892044257722D+00
      Omega(13) = 2.9097642414871805181288133557160335840308D+00
      Omega(14) = 5.1118712221355903872060943982091885118280D+00
C
      Alpha( 1) = 0.0048482592516001103827690205463896155891D+00
      Alpha( 2) = 0.0260686293011070024490058189697849755362D+00
      Alpha( 3) = 0.0665500407740831386566547622773981629507D+00
      Alpha( 4) = 0.1309719715774267940870418530652230515443D+00
      Alpha( 5) = 0.2277953400254295355462910363519846157487D+00
      Alpha( 6) = 0.3712257195456134759318968935337679226905D+00
      Alpha( 7) = 0.5836889969345770847559615179811487450934D+00
      Alpha( 8) = 0.8989929313055353704434505268672239708394D+00
      Alpha( 9) = 1.3669355474682264575334023071206956956303D+00
      Alpha(10) = 2.0606917616485738874079064464339694495720D+00
      Alpha(11) = 3.0896188502677947973051181262960085405211D+00
      Alpha(12) = 4.6249719171086872476425511102604559710016D+00
      Alpha(13) = 6.9662281803325074125247950984629596860032D+00
      Alpha(14) = 10.8012531761208309200386623594170032447437D+00
C
      Xi( 1) = 1.004574409398721246216255875705D+00
      Xi( 2) = 1.041439794412749675648829916774D+00
      Xi( 3) = 1.116618705551927111024532179862D+00
      Xi( 4) = 1.233050686296803127601356964149D+00
      Xi( 5) = 1.395275330904291830914909500194D+00
      Xi( 6) = 1.609635232881006502774975941428D+00
      Xi( 7) = 1.884591731760469213901101892539D+00
      Xi( 8) = 2.231157831100469954212911494196D+00
      Xi( 9) = 2.663449323495960571750451606476D+00
      Xi(10) = 3.199369365994019141514331794340D+00
      Xi(11) = 3.861458454225360590857454856639D+00
      Xi(12) = 4.677958450114408868458770740517D+00
      Xi(13) = 5.684132759612934091084796017768D+00
      Xi(14) = 6.923881926522082252452305883850D+00
      Xi(15) = 8.451669851273091271619264386317D+00
      Xi(16) = 10.334721829133737200367981756699D+00
      Xi(17) = 12.655331836263332695967365193468D+00
      Xi(18) = 15.512852820690443205256525693869D+00
      Xi(19) = 19.024412347503992242944703683349D+00
      Xi(20) = 23.322348472599268891930490710251D+00
      Xi(21) = 28.544519675407910743250572416230D+00
      Xi(22) = 34.810434964067422023564768451820D+00
      Xi(23) = 42.171975004832018905920332230153D+00
      Xi(24) = 50.524527995838974355619077627466D+00
      Xi(25) = 59.472387311023984672597730849475D+00
      Xi(26) = 68.181905048529348800645522032937D+00
      Xi(27) = 75.345409801972011369641801792341D+00
      Xi(28) = 79.457213912610988523765875868321D+00
      GOTO 9999
C
C     ----- K = 14 , R = 9E1 -----
C
 1419 CONTINUE
C
      RIni = 9.0D+01
C
      Omega( 1) = 0.0113813041251049466054515212423692283039D+00
      Omega( 2) = 0.0276524776127700123882970518335300269541D+00
      Omega( 3) = 0.0471009994681223302796281311027870586372D+00
      Omega( 4) = 0.0726615318115644792144815385104283933515D+00
      Omega( 5) = 0.1089398770918274977868313822815249380938D+00
      Omega( 6) = 0.1627455965234474854629216633572497130444D+00
      Omega( 7) = 0.2436082848337518060347853948721663641663D+00
      Omega( 8) = 0.3648123709988585392199455376127659178564D+00
      Omega( 9) = 0.5454701187758314865272266103168874451512D+00
      Omega(10) = 0.8142773411254590846261938730243912232254D+00
      Omega(11) = 1.2172789422143520286171552990950317507668D+00
      Omega(12) = 1.8384268358954554908619299924410483981774D+00
      Omega(13) = 2.8732112163151237922177616557206647485145D+00
      Omega(14) = 5.0700605746327023072399231562457089239615D+00
C
      Alpha( 1) = 0.0044131687819247254553858990627074332380D+00
      Alpha( 2) = 0.0237540107528117159956443154786298599390D+00
      Alpha( 3) = 0.0607645778668153227434893005887150962963D+00
      Alpha( 4) = 0.1199747097378512308168442557221311517424D+00
      Alpha( 5) = 0.2096355719615132448280930832495982940600D+00
      Alpha( 6) = 0.3436674309974069099489057260710112018387D+00
      Alpha( 7) = 0.5441054324421764173047230683799568851100D+00
      Alpha( 8) = 0.8442683196867919211809315471750636561410D+00
      Alpha( 9) = 1.2934459449065358992029053930572501940333D+00
      Alpha(10) = 1.9644120588566827564046268772557368720300D+00
      Alpha(11) = 2.9663588112266798511602178267665408384346D+00
      Alpha(12) = 4.4706848820515293859746697169299523011432D+00
      Alpha(13) = 6.7771978107945288916785786526020274322946D+00
      Alpha(14) = 10.5734181763829215875902245436179782700492D+00
C
      Xi( 1) = 1.004729032416783593298706400709D+00
      Xi( 2) = 1.042850578143463038927572783532D+00
      Xi( 3) = 1.120644850395562548629919963350D+00
      Xi( 4) = 1.241262023107741417605152933756D+00
      Xi( 5) = 1.409572102304617733877813157140D+00
      Xi( 6) = 1.632392438605882554612976076402D+00
      Xi( 7) = 1.918841959530300752970535804742D+00
      Xi( 8) = 2.280827243721750823869653945764D+00
      Xi( 9) = 2.733662096629159128126362876543D+00
      Xi(10) = 3.296841212189401015180204024269D+00
      Xi(11) = 3.995010396018193446401184365513D+00
      Xi(12) = 4.859187146991676860535019910614D+00
      Xi(13) = 5.928293590676148453692123618808D+00
      Xi(14) = 7.251053897509312391560232224919D+00
      Xi(15) = 8.888293048457661208426594523502D+00
      Xi(16) = 10.915623346908362838354744628333D+00
      Xi(17) = 13.426381142357288123005687463518D+00
      Xi(18) = 16.534400662015035598020884499704D+00
      Xi(19) = 20.375627698379672405812335078679D+00
      Xi(20) = 25.106432824227256042637357325020D+00
      Xi(21) = 30.894266557075042121949248752344D+00
      Xi(22) = 37.892497473926202138788665862990D+00
      Xi(23) = 46.185670325825824523990892345182D+00
      Xi(24) = 55.686378646269722496320309090834D+00
      Xi(25) = 65.971578519795400462333301305762D+00
      Xi(26) = 76.091301002290827611151957654556D+00
      Xi(27) = 84.497138578197352687393362202783D+00
      Xi(28) = 89.356599271330439969440728731342D+00
      GOTO 9999
C
C     ----- K = 14 , R = 1E2 -----
C
 1421 CONTINUE
C
      RIni = 1.0D+02
C
      Omega( 1) = 0.0104630611822745417864471734366305089026D+00
      Omega( 2) = 0.0254712518740332508723116320542567692087D+00
      Omega( 3) = 0.0435516943672130260874577646623695414974D+00
      Omega( 4) = 0.0675857109039668863917122050566366198154D+00
      Omega( 5) = 0.1021182439661924676755234281388950989822D+00
      Omega( 6) = 0.1538700796419602742751185989189544045530D+00
      Omega( 7) = 0.2322649456760444986580917484109143344995D+00
      Omega( 8) = 0.3505246753474143100429063352185821145213D+00
      Omega( 9) = 0.5277975088592344113380683268132997909561D+00
      Omega(10) = 0.7929346987934133738815632719276749185155D+00
      Omega(11) = 1.1922412421604374170450504899498866961949D+00
      Omega(12) = 1.8099472513036932155502944929992281686282D+00
      Omega(13) = 2.8415520082506048565700040864356878955732D+00
      Omega(14) = 5.0338617706525944036667630232528836131678D+00
C
      Alpha( 1) = 0.0040562245376395091996092839934512364053D+00
      Alpha( 2) = 0.0218539743092570181921173661301893709208D+00
      Alpha( 3) = 0.0560102456526223489891057934902951487288D+00
      Alpha( 4) = 0.1109232905691386661652868099448010497099D+00
      Alpha( 5) = 0.1946568577743407214288798995349694109791D+00
      Alpha( 6) = 0.3208703362477143840834903354419793686247D+00
      Alpha( 7) = 0.5112372035499269371829067498680387870991D+00
      Alpha( 8) = 0.7986245539444243275793537673035160651125D+00
      Alpha( 9) = 1.2318541070578792897237396797294195494032D+00
      Alpha(10) = 1.8833267887511218623055458953885477058066D+00
      Alpha(11) = 2.8620729340610342446380581771236961685645D+00
      Alpha(12) = 4.3396108706176149492211857605639124813024D+00
      Alpha(13) = 6.6160519092730605914681740742366855556611D+00
      Alpha(14) = 10.3786631164987534587870743330029199569253D+00
C
      Xi( 1) = 1.004867744422024420096646590839D+00
      Xi( 2) = 1.044116774458187907196130006504D+00
      Xi( 3) = 1.124261616172937912989590603097D+00
      Xi( 4) = 1.248647883970209255914685098077D+00
      Xi( 5) = 1.422452844938755765931415186998D+00
      Xi( 6) = 1.652936457790613973312283946360D+00
      Xi( 7) = 1.949832966263220590725536995880D+00
      Xi( 8) = 2.325887705682776740489028921566D+00
      Xi( 9) = 2.797544455295651174555823792467D+00
      Xi(10) = 3.385805976423070912410470678644D+00
      Xi(11) = 4.117318890875324560556236530573D+00
      Xi(12) = 5.025753177986356100021697601576D+00
      Xi(13) = 6.153544413960178281319118864801D+00
      Xi(14) = 7.554072825888110332380465106183D+00
      Xi(15) = 9.294337225410822821147771577444D+00
      Xi(16) = 11.458138137699406099850873985702D+00
      Xi(17) = 14.149662656228357445614696175085D+00
      Xi(18) = 17.497079141823863175989051121917D+00
      Xi(19) = 21.655133140059325434420567368932D+00
      Xi(20) = 26.804466342392660601137710507658D+00
      Xi(21) = 33.142875404431976070485088570194D+00
      Xi(22) = 40.859263677785791095409262396743D+00
      Xi(23) = 50.073963117031069433582324279541D+00
      Xi(24) = 60.721740000069154011980154805883D+00
      Xi(25) = 72.358214095924009510996111060877D+00
      Xi(26) = 83.920238156193154091533337179953D+00
      Xi(27) = 93.611467602814374969200095222277D+00
      Xi(28) = 99.251208591006958893709910540792D+00
      GOTO 9999
C
C     ----- K = 14 , R = 2E2 -----
C
 1422 CONTINUE
C
      RIni = 2.0D+02
C
      Omega( 1) = 0.0059897754697514995871548493591718376194D+00
      Omega( 2) = 0.0147993270032761131379042685018987413059D+00
      Omega( 3) = 0.0260524770457164332042342743539498073346D+00
      Omega( 4) = 0.0422725413703181347437271356271004663085D+00
      Omega( 5) = 0.0675264601300846596407885698287731912615D+00
      Omega( 6) = 0.1078388793634545032024542147996992724757D+00
      Omega( 7) = 0.1719019292760687385337829702347534066575D+00
      Omega( 8) = 0.2725409542163217569029059345053767060563D+00
      Omega( 9) = 0.4291033113493526942206369306775926020237D+00
      Omega(10) = 0.6713790246552777997717138813715820333528D+00
      Omega(11) = 1.0473599692425859896240461499594687211356D+00
      Omega(12) = 1.6432750177593764012327556822690155513556D+00
      Omega(13) = 2.6552117669183682784842664981184157113603D+00
      Omega(14) = 4.8210711548907294781221577473218076193007D+00
C
      Alpha( 1) = 0.0023182288230558957666211321703778058811D+00
      Alpha( 2) = 0.0125823731486630252544487145599516608740D+00
      Alpha( 3) = 0.0327195618552320785285395833563715228820D+00
      Alpha( 4) = 0.0663255367617940227539688295699260578431D+00
      Alpha( 5) = 0.1202545590709986545781143632560805656340D+00
      Alpha( 6) = 0.2063694428248624243463042737811719007368D+00
      Alpha( 7) = 0.3438043885629947898343802825671389200579D+00
      Alpha( 8) = 0.5622945792714698096202013966049548798765D+00
      Alpha( 9) = 0.9074155937680068548679274131885108545248D+00
      Alpha(10) = 1.4488825900310234888381660800504846520198D+00
      Alpha(11) = 2.2943634981023110239271162180685337261821D+00
      Alpha(12) = 3.6159269289529930701404536463883232499938D+00
      Alpha(13) = 5.7157741285866358522428642974233525819727D+00
      Alpha(14) = 9.2805653437018404599070930949267221876653D+00
C
      Xi( 1) = 1.005783423474271097430854382537D+00
      Xi( 2) = 1.052489365036918946808182362407D+00
      Xi( 3) = 1.148254951461565999556134476389D+00
      Xi( 4) = 1.297874142975835953014301926700D+00
      Xi( 5) = 1.508814844920841332504002352266D+00
      Xi( 6) = 1.791671683393926679740866536594D+00
      Xi( 7) = 2.160876387478386478034619866406D+00
      Xi( 8) = 2.635675720287350149405314336803D+00
      Xi( 9) = 3.241395772617858590889522441003D+00
      Xi(10) = 4.011075652554957635467780541205D+00
      Xi(11) = 4.987610740587797831645938684275D+00
      Xi(12) = 6.226585113565167666025562676069D+00
      Xi(13) = 7.800007865684829249600801270859D+00
      Xi(14) = 9.801220827275853267335214979994D+00
      Xi(15) = 12.351309172252640383726784989449D+00
      Xi(16) = 15.607391642043947972695594472725D+00
      Xi(17) = 19.773157609552126628277290976143D+00
      Xi(18) = 25.111797925531063319440350234402D+00
      Xi(19) = 31.960759802441204940198771211612D+00
      Xi(20) = 40.745860291856650380742932426870D+00
      Xi(21) = 51.987661755080454287908375832217D+00
      Xi(22) = 66.282418961142906481398551221673D+00
      Xi(23) = 84.217970766470203421161233592329D+00
      Xi(24) = 106.145066720760969825043673608889D+00
      Xi(25) = 131.677581455554635031512944465248D+00
      Xi(26) = 158.832561217851184115490426052020D+00
      Xi(27) = 183.129616610208101168399963398770D+00
      Xi(28) = 197.985220406843973842136996665886D+00
      GOTO 9999
C
C     ----- K = 14 , R = 3E2 -----
C
 1423 CONTINUE
C
      RIni = 3.0D+02
C
      Omega( 1) = 0.0043105055503077696991761779028717871398D+00
      Omega( 2) = 0.0107629963032836868978568190848010832639D+00
      Omega( 3) = 0.0193446992531293919894731792030739825350D+00
      Omega( 4) = 0.0323686242376243212328105949143752795294D+00
      Omega( 5) = 0.0535802550401220360397575113303725302671D+00
      Omega( 6) = 0.0885650729153326647007452021576057177299D+00
      Omega( 7) = 0.1456078542584206715610048096798401218166D+00
      Omega( 8) = 0.2373064176090271693307697312147475088295D+00
      Omega( 9) = 0.3830591074603137403096781571365525564943D+00
      Omega(10) = 0.6131010801761819402886689867049341273741D+00
      Omega(11) = 0.9763482420250424030530415897555229776117D+00
      Omega(12) = 1.5602761224141383718288025872489299672452D+00
      Omega(13) = 2.5616606516288392130387824296988696914923D+00
      Omega(14) = 4.7144075044545905714063471769748048245674D+00
C
      Alpha( 1) = 0.0016663613283814120935658878094876023113D+00
      Alpha( 2) = 0.0090917173500048858559324685239821928207D+00
      Alpha( 3) = 0.0238907650081464097515017040371243517427D+00
      Alpha( 4) = 0.0492457144648069480704570110435325247522D+00
      Alpha( 5) = 0.0913375036628434229659745727247788238401D+00
      Alpha( 6) = 0.1609683530978087451934364568439939091604D+00
      Alpha( 7) = 0.2757734097349819249304393731225815145081D+00
      Alpha( 8) = 0.4636622369593021248334284434289287446518D+00
      Alpha( 9) = 0.7682880833268509717520755586406266957056D+00
      Alpha(10) = 1.2576815493486096930600762200391784517706D+00
      Alpha(11) = 2.0385090183531704907931442205537564404949D+00
      Alpha(12) = 3.2829551393289981401930194282456909604662D+00
      Alpha(13) = 5.2944340379242350253310989138810782606015D+00
      Alpha(14) = 8.7598988337524299023495455429610956343822D+00
C
      Xi( 1) = 1.006315237360364457479267041862D+00
      Xi( 2) = 1.057363459691149750168531751271D+00
      Xi( 3) = 1.162285463353864100699390504712D+00
      Xi( 4) = 1.326845259291127352113508208742D+00
      Xi( 5) = 1.560057635766641579593873301324D+00
      Xi( 6) = 1.874801561952667291717847253274D+00
      Xi( 7) = 2.288784347834112163069461565179D+00
      Xi( 8) = 2.825870501566240824541792475699D+00
      Xi( 9) = 3.517815375625272366266402257473D+00
      Xi(10) = 4.406544277999540099900743772565D+00
      Xi(11) = 5.547208178319016709696687161824D+00
      Xi(12) = 7.012296877113417459346372817564D+00
      Xi(13) = 8.897174186778193184614582644798D+00
      Xi(14) = 11.327513582117413042105180664976D+00
      Xi(15) = 14.469279784670279704293649469804D+00
      Xi(16) = 18.542064035176400008803065944107D+00
      Xi(17) = 23.836755324296568880546032254131D+00
      Xi(18) = 30.738562568365602854433826784941D+00
      Xi(19) = 39.755848693557840636825240210328D+00
      Xi(20) = 51.553351851322890719853120344851D+00
      Xi(21) = 66.982720580029443922798204624769D+00
      Xi(22) = 87.088441063746228787412206884255D+00
      Xi(23) = 113.031639670528592740705864372330D+00
      Xi(24) = 145.797237152944107990704125654702D+00
      Xi(25) = 185.418779279689918140783255751103D+00
      Xi(26) = 229.381732373208163597300135450041D+00
      Xi(27) = 270.443464948497017869000202949792D+00
      Xi(28) = 296.422536960799892280515877018843D+00
      GOTO 9999
C
C     ----- K = 14 , R = 4E2 -----
C
 1424 CONTINUE
C
      RIni = 4.0D+02
C
      Omega( 1) = 0.0034102793689438034177186941271753273242D+00
      Omega( 2) = 0.0085877067644438304675559613457491714428D+00
      Omega( 3) = 0.0156948222369874568904316372575102889186D+00
      Omega( 4) = 0.0268969875261082530000641618861401127560D+00
      Omega( 5) = 0.0457049261615041172641710383317548149762D+00
      Omega( 6) = 0.0774022197228828850578286752126544456587D+00
      Omega( 7) = 0.1300039887810861542234578194787708582680D+00
      Omega( 8) = 0.2159412580728695688411919104132685731656D+00
      Omega( 9) = 0.3546111560919620659575526855356031319388D+00
      Omega(10) = 0.5765112081449787561986640438416173992664D+00
      Omega(11) = 0.9311714010393723479734606751634373722482D+00
      Omega(12) = 1.5069617811410725934963272809596901424811D+00
      Omega(13) = 2.5012628008061577539536401992847913788864D+00
      Omega(14) = 4.6455970530622682453970873073245684281574D+00
C
      Alpha( 1) = 0.0013171250509087271421003036641315941946D+00
      Alpha( 2) = 0.0072165930695399487539990672577833175083D+00
      Alpha( 3) = 0.0191248799561856207320913745087431756531D+00
      Alpha( 4) = 0.0399561872177887720431496785394864801333D+00
      Alpha( 5) = 0.0754352695183603340863052848230818625552D+00
      Alpha( 6) = 0.1356331022754389202230121402870111069205D+00
      Alpha( 7) = 0.2371598253031562126397734091565183689454D+00
      Alpha( 8) = 0.4066773886612077479861277612416259330530D+00
      Alpha( 9) = 0.6865007575761530051600134427758348465431D+00
      Alpha(10) = 1.1434436672367446008805047519274467049399D+00
      Alpha(11) = 1.8833899153099708375855292352696324087447D+00
      Alpha(12) = 3.0785150126590599113738516701133107744681D+00
      Alpha(13) = 5.0330591708377733087612093676455060631270D+00
      Alpha(14) = 8.4343872183413754663877659467630110157188D+00
C
      Xi( 1) = 1.006687768458224417687622842532D+00
      Xi( 2) = 1.060782789639744152446892977437D+00
      Xi( 3) = 1.172156295232698616976853289717D+00
      Xi( 4) = 1.347309446295021649543828534679D+00
      Xi( 5) = 1.596439169299369165676283355282D+00
      Xi( 6) = 1.934186091099623896723616600646D+00
      Xi( 7) = 2.380813041319493150813763060469D+00
      Xi( 8) = 2.963827965478099218249452562191D+00
      Xi( 9) = 3.720118524505239503532075251435D+00
      Xi(10) = 4.698795731552833488016418517219D+00
      Xi(11) = 5.965055525664543634948439043697D+00
      Xi(12) = 7.605444403980453190773636240607D+00
      Xi(13) = 9.735034757760511523079494100585D+00
      Xi(14) = 12.507213603001360598121483036493D+00
      Xi(15) = 16.127036255524257069016158538943D+00
      Xi(16) = 20.869417193176430210324601155492D+00
      Xi(17) = 27.103866884064095709241937193212D+00
      Xi(18) = 35.327692313473632475945285236207D+00
      Xi(19) = 46.209591656963769922689122537918D+00
      Xi(20) = 60.643879284988232058334345353501D+00
      Xi(21) = 79.809678786824841463520829876188D+00
      Xi(22) = 105.211917927531603808577465031249D+00
      Xi(23) = 138.634083789398782127211262604760D+00
      Xi(24) = 181.819035653450962422561509868046D+00
      Xi(25) = 235.465404790081247646260642625293D+00
      Xi(26) = 296.867425186533363201668578312820D+00
      Xi(27) = 356.119449512395370216477630265217D+00
      Xi(28) = 394.631056323149155384832909021497D+00
      GOTO 9999
C
C     ----- K = 14 , R = 5E2 -----
C
 1425 CONTINUE
C
      RIni = 5.0D+02
C
      Omega( 1) = 0.0028426663110938020009905725362861494965D+00
      Omega( 2) = 0.0072103241609037110839534076393853023745D+00
      Omega( 3) = 0.0133657275195878244502213833856674174427D+00
      Omega( 4) = 0.0233612862120284995966408049528073576084D+00
      Omega( 5) = 0.0405263650731162850112884732389773034811D+00
      Omega( 6) = 0.0699215304569996237322461983543231411886D+00
      Omega( 7) = 0.1193636598582501377941627110357281082997D+00
      Omega( 8) = 0.2011495081893048014103422826970302139671D+00
      Omega( 9) = 0.3346541988448492033192665368046547769154D+00
      Omega(10) = 0.5505486506765955056260514188792853929044D+00
      Omega(11) = 0.8988127429424152543108693380613516410449D+00
      Omega(12) = 1.4685085046890018607669212147293080761301D+00
      Omega(13) = 2.4575387649296610392650463960961815246264D+00
      Omega(14) = 4.5958060777879913907928433047800353961065D+00
C
      Alpha( 1) = 0.0010970334583767882200461239185738937518D+00
      Alpha( 2) = 0.0060323406803296099254263007774512495018D+00
      Alpha( 3) = 0.0161030878430710059278171573590721088465D+00
      Alpha( 4) = 0.0340296553738375912029475930942190586848D+00
      Alpha( 5) = 0.0651971871104269767386034557798968336328D+00
      Alpha( 6) = 0.1191293407601383996688264310659377542834D+00
      Alpha( 7) = 0.2116738402433355986876012747033293237564D+00
      Alpha( 8) = 0.3685611864183791356642822673039816550045D+00
      Alpha( 9) = 0.6310911840532924818396830202171798873678D+00
      Alpha(10) = 1.0651302197216859181036485715310391242383D+00
      Alpha(11) = 1.7759219386721268611663976377812446116877D+00
      Alpha(12) = 2.9355854175023709227727169990274092015170D+00
      Alpha(13) = 4.8489747171770592204000094316285185414017D+00
      Alpha(14) = 8.2038646820468288378691568141221068799496D+00
C
      Xi( 1) = 1.006972908128617223309439476786D+00
      Xi( 2) = 1.063402835956595894996781770381D+00
      Xi( 3) = 1.179735523494981054852710766756D+00
      Xi( 4) = 1.363068745404108952586942626883D+00
      Xi( 5) = 1.624560616562614529271788010867D+00
      Xi( 6) = 1.980294227041889729644917605977D+00
      Xi( 7) = 2.452641125258710084024518782719D+00
      Xi( 8) = 3.072141302588350317932450184166D+00
      Xi( 9) = 3.879990181548534547211967704605D+00
      Xi(10) = 4.931386633383035074384981300000D+00
      Xi(11) = 6.300121770460542964392702813825D+00
      Xi(12) = 8.084896558266141935317206090872D+00
      Xi(13) = 10.418015657705428383787704671448D+00
      Xi(14) = 13.477368131632375061398909643628D+00
      Xi(15) = 17.502983446103226866999436062144D+00
      Xi(16) = 22.819894694028703964819304594869D+00
      Xi(17) = 29.869754296451002642878713722041D+00
      Xi(18) = 39.254168332327877319332110062078D+00
      Xi(19) = 51.793290567937741317172672950164D+00
      Xi(20) = 68.601960242918330327988041261733D+00
      Xi(21) = 91.180267185159977576247580799418D+00
      Xi(22) = 121.496116109682911303468966224273D+00
      Xi(23) = 161.981503757656784783791614756865D+00
      Xi(24) = 215.217085564527138091817093368263D+00
      Xi(25) = 282.746156105924997453682223635951D+00
      Xi(26) = 361.962294243132206655877425305334D+00
      Xi(27) = 440.461597656433940656262038260138D+00
      Xi(28) = 492.647718336578408993009503547000D+00
      GOTO 9999

C     ----- K = 14 , R = 6E2 -----
C
 1426 CONTINUE
C
      RIni = 6.0D+02
C
      Omega( 1) = 0.0024493762987277453332710217166756638818D+00
      Omega( 2) = 0.0062525417642427215862345631341295826289D+00
      Omega( 3) = 0.0117353802142847542872562097322763907670D+00
      Omega( 4) = 0.0208593785557110738476465586904717852690D+00
      Omega( 5) = 0.0368082225463150505831368304587858375498D+00
      Omega( 6) = 0.0644690856910460385444995058679129584789D+00
      Omega( 7) = 0.1115033107571565517817388860088989588348D+00
      Omega( 8) = 0.1900944563293023498363630216512198956025D+00
      Omega( 9) = 0.3195868584534482722821040201477416076159D+00
      Omega(10) = 0.5307743481279922103440588321277004979493D+00
      Omega(11) = 0.8739864827482794822316285177077332946283D+00
      Omega(12) = 1.4388462588113037831857865533891072118422D+00
      Omega(13) = 2.4237120872801188467896005285240335069830D+00
      Omega(14) = 4.5572977662850002100389246173506307968637D+00
C
      Alpha( 1) = 0.0009445997794898587881681367728076842427D+00
      Alpha( 2) = 0.0052106503828448096642906380189519399693D+00
      Alpha( 3) = 0.0139993998059489099533507614099570037780D+00
      Alpha( 4) = 0.0298816538645328309941592631257534229405D+00
      Alpha( 5) = 0.0579751202785818767532666391517626180985D+00
      Alpha( 6) = 0.1073721136819342762436799501579720583777D+00
      Alpha( 7) = 0.1933219116222184735299789021256700038975D+00
      Alpha( 8) = 0.3408201283025550454113244275289318352407D+00
      Alpha( 9) = 0.5903549288936290426622194305039670325641D+00
      Alpha(10) = 1.0070203076871469613376777751945212457940D+00
      Alpha(11) = 1.6955193869260442238178221430189296370372D+00
      Alpha(12) = 2.8278951803735827743701419789701390072878D+00
      Alpha(13) = 4.7094833640003784113077511808143071903032D+00
      Alpha(14) = 8.0284416387115672035859614830144437291892D+00
C
      Xi( 1) = 1.007202914800316681347147618197D+00
      Xi( 2) = 1.065518106264580068900417464395D+00
      Xi( 3) = 1.185864472955849129422525867206D+00
      Xi( 4) = 1.375842005664037344045413924221D+00
      Xi( 5) = 1.647420211021003179883981371034D+00
      Xi( 6) = 2.017906664482518768518465313022D+00
      Xi( 7) = 2.511474653003032098032981656033D+00
      Xi( 8) = 3.161271359374571949391041769672D+00
      Xi( 9) = 4.012222430049692694243451684954D+00
      Xi(10) = 5.124834585716500822295121908567D+00
      Xi(11) = 6.580454177572546470287306874525D+00
      Xi(12) = 8.488553460285218935435502807252D+00
      Xi(13) = 10.996840695516867314300568647667D+00
      Xi(14) = 14.305306950756526665653745045148D+00
      Xi(15) = 18.685818917109797742195920200459D+00
      Xi(16) = 24.509491622579109354126969400056D+00
      Xi(17) = 32.284978109522561379868266229209D+00
      Xi(18) = 42.711832230096845788164250379282D+00
      Xi(19) = 56.754102786836676555054514992094D+00
      Xi(20) = 75.738934911144978306962727288010D+00
      Xi(21) = 101.480179268624615393501464666315D+00
      Xi(22) = 136.406985035249315341143727664530D+00
      Xi(23) = 183.614613895448215277750314555760D+00
      Xi(24) = 246.576710827109510951182791416159D+00
      Xi(25) = 327.819419939240517175571554275848D+00
      Xi(26) = 425.085411433247325113660153306228D+00
      Xi(27) = 523.664787470510884104246684955797D+00
      Xi(28) = 590.496446832911009994226958497165D+00
      GOTO 9999
C
C     ----- K = 14 , R = 7E2 -----
C
 1427 CONTINUE
C
      RIni = 7.0D+02
C
      Omega( 1) = 0.0021594492963257279436261964165877369481D+00
      Omega( 2) = 0.0055442742030058087287469853303434330627D+00
      Omega( 3) = 0.0105226822930893956613025014279472202361D+00
      Omega( 4) = 0.0189804496356877784811717068541581099339D+00
      Omega( 5) = 0.0339808497464531976975593910439421563296D+00
      Omega( 6) = 0.0602711111109771843840981434686776552212D+00
      Omega( 7) = 0.1053851584135917551345658534078797430311D+00
      Omega( 8) = 0.1814084463685945920356301988562996996279D+00
      Omega( 9) = 0.3076509595262928073694085745470871984253D+00
      Omega(10) = 0.5149978645662088606329252760129833177416D+00
      Omega(11) = 0.8540615398602765621634096138148350974006D+00
      Omega(12) = 1.4149345700297016986375989167079580965947D+00
      Omega(13) = 2.3963776420822999551306364818259453386418D+00
      Omega(14) = 4.5261871826738335603847185328874047627323D+00
C
      Alpha( 1) = 0.0008322695037266466425591101576351993341D+00
      Alpha( 2) = 0.0046041779357021047074279007649266759472D+00
      Alpha( 3) = 0.0124421375785095599149015859440081666065D+00
      Alpha( 4) = 0.0267964431863254235673560974331786077585D+00
      Alpha( 5) = 0.0525661414626129625800126082130203641896D+00
      Alpha( 6) = 0.0984912639251911154608665199816641688813D+00
      Alpha( 7) = 0.1793334990446931190370130279609917067773D+00
      Alpha( 8) = 0.3194864137129840428105942429226615786320D+00
      Alpha( 9) = 0.5587657706361327789360525508666199812069D+00
      Alpha(10) = 0.9616156945673737321627315421945780826718D+00
      Alpha(11) = 1.6322722752828760647229172642269645621127D+00
      Alpha(12) = 2.7426945317859525586955821152557177811104D+00
      Alpha(13) = 4.5986103520545233802434281766835511007230D+00
      Alpha(14) = 7.8885292427604850038803940837084383019828D+00
C
      Xi( 1) = 1.007395036308558245916576490142D+00
      Xi( 2) = 1.067286191578751572569226657716D+00
      Xi( 3) = 1.190994301131639571968666435620D+00
      Xi( 4) = 1.386553281899055304693738066568D+00
      Xi( 5) = 1.666635507624293386177462272801D+00
      Xi( 6) = 2.049614185777143367337554957786D+00
      Xi( 7) = 2.561238663574738795656454692917D+00
      Xi( 8) = 3.236949124687638550817991855979D+00
      Xi( 9) = 4.124968053904481232480733199353D+00
      Xi(10) = 5.290525092357551192184805799457D+00
      Xi(11) = 6.821730658703404222547445057145D+00
      Xi(12) = 8.837769165998975847906016234390D+00
      Xi(13) = 11.500328233957221379241131486992D+00
      Xi(14) = 15.029616778645906094089013116388D+00
      Xi(15) = 19.726843000593919557131283859519D+00
      Xi(16) = 26.005941186914756785286684248604D+00
      Xi(17) = 34.438341860266177999611025839499D+00
      Xi(18) = 45.816220339762934548460959760519D+00
      Xi(19) = 61.240996678954597205735144171967D+00
      Xi(20) = 82.244714540690540029166388791282D+00
      Xi(21) = 110.947761605146295177393156450307D+00
      Xi(22) = 150.236827224452757556516502290833D+00
      Xi(23) = 203.878680969143173770663501898071D+00
      Xi(24) = 276.279675957493381005525279192625D+00
      Xi(25) = 371.058887831935104106850786820360D+00
      Xi(26) = 486.525575742089060909956188538672D+00
      Xi(27) = 605.866530293086994141127377133671D+00
      Xi(28) = 688.194268309426277874241151266688D+00
      GOTO 9999
C
C     ----- K = 14 , R = 8E2 -----
C
 1428 CONTINUE
C
      RIni = 8.0D+02
C
      Omega( 1) = 0.0019361209248683744086110819081388012552D+00
      Omega( 2) = 0.0049971806200044293842755464135629850375D+00
      Omega( 3) = 0.0095809917510057843200821196816274749608D+00
      Omega( 4) = 0.0175087402037209378964898722971788025404D+00
      Omega( 5) = 0.0317419702329472813214915772467294985404D+00
      Omega( 6) = 0.0569117246472132654479920369633427768008D+00
      Omega( 7) = 0.1004443163474548991628788538466032775887D+00
      Omega( 8) = 0.1743385660966713684343412649191407126636D+00
      Omega( 9) = 0.2978690083437383052901090141340034733730D+00
      Omega(10) = 0.5019910235145395863078261744494312779352D+00
      Omega(11) = 0.8375524167267300298727814023003190868621D+00
      Omega(12) = 1.3950481721401585319787888583853430191084D+00
      Omega(13) = 2.3735982301456124165563166261705418946804D+00
      Omega(14) = 4.5002652071230410369372110235275386003195D+00
C
      Alpha( 1) = 0.0007457708419311785665749849310712099593D+00
      Alpha( 2) = 0.0041365105869129268884829301808347956637D+00
      Alpha( 3) = 0.0112381125521925794856191731414807311040D+00
      Alpha( 4) = 0.0244007484568962324762367169586618764754D+00
      Alpha( 5) = 0.0483397900781959522131445410736416690156D+00
      Alpha( 6) = 0.0914999138377235021933283655448665427912D+00
      Alpha( 7) = 0.1682345234675917510766334980742975346857D+00
      Alpha( 8) = 0.3024303189839334995167932640391939003166D+00
      Alpha( 9) = 0.5333315311260418145351130947418027972162D+00
      Alpha(10) = 0.9248228623483728231971859323756746107392D+00
      Alpha(11) = 1.5807297911278333924899736073754752396781D+00
      Alpha(12) = 2.6729249179396884245497184595663497930218D+00
      Alpha(13) = 4.5074643277945692153123313783424919165554D+00
      Alpha(14) = 7.7731799603489856813863523310459413551143D+00
C
      Xi( 1) = 1.007559565169001995960963713461D+00
      Xi( 2) = 1.068801236800597058798113769829D+00
      Xi( 3) = 1.195394983364988433522518862517D+00
      Xi( 4) = 1.395756730501326540322404312633D+00
      Xi( 5) = 1.683179314898664369359350401201D+00
      Xi( 6) = 2.076980012415584565246481907330D+00
      Xi( 7) = 2.604310762598350100383709504470D+00
      Xi( 8) = 3.302660903902071089493358568845D+00
      Xi( 9) = 4.223214606111398587273003668585D+00
      Xi(10) = 5.435463824124180364123770647922D+00
      Xi(11) = 7.033658176068005277344064030132D+00
      Xi(12) = 9.145843285778266678123782362775D+00
      Xi(13) = 11.946552820585227116384485768208D+00
      Xi(14) = 15.674667998361805276372193951318D+00
      Xi(15) = 20.658703454100486614566922760616D+00
      Xi(16) = 27.352677294407203746964829527144D+00
      Xi(17) = 36.387268219058119207709456333077D+00
      Xi(18) = 48.642682895969507131384013121078D+00
      Xi(19) = 65.352026262405143154643472058751D+00
      Xi(20) = 88.245534595886393806896119684779D+00
      Xi(21) = 119.743166062967725131460650089821D+00
      Xi(22) = 163.184430354296004489489391175994D+00
      Xi(23) = 223.011753425256878488602829690990D+00
      Xi(24) = 304.593940685787349376134613976319D+00
      Xi(25) = 412.732558070269537980889751338509D+00
      Xi(26) = 546.494984986318698416685180063723D+00
      Xi(27) = 687.169966216998286734352063831466D+00
      Xi(28) = 785.754038180450499972984346186422D+00
      GOTO 9999
C
C     ----- K = 14 , R = 9E2 -----
C
 1429 CONTINUE
C
      RIni = 9.0D+02
C
      Omega( 1) = 0.0017583561916395429822060017142557053482D+00
      Omega( 2) = 0.0045606051634619621230909035022871300669D+00
      Omega( 3) = 0.0088258952559276655398936639942975013184D+00
      Omega( 4) = 0.0163192947691258584884085088700933496852D+00
      Omega( 5) = 0.0299149031953826850005912505250926791689D+00
      Omega( 6) = 0.0541451570576183548560143289629653651218D+00
      Omega( 7) = 0.0963434223496242410426622718166989045585D+00
      Omega( 8) = 0.1684309556033862256650820640735410904654D+00
      Omega( 9) = 0.2896469662987799439663860051608068602036D+00
      Omega(10) = 0.4910021987669276186312151966095740363016D+00
      Omega(11) = 0.8235447309834208091407735385924837601124D+00
      Omega(12) = 1.3781205638528317495824018767969221244130D+00
      Omega(13) = 2.3541736567665158426625876542104265354283D+00
      Omega(14) = 4.4781636543138096371620415769854162135744D+00
C
      Alpha( 1) = 0.0006769402078263382473447450288824706055D+00
      Alpha( 2) = 0.0037638910093806651992535250829743920242D+00
      Alpha( 3) = 0.0102764752572631166838897109373773819385D+00
      Alpha( 4) = 0.0224797445212613789045557948984010998572D+00
      Alpha( 5) = 0.0449316043808094025504196091652664435401D+00
      Alpha( 6) = 0.0858240735352995001098547200868438267207D+00
      Alpha( 7) = 0.1591614518697570559540828763478437224421D+00
      Alpha( 8) = 0.2883947892598595928509083718971339749260D+00
      Alpha( 9) = 0.5122730073838949713159809928875176865404D+00
      Alpha(10) = 0.8941908057078154255367316571856406426377D+00
      Alpha(11) = 1.5376077062076304076279967270046711291798D+00
      Alpha(12) = 2.6143101743202715571759431334442069783108D+00
      Alpha(13) = 4.4306346423300903875248935026576191376080D+00
      Alpha(14) = 7.6757091479698113497057154752667429420399D+00
C
      Xi( 1) = 1.007703131510779039070602181027D+00
      Xi( 2) = 1.070123938594893971674483490464D+00
      Xi( 3) = 1.199240777802830972800958442281D+00
      Xi( 4) = 1.403810760739595360794759737466D+00
      Xi( 5) = 1.697682166981737988302805786311D+00
      Xi( 6) = 2.101020309966497421998166128532D+00
      Xi( 7) = 2.642241663843272448134943952702D+00
      Xi( 8) = 3.360690390160581039399878555685D+00
      Xi( 9) = 4.310241741204164041701557552599D+00
      Xi(10) = 5.564277827666065487106222819946D+00
      Xi(11) = 7.222677715972033789856160401754D+00
      Xi(12) = 9.421658387554165060145949794901D+00
      Xi(13) = 12.347645846092340555892608033872D+00
      Xi(14) = 16.256927789513155738557181884119D+00
      Xi(15) = 21.503582659418284004845811629991D+00
      Xi(16) = 28.579431675928376918940920781154D+00
      Xi(17) = 38.171324390074186028859459796081D+00
      Xi(18) = 51.243525771498495510747339665159D+00
      Xi(19) = 69.155797265770647161420292547973D+00
      Xi(20) = 93.830428988181079658081884531384D+00
      Xi(21) = 127.980389928026863324839190738658D+00
      Xi(22) = 175.392737748039292997037641441693D+00
      Xi(23) = 241.187173599354448605414091844068D+00
      Xi(24) = 331.717331821975056599383080424559D+00
      Xi(25) = 453.041911439697934982362426126201D+00
      Xi(26) = 605.156632010292442058752016009748D+00
      Xi(27) = 767.655839022134085070092623936944D+00
      Xi(28) = 883.185893111137792332865359412608D+00
      GOTO 9999
C
C     ----- K = 14 , R = 1E3 -----
C
 1431 CONTINUE
C
      RIni = 1.0D+03
C
      Omega( 1) = 0.0016132124497670363360598869378707831412D+00
      Omega( 2) = 0.0042033179673286300773109411439698579471D+00
      Omega( 3) = 0.0082051681426387509475769890116003679736D+00
      Omega( 4) = 0.0153343743590184841799105644466472142540D+00
      Omega( 5) = 0.0283888068001417349352845679796852884635D+00
      Omega( 6) = 0.0518157306160420973214076993673682380859D+00
      Omega( 7) = 0.0928668402052448009646951412876525466800D+00
      Omega( 8) = 0.1633932015987313481476632393007974997090D+00
      Omega( 9) = 0.2825994389709872368541231091310095280278D+00
      Omega(10) = 0.4815408164843763549618337743707385811831D+00
      Omega(11) = 0.8114387138683526611471381884133080575339D+00
      Omega(12) = 1.3634496747327067076157799951907634294912D+00
      Omega(13) = 2.3373123988437141795165841173798071395140D+00
      Omega(14) = 4.4589806097080054219858324948688732547453D+00
C
      Alpha( 1) = 0.0006207557929943959426693132709531436353D+00
      Alpha( 2) = 0.0034593749933095318074656649744158531945D+00
      Alpha( 3) = 0.0094888451804602900570287317590090214736D+00
      Alpha( 4) = 0.0209005466717163312898746831136609358737D+00
      Alpha( 5) = 0.0421152120539183988705766396171137344595D+00
      Alpha( 6) = 0.0811052525009715691059583760058604440246D+00
      Alpha( 7) = 0.1515714966963109508647647785672951670222D+00
      Alpha( 8) = 0.2765844306591897970798790695745594803157D+00
      Alpha( 9) = 0.4944571460594123973116172099384613147777D+00
      Alpha(10) = 0.8681492267517379013527963060870007439007D+00
      Alpha(11) = 1.5007904436022078916747324317526590675698D+00
      Alpha(12) = 2.5640827986459930748486257812146504875273D+00
      Alpha(13) = 4.3646061500119444805101398543456525658257D+00
      Alpha(14) = 7.5917614754962411591741877625594270284637D+00
C
      Xi( 1) = 1.007830250880272755483260671561D+00
      Xi( 2) = 1.071295646757362046513038789097D+00
      Xi( 3) = 1.202650501122817269295840747478D+00
      Xi( 4) = 1.410960228390045112100044388814D+00
      Xi( 5) = 1.710575956024697986393098836366D+00
      Xi( 6) = 2.122432998568170951173700311543D+00
      Xi( 7) = 2.676099822013581461949138784462D+00
      Xi( 8) = 3.412616073335494759882074999879D+00
      Xi( 9) = 4.388324774083330777341149797621D+00
      Xi(10) = 5.680190837086783292326358507651D+00
      Xi(11) = 7.393300078694071852079827689153D+00
      Xi(12) = 9.671454294457193075207346311117D+00
      Xi(13) = 12.712182348927038999172878241950D+00
      Xi(14) = 16.788080861856720044436586114500D+00
      Xi(15) = 22.277332668577900672696179285026D+00
      Xi(16) = 29.707546328054613487754109790728D+00
      Xi(17) = 39.819093475399266270770004538804D+00
      Xi(18) = 53.656795027205435311135062192989D+00
      Xi(19) = 72.702510799078205239176497798326D+00
      Xi(20) = 99.065061737622908025202583814917D+00
      Xi(21) = 135.744157475194193174239387644775D+00
      Xi(22) = 186.968934242911004162124299909920D+00
      Xi(23) = 258.536220789325584867857088511300D+00
      Xi(24) = 357.801698996791233825520706091083D+00
      Xi(25) = 492.144128900693246547737302876158D+00
      Xi(26) = 662.639973155541552485292555729757D+00
      Xi(27) = 847.389394762888743639628330583946D+00
      Xi(28) = 980.498069961306024933112013286518D+00
      GOTO 9999
C
C     ----- K = 14 , R = 2E3 -----
C
 1432 CONTINUE
C
      RIni = 2.0D+03
C
      Omega( 1) = 0.0009157569104258143407238324450962002610D+00
      Omega( 2) = 0.0024721937700800369844148039050191201227D+00
      Omega( 3) = 0.0051478470787629635160354869827248336378D+00
      Omega( 4) = 0.0103540882355857959877279915981984625972D+00
      Omega( 5) = 0.0204407992989296262054837484412050940819D+00
      Omega( 6) = 0.0393608842827519610437262635033173197030D+00
      Omega( 7) = 0.0738597067220314193268905847422622912291D+00
      Omega( 8) = 0.1353151948841131961152797064107389246601D+00
      Omega( 9) = 0.2426483067657189412894350177829494441539D+00
      Omega(10) = 0.4271026836177867425840912890544487368061D+00
      Omega(11) = 0.7409052083663800540276958561669573555264D+00
      Omega(12) = 1.2771569837351899401670096700911471998552D+00
      Omega(13) = 2.2376078746285902639889114906779354896571D+00
      Omega(14) = 4.3455745783663723949725998796367321119760D+00
C
      Alpha( 1) = 0.0003510368022941439417818036022346517289D+00
      Alpha( 2) = 0.0019913702978629040273133015293699488168D+00
      Alpha( 3) = 0.0056609555525626375320288213334463112147D+00
      Alpha( 4) = 0.0131208337665729531203289977107917518850D+00
      Alpha( 5) = 0.0279788053246742183667025499413383293756D+00
      Alpha( 6) = 0.0569166589761033866168001092017414421775D+00
      Alpha( 7) = 0.1118433745957511314911107802583245529604D+00
      Alpha( 8) = 0.2135401128441132648114143474660231447615D+00
      Alpha( 9) = 0.3976338058430778222402356136822021426269D+00
      Alpha(10) = 0.7243220713184966184569613112387287401361D+00
      Alpha(11) = 1.2945431454910699557757716715578055755032D+00
      Alpha(12) = 2.2792966515798136375829663347047926436062D+00
      Alpha(13) = 3.9865887787029058773966699869006902190449D+00
      Alpha(14) = 7.1077410216557896983392839107551708366373D+00
C
      Xi( 1) = 1.008632189108310501157381167125D+00
      Xi( 2) = 1.078699122282745128027260650416D+00
      Xi( 3) = 1.224259205672618162972879740646D+00
      Xi( 4) = 1.456458985960483048919568427326D+00
      Xi( 5) = 1.793066161802028811575113187438D+00
      Xi( 6) = 2.260302624265208433149143552043D+00
      Xi( 7) = 2.895737656080540215267071779692D+00
      Xi( 8) = 3.752322270362356595877276199680D+00
      Xi( 9) = 4.903961828884370521917240859366D+00
      Xi(10) = 6.453465919941870727969540633140D+00
      Xi(11) = 8.544038831798453964040673813685D+00
      Xi(12) = 11.375914498639907963990769435014D+00
      Xi(13) = 15.230562231000130036899509189396D+00
      Xi(14) = 20.506116255519682754424048987119D+00
      Xi(15) = 27.769634485956145848392639940982D+00
      Xi(16) = 37.834861577064350164623096617333D+00
      Xi(17) = 51.878864203640160643327128298097D+00
      Xi(18) = 71.618450266567320973420951446542D+00
      Xi(19) = 99.578904116495437867651041585759D+00
      Xi(20) = 139.504710893621180989310204267895D+00
      Xi(21) = 196.984703432065028222419655890008D+00
      Xi(22) = 280.380132645586132661419398459657D+00
      Xi(23) = 402.098544586103016335032478423273D+00
      Xi(24) = 579.907368731582759324716391802212D+00
      Xi(25) = 836.447018090643819976381934111487D+00
      Xi(26) = 1189.547781771136276462996761438262D+00
      Xi(27) = 1612.179929966378417849348636536888D+00
      Xi(28) = 1948.167189464106106577823140924011D+00
      GOTO 9999
C
C     ----- K = 14 , R = 3E3 -----
C
 1433 CONTINUE
C
      RIni = 3.0D+03
C
      Omega( 1) = 0.0006582780968990024278469099211063414456D+00
      Omega( 2) = 0.0018238955102013432973480472581821743017D+00
      Omega( 3) = 0.0039689243401756151851419822385036439982D+00
      Omega( 4) = 0.0083471503569080813148364263176492983121D+00
      Omega( 5) = 0.0170917347246665210408657223709050221316D+00
      Omega( 6) = 0.0339122562898936415295059101232544307436D+00
      Omega( 7) = 0.0652812337962773135973011673360844042691D+00
      Omega( 8) = 0.1222991048085924378522010118330864969494D+00
      Omega( 9) = 0.2236897234768833826398792830048023461131D+00
      Omega(10) = 0.4007368414404818431865263478330874136191D+00
      Omega(11) = 0.7061527331295446461775777713398838386638D+00
      Omega(12) = 1.2340831943571578578362074596341813048639D+00
      Omega(13) = 2.1874725831200839232075494589224717856268D+00
      Omega(14) = 4.2885600188927733692791088859763704022043D+00
C
      Alpha( 1) = 0.0002516319551055987430042176262621686500D+00
      Alpha( 2) = 0.0014463966580943881962319767136085357784D+00
      Alpha( 3) = 0.0042192863421776700386364223650803584320D+00
      Alpha( 4) = 0.0101195652062268073377622880186510467837D+00
      Alpha( 5) = 0.0223521001472474348827746510812164437709D+00
      Alpha( 6) = 0.0469651666520016426651089758853574451791D+00
      Alpha( 7) = 0.0949761173357285567214378617373338187235D+00
      Alpha( 8) = 0.1859949564330128747916253545424680737597D+00
      Alpha( 9) = 0.3542301980022188864283879533578769382984D+00
      Alpha(10) = 0.6583664765398028476052656732253609561667D+00
      Alpha(11) = 1.1980745909205334337986614823279296615510D+00
      Alpha(12) = 2.1438544125498438426297848069523865888186D+00
      Alpha(13) = 3.8044083660706669346510122586835223046364D+00
      Alpha(14) = 6.8722249361801352844636436856262662331574D+00
C
      Xi( 1) = 1.009070286187634341219278300006D+00
      Xi( 2) = 1.082752262413102746821912114594D+00
      Xi( 3) = 1.236136394203963409640575910053D+00
      Xi( 4) = 1.481606971602632887518267990057D+00
      Xi( 5) = 1.838982070848160912945878919800D+00
      Xi( 6) = 2.337698296741422804029958415484D+00
      Xi( 7) = 3.020264301605273550002314641993D+00
      Xi( 8) = 3.947093129638576850838627141371D+00
      Xi( 9) = 5.203271703513132750897746126029D+00
      Xi(10) = 6.908359966048753764058382165203D+00
      Xi(11) = 9.230736714816151771909047418774D+00
      Xi(12) = 12.408663241603011520922117405519D+00
      Xi(13) = 16.781362732081296097394473854081D+00
      Xi(14) = 22.835273644526014512334821660033D+00
      Xi(15) = 31.273460901970722243189970690480D+00
      Xi(16) = 43.120824444532904872601219992134D+00
      Xi(17) = 59.885230718427018205274325879373D+00
      Xi(18) = 83.806858436595493028198333718137D+00
      Xi(19) = 118.248187319778805599856941199732D+00
      Xi(20) = 168.309049111282949179324575084138D+00
      Xi(21) = 241.801324455166201238265344386491D+00
      Xi(22) = 350.781042435316185057470050878692D+00
      Xi(23) = 513.865591281148380464749436669081D+00
      Xi(24) = 759.273411353206116425340610476269D+00
      Xi(25) = 1126.646365958049686772923791977519D+00
      Xi(26) = 1657.203863519424259065360161002900D+00
      Xi(27) = 2333.185126236751510520051056118973D+00
      Xi(28) = 2907.925413012566687065429960057372D+00
      GOTO 9999
C
C     ----- K = 14 , R = 4E3 -----
C
 1434 CONTINUE
C
      RIni = 4.0D+03
C
      Omega( 1) = 0.0005212677491514954218130321807623400732D+00
      Omega( 2) = 0.0014754241414154630869342410938796494868D+00
      Omega( 3) = 0.0033215711612985019576517681585969832270D+00
      Omega( 4) = 0.0072120880495609244083537060082678138428D+00
      Omega( 5) = 0.0151457452900341282069651715515290479885D+00
      Omega( 6) = 0.0306764151466120554822278928414069909536D+00
      Omega( 7) = 0.0600935751154314152806118487504249969788D+00
      Omega( 8) = 0.1143044814464072772076160683374368431942D+00
      Omega( 9) = 0.2118856020224212764887453813722473228154D+00
      Omega(10) = 0.3841247050738851271734289327675071490376D+00
      Omega(11) = 0.6840362336317067256372982664291981791393D+00
      Omega(12) = 1.2064609599820673001974169480376986030024D+00
      Omega(13) = 2.1551812818521882252759125364605097274762D+00
      Omega(14) = 4.2518383632143199927187759978863823562278D+00
C
      Alpha( 1) = 0.0001987978935030617574186219645329502803D+00
      Alpha( 2) = 0.0011552666132272471463946275898465598786D+00
      Alpha( 3) = 0.0034409962587201822635370277456492993373D+00
      Alpha( 4) = 0.0084711218688381321973243252496099664484D+00
      Alpha( 5) = 0.0191961514636296384957847961996724173162D+00
      Alpha( 6) = 0.0412654467728940122616526891413668565178D+00
      Alpha( 7) = 0.0851279932993591171189553569453867254424D+00
      Alpha( 8) = 0.1696344295121822774941250724922703341235D+00
      Alpha( 9) = 0.3280570368244356148541580192601330168145D+00
      Alpha(10) = 0.6180615250656186619336238119437609839224D+00
      Alpha(11) = 1.1384402924603920947326310408875826851727D+00
      Alpha(12) = 2.0593131981714612325010116400392234936589D+00
      Alpha(13) = 3.6898183294620708629583949456076652495540D+00
      Alpha(14) = 6.7232681806355749491201090339842494358891D+00
C
      Xi( 1) = 1.009365790770963671081628110215D+00
      Xi( 2) = 1.085489655663102161295592573698D+00
      Xi( 3) = 1.244177037777732447985414021563D+00
      Xi( 4) = 1.498688212725789823277083856468D+00
      Xi( 5) = 1.870299327967785467711180069639D+00
      Xi( 6) = 2.390754196886843693792279452737D+00
      Xi( 7) = 3.106133356644637027663136130684D+00
      Xi( 8) = 4.082298770277047605600784230262D+00
      Xi( 9) = 5.412577757664872825458385330322D+00
      Xi(10) = 7.229007754815780065899000206286D+00
      Xi(11) = 9.718940188868680564102109364910D+00
      Xi(12) = 13.149632487915218144979012659945D+00
      Xi(13) = 17.904924436335444478188283490283D+00
      Xi(14) = 24.540393461737699832073134054156D+00
      Xi(15) = 33.867127427260030704536664103443D+00
      Xi(16) = 47.080324238993898469662902073196D+00
      Xi(17) = 65.959033777349492022601396001846D+00
      Xi(18) = 93.180058469906382231928709813928D+00
      Xi(19) = 132.817624606849302215660912906969D+00
      Xi(20) = 191.147594376094313806113689224730D+00
      Xi(21) = 277.954733203012668507625448910403D+00
      Xi(22) = 408.658974179089547557053307968999D+00
      Xi(23) = 607.704138264774603617190962268069D+00
      Xi(24) = 913.506450357726390631984969559198D+00
      Xi(25) = 1383.292352342555111777144816187501D+00
      Xi(26) = 2085.366194992934200325152005461860D+00
      Xi(27) = 3022.047473583720196943502855901897D+00
      Xi(28) = 3861.321520022056027920953624743561D+00
      GOTO 9999
C
C     ----- K = 14 , R = 5E3 -----
C
 1435 CONTINUE
C
      RIni = 5.0D+03
C
      Omega( 1) = 0.0004352427581767983618992876244154638377D+00
      Omega( 2) = 0.0012548576180906004206129180809578538280D+00
      Omega( 3) = 0.0029046595378673015925571838967769733308D+00
      Omega( 4) = 0.0064645586114702151104412458735432034374D+00
      Omega( 5) = 0.0138392720069440783444312203197268384969D+00
      Omega( 6) = 0.0284704841665357742747430444194001175617D+00
      Omega( 7) = 0.0565119417861124794805877374209712726838D+00
      Omega( 8) = 0.1087244476673139367912316981257259129734D+00
      Omega( 9) = 0.2035678909477675178652271090196101965830D+00
      Omega(10) = 0.3723216049428287995005839444706552399111D+00
      Omega(11) = 0.6682119500204294828692337671061096671110D+00
      Omega(12) = 1.1865913419286377433877963105679498312384D+00
      Omega(13) = 2.1318813562162843423634928097243346201140D+00
      Omega(14) = 4.2253409304934068796422153013736533466727D+00
C
      Alpha( 1) = 0.0001656550983736506384406475398318389636D+00
      Alpha( 2) = 0.0009719040519362571361550429919085730690D+00
      Alpha( 3) = 0.0029465717920729978218602967228109679176D+00
      Alpha( 4) = 0.0074093762335677205316918764437666977685D+00
      Alpha( 5) = 0.0171307513468537186462675638715102977017D+00
      Alpha( 6) = 0.0374775400711816419706658501387330773014D+00
      Alpha( 7) = 0.0784921372444625901570558305919700359254D+00
      Alpha( 8) = 0.1584754097352566251801593219528108136274D+00
      Alpha( 9) = 0.3100135420433681162971292777363885306841D+00
      Alpha(10) = 0.5900153839041758170916827930252424039281D+00
      Alpha(11) = 1.0966083910583347686045277424859989423567D+00
      Alpha(12) = 1.9996081150871290993799658797236418195098D+00
      Alpha(13) = 3.6084591233871420245474942456453959493956D+00
      Alpha(14) = 6.6171039433964087585204882913103574537672D+00
C
      Xi( 1) = 1.009585803629730712833306727205D+00
      Xi( 2) = 1.087529567163898100564875526253D+00
      Xi( 3) = 1.250179023934667543824922342210D+00
      Xi( 4) = 1.511468018403788665334798457796D+00
      Xi( 5) = 1.893799457175245379075487206499D+00
      Xi( 6) = 2.430707808847820036333639559523D+00
      Xi( 7) = 3.171067336009214557681137880429D+00
      Xi( 8) = 4.185021458606984867505118597109D+00
      Xi( 9) = 5.572423445176040247405618854870D+00
      Xi(10) = 7.475265965197596537954866091624D+00
      Xi(11) = 10.096154571210241612626290486077D+00
      Xi(12) = 13.725868979493426130789046624159D+00
      Xi(13) = 18.784768317560292855486481222727D+00
      Xi(14) = 25.885571274187428361321372527470D+00
      Xi(15) = 35.929568389254655503411983730189D+00
      Xi(16) = 50.255736750058273287766752446259D+00
      Xi(17) = 70.874805007675333271000006618578D+00
      Xi(18) = 100.841377956485743032366553251222D+00
      Xi(19) = 144.854072196638575767413392725302D+00
      Xi(20) = 210.236116070503484667963967069682D+00
      Xi(21) = 308.558668847189461026392365283755D+00
      Xi(22) = 458.344739094307165150343763571072D+00
      Xi(23) = 689.531694272874184525878860085868D+00
      Xi(24) = 1050.420345319029590491766157356324D+00
      Xi(25) = 1615.975892449078113122240552002040D+00
      Xi(26) = 2483.892857072607095014760147932975D+00
      Xi(27) = 3685.033596941953352832399559702026D+00
      Xi(28) = 4809.210718299243535689413420186611D+00
      GOTO 9999
C
C     ----- K = 14 , R = 6E3 -----
C
 1436 CONTINUE
C
      RIni = 6.0D+03
C
      Omega( 1) = 0.0003757964231191720894239006572337333245D+00
      Omega( 2) = 0.0011013938591955864320308362087549869379D+00
      Omega( 3) = 0.0026102631943494946954748291189155051661D+00
      Omega( 4) = 0.0059271464833728378313226672468444489184D+00
      Omega( 5) = 0.0128860046532219340670945327900764887374D+00
      Omega( 6) = 0.0268419977564718356200304527625632999843D+00
      Omega( 7) = 0.0538421401369425328202976670927126789934D+00
      Omega( 8) = 0.1045302971427407760522036543615609360813D+00
      Omega( 9) = 0.1972705244360379961236773294075419471483D+00
      Omega(10) = 0.3633288826899753685680059117624196574070D+00
      Omega(11) = 0.6560911744653233455696131337653298487567D+00
      Omega(12) = 1.1713098067261392495495495036372801678226D+00
      Omega(13) = 2.1139192519073143272049369878473612516245D+00
      Omega(14) = 4.2049130034262412448052559366118430261849D+00
C
      Alpha( 1) = 0.0001427695429054092805663457412044918537D+00
      Alpha( 2) = 0.0008448608471347996179937467201622597335D+00
      Alpha( 3) = 0.0026014627740257254616775862976485456102D+00
      Alpha( 4) = 0.0066596448160948629553583478710154786029D+00
      Alpha( 5) = 0.0156533799168818287282428300399850229496D+00
      Alpha( 6) = 0.0347350649212882243940553762123757408631D+00
      Alpha( 7) = 0.0736359011260507862679553121865172471416D+00
      Alpha( 8) = 0.1502321267433053322815604546658363460665D+00
      Alpha( 9) = 0.2965751784347268078659894813275954561504D+00
      Alpha(10) = 0.5689780633687246379497255699408952978047D+00
      Alpha(11) = 1.0650374633237884148097632253104904975771D+00
      Alpha(12) = 1.9543163700723141009687675029304898544069D+00
      Alpha(13) = 3.5464904138887233075831673101419028171222D+00
      Alpha(14) = 6.5360082020505321270982534986870859938790D+00
C
      Xi( 1) = 1.009759423645731589002511041642D+00
      Xi( 2) = 1.089140442439909128185572539937D+00
      Xi( 3) = 1.254924746323060942934136952065D+00
      Xi( 4) = 1.521590832368567632591239979778D+00
      Xi( 5) = 1.912455651212389301420863096936D+00
      Xi( 6) = 2.462512683048163704109881355642D+00
      Xi( 7) = 3.222921273351894386541283132885D+00
      Xi( 8) = 4.267348507491065362409038419500D+00
      Xi( 9) = 5.701040713877533227735283638538D+00
      Xi(10) = 7.674266617894350726684915553477D+00
      Xi(11) = 10.402397579327837314616678199286D+00
      Xi(12) = 14.196024279887322836792018510010D+00
      Xi(13) = 19.506473792871007164745345008328D+00
      Xi(14) = 26.995290139305076636613844875967D+00
      Xi(15) = 37.641456365932051436606897709680D+00
      Xi(16) = 52.908858369821286385392689055607D+00
      Xi(17) = 75.011344579328146309205038733126D+00
      Xi(18) = 107.338076152353949305195968122462D+00
      Xi(19) = 155.146627326469985913570681645979D+00
      Xi(20) = 226.709021208764812074787364792883D+00
      Xi(21) = 335.235835316554947133083075527793D+00
      Xi(22) = 502.140572942818161933375975536364D+00
      Xi(23) = 762.565150234866601430816501761001D+00
      Xi(24) = 1174.378489516354740151982127827068D+00
      Xi(25) = 1830.239271394546196281005734363134D+00
      Xi(26) = 2858.756200324219738595488138344081D+00
      Xi(27) = 4326.171464965746435549220905159018D+00
      Xi(28) = 5752.152994166337293346913384084473D+00
      GOTO 9999
C
C     ----- K = 14 , R = 7E3 -----
C
 1437 CONTINUE
C
      RIni = 7.0D+03
C
      Omega( 1) = 0.0003320523602446911839109026524973281025D+00
      Omega( 2) = 0.0009877883512064985389095867038440623720D+00
      Omega( 3) = 0.0023894838975166594517009308485527840915D+00
      Omega( 4) = 0.0055180549376580071153728247312225674293D+00
      Omega( 5) = 0.0121516088991089741446678684257121094703D+00
      Omega( 6) = 0.0255755558339171696876826857594555697517D+00
      Omega( 7) = 0.0517496526777221388414870067995465774402D+00
      Omega( 8) = 0.1012210601197211070544036345952854105690D+00
      Omega( 9) = 0.1922728458422537548244387120566223359219D+00
      Omega(10) = 0.3561559590722259835079716722150422469895D+00
      Omega(11) = 0.6463818938212136978069093096177510915368D+00
      Omega(12) = 1.1590285100821415355297830385161717003939D+00
      Omega(13) = 2.0994562104528666176585238734020322226570D+00
      Omega(14) = 4.1884637421121013621180817487754666217370D+00
C
      Alpha( 1) = 0.0001259399039924061092209764387331825952D+00
      Alpha( 2) = 0.0007511606113890968361609065965991266500D+00
      Alpha( 3) = 0.0023452526412624827559112299655078026639D+00
      Alpha( 4) = 0.0060974369005672144819378609290755344219D+00
      Alpha( 5) = 0.0145334809715014825981461901033275907480D+00
      Alpha( 6) = 0.0326353621703552626704661256823447601505D+00
      Alpha( 7) = 0.0698852450077041354036693800166890611081D+00
      Alpha( 8) = 0.1438170738021750018223541417450306312276D+00
      Alpha( 9) = 0.2860481254040681115523043348547460595910D+00
      Alpha(10) = 0.5524038197218400129224688543949639552011D+00
      Alpha(11) = 1.0400418005232401251697674227081336084666D+00
      Alpha(12) = 1.9183099739324458423090150049894475614565D+00
      Alpha(13) = 3.4970663804887805921079585447230897443660D+00
      Alpha(14) = 6.4711800508423940707369848013286173227243D+00
C
      Xi( 1) = 1.009901823360759665873694068683D+00
      Xi( 2) = 1.090462394415279327318686608539D+00
      Xi( 3) = 1.258823195298802149295555596886D+00
      Xi( 4) = 1.529918399423402316388877208997D+00
      Xi( 5) = 1.927830548167165567374423118530D+00
      Xi( 6) = 2.488781324253722668400634621122D+00
      Xi( 7) = 3.265859157088403253062225406467D+00
      Xi( 8) = 4.335716246503774000658343323877D+00
      Xi( 9) = 5.808189936932905584644759588997D+00
      Xi(10) = 7.840628772672288646430893699346D+00
      Xi(11) = 10.659370863600591177247634000480D+00
      Xi(12) = 14.592121141847086769789687554777D+00
      Xi(13) = 20.117119280001803086413425525336D+00
      Xi(14) = 27.938587894110425768195993256882D+00
      Xi(15) = 39.103863869794267798390974277822D+00
      Xi(16) = 55.187487095897049659692923384569D+00
      Xi(17) = 78.584645758482612597217720207254D+00
      Xi(18) = 112.985602168826408296664620678484D+00
      Xi(19) = 164.155421665906010489366373406028D+00
      Xi(20) = 241.236176889214221849555563892409D+00
      Xi(21) = 358.957972477610030737071866724364D+00
      Xi(22) = 541.446129752940204127309442583282D+00
      Xi(23) = 828.794889700391907649823508563713D+00
      Xi(24) = 1288.136425684413921111115541862091D+00
      Xi(25) = 2029.678388672859771224565861302835D+00
      Xi(26) = 3213.976430433580548928773623629240D+00
      Xi(27) = 4948.304895005864610268986325536389D+00
      Xi(28) = 6690.550796528218577385871412843699D+00
      GOTO 9999
C
C     ----- K = 14 , R = 8E3 -----
C
 1438 CONTINUE
C
      RIni = 8.0D+03
C
      Omega( 1) = 0.0002983989963563436384249721199839536223D+00
      Omega( 2) = 0.0008999186728744597156233732903295174932D+00
      Omega( 3) = 0.0022167291373876102643848982681404335970D+00
      Omega( 4) = 0.0051938397507288718906493820131470284807D+00
      Omega( 5) = 0.0115637280016053537487651287941903266443D+00
      Omega( 6) = 0.0245537990526819006199658618899706041816D+00
      Omega( 7) = 0.0500504653739266480064575622760392192845D+00
      Omega( 8) = 0.0985188628895173317413357788607797260738D+00
      Omega( 9) = 0.1881721889260020816905761772974692291882D+00
      Omega(10) = 0.3502457688554311079920117882879893045356D+00
      Omega(11) = 0.6383535275892021627723582621882769672084D+00
      Omega(12) = 1.1488458057525466396702612525970721435442D+00
      Omega(13) = 2.0874455976907589924498886979620237980271D+00
      Omega(14) = 4.1748030920369040923899317530043617807678D+00
C
      Alpha( 1) = 0.0001129997667730506017741767990734003818D+00
      Alpha( 2) = 0.0006789271013530558473627197612482175870D+00
      Alpha( 3) = 0.0021465653166690079212435335711053885355D+00
      Alpha( 4) = 0.0056575785323294799351513835721644296584D+00
      Alpha( 5) = 0.0136491045391653851625568256815401735338D+00
      Alpha( 6) = 0.0309631884975518033795702840110797282591D+00
      Alpha( 7) = 0.0668762843902247179151202005620291402010D+00
      Alpha( 8) = 0.1386378928877147248863816939024040664208D+00
      Alpha( 9) = 0.2775023855512674345210038379105910166800D+00
      Alpha(10) = 0.5388849900439262755134223192854392436857D+00
      Alpha(11) = 1.0195707670958165299299538752997307256010D+00
      Alpha(12) = 1.8887208600165844695623063032030586327892D+00
      Alpha(13) = 3.4563421263845536630585075776878056785790D+00
      Alpha(14) = 6.4176615144620159664895497897418863431085D+00
C
      Xi( 1) = 1.010021876087461640343796720476D+00
      Xi( 2) = 1.091577382099120778133349851657D+00
      Xi( 3) = 1.262114131007728140706179720265D+00
      Xi( 4) = 1.536956603474498366575934360512D+00
      Xi( 5) = 1.940844488310749692749554440319D+00
      Xi( 6) = 2.511056295354720912420612433102D+00
      Xi( 7) = 3.302346913890639810290617006672D+00
      Xi( 8) = 4.393953532044695498536002187251D+00
      Xi( 9) = 5.899705249240140257378089971851D+00
      Xi(10) = 7.983125009359271711563421636271D+00
      Xi(11) = 10.880166731198383961096154415316D+00
      Xi(12) = 14.933595748408965791370617282041D+00
      Xi(13) = 20.645449204978822688172357402436D+00
      Xi(14) = 28.757883568506492941857510814430D+00
      Xi(15) = 40.379322561448514686266442197393D+00
      Xi(16) = 57.183783216903668136094074370135D+00
      Xi(17) = 81.730459890198066286171041383568D+00
      Xi(18) = 117.983894588977728669354494428489D+00
      Xi(19) = 172.174916684593830182303797471377D+00
      Xi(20) = 254.250726574617395767119454319527D+00
      Xi(21) = 380.360409158257794598467782520856D+00
      Xi(22) = 577.188226714989345456086056174172D+00
      Xi(23) = 889.557458674596053360694725142821D+00
      Xi(24) = 1393.576918824347966596732817379234D+00
      Xi(25) = 2216.802704432670242828962159364892D+00
      Xi(26) = 3552.448127287603106605473612944479D+00
      Xi(27) = 5553.570815507726670556820636193152D+00
      Xi(28) = 7624.710568402647891339540819899412D+00
      GOTO 9999
C
C     ----- K = 14 , R = 9E3 -----
C
 1439 CONTINUE
C
      RIni = 9.0D+03
C
      Omega( 1) = 0.0002716356155158930017149898823204667853D+00
      Omega( 2) = 0.0008296965023838813597571069922455289092D+00
      Omega( 3) = 0.0020772138991470568411898948340058024264D+00
      Omega( 4) = 0.0049290823178676729135637619473586656937D+00
      Omega( 5) = 0.0110795308513761144015132356632769017324D+00
      Omega( 6) = 0.0237065990394166789337213269509008206626D+00
      Omega( 7) = 0.0486337493334717352138057283594818613892D+00
      Omega( 8) = 0.0962551893464257460184723205753876129620D+00
      Omega( 9) = 0.1847228594633214214786737228379820408009D+00
      Omega(10) = 0.3452565701047216811597154878521465093399D+00
      Omega(11) = 0.6315558344955877121112280958570295297250D+00
      Omega(12) = 1.1402040860907487576951166441396878781234D+00
      Omega(13) = 2.0772388473542544424198241781454044030397D+00
      Omega(14) = 4.1631936182286419828764423289868545907666D+00
C
      Alpha( 1) = 0.0001027140684192060395268882053794370218D+00
      Alpha( 2) = 0.0006213754330576066933856348433341115767D+00
      Alpha( 3) = 0.0019873999833585964459191006408617097279D+00
      Alpha( 4) = 0.0053024100910098172180148767829166445154D+00
      Alpha( 5) = 0.0129291547668094474290547747120927457587D+00
      Alpha( 6) = 0.0295919326841182352423028972462915042740D+00
      Alpha( 7) = 0.0643932006648959625745554623266198035481D+00
      Alpha( 8) = 0.1343406420206151061187216724790438604487D+00
      Alpha( 9) = 0.2703785774775263907039428434853078897504D+00
      Alpha(10) = 0.5275698886607571892748413944218555116095D+00
      Alpha(11) = 1.0023772597521848588342577301979474668769D+00
      Alpha(12) = 1.8637971960121483040468542968781662239053D+00
      Alpha(13) = 3.4219610092511837523554518725177331361920D+00
      Alpha(14) = 6.3724060819922240706154881184630767165800D+00
C
      Xi( 1) = 1.010125201560989707718490882016D+00
      Xi( 2) = 1.092537381742422711457772555033D+00
      Xi( 3) = 1.264949688541450199060107995042D+00
      Xi( 4) = 1.543027040277458050254559551728D+00
      Xi( 5) = 1.952083353517166921121256395466D+00
      Xi( 6) = 2.530322643622601130625887622116D+00
      Xi( 7) = 3.333963617006754299749299308431D+00
      Xi( 8) = 4.444519836938108185535323269022D+00
      Xi( 9) = 5.979346217389822784517988685238D+00
      Xi(10) = 8.107437542282512363787938536319D+00
      Xi(11) = 11.073296154614077048405906822381D+00
      Xi(12) = 15.233137042066466934446544723158D+00
      Xi(13) = 21.110326514049565119895968123842D+00
      Xi(14) = 29.481168875419115939229275369371D+00
      Xi(15) = 41.509331343108100988853470525441D+00
      Xi(16) = 58.959240998940860505050931550386D+00
      Xi(17) = 84.540024245553541636100103318086D+00
      Xi(18) = 122.468349414466947236435068191440D+00
      Xi(19) = 179.406107152853319552932020997105D+00
      Xi(20) = 266.050786687141597564032480249807D+00
      Xi(21) = 399.884748285057179534307536528104D+00
      Xi(22) = 610.018072850112541394285159412902D+00
      Xi(23) = 945.803693317025847397960802709349D+00
      Xi(24) = 1492.058339914736600273492683754739D+00
      Xi(25) = 2393.457106528536410872121109605359D+00
      Xi(26) = 3876.353956281850738907479581030202D+00
      Xi(27) = 6143.641436161321587317019066176726D+00
      Xi(28) = 8554.875233509914672858087669737870D+00
      GOTO 9999
C
C     ----- K = 14 , R = 1E4 -----
C
 1441 CONTINUE
C
      RIni = 1.0D+04
C
      Omega( 1) = 0.0002497975537203257548106469194482484597D+00
      Omega( 2) = 0.0007721386329534672507321604916288859677D+00
      Omega( 3) = 0.0019617570439813717404042820542306446718D+00
      Omega( 4) = 0.0047078208097291976323748132534502408575D+00
      Omega( 5) = 0.0106718522991160866575884361671933131532D+00
      Omega( 6) = 0.0229891216461233004154999766746603206968D+00
      Omega( 7) = 0.0474281748230232914284056327386407403424D+00
      Omega( 8) = 0.0943209541135432382564325257123805101855D+00
      Omega( 9) = 0.1817649964446676749473897141684552991592D+00
      Omega(10) = 0.3409650001887978614726996962058791496020D+00
      Omega(11) = 0.6256934039371224431642752594751044625809D+00
      Omega(12) = 1.1327363996341790508852118146698728651245D+00
      Omega(13) = 2.0684084060135147842881403468240364418307D+00
      Omega(14) = 4.1531491773382185369971331123650770678069D+00
C
      Alpha( 1) = 0.0000943250416165216137989618163403691270D+00
      Alpha( 2) = 0.0005743348523820048554984620892485680521D+00
      Alpha( 3) = 0.0018566487446835667285619298253439291013D+00
      Alpha( 4) = 0.0050085418932687886600417590931416667033D+00
      Alpha( 5) = 0.0123291275938582904669519159715496137153D+00
      Alpha( 6) = 0.0284417237162022576080406578713510867829D+00
      Alpha( 7) = 0.0622988806916012419749193937401621212757D+00
      Alpha( 8) = 0.1306990307731739439390259219231360532376D+00
      Alpha( 9) = 0.2643170553011205327234404982572613107550D+00
      Alpha(10) = 0.5179082172316745659077473484810383297372D+00
      Alpha(11) = 0.9876519994633035921089390940519336936632D+00
      Alpha(12) = 1.8423980004705068490734182207191338420671D+00
      Alpha(13) = 3.3923836281365733233243708877324706918444D+00
      Alpha(14) = 6.3334194940727337611377500969922493823105D+00
C
      Xi( 1) = 1.010215573145741349797392671928D+00
      Xi( 2) = 1.093377313015473438918140347553D+00
      Xi( 3) = 1.267432169816699997848784275956D+00
      Xi( 4) = 1.548346261731285797183348129646D+00
      Xi( 5) = 1.961942329403673889397750895736D+00
      Xi( 6) = 2.547246361548702414407535643548D+00
      Xi( 7) = 3.361779147118380815274785167190D+00
      Xi( 8) = 4.489086559209831966184983675916D+00
      Xi( 9) = 6.049674242423618930695333739678D+00
      Xi(10) = 8.217449116291621392417499070149D+00
      Xi(11) = 11.244603278428059654064918859717D+00
      Xi(12) = 15.499491141598152142873356940100D+00
      Xi(13) = 21.524801679202383516281704700646D+00
      Xi(14) = 30.127891145613154240401332373267D+00
      Xi(15) = 42.522868854636181872030942230367D+00
      Xi(16) = 60.557087013139544104389777601227D+00
      Xi(17) = 87.077791252557945979317910456530D+00
      Xi(18) = 126.535224264338712396593678022327D+00
      Xi(19) = 185.992772931924037566053797831955D+00
      Xi(20) = 276.851457123846353164386968614963D+00
      Xi(21) = 417.852288133038502038196781995794D+00
      Xi(22) = 640.413851445001130913148301715410D+00
      Xi(23) = 998.239223748491957155515308386384D+00
      Xi(24) = 1584.601549504330486595193860921427D+00
      Xi(25) = 2561.050952771884144043568198867433D+00
      Xi(26) = 4187.400841055936970835915644784109D+00
      Xi(27) = 6719.869978176123153978949176234892D+00
      Xi(28) = 9481.242392326531422952484717825428D+00
      GOTO 9999
C
C     ----- K = 14 , R = 7E5 (Best : R = 669250) -----
C
 1499 CONTINUE
C
      RIni = 7.0D+05
C
      Omega( 1) = 0.0000184753234608245182832378360746668905D+00
      Omega( 2) = 0.0001213861645056389429640852281589141065D+00
      Omega( 3) = 0.0004969107802932144441376842338830427392D+00
      Omega( 4) = 0.0016153616951963146559925689656200820110D+00
      Omega( 5) = 0.0045476671955009153315327798984546436856D+00
      Omega( 6) = 0.0115740755102117943190983846007968693925D+00
      Omega( 7) = 0.0273041506300428478152468418638232350304D+00
      Omega( 8) = 0.0606755970684782298768761375079572317759D+00
      Omega( 9) = 0.1284366973240136942301060571369752949522D+00
      Omega(10) = 0.2611402694575218794929144516814290000184D+00
      Omega(11) = 0.5137270825249222130524581753086010849074D+00
      Omega(12) = 0.9871481159900666525102222448317945691088D+00
      Omega(13) = 1.8941325485709733519130645862382777977473D+00
      Omega(14) = 3.9547772395130827950086166433507628426014D+00
C
      Alpha( 1) = 0.0000057793103608868070374277361758280072D+00
      Alpha( 2) = 0.0000635782132566041314606087872052329196D+00
      Alpha( 3) = 0.0003364910016085842384488450805936701760D+00
      Alpha( 4) = 0.0012987332474693207431947977541740468777D+00
      Alpha( 5) = 0.0041590436292552776221117746564132078291D+00
      Alpha( 6) = 0.0117340040293272211121894041396807040201D+00
      Alpha( 7) = 0.0301622120920925923808104951451269926110D+00
      Alpha( 8) = 0.0721391388152867440501062219759820237641D+00
      Alpha( 9) = 0.1628287566381473312857475904680271128200D+00
      Alpha(10) = 0.3504007079130461677302175310666498830869D+00
      Alpha(11) = 0.7245973001913729933439606289979906250664D+00
      Alpha(12) = 1.4504311395006661014822971211835067606444D+00
      Alpha(13) = 2.8398376292459660175278302052248591280659D+00
      Alpha(14) = 5.5949042429760907193039831675207551597850D+00
C
      Xi( 1) = 1.012202011747291133833374920936D+00
      Xi( 2) = 1.111908257588833184056724412425D+00
      Xi( 3) = 1.322577398019902154033093644170D+00
      Xi( 4) = 1.667625646862291010296373838795D+00
      Xi( 5) = 2.185688370457748422445373726042D+00
      Xi( 6) = 2.936997174736573466445263558278D+00
      Xi( 7) = 4.013522612873748518242528193767D+00
      Xi( 8) = 5.553984255286706629445997451811D+00
      Xi( 9) = 7.767175898508397968895977658832D+00
      Xi(10) = 10.969177894241897386513884260495D+00
      Xi(11) = 15.643182022852918929868570874220D+00
      Xi(12) = 22.536802248036572025868817803484D+00
      Xi(13) = 32.822981032538749612131878663490D+00
      Xi(14) = 48.370851797949895544553866599813D+00
      Xi(15) = 72.211172446181928784303405421952D+00
      Xi(16) = 109.355112104990656117509129074961D+00
      Xi(17) = 168.270238675717422119348931630611D+00
      Xi(18) = 263.627460109292646645018365347823D+00
      Xi(19) = 421.567669726816232317334609547288D+00
      Xi(20) = 690.250435615596447780362154844624D+00
      Xi(21) = 1161.871689858553022167697577060608D+00
      Xi(22) = 2021.330299815971881960408040868060D+00
      Xi(23) = 3661.662830963089881430860828004370D+00
      Xi(24) = 6978.532556574037406083022005987004D+00
      Xi(25) = 14229.940028609742074472421791142551D+00
      Xi(26) = 31896.259376488910900704354389745276D+00
      Xi(27) = 82806.812140749924971316886512795463D+00
      Xi(28) = 291096.000256029446035199725884012878D+00
      GOTO 9999
C
C     ===== K = 15 =====
C
 1500 CONTINUE
C
      GOTO (1511,1511,1511,1512,1513,1514,1515,1516,1517,1518,1519,1521,
     *      1522,1523,1524,1525,1526,1527,1528,1529,1531,1532,1533,1534,
     *      1535,1536,1537,1538,1539,1541,1599),InitR
C
C     ----- K = 15 , R = 1E1 -----
C
 1511 CONTINUE
C
      RIni = 1.0D+01
C
      Omega( 1) = 0.0571511962479834348032161923242044920812D+00
      Omega( 2) = 0.1347609820353712418484776885962972414745D+00
      Omega( 3) = 0.2167924173410872770034491657797914854200D+00
      Omega( 4) = 0.3061300554975517251767719306698722903093D+00
      Omega( 5) = 0.4065242973198527684909695661952255818505D+00
      Omega( 6) = 0.5229143871001192803616049176174129797801D+00
      Omega( 7) = 0.6619515007536857417172124207205285983946D+00
      Omega( 8) = 0.8326876570348027339560359472159234428545D+00
      Omega( 9) = 1.0475551694593583445472978477042147460452D+00
      Omega(10) = 1.3240406119874787986990388732877477195871D+00
      Omega(11) = 1.6881745466647175900418847982464853885176D+00
      Omega(12) = 2.1829224353081294340579027180027082977176D+00
      Omega(13) = 2.8914979429762138795137471625196212698938D+00
      Omega(14) = 4.0201557355540847760351497175435042663594D+00
      Omega(15) = 6.3896647738393501912181060564677181901061D+00
C
      Alpha( 1) = 0.0222363742099818770173831009723253160359D+00
      Alpha( 2) = 0.1179176691974456763197594451331173814879D+00
      Alpha( 3) = 0.2932181611149396961626262259059672032890D+00
      Alpha( 4) = 0.5539290777205830730968785791112907190836D+00
      Alpha( 5) = 0.9091507799137611410024428781895267093205D+00
      Alpha( 6) = 1.3722908032211439577525979860439520052751D+00
      Alpha( 7) = 1.9625022213863007779699118571237193009438D+00
      Alpha( 8) = 2.7067233856900886972921688178317367601267D+00
      Alpha( 9) = 3.6425339141137162792850878734185471330420D+00
      Alpha(10) = 4.8222835928813742919357698824001090542879D+00
      Alpha(11) = 6.3196587298171715231173695137556478584884D+00
      Alpha(12) = 8.2417775368720227408339540353665597649524D+00
      Alpha(13) = 10.7556773355908089108423641988565577776171D+00
      Alpha(14) = 14.1604753425802106564165838697988419880858D+00
      Alpha(15) = 19.1750635168336970078767356184812342689838D+00
C
      Xi( 1) = 1.001742002519599818990622686066D+00
      Xi( 2) = 1.016256515869695547051899942037D+00
      Xi( 3) = 1.044956843621521282270032948691D+00
      Xi( 4) = 1.089034086654614941376550962815D+00
      Xi( 5) = 1.148324279348726706211081260989D+00
      Xi( 6) = 1.224653365407459770287414146939D+00
      Xi( 7) = 1.317686476933486784375929024016D+00
      Xi( 8) = 1.430516793639161399090235782250D+00
      Xi( 9) = 1.562968661557137332461693557395D+00
      Xi(10) = 1.717382025458436712304390037165D+00
      Xi(11) = 1.896294827894254538696618139770D+00
      Xi(12) = 2.101892136857211478787738068164D+00
      Xi(13) = 2.335688311894927313107014898996D+00
      Xi(14) = 2.600941080685604287186379890429D+00
      Xi(15) = 2.902340297336362463006403933363D+00
      Xi(16) = 3.238207746322455995438213682824D+00
      Xi(17) = 3.617001587679771039580967073590D+00
      Xi(18) = 4.036043893618174631734984991738D+00
      Xi(19) = 4.497724617924908842477027537399D+00
      Xi(20) = 5.002668369534430500150667953108D+00
      Xi(21) = 5.550378837989716308571469527422D+00
      Xi(22) = 6.132242691307296785559882712491D+00
      Xi(23) = 6.741834101360709671866416892172D+00
      Xi(24) = 7.363480746778404697865105799082D+00
      Xi(25) = 7.980453048090865568724855405591D+00
      Xi(26) = 8.562938576889232315803024420120D+00
      Xi(27) = 9.083638083161566919784823959105D+00
      Xi(28) = 9.511320235181596404253068222090D+00
      Xi(29) = 9.820249490631424653386272183297D+00
      Xi(30) = 9.979618292913723588861973912145D+00
      GOTO 9999
C
C     ----- K = 15 , R = 2E1 -----
C
 1512 CONTINUE
C
      RIni = 2.0D+01
C
      Omega( 1) = 0.0342570755840630594360332952796888150715D+00
      Omega( 2) = 0.0812389574552505602667430532892733907602D+00
      Omega( 3) = 0.1320880642088497519990664477718134151019D+00
      Omega( 4) = 0.1895620574511707937518821195932439138687D+00
      Omega( 5) = 0.2574347552632545563418026840496466434161D+00
      Omega( 6) = 0.3409324413871414226546215781032955760566D+00
      Omega( 7) = 0.4472725024651988329552326079108226508652D+00
      Omega( 8) = 0.5862378479869452498800599937744948420004D+00
      Omega( 9) = 0.7709416195632693878593728886716718307071D+00
      Omega(10) = 1.0193233364053463209462219163370377827960D+00
      Omega(11) = 1.3575648828950585544081755551282242322486D+00
      Omega(12) = 1.8283241651243346893081914705803114884475D+00
      Omega(13) = 2.5132971297146048884809527157457864632306D+00
      Omega(14) = 3.6130225748024470322147805312340551608941D+00
      Omega(15) = 5.9202936131776174579974258982417723018443D+00
C
      Alpha( 1) = 0.0133199863613047550980807848253206771005D+00
      Alpha( 2) = 0.0708357967215275064081273469207555848470D+00
      Alpha( 3) = 0.1770754400387099266971340644216681425860D+00
      Alpha( 4) = 0.3372094560469260752524033042387685554786D+00
      Alpha( 5) = 0.5596497412025823315457305484610373014220D+00
      Alpha( 6) = 0.8572651840652747578944495310260265341640D+00
      Alpha( 7) = 1.2491004136013469878524126399632621087221D+00
      Alpha( 8) = 1.7626491360944358240212676003366709664988D+00
      Alpha( 9) = 2.4367709098923107514760022507260828206199D+00
      Alpha(10) = 3.3256722098160102919339048410662940113980D+00
      Alpha(11) = 4.5052027161924701617948252341250281460816D+00
      Alpha(12) = 6.0845927692710795584886407372238181778812D+00
      Alpha(13) = 8.2321905231666911532950603458402838441543D+00
      Alpha(14) = 11.2452663855440287169959456825552024383796D+00
      Alpha(15) = 15.8274909305672764536429553139385006943485D+00
C
      Xi( 1) = 1.002479499060150240174853208952D+00
      Xi( 2) = 1.022391750455047769625960274453D+00
      Xi( 3) = 1.062625160624124051296848780801D+00
      Xi( 4) = 1.124003499356398210091978473457D+00
      Xi( 5) = 1.207780198723366982564361715724D+00
      Xi( 6) = 1.315663377120945382275744861555D+00
      Xi( 7) = 1.449851096400061798569192106356D+00
      Xi( 8) = 1.613080412284010575524502428824D+00
      Xi( 9) = 1.808683092560154019481748199905D+00
      Xi(10) = 2.040659750638416931444590884936D+00
      Xi(11) = 2.313741520040654406787511421228D+00
      Xi(12) = 2.633475245137824016479835553817D+00
      Xi(13) = 3.006282198555712859098371092514D+00
      Xi(14) = 3.439522060833853566710546778573D+00
      Xi(15) = 3.941501159331839765527505381471D+00
      Xi(16) = 4.521441417355159078737131261860D+00
      Xi(17) = 5.189323080388879948134789321657D+00
      Xi(18) = 5.955554020001711075342487600182D+00
      Xi(19) = 6.830364172019600505891129271774D+00
      Xi(20) = 7.822753285351927012274825257165D+00
      Xi(21) = 8.938836769060568735595362355184D+00
      Xi(22) = 10.179342570236754901037701248612D+00
      Xi(23) = 11.536099533668944362198305331191D+00
      Xi(24) = 12.987495627551309394787093509649D+00
      Xi(25) = 14.493369771815821333595108999504D+00
      Xi(26) = 15.990565620993585975624651229765D+00
      Xi(27) = 17.391553165581567757386194372771D+00
      Xi(28) = 18.589285048398366650967972191211D+00
      Xi(29) = 19.470958874305620065273125796068D+00
      Xi(30) = 19.939964616609519680590079104832D+00
      GOTO 9999
C
C     ----- K = 15 , R = 3E1 -----
C
 1513 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.0251404605998490003396624243095502926337D+00
      Omega( 2) = 0.0598615044552078848093701769067509488309D+00
      Omega( 3) = 0.0980768773006056569061677542775701255096D+00
      Omega( 4) = 0.1424173537496608342521175333272864804712D+00
      Omega( 5) = 0.1966064249578906485626476202677181959189D+00
      Omega( 6) = 0.2659327500342726060200086030116395363621D+00
      Omega( 7) = 0.3577519270311487996349963935927007696591D+00
      Omega( 8) = 0.4819746544793883449321874462567549812775D+00
      Omega( 9) = 0.6518014776973908524604177372463453821183D+00
      Omega(10) = 0.8853077937671970994051688641501840493220D+00
      Omega(11) = 1.2089526907881880205550620410726025966142D+00
      Omega(12) = 1.6657167955120668526415930732476056164160D+00
      Omega(13) = 2.3371854613987042021544526804177621670533D+00
      Omega(14) = 3.4218312791854920487666369410817424068227D+00
      Omega(15) = 5.7000508866800705478662669722922373694018D+00
C
      Alpha( 1) = 0.0097706839879553685810991379978585058552D+00
      Alpha( 2) = 0.0520656002503765860991156218945397071707D+00
      Alpha( 3) = 0.1306478367526141272201992485357813222890D+00
      Alpha( 4) = 0.2502493155231965290500196347212025216322D+00
      Alpha( 5) = 0.4187480144640803076555059247798595833956D+00
      Alpha( 6) = 0.6484848684705118514832664078362256532273D+00
      Alpha( 7) = 0.9580829940330154019002828480466860128217D+00
      Alpha( 8) = 1.3747544144022870453610007945854931676877D+00
      Alpha( 9) = 1.9371818079908711005216251366256585697556D+00
      Alpha(10) = 2.6994958848433685664504333434621230480843D+00
      Alpha(11) = 3.7376744793080215497997620399672769053723D+00
      Alpha(12) = 5.1613963890033491706786639152682027997798D+00
      Alpha(13) = 7.1396285747356587861496557412266383835231D+00
      Alpha(14) = 9.9693937759796579598653432086052816885058D+00
      Alpha(15) = 14.3484779274818628541296461342824386520078D+00
C
      Xi( 1) = 1.002911899669123286312107101725D+00
      Xi( 2) = 1.026313689141005041976714706742D+00
      Xi( 3) = 1.073687816287695194548794386247D+00
      Xi( 4) = 1.146186639545510977267889218822D+00
      Xi( 5) = 1.245569476402107386529827248722D+00
      Xi( 6) = 1.374246811369371118623748662113D+00
      Xi( 7) = 1.535346148506875486221615045856D+00
      Xi( 8) = 1.732801356476310413292155065967D+00
      Xi( 9) = 1.971463807015423310570190196245D+00
      Xi(10) = 2.257234657950460307053452968518D+00
      Xi(11) = 2.597215293815418540781386758098D+00
      Xi(12) = 2.999875704093432549372441608959D+00
      Xi(13) = 3.475238154369281948002576299039D+00
      Xi(14) = 4.035064925614428033885661140090D+00
      Xi(15) = 4.693036802769865503334734135876D+00
      Xi(16) = 5.464884344761582315574272028513D+00
      Xi(17) = 6.368412773388223142682906807721D+00
      Xi(18) = 7.423316204099501691357487764833D+00
      Xi(19) = 8.650621011282775081954743434665D+00
      Xi(20) = 10.071483516655537285462707064454D+00
      Xi(21) = 11.704963697503043124764632310075D+00
      Xi(22) = 13.564220944764207000057809615612D+00
      Xi(23) = 15.650479657737154826936676332849D+00
      Xi(24) = 17.944198940371834598284106476740D+00
      Xi(25) = 20.393466886603605053157606263881D+00
      Xi(26) = 22.901278629381184189645059845475D+00
      Xi(27) = 25.316342800409824157631399899060D+00
      Xi(28) = 27.435805077602629868660510936706D+00
      Xi(29) = 29.029563787005013233977201991820D+00
      Xi(30) = 29.889333975110803413854920762560D+00
      GOTO 9999
C
C     ----- K = 15 , R = 4E1 -----
C
 1514 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.0201134538823361043219286464767137090348D+00
      Omega( 2) = 0.0480468900161741623712583371813567012509D+00
      Omega( 3) = 0.0792047491252899234882378989852647066527D+00
      Omega( 4) = 0.1161121701594685450795478229291113336785D+00
      Omega( 5) = 0.1624290885458093495466100342206061668549D+00
      Omega( 6) = 0.2234294134956040576680180373414508210317D+00
      Omega( 7) = 0.3064598852976081112429659891738253918447D+00
      Omega( 8) = 0.4213902130844775926945405819523671198112D+00
      Omega( 9) = 0.5814003222278704058117657083037954635074D+00
      Omega(10) = 0.8046842905645213672423746165129898599844D+00
      Omega(11) = 1.1180166798382869076467285851528288276313D+00
      Omega(12) = 1.5647904440413159076047819318056042448006D+00
      Omega(13) = 2.2267586807525969559012901655137284251396D+00
      Omega(14) = 3.3013295338455457977617363285460783117742D+00
      Omega(15) = 5.5613724683790860987830928507946737227030D+00
C
      Alpha( 1) = 0.0078140833353863054459905053937551766552D+00
      Alpha( 2) = 0.0417064349886099036901964884349602868951D+00
      Alpha( 3) = 0.1049725240802041791791758551521951403629D+00
      Alpha( 4) = 0.2020201597069699714137374843558880854744D+00
      Alpha( 5) = 0.3403153505965562537207650534165637168371D+00
      Alpha( 6) = 0.5317496824765323231408442605783193357638D+00
      Alpha( 7) = 0.7944899302065638654196666479379018710461D+00
      Alpha( 8) = 1.1552671367902653680502858546397249028814D+00
      Alpha( 9) = 1.6522434451894113991397838869801262262627D+00
      Alpha(10) = 2.3390686809253648662312863315548838727409D+00
      Alpha(11) = 3.2914576993509841271613042268384674571280D+00
      Alpha(12) = 4.6192066650403851603423621696720147156157D+00
      Alpha(13) = 6.4917334024130807878898152818436528832535D+00
      Alpha(14) = 9.2062044495678152173159802806878815317759D+00
      Alpha(15) = 13.4572444522132288824770030011279686732451D+00
C
      Xi( 1) = 1.003229857361806261315147204005D+00
      Xi( 2) = 1.029200765720327604368054819073D+00
      Xi( 3) = 1.081849244748823186002689711316D+00
      Xi( 4) = 1.162604179074884219757292369568D+00
      Xi( 5) = 1.273652369294515127365850781782D+00
      Xi( 6) = 1.418000961648649064670776875907D+00
      Xi( 7) = 1.599575428591748234297131858739D+00
      Xi( 8) = 1.823342794824100966287125369369D+00
      Xi( 9) = 2.095484842818377407619098717895D+00
      Xi(10) = 2.423588140806265554252560412429D+00
      Xi(11) = 2.816875262889290515129839254094D+00
      Xi(12) = 3.286470548855907088461880682928D+00
      Xi(13) = 3.845709652766320670209954091234D+00
      Xi(14) = 4.510476878570599915639627308650D+00
      Xi(15) = 5.299566074429211033798570174014D+00
      Xi(16) = 6.235029436377318096031835414905D+00
      Xi(17) = 7.342446775895706162582549092477D+00
      Xi(18) = 8.650996654635645408820687762841D+00
      Xi(19) = 10.193114846508562773216843844182D+00
      Xi(20) = 12.003382121351368796123570970025D+00
      Xi(21) = 14.116058058878114126952207918464D+00
      Xi(22) = 16.560385872556048922343840956728D+00
      Xi(23) = 19.352313942618907987094911682391D+00
      Xi(24) = 22.481563079780628417597698742725D+00
      Xi(25) = 25.892502301700924332086839463329D+00
      Xi(26) = 29.460835413519626530334805103450D+00
      Xi(27) = 32.971840195575683459677795994480D+00
      Xi(28) = 36.115528534276363340632665099861D+00
      Xi(29) = 38.519215416158373282290439121311D+00
      Xi(30) = 39.830469749008866213457746852100D+00
      GOTO 9999
C
C     ----- K = 15 , R = 5E1 -----
C
 1515 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.0168880504457705811679482047621414508853D+00
      Omega( 2) = 0.0404523657107779312077841787215204405470D+00
      Omega( 3) = 0.0670338169082042983486528428538342438969D+00
      Omega( 4) = 0.0990695634544053111477518447358647790679D+00
      Omega( 5) = 0.1401553278525078226080533352893020548890D+00
      Omega( 6) = 0.1955184392167879767383559946880566826621D+00
      Omega( 7) = 0.2724397623186262082145472213090542368263D+00
      Omega( 8) = 0.3806988587933995549900632882067696982631D+00
      Omega( 9) = 0.5334384650052095012398231665962100578326D+00
      Omega(10) = 0.7489681402627489304650934553997387865820D+00
      Omega(11) = 1.0543592358076193700325631774639134619065D+00
      Omega(12) = 1.4933981322491246728281777400582086556824D+00
      Omega(13) = 2.1480751959787540564857794000985791171843D+00
      Omega(14) = 3.2151626673716206042397125486331788124517D+00
      Omega(15) = 5.4623017093067981523289711187629791311338D+00
C
      Alpha( 1) = 0.0065589737028395636394760526925035160062D+00
      Alpha( 2) = 0.0350550853541004113653417693236002961044D+00
      Alpha( 3) = 0.0884595857610457954417980917793951078920D+00
      Alpha( 4) = 0.1709281913314045605651768020005754067370D+00
      Alpha( 5) = 0.2895971214269864041769390883285240079204D+00
      Alpha( 6) = 0.4559750943112586425958864672081105595680D+00
      Alpha( 7) = 0.6877928532962935922914597675070780269380D+00
      Alpha( 8) = 1.0112643890368854897570533024619976458780D+00
      Alpha( 9) = 1.4639620939737501783715553638209883047239D+00
      Alpha(10) = 2.0989787920680469870005763199927173445758D+00
      Alpha(11) = 2.9916804481492304287719441457937818995561D+00
      Alpha(12) = 4.2518719910654838274946532017395384173142D+00
      Alpha(13) = 6.0493272187711193718477609149175577840651D+00
      Alpha(14) = 8.6814711877106877273638230185781594627770D+00
      Alpha(15) = 12.8409564351942392032568385218382900347933D+00
C
      Xi( 1) = 1.003481855447011282978127255650D+00
      Xi( 2) = 1.031490732158271747716249411297D+00
      Xi( 3) = 1.088333640314139787261954461783D+00
      Xi( 4) = 1.175680240722669799534327861323D+00
      Xi( 5) = 1.296090183590884389874808846344D+00
      Xi( 6) = 1.453094491106130258157615176007D+00
      Xi( 7) = 1.651319610191417435515054412676D+00
      Xi( 8) = 1.896656937573268690349834186826D+00
      Xi( 9) = 2.196477388512006007244281580171D+00
      Xi(10) = 2.559890491008365413306072100319D+00
      Xi(11) = 2.998053898728377583606657941218D+00
      Xi(12) = 3.524539770453997259303663414087D+00
      Xi(13) = 4.155765896611457538723649607348D+00
      Xi(14) = 4.911493744258011896839516419178D+00
      Xi(15) = 5.815386349036163648757585820093D+00
      Xi(16) = 6.895596746028621872211250565243D+00
      Xi(17) = 8.185322874284366840257065600639D+00
      Xi(18) = 9.723200314621305568110742889587D+00
      Xi(19) = 11.553289726430778155269463169930D+00
      Xi(20) = 13.724215799603990733647462807454D+00
      Xi(21) = 16.286719270811650639094181336475D+00
      Xi(22) = 19.288370022171881075501165270225D+00
      Xi(23) = 22.763612784503022020907159195247D+00
      Xi(24) = 26.716677090822341378295590175185D+00
      Xi(25) = 31.095063981479428792328856268057D+00
      Xi(26) = 35.753698060397073880262164280452D+00
      Xi(27) = 40.417221907703723179916543983836D+00
      Xi(28) = 44.661862908451292482336913991503D+00
      Xi(29) = 47.952660699750523112111499912658D+00
      Xi(30) = 49.764818282386813565881800158763D+00
      GOTO 9999
C
C     ----- K = 15 , R = 6E1 -----
C
 1516 CONTINUE
C
      RIni = 6.0D+01
C
      Omega( 1) = 0.0146259217872521891768382426221584147186D+00
      Omega( 2) = 0.0351175064449876853134877106171751748320D+00
      Omega( 3) = 0.0584601957105404397711031095480338493076D+00
      Omega( 4) = 0.0870165685389104654571547879660098345767D+00
      Omega( 5) = 0.1243207801068556728234714911507374779376D+00
      Omega( 6) = 0.1755396894057373440045134727061082458022D+00
      Omega( 7) = 0.2478661231331974945581598374078069468851D+00
      Omega( 8) = 0.3509786266364572828043398577957390216397D+00
      Omega( 9) = 0.4979850090944644447094481060078763334786D+00
      Omega(10) = 0.7073027698955994890068696423490735014639D+00
      Omega(11) = 1.0062683560617502864092953629437943163794D+00
      Omega(12) = 1.4390235276810452051459918165754459096206D+00
      Omega(13) = 2.0878094629421765564096513667635690580937D+00
      Omega(14) = 3.1489876316118781847657598360612496435351D+00
      Omega(15) = 5.3862800842515326351159221651698771893280D+00
C
      Alpha( 1) = 0.0056788702569020912235681335794579283061D+00
      Alpha( 2) = 0.0303873265325022603842866977807979367299D+00
      Alpha( 3) = 0.0768546492022988189511158163558235401069D+00
      Alpha( 4) = 0.1490327845948825036780094333010993068456D+00
      Alpha( 5) = 0.2537854202184757155798342648234111607053D+00
      Alpha( 6) = 0.4022900808658428740494442110708916970907D+00
      Alpha( 7) = 0.6118736806237610867582367435435486413553D+00
      Alpha( 8) = 0.9082474519754037504135416036810113382671D+00
      Alpha( 9) = 1.3284052305443362714974450677907213957951D+00
      Alpha(10) = 1.9248935323363437296549302568671180324600D+00
      Alpha(11) = 2.7727204966996773344810323091991222099750D+00
      Alpha(12) = 3.9816530453020171492761747344957257155329D+00
      Alpha(13) = 5.7217548099966258393599127796136372126057D+00
      Alpha(14) = 8.2907395887501191293275648597216331836535D+00
      Alpha(15) = 12.3799185668548834054689122829984171403339D+00
C
      Xi( 1) = 1.003690691737168430098040006193D+00
      Xi( 2) = 1.033389874851190462783263879221D+00
      Xi( 3) = 1.093718642683819189430696749987D+00
      Xi( 4) = 1.186560911089355885273365076582D+00
      Xi( 5) = 1.314808791861358351865489424437D+00
      Xi( 6) = 1.482461815451560535735961443571D+00
      Xi( 7) = 1.694778672608641367043277869975D+00
      Xi( 8) = 1.958485097627332217938914138422D+00
      Xi( 9) = 2.282037250536437880633552910403D+00
      Xi(10) = 2.675942969701724920197211110562D+00
      Xi(11) = 3.153148192149421274663168013319D+00
      Xi(12) = 3.729502671580447342855255121741D+00
      Xi(13) = 4.424317147737756152590937741564D+00
      Xi(14) = 5.261024415054952740791044396573D+00
      Xi(15) = 6.267940203075530144691263068069D+00
      Xi(16) = 7.479106120763681706323233733258D+00
      Xi(17) = 8.935155916674657633827494329637D+00
      Xi(18) = 10.684074934984131811585572613410D+00
      Xi(19) = 12.781590917452321275529780741920D+00
      Xi(20) = 15.290692656127236188391693705313D+00
      Xi(21) = 18.279391780428630885288465712790D+00
      Xi(22) = 21.815153227757841315065379106386D+00
      Xi(23) = 25.953542579470569560268211617426D+00
      Xi(24) = 30.717517687357415171606200843257D+00
      Xi(25) = 36.063340223198843699503601811784D+00
      Xi(26) = 41.831395934728776656724713589597D+00
      Xi(27) = 47.689420141103013205813310193548D+00
      Xi(28) = 53.096033085676186246554264513975D+00
      Xi(29) = 57.338084472108027285497389158309D+00
      Xi(30) = 59.693320802696889410682290488808D+00
      GOTO 9999
C
C     ----- K = 15 , R = 7E1 -----
C
 1517 CONTINUE
C
      RIni = 7.0D+01
C
      Omega( 1) = 0.0129431385760811314218342326677613218777D+00
      Omega( 2) = 0.0311433962175995538834011311379867947835D+00
      Omega( 3) = 0.0520576228698372043838453297748269932299D+00
      Omega( 4) = 0.0779840434358426823802275251740212524965D+00
      Omega( 5) = 0.1123987204649185051407301975945340899443D+00
      Omega( 6) = 0.1604023598955141054406888651007179191765D+00
      Omega( 7) = 0.2290926497594118356042900211400059617972D+00
      Omega( 8) = 0.3280493759577526712706285749732160184067D+00
      Omega( 9) = 0.4703507500452662837091537956224485128587D+00
      Omega(10) = 0.6745122915667454001169278665628326052683D+00
      Omega(11) = 0.9681041644892160188224858319294696684665D+00
      Omega(12) = 1.3955858527294683469168259937198683928727D+00
      Omega(13) = 2.0394448609688786832206536248079942197364D+00
      Omega(14) = 3.0957651278694305178353085583609072273248D+00
      Omega(15) = 5.3251821433448008865155220981080219644355D+00
C
      Alpha( 1) = 0.0050242744369184408194854523071332108586D+00
      Alpha( 2) = 0.0269131431088499060335896212806106042592D+00
      Alpha( 3) = 0.0682062998374950968293799029307233894315D+00
      Alpha( 4) = 0.1326861736517988391465287453807597728428D+00
      Alpha( 5) = 0.2269855619840041827220715117685578832152D+00
      Alpha( 6) = 0.3619911699308799959734668505451082864965D+00
      Alpha( 7) = 0.5546589943832142908380911094923249038402D+00
      Alpha( 8) = 0.8302275837852374761613796771975160027068D+00
      Alpha( 9) = 1.2251474742957150349236677966402453421324D+00
      Alpha(10) = 1.7914531635436964926780256668337187875295D+00
      Alpha(11) = 2.6038082335791434180922304664207445057400D+00
      Alpha(12) = 3.7719182048923451738235007590915870423487D+00
      Alpha(13) = 5.4660873571676506076782997567420352424961D+00
      Alpha(14) = 7.9843185170675084285675515705094085205928D+00
      Alpha(15) = 12.0169576607587050556466556905377274233615D+00
C
C
      Xi( 1) = 1.003869021713253574957375180432D+00
      Xi( 2) = 1.035012555911422882045681836960D+00
      Xi( 3) = 1.098325070301832839846437051001D+00
      Xi( 4) = 1.195883988344786225056955164536D+00
      Xi( 5) = 1.330882547499336633839775445587D+00
      Xi( 6) = 1.507745732205119922205403815152D+00
      Xi( 7) = 1.732309131445018534701044132795D+00
      Xi( 8) = 2.012063648685074870913272460093D+00
      Xi( 9) = 2.356466640738977198201928309196D+00
      Xi(10) = 2.777322490733142419227358344003D+00
      Xi(11) = 3.289245690693270381815827008154D+00
      Xi(12) = 3.910223848851538200723301774708D+00
      Xi(13) = 4.662302835038631008881399031551D+00
      Xi(14) = 5.572409455073852398761941051575D+00
      Xi(15) = 6.673321239676869499890532466679D+00
      Xi(16) = 8.004772939028161932335792005944D+00
      Xi(17) = 9.614651387741289116466159025975D+00
      Xi(18) = 11.560153941240544802661338597005D+00
      Xi(19) = 13.908637463799372543243881850117D+00
      Xi(20) = 16.737614017371478167955922700116D+00
      Xi(21) = 20.132875726972504146217168852218D+00
      Xi(22) = 24.182884524734062060835615426413D+00
      Xi(23) = 28.966358540259132967054078733327D+00
      Xi(24) = 34.528318026877213037934000183782D+00
      Xi(25) = 40.838604370819314472174088592737D+00
      Xi(26) = 47.728776377028992096429593772200D+00
      Xi(27) = 54.813878058326717286619089719579D+00
      Xi(28) = 61.432882561411230437631303402668D+00
      Xi(29) = 66.681268392919390133877222126557D+00
      Xi(30) = 69.616639040442253115459347867500D+00
      GOTO 9999
C
C     ----- K = 15 , R = 8E1 -----
C
 1518 CONTINUE
C
      RIni = 8.0D+01
C
      Omega( 1) = 0.0116376959297197557926148675337685300946D+00
      Omega( 2) = 0.0280565631287540469228494266831774872628D+00
      Omega( 3) = 0.0470734626814240072829236081336512498297D+00
      Omega( 4) = 0.0709302331808647406336388978320695031243D+00
      Omega( 5) = 0.1030483747950790365550260974103480293707D+00
      Omega( 6) = 0.1484608343730583948622383030913063350908D+00
      Omega( 7) = 0.2141699383993229090569895486928508887559D+00
      Omega( 8) = 0.3096632904070938441579673161552221927195D+00
      Omega( 9) = 0.4479943584047823081127973066095648846385D+00
      Omega(10) = 0.6477664122253113216059854095885128799637D+00
      Omega(11) = 0.9367556700653058413311678742640964401289D+00
      Omega(12) = 1.3597062225198743081475485916520540286001D+00
      Omega(13) = 1.9993413337952957502209258566772120957467D+00
      Omega(14) = 3.0515526253989090257456168187033540561970D+00
      Omega(15) = 5.2744587859215596903524059246848310067435D+00
C
      Alpha( 1) = 0.0045165373581371195146511878480699575533D+00
      Alpha( 2) = 0.0242166963980014430311528883174587711835D+00
      Alpha( 3) = 0.0614863974926384792736035710125630515677D+00
      Alpha( 4) = 0.1199639025272229584994981496648414065476D+00
      Alpha( 5) = 0.2060825487666099330608549672350093828754D+00
      Alpha( 6) = 0.3304703765697770167825315923426643394123D+00
      Alpha( 7) = 0.5097429488165322793180068983165398321944D+00
      Alpha( 8) = 0.7686998504455993988624014356947355963712D+00
      Alpha( 9) = 1.1432889758299511601645492842038720482378D+00
      Alpha(10) = 1.6850715529907800840176898549138684302307D+00
      Alpha(11) = 2.4683859599626540547168129791089086211286D+00
      Alpha(12) = 3.6028637238078778854723227542677932433435D+00
      Alpha(13) = 5.2590120105798029950978922375526281030034D+00
      Alpha(14) = 7.7351118438454861121582328831181030182051D+00
      Alpha(15) = 11.7207866995992235304921424976498656178592D+00
C
      Xi( 1) = 1.004024611678951089564423848621D+00
      Xi( 2) = 1.036429045999876775410904194974D+00
      Xi( 3) = 1.102350204991204553946898547778D+00
      Xi( 4) = 1.204042429449802128474042439699D+00
      Xi( 5) = 1.344974603354369209972339616055D+00
      Xi( 6) = 1.529962545222354810929565882560D+00
      Xi( 7) = 1.765373718388322756846740835268D+00
      Xi( 8) = 2.059407784772918241066735900446D+00
      Xi( 9) = 2.422453711469328537604292495011D+00
      Xi(10) = 2.867528498348190244774530421878D+00
      Xi(11) = 3.410814667030763296540901707843D+00
      Xi(12) = 4.072320586331676467613249759125D+00
      Xi(13) = 4.876690443441787359973860205287D+00
      Xi(14) = 5.854191786781036360796243167570D+00
      Xi(15) = 7.041895129523705379546300031635D+00
      Xi(16) = 8.485050052605370951054608585906D+00
      Xi(17) = 10.238621125821589888903184739632D+00
      Xi(18) = 12.368868448315555632656403961533D+00
      Xi(19) = 14.954699426572277638769481455938D+00
      Xi(20) = 18.088216348532182266989343233377D+00
      Xi(21) = 21.873325591020155268112690194471D+00
      Xi(22) = 26.420300894221018182458915468658D+00
      Xi(23) = 31.832600608164070669220402365340D+00
      Xi(24) = 38.180038355519728431042292271513D+00
      Xi(25) = 45.450194896975374420772242345379D+00
      Xi(26) = 53.471031272100935448354785961556D+00
      Xi(27) = 61.809268988957774559195312846782D+00
      Xi(28) = 69.683444375138042757578116237482D+00
      Xi(29) = 75.986543115418312147046897564451D+00
      Xi(30) = 79.535272972414583214639804964463D+00
      GOTO 9999
C
C     ----- K = 15 , R = 9E1 -----
C
 1519 CONTINUE
C
      RIni = 9.0D+01
C
      Omega( 1) = 0.0105926344340905414396767953051636812489D+00
      Omega( 2) = 0.0255826077357476880509386859413600845414D+00
      Omega( 3) = 0.0430707967103431928921556792050351347712D+00
      Omega( 4) = 0.0652489962492777935605691753961909995496D+00
      Omega( 5) = 0.0954874765697908190449443542602292467336D+00
      Omega( 6) = 0.1387519341256913148869266139140421500997D+00
      Omega( 7) = 0.2019522835098053309253350248719804937991D+00
      Omega( 8) = 0.2944915569107425864375533058892031590403D+00
      Omega( 9) = 0.4294022509808315279000896780292606536023D+00
      Omega(10) = 0.6253657255484110376233392036837699379248D+00
      Omega(11) = 0.9103406991822887626931645743688648053649D+00
      Omega(12) = 1.3293276205302154366078709601950436081097D+00
      Omega(13) = 1.9652732153735464103519459122537682560505D+00
      Omega(14) = 3.0139343774001751512237046171094334567897D+00
      Omega(15) = 5.2313239633687951277499217361821592930937D+00
C
      Alpha( 1) = 0.0041101270238754405812767959548653662338D+00
      Alpha( 2) = 0.0220571300018926773574333508737277398382D+00
      Alpha( 3) = 0.0560989284387557866623367271857736326979D+00
      Alpha( 4) = 0.1097490263250038257858073875716797829227D+00
      Alpha( 5) = 0.1892657547100838417368333377321931720871D+00
      Alpha( 6) = 0.3050446106979405895738306081499402466761D+00
      Alpha( 7) = 0.4733880519672660061917222884542155725285D+00
      Alpha( 8) = 0.7186895877224460323670456041167398097969D+00
      Alpha( 9) = 1.0764340399712193814540928116230134037323D+00
      Alpha(10) = 1.5977464735297438055234772735779813501722D+00
      Alpha(11) = 2.3566601731303012598472967908946884563193D+00
      Alpha(12) = 3.4627257320720155233783471571129553012725D+00
      Alpha(13) = 5.0866235331733391220911111307856344865286D+00
      Alpha(14) = 7.5268981403874479869584024349649098439841D+00
      Alpha(15) = 11.4726163407158323241999409169977752753766D+00
C
      Xi( 1) = 1.004162582029408795234945439567D+00
      Xi( 2) = 1.037685695187326312513738379995D+00
      Xi( 3) = 1.105924293733365469899303235035D+00
      Xi( 4) = 1.211295943883145047174389186129D+00
      Xi( 5) = 1.357524220232578484029301035818D+00
      Xi( 6) = 1.549786974818863132582341668453D+00
      Xi( 7) = 1.794946143148746320523413688708D+00
      Xi( 8) = 2.101863015200717751560996471305D+00
      Xi( 9) = 2.481799495764264427933795675862D+00
      Xi(10) = 2.948914285225161044465963522399D+00
      Xi(11) = 3.520871665147989925835278479660D+00
      Xi(12) = 4.219599171866740438662651824231D+00
      Xi(13) = 5.072222186071038071975003447278D+00
      Xi(14) = 6.112213647380998372739963064859D+00
      Xi(15) = 7.380786110372765320471882821707D+00
      Xi(16) = 8.928541269462804010233203388935D+00
      Xi(17) = 10.817355927958625714732643352534D+00
      Xi(18) = 13.122403041454299945506312141674D+00
      Xi(19) = 15.934039739813314534577770587020D+00
      Xi(20) = 19.358966647485198544348383720148D+00
      Xi(21) = 23.519425979528879025112764189487D+00
      Xi(22) = 28.548115907535518760707038410374D+00
      Xi(23) = 34.574509264277191612074746629446D+00
      Xi(24) = 41.695498324481614336317614188943D+00
      Xi(25) = 49.920019397288935589923619673414D+00
      Xi(26) = 59.077229894963870584362330973249D+00
      Xi(27) = 68.689937397766962591527661885493D+00
      Xi(28) = 77.856287843186458938471439239493D+00
      Xi(29) = 85.257300548335234142982574923053D+00
      Xi(30) = 89.449614211362739858712789242645D+00
      GOTO 9999
C
C     ----- K = 15 , R = 1E2 -----
C
 1521 CONTINUE
C
      RIni = 1.0D+02
C
      Omega( 1) = 0.0097353035250252180240644063989596812547D+00
      Omega( 2) = 0.0235509356290765406682197842178072555441D+00
      Omega( 3) = 0.0397775784086478460848312552200489733423D+00
      Omega( 4) = 0.0605621311489281332204545355707425713376D+00
      Omega( 5) = 0.0892266961232228339703288245987611304599D+00
      Omega( 6) = 0.1306713372983350476879077853298305456065D+00
      Omega( 7) = 0.1917178993499250586664836917827337003928D+00
      Omega( 8) = 0.2816922494803035432217423739142958538650D+00
      Omega( 9) = 0.4136087055236950441158751828885087320486D+00
      Omega(10) = 0.6062180228309940962258366148329713496423D+00
      Omega(11) = 0.8876415482272522335343949151376108375189D+00
      Omega(12) = 1.3031121145509258732637924982356025793706D+00
      Omega(13) = 1.9357875430219505574541347114880807112058D+00
      Omega(14) = 2.9813305239671294517497801557226466684369D+00
      Omega(15) = 5.1939564819246461309550089158193486582604D+00
C
      Alpha( 1) = 0.0037767636667319992393943901441352717363D+00
      Alpha( 2) = 0.0202847833501038873942951728038264214149D+00
      Alpha( 3) = 0.0516732608069974953441424840583362509960D+00
      Alpha( 4) = 0.1013462127489563033004463070418932346684D+00
      Alpha( 5) = 0.1754064696475346228572754839203362564604D+00
      Alpha( 6) = 0.2840385871952036498256505270276761621062D+00
      Alpha( 7) = 0.4432561489279913867446111797043784008565D+00
      Alpha( 8) = 0.6770770588279350320204397206680368981324D+00
      Alpha( 9) = 1.0205594158683729179987187407618876022752D+00
      Alpha(10) = 1.5244258129485720603974591158547013947100D+00
      Alpha(11) = 2.2624235257354227235197796641585910037975D+00
      Alpha(12) = 3.3440185071922190686473375098586302556214D+00
      Alpha(13) = 4.9400405999879862563306520684136557974853D+00
      Alpha(14) = 7.3492824024054162557542668743337799241999D+00
      Alpha(15) = 11.2603722455184587251594829915291029465152D+00
C
      Xi( 1) = 1.004286491374597387635052292421D+00
      Xi( 2) = 1.038814735511330400170686150130D+00
      Xi( 3) = 1.109137987553001864350459004793D+00
      Xi( 4) = 1.217825521188258722335602535214D+00
      Xi( 5) = 1.368837939395539215060686177861D+00
      Xi( 6) = 1.567690825204025039415860198755D+00
      Xi( 7) = 1.821708863181571224059557279062D+00
      Xi( 8) = 2.140374694778018385991555372350D+00
      Xi( 9) = 2.535773253686438767889183010240D+00
      Xi(10) = 3.023142394209262089803738615679D+00
      Xi(11) = 3.621555637363941586087320856358D+00
      Xi(12) = 4.354770211928382454922431543665D+00
      Xi(13) = 5.252288561299911683315916333115D+00
      Xi(14) = 6.350669235945679522679918616035D+00
      Xi(15) = 7.695131241623746833676461553964D+00
      Xi(16) = 9.341479685750591948657783891719D+00
      Xi(17) = 11.358348485559450371032430560092D+00
      Xi(18) = 13.829675812307640164582067665577D+00
      Xi(19) = 16.857155155790341884147753859224D+00
      Xi(20) = 20.562053289041963013983327357437D+00
      Xi(21) = 25.085103676531217059553213744039D+00
      Xi(22) = 30.581891161466432309951057888497D+00
      Xi(23) = 37.208942577635263544016952508287D+00
      Xi(24) = 45.092168656907410474732333938164D+00
      Xi(25) = 54.265042940772657668185852131160D+00
      Xi(26) = 64.562322586547332983752944812750D+00
      Xi(27) = 75.467256659206015750873675074217D+00
      Xi(28) = 85.958283875905530331540393973455D+00
      Xi(29) = 94.496276014416706254261946185125D+00
      Xi(30) = 99.359980127812882196025601189149D+00
      GOTO 9999
C
C     ----- K = 15 , R = 2E2 -----
C
 1522 CONTINUE
C
      RIni = 2.0D+02
C
      Omega( 1) = 0.0055614445793698374450548323373599846065D+00
      Omega( 2) = 0.0136224431019640811167206750388614100444D+00
      Omega( 3) = 0.0235750487763078402954483245301808835848D+00
      Omega( 4) = 0.0372729718543466537764236382046512119359D+00
      Omega( 5) = 0.0576750231360127275212945803517516907277D+00
      Omega( 6) = 0.0891547071205389258431880071531505649318D+00
      Omega( 7) = 0.1378921395737442449544712247400113369622D+00
      Omega( 8) = 0.2127087840449548338320195917106048000278D+00
      Omega( 9) = 0.3264936972926197210798267278608619790248D+00
      Omega(10) = 0.4983953354087468980441934934821901492796D+00
      Omega(11) = 0.7575386633983470467809326343022746641509D+00
      Omega(12) = 1.1507045886255240824038587055255788982322D+00
      Omega(13) = 1.7626459217391767304210573041878262756654D+00
      Omega(14) = 2.7889482630945156801157563952742179935740D+00
      Omega(15) = 4.9738042967548596619796474893604454337037D+00
C
      Alpha( 1) = 0.0021545257831798762967911889214256415670D+00
      Alpha( 2) = 0.0116436500968750751715380365412799701375D+00
      Alpha( 3) = 0.0300215936149693332600643181154453920101D+00
      Alpha( 4) = 0.0600297576601244908521570232962094593177D+00
      Alpha( 5) = 0.1067865632474907834297927870792488391771D+00
      Alpha( 6) = 0.1790529516984699501594684861727202473958D+00
      Alpha( 7) = 0.2908199897036503363398005084228259420343D+00
      Alpha( 8) = 0.4634860058463758151179896077609399185349D+00
      Alpha( 9) = 0.7291591909739004999271676232819316965106D+00
      Alpha(10) = 1.1357391474843713363284541939535188248556D+00
      Alpha(11) = 1.7548471860364001617118775988579670865875D+00
      Alpha(12) = 2.6951161158064083005030109330135701384279D+00
      Alpha(13) = 4.1281869152446315049571901045766253446345D+00
      Alpha(14) = 6.3546659737279791273223406022907511214726D+00
      Alpha(15) = 10.0614643180308471551165361290713917696849D+00
C
      Xi( 1) = 1.005107922869164440056455323003D+00
      Xi( 2) = 1.046310545611985985594433179457D+00
      Xi( 3) = 1.130535493023324185518987960997D+00
      Xi( 4) = 1.261482275618622608988339539593D+00
      Xi( 5) = 1.444886048402232851394118295918D+00
      Xi( 6) = 1.688814172466229769696381690647D+00
      Xi( 7) = 2.004129361118252874301992605410D+00
      Xi( 8) = 2.405128123552147019655056414678D+00
      Xi( 9) = 2.910359053439188724114738970883D+00
      Xi(10) = 3.543659615510939683282951029319D+00
      Xi(11) = 4.335486604412859002518482176569D+00
      Xi(12) = 5.324633615376642651901423208827D+00
      Xi(13) = 6.560455318249405778224803142962D+00
      Xi(14) = 8.105735776721632048456844810147D+00
      Xi(15) = 10.040372369390934146130001103359D+00
      Xi(16) = 12.466079169573248256776276665647D+00
      Xi(17) = 15.512331663565415596807428588733D+00
      Xi(18) = 19.343746232804397278473573784652D+00
      Xi(19) = 24.168954687767984453039549208597D+00
      Xi(20) = 30.250590684210382804233963049967D+00
      Xi(21) = 37.914995874992405845704812605845D+00
      Xi(22) = 47.557900487020893991879644246978D+00
      Xi(23) = 59.637264311460038802154004500267D+00
      Xi(24) = 74.634464513473317601177647873101D+00
      Xi(25) = 92.946581539845967906421364546077D+00
      Xi(26) = 114.646564926806013524906990852514D+00
      Xi(27) = 139.035396209208599857531929444576D+00
      Xi(28) = 163.995366802160568442392474253211D+00
      Xi(29) = 185.507461262228860315870448971509D+00
      Xi(30) = 198.288101188496654087023429724468D+00
      GOTO 9999
C
C     ----- K = 15 , R = 3E2 -----
C
 1523 CONTINUE
C
      RIni = 3.0D+02
C
      Omega( 1) = 0.0039963763487845303396661785985677362021D+00
      Omega( 2) = 0.0098751009059687778298861852294132646080D+00
      Omega( 3) = 0.0173873966346357320469510942054225388631D+00
      Omega( 4) = 0.0282209747987641637653863282281241353644D+00
      Omega( 5) = 0.0450958506221992216564387057076146447798D+00
      Omega( 6) = 0.0720399636167920214333383957527878749261D+00
      Omega( 7) = 0.1148595378783730234880018118703848273299D+00
      Omega( 8) = 0.1820981220823806225609297823608834221432D+00
      Omega( 9) = 0.2865444576764013857471304763668840109858D+00
      Omega(10) = 0.4475017573487411423637828689070516929860D+00
      Omega(11) = 0.6945970376068834315798350420578799457871D+00
      Omega(12) = 1.0754922855491432295968121102269776656613D+00
      Omega(13) = 1.6759871195093390723466000857833080317505D+00
      Omega(14) = 2.6919852463147228120721177191398965078406D+00
      Omega(15) = 4.8630599466714542812063870602656834307709D+00
C
      Alpha( 1) = 0.0015467042849176954962383268563953786501D+00
      Alpha( 2) = 0.0083952816472138907269735634032736815158D+00
      Alpha( 3) = 0.0218334981051808871674497786879998173504D+00
      Alpha( 4) = 0.0442655123280556494751321848502012556992D+00
      Alpha( 5) = 0.0802749348423698695900101000855197952433D+00
      Alpha( 6) = 0.1377946023687582154495026576235261472902D+00
      Alpha( 7) = 0.2296169463850606289392897174650443048449D+00
      Alpha( 8) = 0.3756070866312024175714696233585598861282D+00
      Alpha( 9) = 0.6061451901696484984233617709925567851315D+00
      Alpha(10) = 0.9674225764379703575634658618831451803999D+00
      Alpha(11) = 1.5296736237746061503727651453843350282114D+00
      Alpha(12) = 2.4008394966710964383953264000837179992232D+00
      Alpha(13) = 3.7528692640339534856985803656925781979226D+00
      Alpha(14) = 5.8874881288607278330068939276031869667349D+00
      Alpha(15) = 9.4913359153416843577016903665821700997185D+00
C
      Xi( 1) = 1.005588000700964566532825961165D+00
      Xi( 2) = 1.050700460413906080670183584846D+00
      Xi( 3) = 1.143117239043653853361459660221D+00
      Xi( 4) = 1.287300366525003658874949652446D+00
      Xi( 5) = 1.490190781701487541133915870617D+00
      Xi( 6) = 1.761611321882728561330662409024D+00
      Xi( 7) = 2.114898299916266595326053145953D+00
      Xi( 8) = 2.567770977709930953040984302760D+00
      Xi( 9) = 3.143453961412974615504539732669D+00
      Xi(10) = 3.872120270522848010279551300705D+00
      Xi(11) = 4.792775182696712790025861394838D+00
      Xi(12) = 5.955736699951859882025406722761D+00
      Xi(13) = 7.425902118476742126782635944338D+00
      Xi(14) = 9.287043146829721957692671541196D+00
      Xi(15) = 11.647443240130051000068289734912D+00
      Xi(16) = 14.647289559199265777077181294175D+00
      Xi(17) = 18.468311302909319914986041588634D+00
      Xi(18) = 23.346241742412311582360873174480D+00
      Xi(19) = 29.586656240765413485027668372140D+00
      Xi(20) = 37.584421748846409350175434838093D+00
      Xi(21) = 47.845911426365556187029204338046D+00
      Xi(22) = 61.010313042861925997284444633806D+00
      Xi(23) = 77.859417012040960090479835287169D+00
      Xi(24) = 99.289393374895234027632806572683D+00
      Xi(25) = 126.185078099482405285247654802561D+00
      Xi(26) = 159.077326811573789830389635824304D+00
      Xi(27) = 197.392752284127000569902143922718D+00
      Xi(28) = 238.160369496851600151776651159707D+00
      Xi(29) = 274.652937169310398690669217103277D+00
      Xi(30) = 296.972486408022178033805360541919D+00
      GOTO 9999
C
C     ----- K = 15 , R = 4E2 -----
C
 1524 CONTINUE
C
      RIni = 4.0D+02
C
      Omega( 1) = 0.0031580665533451926515155902194798009930D+00
      Omega( 2) = 0.0078585696681408003474823896310208493077D+00
      Omega( 3) = 0.0140296686661902682585376881535532689327D+00
      Omega( 4) = 0.0232449783409302702253388566225322797720D+00
      Omega( 5) = 0.0380499461297272989146402329130314257100D+00
      Omega( 6) = 0.0622288885678109713890226312171805034268D+00
      Omega( 7) = 0.1013387615850393206381584859654410024632D+00
      Omega( 8) = 0.1637323506241231536829580650582727230358D+00
      Omega( 9) = 0.2621075905769351542702605417645145280403D+00
      Omega(10) = 0.4158369767907242735846887193851273423206D+00
      Omega(11) = 0.6548585051238816585948525339500037034668D+00
      Omega(12) = 1.0274341017750817477149904988209527800791D+00
      Omega(13) = 1.6201338736773287715897673422205116366968D+00
      Omega(14) = 2.6292153158930104208963057788039918705181D+00
      Omega(15) = 4.7914402331023540574511665468548926583026D+00
C
      Alpha( 1) = 0.0012213088418495537004662592817962052383D+00
      Alpha( 2) = 0.0066522128401389255792803818605196664038D+00
      Alpha( 3) = 0.0174210360526302611842069728229076552850D+00
      Alpha( 4) = 0.0357153196385866714841468647773092826014D+00
      Alpha( 5) = 0.0657608477231415411610950146059639820351D+00
      Alpha( 6) = 0.1149208824288120524064957250764962282119D+00
      Alpha( 7) = 0.1951651703896346168066186846368204044211D+00
      Alpha( 8) = 0.3253123101170385742929996863992414546374D+00
      Alpha( 9) = 0.5345514314848428274692428152636836102829D+00
      Alpha(10) = 0.8678657429638241914264008836088493126226D+00
      Alpha(11) = 1.3944649527164186738345877381028969921317D+00
      Alpha(12) = 2.2217212147441035808518444705228489510773D+00
      Alpha(13) = 3.5217221726634938554385823028525237532449D+00
      Alpha(14) = 5.5969790205896105357227798826613707205979D+00
      Alpha(15) = 9.1341806528607349079107446954139959416352D+00
C
      Xi( 1) = 1.005925747734232925958107462971D+00
      Xi( 2) = 1.053792964858323062660692970294D+00
      Xi( 3) = 1.152003033768909901531779516048D+00
      Xi( 4) = 1.305600412397938297755055347960D+00
      Xi( 5) = 1.522451809161981181595102441406D+00
      Xi( 6) = 1.813738335860105252558480137814D+00
      Xi( 7) = 2.194730223645267108874279404773D+00
      Xi( 8) = 2.685850738406786398105446922457D+00
      Xi( 9) = 3.314060265518352417545358545858D+00
      Xi(10) = 4.114650008002854532883352245420D+00
      Xi(11) = 5.133618600349286557708861078453D+00
      Xi(12) = 6.430834442002402406585864769184D+00
      Xi(13) = 8.084251009730162189359514712805D+00
      Xi(14) = 10.195517362799900396887198272822D+00
      Xi(15) = 12.897451390950307111536721205169D+00
      Xi(16) = 16.363992429124257377975881766474D+00
      Xi(17) = 20.823427766632239457184061492967D+00
      Xi(18) = 26.575895448033863759790262903948D+00
      Xi(19) = 34.016317083385521735650058872125D+00
      Xi(20) = 43.663749518874066371521003304679D+00
      Xi(21) = 56.197286985580479544422072279986D+00
      Xi(22) = 72.495673530117732012711950773109D+00
      Xi(23) = 93.669558817274358585025506584998D+00
      Xi(24) = 121.054980507308547643674501159694D+00
      Xi(25) = 156.089802970307589546905546740163D+00
      Xi(26) = 199.898086658202700019448805335287D+00
      Xi(27) = 252.257887331837507352561011941816D+00
      Xi(28) = 309.590264512515617623122921031609D+00
      Xi(29) = 362.414818019072917348877282961439D+00
      Xi(30) = 395.470504041337537898259668622813D+00
      GOTO 9999
C
C     ----- K = 15 , R = 5E2 -----
C
 1525 CONTINUE
C
      RIni = 5.0D+02
C
      Omega( 1) = 0.0026298460390556482285692014265465754619D+00
      Omega( 2) = 0.0065832442531841001603625115022599967318D+00
      Omega( 3) = 0.0118917603535314008359361306367552035823D+00
      Omega( 4) = 0.0200429042853591930952703069682141467922D+00
      Omega( 5) = 0.0334464729571471454632868719469840357306D+00
      Omega( 6) = 0.0557039547836739697006937059586828553392D+00
      Omega( 7) = 0.0921899550749297709830386814144898011136D+00
      Omega( 8) = 0.1511113941004667048442766663640668411972D+00
      Omega( 9) = 0.2450839575131379099846460831169103755656D+00
      Omega(10) = 0.3935114036889438912799558251132125974436D+00
      Omega(11) = 0.6265467178768755856219964395226185160936D+00
      Omega(12) = 0.9928992278647892220011170194560179425025D+00
      Omega(13) = 1.5797450704374967286620218276382843214378D+00
      Omega(14) = 2.5836775326140330290639046673106804519193D+00
      Omega(15) = 4.7395149729995405176591338758385063556489D+00
C
      Alpha( 1) = 0.0010163654596489672351419041398425113787D+00
      Alpha( 2) = 0.0055523279214756996604944981610341159239D+00
      Alpha( 3) = 0.0146272132671542568534833739235523353273D+00
      Alpha( 4) = 0.0302730130416823495081177306659236947439D+00
      Alpha( 5) = 0.0564510822625912328077502164852541355344D+00
      Alpha( 6) = 0.1000982261047091139777786769626821694601D+00
      Alpha( 7) = 0.1725709802348770900411265710738817347192D+00
      Alpha( 8) = 0.2919086142773966053448348595411587780291D+00
      Alpha( 9) = 0.4864039250725653168189871466964291357726D+00
      Alpha(10) = 0.8001129679399844055404447673573997690255D+00
      Alpha(11) = 1.3014354844107493782605702858923280018644D+00
      Alpha(12) = 2.0972637457991409630150836385098500613822D+00
      Alpha(13) = 3.3597478567908137022518522130098972411361D+00
      Alpha(14) = 5.3919954067373262818554935371651026798645D+00
      Alpha(15) = 8.8808449536686026783843805887386224640068D+00
C
      Xi( 1) = 1.006185134482228618699878319731D+00
      Xi( 2) = 1.056170294653800441364056850180D+00
      Xi( 3) = 1.158846606707571684513093102087D+00
      Xi( 4) = 1.319732090518601833723476290938D+00
      Xi( 5) = 1.547448630422906595046489419687D+00
      Xi( 6) = 1.854292454339231302159508862903D+00
      Xi( 7) = 2.257132513234196740390183344438D+00
      Xi( 8) = 2.778646613633750587623455485797D+00
      Xi( 9) = 3.448931732848780591389054039908D+00
      Xi(10) = 4.307615916055986267797078825836D+00
      Xi(11) = 5.406675184063368204672589412141D+00
      Xi(12) = 6.814220012068835528786653332745D+00
      Xi(13) = 8.619583540384217638929964522276D+00
      Xi(14) = 10.940160941995193255929508246993D+00
      Xi(15) = 13.930605331835086672584433920008D+00
      Xi(16) = 17.795230093903935139273975174490D+00
      Xi(17) = 22.804705821009911772753731007413D+00
      Xi(18) = 29.318505212387904381582393043004D+00
      Xi(19) = 37.814909504949985735722250979052D+00
      Xi(20) = 48.930435047373750816201987134946D+00
      Xi(21) = 63.510051366181364620316918134790D+00
      Xi(22) = 82.666614648042439313091467312233D+00
      Xi(23) = 107.839124642511327741012383540919D+00
      Xi(24) = 140.815561601189142976187085309903D+00
      Xi(25) = 183.626414330187036447195048083358D+00
      Xi(26) = 238.081941187831344838854441547937D+00
      Xi(27) = 304.485675751855637494180939484067D+00
      Xi(28) = 378.875066774860891433140963613369D+00
      Xi(29) = 449.051122170990711673921325086667D+00
      Xi(30) = 493.813224037873900545969618747222D+00
      GOTO 9999
C
C     ----- K = 15 , R = 6E2 -----
C
 1526 CONTINUE
C
      RIni = 6.0D+02
C
      Omega( 1) = 0.0022640577225830361791103777398259655484D+00
      Omega( 2) = 0.0056973303057685757995652294817523486614D+00
      Omega( 3) = 0.0103981057606504449930884324652946570922D+00
      Omega( 4) = 0.0177852165866605353940134871121303561381D+00
      Omega( 5) = 0.0301587536348224576396694421309696565459D+00
      Omega( 6) = 0.0509766295013259356908627128318611809732D+00
      Omega( 7) = 0.0854716363601831396576056186498160371912D+00
      Omega( 8) = 0.1417322693899287700447422316818446574871D+00
      Omega( 9) = 0.2322996708593134201984711437183328541778D+00
      Omega(10) = 0.3765894209406675952208319591019147765110D+00
      Omega(11) = 0.6049134328425471823472217569594988617609D+00
      Omega(12) = 0.9663336751410720168236831439934064746922D+00
      Omega(13) = 1.5485236223280244059015073321283750829025D+00
      Omega(14) = 2.5483852865162736312746455391575750581978D+00
      Omega(15) = 4.6992901048617441548023443420589728702907D+00
C
      Alpha( 1) = 0.0008744958294475131544913113867782683553D+00
      Alpha( 2) = 0.0047897439227468614920500548145371189079D+00
      Alpha( 3) = 0.0126845371835364479919219847861566741187D+00
      Alpha( 4) = 0.0264715244466399859632374210252625523765D+00
      Alpha( 5) = 0.0499047966992862909300129447415228511886D+00
      Alpha( 6) = 0.0895847119517327264103770563530293458143D+00
      Alpha( 7) = 0.1563859893279255842447883484425119604566D+00
      Alpha( 8) = 0.2677347395888903809732690952660760785875D+00
      Alpha( 9) = 0.4512116353157648715921281684648747045685D+00
      Alpha(10) = 0.7501246982213588193133882442431570325425D+00
      Alpha(11) = 1.2322057652185713550905751811903598991194D+00
      Alpha(12) = 2.0039333213152113665964093591753680811962D+00
      Alpha(13) = 3.2374803343641888637039621956503765431989D+00
      Alpha(14) = 5.2364298101630870137419837551107093531755D+00
      Alpha(15) = 8.6878037141362704012803597208858263911679D+00
C
      Xi( 1) = 1.006394949628333896555869297895D+00
      Xi( 2) = 1.058094769159536815278377375193D+00
      Xi( 3) = 1.164394712621561042784762429747D+00
      Xi( 4) = 1.331212710369555388589986322234D+00
      Xi( 5) = 1.567809937489857727765593664593D+00
      Xi( 6) = 1.887431734725242153064700412202D+00
      Xi( 7) = 2.308315630383043569546011597460D+00
      Xi( 8) = 2.855079921818981010138080445238D+00
      Xi( 9) = 3.560540732373147095567167408348D+00
      Xi(10) = 4.468107410429185140376961982156D+00
      Xi(11) = 5.635007538206450348854115173225D+00
      Xi(12) = 7.136647265147478675067355124995D+00
      Xi(13) = 9.072510834617849169968228917327D+00
      Xi(14) = 11.574152398492492466522407745799D+00
      Xi(15) = 14.816034673758016742585452707459D+00
      Xi(16) = 19.030255105054156827479316937968D+00
      Xi(17) = 24.526614737872571635896035147084D+00
      Xi(18) = 31.719891037520669218804925115762D+00
      Xi(19) = 41.166832617257441157693209632384D+00
      Xi(20) = 53.615734679994806032715670696120D+00
      Xi(21) = 70.071262088330390674362213587756D+00
      Xi(22) = 91.874723600885937958848703743797D+00
      Xi(23) = 120.790763010020509561626145256241D+00
      Xi(24) = 159.065016092999220592707487753614D+00
      Xi(25) = 209.346561553769042726691473887968D+00
      Xi(26) = 274.199941891441432817444479042024D+00
      Xi(27) = 354.593579173934398657275224309160D+00
      Xi(28) = 446.383309832086461438338531593217D+00
      Xi(29) = 534.726829720322987760106059340615D+00
      Xi(30) = 592.020706939949882929852265078807D+00
      GOTO 9999
C
C     ----- K = 15 , R = 7E2 -----
C
 1527 CONTINUE
C
      RIni = 7.0D+02
C
      Omega( 1) = 0.0019945380638779841443940742378576613802D+00
      Omega( 2) = 0.0050427907113674821369335269944517152751D+00
      Omega( 3) = 0.0092889724596230179876020223266042119548D+00
      Omega( 4) = 0.0160950930477551877167139218749447593382D+00
      Omega( 5) = 0.0276699612914006832440879171675085501647D+00
      Omega( 6) = 0.0473548764340851198299548602266284547113D+00
      Omega( 7) = 0.0802675698972774872053700771779016776009D+00
      Omega( 8) = 0.1343967267489980976953589664191213159938D+00
      Omega( 9) = 0.2222156818745986620039630995981916328219D+00
      Omega(10) = 0.3631409974153666139469551904506161577046D+00
      Omega(11) = 0.5876075064254369875347758234163109136716D+00
      Omega(12) = 0.9449657115229299850505013946833088311905D+00
      Omega(13) = 1.5233094274329127094458682933897364364384D+00
      Omega(14) = 2.5198228955401023394702203495043590919522D+00
      Omega(15) = 4.6667463424862905396056644846680683258455D+00
C
      Alpha( 1) = 0.0007699970490877884595638549179458937077D+00
      Alpha( 2) = 0.0042272630399960448487346850905688189215D+00
      Alpha( 3) = 0.0112479671026250682619917944243814744709D+00
      Alpha( 4) = 0.0236490750857580692246098265363318802201D+00
      Alpha( 5) = 0.0450157333067533311525749463444245890287D+00
      Alpha( 6) = 0.0816731939775164522392499576175506348363D+00
      Alpha( 7) = 0.1441034743319927689098438119974510129850D+00
      Alpha( 8) = 0.2492317095072272726277934698502214416749D+00
      Alpha( 9) = 0.4240516392151889676529700340390505175492D+00
      Alpha(10) = 0.7112470016203497646706428725327242545973D+00
      Alpha(11) = 1.1779826380950750911152241640067472872033D+00
      Alpha(12) = 1.9303738608286636095276445068691373307956D+00
      Alpha(13) = 3.1405939910687392466887013497967018338386D+00
      Alpha(14) = 5.1126181849943693725167592756264411946177D+00
      Alpha(15) = 8.5336606461802805142710659502824910305208D+00
C
      Xi( 1) = 1.006570622484552558046320969165D+00
      Xi( 2) = 1.059707104699230406316158570323D+00
      Xi( 3) = 1.169048586031044273956039369899D+00
      Xi( 4) = 1.340859496970923495562030836048D+00
      Xi( 5) = 1.584956307384439728644896783205D+00
      Xi( 6) = 1.915411687398691456107185926427D+00
      Xi( 7) = 2.351662563056146397271958803366D+00
      Xi( 8) = 2.920036522584414024174617741103D+00
      Xi( 9) = 3.655754474561851317167965325083D+00
      Xi(10) = 4.605592026462496899573417774576D+00
      Xi(11) = 5.831476847449754423620416066143D+00
      Xi(12) = 7.415385607590361347497137645135D+00
      Xi(13) = 9.466007129254602976697885452495D+00
      Xi(14) = 12.127816592998331805819400841528D+00
      Xi(15) = 15.593475081790864815271813181852D+00
      Xi(16) = 20.120808972607252549172351940854D+00
      Xi(17) = 26.056100760949900310703419847869D+00
      Xi(18) = 33.866121529556390601367610315720D+00
      Xi(19) = 44.181978651464860967379433276392D+00
      Xi(20) = 57.858847845664609296778557734342D+00
      Xi(21) = 76.055565702005308074962197650848D+00
      Xi(22) = 100.336391411221727130997916077604D+00
      Xi(23) = 132.787913973737655648488242832173D+00
      Xi(24) = 176.115722850641647431735314910384D+00
      Xi(25) = 233.605229592472310090611742339206D+00
      Xi(26) = 308.627273030173973150658284225756D+00
      Xi(27) = 402.929885356120519035449945022265D+00
      Xi(28) = 512.369532146612567280463679253444D+00
      Xi(29) = 619.558212002491375869350775928979D+00
      Xi(30) = 690.107176530776788569987445498555D+00
      GOTO 9999
C
C     ----- K = 15 , R = 8E2 -----
C
 1528 CONTINUE
C
      RIni = 8.0D+02
C
      Omega( 1) = 0.0017870212715594367768755893768027609436D+00
      Omega( 2) = 0.0045376002737008805430341375295594241734D+00
      Omega( 3) = 0.0084290300565706590425532336412206557696D+00
      Omega( 4) = 0.0147750621270927640636918989729131634903D+00
      Omega( 5) = 0.0257069116625649650275796821025098282121D+00
      Omega( 6) = 0.0444686874865422564499697205553374601550D+00
      Omega( 7) = 0.0760818511689729804129528854816921779047D+00
      Omega( 8) = 0.1284487982735185641241029697035003209749D+00
      Omega( 9) = 0.2139808776813502482990782260774409451187D+00
      Omega(10) = 0.3520892514244785430257284053467969897611D+00
      Omega(11) = 0.5733069875972591514191531447686145384068D+00
      Omega(12) = 0.9272271445141610459101701169348785924740D+00
      Omega(13) = 1.5023065731054268420827796370886630938912D+00
      Omega(14) = 2.4959879568923889199376364711469022950041D+00
      Omega(15) = 4.6395958902014889061078684040495545559679D+00
C
      Alpha( 1) = 0.0006895611127582455120514943612726466160D+00
      Alpha( 2) = 0.0037937712630365261380920618690740231393D+00
      Alpha( 3) = 0.0101383023615175421161443470620400209015D+00
      Alpha( 4) = 0.0214609580225112899923365760573856775295D+00
      Alpha( 5) = 0.0412052639215424465863278884886211628213D+00
      Alpha( 6) = 0.0754655648230115803606118246582568076519D+00
      Alpha( 7) = 0.1343950641624070088567081959762816723014D+00
      Alpha( 8) = 0.2344981727691005559934206772809694996340D+00
      Alpha( 9) = 0.4022718771382511915063635854483692355643D+00
      Alpha(10) = 0.6798660151460762128333356613119065059436D+00
      Alpha(11) = 1.1339537645619051447164193913152985260240D+00
      Alpha(12) = 1.8703271767807595980228838050685169491771D+00
      Alpha(13) = 3.0611464206734718420353791312749081043876D+00
      Alpha(14) = 5.0107183382202085176505834684945739354589D+00
      Alpha(15) = 8.4064479210506800003349647809613998106215D+00
C
      Xi( 1) = 1.006721380456283441583338611469D+00
      Xi( 2) = 1.061091517463179429944276910280D+00
      Xi( 3) = 1.173048717630027889353636072478D+00
      Xi( 4) = 1.349163118214007318451490291533D+00
      Xi( 5) = 1.599742715330653725991708058451D+00
      Xi( 6) = 1.939594452789270081160231951056D+00
      Xi( 7) = 2.389224015730618983291422274817D+00
      Xi( 8) = 2.976488293178805926266286552284D+00
      Xi( 9) = 3.738771278074708967953040938959D+00
      Xi(10) = 4.725886599094059556565305957321D+00
      Xi(11) = 6.004028122378091374480857389528D+00
      Xi(12) = 7.661167731594584548535792345447D+00
      Xi(13) = 9.814433817982832590487429769155D+00
      Xi(14) = 12.620225238511970131116690652107D+00
      Xi(15) = 16.288094165448597394618701628133D+00
      Xi(16) = 21.099879428735870950245567811976D+00
      Xi(17) = 27.436164885629690362786603508027D+00
      Xi(18) = 35.812882916516620902813716753599D+00
      Xi(19) = 46.932016422984952682545545243897D+00
      Xi(20) = 61.751301777339101615332817019066D+00
      Xi(21) = 81.578810242068995624287186529955D+00
      Xi(22) = 108.196517656471522729477729996006D+00
      Xi(23) = 144.008829305045290447573336223286D+00
      Xi(24) = 192.181578658054551173028912103291D+00
      Xi(25) = 256.648884145825445735056291596266D+00
      Xi(26) = 341.628757450284282204400199134398D+00
      Xi(27) = 449.745499932875416104760546431862D+00
      Xi(28) = 577.021055192766996244646549030222D+00
      Xi(29) = 703.632253977297076108499140900676D+00
      Xi(30) = 788.083377984554580786280553184042D+00
      GOTO 9999
C
C     ----- K = 15 , R = 9E2 -----
C
 1529 CONTINUE
C
      RIni = 9.0D+02
C
      Omega( 1) = 0.0016219084670863559936738798323944965585D+00
      Omega( 2) = 0.0041347531000776380092693500570288911433D+00
      Omega( 3) = 0.0077404596712594351285001176721928262126D+00
      Omega( 4) = 0.0137109856821038252554885092382441769132D+00
      Omega( 5) = 0.0241104755830180065159318390094478345986D+00
      Omega( 6) = 0.0421003703750971199021118068078761709216D+00
      Omega( 7) = 0.0726196964790066316062285695087652470647D+00
      Omega( 8) = 0.1234948615707579749908406711866870608674D+00
      Omega( 9) = 0.2070802540837517613519927831000622120428D+00
      Omega(10) = 0.3427777482480647935046431246108866730538D+00
      Omega(11) = 0.5612008838336487104238477219286806985110D+00
      Omega(12) = 0.9121508407153058013770545453535731894590D+00
      Omega(13) = 1.4844032778764126925818978008919657440856D+00
      Omega(14) = 2.4756385101249463417168866774531466035114D+00
      Omega(15) = 4.6164204896316331934967569505801066043205D+00
C
      Alpha( 1) = 0.0006255780084365577944930178347707205866D+00
      Alpha( 2) = 0.0034485635069017050178818035175223877253D+00
      Alpha( 3) = 0.0092527900519063576451213835871723034643D+00
      Alpha( 4) = 0.0197090055472030227261737069660574306340D+00
      Alpha( 5) = 0.0381394934269198958134755497034751314800D+00
      Alpha( 6) = 0.0704408742294657181496112495489025917550D+00
      Alpha( 7) = 0.1264852816914941078895286488692484283547D+00
      Alpha( 8) = 0.2224163364079585390541528647223579184811D+00
      Alpha( 9) = 0.3843020768245482766045303946178535170475D+00
      Alpha(10) = 0.6538272042959846833674354693588526288295D+00
      Alpha(11) = 1.0972315718134093568336337454205420272046D+00
      Alpha(12) = 1.8200161955372063935561927072726007281744D+00
      Alpha(13) = 2.9943196982235439904725810000840624525154D+00
      Alpha(14) = 4.9247349165318251985705433337869862953085D+00
      Alpha(15) = 8.2988518304647784879987670869638805015711D+00
C
      Xi( 1) = 1.006853176555457203339080662907D+00
      Xi( 2) = 1.062302369921885844810532928406D+00
      Xi( 3) = 1.176550488474161181560485867070D+00
      Xi( 4) = 1.356441416966551330685990250924D+00
      Xi( 5) = 1.612724004189677166377914208084D+00
      Xi( 6) = 1.960865988460154367379814632777D+00
      Xi( 7) = 2.422337979587958453273066616873D+00
      Xi( 8) = 3.026382766910526195009573102723D+00
      Xi( 9) = 3.812350949066554055581421134313D+00
      Xi(10) = 4.832831650830515514156299206050D+00
      Xi(11) = 6.157930403730347043711468302263D+00
      Xi(12) = 7.881141126194877331075788751136D+00
      Xi(13) = 10.127409080838835894050609098826D+00
      Xi(14) = 13.064222849896538832144277098024D+00
      Xi(15) = 16.916925988773785146171424464967D+00
      Xi(16) = 21.989942371927765676561938867906D+00
      Xi(17) = 28.696263145898042862258492391447D+00
      Xi(18) = 37.598595376417315718636791288532D+00
      Xi(19) = 49.466694830718208695380377903916D+00
      Xi(20) = 65.357108688094277620939109851861D+00
      Xi(21) = 86.722585457944927449236427463575D+00
      Xi(22) = 115.557983430261986559917097849848D+00
      Xi(23) = 154.581192990693060101969891206863D+00
      Xi(24) = 207.417307737267877260078918766339D+00
      Xi(25) = 278.657932364932122315526186184798D+00
      Xi(26) = 373.401149784383102586682312562516D+00
      Xi(27) = 495.229957264965882840490607463835D+00
      Xi(28) = 640.482160013576152790548690063588D+00
      Xi(29) = 787.017026953280821321978777405093D+00
      Xi(30) = 885.957745722293947610026521033433D+00
      GOTO 9999
C
C     ----- K = 15 , R = 1E3 -----
C
 1531 CONTINUE
C
      RIni = 1.0D+03
C
      Omega( 1) = 0.0014871445606266374690648170305527775437D+00
      Omega( 2) = 0.0038052862730260423568116905828143115009D+00
      Omega( 3) = 0.0071751648320479873574389824506664092851D+00
      Omega( 4) = 0.0128319872157548835371414751305163548523D+00
      Omega( 5) = 0.0227811357330662754630274790396127215786D+00
      Omega( 6) = 0.0401125718299626029395140922040319253483D+00
      Omega( 7) = 0.0696934521423355049618672714395462008952D+00
      Omega( 8) = 0.1192823247668556892327976909351505696577D+00
      Omega( 9) = 0.2011809721541527566244979968734796216268D+00
      Omega(10) = 0.3347795767086457424199872484615880807723D+00
      Omega(11) = 0.5507588922017481879883117612539678020767D+00
      Omega(12) = 0.8991015649090148173744156678832695206438D+00
      Omega(13) = 1.4688669259558777996929959663852116591443D+00
      Omega(14) = 2.4579547964873604457523847965916274915799D+00
      Omega(15) = 4.5962843562528531344592297447348983041593D+00
C
      Alpha( 1) = 0.0005733678386173403022237547407066043936D+00
      Alpha( 2) = 0.0031665856795662918647250551806501817076D+00
      Alpha( 3) = 0.0085280855110995089536301909903526219381D+00
      Alpha( 4) = 0.0182707634739886854995414644867168796338D+00
      Alpha( 5) = 0.0356113865418937814485283999133002907911D+00
      Alpha( 6) = 0.0662745665478196662695844522394938991283D+00
      Alpha( 7) = 0.1198882170585382153727761235517679239138D+00
      Alpha( 8) = 0.2122814420767166918117005783295248022569D+00
      Alpha( 9) = 0.3691460181815040449419991486212921927290D+00
      Alpha(10) = 0.6317554902189304127609927042019677401186D+00
      Alpha(11) = 1.0659628718482381493862926058469042800425D+00
      Alpha(12) = 1.7770046979357533404068544835574527951394D+00
      Alpha(13) = 2.9369927346261072438182021526742460082460D+00
      Alpha(14) = 4.8507701846577407742318321925978352737729D+00
      Alpha(15) = 8.2061044496867307255702495005778018821729D+00
C
      Xi( 1) = 1.006970073202688794916821735104D+00
      Xi( 2) = 1.063376780618819135437735923233D+00
      Xi( 3) = 1.179660120678657380158781131652D+00
      Xi( 4) = 1.362911892369016381935248327650D+00
      Xi( 5) = 1.624280629534539719662986079207D+00
      Xi( 6) = 1.979835147831775864603678916076D+00
      Xi( 7) = 2.451926363891509618288602756486D+00
      Xi( 8) = 3.071065244455815873613271915765D+00
      Xi( 9) = 3.878407421790135628687778979184D+00
      Xi(10) = 4.929098830293953303199883109187D+00
      Xi(11) = 6.296863303020070916100198177290D+00
      Xi(12) = 8.080323533238966634585642712096D+00
      Xi(13) = 10.411711474326583890155639711139D+00
      Xi(14) = 13.468900109276450950505921166922D+00
      Xi(15) = 17.492097426706340018709440187195D+00
      Xi(16) = 22.807059904425902811114790225133D+00
      Xi(17) = 29.857586391445431261287835056351D+00
      Xi(18) = 39.251016323276317603746532469700D+00
      Xi(19) = 51.822205030656594941518289232363D+00
      Xi(20) = 68.723077412386748143158232693395D+00
      Xi(21) = 91.547000401919210600676368727591D+00
      Xi(22) = 122.497060792932926868992193902841D+00
      Xi(23) = 164.600407649826112041480108416636D+00
      Xi(24) = 221.939527402204062070389944949511D+00
      Xi(25) = 299.769727474212866719094350287378D+00
      Xi(26) = 404.095987991380787812678576642611D+00
      Xi(27) = 539.531615295786462271898642484302D+00
      Xi(28) = 702.867585330041904578823874771842D+00
      Xi(29) = 869.767525781364489290137242960554D+00
      Xi(30) = 983.737118578837241467116569992868D+00
      GOTO 9999
C
C     ----- K = 15 , R = 2E3 -----
C
 1532 CONTINUE
C
      RIni = 2.0D+03
C
      Omega( 1) = 0.0008404149095994821547922493110149844142D+00
      Omega( 2) = 0.0022127172568996186612089172029982364620D+00
      Omega( 3) = 0.0044044157718100057294363658400015992811D+00
      Omega( 4) = 0.0084252777388026441757196138837571197655D+00
      Omega( 5) = 0.0159295973722688583018580743555503431708D+00
      Omega( 6) = 0.0295938800680031685597093371598020894453D+00
      Omega( 7) = 0.0538511112433189692572099716016964521259D+00
      Omega( 8) = 0.0960186198421827258320249143253022339195D+00
      Omega( 9) = 0.1680232026699381431278368381754262372851D+00
      Omega(10) = 0.2891110509850164378420345201448071748018D+00
      Omega(11) = 0.4903019557844095488441382713062921538949D+00
      Omega(12) = 0.8226597473598823428986293038178700953722D+00
      Omega(13) = 1.3770544599011862452186960581457242369652D+00
      Omega(14) = 2.3529519089229520290018626837991178035736D+00
      Omega(15) = 4.4767751352432112454948764934670180082321D+00
C
      Alpha( 1) = 0.0003230240034515234528317284556919730676D+00
      Alpha( 2) = 0.0018095714502588270203298215221821010346D+00
      Alpha( 3) = 0.0050161890329175661332183899787651171209D+00
      Alpha( 4) = 0.0112212887847932644980186722705184365623D+00
      Alpha( 5) = 0.0230170410722087548371916909673018381000D+00
      Alpha( 6) = 0.0451147565214759307949954347805032739416D+00
      Alpha( 7) = 0.0857044983738741855683329617932031396776D+00
      Alpha( 8) = 0.1587370998704414615243507569175562821329D+00
      Alpha( 9) = 0.2876082176004111645717387091281125321984D+00
      Alpha(10) = 0.5110180594894182526388703990960493683815D+00
      Alpha(11) = 0.8923202702949009079702591407112777233124D+00
      Alpha(12) = 1.5349453620910749052086430310737341642380D+00
      Alpha(13) = 2.6106708645312535210791793360840529203415D+00
      Alpha(14) = 4.4258490015686451357623809599317610263824D+00
      Alpha(15) = 7.6696259660669774405050702625885605812073D+00
C
      Xi( 1) = 1.007712242881242170611244546308D+00
      Xi( 2) = 1.070207907157320281754664392793D+00
      Xi( 3) = 1.199485052179940947070235779393D+00
      Xi( 4) = 1.404322731537668511592187314818D+00
      Xi( 5) = 1.698605010097730350082656514132D+00
      Xi( 6) = 2.102551993204441416196459613275D+00
      Xi( 7) = 2.644662161603932748477063796599D+00
      Xi( 8) = 3.364400549289334740876711293822D+00
      Xi( 9) = 4.315818878600615526863748527830D+00
      Xi(10) = 5.572556903872480482675655366620D+00
      Xi(11) = 7.234871749462114642881260806462D+00
      Xi(12) = 9.439540609574310181528744578827D+00
      Xi(13) = 12.373830976687880678355213603936D+00
      Xi(14) = 16.295323830542574938817779184319D+00
      Xi(15) = 21.560143014460372512530739186332D+00
      Xi(16) = 28.663489744676368786713283043355D+00
      Xi(17) = 38.298132791621640080848010256886D+00
      Xi(18) = 51.439448936787506738710362697020D+00
      Xi(19) = 69.469806604976923836147761903703D+00
      Xi(20) = 94.361356140622973498466308228672D+00
      Xi(21) = 128.945200347420382058771792799234D+00
      Xi(22) = 177.305932388017566836424521170557D+00
      Xi(23) = 245.348832019322287578688701614738D+00
      Xi(24) = 341.570568196029398677637800574303D+00
      Xi(25) = 477.944629773722795107460115104914D+00
      Xi(26) = 670.340593697570739095681346952915D+00
      Xi(27) = 936.217380956610213615931570529938D+00
      Xi(28) = 1282.582934856395695533137768507004D+00
      Xi(29) = 1669.807290680049618458724580705166D+00
      Xi(30) = 1957.242597377555057391873560845852D+00
      GOTO 9999
C
C     ----- K = 15 , R = 3E3 -----
C
 1533 CONTINUE
C
      RIni = 3.0D+03
C
      Omega( 1) = 0.0006021998451342176979288824166872129418D+00
      Omega( 2) = 0.0016187261267440271098155723703514163869D+00
      Omega( 3) = 0.0033452393365352763509459692835290578472D+00
      Omega( 4) = 0.0066742118282262347194911810029094301200D+00
      Omega( 5) = 0.0130868192116284632019591839188765725055D+00
      Omega( 6) = 0.0250590910423157118485320117920456262084D+00
      Omega( 7) = 0.0467969004387706694784250429502980850316D+00
      Omega( 8) = 0.0853675374355371224287312818812534231938D+00
      Omega( 9) = 0.1524652909186951293615819302829095249763D+00
      Omega(10) = 0.2672108204861978688227520245535018261762D+00
      Omega(11) = 0.4607495497177949561194883199011229635289D+00
      Omega(12) = 0.7846876596876935452571852347425362950162D+00
      Omega(13) = 1.3308927436829058854042204251832970385294D+00
      Omega(14) = 2.2998057567870609319091862454698116380314D+00
      Omega(15) = 4.4163157384143941802841226706988209116389D+00
C
      Alpha( 1) = 0.0002309479118724317000477037857141156263D+00
      Alpha( 2) = 0.0013072884159532110557924429287968148827D+00
      Alpha( 3) = 0.0037002878105363975695076878989417390464D+00
      Alpha( 4) = 0.0085257832976107738285915377263179948386D+00
      Alpha( 5) = 0.0180659572668002429355059738555477899524D+00
      Alpha( 6) = 0.0365339066158719588569810457490127930669D+00
      Alpha( 7) = 0.0714083393886806190101732873254380962180D+00
      Alpha( 8) = 0.1356885479689384899732677164374550216053D+00
      Alpha( 9) = 0.2515721314522706339163009292558825791275D+00
      Alpha(10) = 0.4563724471127592911205272246943920322337D+00
      Alpha(11) = 0.8120423512111979215421543598640852223980D+00
      Alpha(12) = 1.4209334533330517537754247858750034083641D+00
      Alpha(13) = 2.4545240138803349828119232389767034874239D+00
      Alpha(14) = 4.2199332485013572769001755080608973003109D+00
      Alpha(15) = 7.4072221155356338874627131207972752235946D+00
C
      Xi( 1) = 1.008121593556959773850013883933D+00
      Xi( 2) = 1.073982970148324786159935162466D+00
      Xi( 3) = 1.210481226968480986810883237581D+00
      Xi( 4) = 1.427410523604222229959552492584D+00
      Xi( 5) = 1.740314102870956596263302129657D+00
      Xi( 6) = 2.171961277409063295878305366315D+00
      Xi( 7) = 2.754678113087707313870466929728D+00
      Xi( 8) = 3.533583750261958335986411938734D+00
      Xi( 9) = 4.570995213173794194903842846855D+00
      Xi(10) = 5.952600854260180783094486001161D+00
      Xi(11) = 7.796249000786654706398465242145D+00
      Xi(12) = 10.264479569558108359300652168145D+00
      Xi(13) = 13.582447085113456232072226548269D+00
      Xi(14) = 18.063705211190198193435541007545D+00
      Xi(15) = 24.147529218143520997330586652652D+00
      Xi(16) = 32.453334343040926238815879756316D+00
      Xi(17) = 43.860699401935573860794592704337D+00
      Xi(18) = 59.627926216284838630693476702049D+00
      Xi(19) = 81.569212009265394253132441804155D+00
      Xi(20) = 112.321436402374068085074920730904D+00
      Xi(21) = 155.748476914038984877941551587810D+00
      Xi(22) = 217.555340133537157429333142033556D+00
      Xi(23) = 306.215300407791237269350403948920D+00
      Xi(24) = 434.329701855966633927685904836835D+00
      Xi(25) = 620.453803362545718458331833744523D+00
      Xi(26) = 890.859483103901161116233708980872D+00
      Xi(27) = 1278.348735289984839380572623213084D+00
      Xi(28) = 1807.026081072200998534071914036758D+00
      Xi(29) = 2432.807995726973129935188921990630D+00
      Xi(30) = 2924.603774735733815237637145401095D+00
      GOTO 9999
C
C     ----- K = 15 , R = 4E3 -----
C
 1534 CONTINUE
C
      RIni = 4.0D+03
C
      Omega( 1) = 0.0004756369084263004952714521495968358344D+00
      Omega( 2) = 0.0013003520804791979888086181946053876413D+00
      Omega( 3) = 0.0027672864591600532465548082207765223650D+00
      Omega( 4) = 0.0056928470782110608657607554033974750496D+00
      Omega( 5) = 0.0114501120106099499038605055677153554683D+00
      Omega( 6) = 0.0223884810149639784851015745364977149734D+00
      Omega( 7) = 0.0425633913822724507076359495849615655061D+00
      Omega( 8) = 0.0788704848951632592053109967189516993358D+00
      Omega( 9) = 0.1428378098730392881229780271956641968245D+00
      Omega(10) = 0.2534844400338091604662094785682668884874D+00
      Omega(11) = 0.4420178490944162802921656163501751279910D+00
      Omega(12) = 0.7603902042987330134218675048440871933053D+00
      Omega(13) = 1.3011427010528306348786931057048832371947D+00
      Omega(14) = 2.2654170754188351810838597533148686125060D+00
      Omega(15) = 4.3772021220065278606509151515524536080193D+00
C
      Alpha( 1) = 0.0001820782633968936153992700384776116940D+00
      Alpha( 2) = 0.0010395111289733385836581521692936758150D+00
      Alpha( 3) = 0.0029925180411342110725723009850840483637D+00
      Alpha( 4) = 0.0070545320321262105725314023752653413979D+00
      Alpha( 5) = 0.0153116412226620900789464700328390556194D+00
      Alpha( 6) = 0.0316633381176961214803716824356394354822D+00
      Alpha( 7) = 0.0631367385810755350547128668392549144528D+00
      Alpha( 8) = 0.1221173957421378924965505233246609861908D+00
      Alpha( 9) = 0.2300168021593796421436729435217038997052D+00
      Alpha(10) = 0.4232217341282906642363405985696189759437D+00
      Alpha(11) = 0.7627288970210382018644068580215389374644D+00
      Alpha(12) = 1.3501284949179842941177284720843942977808D+00
      Alpha(13) = 2.3566519253922217683704098956098960115924D+00
      Alpha(14) = 4.0899108319618529120732797199622154948884D+00
      Alpha(15) = 7.2406374424386336823358800707239879557164D+00
C
      Xi( 1) = 1.008399536089696037796485661442D+00
      Xi( 2) = 1.076549175536408965432426387832D+00
      Xi( 3) = 1.217972693848358699828318685299D+00
      Xi( 4) = 1.443188471259828498842806010494D+00
      Xi( 5) = 1.768929003052049432273551410333D+00
      Xi( 6) = 2.219804307045161887888481966336D+00
      Xi( 7) = 2.830928382091189915957529787960D+00
      Xi( 8) = 3.651573113310687771718848759761D+00
      Xi( 9) = 4.750178905489354046470101833322D+00
      Xi(10) = 6.221452381810463394973959028178D+00
      Xi(11) = 8.196544979260530701045683876771D+00
      Xi(12) = 10.857702771770388778041493971216D+00
      Xi(13) = 14.459395840240733399975625594891D+00
      Xi(14) = 19.359032227629489508052840207597D+00
      Xi(15) = 26.061870515171585991134883819598D+00
      Xi(16) = 35.287269826709305186457354963636D+00
      Xi(17) = 48.067285203290741328607804661033D+00
      Xi(18) = 65.894695807480156861390963030090D+00
      Xi(19) = 90.947445315536196770722643378804D+00
      Xi(20) = 126.432109629553380517053007281447D+00
      Xi(21) = 177.114281615756022761343224658503D+00
      Xi(22) = 250.142546888428821502681209665298D+00
      Xi(23) = 356.329570250085996191424797530090D+00
      Xi(24) = 512.113092873752612621540691861810D+00
      Xi(25) = 742.394997837475936641737916943384D+00
      Xi(26) = 1083.912131223594253581765656235802D+00
      Xi(27) = 1585.976659734769377396723655238020D+00
      Xi(28) = 2294.047326326232192128529163710482D+00
      Xi(29) = 3168.504659462954313076110679503472D+00
      Xi(30) = 3887.092106025369653066192654478073D+00
      GOTO 9999
C
C     ----- K = 15 , R = 5E3 -----
C
 1535 CONTINUE
C
      RIni = 5.0D+03
C
      Omega( 1) = 0.0003962677505883613429462231368986466218D+00
      Omega( 2) = 0.0010992903894092523588024720520260613110D+00
      Omega( 3) = 0.0023969412766162124992569204063111443226D+00
      Omega( 4) = 0.0050508717428056814312016932575627459201D+00
      Omega( 5) = 0.0103583403993260641099671804342408076849D+00
      Omega( 6) = 0.0205783676303624219185660869368970082860D+00
      Omega( 7) = 0.0396555901930640382722231557927061729174D+00
      Omega( 8) = 0.0743565312997897404507499294346040130677D+00
      Omega( 9) = 0.1360810125724631901750509219151119566504D+00
      Omega(10) = 0.2437641389423446734556217424527702064552D+00
      Omega(11) = 0.4286480516510043328097610132454420295289D+00
      Omega(12) = 0.7429320310955790694976141885774723050417D+00
      Omega(13) = 1.2796585097760807599870380713014128559735D+00
      Omega(14) = 2.2405122931894283274528723026719490007963D+00
      Omega(15) = 4.3488779989433205140809957089942372476798D+00
C
      Alpha( 1) = 0.0001514563506470363833414023745690263345D+00
      Alpha( 2) = 0.0008711260387687643076721269984070278314D+00
      Alpha( 3) = 0.0025442361670791670130245825005854687362D+00
      Alpha( 4) = 0.0061115723260027562432065333444808352681D+00
      Alpha( 5) = 0.0135201559930968971137509428495637742174D+00
      Alpha( 6) = 0.0284475633694058737973404341753402846393D+00
      Alpha( 7) = 0.0575988032150850329750802099326723038075D+00
      Alpha( 8) = 0.1129165520128868640980959563158148029061D+00
      Alpha( 9) = 0.2152382813196922147522087420679071101404D+00
      Alpha(10) = 0.4002657944347219287585179503530241618137D+00
      Alpha(11) = 0.7282787055024561484635470254289657532354D+00
      Alpha(12) = 1.3002835551871088776154650190619577188045D+00
      Alpha(13) = 2.2873048529384315526256227224521921925771D+00
      Alpha(14) = 3.9973073734529804231820071525049797855900D+00
      Alpha(15) = 7.1215488770918427764884672814815758101759D+00
C
      Xi( 1) = 1.008607548065553133798295804002D+00
      Xi( 2) = 1.078471333272938473789304136830D+00
      Xi( 3) = 1.223592701433025819340340289987D+00
      Xi( 4) = 1.455050730582966612980388987353D+00
      Xi( 5) = 1.790501748102732074976670140209D+00
      Xi( 6) = 2.255993869817991644447993659917D+00
      Xi( 7) = 2.888831206626912981782692724231D+00
      Xi( 8) = 3.741566112838891348035472161904D+00
      Xi( 9) = 4.887511008943874586674116500262D+00
      Xi(10) = 6.428593700672319048283059883531D+00
      Xi(11) = 8.506705591175765453736123955508D+00
      Xi(12) = 11.320115966500891183620092217410D+00
      Xi(13) = 15.147348828771383073138467878493D+00
      Xi(14) = 20.382100579139743773068871313114D+00
      Xi(15) = 27.584741503126332868786607654954D+00
      Xi(16) = 37.558943283391386352976226703504D+00
      Xi(17) = 51.466733475857053891483472440882D+00
      Xi(18) = 71.002995935133440194575449311287D+00
      Xi(19) = 98.662941095424492379817316134449D+00
      Xi(20) = 138.156390239855341187102943933951D+00
      Xi(21) = 195.056436564772982242277343800652D+00
      Xi(22) = 277.823760078639739795614005402058D+00
      Xi(23) = 399.432469610330092707473070845481D+00
      Xi(24) = 579.931423862097276067384399311777D+00
      Xi(25) = 850.331708131169334008436777594397D+00
      Xi(26) = 1257.744948155719920124262500848999D+00
      Xi(27) = 1868.606025334137808568968353029049D+00
      Xi(28) = 2752.647564697004047618023037102830D+00
      Xi(29) = 3882.125886910784884786806969714235D+00
      Xi(30) = 4845.405490236185681585112661196035D+00
      GOTO 9999
C
C     ----- K = 15 , R = 6E3 -----
C
 1536 CONTINUE
C
      RIni = 6.0D+03
C
      Omega( 1) = 0.0003414757206547954683262053514423289835D+00
      Omega( 2) = 0.0009596636276879566365309104716829757820D+00
      Omega( 3) = 0.0021365303957790902949651025335281451234D+00
      Omega( 4) = 0.0045917780797535652267040849790202816649D+00
      Omega( 5) = 0.0095656214187066115749196642051299654952D+00
      Omega( 6) = 0.0192478522560639908174721359954140176285D+00
      Omega( 7) = 0.0374963311723699641913535706566618088686D+00
      Omega( 8) = 0.0709749985723212378787802014079577617167D+00
      Omega( 9) = 0.1309799832669097009464056688421429441860D+00
      Omega(10) = 0.2363753024370086878894019200014753323558D+00
      Omega(11) = 0.4184236484184497331640338602110062993233D+00
      Omega(12) = 0.7295129765209068313947665673779852113512D+00
      Omega(13) = 1.2630808671191903606347006761012607967132D+00
      Omega(14) = 2.2212530740729632757098077622259779673186D+00
      Omega(15) = 4.3269755853895644857193270027551079692785D+00
C
      Alpha( 1) = 0.0001303310340679438245633309440604102880D+00
      Alpha( 2) = 0.0007546147404157980901201372922210836025D+00
      Alpha( 3) = 0.0022321298968959612833828091071550203672D+00
      Alpha( 4) = 0.0054484287226041676528279819168715247812D+00
      Alpha( 5) = 0.0122450024578017911861159351845362852629D+00
      Alpha( 6) = 0.0261311329097184906015960675212950192758D+00
      Alpha( 7) = 0.0535658283228252219509534903102432323863D+00
      Alpha( 8) = 0.1061504639425160218839578642191501245406D+00
      Alpha( 9) = 0.2042761733917355039167294827873178064692D+00
      Alpha(10) = 0.3831072245356744632616575324535190816277D+00
      Alpha(11) = 0.7023544400177715455249605192022954724962D+00
      Alpha(12) = 1.2625539646581452843951756670470842891518D+00
      Alpha(13) = 2.2345533125141977257629294362217819980287D+00
      Alpha(14) = 3.9265876434223163664229833136332104004396D+00
      Alpha(15) = 7.0303439652448463047761106903266181689105D+00
C
      Xi( 1) = 1.008772408116241343792665596801D+00
      Xi( 2) = 1.079995719377837837043773716594D+00
      Xi( 3) = 1.228055056080842786158865975743D+00
      Xi( 4) = 1.464485279493109726482377719314D+00
      Xi( 5) = 1.807695931805995871169065458695D+00
      Xi( 6) = 2.284912192165491337525473625192D+00
      Xi( 7) = 2.935238146811181312451077829451D+00
      Xi( 8) = 3.813937210931115801395077657965D+00
      Xi( 9) = 4.998363927309790924642662890420D+00
      Xi(10) = 6.596474533666179375021587638628D+00
      Xi(11) = 8.759169045930881813272206315446D+00
      Xi(12) = 11.698258701309225672700098375145D+00
      Xi(13) = 15.712703638637112711108623219758D+00
      Xi(14) = 21.227257307300300803731829546450D+00
      Xi(15) = 28.849805069089485847280696084738D+00
      Xi(16) = 39.457250974277042272070348261082D+00
      Xi(17) = 54.325448841075961631946755758804D+00
      Xi(18) = 75.327824329634581163039896978262D+00
      Xi(19) = 105.242487364381162057158647371580D+00
      Xi(20) = 148.232500152187237532630526004596D+00
      Xi(21) = 210.606114800487202343148851468868D+00
      Xi(22) = 302.032690156142679560602459787333D+00
      Xi(23) = 437.503310897037233806461387075615D+00
      Xi(24) = 640.486655729559991812038788339123D+00
      Xi(25) = 947.880459891898872692994615363205D+00
      Xi(26) = 1417.021223231072227566329502224107D+00
      Xi(27) = 2131.793653271545705640832579774724D+00
      Xi(28) = 3188.324684138302722358204732699960D+00
      Xi(29) = 4577.025028872135528068554322089767D+00
      Xi(30) = 5799.999146855984597070943209473626D+00
      GOTO 9999
C
C     ----- K = 15 , R = 7E3 -----
C
 1537 CONTINUE
C
      RIni = 7.0D+03
C
      Omega( 1) = 0.0003011910603773180930263604154978676619D+00
      Omega( 2) = 0.0008564729701576365269785544678295130527D+00
      Omega( 3) = 0.0019419490399198079889074133375960473646D+00
      Omega( 4) = 0.0042438127675699630689655581644581516798D+00
      Omega( 5) = 0.0089572919260013106854156125086197093310D+00
      Omega( 6) = 0.0182166528963363917419782502318814643161D+00
      Omega( 7) = 0.0358089973084950708752329136264069830986D+00
      Omega( 8) = 0.0683137230812789659293985701504769991743D+00
      Omega( 9) = 0.1269403467537262602578143969034307758648D+00
      Omega(10) = 0.2304915026927615025837889350945886235422D+00
      Omega(11) = 0.4102423245034995822537286780384846451852D+00
      Omega(12) = 0.7187313324855581339514060124074035229569D+00
      Omega(13) = 1.2497198359483893011483840451703031249053D+00
      Omega(14) = 2.2057032208277742770657914306298152951058D+00
      Omega(15) = 4.3092920548625502363826456075912574306130D+00
C
      Alpha( 1) = 0.0001148082111075384276075927101853132312D+00
      Alpha( 2) = 0.0006687803525803460729510951823621744960D+00
      Alpha( 3) = 0.0020009370081711234145035142742777289904D+00
      Alpha( 4) = 0.0049528792338325030370404979812581913734D+00
      Alpha( 5) = 0.0112823279837394440386832625278223929399D+00
      Alpha( 6) = 0.0243649356433474875607402693189484921277D+00
      Alpha( 7) = 0.0504631660845739493800269498402499834810D+00
      Alpha( 8) = 0.1009037065377226192512357706709824611835D+00
      Alpha( 9) = 0.1957159032289775951447916707559571136699D+00
      Alpha(10) = 0.3696251344057429575890982159647180083084D+00
      Alpha(11) = 0.6818738938022625279658690822248701124408D+00
      Alpha(12) = 1.2326061877222592045449944819779375393409D+00
      Alpha(13) = 2.1925153876058049134522426326654453987430D+00
      Alpha(14) = 3.8700528921889102919491609799962361648795D+00
      Alpha(15) = 6.9572668947263066707535628108161063209991D+00
C
      Xi( 1) = 1.008908127362487775098565112586D+00
      Xi( 2) = 1.081251304022287459743675852586D+00
      Xi( 3) = 1.231734114590510359627840986096D+00
      Xi( 4) = 1.472274286191225951586809828076D+00
      Xi( 5) = 1.821915543211667064802707083349D+00
      Xi( 6) = 2.308876722519944008678999258777D+00
      Xi( 7) = 2.973788827437679623075933443843D+00
      Xi( 8) = 3.874220815397069276600147680334D+00
      Xi( 9) = 5.090980897802434691420958579045D+00
      Xi(10) = 6.737194374631522964923940888227D+00
      Xi(11) = 8.971532815073269094031938042377D+00
      Xi(12) = 12.017528487409227991485061171062D+00
      Xi(13) = 16.191946054764509448434250948878D+00
      Xi(14) = 21.946733172542159535575723161571D+00
      Xi(15) = 29.931619394205071591624767179951D+00
      Xi(16) = 41.088415407792347463361570802931D+00
      Xi(17) = 56.794532837777393287359561924177D+00
      Xi(18) = 79.083834064138912312014184635700D+00
      Xi(19) = 110.990594989350655014126445507827D+00
      Xi(20) = 157.091701137116315778463615515648D+00
      Xi(21) = 224.372574949777182182208967731185D+00
      Xi(22) = 323.626790368828476635032487251920D+00
      Xi(23) = 471.741810947603555898011151725768D+00
      Xi(24) = 695.440590353829698411214366160493D+00
      Xi(25) = 1037.302986492132237716923270909319D+00
      Xi(26) = 1564.716173465391656383083329728834D+00
      Xi(27) = 2379.177951955116393278899522556458D+00
      Xi(28) = 3604.806434049069072456106255231134D+00
      Xi(29) = 5255.568707099384338388148307785741D+00
      Xi(30) = 6751.198981108692237729229645992746D+00
      GOTO 9999
C
C     ----- K = 15 , R = 8E3 -----
C
 1538 CONTINUE
C
      RIni = 8.0D+03
C
      Omega( 1) = 0.0002702223025865613888936383161923707186D+00
      Omega( 2) = 0.0007767771880651208602113168201919873868D+00
      Omega( 3) = 0.0017901805651350507902784794304722193470D+00
      Omega( 4) = 0.0039690416370657308611156728964108886260D+00
      Omega( 5) = 0.0084718887508346092019530172885499741398D+00
      Omega( 6) = 0.0173869880469622443877852598428934705055D+00
      Omega( 7) = 0.0344420673061209607173889860054494249653D+00
      Omega( 8) = 0.0661449935763909648413303582481503894996D+00
      Omega( 9) = 0.1236312094677251008099428709885447830175D+00
      Omega(10) = 0.2256494857804086714162189811205827538743D+00
      Omega(11) = 0.4034823918551539799601159813624562389123D+00
      Omega(12) = 0.7097924619431462531682526517773368368580D+00
      Omega(13) = 1.2386136266646492081790198458435270367772D+00
      Omega(14) = 2.1927583985724614741227855851946060283808D+00
      Omega(15) = 4.2945711898528852608868111584428106652922D+00
C
      Alpha( 1) = 0.0001028812551570597490250812923917502623D+00
      Alpha( 2) = 0.0006026772000226355159545637738775086811D+00
      Alpha( 3) = 0.0018220066643375311307608615195817447496D+00
      Alpha( 4) = 0.0045663474136871346808813545287142332540D+00
      Alpha( 5) = 0.0105247448017949373921011953572635100329D+00
      Alpha( 6) = 0.0229632113568895219124569835947302642865D+00
      Alpha( 7) = 0.0479820641000258561322216847855326449235D+00
      Alpha( 8) = 0.0966799894828725857719316748306592756990D+00
      Alpha( 9) = 0.1887842738725176050963022733597540536721D+00
      Alpha(10) = 0.3586516625412090184996626029967359272632D+00
      Alpha(11) = 0.6651285739047628965572682424589601168918D+00
      Alpha(12) = 1.2080240390876300740933754762629348533665D+00
      Alpha(13) = 2.1578952260979739972545504711476382908586D+00
      Alpha(14) = 3.8233719362259296979233286206678599228326D+00
      Alpha(15) = 6.8968130432491910890464847927461278231931D+00
C
      Xi( 1) = 1.009022925974474242508983901079D+00
      Xi( 2) = 1.082313804505225142828066242640D+00
      Xi( 3) = 1.234849928760580450729303358948D+00
      Xi( 4) = 1.478878287117131369089208237089D+00
      Xi( 5) = 1.833989011332974671409506983721D+00
      Xi( 6) = 2.329259458577866743252821057908D+00
      Xi( 7) = 3.006643989535890256316677526804D+00
      Xi( 8) = 3.925714789753387274793758332514D+00
      Xi( 9) = 5.170291987700855087205231708225D+00
      Xi(10) = 6.858027474045047244648187767524D+00
      Xi(11) = 9.154418019810454968006319109719D+00
      Xi(12) = 12.293343604743625133678130989523D+00
      Xi(13) = 16.607340928101278137302143633747D+00
      Xi(14) = 22.572571322579393216303800251410D+00
      Xi(15) = 30.876210603182207016167981628030D+00
      Xi(16) = 42.518450554766183475258189972124D+00
      Xi(17) = 58.968566035945209131974298344403D+00
      Xi(18) = 82.406414797020995630094652995012D+00
      Xi(19) = 116.100887969858598533501314875593D+00
      Xi(20) = 165.010501379140145580826093407723D+00
      Xi(21) = 236.750066134443987078683768743304D+00
      Xi(22) = 343.166569791899711577842779774983D+00
      Xi(23) = 502.940653239856677730879042087508D+00
      Xi(24) = 745.904639267252305201427731162767D+00
      Xi(25) = 1120.132762222368250637494213606260D+00
      Xi(26) = 1702.880464722715736392011365296639D+00
      Xi(27) = 2613.319609582309821060874810427777D+00
      Xi(28) = 4004.791123529432345540257642824145D+00
      Xi(29) = 5919.534383978480728494275808770908D+00
      Xi(30) = 7699.252399038355023197510718091507D+00
      GOTO 9999
C
C     ----- K = 15 , R = 9E3 -----
C
 1539 CONTINUE
C
      RIni = 9.0D+03
C
      Omega( 1) = 0.0002456102897028886932859597773967999523D+00
      Omega( 2) = 0.0007131730524152647808370791436462926560D+00
      Omega( 3) = 0.0016679632044389639519804240932673855546D+00
      Omega( 4) = 0.0037453590318413544034607354053789673998D+00
      Omega( 5) = 0.0080731774364872896340873794848724820383D+00
      Omega( 6) = 0.0167006499760849031391811295513272206392D+00
      Omega( 7) = 0.0333046051797057354854048570602653711603D+00
      Omega( 8) = 0.0643311699365383485790261496040098876392D+00
      Omega( 9) = 0.1208512826462217910815865294649196215460D+00
      Omega(10) = 0.2215658405811709383422761915349852301915D+00
      Omega(11) = 0.3977616018252502192744084891229405798185D+00
      Omega(12) = 0.7022056961576497009455168074598674365916D+00
      Omega(13) = 1.2291664486410005130033115672283372532547D+00
      Omega(14) = 2.1817332841699960041797340393188164853200D+00
      Omega(15) = 4.2820334602316590202376878071532928515808D+00
C
      Alpha( 1) = 0.0000934068720182621284586280672491975796D+00
      Alpha( 2) = 0.0005500573606271143280303282958182409246D+00
      Alpha( 3) = 0.0016789264798342002106741011556965970097D+00
      Alpha( 4) = 0.0042550759956803765393242131205575606856D+00
      Alpha( 5) = 0.0099098689489630684085013073020276053171D+00
      Alpha( 6) = 0.0218171234245755086953886744960406929295D+00
      Alpha( 7) = 0.0459401420899353520498271160471537299941D+00
      Alpha( 8) = 0.0931839435803971711893048911201642425794D+00
      Alpha( 9) = 0.1830179615796721870422253397592271539907D+00
      Alpha(10) = 0.3494826683901751471369133417166352728600D+00
      Alpha(11) = 0.6510827012489291688011362746468790874133D+00
      Alpha(12) = 1.1873356150225313516023001314536600148131D+00
      Alpha(13) = 2.1286767960585158574945974230274714500410D+00
      Alpha(14) = 3.7838867200618920482932394655506413982948D+00
      Alpha(15) = 6.8455960408957202303729949743882343682344D+00
C
      Xi( 1) = 1.009122023844475767246145481160D+00
      Xi( 2) = 1.083231329301445447217504269855D+00
      Xi( 3) = 1.237542464500154619695161672954D+00
      Xi( 4) = 1.484590695549148624049137779046D+00
      Xi( 5) = 1.844444936881460817912695149268D+00
      Xi( 6) = 2.346937643605285799550466685481D+00
      Xi( 7) = 3.035188530648495507276163341004D+00
      Xi( 8) = 3.970540103055201205722521407004D+00
      Xi( 9) = 5.239480434723596254803501537367D+00
      Xi(10) = 6.963684535560758376900730715064D+00
      Xi(11) = 9.314733335740850671963242390916D+00
      Xi(12) = 12.535767831380382032226139532582D+00
      Xi(13) = 16.973493003353009556458119710953D+00
      Xi(14) = 23.125903476756856811877294255453D+00
      Xi(15) = 31.714084046546791672768761394252D+00
      Xi(16) = 43.791335855977624912788170874478D+00
      Xi(17) = 60.910906328881381961715124617740D+00
      Xi(18) = 85.386803918233194286846465814733D+00
      Xi(19) = 120.704718237470903959007717531904D+00
      Xi(20) = 172.177979834423158303491341314384D+00
      Xi(21) = 248.010207216664290336138876114092D+00
      Xi(22) = 361.041457459874378377318038246813D+00
      Xi(23) = 531.655741626293901425093224588636D+00
      Xi(24) = 792.666241711624215693365869128684D+00
      Xi(25) = 1197.468966272857073263047311684204D+00
      Xi(26) = 1833.004145658207951763429832681140D+00
      Xi(27) = 2836.113866479956576727161632334173D+00
      Xi(28) = 4390.328081800878949003674733830849D+00
      Xi(29) = 6570.312494082755212954083390286542D+00
      Xi(30) = 8644.355539147717990111630115279695D+00
      GOTO 9999
C
C     ----- K = 15 , R = 1E4 -----
C
 1541 CONTINUE
C
      RIni = 1.0D+04
C
      Omega( 1) = 0.0002255397353676053207922755160881253533D+00
      Omega( 2) = 0.0006611042867166782100040605708868302059D+00
      Omega( 3) = 0.0015670812397325413366538885838637196191D+00
      Omega( 4) = 0.0035589310274069557899554296174354783844D+00
      Omega( 5) = 0.0077382539600035213085181345048224610395D+00
      Omega( 6) = 0.0161205385911687138160550050031138979989D+00
      Omega( 7) = 0.0323382459969524620993670815938259011091D+00
      Omega( 8) = 0.0627833806660786870391163100313391964846D+00
      Omega( 9) = 0.1184699102140543149042604953447010984746D+00
      Omega(10) = 0.2180557225882068321363592075523563096340D+00
      Omega(11) = 0.3928295685505218414498677936830262069634D+00
      Omega(12) = 0.6956484406594744318900157276086559932082D+00
      Omega(13) = 1.2209854833881973573119980591705768802058D+00
      Omega(14) = 2.1721753157989217952565652591800926529686D+00
      Omega(15) = 4.2711641588735770264442792587544772686670D+00
C
      Alpha( 1) = 0.0000856839825970028654958073832074039872D+00
      Alpha( 2) = 0.0005070831754268762750049198586958787427D+00
      Alpha( 3) = 0.0015615827771326924017788347867803644409D+00
      Alpha( 4) = 0.0039981485316142193332287851047224247480D+00
      Alpha( 5) = 0.0093987693813625961427162067846063919774D+00
      Alpha( 6) = 0.0208582420419701535417784151599240871633D+00
      Alpha( 7) = 0.0442219155511724033146426626056380015939D+00
      Alpha( 8) = 0.0902273275897956522284483645013342112406D+00
      Alpha( 9) = 0.1781199701205341328968468142535108711400D+00
      Alpha(10) = 0.3416644402189818329634331262045776611558D+00
      Alpha(11) = 0.6390657647645775942392247503942570574509D+00
      Alpha(12) = 1.1695841604775098689517759398270868587133D+00
      Alpha(13) = 2.1035450462970592940079722632518155478465D+00
      Alpha(14) = 3.7498584977918745053497573405110188105027D+00
      Alpha(15) = 6.8013961440006963335598577646834428378497D+00
C
      Xi( 1) = 1.009208933910929662631193737266D+00
      Xi( 2) = 1.084036269593066855094522227265D+00
      Xi( 3) = 1.239906037642474827937220938079D+00
      Xi( 4) = 1.489609395458397731208319281215D+00
      Xi( 5) = 1.853640840419345385482914612130D+00
      Xi( 6) = 2.362505482913046469129608739834D+00
      Xi( 7) = 3.060362938579087490513844227458D+00
      Xi( 8) = 4.010140331786234245935174325126D+00
      Xi( 9) = 5.300718507532951047356295504187D+00
      Xi(10) = 7.057389212797474001108433139251D+00
      Xi(11) = 9.457224604316827683628265965865D+00
      Xi(12) = 12.751739902731012975614643956135D+00
      Xi(13) = 17.300504874605341535526159546521D+00
      Xi(14) = 23.621403717169431058994777394844D+00
      Xi(15) = 32.466521527743094306661664205649D+00
      Xi(16) = 44.937910036206841608852347036418D+00
      Xi(17) = 62.666224902694419925108793023583D+00
      Xi(18) = 88.089710166909260147249938199820D+00
      Xi(19) = 124.895778451849706613963064683048D+00
      Xi(20) = 178.729665715905407158414774926314D+00
      Xi(21) = 258.349246559639066311842015011280D+00
      Xi(22) = 377.534671007663655301067606728793D+00
      Xi(23) = 558.294837208122016192213266094768D+00
      Xi(24) = 836.308132639007831943267490260041D+00
      Xi(25) = 1270.133443643806666956130868584296D+00
      Xi(26) = 1956.215704141925878389862702988466D+00
      Xi(27) = 3049.016702403769551121825998052373D+00
      Xi(28) = 4763.025285042322653161761536466656D+00
      Xi(29) = 7209.027731618097082133544972748496D+00
      Xi(30) = 9586.668367498586480479616511729546D+00
      GOTO 9999
C
C     ----- K = 15 , R = 2E6 (Best : R = 1175000) -----
C
 1599 CONTINUE
C
      RIni = 2.0D+06
C
      Omega( 1) = 0.0000105228176670261742840088471752704995D+00
      Omega( 2) = 0.0000691367855732188023467920262056778807D+00
      Omega( 3) = 0.0002830208384686293506941606593681898252D+00
      Omega( 4) = 0.0009200464898989470742820623522587042409D+00
      Omega( 5) = 0.0025901722444491988862050644388743125823D+00
      Omega( 6) = 0.0065921351784279965347346512578863286791D+00
      Omega( 7) = 0.0155513212399444333759425873262016715159D+00
      Omega( 8) = 0.0345577940336581698850743678969044125893D+00
      Omega( 9) = 0.0731454596883880376191437722166455159822D+00
      Omega(10) = 0.1486679642694432365698700010403499049971D+00
      Omega(11) = 0.2920249462823511854634838530442308979218D+00
      Omega(12) = 0.5577584133433053689807464003980186362242D+00
      Omega(13) = 1.0451182163084206676361212484671625588817D+00
      Omega(14) = 1.9640378140134062713833412727737481873191D+00
      Omega(15) = 4.0343871534325909029329582100587003878900D+00
C
      Alpha( 1) = 0.0000032916678993378630075590744267626528D+00
      Alpha( 2) = 0.0000362116499423094770823232623331534505D+00
      Alpha( 3) = 0.0001916520413872042188449850493459775125D+00
      Alpha( 4) = 0.0007397073815295242332734412912982613353D+00
      Alpha( 5) = 0.0023688276459558680546775311656984563946D+00
      Alpha( 6) = 0.0066832262130187145182690696548766240426D+00
      Alpha( 7) = 0.0171791919033504756800207575670458304273D+00
      Alpha( 8) = 0.0410873650857746208155845454162991003955D+00
      Alpha( 9) = 0.0927379070681148618668409103205085841637D+00
      Alpha(10) = 0.1995445819374171568023499864774628065334D+00
      Alpha(11) = 0.4124340749566497427288855326388272715121D+00
      Alpha(12) = 0.8239287603035586122677234877098584320265D+00
      Alpha(13) = 1.6008068838318576734705903086286582492903D+00
      Alpha(14) = 3.0544270604919671998137009616414161428111D+00
      Alpha(15) = 5.8841569972355831702123807325222060171654D+00
C
      Xi( 1) = 1.011356431856024272897959237483D+00
      Xi( 2) = 1.104003921877126792355956752534D+00
      Xi( 3) = 1.298967172564399300840019824133D+00
      Xi( 4) = 1.616293799078014142671377850746D+00
      Xi( 5) = 2.088774395352814420342285606225D+00
      Xi( 6) = 2.766850077561704469375838955081D+00
      Xi( 7) = 3.726393897993726684247581260045D+00
      Xi( 8) = 5.080016080628945361781501066645D+00
      Xi( 9) = 6.994131337891847674612805230154D+00
      Xi(10) = 9.715521417601100629193866531352D+00
      Xi(11) = 13.612939230868090786583102236307D+00
      Xi(12) = 19.242846449322077027194177745173D+00
      Xi(13) = 27.454670425888878579692709003268D+00
      Xi(14) = 39.561788791582659235990249158021D+00
      Xi(15) = 57.624125410856096875938936463513D+00
      Xi(16) = 84.923776643453565603214538981547D+00
      Xi(17) = 126.782233409247627554938819827157D+00
      Xi(18) = 191.997991084500353370367875527336D+00
      Xi(19) = 295.438020506877174947701902851804D+00
      Xi(20) = 462.860698788542593656858414874478D+00
      Xi(21) = 740.162676662045127262690158431724D+00
      Xi(22) = 1211.899635043392013278307217660767D+00
      Xi(23) = 2039.943611755279275210561706899171D+00
      Xi(24) = 3548.928812212822196281436504250451D+00
      Xi(25) = 6428.924951436530012660597321882960D+00
      Xi(26) = 12252.483153235444802930942387320101D+00
      Xi(27) = 24984.063501351675157025056250859052D+00
      Xi(28) = 56001.512880335759721361910123960115D+00
      Xi(29) = 145387.166005767926492353581124916673D+00
      Xi(30) = 511088.658350937068661323792184703052D+00
      GOTO 9999
C
C     ===== K = 16 =====
C
 1600 CONTINUE
C
      GOTO (1611,1611,1611,1612,1613,1614,1615,1616,1617,1618,1619,1621,
     *      1622,1623,1624,1625,1626,1627,1628,1629,1631,1632,1633,1634,
     *      1635,1636,1637,1638,1639,1641,1699),InitR
C
C     ----- K = 16 , R = 1E1 -----
C
 1611 CONTINUE
C
      RIni = 1.0D+01
C
      Omega( 1) = 0.0544693993323789184580487575393359378495D+00
      Omega( 2) = 0.1282577437959811045254613498522999748275D+00
      Omega( 3) = 0.2057916770589411241046537068277721971299D+00
      Omega( 4) = 0.2894365923450766548276173739662198158840D+00
      Omega( 5) = 0.3822032310241595546359150842308238793521D+00
      Omega( 6) = 0.4879433905317224986922328838767626280060D+00
      Omega( 7) = 0.6116909136127484010198723796136732744344D+00
      Omega( 8) = 0.7600887458718382676324358537822689640961D+00
      Omega( 9) = 0.9419406889769544439193482698868820079952D+00
      Omega(10) = 1.1690433766271810673556194593025736594427D+00
      Omega(11) = 1.4577672298738048011889495425563723074447D+00
      Omega(12) = 1.8325870880315658182236476791970858357672D+00
      Omega(13) = 2.3346947673352790063924094843272882826568D+00
      Omega(14) = 3.0457271705138052335284232219336786329222D+00
      Omega(15) = 4.1712794331204028318305088163242544396780D+00
      Omega(16) = 6.5349837313441176047529301840199877915438D+00
C
      Alpha( 1) = 0.0211963589052251348014477066658556303480D+00
      Alpha( 2) = 0.1123242174600216008732342896914158103527D+00
      Alpha( 3) = 0.2789480531208277305635963999597137785713D+00
      Alpha( 4) = 0.5259381094628534172061122553554213254756D+00
      Alpha( 5) = 0.8608525827545032450003117951986553180177D+00
      Alpha( 6) = 1.2946560518738649671419832509755565297382D+00
      Alpha( 7) = 1.8427240850238871151941730253120965699054D+00
      Alpha( 8) = 2.5262309525576912049363298429227597807767D+00
      Alpha( 9) = 3.3740205529595078720768441815636151659419D+00
      Alpha(10) = 4.4251473628903414138900662688769216401852D+00
      Alpha(11) = 5.7325633870510128630923474268854533875128D+00
      Alpha(12) = 7.3692105865439201666140800472959426770103D+00
      Alpha(13) = 9.4397718200387649243460197467925354430918D+00
      Alpha(14) = 12.1071304391623297451799357027368841954740D+00
      Alpha(15) = 15.6650471589833527241361621484827537642559D+00
      Alpha(16) = 20.8276833299618638964401107216417585732415D+00
C
      Xi( 1) = 1.001642773023285848659619245993D+00
      Xi( 2) = 1.020632283267804189482753918217D+00
      Xi( 3) = 1.069694067115581228568851890337D+00
      Xi( 4) = 1.079893351595470750436955853502D+00
      Xi( 5) = 1.165776128629064865328823774604D+00
      Xi( 6) = 1.215427455094397498291079739996D+00
      Xi( 7) = 1.309820334060649646076845975529D+00
      Xi( 8) = 1.396768543697270465332033184680D+00
      Xi( 9) = 1.524016153482674934462742788455D+00
      Xi(10) = 1.669566103906232756729291488718D+00
      Xi(11) = 1.739123483722660599612962195604D+00
      Xi(12) = 1.897139976316912667996666952419D+00
      Xi(13) = 2.012573117182326204888803689030D+00
      Xi(14) = 2.225100967753168139297986760639D+00
      Xi(15) = 2.561199438850251413647696185905D+00
      Xi(16) = 2.800852853736864682930585157550D+00
      Xi(17) = 3.121498587220625926938119554244D+00
      Xi(18) = 3.385597622280114333756517730478D+00
      Xi(19) = 3.809646075125514618885808060256D+00
      Xi(20) = 4.262560966502159083644368786636D+00
      Xi(21) = 4.990680098969088600031202007656D+00
      Xi(22) = 5.235494914747392084051341720752D+00
      Xi(23) = 5.679028495515885146399659433314D+00
      Xi(24) = 6.310242140042951990500097458892D+00
      Xi(25) = 6.906489335891794362123213835147D+00
      Xi(26) = 7.346756507326352338659025686951D+00
      Xi(27) = 8.143084406071635463104929630873D+00
      Xi(28) = 8.661425262176749761446059583392D+00
      Xi(29) = 9.103434878560661631767547685712D+00
      Xi(30) = 9.468566991168702428292858197523D+00
      Xi(31) = 9.852011085479216269573943254301D+00
      Xi(32) = 9.982569464695079830551804889183D+00
      GOTO 9999
C
C     ----- K = 16 , R = 2E1 -----
C
 1612 CONTINUE
C
      RIni = 2.0D+01
C
      Omega( 1) = 0.0321225885764125043171516096784934291009D+00
      Omega( 2) = 0.0760016922055371523816134915474407307556D+00
      Omega( 3) = 0.1230393637303320996076211497893826418704D+00
      Omega( 4) = 0.1754101491069476755860646194351559756797D+00
      Omega( 5) = 0.2360231401909021565013475224548145092740D+00
      Omega( 6) = 0.3088188185834776517540989360099956684280D+00
      Omega( 7) = 0.3991595234159666612846605743269989829969D+00
      Omega( 8) = 0.5142450102957285566133809051603265061203D+00
      Omega( 9) = 0.6635919448742423670719453587807379335572D+00
      Omega(10) = 0.8598082538447276580283878955057730308909D+00
      Omega(11) = 1.1202030710426284491497070372467703691655D+00
      Omega(12) = 1.4703548620534354144902614014434050204727D+00
      Omega(13) = 1.9525233214943743469778000831915676371864D+00
      Omega(14) = 2.6485569231618074975340149102009945636382D+00
      Omega(15) = 3.7608140990416505055043278060722400368832D+00
      Omega(16) = 6.0921817077078821980537504110486679564929D+00
C
      Alpha( 1) = 0.0124933559834041096275154218014913354295D+00
      Alpha( 2) = 0.0663634793404057955643017205349387666047D+00
      Alpha( 3) = 0.1655401210940494760050483574742585801687D+00
      Alpha( 4) = 0.3142122161559275833923269433123337535108D+00
      Alpha( 5) = 0.5190964445166673073154064987910771833413D+00
      Alpha( 6) = 0.7903036143386039961630795680491701205028D+00
      Alpha( 7) = 1.1425620524652800269428329360543727943877D+00
      Alpha( 8) = 1.5968455073257302928363063343475403144112D+00
      Alpha( 9) = 2.1824392199329476567555563537581519995001D+00
      Alpha(10) = 2.9396026075790025980805780791982328992162D+00
      Alpha(11) = 3.9233621214215907357317908443761211856327D+00
      Alpha(12) = 5.2097574852041275347835913578364852583036D+00
      Alpha(13) = 6.9076901985583718684459653136542556239874D+00
      Alpha(14) = 9.1850123784656421332614484320799874694785D+00
      Alpha(15) = 12.3393159714999913713004420223739998618839D+00
      Alpha(16) = 17.0792800750442479908869364280121772026177D+00
C
      Xi( 1) = 1.002189500000000066393113229424D+00
      Xi( 2) = 1.019762947701370624990595270543D+00
      Xi( 3) = 1.055229999999999890292201598641D+00
      Xi( 4) = 1.109217220959847719375518382279D+00
      Xi( 5) = 1.182708024034756535448897896146D+00
      Xi( 6) = 1.277016834403164474774465542151D+00
      Xi( 7) = 1.393834079811942746533574843948D+00
      Xi( 8) = 1.535250464041688117709620609030D+00
      Xi( 9) = 1.703819073036006557890498225305D+00
      Xi(10) = 1.902554471272783690361136976765D+00
      Xi(11) = 2.135029299496591970435738216949D+00
      Xi(12) = 2.405404258639855907819793401359D+00
      Xi(13) = 2.718495459904630815730838810929D+00
      Xi(14) = 3.079773970864370405582655099153D+00
      Xi(15) = 3.495481619359764200501844433333D+00
      Xi(16) = 3.972546210309521930170717496367D+00
      Xi(17) = 4.518617597172095341935549317469D+00
      Xi(18) = 5.141874855214699761678243117302D+00
      Xi(19) = 5.850850279044855657955420680238D+00
      Xi(20) = 6.653940585562659133268825772944D+00
      Xi(21) = 7.558725347853058861621183828206D+00
      Xi(22) = 8.570838617733095778083995952557D+00
      Xi(23) = 9.692320370862275815890263253038D+00
      Xi(24) = 10.919306587316860122915951336608D+00
      Xi(25) = 12.239073715550893941585164359509D+00
      Xi(26) = 13.626418554740222651225356464977D+00
      Xi(27) = 15.040170469656139274304551545214D+00
      Xi(28) = 16.420733253759619497263999576830D+00
      Xi(29) = 17.690605814385193066823331875526D+00
      Xi(30) = 18.759784666594276307874378240115D+00
      Xi(31) = 19.537293771777199353351095467701D+00
      Xi(32) = 19.947636191155441776762247130250D+00
      GOTO 9999
C
C     ----- K = 16 , R = 3E1 -----
C
 1613 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.0235609197261937656794168142321277414908D+00
      Omega( 2) = 0.0559413134016891635680453990359106697383D+00
      Omega( 3) = 0.0911632628103368471841698970903333787419D+00
      Omega( 4) = 0.1312859571784798062951855890378993763079D+00
      Omega( 5) = 0.1791533047258759359988446180356191916871D+00
      Omega( 6) = 0.2387270433782989137932529014052285276648D+00
      Omega( 7) = 0.3154669497444695528201585865701872535283D+00
      Omega( 8) = 0.4166893847022847066912006053218675560856D+00
      Omega( 9) = 0.5519940219255625856332320156028004021209D+00
      Omega(10) = 0.7340636247311380832005975327625435511436D+00
      Omega(11) = 0.9803606385561895881488800319480958478380D+00
      Omega(12) = 1.3167257512871589496435131838580900875968D+00
      Omega(13) = 1.7856400770476758351097690269426720988122D+00
      Omega(14) = 2.4686440241868266001670079168306415340339D+00
      Omega(15) = 3.5657298034729491065849410480481651575246D+00
      Omega(16) = 5.8667042716208066843210089391646988588036D+00
C
      Alpha( 1) = 0.0091597849242320682433974644333363190896D+00
      Alpha( 2) = 0.0487412456562638064932481939128816605944D+00
      Alpha( 3) = 0.1219811140457653242323674165625835286164D+00
      Alpha( 4) = 0.2326929954185220091424698279913840792688D+00
      Alpha( 5) = 0.3871221911571684245307288507742526917355D+00
      Alpha( 6) = 0.5948850330998255694957624339913593303208D+00
      Alpha( 7) = 0.8702777453847756691535791917502251635597D+00
      Alpha( 8) = 1.2339548618200209037900141884058768937393D+00
      Alpha( 9) = 1.7149841111626251076958948882378308553598D+00
      Alpha(10) = 2.3534803627430483296753549460333942988655D+00
      Alpha(11) = 3.2044326031056090044613765543957129011687D+00
      Alpha(12) = 4.3440541494478195263390096769740011950489D+00
      Alpha(13) = 5.8816849055963979658358253832517448245198D+00
      Alpha(14) = 7.9856357540459984491881817447733737935778D+00
      Alpha(15) = 10.9528456420038435440225677020009698026115D+00
      Alpha(16) = 15.4852730910841890682103083598519788210979D+00
C
      Xi( 1) = 1.002573734850334118379015790889D+00
      Xi( 2) = 1.023246810174743838886546321021D+00
      Xi( 3) = 1.065036071547796444800547399634D+00
      Xi( 4) = 1.128835801500153776153433093921D+00
      Xi( 5) = 1.216010039164406659876506433537D+00
      Xi( 6) = 1.328419331760795168940678112612D+00
      Xi( 7) = 1.468467515067892203615117219240D+00
      Xi( 8) = 1.639160723274115317202141717257D+00
      Xi( 9) = 1.844179561582496653629643923455D+00
      Xi(10) = 2.087971282324882142059779832621D+00
      Xi(11) = 2.375850111715012873744162780199D+00
      Xi(12) = 2.714107645002727208730780583679D+00
      Xi(13) = 3.110143980894210012731054826318D+00
      Xi(14) = 3.572596476374702785354375844662D+00
      Xi(15) = 4.111475675606219205675262440280D+00
      Xi(16) = 4.738284434211328140576502399028D+00
      Xi(17) = 5.466101472116655788074240929220D+00
      Xi(18) = 6.309585592081037487641875793898D+00
      Xi(19) = 7.284826671866312525639025610147D+00
      Xi(20) = 8.408955894819123388396475693440D+00
      Xi(21) = 9.699317473423044616044619292694D+00
      Xi(22) = 11.172013401263362592820416185901D+00
      Xi(23) = 12.839446921794874787066398358348D+00
      Xi(24) = 14.706508605280594661687820767781D+00
      Xi(25) = 16.764986980257974451269387117236D+00
      Xi(26) = 18.986058317882165179593667581059D+00
      Xi(27) = 21.311359792364128411257251372035D+00
      Xi(28) = 23.644551464645229681485472106317D+00
      Xi(29) = 25.847317894115123357301477824421D+00
      Xi(30) = 27.745714371433567072000592190761D+00
      Xi(31) = 29.152259241143720490396784583709D+00
      Xi(32) = 29.903656300768179401167179420895D+00
      GOTO 9999
C
C     ----- K = 16 , R = 4E1 -----
C
 1614 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.0188412226242910377874828628183556666897D+00
      Omega( 2) = 0.0448605493543371719201160244261394893783D+00
      Omega( 3) = 0.0734930731016188330131737836692362719759D+00
      Omega( 4) = 0.1067040806278245791750857764113025183406D+00
      Omega( 5) = 0.1472715918877373489289132543755833637533D+00
      Omega( 6) = 0.1991328541485264322634819028623454073568D+00
      Omega( 7) = 0.2677423025669556750315552368091331914002D+00
      Omega( 8) = 0.3603915702110762296888485378687505544804D+00
      Omega( 9) = 0.4866411695538599008474967694271029472475D+00
      Omega(10) = 0.6591954549030602137586060118579212030454D+00
      Omega(11) = 0.8956803685351774997266960376407496369211D+00
      Omega(12) = 1.2222440383488672153728532565608588811301D+00
      Omega(13) = 1.6816930580726245238578145380081707571662D+00
      Omega(14) = 2.3555756876213544785941128578699022000364D+00
      Omega(15) = 3.4425924383299705466756485128243525650760D+00
      Omega(16) = 5.7245565688195948557577541482288552288082D+00
C
      Alpha( 1) = 0.0073225601418134040001580017595350646786D+00
      Alpha( 2) = 0.0390193033202446146977791525262135863272D+00
      Alpha( 3) = 0.0979067844659002423532922175508641515762D+00
      Alpha( 4) = 0.1875228759783119811222970585440350532735D+00
      Alpha( 5) = 0.3137542183667890708276444755586709334239D+00
      Alpha( 6) = 0.4858136923194132932740142705174690718195D+00
      Alpha( 7) = 0.7175823469735936248163621353768348853919D+00
      Alpha( 8) = 1.0292869363680080104167871568598968678998D+00
      Alpha( 9) = 1.4495317958082096967270535614602522400673D+00
      Alpha(10) = 2.0179516117564629451488350531107585084101D+00
      Alpha(11) = 2.7891432547193736283131643149957312743936D+00
      Alpha(12) = 3.8391679753262131513397853455060726446391D+00
      Alpha(13) = 5.2775471129346672432419707021722388162743D+00
      Alpha(14) = 7.2729818547780006043890210776226012967527D+00
      Alpha(15) = 10.1223022282442616168185445246763265458867D+00
      Alpha(16) = 14.5237447268137015426309188725895182869863D+00
C
      Xi( 1) = 1.002857171005857524789632584383D+00
      Xi( 2) = 1.025817267547144013986951116557D+00
      Xi( 3) = 1.072287351569654696801375814985D+00
      Xi( 4) = 1.143377809232345467026562002566D+00
      Xi( 5) = 1.240784578294497208822595724076D+00
      Xi( 6) = 1.366830740138873189946054631516D+00
      Xi( 7) = 1.524530992111522788128030736488D+00
      Xi( 8) = 1.717677462397433716854121266149D+00
      Xi( 9) = 1.950950040217017091492487934623D+00
      Xi(10) = 2.230046143600699522918651140913D+00
      Xi(11) = 2.561835105220026761584789509740D+00
      Xi(12) = 2.954535579269154725980464637480D+00
      Xi(13) = 3.417917275365817277900382098466D+00
      Xi(14) = 3.963529860969144456893722971991D+00
      Xi(15) = 4.604950601382712837249255866112D+00
      Xi(16) = 5.358041946476412298316915894247D+00
      Xi(17) = 6.241191647301613178774187806752D+00
      Xi(18) = 7.275491251863214980759309025160D+00
      Xi(19) = 8.484770406912843634071597076485D+00
      Xi(20) = 9.895351467251455564524964536588D+00
      Xi(21) = 11.535301674615324042487418676473D+00
      Xi(22) = 13.432831884931616845367530199695D+00
      Xi(23) = 15.613323064388788258792517549267D+00
      Xi(24) = 18.094252362824670242805180642165D+00
      Xi(25) = 20.877156702410411971693182664467D+00
      Xi(26) = 23.935868424235407380187834469609D+00
      Xi(27) = 27.201080567697457033085606248335D+00
      Xi(28) = 30.543444434751223785090568085820D+00
      Xi(29) = 33.761395182338625502865836480737D+00
      Xi(30) = 36.584850427056804168107762009754D+00
      Xi(31) = 38.707611387791347672820974068486D+00
      Xi(32) = 39.852625297309455229544106202866D+00
      GOTO 9999
C
C     ----- K = 16 , R = 5E1 -----
C
 1615 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.0158137200328224617952739810416984411745D+00
      Omega( 2) = 0.0377409803223574240874571214532373630846D+00
      Omega( 3) = 0.0621065805258739136550242328910220734883D+00
      Omega( 4) = 0.0907986308703175472160397385668062497643D+00
      Omega( 5) = 0.1265341518126833013878351774650354855112D+00
      Omega( 6) = 0.1732081162442415632880766615531165086850D+00
      Omega( 7) = 0.2362277390277097480729417977585171684041D+00
      Omega( 8) = 0.3228120296047016722347598044473571121671D+00
      Omega( 9) = 0.4424551929503152977274198220936085590438D+00
      Omega(10) = 0.6078787063087151399222787806486678618967D+00
      Omega(11) = 0.8368663078342602444094876223967816031291D+00
      Omega(12) = 1.1558545652842125910997675419800145846239D+00
      Omega(13) = 1.6079713225351366139492456652781982029410D+00
      Omega(14) = 2.2748723034512419898574980825500801984163D+00
      Omega(15) = 3.3544410264652535695693164052855195222946D+00
      Omega(16) = 5.6229103116070464694724317933349766462925D+00
C
      Alpha( 1) = 0.0061442796289501532090269618393424400082D+00
      Alpha( 2) = 0.0327791122423217869165921502239147145019D+00
      Alpha( 3) = 0.0824314340868094741013598521217620174184D+00
      Alpha( 4) = 0.1584254595616176457430332100151382235254D+00
      Alpha( 5) = 0.2663633767071307630763289664432846848285D+00
      Alpha( 6) = 0.4151230097894758880718111410113024817292D+00
      Alpha( 7) = 0.6182082205518974109899506264031998625796D+00
      Alpha( 8) = 0.8954087492618855389798354638131172578142D+00
      Alpha( 9) = 1.2748171403732771564832476629547386437480D+00
      Alpha(10) = 1.7955250328439662241622465366219785209978D+00
      Alpha(11) = 2.5116713611733493800946132035178948171961D+00
      Alpha(12) = 3.4990867106993718221950168123512980855594D+00
      Alpha(13) = 4.8673792671259520892755590093514683758258D+00
      Alpha(14) = 6.7855822825558887831549903513206345451181D+00
      Alpha(15) = 9.5506056509052685583177622063999478996266D+00
      Alpha(16) = 13.8582962625927245264856324524771480355412D+00
C
      Xi( 1) = 1.003082187031355445487948441219D+00
      Xi( 2) = 1.027859687592568978427298720391D+00
      Xi( 3) = 1.078057395890052983585156998192D+00
      Xi( 4) = 1.154974233682103314344198630170D+00
      Xi( 5) = 1.260596638410348168985071481973D+00
      Xi( 6) = 1.397653353732658535499115692868D+00
      Xi( 7) = 1.569698023830484381250897907289D+00
      Xi( 8) = 1.781222464089847532854073153263D+00
      Xi( 9) = 2.037799168843202316306340016183D+00
      Xi(10) = 2.346254768002126228762169168895D+00
      Xi(11) = 2.714874319136810658543990104619D+00
      Xi(12) = 3.153641949979435070254563111547D+00
      Xi(13) = 3.674522836983694047859050346716D+00
      Xi(14) = 4.291788356200312000952867830250D+00
      Xi(15) = 5.022386334308347026218061204261D+00
      Xi(16) = 5.886344711498311887357820504363D+00
      Xi(17) = 6.907189247874329671422061194264D+00
      Xi(18) = 8.112326812021093857402176396931D+00
      Xi(19) = 9.533307696033266082948076647341D+00
      Xi(20) = 11.205810438298586911333443172367D+00
      Xi(21) = 13.169075652993514335790947722415D+00
      Xi(22) = 15.464342471304390101358161579981D+00
      Xi(23) = 18.131571280598456772509274870231D+00
      Xi(24) = 21.203392018446972654144300207690D+00
      Xi(25) = 24.694852990502020423715934249742D+00
      Xi(26) = 28.587421489567784713137221785928D+00
      Xi(27) = 32.806440728453080516585638193305D+00
      Xi(28) = 37.194018687447342474267086842588D+00
      Xi(29) = 41.485465403523819716974374571805D+00
      Xi(30) = 45.306503275734785429057493999494D+00
      Xi(31) = 48.214455529116414321533223841243D+00
      Xi(32) = 49.795804671837560206881789781619D+00
      GOTO 9999
C
C     ----- K = 16 , R = 6E1 -----
C
 1616 CONTINUE
C
      RIni = 6.0D+01
C
      Omega( 1) = 0.0136908377299396044207035376752412680901D+00
      Omega( 2) = 0.0327417346325856678237932895908501862436D+00
      Omega( 3) = 0.0540912157388128640461038339898314220022D+00
      Omega( 4) = 0.0795625932745488592432261433506912595703D+00
      Omega( 5) = 0.1118172832095265544061061559200087600630D+00
      Omega( 6) = 0.1547008231015822485966216062736044989379D+00
      Omega( 7) = 0.2135556027713844064370735803537471042546D+00
      Omega( 8) = 0.2955133019832536639664305466723703830212D+00
      Omega( 9) = 0.4100001524248211071165069047106399580116D+00
      Omega(10) = 0.5697558701772300694178043090065699516344D+00
      Omega(11) = 0.7927078885207336509722587936099813532564D+00
      Omega(12) = 1.1055499871864973630798331671876155724021D+00
      Omega(13) = 1.5517056878524865918015859800327405082498D+00
      Omega(14) = 2.2129740071556539273648495091961763137078D+00
      Omega(15) = 3.2866778023493446055612599021600317428238D+00
      Omega(16) = 5.5448487278535236997580037954236331643187D+00
C
      Alpha( 1) = 0.0053182066261874213429777932208436519446D+00
      Alpha( 2) = 0.0284011328189939031852584557999441550180D+00
      Alpha( 3) = 0.0715605398648006798946008370776450391304D+00
      Alpha( 4) = 0.1379484239253294434551745314965209843194D+00
      Alpha( 5) = 0.2329335881567227507148122242486287802876D+00
      Alpha( 6) = 0.3651092254541286646527496750902486155610D+00
      Alpha( 7) = 0.5476393721885706778232903724212832230478D+00
      Alpha( 8) = 0.7998966011997658137069010242381494890651D+00
      Alpha( 9) = 1.1494732580738143478760529547777480274817D+00
      Alpha(10) = 1.6349308451367353002678037277028977314330D+00
      Alpha(11) = 2.3099578735226667029582781065499830219778D+00
      Alpha(12) = 3.2501382678703844913373904867981423194578D+00
      Alpha(13) = 4.5651196769680073232061667809489335922990D+00
      Alpha(14) = 6.4242183428527932921407728183993413040298D+00
      Alpha(15) = 9.1244977187338398849636433851628680713475D+00
      Alpha(16) = 13.3601334728373712015109253670175348815974D+00
C
      Xi( 1) = 1.003268954923555123567163738230D+00
      Xi( 2) = 1.029556008797791147313382964512D+00
      Xi( 3) = 1.082855440022723218199303396592D+00
      Xi( 4) = 1.164634111432245486771573650397D+00
      Xi( 5) = 1.277138088539143851948311214617D+00
      Xi( 6) = 1.423459235187697745068750865283D+00
      Xi( 7) = 1.607636573966127172224775676579D+00
      Xi( 8) = 1.834794677819060208920007304378D+00
      Xi( 9) = 2.111319404311678209987951371218D+00
      Xi(10) = 2.445071479099704985470109530610D+00
      Xi(11) = 2.845641348624389181144744842555D+00
      Xi(12) = 3.324652574133972351196758543423D+00
      Xi(13) = 3.896122654636821812119268360064D+00
      Xi(14) = 4.576888108617195779874442518320D+00
      Xi(15) = 5.387098391416639992478054832503D+00
      Xi(16) = 6.350773440036542298418875684973D+00
      Xi(17) = 7.496409682465592422793976279394D+00
      Xi(18) = 8.857587984687720121523835103972D+00
      Xi(19) = 10.473497990380090252350575585893D+00
      Xi(20) = 12.389204257258051825214828500776D+00
      Xi(21) = 14.655349583419857167551625387603D+00
      Xi(22) = 17.326755222748656678002920905612D+00
      Xi(23) = 20.459029270439212126395100987963D+00
      Xi(24) = 24.101782182788760300687358473226D+00
      Xi(25) = 28.286424404647261036938044043154D+00
      Xi(26) = 33.006081233208052379268782061672D+00
      Xi(27) = 38.185652963770639254509342563892D+00
      Xi(28) = 43.643238869459938741174331866546D+00
      Xi(29) = 49.052443340562168410529109152662D+00
      Xi(30) = 53.929290807526017108297633484426D+00
      Xi(31) = 57.679914983451419423221562254867D+00
      Xi(32) = 59.734007012493412575038220779788D+00
      GOTO 9999
C
C     ----- K = 16 , R = 7E1 -----
C
 1617 CONTINUE
C
      RIni = 7.0D+01
C
      Omega( 1) = 0.0121119422284885791480239553181974532237D+00
      Omega( 2) = 0.0290189532715633909560086142742640902270D+00
      Omega( 3) = 0.0481093108632741291783420454347686145979D+00
      Omega( 4) = 0.0711508422286992170256074777268118225493D+00
      Omega( 5) = 0.1007542599873194719787834576496532434930D+00
      Omega( 6) = 0.1407132096601951245975342663641072249447D+00
      Omega( 7) = 0.1962985762182013438169455796233897615366D+00
      Omega( 8) = 0.2745528922234677530263453892889202734295D+00
      Omega( 9) = 0.3848395792100061791577055025914688712874D+00
      Omega(10) = 0.5399170453962834127802951356134286697852D+00
      Omega(11) = 0.7578406304494023196824219190670390844389D+00
      Omega(12) = 1.0655310964555628802218761874698316205468D+00
      Omega(13) = 1.5066799629278595433305720785455150689813D+00
      Omega(14) = 2.1632416780817367598071382372992843556858D+00
      Omega(15) = 3.2321339880593698366123556198203914391343D+00
      Omega(16) = 5.4820672068151955966253119356679235352203D+00
C
      Alpha( 1) = 0.0047039031601346258206606154757267734823D+00
      Alpha( 2) = 0.0251434613504408551858001997711400399282D+00
      Alpha( 3) = 0.0634624580084015806394608788520983466697D+00
      Alpha( 4) = 0.1226699481548256966310602757153080943908D+00
      Alpha( 5) = 0.2079381850759855593054406530661992746900D+00
      Alpha( 6) = 0.3276138429438437771952941784947554992868D+00
      Alpha( 7) = 0.4945540693857812940433232168357946534343D+00
      Alpha( 8) = 0.7277422312582489759670914541977992939792D+00
      Alpha( 9) = 1.0543005913145042581890997723803593544289D+00
      Alpha(10) = 1.5122947031342729613812014832774366368540D+00
      Alpha(11) = 2.1549896726867369830225873439388806218631D+00
      Alpha(12) = 3.0577259786550075454369945759935944806784D+00
      Alpha(13) = 4.3301629301893997856486018260824266690179D+00
      Alpha(14) = 6.1418596932948431384342857786151625987259D+00
      Alpha(15) = 8.7900619662313506002052077015207487420412D+00
      Alpha(16) = 12.9677092977387981298509345151614979840815D+00
C
      Xi( 1) = 1.003428650850012487471922539850D+00
      Xi( 2) = 1.031007206247950144946921857692D+00
      Xi( 3) = 1.086964401585403750308787829137D+00
      Xi( 4) = 1.172919035756665565180777155430D+00
      Xi( 5) = 1.291352523950442493017899681096D+00
      Xi( 6) = 1.445686855421073847708171578041D+00
      Xi( 7) = 1.640403888683373654668040675375D+00
      Xi( 8) = 1.881208547679956308621394833747D+00
      Xi( 9) = 2.175236252473233881888844964969D+00
      Xi(10) = 2.531305723494001962961194585056D+00
      Xi(11) = 2.960223167304340141474383596432D+00
      Xi(12) = 3.475147698324568867932551907707D+00
      Xi(13) = 4.092030634851775970119158731819D+00
      Xi(14) = 4.830140859978980539916892933405D+00
      Xi(15) = 5.712684090346128815401244294137D+00
      Xi(16) = 6.767518437426713942377254351612D+00
      Xi(17) = 8.027955257164587068649530543496D+00
      Xi(18) = 9.533606681635384107534325615063D+00
      Xi(19) = 11.331193698942411123876949918809D+00
      Xi(20) = 13.475135845050428147137711043513D+00
      Xi(21) = 16.027585820352931588947398289235D+00
      Xi(22) = 19.057298239731036980024181382021D+00
      Xi(23) = 22.636277701185040909029333455749D+00
      Xi(24) = 26.832476998127376190475534833979D+00
      Xi(25) = 31.695907140097002918296964502076D+00
      Xi(26) = 37.234675578505159788261780207108D+00
      Xi(27) = 43.377548401359464658177955698193D+00
      Xi(28) = 49.922998632993908999649912416885D+00
      Xi(29) = 56.485050482773292216537308618740D+00
      Xi(30) = 62.466232288618900061333283346698D+00
      Xi(31) = 67.109010911477774460887069096771D+00
      Xi(32) = 69.667807344614608493216234563761D+00
      GOTO 9999
C
C     ----- K = 16 , R = 8E1 -----
C
 1618 CONTINUE
C
      RIni = 8.0D+01
C
      Omega( 1) = 0.0108873068861372562665142246438043649448D+00
      Omega( 2) = 0.0261282663223343776516766154977000624626D+00
      Omega( 3) = 0.0434552850078543593980904234663764640345D+00
      Omega( 4) = 0.0645878894987122491969946079526376792046D+00
      Omega( 5) = 0.0920903496023600561648904741507415394608D+00
      Omega( 6) = 0.1297042298984791642354075930088086465730D+00
      Omega( 7) = 0.1826274887652090326948953538654940587094D+00
      Omega( 8) = 0.2578168051849726095697996897104786739874D+00
      Omega( 9) = 0.3645792437854544994267711605084869574966D+00
      Omega(10) = 0.5156912792016989570496182659908868117782D+00
      Omega(11) = 0.7293212975503779889590444396318957842595D+00
      Omega(12) = 1.0325907226271625882763061099467449821532D+00
      Omega(13) = 1.4694338205014248436176133028041590478097D+00
      Omega(14) = 2.1219626037538278065432834074144352598523D+00
      Omega(15) = 3.1867916413473558830040527345417444848863D+00
      Omega(16) = 5.4299136597940658287980708607278756971937D+00
C
      Alpha( 1) = 0.0042274941327305500965867126814123899692D+00
      Alpha( 2) = 0.0226156451548393151356538145063668210355D+00
      Alpha( 3) = 0.0571724048914840694760443415983752757370D+00
      Alpha( 4) = 0.1107855007815737983603598428017367183429D+00
      Alpha( 5) = 0.1884582617757677853434408049571047172321D+00
      Alpha( 6) = 0.2983204706033127306074020229642584922658D+00
      Alpha( 7) = 0.4529507102131609971406960013506903806046D+00
      Alpha( 8) = 0.6709720585108713596447782556708716583671D+00
      Alpha( 9) = 0.9790711090780677519932947983072324404930D+00
      Alpha(10) = 1.4148557088957122064446295106598938673415D+00
      Alpha(11) = 2.0311982614953904605591128440522652454092D+00
      Alpha(12) = 2.9032052654708691739274956100658187096997D+00
      Alpha(13) = 4.1405299017812756882765679034008599046501D+00
      Alpha(14) = 5.9129417512000719144536764471098422291107D+00
      Alpha(15) = 8.5178777748218142407132802418345818296075D+00
      Alpha(16) = 12.6473263671162415074794571623328920395579D+00
C
      Xi( 1) = 1.003568146725075832287506216733D+00
      Xi( 2) = 1.032275421951231643576639418480D+00
      Xi( 3) = 1.090558429595056287738329137138D+00
      Xi( 4) = 1.180175123042378194666152113612D+00
      Xi( 5) = 1.303822609129984488006426257556D+00
      Xi( 6) = 1.465226302116329349235594736722D+00
      Xi( 7) = 1.669276518959938414949847884650D+00
      Xi( 8) = 1.922215602583746199344058580571D+00
      Xi( 9) = 2.231876104194821583978983192686D+00
      Xi(10) = 2.607971760466308256722067770994D+00
      Xi(11) = 3.062449793423694131182591360307D+00
      Xi(12) = 3.609917387686896413740919586921D+00
      Xi(13) = 4.268159232181218664618077562878D+00
      Xi(14) = 5.058762368842223879333158675919D+00
      Xi(15) = 6.007861986970951318539485574988D+00
      Xi(16) = 7.147016996702025953477899200550D+00
      Xi(17) = 8.514211259033158024460419266433D+00
      Xi(18) = 10.154950044864726479108829604314D+00
      Xi(19) = 12.123369696286200031601354432098D+00
      Xi(20) = 14.483180337865166581283460356389D+00
      Xi(21) = 17.308084416454160245693594255556D+00
      Xi(22) = 20.680995355834386979776429171807D+00
      Xi(23) = 24.690858225061783201223497563603D+00
      Xi(24) = 29.425016953835080223134124111084D+00
      Xi(25) = 34.953889483568182535572210767327D+00
      Xi(26) = 41.303361521813087238463868544613D+00
      Xi(27) = 48.409874077820592781079556132795D+00
      Xi(28) = 56.056430700655069294252585976324D+00
      Xi(29) = 63.799997587513382624191393688307D+00
      Xi(30) = 70.927009405821734140673573421054D+00
      Xi(31) = 76.505504680732721367597548578487D+00
      Xi(32) = 79.597637690856747637313528542791D+00
      GOTO 9999
C
C     ----- K = 16 , R = 9E1 -----
C
 1619 CONTINUE
C
      RIni = 9.0D+01
C
      Omega( 1) = 0.0099070942053584087306917073216527835910D+00
      Omega( 2) = 0.0238121984392840396955234177900351255630D+00
      Omega( 3) = 0.0397197022388286855612516898506036966410D+00
      Omega( 4) = 0.0593065074157296978656821531683540627000D+00
      Omega( 5) = 0.0850941121309136351742469922543321558805D+00
      Omega( 6) = 0.1207729487547143815568675626059835792603D+00
      Omega( 7) = 0.1714692310620858691768775630692722700132D+00
      Omega( 8) = 0.2440589639770963612403755979562980371611D+00
      Omega( 9) = 0.3477989365091591634755827794345250936203D+00
      Omega(10) = 0.4954823766957529564093284935033523197490D+00
      Omega(11) = 0.7053775625652202247361718945306563455233D+00
      Omega(12) = 1.0047843223684194823453613709673959419888D+00
      Omega(13) = 1.4378576455423156901531483997658256157592D+00
      Omega(14) = 2.0868647293083933649988914726947086819564D+00
      Omega(15) = 3.1481875148259403213429202805784257179766D+00
      Omega(16) = 5.3855376972745262007663535896995199436788D+00
C
      Alpha( 1) = 0.0038462156898872527546131037468404856305D+00
      Alpha( 2) = 0.0205915676059204840877094912221512146289D+00
      Alpha( 3) = 0.0521312367564103167914054282744418244988D+00
      Alpha( 4) = 0.1012481384043264893461623772641377883019D+00
      Alpha( 5) = 0.1727980588057935402480653347412342668576D+00
      Alpha( 6) = 0.2747175043007444109066390175444283272554D+00
      Alpha( 7) = 0.4193307805231102605074498118620240916243D+00
      Alpha( 8) = 0.6249280273155943181786521611886797700208D+00
      Alpha( 9) = 0.9177937594965698797329703428893310501735D+00
      Alpha(10) = 1.3351147388188597902019399965567458821170D+00
      Alpha(11) = 1.9294001189108118012168965882402460465528D+00
      Alpha(12) = 2.7755336610437487396881722778729795209074D+00
      Alpha(13) = 3.9831504408499844558950303552435912024521D+00
      Alpha(14) = 5.7222039389716857524063875484188201880897D+00
      Alpha(15) = 8.2903222456109351128777928963131671480369D+00
      Alpha(16) = 12.3787400479130052528137229117533024691511D+00
C
      Xi( 1) = 1.003691976916175615866522186082D+00
      Xi( 2) = 1.033401659367964465748371272191D+00
      Xi( 3) = 1.093752631786901880362600225638D+00
      Xi( 4) = 1.186631336597799261695258821181D+00
      Xi( 5) = 1.314934451540161665524485135048D+00
      Xi( 6) = 1.482668631894253555097117069916D+00
      Xi( 7) = 1.695103987050068696678696567037D+00
      Xi( 8) = 1.958984485408930278200166641422D+00
      Xi( 9) = 2.282795600848628430170131342614D+00
      Xi(10) = 2.677092955809774674647699699293D+00
      Xi(11) = 3.154901093114184340918115800889D+00
      Xi(12) = 3.732200333833114767791844235134D+00
      Xi(13) = 4.428521835716363213783508534505D+00
      Xi(14) = 5.267671209050016853294579055600D+00
      Xi(15) = 6.278601228720268359086154053017D+00
      Xi(16) = 7.496448588479154036007950923093D+00
      Xi(17) = 8.963739143018941191859427775057D+00
      Xi(18) = 10.731740206947994127444079293099D+00
      Xi(19) = 12.861885824280574017186573598792D+00
      Xi(20) = 15.427096091462101036875798065751D+00
      Xi(21) = 18.512620100932913286967407273664D+00
      Xi(22) = 22.215675943845266181869702037943D+00
      Xi(23) = 26.642549964022028703009015160319D+00
      Xi(24) = 31.900809183562342526141453191713D+00
      Xi(25) = 38.082764344011690707536565980718D+00
      Xi(26) = 45.234470509062490239243770062672D+00
      Xi(27) = 53.303434122760849408345640298990D+00
      Xi(28) = 62.061106950823038484743587872572D+00
      Xi(29) = 71.010132921310621989530176279004D+00
      Xi(30) = 79.319141105841021607836260898239D+00
      Xi(31) = 85.872336017797033375087423223704D+00
      Xi(32) = 89.523836686630239280193510609251D+00
      GOTO 9999
C
C     ----- K = 16 , R = 1E2 -----
C
 1621 CONTINUE
C
      RIni = 1.0D+02
C
      Omega( 1) = 0.0091030846063941373947308146122314731485D+00
      Omega( 2) = 0.0219107094627195211883731798541141699843D+00
      Omega( 3) = 0.0366477217677682609360426211481120972735D+00
      Omega( 4) = 0.0549529774439788038923107842820270363404D+00
      Omega( 5) = 0.0793083736311463438007816027786045509629D+00
      Omega( 6) = 0.1133547294055116984719212488985640163719D+00
      Omega( 7) = 0.1621489723348042820026335131267103406572D+00
      Omega( 8) = 0.2324921039907123551090722610168448625245D+00
      Omega( 9) = 0.3335959219735489658447899485871346314525D+00
      Omega(10) = 0.4782687636968508076529098260731842628957D+00
      Omega(11) = 0.6848674256397950935236311476028703282282D+00
      Omega(12) = 0.9808514234763911221752670510642957424352D+00
      Omega(13) = 1.4105774826377881514018197695037315497757D+00
      Omega(14) = 2.0564635512053020538481007406517164781690D+00
      Omega(15) = 3.1147098104802878920822906128051954510738D+00
      Omega(16) = 5.3470752639906505886482901424017200042726D+00
C
      Alpha( 1) = 0.0035335098097274502238478871302823636213D+00
      Alpha( 2) = 0.0189307470173197830927468501838184522512D+00
      Alpha( 3) = 0.0479913291058656280866289659675638645808D+00
      Alpha( 4) = 0.0934063314619771164137255331505471644959D+00
      Alpha( 5) = 0.1599009758991468320513942499849768807962D+00
      Alpha( 6) = 0.2552377205651998341359980998488410364189D+00
      Alpha( 7) = 0.3915075062326464324703213792933098602589D+00
      Alpha( 8) = 0.5866925245564957031702522927130161178866D+00
      Alpha( 9) = 0.8667062212594494830723672318395500724364D+00
      Alpha(10) = 1.2683475596560744691043004506170177592139D+00
      Alpha(11) = 1.8437892243513303981803513886816858757811D+00
      Alpha(12) = 2.6677035867305717634339473631754913185432D+00
      Alpha(13) = 3.8496992950460036169921251847370058385422D+00
      Alpha(14) = 5.5598919048823978901474607283716977690347D+00
      Alpha(15) = 8.0960961548375747339964525473021694779163D+00
      Alpha(16) = 12.1489357832949838493855065912896407098742D+00
C
      Xi( 1) = 1.003803294380646768694728687699D+00
      Xi( 2) = 1.034414459604964373409807798510D+00
      Xi( 3) = 1.096627116167862222044157571865D+00
      Xi( 4) = 1.192447330001823254546668151566D+00
      Xi( 5) = 1.324957592558943276833324786956D+00
      Xi( 6) = 1.498427125021005234173018805244D+00
      Xi( 7) = 1.718481564209594093881744436558D+00
      Xi( 8) = 1.992335948611406212685297212328D+00
      Xi( 9) = 2.329090952993635296611532603528D+00
      Xi(10) = 2.740098264961065148729787988024D+00
      Xi(11) = 3.239404450340402310919152295554D+00
      Xi(12) = 3.844298179747094311095589880267D+00
      Xi(13) = 4.575980668529397538298042480420D+00
      Xi(14) = 5.460388213371944769389254581426D+00
      Xi(15) = 6.529191337570760817875931136811D+00
      Xi(16) = 7.820993488808169717772106332987D+00
      Xi(17) = 9.382742725529808837207912386447D+00
      Xi(18) = 11.271345248068873026031644357658D+00
      Xi(19) = 13.555416450399780177622777177504D+00
      Xi(20) = 16.316997190581468463524417167321D+00
      Xi(21) = 19.652855825131246635811832490504D+00
      Xi(22) = 23.674600448775685208879782095437D+00
      Xi(23) = 28.506167398102847601246145892162D+00
      Xi(24) = 34.276011964813860175699256060966D+00
      Xi(25) = 41.099581606579022291514391440614D+00
      Xi(26) = 49.045173894371357652122167536390D+00
      Xi(27) = 58.074399576032853211721151964753D+00
      Xi(28) = 67.950839319041810432375783079806D+00
      Xi(29) = 78.125663428646631089580232298886D+00
      Xi(30) = 87.648656297381219167086374710607D+00
      Xi(31) = 95.211876155386198909702510917441D+00
      Xi(32) = 99.446678144653096610705578228817D+00
      GOTO 9999
C
C     ----- K = 16 , R = 2E2 -----
C
 1622 CONTINUE
C
      RIni = 2.0D+02
C
      Omega( 1) = 0.0051910050391620038140489551154593828031D+00
      Omega( 2) = 0.0126277612849116577841408043912652114926D+00
      Omega( 3) = 0.0215606407131872894500943058787578721081D+00
      Omega( 4) = 0.0333846640063611133550348052112521912704D+00
      Omega( 5) = 0.0502953701646382063748634289740424563320D+00
      Omega( 6) = 0.0755353808513282801251971683431651882756D+00
      Omega( 7) = 0.1136336810394631888142757090987000623272D+00
      Omega( 8) = 0.1708798850219694833184108304080517370949D+00
      Omega( 9) = 0.2561954432329190103000918915521211260966D+00
      Omega(10) = 0.3824592769646401152450979760155647113606D+00
      Omega(11) = 0.5685478192743596877059900074335985209473D+00
      Omega(12) = 0.8429296732276469761584343498572735597918D+00
      Omega(13) = 1.2513442211943193828945636303373589726107D+00
      Omega(14) = 1.8774322383842335613957086337677537812851D+00
      Omega(15) = 2.9167470741005022236320975981271885757451D+00
      Omega(16) = 5.1200362418007544676283593343413258480723D+00
C
      Alpha( 1) = 0.0020125660928861264092960586986891247108D+00
      Alpha( 2) = 0.0108393025800470222817957488548046285359D+00
      Alpha( 3) = 0.0277607372768341410946277888344146944632D+00
      Alpha( 4) = 0.0549153548548539399454552697998899191134D+00
      Alpha( 5) = 0.0962139067787890126429270649943692461648D+00
      Alpha( 6) = 0.1582685859013296521361499960556251664912D+00
      Alpha( 7) = 0.2515517622186365606119784477101575248525D+00
      Alpha( 8) = 0.3918898940757439454808490342241356074737D+00
      Alpha( 9) = 0.6026248579910799004323920802228542470402D+00
      Alpha(10) = 0.9178653621745612205739747735400158035191D+00
      Alpha(11) = 1.3873828979151186895080183347772617707960D+00
      Alpha(12) = 2.0842135216668545253260874661194179680024D+00
      Alpha(13) = 3.1175469236537975187186638725833631724527D+00
      Alpha(14) = 4.6584561428876707965887615792155429517152D+00
      Alpha(15) = 7.0062440250979758607670355718965993219172D+00
      Alpha(16) = 10.8487716247577870394211418947350011876551D+00
C
      Xi( 1) = 1.004543963223910764100479942051D+00
      Xi( 2) = 1.041162209364920160662923431794D+00
      Xi( 3) = 1.115827713131373046559317863125D+00
      Xi( 4) = 1.231441193971959971410409795389D+00
      Xi( 5) = 1.392482124242125950618585716878D+00
      Xi( 6) = 1.605208525760429899716628943107D+00
      Xi( 7) = 1.877967940703037661690118220736D+00
      Xi( 8) = 2.221625241230130186993393937023D+00
      Xi( 9) = 2.650109627811283131634811915944D+00
      Xi(10) = 3.181098745057285561051163558233D+00
      Xi(11) = 3.836876209790531579027841879714D+00
      Xi(12) = 4.645422392240050664391493029370D+00
      Xi(13) = 5.641798797641391978452407585110D+00
      Xi(14) = 6.869906621767563695887121033579D+00
      Xi(15) = 8.384711159585907115816827417376D+00
      Xi(16) = 10.255043690692088445559626563863D+00
      Xi(17) = 12.567108023170744343297022815875D+00
      Xi(18) = 15.428825091440629301529907646540D+00
      Xi(19) = 18.975125907624490443834086583763D+00
      Xi(20) = 23.374196278342412529116511876737D+00
      Xi(21) = 28.834431487391838569780966317069D+00
      Xi(22) = 35.611266762769640249058911152247D+00
      Xi(23) = 44.011799056629079002905813311486D+00
      Xi(24) = 54.392539367342489734641342025157D+00
      Xi(25) = 67.140738711682966172178410602100D+00
      Xi(26) = 82.620841953775863229780807017733D+00
      Xi(27) = 101.054169044903487922892892925120D+00
      Xi(28) = 122.286579577677779903654986526362D+00
      Xi(29) = 145.409716504364367142820846368068D+00
      Xi(30) = 168.301902381474210376732791871746D+00
      Xi(31) = 187.425817576227922295339922698076D+00
      Xi(32) = 198.527536791589821002501636826310D+00
      GOTO 9999
C
C     ----- K = 16 , R = 3E2 -----
C
 1623 CONTINUE
C
      RIni = 3.0D+02
C
      Omega( 1) = 0.0037255564169910067277597943676339942165D+00
      Omega( 2) = 0.0091302987056417856438241247712950254822D+00
      Omega( 3) = 0.0158171136039656348763288119460557190621D+00
      Omega( 4) = 0.0250466904073013040228428641859048653373D+00
      Omega( 5) = 0.0388327073275191414696656724142576067038D+00
      Omega( 6) = 0.0601510240904654430165875329816760341828D+00
      Omega( 7) = 0.0932081506757678266748639271666032612984D+00
      Omega( 8) = 0.1440089851094002000250499789801850170079D+00
      Omega( 9) = 0.2213025039554186542865247405709183681211D+00
      Omega(10) = 0.3379614198919181375997851379811010019694D+00
      Omega(11) = 0.5130898999076860187416350422839883549386D+00
      Omega(12) = 0.7756846839561412798483112929215366193603D+00
      Omega(13) = 1.1723016787490888693809912290966224190925D+00
      Omega(14) = 1.7874363147483472139985180571208900346392D+00
      Omega(15) = 2.8166376845013903116275721938421838785871D+00
      Omega(16) = 5.0054814386377507826747512709175680356566D+00
C
      Alpha( 1) = 0.0014432114855624366430164606540237359766D+00
      Alpha( 2) = 0.0078015317798097004564905892088494399061D+00
      Alpha( 3) = 0.0201254698329391623609277558668018137666D+00
      Alpha( 4) = 0.0402748735892814804047793765529039689000D+00
      Alpha( 5) = 0.0717270043310115766298511348619815208849D+00
      Alpha( 6) = 0.1204378422197703271513485011323663087524D+00
      Alpha( 7) = 0.1959225086659849674689676257699488814978D+00
      Alpha( 8) = 0.3127387451926994034500425362343989377223D+00
      Alpha( 9) = 0.4927264320226538121989812157863752872800D+00
      Alpha(10) = 0.7683981194605783673250501419893510046677D+00
      Alpha(11) = 1.1880336160297062512341062512355449598545D+00
      Alpha(12) = 1.8235512816513078762680269417906231410598D+00
      Alpha(13) = 2.7837030596469976369451704290014504294959D+00
      Alpha(14) = 4.2399974599818059570709161132384679149254D+00
      Alpha(15) = 6.4927254010870050415435428270427564712008D+00
      Alpha(16) = 10.2289376129522341478395031089121403056197D+00
C
      Xi( 1) = 1.004979210341979526917646770023D+00
      Xi( 2) = 1.045134743033515391371843228363D+00
      Xi( 3) = 1.127172042944499245854458879812D+00
      Xi( 4) = 1.254599420629432444105645638732D+00
      Xi( 5) = 1.432850797631794785135656011832D+00
      Xi( 6) = 1.669557895055314880638011598712D+00
      Xi( 7) = 1.974975462656173537455243927052D+00
      Xi( 8) = 2.362565861687259726907003876128D+00
      Xi( 9) = 2.849750598893955735825198050115D+00
      Xi(10) = 3.458858686311279327825485285786D+00
      Xi(11) = 4.218338541652051693585379066320D+00
      Xi(12) = 5.164320825089354523133866603501D+00
      Xi(13) = 6.342640395703774114453260546398D+00
      Xi(14) = 7.811444110011478200034096275317D+00
      Xi(15) = 9.644553699667076237037055541634D+00
      Xi(16) = 11.935793682202448600347677309941D+00
      Xi(17) = 14.804545904806895795929877479580D+00
      Xi(18) = 18.402846576447768963888385052030D+00
      Xi(19) = 22.924367560072551247563055198952D+00
      Xi(20) = 28.615602890090455294111460027295D+00
      Xi(21) = 35.789364298836023884492485436226D+00
      Xi(22) = 44.840078460752617432738542646575D+00
      Xi(23) = 56.258867284198758970670262158364D+00
      Xi(24) = 70.642906086139508434196265795890D+00
      Xi(25) = 88.685979443723195511650736477804D+00
      Xi(26) = 111.122037340557960498343970101587D+00
      Xi(27) = 138.565411870331778212972473340869D+00
      Xi(28) = 171.151311059692604610305011192395D+00
      Xi(29) = 207.857682225517994667463028690690D+00
      Xi(30) = 245.514446222413214526714142493802D+00
      Xi(31) = 278.044557713935721832143599385745D+00
      Xi(32) = 297.404850809861197785677333627063D+00
      GOTO 9999
C
C     ----- K = 16 , R = 4E2 -----
C
 1624 CONTINUE
C
      RIni = 4.0D+02
C
      Omega( 1) = 0.0029411704155330162737393910176580957838D+00
      Omega( 2) = 0.0072506379760886996575654608095801156864D+00
      Omega( 3) = 0.0127075060808783997139066778217508790760D+00
      Omega( 4) = 0.0204817457415186981432056153099030559872D+00
      Omega( 5) = 0.0324552336489367649763478854835208053942D+00
      Omega( 6) = 0.0514120457313516375765709114087353981404D+00
      Omega( 7) = 0.0813399962135207765340184568603199011250D+00
      Omega( 8) = 0.1280531259067858902434824605598429059228D+00
      Omega( 9) = 0.2001715994871604321903024725770592340268D+00
      Omega(10) = 0.3105368092541307178716582931832235203728D+00
      Omega(11) = 0.4783756936706182759703152862496011721305D+00
      Omega(12) = 0.7330239927551988701227847489150235560373D+00
      Omega(13) = 1.1216059739542480910455199416553284663678D+00
      Omega(14) = 1.7292644544768973737056821926572069969552D+00
      Omega(15) = 2.7516812909019973754434773560717530926922D+00
      Omega(16) = 4.9312407184718789086505519314584944368107D+00
C
      Alpha( 1) = 0.0011386076066376620912603070696206275514D+00
      Alpha( 2) = 0.0061729978126926382157131298344943104439D+00
      Alpha( 3) = 0.0160169098352160544780869826098079755639D+00
      Alpha( 4) = 0.0323524323481128600355502880625557793337D+00
      Alpha( 5) = 0.0583702583176669546740700841164217749224D+00
      Alpha( 6) = 0.0995777894307948640905759143948650091716D+00
      Alpha( 7) = 0.1648315575629028856323660470750702700116D+00
      Alpha( 8) = 0.2678221908723548319683927659839994817048D+00
      Alpha( 9) = 0.4293602681830135254224031088332580452516D+00
      Alpha(10) = 0.6808451714576150000748401558414713008460D+00
      Alpha(11) = 1.0694838116053031123542718661134642843535D+00
      Alpha(12) = 1.6663482146929018885318785070381863988587D+00
      Alpha(13) = 2.5798010052893875195326178761945357109653D+00
      Alpha(14) = 3.9815934666964298322029691679091456535389D+00
      Alpha(15) = 6.1727341131020466791570755482609911268810D+00
      Alpha(16) = 9.8399722788013861806549753552531001332682D+00
C
      Xi( 1) = 1.005286566415155632557738851762D+00
      Xi( 2) = 1.047943318089983249109115037712D+00
      Xi( 3) = 1.135210829091119112297866844408D+00
      Xi( 4) = 1.271063636152130822644032437374D+00
      Xi( 5) = 1.461671315433593978721713013691D+00
      Xi( 6) = 1.715731492895732926699976783613D+00
      Xi( 7) = 2.044992742814055935356501447764D+00
      Xi( 8) = 2.464972706047913258813053527518D+00
      Xi( 9) = 2.995883140373143346769577166633D+00
      Xi(10) = 3.663810016421250367836190409143D+00
      Xi(11) = 4.502235087972342951999349391912D+00
      Xi(12) = 5.554025676793504075721974455604D+00
      Xi(13) = 6.874030818543077030104843094804D+00
      Xi(14) = 8.532470992427824087525611851390D+00
      Xi(15) = 10.619360983407052111127111260203D+00
      Xi(16) = 13.250281458211545110477402875659D+00
      Xi(17) = 16.573901537800292628824228025053D+00
      Xi(18) = 20.781764408042251687444568197805D+00
      Xi(19) = 26.120951319989963723688930485878D+00
      Xi(20) = 32.910312071354180987053927864849D+00
      Xi(21) = 41.560819307413242679194453899072D+00
      Xi(22) = 52.600107250088324689629715180672D+00
      Xi(23) = 66.699641276715551117815206083606D+00
      Xi(24) = 84.698923618478650476859570517263D+00
      Xi(25) = 107.611537314413924532052746485533D+00
      Xi(26) = 136.576865916434009806668115061257D+00
      Xi(27) = 172.678517322811636527934986418131D+00
      Xi(28) = 216.477712329128258686439245650490D+00
      Xi(29) = 267.033458290465157913828875280160D+00
      Xi(30) = 320.286188490862217620458451960985D+00
      Xi(31) = 367.483131109660798974125839322369D+00
      Xi(32) = 396.127659252490586572204023241284D+00
      GOTO 9999
C
C     ----- K = 16 , R = 5E2 -----
C
 1625 CONTINUE
C
      RIni = 5.0D+02
C
      Omega( 1) = 0.0024472131936283410200325512872873701120D+00
      Omega( 2) = 0.0060630961621274316105539587545836255345D+00
      Omega( 3) = 0.0107312550216339942659886158224380459103D+00
      Omega( 4) = 0.0175540514820946540683208264913006679819D+00
      Omega( 5) = 0.0283110517763022486080823217693969162667D+00
      Omega( 6) = 0.0456402961035302483781047472310987700439D+00
      Omega( 7) = 0.0733688359935863446185900713092387093184D+00
      Omega( 8) = 0.1171684766792688380638524231480701587316D+00
      Omega( 9) = 0.1855546600295324656461782436567276022288D+00
      Omega(10) = 0.2913293199019893982065418369664655529050D+00
      Omega(11) = 0.4537929108325465425504644662035858004856D+00
      Omega(12) = 0.7025225505971376122721881729571435926118D+00
      Omega(13) = 1.0850730482430842483310834412080225774844D+00
      Omega(14) = 1.6871056785919717904784903561576925312693D+00
      Omega(15) = 2.7044720053839088518346511769152584747644D+00
      Omega(16) = 4.8773253290853238362152766161017325430294D+00
C
      Alpha( 1) = 0.0009468599039160355319218457159548968383D+00
      Alpha( 2) = 0.0051461608873027619992851695342184203952D+00
      Alpha( 3) = 0.0134185711537655366997178743235230591324D+00
      Alpha( 4) = 0.0273192349904864276411927976467852730025D+00
      Alpha( 5) = 0.0498287460342099060398328089216413161466D+00
      Alpha( 6) = 0.0861193605778245913263567110074170329881D+00
      Alpha( 7) = 0.1445556956959863234243347795970890956596D+00
      Alpha( 8) = 0.2381830158803071012242570311800982096884D+00
      Alpha( 9) = 0.3870414798533882194718838687386153196712D+00
      Alpha(10) = 0.6216847902584319523577698063387941829205D+00
      Alpha(11) = 0.9884865630628114253224993090984895616202D+00
      Alpha(12) = 1.5578388987534183036812521216951665792294D+00
      Alpha(13) = 2.4377619194799008527779760058251667942386D+00
      Alpha(14) = 3.8001562576159002308697626437705707758141D+00
      Alpha(15) = 5.9465865778449154555040934155840659514070D+00
      Alpha(16) = 9.5636980300421801154051038018621966330102D+00
C
      Xi( 1) = 1.005523302139058109761285408901D+00
      Xi( 2) = 1.050108461063545093423440834890D+00
      Xi( 3) = 1.141418414873557743926223151387D+00
      Xi( 4) = 1.283808123736142587463168740491D+00
      Xi( 5) = 1.484048964244604059629223380412D+00
      Xi( 6) = 1.751715896121533585778722030835D+00
      Xi( 7) = 2.099794962973845620325327732125D+00
      Xi( 8) = 2.545518284900301884694090448491D+00
      Xi( 9) = 3.111443809952679282479839906195D+00
      Xi(10) = 3.826838263523562360529650661078D+00
      Xi(11) = 4.729483845715898698500168073622D+00
      Xi(12) = 5.868051273980675961849245991786D+00
      Xi(13) = 7.305226942313801762572678111241D+00
      Xi(14) = 9.121828661475524921073421147355D+00
      Xi(15) = 11.422223160509220610259184525148D+00
      Xi(16) = 14.341464492801081471659574440025D+00
      Xi(17) = 18.054709049381179824730891958495D+00
      Xi(18) = 22.789617009219566576630477960919D+00
      Xi(19) = 28.842636541390743329382750914647D+00
      Xi(20) = 36.600283997038238807675281094589D+00
      Xi(21) = 46.566491235160331074033335241324D+00
      Xi(22) = 59.396774874873065046959608181965D+00
      Xi(23) = 75.938463956485049821676369674606D+00
      Xi(24) = 97.271796448065084995937557010848D+00
      Xi(25) = 124.735614845963427271824386366461D+00
      Xi(26) = 159.895284599243062084727995397770D+00
      Xi(27) = 204.353454151061048646442763754294D+00
      Xi(28) = 259.196002266412646453108692412570D+00
      Xi(29) = 323.720422719421398571881809402839D+00
      Xi(30) = 393.136876103394547282698923140742D+00
      Xi(31) = 455.962930007404209614696455332705D+00
      Xi(32) = 494.722418110257492418480040896611D+00
      GOTO 9999
C
C     ----- K = 16 , R = 6E2 -----
C
 1626 CONTINUE
C
      RIni = 6.0D+02
C
      Omega( 1) = 0.0021053197367968382995978809512499507761D+00
      Omega( 2) = 0.0052388810322415410763938214706164631096D+00
      Omega( 3) = 0.0093527407563119556425679798974948120360D+00
      Omega( 4) = 0.0154958369642747481409480558682545403570D+00
      Omega( 5) = 0.0253648437706326759556062959768962805640D+00
      Omega( 6) = 0.0414818886569631436331208503009992938360D+00
      Omega( 7) = 0.0675490862157180216787254363464132467243D+00
      Omega( 8) = 0.1091252814085100464563775346515939190795D+00
      Omega( 9) = 0.1746371114059133875472378666371398026058D+00
      Omega(10) = 0.2768450406296747271536577705841608576520D+00
      Omega(11) = 0.4350961478436266127289150373869119903247D+00
      Omega(12) = 0.6791506110084174656967709315935621816607D+00
      Omega(13) = 1.0569067323081625278495543351908736440237D+00
      Omega(14) = 1.6544567764087873979741635799278753893304D+00
      Omega(15) = 2.6678292216763005312036433069877716661722D+00
      Omega(16) = 4.8355004456198803571183308935843569997814D+00
C
      Alpha( 1) = 0.0008141838379395475550614482643935199135D+00
      Alpha( 2) = 0.0044346795523015622871485191640317236050D+00
      Alpha( 3) = 0.0116136389856696046680349871174642295557D+00
      Alpha( 4) = 0.0238092630968598776858307806622572222466D+00
      Alpha( 5) = 0.0438383776182723896082556441083921683344D+00
      Alpha( 6) = 0.0766089872341315043370959560198318882840D+00
      Alpha( 7) = 0.1300986481276461124610092229270108532546D+00
      Alpha( 8) = 0.2168455261816722999849375158598618895667D+00
      Alpha( 9) = 0.3562805725175744084559214391871861948857D+00
      Alpha(10) = 0.5782804460787271140227543131384635444192D+00
      Alpha(11) = 0.9285403059074989504475573787178888096605D+00
      Alpha(12) = 1.4768849008164217853503533128822766684607D+00
      Alpha(13) = 2.3310307802189951843514076346153274243989D+00
      Alpha(14) = 3.6629754895839110103317876054873636348930D+00
      Alpha(15) = 5.7747339253388481326335901488988611163222D+00
      Alpha(16) = 9.3529386861084840504645954162299403833458D+00
C
      Xi( 1) = 1.005715257024971536078082612331D+00
      Xi( 2) = 1.051865261327573555262537841060D+00
      Xi( 3) = 1.146461986814843870500363331999D+00
      Xi( 4) = 1.294182500205798182950703290039D+00
      Xi( 5) = 1.502309193265005543408731336630D+00
      Xi( 6) = 1.781165246115183020637548483300D+00
      Xi( 7) = 2.144797001799990852910493543959D+00
      Xi( 8) = 2.611915402446914854497558855506D+00
      Xi( 9) = 3.207111396122765457862222038266D+00
      Xi(10) = 3.962428457963935397650664937963D+00
      Xi(11) = 4.919423697530920916876767678971D+00
      Xi(12) = 6.131899175222163090887700365172D+00
      Xi(13) = 7.669520950434395173585244709358D+00
      Xi(14) = 9.622613902487533195546920428143D+00
      Xi(15) = 12.108520278443703821480426086055D+00
      Xi(16) = 15.280043428652557609137563332524D+00
      Xi(17) = 19.336671649158674922072687074603D+00
      Xi(18) = 24.539526368547471881290111817009D+00
      Xi(19) = 31.231211134348567231081017503413D+00
      Xi(20) = 39.862071504431753059261023253157D+00
      Xi(21) = 51.024602316323522666063894348554D+00
      Xi(22) = 65.497477446429079608880563512230D+00
      Xi(23) = 84.299401550706269052593011537056D+00
      Xi(24) = 108.748525176769570574353274139412D+00
      Xi(25) = 140.510815862846447976686192760098D+00
      Xi(26) = 181.590258826201451869408742822998D+00
      Xi(27) = 234.143479946847548694988105921766D+00
      Xi(28) = 299.856260581767213896364765446378D+00
      Xi(29) = 378.395246141377767284952327031533D+00
      Xi(30) = 464.391670671157115651839220049624D+00
      Xi(31) = 543.625273039097820670928484787510D+00
      Xi(32) = 593.206156763596342440525432948562D+00
      GOTO 9999
C
C     ----- K = 16 , R = 7E2 -----
C
 1627 CONTINUE
C
      RIni = 7.0D+02
C
      Omega( 1) = 0.0018535144254540457011458598956740728703D+00
      Omega( 2) = 0.0046303911519224061630527157149006534098D+00
      Omega( 3) = 0.0083305478664809014863852024840307208819D+00
      Omega( 4) = 0.0139590364258402108302233078987453218645D+00
      Omega( 5) = 0.0231433556018734578973674491722545276673D+00
      Omega( 6) = 0.0383107456954635583907262455505282616741D+00
      Omega( 7) = 0.0630622860656869282578156771235100563899D+00
      Omega( 8) = 0.1028633169367928892955037223355674314007D+00
      Omega( 9) = 0.1660632135759762743307298496486801298033D+00
      Omega(10) = 0.2653813785938354123422784625052628371122D+00
      Omega(11) = 0.4201953469238532187909809095804902767668D+00
      Omega(12) = 0.6604101630611989827640898043092931857245D+00
      Omega(13) = 1.0342079766011074785361442684994415230904D+00
      Omega(14) = 1.6280489633923891337925760058880086944555D+00
      Omega(15) = 2.6381353298352225220622935264458419624134D+00
      Omega(16) = 4.8016213243881142968931541314958622024278D+00
C
      Alpha( 1) = 0.0007164949012989691333421935718478446020D+00
      Alpha( 2) = 0.0039101863821661210702473291268149724331D+00
      Alpha( 3) = 0.0102801072464593958679624180301004354732D+00
      Alpha( 4) = 0.0212070464670612541331542327815995463425D+00
      Alpha( 5) = 0.0393748784721239822396434901441431009061D+00
      Alpha( 6) = 0.0694755195241083988366356805066459401132D+00
      Alpha( 7) = 0.1191707702350980214441944840531029115027D+00
      Alpha( 8) = 0.2005851272064654881112409981480659837416D+00
      Alpha( 9) = 0.3326494625651600446408686501120754996919D+00
      Alpha(10) = 0.5446788598533015113940455864671719155012D+00
      Alpha(11) = 0.8817981251009138986794722026019144323072D+00
      Alpha(12) = 1.4133459060584151906767372275375294066180D+00
      Alpha(13) = 2.2467669668753898573089478052366985139088D+00
      Alpha(14) = 3.5541234904565744520850423260327488605981D+00
      Alpha(15) = 5.6378062888905456332029608734757175625418D+00
      Alpha(16) = 9.1844824112248478770875736998391403176356D+00
C
      Xi( 1) = 1.005876309005510832483322458675D+00
      Xi( 2) = 1.053340084448622940402513770142D+00
      Xi( 3) = 1.150700637067471238149732326583D+00
      Xi( 4) = 1.302914865532209731527682472496D+00
      Xi( 5) = 1.517710126543378411898867208496D+00
      Xi( 6) = 1.806062687551509105790721598783D+00
      Xi( 7) = 2.182950165131476624822554066796D+00
      Xi( 8) = 2.668385994970611529292225938548D+00
      Xi( 9) = 3.288763099913594485949538959169D+00
      Xi(10) = 4.078594878129303225207380467765D+00
      Xi(11) = 5.082817271925541693228245421210D+00
      Xi(12) = 6.359850953453854601350042630159D+00
      Xi(13) = 7.985681648838154654285875766462D+00
      Xi(14) = 10.059296353077444365585468855073D+00
      Xi(15) = 12.709929661761720182631962838826D+00
      Xi(16) = 16.106761007607586591930393105088D+00
      Xi(17) = 20.471892668824389804188568220411D+00
      Xi(18) = 26.097744473175567116521311739064D+00
      Xi(19) = 33.370410916040415448546863075308D+00
      Xi(20) = 42.800865993629198257702794805368D+00
      Xi(21) = 55.066376719235140806824047032819D+00
      Xi(22) = 71.064550369022659426221366629761D+00
      Xi(23) = 91.981228843948959053578029454457D+00
      Xi(24) = 119.369175386683191796011183072324D+00
      Xi(25) = 155.221425507736776452927252023528D+00
      Xi(26) = 201.988642197331957670036750585041D+00
      Xi(27) = 262.407106944497138439897909734100D+00
      Xi(28) = 338.822143677454432147966301158704D+00
      Xi(29) = 431.380209614188204869655862694344D+00
      Xi(30) = 534.275379812551198699388521617948D+00
      Xi(31) = 630.569530725390331893365214455116D+00
      Xi(32) = 691.590930201107461305909396287461D+00
      GOTO 9999
C
C     ----- K = 16 , R = 8E2 -----
C
 1628 CONTINUE
C
      RIni = 8.0D+02
C
      Omega( 1) = 0.0016597110979603259690510122363084344332D+00
      Omega( 2) = 0.0041610632624492284243526794146328384727D+00
      Omega( 3) = 0.0075390201458541255498857405065171732872D+00
      Omega( 4) = 0.0127615706769678872249652000492314840585D+00
      Omega( 5) = 0.0213972122588712194549739453036663050511D+00
      Omega( 6) = 0.0357936419467514498914443264732110350224D+00
      Omega( 7) = 0.0594677802105419486269292234092986859650D+00
      Omega( 8) = 0.0978053035397581935553312781905788142467D+00
      Omega( 9) = 0.1590871684802288474303803150755598494470D+00
      Omega(10) = 0.2559931002217505570557609662074582956848D+00
      Omega(11) = 0.4079207043380653206527015897098209507021D+00
      Omega(12) = 0.6448932537552953053673794603373181644201D+00
      Omega(13) = 1.0153335202525769788688830841394405979372D+00
      Omega(14) = 1.6060220091447995983740745784551506858406D+00
      Omega(15) = 2.6133276848398222708992622020929275095114D+00
      Omega(16) = 4.7733264900750258710043771870346063224133D+00
C
      Alpha( 1) = 0.0006413268094817407344158084552508680698D+00
      Alpha( 2) = 0.0035061734626053672457980382422618181870D+00
      Alpha( 3) = 0.0092508507448277612239634858613823720219D+00
      Alpha( 4) = 0.0191923279358940297880222261572846775834D+00
      Alpha( 5) = 0.0359033684018405671501577344079114695319D+00
      Alpha( 6) = 0.0638944507354568748034272240854392066467D+00
      Alpha( 7) = 0.1105628524589580313801850594557230067494D+00
      Alpha( 8) = 0.1876862369295839864030217905566111369353D+00
      Alpha( 9) = 0.3137737097128846019478739387542631789074D+00
      Alpha(10) = 0.5176624942979187261925602681067459798214D+00
      Alpha(11) = 0.8439866686004004246268554723098276326709D+00
      Alpha(12) = 1.3616601833297931429742066988808346650330D+00
      Alpha(13) = 2.1778821657413933766695574023586345902004D+00
      Alpha(14) = 3.4647582889019940316412821124103516012838D+00
      Alpha(15) = 5.5250005152878595270776740377982605423313D+00
      Alpha(16) = 9.0453357535582769796689284191870683571324D+00
C
      Xi( 1) = 1.006014770006011066765146122481D+00
      Xi( 2) = 1.054608649958689026268805755127D+00
      Xi( 3) = 1.154349912463906832247283185122D+00
      Xi( 4) = 1.310443042542520201824453318640D+00
      Xi( 5) = 1.531009803243571506885765953765D+00
      Xi( 6) = 1.827607135648407961814494726571D+00
      Xi( 7) = 2.216043517152185178759396522175D+00
      Xi( 8) = 2.717499782918300837942712777640D+00
      Xi( 9) = 3.359988834639446199283735472463D+00
      Xi(10) = 4.180256279270505156194642770728D+00
      Xi(11) = 5.226303654756044108319057439616D+00
      Xi(12) = 6.560763652753605687864557616251D+00
      Xi(13) = 8.265412865630715556590490944444D+00
      Xi(14) = 10.447215904516594989909217261470D+00
      Xi(15) = 13.246427732457084346096021221229D+00
      Xi(16) = 16.847480533181990430824370719165D+00
      Xi(17) = 21.493649889102426278966739836562D+00
      Xi(18) = 27.506867419054068542652591311537D+00
      Xi(19) = 35.314460273588206010164647352667D+00
      Xi(20) = 45.485262313086833711045509076598D+00
      Xi(21) = 58.778035298133601598696840184743D+00
      Xi(22) = 76.205516395195813147966923395416D+00
      Xi(23) = 99.116695413420177446228720441468D+00
      Xi(24) = 129.295537243414167843935658197552D+00
      Xi(25) = 169.060988432439411974228526958086D+00
      Xi(26) = 221.315575358555647347835737548394D+00
      Xi(27) = 289.394717428705165518376674071988D+00
      Xi(28) = 376.351687843882334871814521193301D+00
      Xi(29) = 482.908399988066930896835060593730D+00
      Xi(30) = 602.953743961086938318860717345160D+00
      Xi(31) = 716.870063847640778764702673697684D+00
      Xi(32) = 789.885808266521592280184904666385D+00
      GOTO 9999
C
C     ----- K = 16 , R = 9E2 -----
C
 1629 CONTINUE
C
      RIni = 9.0D+02
C
      Omega( 1) = 0.0015055631595250562666732919406449575650D+00
      Omega( 2) = 0.0037870450506321845648883704562454033393D+00
      Omega( 3) = 0.0069059686538486601540820237609547005420D+00
      Omega( 4) = 0.0117983671303826509692803882652090141647D+00
      Omega( 5) = 0.0199815601654048778368355770290731854288D+00
      Omega( 6) = 0.0337352970754505528657785692198123328467D+00
      Omega( 7) = 0.0565047866336671157529018046788982232442D+00
      Omega( 8) = 0.0936063868568455301861445244648862740178D+00
      Omega( 9) = 0.1532596650407880263462844297461806775118D+00
      Omega(10) = 0.2481064208093561478975230816246266840608D+00
      Omega(11) = 0.3975573296197845333881885215854268267321D+00
      Omega(12) = 0.6317344227996357830978886582773412783354D+00
      Omega(13) = 0.9992684300875393019073303357835413862631D+00
      Omega(14) = 1.5872230602253215233114411186754466598359D+00
      Omega(15) = 2.5921258825123527603454670620308775141893D+00
      Omega(16) = 4.7491508522890109713146533643168822891312D+00
C
      Alpha( 1) = 0.0005815528588767195223618469517284229608D+00
      Alpha( 2) = 0.0031845872823828756917046448266309099928D+00
      Alpha( 3) = 0.0084300984840787595718698821389835984519D+00
      Alpha( 4) = 0.0175811659326021210709409302655470419552D+00
      Alpha( 5) = 0.0331156506410349317250628193437522739373D+00
      Alpha( 6) = 0.0593885926005389108910754916020868776627D+00
      Alpha( 7) = 0.1035710999862279087105690345005371000298D+00
      Alpha( 8) = 0.1771438940187624976182675928804144405149D+00
      Alpha( 9) = 0.2982529920600857122872564403115447362325D+00
      Alpha(10) = 0.4953210601193777047124896378837988208943D+00
      Alpha(11) = 0.8125524576330856438071900682551529371267D+00
      Alpha(12) = 1.3184844662501273938664744789228677745996D+00
      Alpha(13) = 2.1200923357897898953016574408181327271450D+00
      Alpha(14) = 3.3895107039352260140489814244091348882648D+00
      Alpha(15) = 5.4297307903033270120272957282026027314714D+00
      Alpha(16) = 8.9275533544404011261977349711571605439531D+00
C
      Xi( 1) = 1.006136012409362691789709021162D+00
      Xi( 2) = 1.055719931445410426961231675769D+00
      Xi( 3) = 1.157549330324401512287875448681D+00
      Xi( 4) = 1.317050855184951755109078186390D+00
      Xi( 5) = 1.542700605008748187488981651327D+00
      Xi( 6) = 1.846578984154591802132466815678D+00
      Xi( 7) = 2.245245130934919511583175433245D+00
      Xi( 8) = 2.760939294470852008438893632203D+00
      Xi( 9) = 3.423148362551289448007726945455D+00
      Xi(10) = 4.270657235986691423797884104019D+00
      Xi(11) = 5.354279482436550032398853149473D+00
      Xi(12) = 6.740525940807517171326351812510D+00
      Xi(13) = 8.516531260420975283156885815572D+00
      Xi(14) = 10.796675502208701058828865804173D+00
      Xi(15) = 13.731498439456119056018001955977D+00
      Xi(16) = 17.519740105992699359324649144298D+00
      Xi(17) = 22.424651605042061531039565558387D+00
      Xi(18) = 28.796117372762449860418287883590D+00
      Xi(19) = 37.100772684586712478471692833182D+00
      Xi(20) = 47.962900980259937141342296840207D+00
      Xi(21) = 62.219816526658292429180496796448D+00
      Xi(22) = 80.996014694373807858707170481694D+00
      Xi(23) = 105.799881084980490091040916666998D+00
      Xi(24) = 138.643125702413568964277423845033D+00
      Xi(25) = 182.168901573139119215438697096943D+00
      Xi(26) = 239.734799690210702946546739156020D+00
      Xi(27) = 315.290477426446540731852508088195D+00
      Xi(28) = 412.637447746722082458914471203570D+00
      Xi(29) = 533.156399397003743667600161870723D+00
      Xi(30) = 670.554597887061645367090534364252D+00
      Xi(31) = 802.584897091940211777405522752815D+00
      Xi(32) = 888.097908151572883517221868032721D+00
      GOTO 9999
C
C     ----- K = 16 , R = 1E3 -----
C
 1631 CONTINUE
C
      RIni = 1.0D+03
C
      Omega( 1) = 0.0013797888502005507772685364165782262447D+00
      Omega( 2) = 0.0034813296515988117510506102492487401356D+00
      Omega( 3) = 0.0063868138275758572432797531299303317454D+00
      Omega( 4) = 0.0110042750893626431830035284145652418886D+00
      Omega( 5) = 0.0188060416343505131900348294193836551358D+00
      Omega( 6) = 0.0320129266794980219734613902798914253367D+00
      Omega( 7) = 0.0540079284084068265886003669395964976729D+00
      Omega( 8) = 0.0900461102030354742912533760890170242419D+00
      Omega( 9) = 0.1482913872171991381761256903470957979607D+00
      Omega(10) = 0.2413494780456323231487792899119249057094D+00
      Omega(11) = 0.3886392366518692786925542187459470255817D+00
      Omega(12) = 0.6203667158670121572648301078167776267946D+00
      Omega(13) = 0.9853451315581527809373853232788320610780D+00
      Omega(14) = 1.5708916426203869327361817287425083122798D+00
      Omega(15) = 2.5736841409626707883207680227499736247410D+00
      Omega(16) = 4.7281271127462638696574426244723099443945D+00
C
      Alpha( 1) = 0.0005327914492434099262305651010639823140D+00
      Alpha( 2) = 0.0029220140733091449616117170340334752154D+00
      Alpha( 3) = 0.0077588419683956976036238583228216159071D+00
      Alpha( 4) = 0.0162599925842890798076650731018855111643D+00
      Alpha( 5) = 0.0308208827514734286988985810865071535147D+00
      Alpha( 6) = 0.0556612598231712853493206773908275053486D+00
      Alpha( 7) = 0.0977557047572212667619902075333282454039D+00
      Alpha( 8) = 0.1683264855900836211966649022531150592386D+00
      Alpha( 9) = 0.2852019800763537923861574985284761396542D+00
      Alpha(10) = 0.4764397276834995646332005653222907426425D+00
      Alpha(11) = 0.7858625488052698108542432064993477069947D+00
      Alpha(12) = 1.2816696415764265190410930994602267674054D+00
      Alpha(13) = 2.0706307332283275014512408107592023043253D+00
      Alpha(14) = 3.3248991112367181318450887195226073345111D+00
      Alpha(15) = 5.3477122672426787952475180087930084482650D+00
      Alpha(16) = 8.8259523750929880087090495521806587930769D+00
C
      Xi( 1) = 1.006243706349369666569124048827D+00
      Xi( 2) = 1.056707400670955050996265012220D+00
      Xi( 3) = 1.160394330619598642897680185193D+00
      Xi( 4) = 1.322932700258313029866713561589D+00
      Xi( 5) = 1.553120395588735876829522553066D+00
      Xi( 6) = 1.863514697251501866875032786819D+00
      Xi( 7) = 2.271360233615394187169089779132D+00
      Xi( 8) = 2.799867055723205390514035650895D+00
      Xi( 9) = 3.479877277249979421582318228623D+00
      Xi(10) = 4.352054420590895917123064329246D+00
      Xi(11) = 5.469813169292706701418022019467D+00
      Xi(12) = 6.903265503925697304816316846043D+00
      Xi(13) = 8.744539676385018759155465239985D+00
      Xi(14) = 11.114951975751633667273410299714D+00
      Xi(15) = 14.174706306404383061338381200045D+00
      Xi(16) = 18.136049297776665786829930482327D+00
      Xi(17) = 23.281151209986575804575248405115D+00
      Xi(18) = 29.986529513007290685677741493009D+00
      Xi(19) = 38.756415836380956937284869567861D+00
      Xi(20) = 50.268415213302309007409318653004D+00
      Xi(21) = 65.435796873362536800466937592091D+00
      Xi(22) = 85.491662295261926479694736258352D+00
      Xi(23) = 112.100398226085366817916799675459D+00
      Xi(24) = 147.498040056546388928970792164819D+00
      Xi(25) = 194.649827778459253449505617084014D+00
      Xi(26) = 257.370255453781749288566516042920D+00
      Xi(27) = 340.235188693453923319820475512643D+00
      Xi(28) = 447.828326319095989110907751751256D+00
      Xi(29) = 582.262922250848045713222944641529D+00
      Xi(30) = 737.179619506007345774989403253130D+00
      Xi(31) = 887.760886989793165269269081818493D+00
      Xi(32) = 986.233000349415966367949692994443D+00
      GOTO 9999
C
C     ----- K = 16 , R = 2E3 -----
C
 1632 CONTINUE
C
      RIni = 2.0D+03
C
      Omega( 1) = 0.0007768847420693003804794456927489132436D+00
      Omega( 2) = 0.0020065660238064882712827270748803809575D+00
      Omega( 3) = 0.0038523102169686026524150345307778708914D+00
      Omega( 4) = 0.0070518785440354660497554411917320305037D+00
      Omega( 5) = 0.0128051535867035469926666934638304073957D+00
      Omega( 6) = 0.0229904356331946649850896287337054246791D+00
      Omega( 7) = 0.0406223196249400557508322986283166855515D+00
      Omega( 8) = 0.0705684431621601551419043808766051739667D+00
      Omega( 9) = 0.1206153238244931053891120797949554344086D+00
      Omega(10) = 0.2030907673714956750333658283547144662862D+00
      Omega(11) = 0.3373950010841398567469802405316414706249D+00
      Omega(12) = 0.5541903301702226011934623861598936400696D+00
      Omega(13) = 0.9034013813719590036484426154306248690773D+00
      Omega(14) = 1.4739942115085553971543658180465286022809D+00
      Omega(15) = 2.4637954099354931818913888585953486654034D+00
      Omega(16) = 4.6029358609429098189193274581754167229519D+00
C
      Alpha( 1) = 0.0002992241208797295031346425477894110379D+00
      Alpha( 2) = 0.0016602631356552019942533424123887186852D+00
      Alpha( 3) = 0.0045138170003323857566210179376869904289D+00
      Alpha( 4) = 0.0098111412969511009946667898576344590822D+00
      Alpha( 5) = 0.0194616438373382750691399038084350436861D+00
      Alpha( 6) = 0.0368864680879441863142482582044179295622D+00
      Alpha( 7) = 0.0679045674012652014929730497072579709084D+00
      Alpha( 8) = 0.1222041350629109780914034397159007028222D+00
      Alpha( 9) = 0.2156923865273337429540557660945232498761D+00
      Alpha(10) = 0.3741677069120103561001774994387858441769D+00
      Alpha(11) = 0.6390292853494859240956818546841589068208D+00
      Alpha(12) = 1.0762526604724020765427305867945051431889D+00
      Alpha(13) = 1.7911592740081455582632635747941662884841D+00
      Alpha(14) = 2.9558684609147227069503305330044895526953D+00
      Alpha(15) = 4.8751392319118493660565216707425406639231D+00
      Alpha(16) = 8.2366778447943210109474798663597994163865D+00
C
      Xi( 1) = 1.006931202362213098188033211233D+00
      Xi( 2) = 1.063019469384929592374600115789D+00
      Xi( 3) = 1.178625720298270394695641649641D+00
      Xi( 4) = 1.360758809358223196716089486991D+00
      Xi( 5) = 1.620433485216211504163601186956D+00
      Xi( 6) = 1.973517348237369513938403720044D+00
      Xi( 7) = 2.442066143829587792279597890932D+00
      Xi( 8) = 3.056165694352967220071004561532D+00
      Xi( 9) = 3.856366173641724338222447343583D+00
      Xi(10) = 4.896955873372914460044807993810D+00
      Xi(11) = 6.250445492056464633773282812967D+00
      Xi(12) = 8.013740354402243216795642677397D+00
      Xi(13) = 10.316640688505583998285652924665D+00
      Xi(14) = 13.333568352935279436623228210834D+00
      Xi(15) = 17.299840002468369401858461209542D+00
      Xi(16) = 22.534299577395717753200421462623D+00
      Xi(17) = 29.471029029283412697948207892296D+00
      Xi(18) = 38.703928997986791538110651345050D+00
      Xi(19) = 51.049828735696785243086681660429D+00
      Xi(20) = 67.638080241728973672488134383229D+00
      Xi(21) = 90.038414614460910108395541584514D+00
      Xi(22) = 120.443318367013035181944946572230D+00
      Xi(23) = 161.927067404595660893296127369467D+00
      Xi(24) = 218.808000562085702991876878797939D+00
      Xi(25) = 297.133859138640721020419022124770D+00
      Xi(26) = 405.264591321704776033518768940667D+00
      Xi(27) = 554.347801774285637022021688835594D+00
      Xi(28) = 757.909757688632282901508574468608D+00
      Xi(29) = 1028.168681040490993860991864039534D+00
      Xi(30) = 1363.125723846814662976179022280121D+00
      Xi(31) = 1716.193156804617456989703327963070D+00
      Xi(32) = 1964.134483811259450303232654277963D+00
      GOTO 9999
C
C     ----- K = 16 , R = 3E3 -----
C
 1633 CONTINUE
C
      RIni = 3.0D+03
C
      Omega( 1) = 0.0005552467027450946637557560830837644072D+00
      Omega( 2) = 0.0014584383837619173779302032801195865286D+00
      Omega( 3) = 0.0028902827511056797591154930498688613305D+00
      Omega( 4) = 0.0055003742951535241132696206836638630477D+00
      Omega( 5) = 0.0103516891504137027865415699758033785116D+00
      Omega( 6) = 0.0191571445523666846210585805801440706375D+00
      Omega( 7) = 0.0347438857823840648833720046173567652659D+00
      Omega( 8) = 0.0617661025400090076408908448118983969266D+00
      Omega( 9) = 0.1077870257402970902431589561576430469358D+00
      Omega(10) = 0.1849494100974524521819718908544061264365D+00
      Omega(11) = 0.3125950216953475513045630079789205524321D+00
      Omega(12) = 0.5215821469062483484689651813681621206342D+00
      Omega(13) = 0.8624104031684323868472974439214340236504D+00
      Omega(14) = 1.4249765739881087528526798502426231607387D+00
      Omega(15) = 2.4078703105469664194453194294709419409628D+00
      Omega(16) = 4.5392702166529680200823815283683870802633D+00
C
      Alpha( 1) = 0.0002134705217666186853756709926430330804D+00
      Alpha( 2) = 0.0011944382121612197912085697013191698534D+00
      Alpha( 3) = 0.0033030929481546051993269118990381727485D+00
      Alpha( 4) = 0.0073633574898127585851037601484547767683D+00
      Alpha( 5) = 0.0150440187979848748675706449420830246311D+00
      Alpha( 6) = 0.0293726979281437198497910646272091028663D+00
      Alpha( 7) = 0.0555989091923544578093484801184498422799D+00
      Alpha( 8) = 0.1026407267113996364378634279512780835830D+00
      Alpha( 9) = 0.1854143263568294130013086942754263475308D+00
      Alpha(10) = 0.3285173216429233088295078224483347639762D+00
      Alpha(11) = 0.5720171780433248305921568876986071927604D+00
      Alpha(12) = 0.9806147524370287133508097832113037384261D+00
      Alpha(13) = 1.6587347438742525073882005948178175458452D+00
      Alpha(14) = 2.7783675668309569231542027756276524996792D+00
      Alpha(15) = 4.6450720342179955234251931273092850460671D+00
      Alpha(16) = 7.9472026835846368108921822237533660882036D+00
C
      Xi( 1) = 1.007313518843293634764252120206D+00
      Xi( 2) = 1.066535862309835448508317512495D+00
      Xi( 3) = 1.188816594963225726860021891351D+00
      Xi( 4) = 1.382003997960557570312024877612D+00
      Xi( 5) = 1.658469578868685293773950484297D+00
      Xi( 6) = 2.036129981586481010272932090643D+00
      Xi( 7) = 2.540058670835582692445683838933D+00
      Xi( 8) = 3.204711797327240542206452933449D+00
      Xi( 9) = 4.076896440729057241592059313007D+00
      Xi(10) = 5.219814824401254844260522602895D+00
      Xi(11) = 6.718675885724491570871047541047D+00
      Xi(12) = 8.688507703690568208937383687385D+00
      Xi(13) = 11.285044312162827710721568763574D+00
      Xi(14) = 14.719953669818344563213763365184D+00
      Xi(15) = 19.282243192993958297992707073831D+00
      Xi(16) = 25.368500856855381066223831254547D+00
      Xi(17) = 33.525907356905569393218380369603D+00
      Xi(18) = 44.513791278037670100242717552419D+00
      Xi(19) = 59.392255900397037244459186311474D+00
      Xi(20) = 79.650749218774877007109846971389D+00
      Xi(21) = 107.395501977541313345931861533700D+00
      Xi(22) = 145.624118898835682209269570819288D+00
      Xi(23) = 198.628445768431519641294968891998D+00
      Xi(24) = 272.582126817676001551138398326657D+00
      Xi(25) = 376.379417355359462415975713156513D+00
      Xi(26) = 522.761652063993322581136879989572D+00
      Xi(27) = 729.578496466861973479556269239765D+00
      Xi(28) = 1020.272640770886480754420233552082D+00
      Xi(29) = 1420.156514258006317574789534319279D+00
      Xi(30) = 1938.106130483476449266611041366559D+00
      Xi(31) = 2513.149382258735478057332102252985D+00
      Xi(32) = 2937.145917315011803161439729592530D+00
      GOTO 9999
C
C     ----- K = 16 , R = 4E3 -----
C
 1634 CONTINUE
C
      RIni = 4.0D+03
C
      Omega( 1) = 0.0004376510700857689912567819958853429507D+00
      Omega( 2) = 0.0011653666641048586782173284792998857462D+00
      Omega( 3) = 0.0023680456391125560362919923651156550726D+00
      Omega( 4) = 0.0046379834344294636243877781069111509993D+00
      Omega( 5) = 0.0089518482939726288836161530518276130408D+00
      Omega( 6) = 0.0169190097024326531614162125866096531013D+00
      Omega( 7) = 0.0312441751584375854244819475588612611716D+00
      Omega( 8) = 0.0564366963334440862069471545048315874737D+00
      Omega( 9) = 0.0999033373206869644461464337326095375147D+00
      Omega(10) = 0.1736501546243619733021290932328817291364D+00
      Omega(11) = 0.2969613931046048855015746792407327347973D+00
      Omega(12) = 0.5008065845528724652582254484034507413526D+00
      Omega(13) = 0.8360592730493393466692161886122391933895D+00
      Omega(14) = 1.3932536792220334117067825907554379227804D+00
      Omega(15) = 2.3715445629975790188277012804363153009035D+00
      Omega(16) = 4.4979306792040948293105728783558561190148D+00
C
      Alpha( 1) = 0.0001680125928201301219977655484224925153D+00
      Alpha( 2) = 0.0009465404888159955446108676782243640524D+00
      Alpha( 3) = 0.0026538938383495571741655440775944918563D+00
      Alpha( 4) = 0.0060343511107069091400544535389732780928D+00
      Alpha( 5) = 0.0126044257319316648984378688609642171770D+00
      Alpha( 6) = 0.0251439404415568449864206877417127650176D+00
      Alpha( 7) = 0.0485422220818330998734133324007400389632D+00
      Alpha( 8) = 0.0912233302740963573004037527602871904264D+00
      Alpha( 9) = 0.1674571159436080884457508088924981848322D+00
      Alpha(10) = 0.3010446184572926708151400371216155349430D+00
      Alpha(11) = 0.5311533323793717916816263524548702434913D+00
      Alpha(12) = 0.9216025899928024695028153101539203362336D+00
      Alpha(13) = 1.5761719145679200359405625064823652792256D+00
      Alpha(14) = 2.6667207963614877679840026170055011789373D+00
      Alpha(15) = 4.4993322323555962252011886093328030256089D+00
      Alpha(16) = 7.7628668850489034132175691915733750647632D+00
C
      Xi( 1) = 1.007574569294012870328443309820D+00
      Xi( 2) = 1.068939445826771577870498153295D+00
      Xi( 3) = 1.195796686120764102658972638515D+00
      Xi( 4) = 1.396597582035634397060186451256D+00
      Xi( 5) = 1.684692532946183825921446231000D+00
      Xi( 6) = 2.079486669305312379792127774891D+00
      Xi( 7) = 2.608262938092970148690544207604D+00
      Xi( 8) = 3.308703009241117815606228469427D+00
      Xi( 9) = 4.232270880775881090156420594184D+00
      Xi(10) = 5.448864721892181405080851641642D+00
      Xi(11) = 7.053326496492226941638359338782D+00
      Xi(12) = 9.174574041637437877769190031074D+00
      Xi(13) = 11.988440511077821115822061859291D+00
      Xi(14) = 15.735778913915754973276706074614D+00
      Xi(15) = 20.748179556649722623737353011109D+00
      Xi(16) = 27.484647354795787521641869943778D+00
      Xi(17) = 36.584363092090761336583337737238D+00
      Xi(18) = 48.943045590636358243424863800897D+00
      Xi(19) = 65.824300738152503126299208702221D+00
      Xi(20) = 89.023331061222415204470248539792D+00
      Xi(21) = 121.109255467692712819605116436605D+00
      Xi(22) = 165.786268551979793922646067017013D+00
      Xi(23) = 228.434385063764427259425104921320D+00
      Xi(24) = 316.918677395432000221120460992097D+00
      Xi(25) = 442.786666521232530724372367103570D+00
      Xi(26) = 622.972120263308429599646842689253D+00
      Xi(27) = 881.953039206869549737621838403356D+00
      Xi(28) = 1253.450674381069734564242423857650D+00
      Xi(29) = 1777.482136531094566778143928331701D+00
      Xi(30) = 2478.162137296291906984890829335200D+00
      Xi(31) = 3286.862161306532813975778140047623D+00
      Xi(32) = 3906.325460007557433206315522511431D+00
      GOTO 9999
C
C     ----- K = 16 , R = 5E3 -----
C
 1635 CONTINUE
C
      RIni = 5.0D+03
C
      Omega( 1) = 0.0003639842736266964178310734200202589128D+00
      Omega( 2) = 0.0009806443211126502088166863421898888920D+00
      Omega( 3) = 0.0020348019995182053730111034189204088918D+00
      Omega( 4) = 0.0040773537715263094949365708756505721766D+00
      Omega( 5) = 0.0080241245702683121638685410058344782414D+00
      Omega( 6) = 0.0154111678077861830196034474173283346232D+00
      Omega( 7) = 0.0288537422675381651965184062400004982862D+00
      Omega( 8) = 0.0527529528046750570404127642781810791917D+00
      Omega( 9) = 0.0943962899936182988163920794255634660885D+00
      Omega(10) = 0.1656821898372087954162722514550587504800D+00
      Omega(11) = 0.2858429295176569531538798840664128420030D+00
      Omega(12) = 0.4859200114387559206441911940910216571865D+00
      Omega(13) = 0.8170578475224640865501833653095076215322D+00
      Omega(14) = 1.3702697547901459369443233349272759369342D+00
      Omega(15) = 2.3451567673310105685210991133260449714726D+00
      Omega(16) = 4.4679066019423948753590269955537905843812D+00
C
      Alpha( 1) = 0.0001395561408272046871737763002385354638D+00
      Alpha( 2) = 0.0007908754043432692235792866254553284966D+00
      Alpha( 3) = 0.0022437319728956972643010923506440290609D+00
      Alpha( 4) = 0.0051861449510816475890272018740151382943D+00
      Alpha( 5) = 0.0110265465390968265123897368643057781412D+00
      Alpha( 6) = 0.0223694637981097021065608927703483033156D+00
      Alpha( 7) = 0.0438480579093586883351212936388296981249D+00
      Alpha( 8) = 0.0835312227650857650027189926777992923235D+00
      Alpha( 9) = 0.1552186994022961743486038396766524272152D+00
      Alpha(10) = 0.2821252956665503958794648242669467208543D+00
      Alpha(11) = 0.5027477680571566028976289441931868395841D+00
      Alpha(12) = 0.8802381938407273885690770831580209687672D+00
      Alpha(13) = 1.5178750221968237785891142621608196350280D+00
      Alpha(14) = 2.5873972191280157082240998001054776977981D+00
      Alpha(15) = 4.3952693534984634562111371902659584520734D+00
      Alpha(16) = 7.6307623342543931983794625040218306821771D+00
C
      Xi( 1) = 1.007770795780931430678933891087D+00
      Xi( 2) = 1.070747566096610958611204389435D+00
      Xi( 3) = 1.201055232665560991260622625276D+00
      Xi( 4) = 1.407614329043466570032597240392D+00
      Xi( 5) = 1.704539598973179336037324371755D+00
      Xi( 6) = 2.112404195000717510837667800772D+00
      Xi( 7) = 2.660234480703984382820442844420D+00
      Xi( 8) = 3.388271982928642332612984011675D+00
      Xi( 9) = 4.351697965542117357059387350304D+00
      Xi(10) = 5.625791153667208028144736120524D+00
      Xi(11) = 7.313189251692526079984268960743D+00
      Xi(12) = 9.554136392540210479841500390563D+00
      Xi(13) = 12.540973361192514306389211853343D+00
      Xi(14) = 16.538737305833064141630250887260D+00
      Xi(15) = 21.914584017855716394432286087124D+00
      Xi(16) = 29.180138251998641833392933442326D+00
      Xi(17) = 39.052865128266937247425438783921D+00
      Xi(18) = 52.545679566284073297444123795685D+00
      Xi(19) = 71.098914463001920728546512151524D+00
      Xi(20) = 96.776240058579681938000227603425D+00
      Xi(21) = 132.558018298638639761999336030840D+00
      Xi(22) = 182.784080056747836526276351776232D+00
      Xi(23) = 253.826595804328427108398358313934D+00
      Xi(24) = 355.115530315099602431949676883960D+00
      Xi(25) = 500.693591776281145433458164006879D+00
      Xi(26) = 711.511993913223033003667694629257D+00
      Xi(27) = 1018.549699793377793732851444019616D+00
      Xi(28) = 1465.938683276471933170093109310983D+00
      Xi(29) = 2109.391908448062716407989114486554D+00
      Xi(30) = 2991.442293428318250025910174372257D+00
      Xi(31) = 4041.741988572836854709535714391677D+00
      Xi(32) = 4872.251104496301511215250457098591D+00
      GOTO 9999
C
C     ----- K = 16 , R = 6E3 -----
C
 1636 CONTINUE
C
      RIni = 6.0D+03
C
      Omega( 1) = 0.0003131740893025812162727879048524162742D+00
      Omega( 2) = 0.0008525743777120362815273941689813569367D+00
      Omega( 3) = 0.0018013124768756676217053899063946786896D+00
      Omega( 4) = 0.0036784434068537452545763754294463887540D+00
      Omega( 5) = 0.0073538954069976066201639480247875813035D+00
      Omega( 6) = 0.0143079355017790147446262523540927436017D+00
      Omega( 7) = 0.0270861317219600750889917963049047244795D+00
      Omega( 8) = 0.0500039451671855890466740404004530873294D+00
      Omega( 9) = 0.0902531713884734725832168639181318781084D+00
      Omega(10) = 0.1596439284804761997286527697159641547842D+00
      Omega(11) = 0.2773620927927093378633112419340278620439D+00
      Omega(12) = 0.4744994636619417472235635951927079645429D+00
      Omega(13) = 0.8024095949928360991892060949393084001713D+00
      Omega(14) = 1.3524864442578875393420612116557322224253D+00
      Omega(15) = 2.3246983701212194409983086140769614758028D+00
      Omega(16) = 4.4446320157087978023648100478482092512422D+00
C
      Alpha( 1) = 0.0001199406337778443636857985098989262673D+00
      Alpha( 2) = 0.0006832918618089672388053348428115646440D+00
      Alpha( 3) = 0.0019587716841422183862100490567281774190D+00
      Alpha( 4) = 0.0045917452487347416349730230313019863075D+00
      Alpha( 5) = 0.0099085570350233827150601239978910950157D+00
      Alpha( 6) = 0.0203808716667070842385535477461755426987D+00
      Alpha( 7) = 0.0404466664137481064933844331771295443900D+00
      Alpha( 8) = 0.0779018628897185319790168363252025329757D+00
      Alpha( 9) = 0.1461816622304678183863046320167811131796D+00
      Alpha(10) = 0.2680422516895201304264022279522095004722D+00
      Alpha(11) = 0.4814506743542380286259886601474278222668D+00
      Alpha(12) = 0.8490259511837918704729027963296061898291D+00
      Alpha(13) = 1.4736387032169466420683870411245663945010D+00
      Alpha(14) = 2.5269188569364048871281752672857123798167D+00
      Alpha(15) = 4.3156266381570576305161390795461784364306D+00
      Alpha(16) = 7.5293759519457498950376073132417786837323D+00
C
      Xi( 1) = 1.007926880878180067591110002923D+00
      Xi( 2) = 1.072186661890797544074635516687D+00
      Xi( 3) = 1.205245264343925420112181579047D+00
      Xi( 4) = 1.416406408431956687773815850573D+00
      Xi( 5) = 1.720410532419148070897539637425D+00
      Xi( 6) = 2.138790611290749044993975136464D+00
      Xi( 7) = 2.702011854768343456596399465575D+00
      Xi( 8) = 3.452437690046877283951351489399D+00
      Xi( 9) = 4.448344337372049695205483699212D+00
      Xi(10) = 5.769514515083047219781708925268D+00
      Xi(11) = 7.525145864411813474790374778323D+00
      Xi(12) = 9.865065558558741913922074395771D+00
      Xi(13) = 12.995677729521920621379160731834D+00
      Xi(14) = 17.202730517444799317877679456501D+00
      Xi(15) = 22.884064885755052969756873082474D+00
      Xi(16) = 30.597010199494975324699397312145D+00
      Xi(17) = 41.127511184094246686482332364676D+00
      Xi(18) = 55.591839301494131185543690065742D+00
      Xi(19) = 75.587438261880676916570909185111D+00
      Xi(20) = 103.418740278164994830789602531240D+00
      Xi(21) = 142.438347995312840466031190089780D+00
      Xi(22) = 197.567170711043371827964065801098D+00
      Xi(23) = 276.094099660891465264356270381541D+00
      Xi(24) = 388.911863069203033865806062863157D+00
      Xi(25) = 552.425595514388325379950828164510D+00
      Xi(26) = 791.446501710242535265749097561638D+00
      Xi(27) = 1143.313849906627009955784046724148D+00
      Xi(28) = 1662.593202381695404956651884731400D+00
      Xi(29) = 2421.343267206676987290947522524220D+00
      Xi(30) = 3482.956847843122484276179307016719D+00
      Xi(31) = 4780.612701110991521691317984732450D+00
      Xi(32) = 5835.298406764837678473156756808748D+00
      GOTO 9999
C
C     ----- K = 16 , R = 7E3 -----
C
 1637 CONTINUE
C
      RIni = 7.0D+03
C
      Omega( 1) = 0.0002758456244211198484029873463641013842D+00
      Omega( 2) = 0.0007580596968981613001793861046119504810D+00
      Omega( 3) = 0.0016273891617054744144489056203312093984D+00
      Omega( 4) = 0.0033773574743788997862165452432461898447D+00
      Omega( 5) = 0.0068416530214521522122995110266191476356D+00
      Omega( 6) = 0.0134560408443224474344041604462196559666D+00
      Omega( 7) = 0.0257094629782190379569734766176947182714D+00
      Omega( 8) = 0.0478470046371534687721445607000436694989D+00
      Omega( 9) = 0.0869809853452546726990633751963599706869D+00
      Omega(10) = 0.1548469094737899646985722357239723123712D+00
      Omega(11) = 0.2705890951533227137845803028648461463490D+00
      Omega(12) = 0.4653363336543718229733595376229615681041D+00
      Omega(13) = 0.7906106418418388792081728799754358760765D+00
      Omega(14) = 1.3381198015624958747374351197834130289266D+00
      Omega(15) = 2.3081433515491244952422267289193769101985D+00
      Omega(16) = 4.4257998198618129855226588365013640213874D+00
C
      Alpha( 1) = 0.0001055373074657432837801094713465251823D+00
      Alpha( 2) = 0.0006041148903459930011925159410151459083D+00
      Alpha( 3) = 0.0017480808776888785328212407021997165657D+00
      Alpha( 4) = 0.0041489235133947849417641688934271781974D+00
      Alpha( 5) = 0.0090677748285274670356467308952874706307D+00
      Alpha( 6) = 0.0188708736505370171843322260954445290793D+00
      Alpha( 7) = 0.0378405483771319748233371519502421520542D+00
      Alpha( 8) = 0.0735534743950406653969728044961673418811D+00
      Alpha( 9) = 0.1391499701720789750611969454174854377015D+00
      Alpha(10) = 0.2570126196928731625095890939158849164414D+00
      Alpha(11) = 0.4646737063801709214359327015930745119476D+00
      Alpha(12) = 0.8243107746878442959105083576165640124600D+00
      Alpha(13) = 1.4384517778893632151309203703881678393373D+00
      Alpha(14) = 2.4786279831485562544397815987196054265951D+00
      Alpha(15) = 4.2518384035905261455166870288024938417948D+00
      Alpha(16) = 7.4479905527901974220290892692020179310930D+00
C
      Xi( 1) = 1.008055779207782207780239036587D+00
      Xi( 2) = 1.073375671584851511462278816911D+00
      Xi( 3) = 1.208710313926687793728217967804D+00
      Xi( 4) = 1.423686547017540950078005879131D+00
      Xi( 5) = 1.733573479654473688000432263223D+00
      Xi( 6) = 2.160717628625757772385662658365D+00
      Xi( 7) = 2.736808038339744198266800800923D+00
      Xi( 8) = 3.506018171653663538555115164463D+00
      Xi( 9) = 4.529277527399502834264816142351D+00
      Xi(10) = 5.890241476602233601424923881140D+00
      Xi(11) = 7.703775160018586064866047413702D+00
      Xi(12) = 10.128028644161589948494150981162D+00
      Xi(13) = 13.381664314000179407879165260908D+00
      Xi(14) = 17.768600388391513886232608854243D+00
      Xi(15) = 23.713719700629780573475358096402D+00
      Xi(16) = 31.814861739974554624352243159535D+00
      Xi(17) = 42.919065660375737857046285483875D+00
      Xi(18) = 58.235365994749826740878306807758D+00
      Xi(19) = 79.503158873587560423468190151652D+00
      Xi(20) = 109.245995836587721304622444762344D+00
      Xi(21) = 151.157855596998021305288339988238D+00
      Xi(22) = 210.696927268345853390130173465877D+00
      Xi(23) = 296.007248588640691083773148761793D+00
      Xi(24) = 419.359245334154272927262852022068D+00
      Xi(25) = 599.406688185068775298613985569318D+00
      Xi(26) = 864.679841780689235786194757338308D+00
      Xi(27) = 1258.735317870830061504072716616065D+00
      Xi(28) = 1846.539033092342188324685992029117D+00
      Xi(29) = 2716.937133125949431988743754118332D+00
      Xi(30) = 3956.119117622416815338581841388077D+00
      Xi(31) = 5505.466470836707039815394182369346D+00
      Xi(32) = 6795.735316804062661422136670807959D+00
      GOTO 9999
C
C     ----- K = 16 , R = 8E3 -----
C
 1638 CONTINUE
C
      RIni = 8.0D+03
C
      Omega( 1) = 0.0002471688129707959323570824706692294903D+00
      Omega( 2) = 0.0006851572503816519147648152698003298777D+00
      Omega( 3) = 0.0014921087303356729843801153825161298627D+00
      Omega( 4) = 0.0031404544653767500959818303931520011574D+00
      Omega( 5) = 0.0064343087795187726742063023116502407106D+00
      Omega( 6) = 0.0127727320392762643041777051271246090636D+00
      Omega( 7) = 0.0245972718527957374620458025349378772262D+00
      Omega( 8) = 0.0460935944652694077526772567510171541016D+00
      Omega( 9) = 0.0843063411502943351981940491024491990402D+00
      Omega(10) = 0.1509066144125360964657698442659139459465D+00
      Omega(11) = 0.2650012689108123447917410991925279972747D+00
      Omega(12) = 0.4577473064340514511837056804699486178833D+00
      Omega(13) = 0.7808066064884537147545127966363764926427D+00
      Omega(14) = 1.3261526510468241279311341074276242579799D+00
      Omega(15) = 2.2943342451891662763831247762169596171589D+00
      Omega(16) = 4.4100922622450336430498241124809055691003D+00
C
      Alpha( 1) = 0.0000944773691119611902426450944308088109D+00
      Alpha( 2) = 0.0005431933718162297242877610526333442831D+00
      Alpha( 3) = 0.0015852907600157036040126309677997240577D+00
      Alpha( 4) = 0.0038044511901996254332009391894639516174D+00
      Alpha( 5) = 0.0084083088894682432245410836323118441982D+00
      Alpha( 6) = 0.0176766622119811988063381317029532091567D+00
      Alpha( 7) = 0.0357636322027086082402298240767724024636D+00
      Alpha( 8) = 0.0700642034135144223044259353949669844042D+00
      Alpha( 9) = 0.1334728750534660843214728245809119755450D+00
      Alpha(10) = 0.2480589770841020254066094583023094344298D+00
      Alpha(11) = 0.4509880134678298451529274148974479885510D+00
      Alpha(12) = 0.8040623195001615163262503882535270349763D+00
      Alpha(13) = 1.4095152073454303062174583494048363263573D+00
      Alpha(14) = 2.4387882648489325467747873554813509144878D+00
      Alpha(15) = 4.1990791711593300687514751512452448878321D+00
      Alpha(16) = 7.3805513886962790482890173215935192274628D+00
C
      Xi( 1) = 1.008165110648253801062246282072D+00
      Xi( 2) = 1.074384597037245452273192258819D+00
      Xi( 3) = 1.211652804915005441363864968629D+00
      Xi( 4) = 1.429875405649688178402398341937D+00
      Xi( 5) = 1.744778502878409746543707237176D+00
      Xi( 6) = 2.179413638528453895943509666111D+00
      Xi( 7) = 2.766533085561071399509011636830D+00
      Xi( 8) = 3.551889198475691804398821083133D+00
      Xi( 9) = 4.598729763476446650693335582716D+00
      Xi(10) = 5.994109407066016876903385979247D+00
      Xi(11) = 7.857885718551512398057784514194D+00
      Xi(12) = 10.355562038617222487928148932923D+00
      Xi(13) = 13.716691201702225162521731061105D+00
      Xi(14) = 18.261379363073626068761368124171D+00
      Xi(15) = 24.438738851383566025310400959825D+00
      Xi(16) = 32.883065798256496028301176792752D+00
      Xi(17) = 44.496644198750315540857869223146D+00
      Xi(18) = 60.572853238664348046044194617821D+00
      Xi(19) = 82.980927397131630282589576097507D+00
      Xi(20) = 114.446023333009322210662705998629D+00
      Xi(21) = 158.978262123559724389165737079566D+00
      Xi(22) = 222.536874297237396921889107659354D+00
      Xi(23) = 314.069373225468011751049246527145D+00
      Xi(24) = 447.151233986197667480411155338516D+00
      Xi(25) = 642.585980326371911719451901490174D+00
      Xi(26) = 932.495580759485205857295397890994D+00
      Xi(27) = 1366.515654488450761561146862277383D+00
      Xi(28) = 2019.946039647338810851806556456722D+00
      Xi(29) = 2998.725418032073378071800107136369D+00
      Xi(30) = 4413.413095771435769254509295933531D+00
      Xi(31) = 6217.794961798526415908128228693386D+00
      Xi(32) = 7753.764950412754636488443793496117D+00
      GOTO 9999
C
C     ----- K = 16 , R = 9E3 -----
C
 1639 CONTINUE
C
      RIni = 9.0D+03
C
      Omega( 1) = 0.0002243920605751591336424850472996972428D+00
      Omega( 2) = 0.0006270417067617411212975996796879091244D+00
      Omega( 3) = 0.0013834406617184846414476335802086481852D+00
      Omega( 4) = 0.0029481984836296284451914099904168420352D+00
      Omega( 5) = 0.0061006853878017267386307657429933559712D+00
      Omega( 6) = 0.0122089286671955113017460086040355538017D+00
      Omega( 7) = 0.0236739223810270169472055054940562435206D+00
      Omega( 8) = 0.0446301312678028930030701970643747245049D+00
      Omega( 9) = 0.0820634642574863223097821930684947488999D+00
      Omega(10) = 0.1475885080271769715372191800195622590763D+00
      Omega(11) = 0.2602781043968390923865237340795175668973D+00
      Omega(12) = 0.4513113627618848883095870744286415288116D+00
      Omega(13) = 0.7724689548824036639928375391761505852628D+00
      Omega(14) = 1.3159538709954992786576449814717193476099D+00
      Omega(15) = 2.2825517326481741162656768917571525889798D+00
      Omega(16) = 4.3966905735612235627332577969639260118129D+00
C
      Alpha( 1) = 0.0000856966014984952755532602313326851623D+00
      Alpha( 2) = 0.0004947372328309368735756184785531952564D+00
      Alpha( 3) = 0.0014553149185678176834364362982575380467D+00
      Alpha( 4) = 0.0035277193978652813810531198258566010217D+00
      Alpha( 5) = 0.0078746243803636643179873623281384009687D+00
      Alpha( 6) = 0.0167031895938322041453679344877314427720D+00
      Alpha( 7) = 0.0340593424378246069270895646924346422679D+00
      Alpha( 8) = 0.0671839347162209137161951316707941828099D+00
      Alpha( 9) = 0.1287618816032252760863264368085268074537D+00
      Alpha(10) = 0.2405941184184795421716193328243171833947D+00
      Alpha(11) = 0.4395302550885714915464373003795017780249D+00
      Alpha(12) = 0.7870474597304489113502520836185283314990D+00
      Alpha(13) = 1.3851212211488514633971058653827412854298D+00
      Alpha(14) = 2.4051111576427746828119036415216669411166D+00
      Alpha(15) = 4.1543839778031023445351022083116276917281D+00
      Alpha(16) = 7.3233295014735437632874548441463957715314D+00
C
      Xi( 1) = 1.008259723879726138789231226767D+00
      Xi( 2) = 1.075258004261145553404760422112D+00
      Xi( 3) = 1.214201731916668916913017162162D+00
      Xi( 4) = 1.435241413958034460847910940284D+00
      Xi( 5) = 1.754505053750317001420855334004D+00
      Xi( 6) = 2.195665201139894474446218053387D+00
      Xi( 7) = 2.792414039715915654400399392721D+00
      Xi( 8) = 3.591902005587370458631821246165D+00
      Xi( 9) = 4.659435744122105359579788519397D+00
      Xi(10) = 6.085097732474222439243843885137D+00
      Xi(11) = 7.993205967130740433566893354822D+00
      Xi(12) = 10.555856697719727954609680864451D+00
      Xi(13) = 14.012398949955290462440760101970D+00
      Xi(14) = 18.697561378800773526642764466033D+00
      Xi(15) = 25.082410883299436812898797199978D+00
      Xi(16) = 33.834433326273254095950537845283D+00
      Xi(17) = 45.906429557701034872840839540231D+00
      Xi(18) = 62.669241316123736631332752367030D+00
      Xi(19) = 86.111937187298724968864682693948D+00
      Xi(20) = 119.146741352932677100728842134458D+00
      Xi(21) = 166.078778721825472852735572359961D+00
      Xi(22) = 233.337675973566279430393421989720D+00
      Xi(23) = 330.630228643207634497969138465123D+00
      Xi(24) = 472.773955560221967292511280334111D+00
      Xi(25) = 682.634311241447989671016216561839D+00
      Xi(26) = 995.810103863546314306720574904830D+00
      Xi(27) = 1467.882979342089860841369386434963D+00
      Xi(28) = 2184.401544057028329515190989695839D+00
      Xi(29) = 3268.607022735769193300114920930355D+00
      Xi(30) = 4856.732194515447875726721349565196D+00
      Xi(31) = 6918.766716127081207066140677852673D+00
      Xi(32) = 8709.547740942938796493422159983311D+00
      GOTO 9999
C
C     ----- K = 16 , R = 1E4 -----
C
 1641 CONTINUE
C
      RIni = 1.0D+04
C
      Omega( 1) = 0.0002058282962339966570492447403323450772D+00
      Omega( 2) = 0.0005795163034704684972347108991904546249D+00
      Omega( 3) = 0.0012939465354750554289245856494361408551D+00
      Omega( 4) = 0.0027884026266817487573487446437733860982D+00
      Omega( 5) = 0.0058211432932475670054928791521259290320D+00
      Omega( 6) = 0.0117334510546047818806762968839307870184D+00
      Omega( 7) = 0.0228910200924036787596406146774730139271D+00
      Omega( 8) = 0.0433834862866021760988372111406130571254D+00
      Omega( 9) = 0.0801450248378290806266642694322133166906D+00
      Omega(10) = 0.1447399726413308090276202064194954743925D+00
      Omega(11) = 0.2562100884259844294811581111304121805006D+00
      Omega(12) = 0.4457521692955299713254091464875727979233D+00
      Omega(13) = 0.7652495837273190427605361330076760850716D+00
      Omega(14) = 1.3071066983128932045181772036990253127442D+00
      Omega(15) = 2.2723201246218485212409465834326738331583D+00
      Omega(16) = 4.3850533669438823197381238649938950402429D+00
C
      Alpha( 1) = 0.0000785427260255526335381718684768102140D+00
      Alpha( 2) = 0.0004551924617865591627534382120762179724D+00
      Alpha( 3) = 0.0013488674597178575663637117702277345987D+00
      Alpha( 4) = 0.0032998006170175148886844929563971762221D+00
      Alpha( 5) = 0.0074321614793527533236597911597831611630D+00
      Alpha( 6) = 0.0158908890315543802752948788353792863859D+00
      Alpha( 7) = 0.0326288715086645033290298819478314484854D+00
      Alpha( 8) = 0.0647538075936886720816677495693447497160D+00
      Alpha( 9) = 0.1247687659706603474505303905117692409021D+00
      Alpha(10) = 0.2342408160070870709053942812505333392892D+00
      Alpha(11) = 0.4297430785088648099107402900109775600868D+00
      Alpha(12) = 0.7724665951175793187429181874126271623027D+00
      Alpha(13) = 1.3641581106829700108717604023489400333347D+00
      Alpha(14) = 2.3761018937333028314910082778865785257949D+00
      Alpha(15) = 4.1158110002476520194319986600817173894029D+00
      Alpha(16) = 7.2738777917555991910612822604065286213881D+00
C
      Xi( 1) = 1.008342889724298546520742103638D+00
      Xi( 2) = 1.076025969828900324021421497367D+00
      Xi( 3) = 1.216444223911958485550427466748D+00
      Xi( 4) = 1.439966112440204014421077371999D+00
      Xi( 5) = 1.763077677382567077742948902230D+00
      Xi( 6) = 2.210006274817696659672899839322D+00
      Xi( 7) = 2.815285079582492041780361624603D+00
      Xi( 8) = 3.627318434126628444785431226727D+00
      Xi( 9) = 4.713263842711685328631249225495D+00
      Xi(10) = 6.165933069442765533749867268298D+00
      Xi(11) = 8.113673080220766232681373075764D+00
      Xi(12) = 10.734560325839581347226459051569D+00
      Xi(13) = 14.276844759455947475243475786044D+00
      Xi(14) = 19.088598358000106149784391718072D+00
      Xi(15) = 25.660981461558588253082935359828D+00
      Xi(16) = 34.691962651683783212114287053396D+00
      Xi(17) = 47.180913850895391293410741084813D+00
      Xi(18) = 64.570411096551821504563317688508D+00
      Xi(19) = 88.960948595539069416737731899048D+00
      Xi(20) = 123.439503538840588771352546615390D+00
      Xi(21) = 172.588155100055261406688522640707D+00
      Xi(22) = 243.280490027515989531758933139827D+00
      Xi(23) = 345.944111231481443657909125022343D+00
      Xi(24) = 496.583338359364417052832507692983D+00
      Xi(25) = 720.047273570623065530593720495744D+00
      Xi(26) = 1055.305720940396589591081522030436D+00
      Xi(27) = 1563.760916773524789480376284700469D+00
      Xi(28) = 2341.113443378716730380872945715964D+00
      Xi(29) = 3528.049719157751417775159552547848D+00
      Xi(30) = 5287.568848380935123021373556184699D+00
      Xi(31) = 7609.323569634084918344996140149306D+00
      Xi(32) = 9663.214336891674749452363357704598D+00
      GOTO 9999
C
C     ----- K = 16 , R = 3E6 (Best : R = 2080000) -----
C
 1699 CONTINUE
C
      RIni = 3.0D+06
C
      Omega( 1) = 0.0000061003972863863402651743415894584292D+00
      Omega( 2) = 0.0000400806962962503114856435918063846946D+00
      Omega( 3) = 0.0001640757836358783800369896591054251489D+00
      Omega( 4) = 0.0005333789180131260076594652082184605524D+00
      Omega( 5) = 0.0015016015641512588418497568549386800285D+00
      Omega( 6) = 0.0038216606296993014147883241512712593213D+00
      Omega( 7) = 0.0090155651757342924576852727716613777886D+00
      Omega( 8) = 0.0200340931971503289075645743919085184359D+00
      Omega( 9) = 0.0424034320065246971366340524717264415244D+00
      Omega(10) = 0.0861757293982789379928571675548720776305D+00
      Omega(11) = 0.1691970111509645595203711956888703582536D+00
      Omega(12) = 0.3225876471954512571087281269655022697407D+00
      Omega(13) = 0.6004239149409639519677957253396272108148D+00
      Omega(14) = 1.1003842126562718536008853020824460600124D+00
      Omega(15) = 2.0300383954052895140417189789872054461739D+00
      Omega(16) = 4.1094986543741171998445071977812403929420D+00
C
      Alpha( 1) = 0.0000019082799452525089727123312353175870D+00
      Alpha( 2) = 0.0000209929942749567198929778781085970021D+00
      Alpha( 3) = 0.0001111065144474894622336291847425338020D+00
      Alpha( 4) = 0.0004288308553325246431604475684908409083D+00
      Alpha( 5) = 0.0013732813892174006607116380621926021721D+00
      Alpha( 6) = 0.0038744692237789735938431894164690749705D+00
      Alpha( 7) = 0.0099592966839161908000036156629608541380D+00
      Alpha( 8) = 0.0238195511381459912370105267559461381666D+00
      Alpha( 9) = 0.0537624744822230793628780806669986702673D+00
      Alpha(10) = 0.1156769011058406200660730878126969400910D+00
      Alpha(11) = 0.2390543762941218424247519919179083558447D+00
      Alpha(12) = 0.4772836329010108079079401782740532667049D+00
      Alpha(13) = 0.9252667046693414284173374861897087839679D+00
      Alpha(14) = 1.7511652086513542147475780419441093727073D+00
      Alpha(15) = 3.2656590749026312514666919861738847430388D+00
      Alpha(16) = 6.1657963486574848834508022488165579488850D+00
C
      Xi( 1) = 1.010619076865009477754139832761D+00
      Xi( 2) = 1.097130811991455252374719209207D+00
      Xi( 3) = 1.278543967005584003045551289013D+00
      Xi( 4) = 1.572208435851447816183554251346D+00
      Xi( 5) = 2.006295143517123847655414525804D+00
      Xi( 6) = 2.623643844654697781593535865774D+00
      Xi( 7) = 3.487852850749272319965180177981D+00
      Xi( 8) = 4.692003643084871500334998017934D+00
      Xi( 9) = 6.371504907418426590353516258958D+00
      Xi(10) = 8.723643303268770113692576195774D+00
      Xi(11) = 12.037534791178474512587937006458D+00
      Xi(12) = 16.740308053138351525312987888583D+00
      Xi(13) = 23.469074329074754535831570656512D+00
      Xi(14) = 33.184443208950699554654617529081D+00
      Xi(15) = 47.352164865604347267213913852402D+00
      Xi(16) = 68.238125166270907559495917382719D+00
      Xi(17) = 99.395846543504521516942951109286D+00
      Xi(18) = 146.486945151395616310230352041799D+00
      Xi(19) = 218.690811018864780301829142672432D+00
      Xi(20) = 331.184415657320374265504270283600D+00
      Xi(21) = 509.612454553907175014426655934585D+00
      Xi(22) = 798.406621704696891683727955069116D+00
      Xi(23) = 1276.735904045949239304924560656218D+00
      Xi(24) = 2090.453775548968897268764521868434D+00
      Xi(25) = 3518.779759495649122946758780017262D+00
      Xi(26) = 6121.688303599042539016039654597989D+00
      Xi(27) = 11089.508083923279960991692405514186D+00
      Xi(28) = 21134.795018521869595318207757372875D+00
      Xi(29) = 43096.003825583163386880869438755326D+00
      Xi(30) = 96599.234684056726585765773052116856D+00
      Xi(31) = 250784.099339365001043233860400505364D+00
      Xi(32) = 881597.133850978159500755282351747155D+00
      GOTO 9999
C
C     ===== K = 17 =====
C
 1700 CONTINUE
C
      GOTO (1712,1712,1712,1712,1713,1714,1715,1716,1717,1718,1719,1721,
     *      1722,1723,1724,1725,1726,1727,1728,1729,1731,1732,1733,1734,
     *      1735,1736,1737,1738,1739,1741,1799),InitR
C
C     ----- K = 17 , R = 2E1 -----
C
 1712 CONTINUE
      Omega( 1) = 0.0302397699486285961049871363198748852597D+00
      Omega( 2) = 0.0714111729693463297876233723815442289151D+00
      Omega( 3) = 0.1151983413308698069545316956807656794126D+00
      Omega( 4) = 0.1633463455699484639498933122303547804677D+00
      Omega( 5) = 0.2181445810861671799195816295879879476161D+00
      Omega( 6) = 0.2826292106846990995726309758850902653649D+00
      Omega( 7) = 0.3608685543158396038924441706630119597321D+00
      Omega( 8) = 0.4582722819283937388363473625085831031356D+00
      Omega( 9) = 0.5819243715708696990151008177072355920245D+00
      Omega(10) = 0.7410321325985205146835016476458690704021D+00
      Omega(11) = 0.9477492570824826021358075478495663901413D+00
      Omega(12) = 1.2188845101312763704280717202976802582270D+00
      Omega(13) = 1.5795805617706500316010237439101615564141D+00
      Omega(14) = 2.0718714460365349940701440445423031633254D+00
      Omega(15) = 2.7779101419370363975497317809981723257806D+00
      Omega(16) = 3.9019593558207467727961398340497112258163D+00
      Omega(17) = 6.2568211133691469119991190872553943336243D+00
C
      Alpha( 1) = 0.0117636499669930626207187650267282919003D+00
      Alpha( 2) = 0.0624282057373425173215709168411668983367D+00
      Alpha( 3) = 0.1554499074025039259292976789872398057923D+00
      Alpha( 4) = 0.2942728713790734211899757788444276229711D+00
      Alpha( 5) = 0.4843511156615427304611767866937555027107D+00
      Alpha( 6) = 0.7337793906644703758050847508531688845324D+00
      Alpha( 7) = 1.0541795508192826758128365916000745983183D+00
      Alpha( 8) = 1.4618861427706882925641948545170123452408D+00
      Alpha( 9) = 1.9794504252494227664656983867175199520716D+00
      Alpha(10) = 2.6375217858538901655098690701706232175638D+00
      Alpha(11) = 3.4773350777344393465646821272940769631532D+00
      Alpha(12) = 4.5544058764435397814417105788464823490358D+00
      Alpha(13) = 5.9447852881084881178017087499654280691175D+00
      Alpha(14) = 7.7570395281938202330582199373054663738003D+00
      Alpha(15) = 10.1586829246453062003396894397155847400427D+00
      Alpha(16) = 13.4479116052396606220686869903602200793102D+00
      Alpha(17) = 18.3386844672637758790756379312369972467422D+00
C
      Xi( 1) = 1.001944992041143801786949596444D+00
      Xi( 2) = 1.017573053775527585772427141553D+00
      Xi( 3) = 1.049054516638138171043259649906D+00
      Xi( 4) = 1.096933820715218840607073358395D+00
      Xi( 5) = 1.161946276668325532619580053861D+00
      Xi( 6) = 1.245136996763966193602253540451D+00
      Xi( 7) = 1.347820890830599597641666076431D+00
      Xi( 8) = 1.471549417805818207633135330514D+00
      Xi( 9) = 1.618489190546147302065919937331D+00
      Xi(10) = 1.790745220583274777289249646284D+00
      Xi(11) = 1.991282425182284036521175585310D+00
      Xi(12) = 2.223219853512135877953923013273D+00
      Xi(13) = 2.490140598951331849485180125470D+00
      Xi(14) = 2.796239012409017054942850233346D+00
      Xi(15) = 3.146287033789434546071986376070D+00
      Xi(16) = 3.545673539110263757087246339061D+00
      Xi(17) = 4.000046161168884129629569335052D+00
      Xi(18) = 4.515993532901835928825701182987D+00
      Xi(19) = 5.100193050568134361683753086325D+00
      Xi(20) = 5.759757577923325344917970802783D+00
      Xi(21) = 6.501659436276989377859192309472D+00
      Xi(22) = 7.332452451966076140913580738712D+00
      Xi(23) = 8.256903948714259114556590635203D+00
      Xi(24) = 9.277899381065644061374053652713D+00
      Xi(25) = 10.393855072285290004792390750055D+00
      Xi(26) = 11.597510360067749908899448207933D+00
      Xi(27) = 12.872698832376013395629998914416D+00
      Xi(28) = 14.192600960322230821755717933286D+00
      Xi(29) = 15.516222048844071365082675217906D+00
      Xi(30) = 16.789073787817390622650881581990D+00
      Xi(31) = 17.942660053233430389116320746723D+00
      Xi(32) = 18.901622259349553395457732563045D+00
      Xi(33) = 19.591956786301745608094848627445D+00
      Xi(34) = 19.953925505681302989685610427273D+00
      GOTO 9999
C
C    ----- K = 17 , R = 3E1 -----
C
 1713 CONTINUE
C
      RIni = 3.0D+01
C
      Omega( 1) = 0.0221689954347680469674537837715400101501D+00
      Omega( 2) = 0.0525136316982886899500462558193492057512D+00
      Omega( 3) = 0.0852024918822239950270025672651730275220D+00
      Omega( 4) = 0.1218775877465483974658195381879366436806D+00
      Omega( 5) = 0.1647575454691382380742687221997222479786D+00
      Omega( 6) = 0.2168726754863256642955884542778299817201D+00
      Omega( 7) = 0.2823530993770712014016236596214426413098D+00
      Omega( 8) = 0.3667057338149049161731257012730722522065D+00
      Omega( 9) = 0.4770978458889288356561779458342442694629D+00
      Omega(10) = 0.6227933403263984189356279597138410508705D+00
      Omega(11) = 0.8160260402127402867920175077998123924772D+00
      Omega(12) = 1.0737693823036798862393531583947492435982D+00
      Omega(13) = 1.4213969799562429773562635682004895443242D+00
      Omega(14) = 1.9010620007560267858684119324230721304048D+00
      Omega(15) = 2.5944356141359594832267870812003707214899D+00
      Omega(16) = 3.7031544721692649809428876661598906139261D+00
      Omega(17) = 6.0263079220428109991809784151683970776503D+00
C
      Alpha( 1) = 0.0086209575744671239913921589881340423744D+00
      Alpha( 2) = 0.0458206720026962791509413542440087852015D+00
      Alpha( 3) = 0.1144226474825498924414854048925427321137D+00
      Alpha( 4) = 0.2175483194524849357882889666626624602941D+00
      Alpha( 5) = 0.3602371351111766857621462184502192371838D+00
      Alpha( 6) = 0.5501295551911949390714966223736581696357D+00
      Alpha( 7) = 0.7984222655089383742250258579886690313288D+00
      Alpha( 8) = 1.1211082157416934496768634588192981027532D+00
      Alpha( 9) = 1.5404908164489909015368307443338835582836D+00
      Alpha(10) = 2.0870398051392735324152671871900111000286D+00
      Alpha(11) = 2.8018727807909891906572308339917753983173D+00
      Alpha(12) = 3.7405073580248508196156170813040375833225D+00
      Alpha(13) = 4.9792014542644435345572495599242301977938D+00
      Alpha(14) = 6.6269271013874555863523707621709490922512D+00
      Alpha(15) = 8.8514862857485870194923505138717700901907D+00
      Alpha(16) = 11.9500387904471270811984640047853645228315D+00
      Alpha(17) = 16.6290535199854737982705366050595330307260D+00
C
      Xi( 1) = 1.002291568893694707334182658620D+00
      Xi( 2) = 1.020685218992778364949583491672D+00
      Xi( 3) = 1.057827576467378746490413665704D+00
      Xi( 4) = 1.114413507840623477089178616772D+00
      Xi( 5) = 1.191528289563124709841621107209D+00
      Xi( 6) = 1.290624288731503517633353306238D+00
      Xi( 7) = 1.413584180873620381587926764233D+00
      Xi( 8) = 1.562739803779250570992670477111D+00
      Xi( 9) = 1.740949844069554654397979920333D+00
      Xi(10) = 1.951622386802114737921684528565D+00
      Xi(11) = 2.198829269235813942259469278007D+00
      Xi(12) = 2.487340809060435524880522395463D+00
      Xi(13) = 2.822746253318904027405575329901D+00
      Xi(14) = 3.211534714095589648817988259744D+00
      Xi(15) = 3.661191678398682642367814588091D+00
      Xi(16) = 4.180297123045072366521318629751D+00
      Xi(17) = 4.778612938301576774490636578818D+00
      Xi(18) = 5.467118211613049514989437493995D+00
      Xi(19) = 6.258040356889422471675654691126D+00
      Xi(20) = 7.164671300680670553828049973433D+00
      Xi(21) = 8.201177916486034297043061780386D+00
      Xi(22) = 9.381898130731370829930859667201D+00
      Xi(23) = 10.720507977526170022460061392877D+00
      Xi(24) = 12.228219924802319389968430218829D+00
      Xi(25) = 13.911459749000607544026919537572D+00
      Xi(26) = 15.768042709301844005176373997301D+00
      Xi(27) = 17.782453404170050157914739763498D+00
      Xi(28) = 19.919486830047647271321742401540D+00
      Xi(29) = 22.117910203168493023240248618322D+00
      Xi(30) = 24.285100625197177140815463225465D+00
      Xi(31) = 26.296600456185529319508287215967D+00
      Xi(32) = 28.003802500104567398467314554011D+00
      Xi(33) = 29.253240477906523528406346201791D+00
      Xi(34) = 29.915367900093881785988925159359D+00
      GOTO 9999
C
C     ----- K = 17 , R = 4E1 -----
C
 1714 CONTINUE
C
      RIni = 4.0D+01
C
      Omega( 1) = 0.0177211717809415518640002830744184514344D+00
      Omega( 2) = 0.0420804499893616642852874065727775132473D+00
      Omega( 3) = 0.0685892521116945304979141580237200770398D+00
      Omega( 4) = 0.0988069670354228814801726861472008778264D+00
      Omega( 5) = 0.1348877148729152561400842363914520660728D+00
      Omega( 6) = 0.1798302894659236093774208956053151098331D+00
      Omega( 7) = 0.2377570966773244348510746651315095334667D+00
      Omega( 8) = 0.3141621682693044888387470109369914439412D+00
      Omega( 9) = 0.4161787636643034244527360188126152706900D+00
      Omega(10) = 0.5530505032238192121801616252252387084809D+00
      Omega(11) = 0.7370736963599462467392961495527003989991D+00
      Omega(12) = 0.9854093744573996703451854162203460418823D+00
      Omega(13) = 1.3237121967377334781591421974944466910529D+00
      Omega(14) = 1.7943741588876145828801433879107207758352D+00
      Omega(15) = 2.4789350252740321232762521486137075044098D+00
      Omega(16) = 3.5775455854296866070336086140812881239981D+00
      Omega(17) = 5.8808761560584260018845481621241333414218D+00
C
      Alpha( 1) = 0.0068893777628398834758633286282081265028D+00
      Alpha( 2) = 0.0366619080357459382424421129309521916184D+00
      Alpha( 3) = 0.0917603715074955805243960005324055373421D+00
      Alpha( 4) = 0.1750703781562064611029708519063241567437D+00
      Alpha( 5) = 0.2913192809176047665088108568065550230131D+00
      Alpha( 6) = 0.4477872044599345537009733864142191350766D+00
      Alpha( 7) = 0.6552926902022263624925622893524490564232D+00
      Alpha( 8) = 0.9294428610717238099423430364165454875547D+00
      Alpha( 9) = 1.2921313629933525301152902331480731845659D+00
      Alpha(10) = 1.7733859123770382129527817260949973388051D+00
      Alpha(11) = 2.4138975752788709510602421781655380073062D+00
      Alpha(12) = 3.2688815574094914609068474042885554808890D+00
      Alpha(13) = 4.4145341217529551320307434902900922679692D+00
      Alpha(14) = 5.9600424918313773198566551947408242995152D+00
      Alpha(15) = 8.0735188590298359763430813273998865042813D+00
      Alpha(16) = 11.0517695651505124913860367286133623565547D+00
      Alpha(17) = 15.5970252200462439961733962512369089381536D+00
C
      Xi( 1) = 1.002545391012707700273062783047D+00
      Xi( 2) = 1.022989700653836702077051357129D+00
      Xi( 3) = 1.064312733992968879720951436152D+00
      Xi( 4) = 1.127389942221722018284024713353D+00
      Xi( 5) = 1.213557477075983313937954100048D+00
      Xi( 6) = 1.324639555837989256551964767361D+00
      Xi( 7) = 1.462990572970611438197510723036D+00
      Xi( 8) = 1.631561119558584901771160213801D+00
      Xi( 9) = 1.833961456632768107833687809549D+00
      Xi(10) = 2.074562680292959938746846781932D+00
      Xi(11) = 2.358595575819610036267387487285D+00
      Xi(12) = 2.692272446945528910008937906184D+00
      Xi(13) = 3.082923335884153877208685434752D+00
      Xi(14) = 3.539158400714016342253689928476D+00
      Xi(15) = 4.071029753543150548570489499767D+00
      Xi(16) = 4.690223124743389164839624250192D+00
      Xi(17) = 5.410239559896043670846055295698D+00
      Xi(18) = 6.246568023375087223749563269948D+00
      Xi(19) = 7.216801269083062290998764609640D+00
      Xi(20) = 8.340650732776270932442019834241D+00
      Xi(21) = 9.639766686236018726485763608025D+00
      Xi(22) = 11.137225750524117250973843429307D+00
      Xi(23) = 12.856445313725064712000178968054D+00
      Xi(24) = 14.819256411371155610079419773228D+00
      Xi(25) = 17.042638573235718358819812401350D+00
      Xi(26) = 19.533631219514500466927220401203D+00
      Xi(27) = 22.281835512529508855003590461763D+00
      Xi(28) = 25.249330361463563417168121105760D+00
      Xi(29) = 28.358601719426040955324141989990D+00
      Xi(30) = 31.481047768266854563340428896367D+00
      Xi(31) = 34.431395403085794500175342847115D+00
      Xi(32) = 36.976004843928678710723856326581D+00
      Xi(33) = 38.862502856911729828826018717791D+00
      Xi(34) = 39.870707818741036188719961685933D+00
      GOTO 9999
C
C     ----- K = 17 , R = 5E1 -----
C
 1715 CONTINUE
C
      RIni = 5.0D+01
C
      Omega( 1) = 0.0148686660388008807532681768322213589073D+00
      Omega( 2) = 0.0353795804560981431076025719578703743196D+00
      Omega( 3) = 0.0578913354783317874690097646489528315783D+00
      Omega( 4) = 0.0838960166520241401523667879036594285935D+00
      Omega( 5) = 0.1154909243595618800322876416897255325011D+00
      Omega( 6) = 0.1556347058668119319956545110184009672594D+00
      Omega( 7) = 0.2084132074967702798830348001613010922028D+00
      Omega( 8) = 0.2792663830727312734858187176500976534044D+00
      Omega( 9) = 0.3752605918790274696684745175945252526617D+00
      Omega(10) = 0.5056075720898941724132727437712730989006D+00
      Omega(11) = 0.6826650143959483072971829165087598312311D+00
      Omega(12) = 0.9237725888705498888692907444308843878389D+00
      Omega(13) = 1.2548501525981011695464947819367296233395D+00
      Omega(14) = 1.7185396923469535971663868512138151345425D+00
      Omega(15) = 2.3963756361848144599020066225492087141902D+00
      Omega(16) = 3.4875378011820311698922003618150711190538D+00
      Omega(17) = 5.7767967602680691920416511386804359062808D+00
C
      Alpha( 1) = 0.0057790612680969588938700127415254437224D+00
      Alpha( 2) = 0.0307848490694564320059887834943324058656D+00
      Alpha( 3) = 0.0771991247235623547496272990608989772454D+00
      Alpha( 4) = 0.1477253347274733382305531362299610975697D+00
      Alpha( 5) = 0.2468449877068841172305822470600666918017D+00
      Alpha( 6) = 0.3815442037677632649477019871619987156919D+00
      Alpha( 7) = 0.5623109577640680386439488158689670171952D+00
      Alpha( 8) = 0.8043777658932390478303126046810689331323D+00
      Alpha( 9) = 1.1292008564500714781243806417165842503891D+00
      Alpha(10) = 1.5663159266865059626207268284048268469633D+00
      Alpha(11) = 2.1559328328586830575506333440571893333981D+00
      Alpha(12) = 2.9529057092101003143859438981877474361681D+00
      Alpha(13) = 4.0332967734518776884745927535647069817060D+00
      Alpha(14) = 5.5064292403364440953682235058153082718491D+00
      Alpha(15) = 7.5407037530380250545287978969355435765465D+00
      Alpha(16) = 10.4328308991523239354290431002425521000987D+00
      Alpha(17) = 14.8822667559735207229790199257024596590782D+00
C
      Xi( 1) = 1.002747516529837833957478077096D+00
      Xi( 2) = 1.024822470846904157604911889123D+00
      Xi( 3) = 1.069480378717986563189001891772D+00
      Xi( 4) = 1.137747078914491229606649469464D+00
      Xi( 5) = 1.231188066139299916012389557451D+00
      Xi( 6) = 1.351946282217304978420270578088D+00
      Xi( 7) = 1.502798888520101475369319943276D+00
      Xi( 8) = 1.687235795879166483216861360805D+00
      Xi( 9) = 1.909553499731967626589987374608D+00
      Xi(10) = 2.174978286234729048720035193831D+00
      Xi(11) = 2.489802479293736500644967923890D+00
      Xi(12) = 2.861547482390371150220986784518D+00
      Xi(13) = 3.299155511449299699481760295150D+00
      Xi(14) = 3.813203198815979375144863405822D+00
      Xi(15) = 4.416157890949495515124645361738D+00
      Xi(16) = 5.122646892908169101530507782982D+00
      Xi(17) = 5.949761203893470298213058677916D+00
      Xi(18) = 6.917353421690117099788347454492D+00
      Xi(19) = 8.048309539958687430956374431190D+00
      Xi(20) = 9.368725948256321439319704857951D+00
      Xi(21) = 10.907896603463473500852942699169D+00
      Xi(22) = 12.697927722935548186175558349476D+00
      Xi(23) = 14.772704548227969807705495242089D+00
      Xi(24) = 17.165788871667563860942684605959D+00
      Xi(25) = 19.906596463527166234255827426125D+00
      Xi(26) = 23.014021556751030934614976608543D+00
      Xi(27) = 26.486506433188673752254782556292D+00
      Xi(28) = 30.287761291592585566334605573502D+00
      Xi(29) = 34.328413013627400472010942067413D+00
      Xi(30) = 38.446466616132194189792148364404D+00
      Xi(31) = 42.394153189670192732191322804169D+00
      Xi(32) = 45.844150908129016632613916826244D+00
      Xi(33) = 48.429519596735885651056507938961D+00
      Xi(34) = 49.821049698729165840227484096658D+00
      GOTO 9999
C
C     ----- K = 17 , R = 6E1 -----
C
 1716 CONTINUE
C
      RIni = 6.0D+01
C
      Omega( 1) = 0.0128688641747443431470488014144171184938D+00
      Omega( 2) = 0.0306759431049708882359357839733626782674D+00
      Omega( 3) = 0.0503652577739043404976375644527264441308D+00
      Omega( 4) = 0.0733727538326730014271097150835299061100D+00
      Omega( 5) = 0.1017455651731470538470542691600329021639D+00
      Omega( 6) = 0.1383996020235092485434983539116515771639D+00
      Omega( 7) = 0.1873716447221103687041350929765037847119D+00
      Omega( 8) = 0.2540325726980336789554305171323633771863D+00
      Omega( 9) = 0.3453754544025244804892216948788430386230D+00
      Omega(10) = 0.4705821147337007801468838213976297879526D+00
      Omega(11) = 0.6420685909054469359276938467129269838551D+00
      Omega(12) = 0.8773350492032141751536089557816922024358D+00
      Omega(13) = 1.2025388662231339082864592371358014588623D+00
      Omega(14) = 1.6605588690218929312633275308108693479880D+00
      Omega(15) = 2.3329791005813414442489339695185890377616D+00
      Omega(16) = 3.4182913257221446416862786765733517313492D+00
      Omega(17) = 5.6968099048043257253033655462814976999653D+00
C
      Alpha( 1) = 0.0050007674880163409831774297322620759232D+00
      Alpha( 2) = 0.0266626502692728805931916390842273401063D+00
      Alpha( 3) = 0.0669742870912714411999280307574711201823D+00
      Alpha( 4) = 0.1284926354669767219635620816853283088221D+00
      Alpha( 5) = 0.2154985496957080209131252940268907991594D+00
      Alpha( 6) = 0.3347316344380094074533425752004944797591D+00
      Alpha( 7) = 0.4963890092080094549025631733840668857738D+00
      Alpha( 8) = 0.7153535743415229191874415981544643727830D+00
      Alpha( 9) = 1.0126577716550769446817586993958570928953D+00
      Alpha(10) = 1.4173574351365213142483687192019203848758D+00
      Alpha(11) = 1.9691940995077596654678001031335554671386D+00
      Alpha(12) = 2.7226634022245192986218936814779567612277D+00
      Alpha(13) = 3.7536719703944716284172572473565310247068D+00
      Alpha(14) = 5.1716341738480789531028081196240009376197D+00
      Alpha(15) = 7.1452085392034963683832726744071806024294D+00
      Alpha(16) = 9.9711244754619540453521864264985197223723D+00
      Alpha(17) = 14.3468619350247799807199222676956651412183D+00
C
      Xi( 1) = 1.002915499109157203240158540147D+00
      Xi( 2) = 1.026346630989126865705907132309D+00
      Xi( 3) = 1.073782732619288837945005499197D+00
      Xi( 4) = 1.146383008420414932799526186447D+00
      Xi( 5) = 1.245918789443741162893154739155D+00
      Xi( 6) = 1.374819018650134055969415369791D+00
      Xi( 7) = 1.536240708054906436551972914728D+00
      Xi( 8) = 1.734161982673499102910671942901D+00
      Xi( 9) = 1.973503827098548688815000951013D+00
      Xi(10) = 2.260274464933970315228678460429D+00
      Xi(11) = 2.601744180092282813371307170769D+00
      Xi(12) = 3.006647453358342283145920870169D+00
      Xi(13) = 3.485425441718501938766563985617D+00
      Xi(14) = 4.050507024804020406432564227472D+00
      Xi(15) = 4.716636777895760899905613916516D+00
      Xi(16) = 5.501248259550373111061855047055D+00
      Xi(17) = 6.424879686084404700145006339440D+00
      Xi(18) = 7.511615350987158668590731869230D+00
      Xi(19) = 8.789522694599506495369278802343D+00
      Xi(20) = 10.291020111543749447104156846677D+00
      Xi(21) = 12.053065680014858375435071735637D+00
      Xi(22) = 14.116964761606295349419259999024D+00
      Xi(23) = 16.527467776378566692921379122083D+00
      Xi(24) = 19.330626819368343607963289798590D+00
      Xi(25) = 22.569577309641494939143613329691D+00
      Xi(26) = 26.277070914326466520224978573594D+00
      Xi(27) = 30.463249836486540004371059708888D+00
      Xi(28) = 35.097156787527594493797389318956D+00
      Xi(29) = 40.081587760813652453195654601359D+00
      Xi(30) = 45.224179044115468292103221514822D+00
      Xi(31) = 50.214331710936857079696782335532D+00
      Xi(32) = 54.624695735134070985472565240570D+00
      Xi(33) = 57.960537309272782375418886502416D+00
      Xi(34) = 59.767106150282154120034761035640D+00
      GOTO 9999
C
C     ----- K = 17 , R = 7E1 -----
C
 1717 CONTINUE
C
      RIni = 7.0D+01
C
      Omega( 1) = 0.0113817599792177474883739024189011068700D+00
      Omega( 2) = 0.0271743593440448312882964931880280801124D+00
      Omega( 3) = 0.0447515467109071490308292058657446688130D+00
      Omega( 4) = 0.0655014490394985475404379324237957860078D+00
      Omega( 5) = 0.0914265035459931564443681988041401353939D+00
      Omega( 6) = 0.1253997678698674680237111489500279049025D+00
      Omega( 7) = 0.1714039191196946684581396344593251512833D+00
      Omega( 8) = 0.2347367696287157425250264122795762489204D+00
      Omega( 9) = 0.3223205037789023950413076075038887324808D+00
      Omega(10) = 0.4433112013591309012565128827576543812938D+00
      Omega(11) = 0.6101784217262353503577997082540917972437D+00
      Omega(12) = 0.8405640412966286443554805862810042071942D+00
      Omega(13) = 1.1608360910097696578864925887053516362357D+00
      Omega(14) = 1.6140922256599763499581093317836177902791D+00
      Omega(15) = 2.2819924145495760149562197716122113888559D+00
      Omega(16) = 3.3625148353540114170776020863229405222228D+00
      Omega(17) = 5.6324406740100084454471451866908182637417D+00
C
      Alpha( 1) = 0.0044220826490264052288574705452950386331D+00
      Alpha( 2) = 0.0235959842136256114419402048903995350315D+00
      Alpha( 3) = 0.0593600889932688150471323720175353955142D+00
      Alpha( 4) = 0.1141499987320069049107963167488666300642D+00
      Alpha( 5) = 0.1920783153601949629779317013644757139446D+00
      Alpha( 6) = 0.2996733309241805377580707830942330360813D+00
      Alpha( 7) = 0.4468736581015504913947430226661694518953D+00
      Alpha( 8) = 0.6482397686822209835390967702650755200011D+00
      Alpha( 9) = 0.9244066057150568037304935731235389084759D+00
      Alpha(10) = 1.3039805390235891479049773478138263271831D+00
      Alpha(11) = 1.8262656613661149404472666679666303934937D+00
      Alpha(12) = 2.5454181581362931795740889295842634965084D+00
      Alpha(13) = 3.5371843477475413788143482340942114205973D+00
      Alpha(14) = 4.9110406570491912229582676108208261211985D+00
      Alpha(15) = 6.8358741981674408794145136880615609697998D+00
      Alpha(16) = 9.6084889100046543333744075710001197876409D+00
      Alpha(17) = 13.9248776999019524772821521096943797601853D+00
C
      Xi( 1) = 1.003059306218734375798842806127D+00
      Xi( 2) = 1.027652007831690134770884215154D+00
      Xi( 3) = 1.077470832132708706109849861043D+00
      Xi( 4) = 1.153795992652397408304368320309D+00
      Xi( 5) = 1.258585453713018532614399680458D+00
      Xi( 6) = 1.394528608539600317045743793631D+00
      Xi( 7) = 1.565128237977414441656730248464D+00
      Xi( 8) = 1.774811888247491756297538612053D+00
      Xi( 9) = 2.029073788523009899860405358396D+00
      Xi(10) = 2.334647524299378020017428636557D+00
      Xi(11) = 2.699711470502097960741383353422D+00
      Xi(12) = 3.134134981167966560499163475306D+00
      Xi(13) = 3.649768331014000715285111597908D+00
      Xi(14) = 4.260789544679054438418297490188D+00
      Xi(15) = 4.984111036106225256676893753394D+00
      Xi(16) = 5.839850856509280158622432477245D+00
      Xi(17) = 6.851869206681649118566290734833D+00
      Xi(18) = 8.048356757322034505178209862208D+00
      Xi(19) = 9.462448801068054722064604167286D+00
      Xi(20) = 11.132800568353531299632674933520D+00
      Xi(21) = 13.104007107330691340313366666237D+00
      Xi(22) = 15.426648606116144035911574849962D+00
      Xi(23) = 18.156587826522455474120842922048D+00
      Xi(24) = 21.352892433095730553849200816785D+00
      Xi(25) = 25.073360231710072847446935107030D+00
      Xi(26) = 29.366141741389968937078758237647D+00
      Xi(27) = 34.255394316438924642853613988791D+00
      Xi(28) = 39.718638441445942799390422806027D+00
      Xi(29) = 45.654496890002611110548347284066D+00
      Xi(30) = 51.843356484674942806500919090240D+00
      Xi(31) = 57.912297647083303200887804962349D+00
      Xi(32) = 63.329143135346385548589687175536D+00
      Xi(33) = 67.459959939941881126856948469594D+00
      Xi(34) = 69.709379552572190680281849495259D+00
      GOTO 9999
C
C     ----- K = 17 , R = 8E1 -----
C
 1718 CONTINUE
      Omega( 1) = 0.0102284904090335036154122382023867743328D+00
      Omega( 2) = 0.0244561615921032835244597249818587769710D+00
      Omega( 3) = 0.0403860818642373749303667741089585163650D+00
      Omega( 4) = 0.0593649342842641226417022774081155844783D+00
      Omega( 5) = 0.0833549435940638370153336032808422118023D+00
      Omega( 6) = 0.1151872860033455609065451349937170277826D+00
      Omega( 7) = 0.1587891810315065413005027050719064618534D+00
      Omega( 8) = 0.2193862206997807445173166371321649137371D+00
      Omega( 9) = 0.3038348393048518613376949432103302228825D+00
      Omega(10) = 0.4212691673947276489384239361868367268471D+00
      Omega(11) = 0.5842068690607538775159783550616765523955D+00
      Omega(12) = 0.8104146081742157019375469295674463410251D+00
      Omega(13) = 1.1264476636364262366555438754645024346246D+00
      Omega(14) = 1.5756047826778999358795715912684443082981D+00
      Omega(15) = 2.2396347826394381254518084478632999889669D+00
      Omega(16) = 3.3161182846423878898840104101353176702105D+00
      Omega(17) = 5.5789379346348069691093052302477417470072D+00
C
      Alpha( 1) = 0.0039733564241690171219145038573883343958D+00
      Alpha( 2) = 0.0212168432499748764094165158142213378767D+00
      Alpha( 3) = 0.0534476838381714172498222412133195291517D+00
      Alpha( 4) = 0.1029986320484459273082384136999145596292D+00
      Alpha( 5) = 0.1738382188340671212229554543071685657196D+00
      Alpha( 6) = 0.2723103794523001048417075453667024476090D+00
      Alpha( 7) = 0.4081217168328174226845077993131027938034D+00
      Alpha( 8) = 0.5955361059459275052869499378349615881234D+00
      Alpha( 9) = 0.8548199579708161325143980779905916733696D+00
      Alpha(10) = 1.2141643235424078605712294720575528117479D+00
      Alpha(11) = 1.7124696860301575545649024889804934446147D+00
      Alpha(12) = 2.4035753152187838668947528208974517838215D+00
      Alpha(13) = 3.3630688929233138084913534715525429419358D+00
      Alpha(14) = 4.7004686318565189542817428502985421800986D+00
      Alpha(15) = 6.5848644957529141393155758255062437456218D+00
      Alpha(16) = 9.3131630039612661184703079619850996095920D+00
      Alpha(17) = 13.5801936423355342027294301310291757545201D+00
C
      Xi( 1) = 1.003185049342236287504663561077D+00
      Xi( 2) = 1.028793990287334054328782784182D+00
      Xi( 3) = 1.080699748461550579949237704813D+00
      Xi( 4) = 1.160293481503308493326710737481D+00
      Xi( 5) = 1.269704561755699912472226953408D+00
      Xi( 6) = 1.411861685687981412011191006961D+00
      Xi( 7) = 1.590586957135407310866964669671D+00
      Xi( 8) = 1.810723906681402168216896142461D+00
      Xi( 9) = 2.078299602443752740980192905695D+00
      Xi(10) = 2.400724593438739941321166670996D+00
      Xi(11) = 2.787029119941703737437996291071D+00
      Xi(12) = 3.248150538939510043046077214512D+00
      Xi(13) = 3.797272462402132554371361750079D+00
      Xi(14) = 4.450236221724152312160388555995D+00
      Xi(15) = 5.226024730023481232668580709344D+00
      Xi(16) = 6.147335538281357032123347616803D+00
      Xi(17) = 7.241237963009023085800275332247D+00
      Xi(18) = 8.539913938899198962186043448952D+00
      Xi(19) = 10.081456310334636464025315216730D+00
      Xi(20) = 11.910664040008737896841894965405D+00
      Xi(21) = 14.079717653115598969063015921677D+00
      Xi(22) = 16.648495244969107535737418146482D+00
      Xi(23) = 19.684131952622105739963975779006D+00
      Xi(24) = 23.259078536592308302996534585816D+00
      Xi(25) = 27.446514278793275869172685332842D+00
      Xi(26) = 32.311219585524162591777530550985D+00
      Xi(27) = 37.893276819644083831112313376366D+00
      Xi(28) = 44.181388382686948908262269952729D+00
      Xi(29) = 51.073333552978532983573911607778D+00
      Xi(30) = 58.325346325847006980630515826647D+00
      Xi(31) = 65.503209781983478085520822276067D+00
      Xi(32) = 71.966253063518017804578441598551D+00
      Xi(33) = 76.931402511047548453682054514502D+00
      Xi(34) = 79.647416004628414316901530156656D+00
      GOTO 9999
C
C     ----- K = 17 , R = 9E1 -----
C
 1719 CONTINUE
C
      RIni = 9.0D+01
C
      Omega( 1) = 0.0093055452799273397626321252702752406805D+00
      Omega( 2) = 0.0222788871640996847180458744971942586233D+00
      Omega( 3) = 0.0368837610336946389418195154513702149757D+00
      Omega( 4) = 0.0544304246766022368402351827576524101460D+00
      Omega( 5) = 0.0768446068104137880862107568002983271072D+00
      Omega( 6) = 0.1069170503299141731774844071845009807475D+00
      Omega( 7) = 0.1485200449520121921163440595337057459346D+00
      Omega( 8) = 0.2068099190615291706755096354242162703940D+00
      Omega( 9) = 0.2885830163742097357704001481248923255407D+00
      Omega(10) = 0.4029545064879059731695679080765160051669D+00
      Omega(11) = 0.5624848391441751290571911259252857462343D+00
      Omega(12) = 0.7850508763672496834792186026774629681313D+00
      Omega(13) = 1.0973753054331811402624316520437730559934D+00
      Omega(14) = 1.5429418443905634444977131858323105007003D+00
      Omega(15) = 2.2035942157384123282846261071554749833012D+00
      Omega(16) = 3.2765969845607466565205051489684251464496D+00
      Omega(17) = 5.5333942237095084396361621692506105318898D+00
C
      Alpha( 1) = 0.0036142847601028185659230223744642351846D+00
      Alpha( 2) = 0.0193121979363516552036001035219658916731D+00
      Alpha( 3) = 0.0487106256560613163639824208656081694357D+00
      Alpha( 4) = 0.0940536129957006666649200164742605778656D+00
      Alpha( 5) = 0.1591842932859885571041666865732544522416D+00
      Alpha( 6) = 0.2502835048715152584130651247651400126415D+00
      Alpha( 7) = 0.3768478515919123995111129216128986740841D+00
      Alpha( 8) = 0.5528681138914166540328232191114921079134D+00
      Alpha( 9) = 0.7982702270194615650324501676671218319825D+00
      Alpha(10) = 1.1408628740479669352262531711872384221351D+00
      Alpha(11) = 1.6191750225339964947305834708224381301989D+00
      Alpha(12) = 2.2867510241830227718257823044112342358858D+00
      Alpha(13) = 3.2190241844884033011083002140395592505229D+00
      Alpha(14) = 4.5255397708238691554562260765948167318129D+00
      Alpha(15) = 6.3755685758961003579622928150172356254188D+00
      Alpha(16) = 9.0661336228983590247373913761919084208785D+00
      Alpha(17) = 13.2911275294674962061530765211614379950333D+00
C
      Xi( 1) = 1.003296779115244159925370015518D+00
      Xi( 2) = 1.029808878893360909144438775087D+00
      Xi( 3) = 1.083571537207537143713415028756D+00
      Xi( 4) = 1.166078406934612099118964145816D+00
      Xi( 5) = 1.279617310379891032901325531590D+00
      Xi( 6) = 1.427339244833476762154035599739D+00
      Xi( 7) = 1.613363210094094657583684016000D+00
      Xi( 8) = 1.842920671002319024460967822332D+00
      Xi( 9) = 2.122538146114561116153604536372D+00
      Xi(10) = 2.460261767966067304952773597826D+00
      Xi(11) = 2.865926179907826641841961379242D+00
      Xi(12) = 3.351478973975718224625336416622D+00
      Xi(13) = 3.931372321603539599131973969470D+00
      Xi(14) = 4.623035084321579500730298173394D+00
      Xi(15) = 5.447437774494031566870771987610D+00
      Xi(16) = 6.429762708165225281592425332633D+00
      Xi(17) = 7.600186972482000267829771456007D+00
      Xi(18) = 8.994777496566889835212521564500D+00
      Xi(19) = 10.656479258892964868260522237620D+00
      Xi(20) = 12.636141833151536209220444639811D+00
      Xi(21) = 14.993463705687141253493355197435D+00
      Xi(22) = 17.797615773195354276284274597231D+00
      Xi(23) = 21.127093376418113003512022629593D+00
      Xi(24) = 25.068012988698169147411265456071D+00
      Xi(25) = 29.709490481022010189768312038439D+00
      Xi(26) = 35.133926553450054650062917005471D+00
      Xi(27) = 41.398943821547749412348427000552D+00
      Xi(28) = 48.506794614204075954955186489315D+00
      Xi(29) = 56.357422262427981987653602402588D+00
      Xi(30) = 64.685922339974741085422138553440D+00
      Xi(31) = 72.998129197201843718079139478050D+00
      Xi(32) = 80.542045616521516282482195947523D+00
      Xi(33) = 86.376471319539266387199649699369D+00
      Xi(34) = 89.584005199299913783927973298660D+00
      GOTO 9999
C
C    ----- K = 17 , R = 1E2 -----
C
 1721 CONTINUE
C
      RIni = 1.0D+02
C
      Omega( 1) = 0.0085486027112414137163487210663292970025D+00
      Omega( 2) = 0.0204917524282931772402181497955142575051D+00
      Omega( 3) = 0.0340047708324275376429266393857320238681D+00
      Omega( 4) = 0.0503655153946454480856859473036912788757D+00
      Omega( 5) = 0.0714663291839928747361285026051813673575D+00
      Omega( 6) = 0.1000591584195262242389448670287777076737D+00
      Omega( 7) = 0.1399628625763894228896172186593638286922D+00
      Omega( 8) = 0.1962682990482164567346392083924122573535D+00
      Omega( 9) = 0.2757171017009835744322916728510364237081D+00
      Omega(10) = 0.3874077253288334146360573362333212799058D+00
      Omega(11) = 0.5439384928815124032198263792636794278224D+00
      Omega(12) = 0.7632840773608892314056398320509799759748D+00
      Omega(13) = 1.0723178584501672561731661081907418520132D+00
      Omega(14) = 1.5146942602000858187939746901662374511943D+00
      Omega(15) = 2.1723542263902079296842423383750997345487D+00
      Omega(16) = 3.2423058243498171153255948961913190942141D+00
      Omega(17) = 5.4939008854835959206816542454987484234152D+00
C
      Alpha( 1) = 0.0033198247340284218243658512269636529091D+00
      Alpha( 2) = 0.0177496313316968076455022815945961056627D+00
      Alpha( 3) = 0.0448214559364190146659623521785409394624D+00
      Alpha( 4) = 0.0867017057150739512905103685558305670611D+00
      Alpha( 5) = 0.1471228721624464812976198699323582275156D+00
      Alpha( 6) = 0.2321198066274996981334143367203814989352D+00
      Alpha( 7) = 0.3509975895016110316833920362844523310741D+00
      Alpha( 8) = 0.5174948868641361680777851494994479253364D+00
      Alpha( 9) = 0.7512229986159361620886784627337817710213D+00
      Alpha(10) = 1.0796383396633986272031627273726428484224D+00
      Alpha(11) = 1.5409273193095968162644182619658295152476D+00
      Alpha(12) = 2.1883595496344136345067549198617484762508D+00
      Alpha(13) = 3.0972191923966147315740865542110782371310D+00
      Alpha(14) = 4.3770675431933018500324505950516140728723D+00
      Alpha(15) = 6.1973374899988810981035780578451976907672D+00
      Alpha(16) = 8.8551748034756879732545820083089438412571D+00
      Alpha(17) = 13.0437004598485047808520409517996085924096D+00
C
      Xi( 1) = 1.003397304594680938902989675299D+00
      Xi( 2) = 1.030722356774342372267780354456D+00
      Xi( 3) = 1.086157905417406153926636325213D+00
      Xi( 4) = 1.171293105508434075306793686888D+00
      Xi( 5) = 1.288563641148398332533439591963D+00
      Xi( 6) = 1.441327846477146027641229319993D+00
      Xi( 7) = 1.633983056430369042058385609106D+00
      Xi( 8) = 1.872124856731489036393289993043D+00
      Xi( 9) = 2.162750234354115420063249808358D+00
      Xi(10) = 2.514506306061629360530662591344D+00
      Xi(11) = 2.937989383136705290235773935237D+00
      Xi(12) = 3.446109522244970960495533129908D+00
      Xi(13) = 4.054529740004681303677108727967D+00
      Xi(14) = 4.782200741792271876315356671228D+00
      Xi(15) = 5.652005271429134055077153631075D+00
      Xi(16) = 6.691527239140495126549362803203D+00
      Xi(17) = 7.933959967695566292342213010613D+00
      Xi(18) = 9.419157721786756060078360519583D+00
      Xi(19) = 11.194818659897883121080719437401D+00
      Xi(20) = 13.317749785224979691862501329780D+00
      Xi(21) = 15.855096568877923896359605215878D+00
      Xi(22) = 18.885291875017808239933803093891D+00
      Xi(23) = 22.498252544423162078632283389368D+00
      Xi(24) = 26.793960055960475894615280267885D+00
      Xi(25) = 31.877911764722284731002344138062D+00
      Xi(26) = 37.850957461647661145276444472074D+00
      Xi(27) = 44.789632960802394250066083003503D+00
      Xi(28) = 52.711817335957233870979976586568D+00
      Xi(29) = 61.522385107045003675707572909914D+00
      Xi(30) = 70.938197102818303187155013489473D+00
      Xi(31) = 80.406727242843933162186331742305D+00
      Xi(32) = 89.062467800429165006026455131405D+00
      Xi(33) = 95.798208021227316426304820851101D+00
      Xi(34) = 99.516891069178648451809543473701D+00
      GOTO 9999
C
C     ----- K = 17 , R = 2E2 -----
C
 1722 CONTINUE
C
      RIni = 2.0D+02
C
      Omega( 1) = 0.0048673410480669797478294228646605867894D+00
      Omega( 2) = 0.0117746330456780427921052094124287901167D+00
      Omega( 3) = 0.0198870791428627719878467153963308433617D+00
      Omega( 4) = 0.0302776004275052840479694193920323641578D+00
      Omega( 5) = 0.0446058663311346083678147773216027260901D+00
      Omega( 6) = 0.0653179331698807654382928078451442388541D+00
      Omega( 7) = 0.0958124822046568227828589395245195703410D+00
      Omega( 8) = 0.1407135799336433006312104335155943601876D+00
      Omega( 9) = 0.2064043244858754213232835544866716759316D+00
      Omega(10) = 0.3018653172872777795817614970852460487549D+00
      Omega(11) = 0.4399011184015803755254902379645542964681D+00
      Omega(12) = 0.6390746417174834501334461378529994135533D+00
      Omega(13) = 0.9272349393169635067444775411438229184569D+00
      Omega(14) = 1.3492433066001499579337000178647087977879D+00
      Omega(15) = 1.9879267827114792103635962372543133369618D+00
      Omega(16) = 3.0391568135442928416493912902041074630688D+00
      Omega(17) = 5.2603872308116383059439780112143125734292D+00
C
      Alpha( 1) = 0.0018882627435518682409133786278578048812D+00
      Alpha( 2) = 0.0101417169947529204940798877934704513848D+00
      Alpha( 3) = 0.0258347276183185878667663681190758140360D+00
      Alpha( 4) = 0.0506687265340891762473672903926980026768D+00
      Alpha( 5) = 0.0876936725177892334700592048835066094625D+00
      Alpha( 6) = 0.1419979493343611348524571647011782715708D+00
      Alpha( 7) = 0.2215771340255664241270525463556317902203D+00
      Alpha( 8) = 0.3384059441793975349984519018775586118863D+00
      Alpha( 9) = 0.5099057820568959907602327175091261324269D+00
      Alpha(10) = 0.7610963644983120443511179220585916027630D+00
      Alpha(11) = 1.1277658774087937613474219578080237624818D+00
      Alpha(12) = 1.6611792947718912516258046863093511547049D+00
      Alpha(13) = 2.4354020371117624408838220340101088368101D+00
      Alpha(14) = 3.5598923622