/*  -*- Mode: c++ -*-
 *
 *   Class AVFormatWriter
 *
 *   Copyright (C) Chris Pinkham 2011
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301 USA
 */

#include "mythlogging.h"
#include "mythcorecontext.h"
#include "avformatwriter.h"
#include "audiooutpututil.h"
#include "mythavutil.h"

extern "C" {
#if HAVE_BIGENDIAN
#include "bswap.h"
#endif
#include "libavutil/opt.h"
#include "libavutil/samplefmt.h"
#include "libavutil/imgutils.h"
}

#define LOC QString("AVFW(%1): ").arg(m_filename)
#define LOC_ERR QString("AVFW(%1) Error: ").arg(m_filename)
#define LOC_WARN QString("AVFW(%1) Warning: ").arg(m_filename)

AVFormatWriter::~AVFormatWriter()
{
    QMutexLocker locker(avcodeclock);

    if (m_ctx)
    {
        (void)av_write_trailer(m_ctx);
        avio_closep(&m_ctx->pb);
        for(unsigned int i = 0; i < m_ctx->nb_streams; i++) {
            av_freep(&m_ctx->streams[i]);
        }
        av_freep(&m_ctx);
    }

    if (m_audioInBuf)
        av_freep(&m_audioInBuf);

    if (m_audioInPBuf)
        av_freep(&m_audioInPBuf);

    if (m_audPicture)
        av_frame_free(&m_audPicture);

    Cleanup();

    av_frame_free(&m_picture);
}

bool AVFormatWriter::Init(void)
{
    AVOutputFormat *fmt = av_guess_format(m_container.toLatin1().constData(),
                                          nullptr, nullptr);
    if (!fmt)
    {
        LOG(VB_RECORD, LOG_ERR, LOC +
            QString("Init(): Unable to guess AVOutputFormat from container %1")
                    .arg(m_container));
        return false;
    }

    m_fmt = *fmt;

    if (m_width && m_height)
    {
        m_avVideoCodec = avcodec_find_encoder_by_name(
            m_videoCodec.toLatin1().constData());
        if (!m_avVideoCodec)
        {
            LOG(VB_RECORD, LOG_ERR, LOC +
                QString("Init(): Unable to find video codec %1").arg(m_videoCodec));
            return false;
        }

        m_fmt.video_codec = m_avVideoCodec->id;
    }
    else
        m_fmt.video_codec = AV_CODEC_ID_NONE;

    m_avAudioCodec = avcodec_find_encoder_by_name(
        m_audioCodec.toLatin1().constData());
    if (!m_avAudioCodec)
    {
        LOG(VB_RECORD, LOG_ERR, LOC +
            QString("Init(): Unable to find audio codec %1").arg(m_audioCodec));
        return false;
    }

    m_fmt.audio_codec = m_avAudioCodec->id;

    m_ctx = avformat_alloc_context();
    if (!m_ctx)
    {
        LOG(VB_RECORD, LOG_ERR,
            LOC + "Init(): Unable to allocate AVFormatContext");
        return false;
    }

    m_ctx->oformat = &m_fmt;

    if (m_container == "mpegts")
        m_ctx->packet_size = 2324;

    QByteArray filename = m_filename.toLatin1();
    auto size = static_cast<size_t>(filename.size());
    m_ctx->url = static_cast<char*>(av_malloc(size));
    memcpy(m_ctx->url, filename.constData(), size);

    if (m_fmt.video_codec != AV_CODEC_ID_NONE)
        m_videoStream = AddVideoStream();
    if (m_fmt.audio_codec != AV_CODEC_ID_NONE)
        m_audioStream = AddAudioStream();

    if ((m_videoStream) && (!OpenVideo()))
    {
        LOG(VB_RECORD, LOG_ERR, LOC + "Init(): OpenVideo() failed");
        return false;
    }

    if ((m_audioStream) && (!OpenAudio()))
    {
        LOG(VB_RECORD, LOG_ERR, LOC + "Init(): OpenAudio() failed");
        return false;
    }

    return true;
}

bool AVFormatWriter::OpenFile(void)
{
    if (!(m_fmt.flags & AVFMT_NOFILE))
    {
        if (avio_open(&m_ctx->pb, m_filename.toLatin1().constData(),
                      AVIO_FLAG_WRITE) < 0)
        {
            LOG(VB_RECORD, LOG_ERR, LOC + "OpenFile(): avio_open() failed");
            return false;
        }
    }

    m_ringBuffer = RingBuffer::Create(m_filename, true);

    if (!m_ringBuffer || !m_ringBuffer->GetLastError().isEmpty())
    {
        LOG(VB_RECORD, LOG_ERR, LOC +
            QString("OpenFile(): RingBuffer::Create() failed: '%1'")
            .arg(m_ringBuffer ? m_ringBuffer->GetLastError() : ""));
        Cleanup();
        return false;
    }

    m_avfRingBuffer     = new AVFRingBuffer(m_ringBuffer);
    auto *uc            = (URLContext *)m_ctx->pb->opaque;
    uc->prot            = AVFRingBuffer::GetRingBufferURLProtocol();
    uc->priv_data       = (void *)m_avfRingBuffer;

    if (avformat_write_header(m_ctx, nullptr) < 0)
    {
        Cleanup();
        return false;
    }

    return true;
}

void AVFormatWriter::Cleanup(void)
{
    if (m_ctx && m_ctx->pb)
    {
        avio_closep(&m_ctx->pb);
    }
    delete m_avfRingBuffer;
    m_avfRingBuffer = nullptr;
    delete m_ringBuffer;
    m_ringBuffer = nullptr;
}

bool AVFormatWriter::CloseFile(void)
{
    if (m_ctx)
    {
        (void)av_write_trailer(m_ctx);
        avio_close(m_ctx->pb);
        for(unsigned int i = 0; i < m_ctx->nb_streams; i++) {
            av_freep(&m_ctx->streams[i]);
        }

        av_freep(&m_ctx);
    }

    return true;
}

bool AVFormatWriter::NextFrameIsKeyFrame(void)
{
    return (m_bufferedVideoFrameTypes.isEmpty()) ||
           (m_bufferedVideoFrameTypes.first() == AV_PICTURE_TYPE_I);
}

int AVFormatWriter::WriteVideoFrame(VideoFrame *frame)
{
    int framesEncoded = m_framesWritten + m_bufferedVideoFrameTimes.size();

    av_frame_unref(m_picture);
    AVPictureFill(m_picture, frame);
    m_picture->pts = framesEncoded + 1;

    if ((framesEncoded % m_keyFrameDist) == 0)
        m_picture->pict_type = AV_PICTURE_TYPE_I;
    else
        m_picture->pict_type = AV_PICTURE_TYPE_NONE;

    int got_pkt = 0;
    int ret = 0;

    m_bufferedVideoFrameTimes.push_back(frame->timecode);
    m_bufferedVideoFrameTypes.push_back(m_picture->pict_type);

    AVPacket pkt;
    av_init_packet(&pkt);
    pkt.data = nullptr;
    pkt.size = 0;
    AVCodecContext *avctx = gCodecMap->getCodecContext(m_videoStream);
    {
        QMutexLocker locker(avcodeclock);
        ret = avcodec_encode_video2(avctx, &pkt,
                                    m_picture, &got_pkt);
    }

    if (ret < 0)
    {
        LOG(VB_RECORD, LOG_ERR, "avcodec_encode_video2() failed");
        return ret;
    }

    if (!got_pkt)
    {
        //LOG(VB_RECORD, LOG_DEBUG, QString("WriteVideoFrame(): Frame Buffered: cs: %1, mfw: %2, f->tc: %3, fn: %4, pt: %5").arg(pkt.size).arg(m_framesWritten).arg(frame->timecode).arg(frame->frameNumber).arg(m_picture->pict_type));
        return ret;
    }

    long long tc = frame->timecode;

    if (!m_bufferedVideoFrameTimes.isEmpty())
        tc = m_bufferedVideoFrameTimes.takeFirst();
    if (!m_bufferedVideoFrameTypes.isEmpty())
    {
        int pict_type = m_bufferedVideoFrameTypes.takeFirst();
        if (pict_type == AV_PICTURE_TYPE_I)
            pkt.flags |= AV_PKT_FLAG_KEY;
    }

    if (m_startingTimecodeOffset == -1)
        m_startingTimecodeOffset = tc - 1;
    tc -= m_startingTimecodeOffset;

    pkt.pts = tc * m_videoStream->time_base.den / m_videoStream->time_base.num / 1000;
    pkt.dts = AV_NOPTS_VALUE;
    pkt.stream_index= m_videoStream->index;

    //LOG(VB_RECORD, LOG_DEBUG, QString("WriteVideoFrame(): cs: %1, mfw: %2, pkt->pts: %3, tc: %4, fn: %5, pic->pts: %6, f->tc: %7, pt: %8").arg(pkt.size).arg(m_framesWritten).arg(pkt.pts).arg(tc).arg(frame->frameNumber).arg(m_picture->pts).arg(frame->timecode).arg(m_picture->pict_type));
    ret = av_interleaved_write_frame(m_ctx, &pkt);
    if (ret != 0)
        LOG(VB_RECORD, LOG_ERR, LOC + "WriteVideoFrame(): "
                "av_interleaved_write_frame couldn't write Video");

    frame->timecode = tc + m_startingTimecodeOffset;
    m_framesWritten++;

    av_packet_unref(&pkt);

    return 1;
}

#if HAVE_BIGENDIAN
static void bswap_16_buf(short int *buf, int buf_cnt, int audio_channels)
    __attribute__ ((unused)); /* <- suppress compiler warning */

static void bswap_16_buf(short int *buf, int buf_cnt, int audio_channels)
{
    for (int i = 0; i < audio_channels * buf_cnt; i++)
        buf[i] = bswap_16(buf[i]);
}
#endif

int AVFormatWriter::WriteAudioFrame(unsigned char *buf, int /*fnum*/, long long &timecode)
{
#if HAVE_BIGENDIAN
    bswap_16_buf((short int*) buf, m_audioFrameSize, m_audioChannels);
#endif

    bool got_packet = false;
    int ret = 0;
    AVCodecContext *avctx = gCodecMap->getCodecContext(m_audioStream);
    int samples_per_avframe  = m_audioFrameSize * m_audioChannels;
    int sampleSizeIn   = AudioOutputSettings::SampleSize(FORMAT_S16);
    AudioFormat format =
        AudioOutputSettings::AVSampleFormatToFormat(avctx->sample_fmt);
    int sampleSizeOut  = AudioOutputSettings::SampleSize(format);

    AVPacket pkt;
    av_init_packet(&pkt);
    pkt.data          = nullptr;
    pkt.size          = 0;

    if (av_get_packed_sample_fmt(avctx->sample_fmt) == AV_SAMPLE_FMT_FLT)
    {
        AudioOutputUtil::toFloat(FORMAT_S16, (void *)m_audioInBuf, (void *)buf,
                                 samples_per_avframe * sampleSizeIn);
        buf = m_audioInBuf;
    }
    if (av_sample_fmt_is_planar(avctx->sample_fmt))
    {
        AudioOutputUtil::DeinterleaveSamples(format,
                                             m_audioChannels,
                                             m_audioInPBuf,
                                             buf,
                                             samples_per_avframe * sampleSizeOut);

        // init AVFrame for planar data (input is interleaved)
        for (int j = 0, jj = 0; j < m_audioChannels; j++, jj += m_audioFrameSize)
        {
            m_audPicture->data[j] = m_audioInPBuf + jj * sampleSizeOut;
        }
    }
    else
    {
        m_audPicture->data[0] = buf;
    }

    m_audPicture->linesize[0] = m_audioFrameSize;
    m_audPicture->nb_samples = m_audioFrameSize;
    m_audPicture->format = avctx->sample_fmt;
    m_audPicture->extended_data = m_audPicture->data;

    m_bufferedAudioFrameTimes.push_back(timecode);

    {
        QMutexLocker locker(avcodeclock);
        //  SUGGESTION
        //  Now that avcodec_encode_audio2 is deprecated and replaced
        //  by 2 calls, this could be optimized
        //  into separate routines or separate threads.
        got_packet = false;
        ret = avcodec_receive_packet(avctx, &pkt);
        if (ret == 0)
            got_packet = true;
        if (ret == AVERROR(EAGAIN))
            ret = 0;
        if (ret == 0)
            ret = avcodec_send_frame(avctx, m_audPicture);
        // if ret from avcodec_send_frame is AVERROR(EAGAIN) then
        // there are 2 packets to be received while only 1 frame to be
        // sent. The code does not cater for this. Hopefully it will not happen.
    }

    if (ret < 0)
    {
        char error[AV_ERROR_MAX_STRING_SIZE];
        LOG(VB_GENERAL, LOG_ERR, LOC +
            QString("audio encode error: %1 (%2)")
            .arg(av_make_error_string(error, sizeof(error), ret))
            .arg(got_packet));
        return ret;
    }

    if (!got_packet)
    {
        //LOG(VB_RECORD, LOG_ERR, QString("WriteAudioFrame(): Frame Buffered: cs: %1, mfw: %2, f->tc: %3, fn: %4").arg(m_audPkt->size).arg(m_framesWritten).arg(timecode).arg(fnum));
        return ret;
    }

    long long tc = timecode;

    if (!m_bufferedAudioFrameTimes.empty())
        tc = m_bufferedAudioFrameTimes.takeFirst();

    if (m_startingTimecodeOffset == -1)
        m_startingTimecodeOffset = tc - 1;
    tc -= m_startingTimecodeOffset;

    if (m_avVideoCodec)
        pkt.pts = tc * m_videoStream->time_base.den / m_videoStream->time_base.num / 1000;
    else
        pkt.pts = tc * m_audioStream->time_base.den / m_audioStream->time_base.num / 1000;

    pkt.dts = AV_NOPTS_VALUE;
    pkt.flags |= AV_PKT_FLAG_KEY;
    pkt.stream_index = m_audioStream->index;

    //LOG(VB_RECORD, LOG_ERR, QString("WriteAudioFrame(): cs: %1, mfw: %2, pkt->pts: %3, tc: %4, fn: %5, f->tc: %6").arg(m_audPkt->size).arg(m_framesWritten).arg(m_audPkt->pts).arg(tc).arg(fnum).arg(timecode));

    ret = av_interleaved_write_frame(m_ctx, &pkt);
    if (ret != 0)
        LOG(VB_RECORD, LOG_ERR, LOC + "WriteAudioFrame(): "
                "av_interleaved_write_frame couldn't write Audio");
    timecode = tc + m_startingTimecodeOffset;

    av_packet_unref(&pkt);

    return 1;
}

int AVFormatWriter::WriteTextFrame(int /*vbimode*/, unsigned char */*buf*/, int /*len*/,
                                   long long /*timecode*/, int /*pagenr*/)
{
    return 1;
}

bool AVFormatWriter::ReOpen(const QString& filename)
{
    bool result = m_ringBuffer->ReOpen(filename);

    if (result)
        m_filename = filename;

    return result;
}

AVStream* AVFormatWriter::AddVideoStream(void)
{
    AVStream *st = avformat_new_stream(m_ctx, nullptr);
    if (!st)
    {
        LOG(VB_RECORD, LOG_ERR,
            LOC + "AddVideoStream(): avformat_new_stream() failed");
        return nullptr;
    }
    st->id = 0;

    AVCodec *codec = avcodec_find_encoder(m_ctx->oformat->video_codec);
    if (!codec)
    {
        LOG(VB_RECORD, LOG_ERR,
            LOC + "AddVideoStream(): avcodec_find_encoder() failed");
        return nullptr;
    }

    gCodecMap->freeCodecContext(st);
    AVCodecContext *c = gCodecMap->getCodecContext(st, codec);

    c->codec                      = codec;
    c->codec_id                   = m_ctx->oformat->video_codec;
    c->codec_type                 = AVMEDIA_TYPE_VIDEO;

    c->bit_rate                   = m_videoBitrate;
    c->width                      = m_width;
    c->height                     = m_height;

    // c->sample_aspect_ratio.num    = (int)floor(m_aspect * 10000);
    // c->sample_aspect_ratio.den    = 10000;

    c->time_base                  = GetCodecTimeBase();

    st->time_base.den             = 90000;
    st->time_base.num             = 1;
    st->r_frame_rate.num          = 0;
    st->r_frame_rate.den          = 0;

    c->gop_size                   = m_keyFrameDist;
    c->pix_fmt                    = AV_PIX_FMT_YUV420P;
    c->thread_count               = m_encodingThreadCount;
    c->thread_type                = FF_THREAD_SLICE;

    if (c->codec_id == AV_CODEC_ID_MPEG2VIDEO) {
        c->max_b_frames          = 2;
    }
    else if (c->codec_id == AV_CODEC_ID_MPEG1VIDEO)
    {
        c->mb_decision           = 2;
    }
    else if (c->codec_id == AV_CODEC_ID_H264)
    {

        // Try to provide the widest software/device support by automatically using
        // the Baseline profile where the given bitrate and resolution permits

        if ((c->height > 720) || // Approximate highest resolution supported by Baseline 3.1
            (c->bit_rate > 1000000)) // 14,000 Kbps aka 14Mbps maximum permissable rate for Baseline 3.1
        {
            c->level = 40;
            // AVCodecContext AVOptions:
            av_opt_set(c->priv_data, "profile", "main", 0);
        }
        else if ((c->height > 576) || // Approximate highest resolution supported by Baseline 3.0
            (c->bit_rate > 1000000))  // 10,000 Kbps aka 10Mbps maximum permissable rate for Baseline 3.0
        {
            c->level = 31;
            // AVCodecContext AVOptions:
            av_opt_set(c->priv_data, "profile", "baseline", 0);
        }
        else
        {
            c->level = 30; // Baseline 3.0 is the most widely supported, but it's limited to SD
            // AVCodecContext AVOptions:
            av_opt_set(c->priv_data, "profile", "baseline", 0);
        }

        // AVCodecContext AVOptions:
        // c->coder_type            = 0;
        av_opt_set_int(c, "coder", FF_CODER_TYPE_VLC, 0);
        c->max_b_frames          = 0;
        c->slices                = 8;

        c->flags                |= AV_CODEC_FLAG_LOOP_FILTER;
        c->me_cmp               |= 1;
        // c->me_method             = ME_HEX;
        // av_opt_set_int(c, "me_method", ME_HEX, 0);
        av_opt_set(c, "me_method", "hex", 0);
        c->me_subpel_quality     = 6;
        c->me_range              = 16;
        c->keyint_min            = 25;
        // c->scenechange_threshold = 40;
        av_opt_set_int(c, "sc_threshold", 40, 0);
        c->i_quant_factor        = 0.71;
        // c->b_frame_strategy      = 1;
        av_opt_set_int(c, "b_strategy", 1, 0);
        c->qcompress             = 0.6;
        c->qmin                  = 10;
        c->qmax                  = 51;
        c->max_qdiff             = 4;
        c->refs                  = 3;
        c->trellis               = 0;

        // libx264 AVOptions:
        av_opt_set(c, "partitions", "i8x8,i4x4,p8x8,b8x8", 0);
        av_opt_set_int(c, "direct-pred", 1, 0);
        av_opt_set_int(c, "rc-lookahead", 0, 0);
        av_opt_set_int(c, "fast-pskip", 1, 0);
        av_opt_set_int(c, "mixed-refs", 1, 0);
        av_opt_set_int(c, "8x8dct", 0, 0);
        av_opt_set_int(c, "weightb", 0, 0);

        // libx264 AVOptions:
        av_opt_set(c->priv_data, "preset",
                   m_encodingPreset.toLatin1().constData(), 0);
        av_opt_set(c->priv_data, "tune",
                   m_encodingTune.toLatin1().constData(), 0);
    }

    if(m_ctx->oformat->flags & AVFMT_GLOBALHEADER)
        c->flags |= AV_CODEC_FLAG_GLOBAL_HEADER;

    return st;
}

bool AVFormatWriter::OpenVideo(void)
{
    AVCodecContext *c = gCodecMap->getCodecContext(m_videoStream);

    if (!m_width || !m_height)
        return false;

    if (avcodec_open2(c, nullptr, nullptr) < 0)
    {
        LOG(VB_RECORD, LOG_ERR,
            LOC + "OpenVideo(): avcodec_open() failed");
        return false;
    }

    if (!m_picture)
    {
        m_picture = AllocPicture(c->pix_fmt);
        if (!m_picture)
        {
            LOG(VB_RECORD, LOG_ERR,
                LOC + "OpenVideo(): AllocPicture() failed");
            return false;
        }
    }
    else
    {
        av_frame_unref(m_picture);
    }

    return true;
}

AVStream* AVFormatWriter::AddAudioStream(void)
{
    AVStream *st = avformat_new_stream(m_ctx, nullptr);
    if (!st)
    {
        LOG(VB_RECORD, LOG_ERR,
            LOC + "AddAudioStream(): avformat_new_stream() failed");
        return nullptr;
    }
    st->id = 1;

    AVCodecContext *c = gCodecMap->getCodecContext(st, nullptr, true);

    c->codec_id     = m_ctx->oformat->audio_codec;
    c->codec_type   = AVMEDIA_TYPE_AUDIO;
    c->bit_rate     = m_audioBitrate;
    c->sample_rate  = m_audioFrameRate;
    c->channels     = m_audioChannels;
    c->channel_layout = static_cast<uint64_t>(av_get_default_channel_layout(m_audioChannels));

    // c->flags |= CODEC_FLAG_QSCALE; // VBR
    // c->global_quality = blah;

    if (!m_avVideoCodec)
    {
        c->time_base      = GetCodecTimeBase();
        st->time_base.den = 90000;
        st->time_base.num = 1;
    }

    if(m_ctx->oformat->flags & AVFMT_GLOBALHEADER)
        c->flags |= AV_CODEC_FLAG_GLOBAL_HEADER;

    return st;
}

bool AVFormatWriter::FindAudioFormat(AVCodecContext *ctx, AVCodec *c, AVSampleFormat format)
{
    if (c->sample_fmts)
    {
        for (int i = 0; c->sample_fmts[i] != AV_SAMPLE_FMT_NONE; i++)
        {
            if (av_get_packed_sample_fmt(c->sample_fmts[i]) == format)
            {
                ctx->sample_fmt = c->sample_fmts[i];
                return true;
            }
        }
    }
    return false;
}

bool AVFormatWriter::OpenAudio(void)
{
    AVCodecContext *c = gCodecMap->getCodecContext(m_audioStream);

    c->strict_std_compliance = FF_COMPLIANCE_EXPERIMENTAL;

    AVCodec *codec = avcodec_find_encoder(c->codec_id);
    if (!codec)
    {
        LOG(VB_RECORD, LOG_ERR,
            LOC + "OpenAudio(): avcodec_find_encoder() failed");
        return false;
    }

    // try to find suitable format we can use. avcodec_open2 will fail if we don't
    // find one, so no need to worry otherwise. Can only handle S16 or FLOAT
    // we give priority to S16 as libmp3lame requires aligned floats which we can't guarantee
    if (!FindAudioFormat(c, codec, AV_SAMPLE_FMT_S16))
    {
        FindAudioFormat(c, codec, AV_SAMPLE_FMT_FLT);
    }

    if (avcodec_open2(c, codec, nullptr) < 0)
    {
        LOG(VB_RECORD, LOG_ERR,
            LOC + "OpenAudio(): avcodec_open() failed");
        return false;
    }

    m_audioFrameSize = c->frame_size; // number of *samples* per channel in an AVFrame

    m_audPicture = av_frame_alloc();
    if (!m_audPicture)
    {
        LOG(VB_RECORD, LOG_ERR,
            LOC + "OpenAudio(): alloc_frame() failed");
        return false;
    }

    int samples_per_frame = m_audioFrameSize * m_audioChannels;
    int bps = av_get_bytes_per_sample(c->sample_fmt);
    if (av_get_packed_sample_fmt(c->sample_fmt) == AV_SAMPLE_FMT_FLT)
    {
        // allocate buffer to convert from S16 to float
        if (!(m_audioInBuf = (unsigned char*)av_malloc(bps * samples_per_frame)))
            return false;
    }
    if (av_sample_fmt_is_planar(c->sample_fmt))
    {
        // allocate buffer to convert interleaved to planar audio
        if (!(m_audioInPBuf = (unsigned char*)av_malloc(bps * samples_per_frame)))
            return false;
    }
    return true;
}

AVFrame* AVFormatWriter::AllocPicture(enum AVPixelFormat pix_fmt)
{
    AVFrame *picture = av_frame_alloc();
    if (!picture)
    {
        LOG(VB_RECORD, LOG_ERR,
            LOC + "AllocPicture(): avcodec_alloc_frame() failed");
        return nullptr;
    }
    int size = av_image_get_buffer_size(pix_fmt, m_width, m_height, IMAGE_ALIGN);
    auto *picture_buf = (unsigned char *)av_malloc(size);
    if (!picture_buf)
    {
        LOG(VB_RECORD, LOG_ERR, LOC + "AllocPicture(): av_malloc() failed");
        av_frame_free(&picture);
        return nullptr;
    }
    av_image_fill_arrays(picture->data, picture->linesize,
        picture_buf, pix_fmt, m_width, m_height, IMAGE_ALIGN);
    return picture;
}

AVRational AVFormatWriter::GetCodecTimeBase(void)
{
    AVRational result;

    result.den = (int)floor(m_frameRate * 100);
    result.num = 100;

    if (m_avVideoCodec && m_avVideoCodec->supported_framerates) {
        const AVRational *p= m_avVideoCodec->supported_framerates;
        AVRational req = {result.den, result.num};
        const AVRational *best = nullptr;
        AVRational best_error= {INT_MAX, 1};
        for(; p->den!=0; p++) {
            AVRational error = av_sub_q(req, *p);
            if (error.num <0)
                error.num *= -1;
            if (av_cmp_q(error, best_error) < 0) {
                best_error = error;
                best = p;
            }
        }

        if (best && best->num && best->den)
        {
            result.den = best->num;
            result.num = best->den;
        }
    }

    if (result.den == 2997)
    {
         result.den = 30000;
         result.num = 1001;
    }
    else if (result.den == 5994)
    {
         result.den = 60000;
         result.num = 1001;
    }

    return result;
}

/* vim: set expandtab tabstop=4 shiftwidth=4: */
