/*
 * Copyright (c) 1999, 2003, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package javax.management;


/**
 * Represents a notification emitted by the MBean server through the MBeanServerDelegate MBean.
 * The MBean Server emits the following types of notifications: MBean registration, MBean
 * de-registration.
 * <P>
 * To receive to MBeanServerNotifications, you need to be declared as listener to
 * the {@link javax.management.MBeanServerDelegate javax.management.MBeanServerDelegate} MBean
 * that represents the MBeanServer. The ObjectName of the MBeanServerDelegate is:
 * <CODE>JMImplementation:type=MBeanServerDelegate</CODE>.
 *
 * @since 1.5
 */
 public class MBeanServerNotification extends Notification   {


     /* Serial version */
     private static final long serialVersionUID = 2876477500475969677L;

     /**
      * Notification type denoting that an MBean has been registered. Value is "JMX.mbean.registered".
      */
     public static final String REGISTRATION_NOTIFICATION = "JMX.mbean.registered" ;

     /**
      * Notification type denoting that an MBean has been unregistered. Value is "JMX.mbean.unregistered".
      */
     public static final String UNREGISTRATION_NOTIFICATION = "JMX.mbean.unregistered" ;


     /**
      * @serial The object names of the MBeans concerned by this notification
      */
     private final ObjectName objectName;


     /**
      * Creates an MBeanServerNotification object specifying object names of
      * the MBeans that caused the notification and the specified notification type.
      *
      * @param type A string denoting the type of the
      * notification. Set it to one these values: {@link
      * #REGISTRATION_NOTIFICATION}, {@link
      * #UNREGISTRATION_NOTIFICATION}.
      * @param source The MBeanServerNotification object responsible
      * for forwarding MBean server notification.
      * @param sequenceNumber A sequence number that can be used to order
      * received notifications.
      * @param objectName The object name of the MBean that caused the notification.
      *
      */
     public MBeanServerNotification(String type, Object source, long sequenceNumber, ObjectName objectName ) {
         super (type,source,sequenceNumber) ;
         this.objectName =  objectName ;
     }

     /**
      * Returns the  object name of the MBean that caused the notification.
      *
      * @return the object name of the MBean that caused the notification.
      */
     public ObjectName getMBeanName()  {
         return objectName ;
     }

 }
