///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <scripting/Scripting.h>
#include <scripting/engine/ScriptEngine.h>
#include <scripting/interface/ContainerWrappers.h>
#include "../CrystalAnalysis.h"
#include "../deformation/CalculateExtrinsicStrainModifier.h"
#include "../deformation/CalculateIntrinsicStrainModifier.h"
#include "../grains/AnalyzeMicrostructureModifier.h"

using namespace boost::python;
using namespace CrystalAnalysis;
using namespace AtomViz;
using namespace Scripting;

BOOST_PYTHON_MODULE(CrystalAnalysis)
{
	class_<CalculateExtrinsicStrainModifier, bases<AtomsObjectAnalyzerBase>, intrusive_ptr<CalculateExtrinsicStrainModifier>, noncopyable>("CalculateExtrinsicStrainModifier", init<>())
		.def("Calculate", &CalculateExtrinsicStrainModifier::calculate)
		.add_property("DeformationGradients", make_function(&CalculateExtrinsicStrainModifier::deformationGradients, return_internal_reference<>()))
		.add_property("StrainTensors", make_function(&CalculateExtrinsicStrainModifier::strainTensors, return_internal_reference<>()))
		.add_property("HydrostaticStrains", make_function(&CalculateExtrinsicStrainModifier::hydrostaticStrains, return_internal_reference<>()))
		.add_property("ShearStrains", make_function(&CalculateExtrinsicStrainModifier::shearStrains, return_internal_reference<>()))
	;

	class_<CalculateIntrinsicStrainModifier, bases<AtomsObjectAnalyzerBase>, intrusive_ptr<CalculateIntrinsicStrainModifier>, noncopyable>("CalculateIntrinsicStrainModifier", init<>())
		.def("Calculate", &CalculateIntrinsicStrainModifier::calculate)
		.add_property("DeformationGradients", make_function(&CalculateIntrinsicStrainModifier::deformationGradients, return_internal_reference<>()))
		.add_property("StrainTensors", make_function(&CalculateIntrinsicStrainModifier::strainTensors, return_internal_reference<>()))
		.add_property("HydrostaticStrains", make_function(&CalculateIntrinsicStrainModifier::hydrostaticStrains, return_internal_reference<>()))
		.add_property("ShearStrains", make_function(&CalculateIntrinsicStrainModifier::shearStrains, return_internal_reference<>()))
		.add_property("LatticeConstant", &CalculateIntrinsicStrainModifier::latticeConstant, &CalculateIntrinsicStrainModifier::setLatticeConstant)
	;

	class_<QVector<FindGrains::GrainInfo>, noncopyable>("GrainInfoList", no_init)
		.def(QVector_readonly_indexing_suite< QVector<FindGrains::GrainInfo>, return_internal_reference<> >())
	;

	class_<QVector<FindGrains::GrainBoundaryInfo>, noncopyable>("GrainBoundaryInfoList", no_init)
		.def(QVector_readonly_indexing_suite< QVector<FindGrains::GrainBoundaryInfo>, return_internal_reference<> >())
	;

	class_<AnalyzeMicrostructureModifier, bases<AtomsObjectAnalyzerBase>, intrusive_ptr<AnalyzeMicrostructureModifier>, noncopyable>("AnalyzeMicrostructureModifier", init<>())
		.def("Calculate", &AnalyzeMicrostructureModifier::calculate)
		.add_property("ClusterChannel", make_function(&AnalyzeMicrostructureModifier::clusterChannel, return_internal_reference<>()))
		.add_property("MisorientationChannel", make_function(&AnalyzeMicrostructureModifier::misorientationChannel, return_internal_reference<>()))
		.add_property("GrainMisorientationThreshold", &AnalyzeMicrostructureModifier::grainMisorientationThreshold, &AnalyzeMicrostructureModifier::setGrainMisorientationThreshold)
		.add_property("MinimumGrainSize", &AnalyzeMicrostructureModifier::minimumGrainSize, &AnalyzeMicrostructureModifier::setMinimumGrainSize)
		.add_property("AssignRandomColors", &AnalyzeMicrostructureModifier::assignRandomColors, &AnalyzeMicrostructureModifier::setAssignRandomColors)
		.add_property("CalculateAtomicMisorientations", &AnalyzeMicrostructureModifier::calculateAtomicMisorientations, &AnalyzeMicrostructureModifier::setCalculateAtomicMisorientations)
		.add_property("Grains", make_function(&AnalyzeMicrostructureModifier::grains, return_internal_reference<>()))
		.add_property("GrainBoundaries", make_function(&AnalyzeMicrostructureModifier::grainBoundaries, return_internal_reference<>()))
	;

	class_<FindGrains::GrainInfo, noncopyable>("GrainInfo", no_init)
		.add_property("Id", &FindGrains::GrainInfo::id)
		.add_property("AtomCount", &FindGrains::GrainInfo::atomCount)
		.add_property("CrystallineCount", &FindGrains::GrainInfo::crystallineCount)
		.add_property("Color", make_function(&FindGrains::GrainInfo::color, return_value_policy<copy_const_reference>()))
		.add_property("AverageOrientation", make_function(&FindGrains::GrainInfo::averageOrientation, return_value_policy<copy_const_reference>()))
	;

	class_<FindGrains::GrainBoundaryInfo, noncopyable>("GrainBoundaryInfo", no_init)
		.add_property("Id", &FindGrains::GrainBoundaryInfo::id)
		.add_property("GrainA", &FindGrains::GrainBoundaryInfo::grainA)
		.add_property("GrainB", &FindGrains::GrainBoundaryInfo::grainB)
	;
}

OVITO_REGISTER_PLUGIN_PYTHON_INTERFACE(CrystalAnalysis);
