/*
 * Copyright (C) 2014 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "applicationcontroller.h"

// unity-mir
#include <logging.h>

// upstart
extern "C" {
    #include "upstart-app-launch.h"
}

namespace unitymir
{
namespace upstart
{

struct ApplicationController::Private
{
    upstart_app_launch_app_observer_t preStartCallback = nullptr;
    upstart_app_launch_app_observer_t startedCallback = nullptr;
    upstart_app_launch_app_observer_t stopCallback = nullptr;
    upstart_app_launch_app_observer_t focusCallback = nullptr;
    upstart_app_launch_app_observer_t resumeCallback = nullptr;
    upstart_app_launch_app_failed_observer_t failureCallback = nullptr;
};

ApplicationController::ApplicationController()
    : unitymir::ApplicationController(),
      impl(new Private())
{
    impl->preStartCallback = [](const gchar * appId, gpointer userData) {
        auto thiz = static_cast<ApplicationController*>(userData);
        Q_EMIT(thiz->applicationAboutToBeStarted(appId));
    };

    impl->startedCallback = [](const gchar * appId, gpointer userData) {
        auto thiz = static_cast<ApplicationController*>(userData);
        Q_EMIT(thiz->applicationStarted(appId));
    };

    impl->stopCallback = [](const gchar * appId, gpointer userData) {
        auto thiz = static_cast<ApplicationController*>(userData);
        Q_EMIT(thiz->applicationStopped(appId));
    };

    impl->focusCallback = [](const gchar * appId, gpointer userData) {
        auto thiz = static_cast<ApplicationController*>(userData);
        Q_EMIT(thiz->applicationFocusRequest(appId));
    };

    impl->resumeCallback = [](const gchar * appId, gpointer userData) {
        auto thiz = static_cast<ApplicationController*>(userData);
        Q_EMIT(thiz->applicationResumeRequest(appId));
    };

    impl->failureCallback = [](const gchar * appId, upstart_app_launch_app_failed_t failureType, gpointer userData) {
        ApplicationController::Error error;
        switch(failureType)
        {
        case UPSTART_APP_LAUNCH_APP_FAILED_CRASH: error = ApplicationController::Error::APPLICATION_CRASHED;
        case UPSTART_APP_LAUNCH_APP_FAILED_START_FAILURE: error = ApplicationController::Error::APPLICATION_FAILED_TO_START;
        }

        auto thiz = static_cast<ApplicationController*>(userData);
        Q_EMIT(thiz->applicationError(appId, error));
    };

    upstart_app_launch_observer_add_app_starting(impl->preStartCallback, this);
    upstart_app_launch_observer_add_app_started(impl->startedCallback, this);
    upstart_app_launch_observer_add_app_stop(impl->stopCallback, this);
    upstart_app_launch_observer_add_app_focus(impl->focusCallback, this);
    upstart_app_launch_observer_add_app_resume(impl->resumeCallback, this);
    upstart_app_launch_observer_add_app_failed(impl->failureCallback, this);
}

ApplicationController::~ApplicationController()
{
    upstart_app_launch_observer_delete_app_starting(impl->preStartCallback, this);
    upstart_app_launch_observer_delete_app_started(impl->startedCallback, this);
    upstart_app_launch_observer_delete_app_stop(impl->stopCallback, this);
    upstart_app_launch_observer_delete_app_focus(impl->focusCallback, this);
    upstart_app_launch_observer_delete_app_resume(impl->resumeCallback, this);
    upstart_app_launch_observer_delete_app_failed(impl->failureCallback, this);
}

pid_t ApplicationController::primaryPidForAppId(const QString& appId)
{
    GPid pid = upstart_app_launch_get_primary_pid(appId.toLatin1().constData());
    DLOG_IF(!pid, "ApplicationController::stopApplication appId='%s' FAILED", qPrintable(appId));

    return pid;
}

bool ApplicationController::appIdHasProcessId(pid_t pid, const QString& appId)
{
    return upstart_app_launch_pid_in_app_id(pid, appId.toLatin1().constData());
}

bool ApplicationController::stopApplicationWithAppId(const QString& appId)
{
    auto result = upstart_app_launch_stop_application(appId.toLatin1().constData());
    DLOG_IF(!result, "ApplicationController::stopApplicationWithAppId appId='%s' FAILED", qPrintable(appId));
    return result;
}

bool ApplicationController::startApplicationWithAppIdAndArgs(const QString& appId, const QStringList& arguments)
{
    // Convert arguments QStringList into format suitable for upstart-app-launch
    // The last item should be null, which is done by g_new0, we just don't fill it.
    auto upstartArgs = g_new0(gchar *, arguments.length() + 1);

    for (int i=0; i<arguments.length(); i++) {
        upstartArgs[i] = g_strdup(arguments.at(i).toUtf8().data());
    }

    auto result = upstart_app_launch_start_application(
                appId.toLatin1().constData(),
                static_cast<const gchar * const *>(upstartArgs));

    g_strfreev(upstartArgs);

    DLOG_IF(!result, "Application::Controller::startApplicationWithAppIdAndArgs appId='%s' FAILED", qPrintable(appId));
    return result;
}

} // namespace upstart
} // namespace unitymir
