 /*
 *  Copyright (c) 2008,2009,2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTLCore_TYPE_P_H_
#define _GTLCore_TYPE_P_H_

#include "Debug.h"
#include "Type.h"
#include "Parameter.h"
namespace llvm {
  class Type;
  class LLVMContext;
  class FunctionType;
}

namespace LLVMBackend {
  class Visitor;
}

namespace GTLCore {
  class Function;
  /**
   * @internal
   * @ingroup GTLCore
   */
  class GTLCORE_EXPORT Type::StructDataMember::Information {
    public:
      /**
       * @param setter is the name of a function that will take a pointer
       *               to the structure as first parameter, and the value
       *               as second parameter, and that will be called in operation=
       *               to change the value.
       */
      Information(Function* _setter) : m_setter(_setter) {
      }
      ~Information();
      const Function* setter() const {
        return m_setter;
      }
      void setSetter(Function* function)
      {
        GTL_ASSERT(m_setter == 0);
        m_setter = function;
      }
    private:
      Function* m_setter;
  };
  /**
   * @internal
   * @ingroup GTLCore
   *
   * Represent a function member of a structure.
   */
  class GTLCORE_EXPORT Type::StructFunctionMember {
    public:
      /**
       * 
       * @param _name 
       * @param _symbolName the name of the symbol to call for executing this member,
       *                    the first parameter of the function will contains a pointer
       *                    to this structure
       * @param _type 
       * @param _parameters 
       */
      StructFunctionMember(Function* _function);
      StructFunctionMember(const StructFunctionMember& _rhs);
      StructFunctionMember& operator=(const StructFunctionMember& _rhs);
      ~StructFunctionMember();
      /**
        * @return the name of the member.
        */
      const GTLCore::String& name() const;
      /**
        * @return the type of the function.
        */
      const Type* returnType() const;
      /**
        * @return the list of arguments
        */
      const std::vector<Parameter>& parameters() const;
      
      /**
       * @return the function associated with this member
       */
      const Function* function() const;
    private:
      struct Private;
      Private* const d;
  };
  
  /**
   * @internal
   * @ingroup GTLCore
   * Private information of the type object
   */
  struct GTLCORE_EXPORT Type::Private {
    friend class Type;
    public:
      /**
      * @internal
      * @ingroup GTLCore
      *
      * Allow to create a specific type. This use for instance to create
      * the function type used by function member @ref Type::StructFunctionMember .
      */
      class GTLCORE_EXPORT AribtraryTypeFactory {
        public:
          virtual ~AribtraryTypeFactory();
          virtual llvm::Type* createType(llvm::LLVMContext& context) = 0;
      };
    public:
      static inline const Private* d( const GTLCore::Type* _type )
      {
        return _type->d;
      }
    public:
      Private() : structDataMembers(0), structFunctionMembers(0), structPrivateFunctionMembers(0), m_visitor(0), m_factory(0) {}
      Private( Type::DataType _dataType );
      ~Private();
      llvm::Type * type(llvm::LLVMContext& _context) const;
      /**
       * @return a pointer version of the type a.k.a. llvm::PointerType::get( type(), 0)
       */
      llvm::Type * pointerType(llvm::LLVMContext& _context) const;
      /**
       * @return how the type will be used as function argument (usually return type(),
       *         except for ARRAY and STRUCTURE where it return pointerType() )
       */
      llvm::Type * asArgumentType(llvm::LLVMContext& _context) const;
      void addFunctionMember( const StructFunctionMember& );
      void addPrivateFunctionMember( const StructFunctionMember& );
      int memberToIndex(const GTLCore::String&);
      /**
       * @return the function member associated with that name, or null if none
       */
      std::vector<const Type::StructFunctionMember*> functionMembers( const GTLCore::String& _name) const;
      /**
       * @return the private function member associated with that name, or null if none
       */
      const Type::StructFunctionMember* privateFunctionMember( const GTLCore::String&) const;
      const LLVMBackend::Visitor* visitor() const;
      void setVisitor( const LLVMBackend::Visitor* );
      /**
        * Select the type that has the higher priority
        */
      static const Type* selectType(const Type* type1, const Type* type2);
      /**
       * @return true if this a complex structure (a structure where one of its element
       *         is an array or an other structure)
       * 
       * @code
       * Type* type1 = new Type( struct { int a; int b; } );
       * Type* type2 = new Type( struct { int a; int b[2]; } );
       * Type* type3 = new Type( struct { int a; struct { int b; } } );
       * type1->d->isComplexStructure() == false;
       * type2->d->isComplexStructure() == true;
       * type3->d->isComplexStructure() == true;
       * @endcode
       */
      bool isComplexStructure();
      /**
       * @return true if this a "nested" array (ie int a[2][3]; is nested, while int a[2]; isn't)
       */
      bool isNestedArray();
      /**
       * @return the subtype at a given index
       */
      const GTLCore::Type* subtypeAt( unsigned int _index );
      const GTLCore::String& symbolName() const;
      static const GTLCore::Type* createArbitraryType( AribtraryTypeFactory* _factory);
      static std::vector< llvm::Type*> createFunctionParams(llvm::LLVMContext& _context, const Function* function);
      llvm::FunctionType* createFunctionFunctionType(llvm::LLVMContext& _context, const GTLCore::Function* function);
      static void deleteTypes();
    private:
      DataType dataType;
      unsigned int vectorSize;
      const Type* arrayType;
      GTLCore::String structName;
      GTLCore::String m_symbolName;
      std::vector<StructDataMember>* structDataMembers;
      std::vector<Type::StructFunctionMember>* structFunctionMembers;
      std::vector<Type::StructFunctionMember>* structPrivateFunctionMembers;
      const LLVMBackend::Visitor* m_visitor;
      AribtraryTypeFactory* m_factory;
  };
  
}

#endif
