% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model-matrix.R
\name{model_matrix}
\alias{model_matrix}
\title{Construct a design matrix}
\usage{
model_matrix(terms, data, ..., call = current_env())
}
\arguments{
\item{terms}{A terms object to construct a model matrix with. This is
typically the terms object returned from the corresponding call to
\code{\link[=model_frame]{model_frame()}}.}

\item{data}{A tibble to construct the design matrix with. This is
typically the tibble returned from the corresponding call to
\code{\link[=model_frame]{model_frame()}}.}

\item{...}{These dots are for future extensions and must be empty.}

\item{call}{The call used for errors and warnings.}
}
\value{
A tibble containing the design matrix.
}
\description{
\code{model_matrix()} is a stricter version of \code{\link[stats:model.matrix]{stats::model.matrix()}}. Notably,
\code{model_matrix()} will \emph{never} drop rows, and the result will be a tibble.
}
\details{
The following explains the rationale for some of the difference in arguments
compared to \code{\link[stats:model.matrix]{stats::model.matrix()}}:
\itemize{
\item \code{contrasts.arg}: Set the contrasts argument, \code{options("contrasts")}
globally, or assign a contrast to the factor of interest directly using
\code{\link[stats:contrasts]{stats::contrasts()}}. See the examples section.
\item \code{xlev}: Not allowed because \code{model.frame()} is never called, so it is
unnecessary.
\item \code{...}: Not allowed because the default method of \code{model.matrix()} does
not use it, and the \code{lm} method uses it to pass potential offsets and
weights through, which are handled differently in hardhat.
}
}
\examples{
# ---------------------------------------------------------------------------
# Example usage

framed <- model_frame(Sepal.Width ~ Species, iris)

model_matrix(framed$terms, framed$data)

# ---------------------------------------------------------------------------
# Missing values never result in dropped rows

iris2 <- iris
iris2$Species[1] <- NA

framed2 <- model_frame(Sepal.Width ~ Species, iris2)

model_matrix(framed2$terms, framed2$data)

# ---------------------------------------------------------------------------
# Contrasts

# Default contrasts
y <- factor(c("a", "b"))
x <- data.frame(y = y)
framed <- model_frame(~y, x)

# Setting contrasts directly
y_with_contrast <- y
contrasts(y_with_contrast) <- contr.sum(2)
x2 <- data.frame(y = y_with_contrast)
framed2 <- model_frame(~y, x2)

# Compare!
model_matrix(framed$terms, framed$data)
model_matrix(framed2$terms, framed2$data)

# Also, can set the contrasts globally
global_override <- c(unordered = "contr.sum", ordered = "contr.poly")

rlang::with_options(
  .expr = {
    model_matrix(framed$terms, framed$data)
  },
  contrasts = global_override
)
}
