///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// Field: two dof vectors and the associated finite
//     element space
//
// contents:
//	* basic linear algebra (not complete)
//      * set boundary value with:
//         - a field
//         - a function
//         - a Float 
//      * interpolation of continuous fct with:
//         - P0, P1, P2, P1d
//         - Q0, Q1, Q2, Q1d
//      * input/output from/to file
//
// authors:
//      Pierre.Saramito@imag.fr
//	Jocelyn.Etienne@imag.fr
//
// date: 12 may 1997  update: 26 mar 2002
//

#include "rheolef/field.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h"
#include "rheolef/form_diag.h"
#include "rheolef/vec.h"
#include "rheolef/blas1-dense.h"
#include "rheolef/tiny_matvec.h"
#include "rheolef/geomap.h"
#include "rheolef/piola_v1.h"
using namespace rheolef;
using namespace std;
namespace rheolef { 

// ==========================================================================
// allocators
// ==========================================================================
field::field (const class form_diag& D)
: _V(D.get_space()),
   u(vec<Float>(D.uu)),
   b(vec<Float>(D.bb))
{
  check_macro(!_V.has_locked(), "Not implemented for spaces with locked components");
}
// conversion from field_component into field
field
field::operator = (const field_component& ui)
{
    check_macro(!_V.has_locked(), "Not implemented for spaces with locked components");
    *this = field(ui);
    return *this;
}
field
field::operator = (const const_field_component& ui)
{
    check_macro(!_V.has_locked(), "Not implemented for spaces with locked components");
    *this = field(ui);
    return *this;
}
field::field (const field_component& ui)
: _V(),
   u(),
   b()
{
    check_macro(!_V.has_locked(), "Not implemented for spaces with locked components");
    const field& w   = *ui._px;
    size_type i_comp =  ui._i_comp;
    const space& V = w.get_space();
    space Vi = space(V, i_comp); // V[i_comp]
    *this = w.get_comp(Vi, i_comp);
}
field::field (const const_field_component& ui)
: _V(),
   u(),
   b()
{
    check_macro(!_V.has_locked(), "Not implemented for spaces with locked components");
    const field& w   = *ui._px;
    size_type i_comp =  ui._i_comp;
    const space& V = w.get_space();
    space Vi = space(V, i_comp); // V[i_comp]
    *this = w.get_comp(Vi, i_comp);
}

// =================================================================================
// evaluation
// =================================================================================

Float 
field::operator() (const point& x, size_type i_comp) const
{
    Float v;
    bool is_inside;
    evaluate(x, v, is_inside, i_comp);
    check_macro(is_inside,
	"evaluate: point (" << x << ") is outside the mesh.");
    return v;
}
  
void
field::evaluate (const point& x, Float& result, bool& is_inside, size_type i_comp) const
{
    const geo& Th = get_geo();
    Th.init_localizer (Th.boundary());
    size_type K_idx;
    is_inside=Th.localize (x, K_idx);
    if (!is_inside) { result=numeric_limits<Float>::max(); return; }
    meshpoint hat_x_in_K = Th.hatter (x, K_idx);
    result = evaluate (hat_x_in_K, i_comp);
}
void 
field::evaluate (const point& x, Float& result) const
{
    check_macro (get_valued_type() == fem_helper::scalar,
	"evaluate: scalar field expected, get " << get_valued() << " one.");
    result = operator() (x);
}
void 
field::evaluate (const point& x, point& result) const
{
    bool is_inside;
    evaluate (x, result, is_inside);
    check_macro(is_inside,
	"evaluate: point (" << x << ") is outside the mesh.");
}
void 
field::evaluate (const point& x, point& result, bool& is_inside) const
{
    check_macro (get_valued_type() == fem_helper::vectorial ||
	   (get_valued_type() == fem_helper::scalar && dimension() == 1),
	"evaluate: vector field expected, get " << get_valued() << " one.");
    for (size_type i_comp = 0; i_comp < n_component(); i_comp++)
        { evaluate(x, result [i_comp], is_inside, i_comp); }
    for (size_type i_comp = n_component(); i_comp < 3; i_comp++)
        result [i_comp] = 0;
}
void 
field::evaluate (const point& x, tensor& result) const
{
    bool is_inside;
    evaluate (x, result, is_inside);
    check_macro(is_inside,
	"evaluate: point (" << x << ") is outside the mesh.");
}
void 
field::evaluate (const point& x, tensor& result, bool& is_inside) const
{
    fem_helper::coordinate_type   sys_coord = coordinate_system_type();
    fem_helper::valued_field_type valued    = get_valued_type();
    check_macro (get_valued_type() == fem_helper::tensorial ||
		 get_valued_type() == fem_helper::unsymmetric_tensorial ||
	         (get_valued_type() == fem_helper::scalar && dimension() == 1),
	"evaluate: tensor field expected, get " << get_valued() << " one.");
    result = Float(0);
    for (size_type i_comp = 0; i_comp < n_component(); i_comp++) {
      fem_helper::pair_size_type ij_comp = fem_helper::tensor_subscript (valued, sys_coord, i_comp);
      evaluate(x, result(ij_comp.first, ij_comp.second), is_inside, i_comp); 
      if (ij_comp.first != ij_comp.second)
	result (ij_comp.second, ij_comp.first) = result (ij_comp.first, ij_comp.second);
    }
}

void
field::get_dof_value (const basis& b, const geo_element& K, tiny_vector<Float>& dofv, size_type i_comp) const
{
    size_t nb_dof = b.size (K.variant()) ;
    tiny_vector<size_type> dof  (nb_dof);
    dofv.resize (nb_dof);
    
    // get the global dof numbers for triangle K
    get_space().set_dof (K, dof, i_comp);

    // get f value at these dof
    for (size_type iloc = 0; iloc < nb_dof; iloc++) {
      	dofv[iloc] = at(dof[iloc]);
    }
}
// current field is defined on a domain of a global mesh
// S is a side on the global mesh
void
field::get_dof_value_from_global (const basis& b, const geo_element& S,
	tiny_vector<Float>& dofv, size_type i_comp) const
{
    size_t nb_dof = b.size (S.variant()) ;
    tiny_vector<size_type> msh_dof  (nb_dof);
    tiny_vector<size_type> dmn_dof  (nb_dof);
    dofv.resize (nb_dof);

    // get the mesh dof numbers for element S
    get_space().set_global_dof (S, msh_dof, i_comp);
    for (size_type iloc = 0; iloc < nb_dof; iloc++) {
    	dmn_dof[iloc] = get_space().domain_dof(msh_dof[iloc]);
    }
    // get f value at these dof
    for (size_type iloc = 0; iloc < nb_dof; iloc++) {
      	dofv[iloc] = at(dmn_dof[iloc]);
    }
}
Float
field::evaluate (const meshpoint& S, size_type i_comp) const
{
    const geo_element& K = get_geo().begin()[S.element()];
    const space&       V = get_space();
    const basis&       b = V.get_basis(i_comp);
    piola transformation(V);
    tiny_vector<Float> dofv;
    get_dof_value (b, K, dofv, i_comp);
    return transformation.reference_eval (K, dofv, S.hat(), i_comp);
}
void
field::evaluate (const meshpoint& x, point& result) const
{
    check_macro (get_valued_type() == fem_helper::vectorial ||
	   (get_valued_type() == fem_helper::scalar && dimension() == 1),
	"evaluate: vector field expected, get " << get_valued() << " one.");
    for (size_type i_comp = 0; i_comp < n_component(); i_comp++)
        result [i_comp] = evaluate (x, i_comp);
    for (size_type i_comp = n_component(); i_comp < 3; i_comp++)
        result [i_comp] = 0;
}
Float
field::evaluate_d_dxi (size_type i, const meshpoint& S, size_type i_comp) const
{
    const geo_element& K = get_geo().begin()[S.element()];
    const space&       V = get_space();
    const basis&       b = V.get_basis(i_comp);
    piola transformation(V);
    tiny_vector<Float> dofv;
    get_dof_value (b, K, dofv, i_comp);
    return transformation.reference_d_dxi_eval (i, K, dofv, S.hat(), i_comp);
}
// =================================================================================
// linear algebra (partial)
// =================================================================================
/* 
void
check_same_locked(x,y)
 {
    if (!x.get_space().has_locked()&& !y.get_space().has_locked()) return;		
    if (x.locked_boundaries_set() && !y.locked_boundaries_set()) y.set_locked_boundaries();
    if (!x.locked_boundaries_set() && y.locked_boundaries_set()) x.set_locked_boundaries();
 }
*/
field
sqr (const field& x)
{
    return compose (sqr, x);
}
field
sqrt (const field& x)
{
    field z (x.get_space());
    z.u = sqrt(x.u);
    z.b = sqrt(x.b);
    return z;
}
field
abs (const field& x)
{
    field z (x.get_space());
    z.u = abs(x.u);
    z.b = abs(x.b);
    return z;
}
field
pow (const field& x, Float alpha)
{
    field z (x.get_space());
    z.u = pow(x.u, alpha);
    z.b = pow(x.b, alpha);
    return z;
}
Float
dot (const field& x, const field& y)
{
    return dot(x.u,y.u) + dot (x.b,y.b);
}
field
operator - (const field& x)
{
  field y(x.get_space());
  y.u = -x.u ;
  y.b = -x.b ;
  return y ;
}
field
operator + (const Float& lambda, const field& x)
{
  field y(x.get_space());
  y.u = lambda+x.u ;
  y.b = lambda+x.b ;
  return y ;
}
field
operator + (const field& x, const Float& lambda)
{
  field y(x.get_space());
  y.u = x.u+lambda;
  y.b = x.b+lambda;
  return y ;
}
field
operator - (const Float& lambda, const field& x)
{
  field y(x.get_space());
  y.u = lambda-x.u ;
  y.b = lambda-x.b ;
  return y ;
}
field
operator - (const field& x, const Float& lambda)
{
  field y(x.get_space());
  y.u = x.u-lambda;
  y.b = x.b-lambda;
  return y ;
}
field
operator * (const Float& lambda, const field& x)
{
  field y(x.get_space());
  y.u = lambda*x.u ;
  y.b = lambda*x.b ;
  return y ;
}
field
operator * (const field& x, const Float& lambda)
{
  field y(x.get_space());
  y.u = x.u*lambda;
  y.b = x.b*lambda;
  return y ;
}
field
operator / (const Float& lambda, const field& x)
{
  field y(x.get_space());
  y.u = lambda/x.u ;
  y.b = lambda/x.b ;
  return y ;
}
field
operator / (const field& x, const Float& lambda)
{
  field y(x.get_space());
  y.u = x.u/lambda;
  y.b = x.b/lambda;
  return y ;
}
// TODO: a template function
#define field_linear_operator(OP, NAME)						\
static										\
field										\
field_linear_operator_ ## NAME (const field& x, const field& y)			\
{										\
    typedef field::size_type size_type;						\
    if (x.get_valued_type() == y.get_valued_type()) {				\
        check_macro (x.get_space() == y.get_space(),				\
	    "incompatible spaces for field-field operator");			\
        field z(x.get_space());							\
        z.u = x.u OP y.u;							\
        z.b = x.b OP y.b;							\
        return z;								\
    }										\
    fem_helper::valued_field_type xt = x.get_valued_type();			\
    fem_helper::valued_field_type yt = y.get_valued_type();			\
    check_macro ((xt == fem_helper::tensorial && 				\
		  yt == fem_helper::unsymmetric_tensorial) ||			\
                 (yt == fem_helper::tensorial && 				\
		  xt == fem_helper::unsymmetric_tensorial),			\
	"unexpected tensorial field operands: "					\
	<< x.get_valued() << " and " << y.get_valued());			\
    space Zh;									\
    if (xt == fem_helper::unsymmetric_tensorial)				\
      Zh = x.get_space();							\
    else									\
      Zh = y.get_space();							\
    field z (Zh);								\
    size_type n_comp = z.n_component();						\
    for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {			\
      fem_helper::pair_size_type ij_comp = fem_helper::tensor_subscript (	\
		fem_helper::unsymmetric_tensorial,				\
		z.coordinate_system_type(), i_comp);				\
      size_type i = ij_comp.first;						\
      size_type j = ij_comp.second;						\
      z(i,j) = x(i,j) OP y(i,j);						\
    }										\
    return z;									\
}
field_linear_operator(+, add)
field_linear_operator(-, sub)
#undef field_linear_operator
field 
operator - (const field& x, const field& y)
{
  return field_linear_operator_sub (x,y);
}
field 
operator + (const field& x, const field& y)
{
  return field_linear_operator_add (x,y);
}
field
operator * (const field& x, const field& y)
{
    typedef field::size_type size_type;
    fem_helper::valued_field_type xt = x.get_valued_type();
    fem_helper::valued_field_type yt = y.get_valued_type();
    if (xt == fem_helper::scalar && yt == fem_helper::scalar) {
        check_macro (x.get_space() == y.get_space(),
	    "incompatible spaces for field-field operator");
        field z(x.get_space());
        z.u = x.u * y.u;
        z.b = x.b * y.b;
        return z;
    }
    if (xt == fem_helper::scalar) { // and y is vectorial, tensorial, etc
        check_macro (x.get_geo() == y.get_geo(),
	    "incompatible meshes for scalar_field*multi_field operator");
        check_macro (x.get_approx() == y.get_approx(),
	    "incompatible approx for scalar_field*multi_field operator");
        field z(y.get_space());
	size_type n_comp = y.n_component();
	for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
          z[i_comp] = x * y[i_comp];
        } 
        return z;
    }
    if (yt == fem_helper::scalar) { // and x is vectorial, tensorial, etc
        check_macro (x.get_geo() == y.get_geo(),
	    "incompatible meshes for multi_field*scalar_field operator");
        check_macro (x.get_approx() == y.get_approx(),
	    "incompatible approx for multi_field*scalar_field operator");
        field z(x.get_space());
	size_type n_comp = x.n_component();
	for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
          z[i_comp] = x[i_comp] * y;
        } 
        return z;
    }
    size_type d = x.dimension(); // get physical dimension
    // both x and y vectorial ? then scalar product
    if ((xt == fem_helper::vectorial) && (yt == fem_helper::vectorial)) {
      field z (y.get_space()[0], Float(0));
      for (size_type j = 0; j < d; j++)
          z += x[j] * y[j];
      return z;
    }
    // x vectorial and y tensorial ?
    if ((xt == fem_helper::tensorial || xt == fem_helper::unsymmetric_tensorial) &&
        (yt == fem_helper::vectorial)) {
      field z (y.get_space(), Float(0));
      for (size_type i = 0; i < d; i++)
        for (size_type j = 0; j < d; j++)
          z[i] += x(i,j) * y[j];
      return z;
    }
    // x tensorial and y vectorial ?
    if ((yt == fem_helper::tensorial || yt == fem_helper::unsymmetric_tensorial) &&
        (xt == fem_helper::vectorial)) {
      field z (x.get_space(), Float(0));
      for (size_type i = 0; i < d; i++)
        for (size_type j = 0; j < d; j++)
          z[j] += x[i] * y(i,j);
      return z;
    }
    // both x and y are tensorial ?
    check_macro ((xt == fem_helper::tensorial ||
		  xt == fem_helper::unsymmetric_tensorial) &&
                 (yt == fem_helper::tensorial ||
		  yt == fem_helper::unsymmetric_tensorial),
	"unsupported tensorial field operands for `*': "
	<< x.get_valued() << " and " << y.get_valued());
    space Zh;
    if (xt == fem_helper::unsymmetric_tensorial)
      Zh = x.get_space();
    else
      Zh = y.get_space();
    field z (Zh, Float(0));
    for (size_type i = 0; i < d; i++)
      for (size_type j = 0; j < d; j++)
        for (size_type k = 0; k < d; k++)
          z(i,j) += x(i,k) * y(k,j);

    if (z.coordinate_system_type() == fem_helper::axisymmetric_rz) {
          z(2,2) = x(2,2) * y(2,2);
    }
    return z;
}
field
operator / (const field& x, const field& y)
{
    check_macro (x.get_space() == y.get_space(),
	"incompatible spaces for field/field operator");
    field z (x.get_space());
    if (x.n_component()>1 && y.n_component()==1) 
     {
	for (size_t i=0; i<x.n_component(); i++)
	 {
	    z[i] = x[i]/y;    
	 }
     }
    else
     {
	z.u = x.u / y.u;
	z.b = x.b / y.b;
     }
    return z;
}
// tensor field transposition
field trans (const field& x)
{
    typedef field::size_type size_type;
    fem_helper::valued_field_type xt = x.get_valued_type();
    if (xt != fem_helper::unsymmetric_tensorial)
       return x;
    field y = x;
    size_type n_comp = y.n_component();
    for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
      fem_helper::pair_size_type ij_comp = fem_helper::tensor_subscript (
		xt, y.coordinate_system_type(), i_comp);
      size_type i = ij_comp.first;
      size_type j = ij_comp.second;
      if (i != j) {
	  y(i,j) = x(j,i);
      }
    }
    return y;
}
// tensor field symmetrization on space Sh of symmetric tensors
// where x belongs to an unsymmetric tensor field Th
field
tensor_symmetrize (const space& Sh, const field& x)
{
    typedef field::size_type size_type;
    fem_helper::valued_field_type xt = x.get_valued_type();
    if (xt != fem_helper::unsymmetric_tensorial)
       return x;
    field y (Sh);
    size_type d = x.dimension();
    for (size_type i = 0; i < d; i++)
      for (size_type j = 0; j < d; j++)
	if (i <= j)
	  y(i,j) = x(i,j);
    
    if (x.coordinate_system_type() == fem_helper::axisymmetric_rz) {
          y(2,2) = field(x(2,2));
    }
    return y;
}
field 
field::operator = (Float lambda)
{
    u = lambda;
    b = lambda;
    return *this;
}
field
field::operator *= (Float lambda)
{
    u *= lambda;
    b *= lambda;
    return *this;
}
field
field::operator /= (Float lambda)
{
    u /= lambda;
    b /= lambda;
    return *this;
}
// 
// interpolation: uh on a old mesh
// is reinterpolated in Vh on a new mesh
template <> 
field
interpolate (const space& Vh, field uh)
{
    geomap to_new_mesh (uh.get_geo(), Vh);
    return compose (uh, to_new_mesh);
}
field
fld_cat2(const field& f1, const field& f2)
{
  typedef field::size_type size_type;

  space X = f1.get_space() * f2.get_space();
  field f(X);

  size_type iu1 = f1.n_unknown();
  for (size_type i=0 ; i<iu1 ; i++)
    f.u(i) = f1.u(i);

  size_type iu2 = f2.n_unknown() ;
  for (size_type i=0 ; i<iu2 ; i++ )
    f.u(i+iu1) = f2.u(i) ;

  size_type ib1 = f1.n_blocked() ;
  for (size_type i=0 ; i<ib1 ; i++ )
    f.b(i) = f1.b(i) ;
  
  size_type ib2 = f2.n_blocked() ;
  for (size_type i=0 ; i<ib2 ; i++ )
    f.b(i+ib1) = f2.b(i) ;
  
  return f ;
}

field
fld_cat3(const field& f1, const field& f2, const field& f3)
{
  typedef field::size_type size_type;

  //
  // it is assumed here that the three fields are defined on the same space
  //

  space X1 = f1.get_space() ;
  space X  = X1*X1*X1 ;
  field f(X) ;

  size_type iu = f1.n_unknown() ;
  for (size_type i=0 ; i<iu ; i++ )
    {
      f.u(i)      = f1.u(i) ;
      f.u(i+iu)   = f2.u(i) ;
      f.u(i+2*iu) = f3.u(i) ;
    } 
     
  size_type ib = f1.n_blocked() ;
  for (size_type i=0 ; i<ib ; i++ )
    {
      f.b(i)      = f1.b(i) ;
      f.b(i+ib)   = f2.b(i) ;
      f.b(i+2*ib) = f3.b(i) ;
    }

  return f ;

}
// catch "\nfield" or "\nmfield"
// return: "field", "mfield", or "" when failed
static
string 
get_field_mark (istream& in)
{
    unsigned int optional_position = 1; // optional 'm'
    string ch = "\nmfield";
    unsigned int l = ch.length();

    // begining of stream <==> begining of a line, e.g. 
    //   we look at "\nfield" or "\nmfield" while file starts 
    //   with string "field" or "mfield"; it's ok
    unsigned int state = 0;

    char c = '\0';
    in.get(c);
    bool has_skip_optional = false;
    const char *p = ch.c_str();
    if (*p == '\n') {
        // begining of stream <==> begining of a line, e.g.
        //   we look at "\nmfield" while file starts
        //   with string "mfield"; it's ok
        state++;
        p++;
    }
    do {
        if (*p == c) {
            // advance in the string
            state++;
            p++;
        } else if (state == optional_position && ch[optional_position+1] == c) {
	    state += 2;
            p     += 2;
	    has_skip_optional = true;
        } else if (state != 0 && ch[0] == c) {
            // come back to the second position
            state = 1;
            p     = ch.c_str() + 1;
	    has_skip_optional = false;
        } else if (state != 0) {
            // come back to the begining of the string
            state = 0;
            p     = ch.c_str();
	    has_skip_optional = false;
        }
    }
    while (state < l && in.get(c) && in.good());

    if (state == l && in.good()) {
	if (has_skip_optional) {
	    return "field";
	} else {
	    return "mfield";
	}
    } else {
	    return "";
    }
}
//
// load file data
//
istream&
load_field_data (istream& s, field& x)
{
  typedef field::size_type size_type;

  const space& V = x.get_space();
  vec<Float>::iterator xu = x.u.begin();
  vec<Float>::iterator xb = x.b.begin();
  size_type ndof = x.size();

  for (size_type i = 0; i < ndof; i++) {
        Float value;
        s >> value;
	size_type idx = V.index(i);
	if (V.is_blocked(i)) {
	         xb [idx] = value;
	} else {
	         xu [idx] = value;
	}
  }
  return s;
}
//
// text input/output without specifield space
// => build also the space
//
istream&
load_scalar_field (istream& s, field& x)
{
  typedef field::size_type size_type;

  int version;
  size_type ndof;
  string geo_name;
  string app_name;
  s >> version >> ndof
    >> geo_name
    >> app_name;
  check_macro (version == 1, "field format version " << version << " not defined");
  bool fastfieldload = iorheo::getfastfieldload(s);
  geo g;
  bool have_new_geo = false;
  if (!fastfieldload || x.size() == 0 || x.get_geo().name() != geo_name) {
      g = geo(geo_name);
      have_new_geo = true;
  } else {
      // reuse previously loaded mesh
      g = x.get_geo();
  }
  space V;
  bool have_new_space = false;
  if (have_new_geo || x.get_approx() != app_name) {
      V = space (g, app_name);
      have_new_space = true;
  } else {
      // reuse previously builded space
      V = x.get_space();
  }
  V.freeze();
  if (V.size() != ndof) {
      error_macro ("geo `" << geo_name << "' size is incompatible with (old?) field data size.");
  }
  x._V = V;
  x.u.resize(ndof);
  x.b.resize(0);
  return load_field_data (s, x);
}
istream&
load_multi_field (istream& s, field& x)
{
  typedef field::size_type size_type;
  bool fastfieldload = iorheo::getfastfieldload(s);
  // 
  // load values
  //
  int version;
  size_type n_comp;
  s >> version >> n_comp;
  check_macro (version == 1, "multi-field format version " << version << " not defined");
  vector<pair<string,string> > approx(n_comp); // (geo_name, approx_name)
  vector<vector<Float> >       value(n_comp);
  for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
      check_macro(scatch(s,"\nfield"), "input stream does not contains a field.");
      size_type ndof_i, version_i;
      s >> version_i >> ndof_i
        >> approx[i_comp].first
        >> approx[i_comp].second;
      check_macro (version == 1, "field format version " << version << " not defined");
      check_macro (approx[i_comp].first == approx[0].first, 
	"geo #" << i_comp << " differs: not yet supported");
      value[i_comp].resize(ndof_i);

#ifdef _RHEOLEF_HAVE_A_BUG_IN_STL
      // why  istream_iterator<Float>(s) does not works ??
      copy_n (istream_iterator<Float>(s), ndof_i, value[i_comp].begin());
#else
      for (size_type j = 0; j < ndof_i; j++) {
	  s >> value[i_comp][j];
      }
#endif // _RHEOLEF_HAVE_A_BUG_IN_STL
  }
  // 
  // determines if space may be buided
  //
  bool may_rebuild = false;
  if (!fastfieldload || x._V.size() == 0 || x._V.n_component() != n_comp) {
      may_rebuild = true;
  } else {
      for (size_type i_comp = 0; !may_rebuild && i_comp < n_comp; i_comp++) {
	if (x._V.get_approx(i_comp) != approx[i_comp].second ||
	    x._V.get_geo().name()    != approx[0].first) {
		may_rebuild = true;
	}
      }
  }
  //
  // build space
  //
  if (may_rebuild) {
      space V;
      geo g;
      for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
          // different geos not yet supported
          if (i_comp == 0) g = geo(approx[0].first);
	  space Vi = space (g, approx[i_comp].second);
          if (i_comp == 0) V = Vi;
	  else             V = V*Vi;
      }
      x = field(V);
  }
  //
  // store values
  //
  size_type dof = 0;
  for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
      for (vector<Float>::const_iterator p = value[i_comp].begin(); p != value[i_comp].end(); p++) {
	x.at(dof++) = *p;
      }
  }
  return s;
}
istream& 
operator >> (istream& s, field& x)
{
  iorheo::flag_type fmt = iorheo::flags(s) & iorheo::format_field;
  if      (fmt [iorheo::bamg])     { return x.get_bamg_metric (s); }
  if      (fmt [iorheo::mmg3d])    { return x.get_mmg3d_metric (s); }
  else if (fmt [iorheo::cemagref]) { return x.get_cemagref (s); }
  check_macro(fmt [iorheo::rheo], "Unknown input format !") 
  // else continue

  // rheolef input file format:
  string fname = iorheo::getmark(s);
  string mark = get_field_mark(s);
  if (mark == "field") {
      load_scalar_field (s, x);
  } else if (mark == "mfield") {
      // vector-valued input, in multi-field format
      load_multi_field (s, x);
  } else {
      error_macro("field: input stream does not contains a field.");
  }
  // prevent next field read
  iorheo::setmark(s,"");
  return s;
}
//
// cstor from file name
//
field::field(const string& file_name) : _locked_boundaries_set(false)
{
  irheostream is(file_name, "field");
  if (!is) error_macro("\"" << file_name << "[.field[.gz]]\" not found.");
  is >> *this; // load data
}
//
// text input/output with specified space
//
static
void
load_field_data_with_specified_space(
    istream&     s, 
    const space& V,
    field&       x)
{
  typedef field::size_type size_type;

  V.freeze();
  
  bool verbose = iorheo::getverbose(s);
  if (!scatch(s,"\nfield"))
    error_macro("input stream does not contains a field.");
  int version;
  size_type ndof;
  s >> version >> ndof;
  char geo_name [1024];
  s >> geo_name;
  if (V.get_geo().name()!=geo_name) {
    error_macro("field: input stream has incompatible geo ("
         << "\"" << geo_name << "\", \"" << V.get_geo().name() << "\").");
  }
  char app_name [1024];
  s >> app_name;
  //
  // ATTENTION : ce test ne marche pas : 
  //
  // if (strcmp(V.get_approx(), app_name)!=0)
  //   error_macro("field: input stream has incompatible approx ("
  //         << "\"" << app_name << "\", \"" << V.get_approx() << "\").");
  //
  if (V.size() != ndof)
    error_macro("field: geo `" << geo_name << "' has incompatible size.");

  load_field_data (s, x);
}
field::field (const space& V, istream& s)
: _V(V), _locked_boundaries_set(false), u(V.n_unknown()), b(V.n_blocked())
{
    load_field_data_with_specified_space(s, V, *this);
}
field::field(const space& V, const string& file_name)
: _V(V), _locked_boundaries_set(false), u(V.n_unknown()), b(V.n_blocked())
{
  irheostream is(file_name, "field");
  if (!is) error_macro("\"" << file_name << "[.field[.gz]]\" not found.");
  load_field_data_with_specified_space(is, V, *this);
}

//
// allocator : from a different space of same mesh and approximation,
//   only blocked domains differ
//
field::field(const space& V, const field& w)
: _V(V), _locked_boundaries_set(false)
{
  V.freeze();
  u.resize(V.n_unknown());
  b.resize(V.n_blocked());
  const space& W=w.get_space();
  check_macro(!W.has_locked(), "Not implemented for spaces with locked components");
  const geo&  g = V.get_geo() ;  
  const basis& b_V = V.get_basis() ;  
  check_macro (V.n_component() == 1, "expect scalar space, get "
	<< V.n_component() << "D vector-valued space.");
  check_macro (W.n_component() == 1, "expect scalar space, get "
	<< W.n_component() << "D vector-valued space.");
  check_macro (g.familyname()==W.get_geo().familyname() && g.version()==W.get_geo().version(),
  	"Spaces based on different mesh. Use interpolate to transfer between these meshes.");
  check_macro (get_approx()==W.get_approx(),
  	"Spaces using different approximations. Use a projection to transfer between these spaces.");
  std::vector<bool> marked (V.size(), false);
  vec<Float>::iterator xu = u.begin();
  vec<Float>::iterator xb = b.begin();
  for (geo::const_iterator iter_K = g.begin(); iter_K != g.end(); iter_K++) {
      const geo_element& K = *iter_K;
      size_type nb_dof = b_V.size(K.variant()) ;
      tiny_vector<space::size_type> dof_V(nb_dof), dof_W(nb_dof);
      V.set_dof(K, dof_V, 0) ;
      W.set_dof(K, dof_W, 0) ;
      for(size_type i = 0; i < nb_dof; i++) {
	  size_type i_dof = dof_V[i];
	  if (marked [i_dof]) continue; else marked [i_dof] = true;
	  Float value = w.at(dof_W[i]);
	  size_type idx = V.index(i_dof);
	  if (V.is_blocked(i_dof))
	    xb [idx] = value;
	  else
	    xu [idx] = value;
      }
  }
} 	

//
// outputs
//
ostream& 
field::put_scalar (ostream& s) const
{
    s << "\nfield\n";
    s << "1 " << size() << endl;
    s << get_geo().name() << endl;
    s << get_approx() << endl;
    s << endl;
    for (size_type i = 0; i < size() && s.good(); i++) {
	Float value = at(i);

#ifndef _RHEOLEF_HAVE_DOUBLEDOUBLE
	// TODO: use default n = numeric_limits<Float>::digit10
	// and: stream << setrheoprecision(n) << field_u;
	//  => stream.precision(min(n,numeric_limits<Float>::digit10))
	s << value << endl;
#else
	// avoid variable format in non-regression test...
	// doubledouble does not head stream.precision(n) and such
	s << double(value) << endl;
#endif // _RHEOLEF_HAVE_DOUBLEDOUBLE
    
    }
    s << endl;
    return s;
}
ostream&
put_component (ostream& os, const field& u, field::size_type i_comp)
{
    check_macro (!u.locked_boundaries_set(), 
    	"Components are still locked! Use field::unset_locked_boundaries() first.");
    typedef field::size_type size_type;
    const space& V = u.get_space();
    size_type first = V.start(i_comp);
    size_type last  = V.start(i_comp+1);
    os << "field\n";
    os << "1 " << last-first << endl;
    os << u.get_geo().name() << endl;
    os << u.get_approx(i_comp) << endl;
    os << endl;
    for (size_type i = first; i < last; i++) {
	os << u.at(i) << endl;
    }
    os << endl;
    return os;
}
ostream& 
field::put_multi (ostream& os) const
{
    os << "mfield"              << endl
       << "1 " << n_component() << endl 
       << endl;
    string fname = iorheo::getmark(os);
    size_type dim = dimension();
    string valued = get_valued();
    string sys_coord = coordinate_system();
    for (size_type i_comp = 0; i_comp < n_component() && os.good(); i_comp++) {
        if (fname != "") {
	  if (valued == "tensor" || valued == "unsymmetric_tensor") {
            os << "#" << fname << fem_helper::tensor_subscript_name (valued,sys_coord,i_comp) << endl;
	  } else {
            os << "#" << fname << i_comp << endl;
          }
        }
	put_component (os, *this, i_comp);
    }
    return os;
}
ostream& 
field::put (ostream& os) const
{
    string fname = iorheo::getmark(os);
    if (n_component() <= 1) {
        if (fname == "") { os << "#!field\n"; }
        put_scalar(os);
    } else {
        if (fname == "") { os << "#!mfield\n"; }
        put_multi(os);
    }
    // prevent mark propagation for next output
    iorheo::setmark(os,"");
    return os;
}
ostream&
operator << (ostream& os, const const_field_component& ui)
{
  return os << field(ui);
}
ostream&
operator << (std::ostream& os, const field_component& ui)
{
  return os << field(ui);
}
//
// set boundary values
//  u[domain]    := y;
//  u[domain][i] := y;
//
void 
field::boundary_val( const field&  y, 
		     size_type     i_comp,
		     const domain& d)
{
  check_macro (y.n_component() == 1, "invalid rhs: multi-component field");
  const space&  X  = get_space();
  const space&  Y  = y.get_space();
  check_macro (X.get_approx(i_comp) == Y.get_approx(), "incompatible approximation: lhs[" 
	<< i_comp << "](" << X.get_approx(i_comp) << ") and rhs(" 
	<< Y.get_approx() << ")");
  vec<Float>::iterator xu = u.begin();
  vec<Float>::iterator xb = b.begin();
  vec<Float>::const_iterator yu = y.u.begin();
  vec<Float>::const_iterator yb = y.b.begin();

  tiny_vector<size_type> idx;
  tiny_vector<size_type> idy;
  for (domain::const_iterator p_side = d.begin(); p_side != d.end(); p_side++) {
      const geo_element& S = *p_side;
      X.set_global_dof (S, idx, i_comp);
      Y.set_global_dof (S, idy);
      for (size_type i = 0 ; i < idx.size() ; i++) {
	  size_type ix_dof = X.domain_dof (idx(i));
	  size_type iy_dof = Y.domain_dof (idy(i));
          size_type ix  = X.index(ix_dof);
          size_type iy  = Y.index(iy_dof);

	  Float value;
	  if (Y.is_blocked(iy_dof))
	    value = yb [iy];
	  else
	    value = yu [iy];

	  if (X.is_blocked(ix_dof))
	    xb [ix] = value;
	  else
	    xu [ix] = value;
      }
  }
}
void 
field::boundary_val( Float         lambda,
		     size_type     i_comp,
		     const domain& d)
{
  const space&  X  = get_space();
  vec<Float>::iterator xu = u.begin();
  vec<Float>::iterator xb = b.begin();
  tiny_vector<size_type> idx;
  for (domain::const_iterator iter_K = d.begin(); iter_K != d.end(); iter_K++) {
      const geo_element& K = *iter_K;
      X.set_global_dof (K, idx, i_comp);
      for (size_type i = 0 ; i < idx.size() ; i++) {
	  size_type ix_dof = X.domain_dof (idx(i));
          size_type ix     = X.index(ix_dof);

	  if (X.is_blocked(ix_dof)) {
	    assert_macro (ix < b.size(), "out of range");
	    xb [ix] = lambda;
	  } else {
	    assert_macro (ix < u.size(), "out of range");
	    xu [ix] = lambda;
	  }
      }
  }
}
void 
field::boundary_val( Float        (*f)(const point& x),
		     const space&  V0           , 
		     size_type  i_comp      ,
		     const string& dom_name     )
  //
  // *** A supprimer...: utiliser
  ///  u[domain]    := interpolate(V,f);
  ///  u[domain][i] := interpolate(V,f);
  //
{
  field y = interpolate(V0, f);
  field::boundary_val(y, i_comp, dom_name) ;
}
field
field::get_comp(const space& Vi, size_type i_comp) const
{
  // assume: _V[i_comp] == Vi
  // Vi est l'espace sur lequel la composante est construite, 
  // Vi est en parametre pour eviter de le construire a` chaque fois
  size_type first = _V.start(i_comp); 
  size_type last  = _V.start(i_comp+1);
  if (last-first != Vi.size()) {
      error_macro ("get_comp: incompatible spaces: component "<<i_comp<<" has size "<< last-first
	<< "while destination space has size=" << Vi.size());
  }
  field ui(Vi);
  vec<Float>::iterator ui_u = ui.u.begin();
  vec<Float>::iterator ui_b = ui.b.begin();
  for (size_type p = first; p < last; p++) {
      size_type dof   = p-first;
      size_type num   = Vi.index(dof);
      Float value     = at(p);
      if (Vi.is_blocked(dof)) {
	 ui_b [num] = value;
      } else {
	 ui_u [num] = value;
      }
  }
  return ui;
}
//
// set from boundary values
//  u = y[domain]
//  u = y[domain][i]
//
void 
field::from_boundary_val( const field&  y, 
		     size_type     i_comp,
		     const domain& d)
{
  check_macro (n_component() == 1, "invalid lhs: multi-component field");
  const space&  X  = get_space();
  const space&  Y  = y.get_space();
  check_macro (X.get_approx() == Y.get_approx(i_comp), "incompatible approximation: rhs[" 
	<< i_comp << "](" << Y.get_approx(i_comp) << ") and lhs(" 
	<< X.get_approx() << ")");
  vec<Float>::iterator xu = u.begin();
  vec<Float>::iterator xb = b.begin();
  vec<Float>::const_iterator yu = y.u.begin();
  vec<Float>::const_iterator yb = y.b.begin();

  tiny_vector<size_type> idx;
  tiny_vector<size_type> idy;
  for (domain::const_iterator p_side = d.begin(); p_side != d.end(); p_side++) {
      const geo_element& S = *p_side;
      X.set_global_dof (S, idx);
      Y.set_global_dof (S, idy, i_comp);
      for (size_type i = 0 ; i < idx.size() ; i++) {
	  size_type ix_dof = X.domain_dof (idx(i));
	  size_type iy_dof = Y.domain_dof (idy(i));
          size_type ix  = X.index(ix_dof);
          size_type iy  = Y.index(iy_dof);

	  Float value;
	  if (Y.is_blocked(iy_dof))
	    value = yb [iy];
	  else
	    value = yu [iy];

	  if (X.is_blocked(ix_dof))
	    xb [ix] = value;
	  else
	    xu [ix] = value;
      }
  }
}
void
field::check() const
{
    warning_macro ("checking field...");
    _V.check();
    warning_macro ("checking field done.");
}
vec<Float>::const_iterator 
field_component::u_begin() const
{
    const space& X = (*_px).get_space();
    return (*_px).u.begin() + X.start_unknown(_i_comp);
}
vec<Float>::const_iterator 
field_component::b_begin() const
{
    const space& X = (*_px).get_space();
    return (*_px).b.begin() + X.start_blocked(_i_comp);
}
vec<Float>::const_iterator 
field_component::u_end() const
{
    const space& X = (*_px).get_space();
    return (*_px).u.begin() + X.start_unknown(_i_comp+1);
}
vec<Float>::const_iterator 
field_component::b_end() const
{
    const space& X = (*_px).get_space();
    return (*_px).b.begin() + X.start_blocked(_i_comp+1);
}
vec<Float>::iterator 
field_component::u_begin()
{
    const space& X = (*_px).get_space();
    return (*_px).u.begin() + X.start_unknown(_i_comp);
}
vec<Float>::iterator 
field_component::b_begin()
{
    const space& X = (*_px).get_space();
    return (*_px).b.begin() + X.start_blocked(_i_comp);
}
vec<Float>::iterator 
field_component::u_end()
{
    const space& X = (*_px).get_space();
    return (*_px).u.begin() + X.start_unknown(_i_comp+1);
}
vec<Float>::iterator 
field_component::b_end()
{
    const space& X = (*_px).get_space();
    return (*_px).b.begin() + X.start_blocked(_i_comp+1);
}
field_component::size_type
field_component::u_size() const
{
    const space& X = (*_px).get_space();
    return X.n_unknown_component(_i_comp);
}
field_component::size_type
field_component::b_size() const
{
    const space& X = (*_px).get_space();
    return X.n_blocked_component(_i_comp);
}
vec<Float>::const_iterator 
const_field_component::u_begin() const
{
    const space& X = (*_px).get_space();
    return (*_px).u.begin() + X.start_unknown(_i_comp);
}
vec<Float>::const_iterator 
const_field_component::b_begin() const
{
    const space& X = (*_px).get_space();
    return (*_px).b.begin() + X.start_blocked(_i_comp);
}
vec<Float>::const_iterator 
const_field_component::u_end() const
{
    const space& X = (*_px).get_space();
    return (*_px).u.begin() + X.start_unknown(_i_comp+1);
}
vec<Float>::const_iterator 
const_field_component::b_end() const
{
    const space& X = (*_px).get_space();
    return (*_px).b.begin() + X.start_blocked(_i_comp+1);
}
const_field_component::size_type
const_field_component::u_size() const
{
    const space& X = (*_px).get_space();
    return X.n_unknown_component(_i_comp);
}
const_field_component::size_type
const_field_component::b_size() const
{
    const space& X = (*_px).get_space();
    return X.n_blocked_component(_i_comp);
}
// x["boundary"] = 1;
const Float&
field_on_domain::operator = (const Float& lambda)
{
    if (_i_comp != numeric_limits<size_type>::max()) {
            (*_px).boundary_val (lambda, _i_comp, _d);
    } else {
        for (size_type i = 0; i < (*_px).n_component(); i++) {
            (*_px).boundary_val (lambda, i, _d);
	}
    }
    return lambda;
}
// x["boundary"] = y;
const field&
field_on_domain::operator = (const field& y)
{
    if (_i_comp != numeric_limits<size_type>::max()) {
         (*_px).boundary_val (y, _i_comp, _d);
         return y;
    }
    size_type n_comp = (*_px).n_component();
    if (y.n_component() == 1) {
        // x[i]["boundary"] = y,  for i=1..n_comp
        for (size_type i = 0; i < n_comp; i++) {
            (*_px).boundary_val (y, i, _d);
        }
        return y;
    }
    check_macro (y.n_component() == n_comp, "incompatible multi-component field affectation on domain: expect " << n_comp << ", get " << y.n_component());
    // x[i]["boundary"] = y[i],  for i=1..n_comp
    for (size_type i = 0; i < n_comp; i++) {
        (*_px).boundary_val (y[i], i, _d);
    }
    return y;
}
// x[i] = 1;
const Float& 
field_component::operator = (const Float& lambda)
{
  field& x = *_px;
  const space& X = x.get_space();
  size_type first = X.start(_i_comp); 
  size_type last  = X.start(_i_comp+1);
  vec<Float>::iterator xi_u = x.u.begin();
  vec<Float>::iterator xi_b = x.b.begin();
  for (size_type dof = first; dof < last; dof++) {
      size_type num   = X.index(dof);
      if (X.is_blocked(dof)) {
	 xi_b [num] = lambda;
      } else {
	 xi_u [num] = lambda;
      }
  }
  return lambda;
}
// x[i] = y;
const field& 
field_component::operator = (const field& y)
{
  field& x = *_px;
  const space& X = x.get_space();
  size_type first = X.start(_i_comp); 
  size_type last  = X.start(_i_comp+1);
  if (last-first != y.size()) {
      error_macro ("field_component::operator=(const field&): component #"
	<<_i_comp<<" size(" << last-first << ") and rhs field size(" << y.size()
	<< ") have incompatible sizes");
  }
  vec<Float>::iterator xi_u = x.u.begin();
  vec<Float>::iterator xi_b = x.b.begin();
  for (size_type dof = first; dof < last; dof++) {
      Float value = y.at(dof-first);
      size_type num   = X.index(dof);
      if (X.is_blocked(dof)) {
	 xi_b [num] = value;
      } else {
	 xi_u [num] = value;
      }
  }
  return y;
}
field
field_component::operator = (const field_component& y)
{
    field tmp = y;
    *this = tmp;
    return tmp;
}

// x[i] += y;
field_component&
field_component::operator += (const field& y)
{
  field& x = *_px;
  const space& X = x.get_space();
  size_type first = X.start(_i_comp); 
  size_type last  = X.start(_i_comp+1);
  if (last-first != y.size()) {
      error_macro ("field_component::operator+=(const field&): component #"
	<<_i_comp<<" and rhs field have incompatible sizes");
  }
  vec<Float>::iterator xi_u = x.u.begin();
  vec<Float>::iterator xi_b = x.b.begin();
  for (size_type dof = first; dof < last; dof++) {
      Float value = y.at(dof-first);
      size_type num   = X.index(dof);
      if (X.is_blocked(dof)) {
	 xi_b [num] += value;
      } else {
	 xi_u [num] += value;
      }
  }
  return *this;
}
// sigma(i,j)
const_field_component 
field::operator () (size_type i_comp, size_type j_comp) const
{
    check_macro (!_locked_boundaries_set, "Components are still locked! Use field::unset_locked_boundaries() first.");
    size_type k_comp = fem_helper::tensor_index(get_valued(), coordinate_system(), i_comp, j_comp);
    return const_field_component (*this, k_comp);
}
field_component 
field::operator () (size_type i_comp, size_type j_comp)
{
    check_macro (!_locked_boundaries_set, "Components are still locked! Use field::unset_locked_boundaries() first.");
    size_type k_comp = fem_helper::tensor_index(get_valued(), coordinate_system(), i_comp, j_comp);
    return field_component (*this, k_comp);
}
// A(i,j,k,l)
const_field_component
field::operator () (size_type i, size_type j, size_type k, size_type l) const
{
    check_macro (!_locked_boundaries_set, "Components are still locked! Use field::unset_locked_boundaries() first.");
    size_type ijlk_comp = fem_helper::tensor4_index(get_valued(), coordinate_system(), i, j, k, l);
    return const_field_component (*this, ijlk_comp);
}
field_component
field::operator () (size_type i, size_type j, size_type k, size_type l)
{
    check_macro (!_locked_boundaries_set, "Components are still locked! Use field::unset_locked_boundaries() first.");
    size_type ijlk_comp = fem_helper::tensor4_index(get_valued(), coordinate_system(), i, j, k, l);
    return field_component (*this, ijlk_comp);
}
// z = x[i]*y[j];
// TODO: a template function
#define field_component_operator(FieldCompType, OP)				\
field										\
operator OP (const FieldCompType& xi, const FieldCompType& yj)			\
{										\
  typedef field::size_type size_type;						\
  const field& x = *(xi._px);							\
  const field& y = *(yj._px);							\
  const space& X = x.get_space();						\
  const space& Y = y.get_space();						\
  size_type i = xi._i_comp;							\
  size_type j = yj._i_comp;							\
  size_type first_x = X.start(i); 						\
  size_type last_x  = X.start(i+1);						\
  size_type first_y = Y.start(j); 						\
  size_type last_y  = Y.start(j+1);						\
  if (last_x-first_x != last_y-first_y) {					\
      error_macro ("field component operands have incompatible sizes");		\
  }										\
  space Z = space(X[i]);							\
  field z(Z);									\
  for (size_type dof = 0; dof < last_x-first_x; dof++) {			\
      z.at(dof) = x.at(first_x+dof) OP y.at(first_y+dof);			\
  }										\
  return z;									\
}
field_component_operator(const_field_component, +)
field_component_operator(const_field_component, -)
field_component_operator(const_field_component, *)
field_component_operator(const_field_component, /)
field_component_operator(field_component, +)
field_component_operator(field_component, -)
field_component_operator(field_component, *)
field_component_operator(field_component, /)
#undef field_component_operator

// ---------------------------------------------------------------------
// general function transform: y := f(x[i]) <==> y := transform(x[i],f)
// ---------------------------------------------------------------------
void
transform (const const_field_component& x, Float (*f)(const Float&), field& y)
{
    transform(x.u_begin(), x.u_end(), y.u.begin(), f);
    transform(x.b_begin(), x.b_end(), y.b.begin(), f);
}
field
compose (Float (*f)(const Float&), const const_field_component& x)
{
    const space& X = (*x._px).get_space();
    space Y = space(X[x._i_comp]); // TODO: store Y space in X as basic scalar space
    field y (Y);
    transform (x, f, y);
    return y;
}
field
transform (const const_field_component& x, Float (*f)(const Float&))
{
    warning_macro("vh = transform(uh[i],f) is obsolete: please, use vh = compose(f,uh[i]) instead.");
    return compose (f, x);
}
field
sqr (const const_field_component& xi)
{
    return compose(sqr, xi);
}
#ifdef TODO
field
sqrt (const const_field_component& xi)
{
    return compose(sqrt, xi);
}
field
abs (const const_field_component& xi)
{
    return compose(abs, xi);
}
#endif // TODO
// -----------------------------------------------------
// the same with field_component
// -----------------------------------------------------
void
transform (const field_component& x, Float (*f)(const Float&), field& y)
{
    transform(x.u_begin(), x.u_end(), y.u.begin(), f);
    transform(x.b_begin(), x.b_end(), y.b.begin(), f);
}
field
compose (Float (*f)(const Float&), const field_component& x)
{
    const space& X = (*x._px).get_space();
    space Y = space(X[x._i_comp]); // TODO: store Y space in X as basic scalar space
    field y (Y);
    transform (x, f, y);
    return y;
}
field
transform (const field_component& x, Float (*f)(const Float&))
{
    warning_macro("vh = transform(uh[i],f) is obsolete: please, use vh = compose(f,uh[i]) instead.");
    return compose(f,x);
}
field
sqr (const field_component& xi)
{
    return compose(sqr, xi);
}
#ifdef TODO
field
sqrt (const field_component& xi)
{
    return compose(sqrt, xi);
}
field
abs (const field_component& xi)
{
    return compose(abs, xi);
}
#endif // TODO
// -----------------------------------------------------
void
field::save (string fname, string username) const
{
    if (fname.length() == 0) {
        fname = get_geo().name();
    }
    if (n_component() == 1) {
        orheostream os (fname, "field");
	os << *this;
    } else {
        orheostream os (fname, "mfield");
	if (username.length()==0) 
		os << catchmark(fname) << *this;
	else
		os << catchmark(username) << *this;
    }
}
void 
field::set_locked_boundaries ()
{
	// upon init, the fields with locked boundaries have 'wrong' values on the
	// locked boundary:
	// namely, f_// has been attributed the value of f_0 (in b[i]) 
	// and f_orth the one of f_1 (in u[locked_with(i)])
	// this calculates the actual f_// and f_ortho
    if (!_V.has_locked() || _locked_boundaries_set) return;
    for (size_type i0=0; i0<_V.size_component(0); i0++)
     	if (_V.is_locked(i0))
     	{   
	    size_type i1=_V.locked_with(i0);
	    size_type idx0=_V.index(i0);
	    size_type idx1=_V.index(i1);
	    Float u0;
	    // The component 0 can be either blocked or not, while the component 1 is always blocked
	    if (_V.is_blocked(i0)) u0=b.at(idx0); else u0=u.at(idx0); 
	    Float f_ortho 
	    	=u0*_V.unlocked_component(i0,0)
		+b.at(idx1)*_V.unlocked_component(i0,1);  // f_ortho
	    b.at(idx1)=u0*_V.locked_component(i0,0)
		+b.at(idx1)*_V.locked_component(i0,1); // f_//
	    if (_V.is_blocked(i0)) b.at(idx0)=f_ortho; else u.at(idx0)=f_ortho; 
     	}
    _locked_boundaries_set=true;
}
void 
field::unset_locked_boundaries ()
{
	// projects back to cartesian/cylindric components
    if (!_V.has_locked() || !_locked_boundaries_set) return;
    for (size_type i0=0; i0<_V.size_component(0); i0++)
     	if (_V.is_locked(i0))
     	{   
	    size_type i1=_V.locked_with(i0);
	    size_type idx0=_V.index(i0);
	    size_type idx1=_V.index(i1);
	    Float u_ortho;
	    // The component 0 (ortho) can be either blocked or not, while the component 1 (normal) is always blocked
	    if (_V.is_blocked(i0)) u_ortho=b.at(idx0); else u_ortho=u.at(idx0); 
	    Float f0 
	    	=u_ortho*_V.unlocked_component(i0,0)
		+b.at(idx1)*_V.locked_component(i0,0);  // f_ortho
	    b.at(idx1)=u_ortho*_V.unlocked_component(i0,1)
		+b.at(idx1)*_V.locked_component(i0,1); // f_//
	    if (_V.is_blocked(i0)) b.at(idx0)=f0; else u.at(idx0)=f0; 
     	}
    _locked_boundaries_set=false;
}

field
field::piecewise_to_discontinuous() const
{
    check_macro(get_approx()=="P1pw", 
    	"Not a supported piecewise continuous approximation " << get_approx());
    const geo& g=get_geo();
    space Vd (g, "P1d");
    if (_V.n_component()>0)
      for (size_type i_comp=1; i_comp<_V.n_component(); i_comp++) 
      	Vd = Vd * space(g, "P1d");
    // ! the sequel assumes Vd has no blocked boundaries, thus Vd.index[i]=i.
    field fd (Vd);
    geo::const_iterator K=g.begin();
    geo::const_iterator eK=g.end();
    tiny_vector<size_type> idx_from;
    tiny_vector<size_type> idx_to;
    for (; K!=eK; K++) 
      for (size_type i_comp=0; i_comp<_V.n_component(); i_comp++) {
    	_V.set_dof(*K,idx_from,i_comp);
    	Vd.set_dof(*K,idx_to,i_comp);
    	for (size_type i=0; i<(*K).size(); i++)
	    fd.u.at(idx_to[i])=at(idx_from[i]);
      }
    return fd;
}

field 
euclidian_norm2 (const field& x)
{
    field nx(x.get_space()[0], Float(0));
    for (field::size_type i=0; i<x.n_component(); i++) {
    	nx += sqr(x[i]);
    }
    return nx;
}
//! for a tensor field, returns field: sqrt(tau(0,0)^2+2*tau(0,1)^2+..)
field tensor_norm (const field& tau)
{ 
    check_macro (tau.get_valued_type() == fem_helper::tensorial ||
		 tau.get_valued_type() == fem_helper::unsymmetric_tensorial,
	"evaluate: tensor field expected, get " << tau.get_valued() << " one.");
    if (tau.get_valued_type() == fem_helper::tensorial) {
      if (tau.coordinate_system_type() == fem_helper::axisymmetric_rz ||
          tau.coordinate_system_type() == fem_helper::axisymmetric_rz) {
        return sqrt(sqr(tau(0,0)) + sqr(tau(1,1)) + sqr(tau(2,2))
               +2*sqr(tau(0,1)));
      }
      switch (tau.dimension()) {
      case 1 : return abs(tau);
      case 2 : return sqrt(sqr(tau(0,0)) + sqr(tau(1,1)) + 2*sqr(tau(0,1)));
      default : return sqrt(sqr(tau(0,0)) + sqr(tau(1,1)) + sqr(tau(2,2))
                 + 2*(sqr(tau(0,1)) + sqr(tau(0,2)) + sqr(tau(1,2))));
      }
    }
    if (tau.get_valued_type() == fem_helper::unsymmetric_tensorial) {
      switch (tau.dimension()) {
      case 1 : return abs(tau);
      case 2 : return sqrt(sqr(tau(0,0)) + sqr(tau(0,1))
                         + sqr(tau(1,0)) + sqr(tau(1,1)));
      default: return sqrt(sqr(tau(0,0)) + sqr(tau(0,1)) + sqr(tau(0,2))
                         + sqr(tau(1,0)) + sqr(tau(1,1)) + sqr(tau(1,2))
                         + sqr(tau(2,0)) + sqr(tau(2,1)) + sqr(tau(2,2)));
      }
    }
    error_macro ("not yet supported: tensor_norm (" << tau.get_valued() << ")");
}
}// namespace rheolef
