///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef.h"
#include "rheolef/uzawa_abtb.h"
using namespace rheolef;
using namespace std;
class poiseuille : public unary_function<point,point> {
  Float theta;
 public:
  poiseuille(Float t) : theta(t) {}
  point operator() (const point& x) 
   { 
	Float z=dot(x,point(-sin(theta),cos(theta)));
   	return (1-sqr(z))*point(cos(theta),sin(theta)); 
   }
};
geo rotate(const geo& omega, Float angle) {
  geo::nodelist_type p;
  p.resize(omega.n_node());
  geo::nodelist_type::iterator iter_p = p.begin();
  geo::const_iterator_node      sommets = omega.begin_node();
  for(geo::size_type i=0; i!=omega.n_node(); i++)
        {
        *iter_p = point(sommets[i][0]*cos(angle)-sommets[i][1]*sin(angle), 
			sommets[i][1]*cos(angle)+sommets[i][0]*sin(angle));
        iter_p++;
        }
  geo g (p,omega);
  return g;
}
class normal : public unary_function<point,point> {
  Float theta;
 public:
  normal(Float t) : theta(t) {}
  point operator() (const point& x) { return point(-sin(theta),cos(theta)); }
};
space the_space(geo g,string app,Float theta) {
  space Vh(g,app,"vector");
  Vh.block("left");
  Vh.block("top");
  Vh.lock_components("bottom",normal(theta));
  Vh.lock_components("right",point(-sin(theta),cos(theta)));
  return Vh;
 }
int main(int argc, char**argv) {
  if (argc==1) cerr << argv[0] << " <mesh file> [<tilt angle>] [-quiet]" << endl;
  geo  omega_h (argv[1]);
  Float theta=(argc>2)?atof(argv[2])*acos(-1.)/180:0;
  bool quiet=(argc>3);
  omega_h=rotate(omega_h, theta);
  space Vh = the_space (omega_h, "P2", theta);
  space Qh (omega_h, "P1");
  field uh = interpolate(Vh,poiseuille(theta));
  field ph (Qh, Float(0));
  form a (Vh, Vh, "2D_D", true);
  form b (Vh, Qh, "div", true);  b = -b;
  int   max_iter = 50;
  Float tol      = 1e-12;
  Float r        = 1e+7;
  form ar = a + r*trans(b)*b;
  ssk<Float> fact = ldlt(ar.uu);
  uh.set_locked_boundaries();
  uzawa_abtb (ar.uu, fact, b.uu, uh.u, ph.u, -(ar.ub*uh.b), -(b.ub*uh.b), r, max_iter, tol);
  uh.unset_locked_boundaries();
  field u=interpolate(Vh,poiseuille(theta));
  Float error=(uh-u).max_abs();
  if (!quiet)
   {
  	cout << catchmark("uh") << uh 
	     << catchmark("ph") << ph
	     << catchmark("error") << uh-u;
  	cerr << "Error is " << error << endl;
   }
  Float tol2 = 1e-10;
  if (error > tol2) {
   	cerr << "Error is " << error << " > " << tol2 << endl;
	exit(1);
   }
  return 0;
}
