/* ====================================================================
 * Copyright (c) 2003-2006, 2008  Martin Hauner
 *                                http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

#ifndef _SC_UTIL_ERROR_H
#define _SC_UTIL_ERROR_H

// sc
#include "String.h"

// boost
#include <boost/shared_ptr.hpp>


namespace sc
{

/**
 * general error interface.
 */
class Error
{
public:
  virtual ~Error() {}

  /**
   * \return the error code.
   */
  virtual long getCode() const = 0;

  /**
   * \return the error message of \b only this error. It does ignore
   * the nested error.
   */
  virtual sc::String getMessage() const = 0;

  /**
   * \return the error message of this error and of the nested error.
   */
  virtual sc::String getMessages() const = 0;

  /**
   * \return a nested Error or 0 if the there is none.
   */
  virtual const Error* getNested() const = 0;
};

/**
 * the success error.
 */
class Error;
extern Error* Success;

/**
 * create a new Error.
 * \param code the error code.
 * \param msg  the error msg.
 * \return a new Error.
 */
const Error* createError( long code, const sc::String& msg );

/**
 * create a new Error with a nested Error.
 * \param code the error code.
 * \param msg  the error msg.
 * \param err  the nested error.
 * \return a new Error.
 */
const Error* createError( long code, const sc::String& msg, const Error* err );


/**
 * a helper macro for handling errors
 */
#define SC_ERR(exp)\
{\
  const sc::Error* sc_err = (exp);\
  if( sc_err != sc::Success )\
  {\
    return sc_err;\
  }\
}

} // namespace



typedef boost::shared_ptr<const sc::Error> ScErrorPtr;


#endif //  _SC_UTIL_ERROR_H
